/**
 * @file IPhoneBookRequestIf.h
 *
 * @swcomponent PhoneCallManager
 *
 * @brief This file contains the definition of the IPhoneBookRequestIf class
 *
 * @copyright (C) 2016 Robert Bosch GmbH.
 *            The reproduction, distribution and utilization of this file as
 *            well as the communication of its contents to others without express
 *            authorization is prohibited. Offenders will be held liable for the
 *            payment of damages. All rights reserved in the event of the grant
 *            of a patent, utility model or design.
 *
 * @details This file holds the class definition for IPhoneBookRequestIf class
 *
 * @ingroup IpcWrapper
 */

#ifndef IPhoneBookRequestIf_h
#define IPhoneBookRequestIf_h

#include <string>

namespace com
{
namespace bosch
{
namespace pmapp
{

class IPhoneBookCallBackIf;

class IPhoneBookRequestIf
{
public:

   /**
    * Virtual destructor of IPhoneBookRequestIf class
    *
    * @param[in]
    * @param[out]
    * @param[in,out]
    *
    * @return
    *
    */
   virtual ~IPhoneBookRequestIf() {}

   /**
    * This function is called to set the callback interface object of the Phonebook Application.
    *
    * @param[in] IPhoneBookCallBackIf* - Pointer to the Callback Interface of phonebook client.
    *
    * @param[out]
    * @param[in,out]
    *
    * @return
    *
    */
   virtual void setPhoneBookCallBackIf(IPhoneBookCallBackIf*) = 0;

   /**
    * This function is used to call the searchPhoneBook method of the Phonebook Application interface
    * to get the contact details of the provided phone number.
    *
    * @param[in] std::string deviceAddress - Device Address of the device in which the contact needs to be queried.
    *            string phoneNumber - Phone number for which the contact details need to be queried.
    *
    * @param[out] act_t - Asynchronous Callback Token received from the ASF FW.
    *
    * @param[in,out]
    *
    * @return
    *
    */
   virtual act_t searchPhoneBook(const std::string deviceAddress, const std::string phoneNumber) = 0;

   /**
    * This function is used to call the searchPhoneBook method of the Phonebook Application interface
    * to get the contact details of the provided phone number.
    *
    * @param[in] - uint8 deviceHandle - Device handle of the device in which the contact needs to be queried.
    *              string phoneNumber - Phone number for which the contact details need to be queried.
    *
    * @param[out] act_t - Asynchronous Callback Token received from the ASF FW.
    *
    * @param[in,out]
    *
    * @return
    *
    */
   virtual act_t searchPhoneBook(const uint8 deviceHandle, const std::string phoneNumber) = 0;
   
protected:
   /**
    * Parameterized constructor of IPhoneBookRequestIf
    *
    * @param[in]
    * @param[out]
    * @param[in,out]
    *
    * @return
    *
    */
   IPhoneBookRequestIf(bool serviceAvailability){_serviceAvailability = serviceAvailability;}

   /**
    * Once the service is available, this function is called to register
    * the required properties of Phonebook application.
    *
    * @param[in]
    * @param[out]
    * @param[in,out]
    *
    * @return
    *
    */
   virtual void registerProperties() = 0;

   /**
    * Once the service becomes unavailable, this function is called to de-register
    * the registered properties of Phonebook application.
    *
    * @param[in]
    * @param[out]
    * @param[in,out]
    *
    * @return
    *
    */
   virtual void deregisterProperties() = 0;

   bool _serviceAvailability;
};

} // namespace pmapp
} // namespace bosch
} // namespace com

#endif //IPhoneBookRequestIf_h
