/**
 * @file CcaMessageHandler.cpp
 *
 * @swcomponent PhoneCallManager
 *
 * @brief This file contains the definition of the CcaMessageHandler class
 *
 * @copyright (C) 2016 Robert Bosch GmbH.
 *            The reproduction, distribution and utilization of this file as
 *            well as the communication of its contents to others without express
 *            authorization is prohibited. Offenders will be held liable for the
 *            payment of damages. All rights reserved in the event of the grant
 *            of a patent, utility model or design.
 *
 * @details
 *
 * @ingroup IpcWrapper
 */

#include "CcaMessageHandler.h"
#include "CcaCallStatusListHandler.h"
#include "PmAppTrace.h"

#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#define ETG_DEFAULT_TRACE_CLASS   TR_CLASS_PM_SERVICE
#ifdef VARIANT_S_FTR_ENABLE_FW_ETG_USAGE
#include "trcGenProj/Header/CcaMessageHandler.cpp.trc.h"
#define ETG_DEFAULT_TRACE_CLASS   TR_CLASS_PM_SERVICE
#endif
#endif

#define FUNCTION_ID_PLACE_HOLDER 0xFFFF0
#define FUNCTION_ID_POS_EXTRACTOR (unsigned char)0x04

CcaMessageHandler::CcaMessageHandler():
   _pAhlService(nullptr),
   _ccaActiveCallInHandsetHandler(nullptr),
   _ccaActivePhoneDeviceHandler(nullptr),
   _ccaAudioTransferFailedEventHandler(nullptr),
   _ccaAutoWaitingModeOnOffListHandler(nullptr),
   _ccaCallOperationFailedEventHandler(nullptr),
   _ccaCallStatusNoticeExtendedHandler(nullptr),
   _ccaCallStatusNoticeHandler(nullptr),
   _ccaCellDeviceNameHandler(nullptr),
   _ccaCellNetworkStatusHandler(nullptr),
   _ccaCellSignalStrengthHandler(nullptr),
   _ccaCellStateOfChargeHandler(nullptr),
   _ccaDeviceRingtoneListHandler(nullptr),
   _ccaExternalVRStateHandler(nullptr),
   _ccaHandsFreeEnhancedCallControlHandler(nullptr),
   _ccaHFServiceHandler(nullptr),
   _ccaInComingCallEventHandler(nullptr),
   _ccaIncomingCallRejectedEventHandler(nullptr),
   _ccaMicrophoneMuteStateHandler(nullptr),
   _ccaNetworkOperatorHandler(nullptr),
   _ccaOutgoingCallFailedEventHandler(nullptr),
   _ccaRingtoneMuteStateHandler(nullptr),
   _ccaRingtonePlaybackStatusHandler(nullptr),
   _ccaSendDTMFFailedEventHandler(nullptr),
   _ccaSlcStatusHandler(nullptr),
   _ccaSuppressRingtoneOnOffListHandler(nullptr),
   _ccaSystemWideRingtonesListHandler(nullptr),
   _ccaVehicleAudioHandler(nullptr),
   _ccaVoiceRecognitionExtendedHandler(nullptr),
   _ccaVoiceRecognitionHandler(nullptr),
   _ccaWaitingModeFilePathHandler(nullptr),
   _ccaWaitingModeStateHandler(nullptr),
   _ccaAcceptCallReqHandler(nullptr),
   _ccaActivateExternalVRReqHandler(nullptr),
   _ccaAutoWaitingModeOnOffReqHandler(nullptr),
   _ccaCancelOutgoingCallReqHandler(nullptr),
   _ccaConferenceInReqHandler(nullptr),
   _ccaDeActivateExternalVRReqHandler(nullptr),
   _ccaDialExtendedReqHandler(nullptr),
   _ccaDialReqHandler(nullptr),
   _ccaHangUpCallReqHandler(nullptr),
   _ccaHoldCallReqHandler(nullptr),
   _ccaIgnoreCallReqHandler(nullptr),
   _ccaPlayRingtoneReqHandler(nullptr),
   _ccaQueryTelephoneNumberReqHandler(nullptr),
   _ccaRedialReqHandler(nullptr),
   _ccaRejectCallReqHandler(nullptr),
   _ccaResumeCallReqHandler(nullptr),
   _ccaSendDTMFReqHandler(nullptr),
   _ccaSetActivePhoneReqHandler(nullptr),
   _ccaSetRingtoneReqHandler(nullptr),
   _ccaSpeedDialReqHandler(nullptr),
   _ccaSplitCallReqHandler(nullptr),
   _ccaStartStopWaitingModeReqHandler(nullptr),
   _ccaStopRingtoneReqHandler(nullptr),
   _ccaSuppressRingtoneOnOffReqHandler(nullptr),
   _ccaSwapCallReqHandler(nullptr),
   _ccaTransferCallToHandsetReqHandler(nullptr),
   _ccaTransferCallToVehicleReqHandler(nullptr),
   _ccaVoiceMailReqHandler(nullptr)
{
   ETG_TRACE_USR4(("CcaMessageHandler::CcaMessageHandler entered"));

   // Instantiating "CcaCallStatusListHandler"
   CcaCallStatusListHandler::getInstance();
}

CcaMessageHandler::CcaMessageHandler(AhlService* pAhlService)
{
   ETG_TRACE_USR4(("CcaMessageHandler::CcaMessageHandler entered"));

   _pAhlService = pAhlService;

   createCcaMsgHandlers();
}

CcaMessageHandler::~CcaMessageHandler()
{
   ETG_TRACE_USR4(("CcaMessageHandler::~CcaMessageHandler entered"));

   _functionMap.clear();

   deleteCcaMsgHandlers();
}

CcaFunction* CcaMessageHandler::getMsgHandlerFromFktId(tU16 u16FktId)
{
   ETG_TRACE_USR4(("CcaMessageHandler::getMsgHandlerFromFktId entered"));

   CcaFunction* msgHandler = nullptr;

   std::map<tU16, CcaFunction*>::const_iterator iterFunctionMap = _functionMap.find(u16FktId);

   if(iterFunctionMap != _functionMap.end())
   {
      msgHandler = iterFunctionMap->second;
   }

   return msgHandler;
}

CcaFunction* CcaMessageHandler::getMsgHandlerFromTokenId(tU64 tokenId)
{
   ETG_TRACE_USR4(("CcaMessageHandler::getMsgHandlerFromTokenId entered"));

   tU16 u16FktId = (tokenId & FUNCTION_ID_PLACE_HOLDER) >> FUNCTION_ID_POS_EXTRACTOR;
   return (getMsgHandlerFromFktId(u16FktId));
}

void CcaMessageHandler::addFunction(tU16 u16FktId, CcaFunction* msgHandler)
{
   ETG_TRACE_USR4(("CcaMessageHandler::addFunction entered"));
   _functionMap.insert(std::pair<tU16, CcaFunction*>(u16FktId, msgHandler));
}

void CcaMessageHandler::processMessage(amt_tclServiceData* pInMessage)
{
   ETG_TRACE_USR4(("CcaMessageHandler::processMessage entered for CCA message"));
   ETG_TRACE_USR4(("fkt ID: %u", pInMessage->u16GetFunctionID()));

   CcaFunction* msgHandler = getMsgHandlerFromFktId(pInMessage->u16GetFunctionID());

   if(msgHandler)
   {
      msgHandler->handleMessage(pInMessage);
   }
   else
   {
      // Requesting for a function that is not registered by the handler
      ETG_TRACE_ERR(("CcaMessageHandler::Message Handler not registered for fkt: %u", pInMessage->u16GetFunctionID()));
   }
}

void CcaMessageHandler::processMessage(PmCoreResponseData* pmCoreResponseData)
{
   ETG_TRACE_USR4(("CcaMessageHandler::processMessage entered for CPP Response message"));

   CcaFunction* msgHandler = getMsgHandlerFromTokenId(pmCoreResponseData->getTokenId());

   if(msgHandler)
   {
      msgHandler->handleMessage(pmCoreResponseData);
   }
   else
   {
      // CPP Response for a function that is not registered by the handler
      ETG_TRACE_ERR(("CcaMessageHandler::Message Handler not found"));
   }
}

void CcaMessageHandler::onPropertyUpdate(PropertyUpdate* pPropertyUpdate)
{
   ETG_TRACE_USR4(("CcaMessageHandler::onPropertyUpdate entered"));

   CcaFunction* msgHandler = getMsgHandlerFromFktId(pPropertyUpdate->u16GetFunctionID());

   if(msgHandler)
   {
      msgHandler->onPropertyUpdate(pPropertyUpdate);
   }
   else
   {
      // Update received for unregistered property
      ETG_TRACE_ERR(("Update for property not registered : %d", pPropertyUpdate->u16GetFunctionID()));
   }
}

void CcaMessageHandler::createCcaMsgHandlers()
{
   ETG_TRACE_USR4(("CcaMessageHandler::createCcaMsgHandlers entered"));

   // Creating and registering property handlers
   _ccaActiveCallInHandsetHandler = new CcaActiveCallInHandsetHandler(_pAhlService);
   addFunction(MOST_TELFI_C_U16_ACTIVECALLINHANDSET, _ccaActiveCallInHandsetHandler);

   _ccaActivePhoneDeviceHandler = new CcaActivePhoneDeviceHandler(_pAhlService);
   addFunction(MOST_TELFI_C_U16_ACTIVEPHONEDEVICE, _ccaActivePhoneDeviceHandler);

   _ccaAudioTransferFailedEventHandler = new CcaAudioTransferFailedEventHandler(_pAhlService);
   addFunction(MOST_TELFI_C_U16_AUDIOTRANSFERFAILEDEVENT, _ccaAudioTransferFailedEventHandler);

   _ccaAutoWaitingModeOnOffListHandler = new CcaAutoWaitingModeOnOffListHandler(_pAhlService);
   addFunction(MOST_TELFI_C_U16_AUTOWAITINGMODEONOFFLIST, _ccaAutoWaitingModeOnOffListHandler);

   _ccaCallOperationFailedEventHandler = new CcaCallOperationFailedEventHandler(_pAhlService);
   addFunction(MOST_TELFI_C_U16_CALLOPERATIONFAILEDEVENT, _ccaCallOperationFailedEventHandler);

   _ccaCallStatusNoticeExtendedHandler = new CcaCallStatusNoticeExtendedHandler(_pAhlService);
   addFunction(MOST_TELFI_C_U16_CALLSTATUSNOTICEEXTENDED, _ccaCallStatusNoticeExtendedHandler);

   _ccaCallStatusNoticeHandler = new CcaCallStatusNoticeHandler (_pAhlService);
   addFunction(MOST_TELFI_C_U16_CALLSTATUSNOTICE, _ccaCallStatusNoticeHandler);

   _ccaCellDeviceNameHandler = new CcaCellDeviceNameHandler (_pAhlService);
   addFunction(MOST_TELFI_C_U16_CELLDEVICENAME, _ccaCellDeviceNameHandler);

   _ccaCellNetworkStatusHandler = new CcaCellNetworkStatusHandler(_pAhlService);
   addFunction(MOST_TELFI_C_U16_CELLNETWORKSTATUS, _ccaCellNetworkStatusHandler);

   _ccaCellSignalStrengthHandler = new CcaCellSignalStrengthHandler(_pAhlService);
   addFunction(MOST_TELFI_C_U16_CELLSIGNALSTRENGTH, _ccaCellSignalStrengthHandler);

   _ccaCellStateOfChargeHandler = new CcaCellStateOfChargeHandler(_pAhlService);
   addFunction(MOST_TELFI_C_U16_CELLSTATEOFCHARGE, _ccaCellStateOfChargeHandler);

   _ccaDeviceRingtoneListHandler = new CcaDeviceRingtoneListHandler(_pAhlService);
   addFunction(MOST_TELFI_C_U16_DEVICERINGTONELIST, _ccaDeviceRingtoneListHandler);

   _ccaExternalVRStateHandler = new CcaExternalVRStateHandler(_pAhlService);
   addFunction(MOST_TELFI_C_U16_EXTERNALVRSTATE, _ccaExternalVRStateHandler);

   _ccaHandsFreeEnhancedCallControlHandler = new CcaHandsFreeEnhancedCallControlHandler(_pAhlService);
   addFunction(MOST_TELFI_C_U16_HANDSFREEENHANCEDCALLCONTROL, _ccaHandsFreeEnhancedCallControlHandler);

   _ccaHFServiceHandler = new CcaHFServiceHandler(_pAhlService);
   addFunction(MOST_TELFI_C_U16_HFSERVICE, _ccaHFServiceHandler);

   _ccaInComingCallEventHandler = new CcaInComingCallEventHandler(_pAhlService);
   addFunction(MOST_TELFI_C_U16_INCOMINGCALLEVENT, _ccaInComingCallEventHandler);

   _ccaIncomingCallRejectedEventHandler = new CcaIncomingCallRejectedEventHandler(_pAhlService);
   addFunction(MOST_TELFI_C_U16_INCOMINGCALLREJECTEDEVENT, _ccaIncomingCallRejectedEventHandler);

   _ccaMicrophoneMuteStateHandler = new CcaMicrophoneMuteStateHandler(_pAhlService);
   addFunction(MOST_TELFI_C_U16_MICROPHONEMUTESTATE, _ccaMicrophoneMuteStateHandler);

   _ccaNetworkOperatorHandler = new CcaNetworkOperatorHandler(_pAhlService);
   addFunction(MOST_TELFI_C_U16_NETWORKOPERATOR, _ccaNetworkOperatorHandler);

   _ccaOutgoingCallFailedEventHandler = new CcaOutgoingCallFailedEventHandler(_pAhlService);
   addFunction(MOST_TELFI_C_U16_OUTGOINGCALLFAILEDEVENT, _ccaOutgoingCallFailedEventHandler);

   _ccaRingtoneMuteStateHandler = new CcaRingtoneMuteStateHandler(_pAhlService);
   addFunction(MOST_TELFI_C_U16_RINGTONEMUTESTATE, _ccaRingtoneMuteStateHandler);

   _ccaRingtonePlaybackStatusHandler = new CcaRingtonePlaybackStatusHandler(_pAhlService);
   addFunction(MOST_TELFI_C_U16_RINGTONEPLAYBACKSTATUS, _ccaRingtonePlaybackStatusHandler);

   _ccaSendDTMFFailedEventHandler = new CcaSendDTMFFailedEventHandler(_pAhlService);
   addFunction(MOST_TELFI_C_U16_SENDDTMFFAILEDEVENT, _ccaSendDTMFFailedEventHandler);

   _ccaSlcStatusHandler = new CcaSlcStatusHandler(_pAhlService);
   addFunction(MOST_TELFI_C_U16_SLCSTATUS, _ccaSlcStatusHandler);

   _ccaSuppressRingtoneOnOffListHandler = new CcaSuppressRingtoneOnOffListHandler(_pAhlService);
   addFunction(MOST_TELFI_C_U16_SUPPRESSRINGTONEONOFFLIST, _ccaSuppressRingtoneOnOffListHandler);

   _ccaSystemWideRingtonesListHandler = new CcaSystemWideRingtonesListHandler(_pAhlService);
   addFunction(MOST_TELFI_C_U16_SYSTEMWIDERINGTONESLIST, _ccaSystemWideRingtonesListHandler);

   _ccaVehicleAudioHandler = new CcaVehicleAudioHandler(_pAhlService);
   addFunction(MOST_TELFI_C_U16_VEHICLEAUDIO, _ccaVehicleAudioHandler);

   _ccaVoiceRecognitionExtendedHandler = new CcaVoiceRecognitionExtendedHandler(_pAhlService);
   addFunction(MOST_TELFI_C_U16_BTDEVICEVOICERECOGNITIONEXTENDED, _ccaVoiceRecognitionExtendedHandler);

   _ccaVoiceRecognitionHandler = new CcaVoiceRecognitionHandler(_pAhlService);
   addFunction(MOST_TELFI_C_U16_BTDEVICEVOICERECOGNITION, _ccaVoiceRecognitionHandler);

   _ccaWaitingModeFilePathHandler = new CcaWaitingModeFilePathHandler(_pAhlService);
   addFunction(MOST_TELFI_C_U16_WAITINGMODEFILEPATH, _ccaWaitingModeFilePathHandler);

   _ccaWaitingModeStateHandler = new CcaWaitingModeStateHandler(_pAhlService);
   addFunction(MOST_TELFI_C_U16_WAITINGMODESTATE, _ccaWaitingModeStateHandler);

   // Creating and registering method handlers
   _ccaAcceptCallReqHandler = new CcaAcceptCallReqHandler(_pAhlService);
   addFunction(MOST_TELFI_C_U16_ACCEPTCALL, _ccaAcceptCallReqHandler);

   _ccaActivateExternalVRReqHandler = new CcaActivateExternalVRReqHandler(_pAhlService);
   addFunction(MOST_TELFI_C_U16_ACTIVATEEXTERNALVR, _ccaActivateExternalVRReqHandler);

   _ccaAutoWaitingModeOnOffReqHandler = new CcaAutoWaitingModeOnOffReqHandler(_pAhlService);
   addFunction(MOST_TELFI_C_U16_AUTOWAITINGMODEONOFF, _ccaAutoWaitingModeOnOffReqHandler);

   _ccaCancelOutgoingCallReqHandler = new CcaCancelOutgoingCallReqHandler(_pAhlService);
   addFunction(MOST_TELFI_C_U16_CANCELOUTGOINGCALL, _ccaCancelOutgoingCallReqHandler);

   _ccaDeActivateExternalVRReqHandler = new CcaDeActivateExternalVRReqHandler(_pAhlService);
   addFunction(MOST_TELFI_C_U16_DEACTIVATEEXTERNALVR, _ccaDeActivateExternalVRReqHandler);

   _ccaConferenceInReqHandler = new CcaConferenceInReqHandler(_pAhlService);
   addFunction(MOST_TELFI_C_U16_CONFERENCEIN, _ccaConferenceInReqHandler);

   _ccaDialExtendedReqHandler = new CcaDialExtendedReqHandler(_pAhlService);
   addFunction(MOST_TELFI_C_U16_DIALEXTENDED, _ccaDialExtendedReqHandler);

   _ccaDialReqHandler = new CcaDialReqHandler(_pAhlService);
   addFunction(MOST_TELFI_C_U16_DIAL, _ccaDialReqHandler);

   _ccaHangUpCallReqHandler = new CcaHangUpCallReqHandler(_pAhlService);
   addFunction(MOST_TELFI_C_U16_HANGUPCALL, _ccaHangUpCallReqHandler);

   _ccaHoldCallReqHandler = new CcaHoldCallReqHandler(_pAhlService);
   addFunction(MOST_TELFI_C_U16_HOLDCALL, _ccaHoldCallReqHandler);

   _ccaIgnoreCallReqHandler = new CcaIgnoreCallReqHandler(_pAhlService);
   addFunction(MOST_TELFI_C_U16_IGNORECALL, _ccaIgnoreCallReqHandler);

   _ccaPlayRingtoneReqHandler = new CcaPlayRingtoneReqHandler(_pAhlService);
   addFunction(MOST_TELFI_C_U16_PLAYRINGTONE, _ccaPlayRingtoneReqHandler);

   _ccaQueryTelephoneNumberReqHandler = new CcaQueryTelephoneNumberReqHandler(_pAhlService);
   addFunction(MOST_TELFI_C_U16_QUERYTELEPHONENUMBER, _ccaQueryTelephoneNumberReqHandler);

   _ccaRedialReqHandler = new CcaRedialReqHandler(_pAhlService);
   addFunction(MOST_TELFI_C_U16_REDIAL, _ccaRedialReqHandler);

   _ccaRejectCallReqHandler = new CcaRejectCallReqHandler(_pAhlService);
   addFunction(MOST_TELFI_C_U16_REJECTCALL, _ccaRejectCallReqHandler);

   _ccaResumeCallReqHandler = new CcaResumeCallReqHandler(_pAhlService);
   addFunction(MOST_TELFI_C_U16_RESUMECALL, _ccaResumeCallReqHandler);

   _ccaSendDTMFReqHandler = new CcaSendDTMFReqHandler(_pAhlService);
   addFunction(MOST_TELFI_C_U16_SENDDTMF, _ccaSendDTMFReqHandler);

   _ccaSetActivePhoneReqHandler = new CcaSetActivePhoneReqHandler(_pAhlService);
   addFunction(MOST_TELFI_C_U16_SETACTIVEPHONE, _ccaSetActivePhoneReqHandler);

   _ccaSetRingtoneReqHandler = new CcaSetRingtoneReqHandler(_pAhlService);
   addFunction(MOST_TELFI_C_U16_SETRINGTONE, _ccaSetRingtoneReqHandler);

   _ccaSpeedDialReqHandler = new CcaSpeedDialReqHandler(_pAhlService);
   addFunction(MOST_TELFI_C_U16_SPEEDDIAL, _ccaSpeedDialReqHandler);

   _ccaSplitCallReqHandler = new CcaSplitCallReqHandler(_pAhlService);
   addFunction(MOST_TELFI_C_U16_SPLITCALL, _ccaSplitCallReqHandler);

   _ccaStartStopWaitingModeReqHandler = new CcaStartStopWaitingModeReqHandler(_pAhlService);
   addFunction(MOST_TELFI_C_U16_STARTSTOPWAITINGMODE, _ccaStartStopWaitingModeReqHandler);

   _ccaStopRingtoneReqHandler = new CcaStopRingtoneReqHandler(_pAhlService);
   addFunction(MOST_TELFI_C_U16_STOPRINGTONE, _ccaStopRingtoneReqHandler);

   _ccaSuppressRingtoneOnOffReqHandler = new CcaSuppressRingtoneOnOffReqHandler(_pAhlService);
   addFunction(MOST_TELFI_C_U16_SUPPRESSRINGTONEONOFF, _ccaSuppressRingtoneOnOffReqHandler);

   _ccaSwapCallReqHandler = new CcaSwapCallReqHandler(_pAhlService);
   addFunction(MOST_TELFI_C_U16_SWAPCALL, _ccaSwapCallReqHandler);

   _ccaTransferCallToHandsetReqHandler = new CcaTransferCallToHandsetReqHandler(_pAhlService);
   addFunction(MOST_TELFI_C_U16_TRANSFERCALLTOHANDSET, _ccaTransferCallToHandsetReqHandler);

   _ccaTransferCallToVehicleReqHandler = new CcaTransferCallToVehicleReqHandler(_pAhlService);
   addFunction(MOST_TELFI_C_U16_TRANSFERCALLTOVEHICLE, _ccaTransferCallToVehicleReqHandler);

   _ccaVoiceMailReqHandler = new CcaVoiceMailReqHandler(_pAhlService);
   addFunction(MOST_TELFI_C_U16_VOICEMAIL, _ccaVoiceMailReqHandler);
}

void CcaMessageHandler::deleteCcaMsgHandlers()
{
   ETG_TRACE_USR4(("CcaMessageHandler::deleteCcaMsgHandlers entered"));

   // Deleting property handlers
   if(_ccaActiveCallInHandsetHandler)
   {
      delete _ccaActiveCallInHandsetHandler;
      _ccaActiveCallInHandsetHandler = nullptr;
   }

   if(_ccaActivePhoneDeviceHandler)
   {
      delete _ccaActivePhoneDeviceHandler;
      _ccaActivePhoneDeviceHandler = nullptr;
   }

   if(_ccaAudioTransferFailedEventHandler)
   {
      delete _ccaAudioTransferFailedEventHandler;
      _ccaAudioTransferFailedEventHandler = nullptr;
   }

   if(_ccaAutoWaitingModeOnOffListHandler)
   {
      delete _ccaAutoWaitingModeOnOffListHandler;
      _ccaAutoWaitingModeOnOffListHandler = nullptr;
   }

   if(_ccaCallOperationFailedEventHandler)
   {
      delete _ccaCallOperationFailedEventHandler;
      _ccaCallOperationFailedEventHandler = nullptr;
   }

   if(_ccaCallStatusNoticeExtendedHandler)
   {
      delete _ccaCallStatusNoticeExtendedHandler;
      _ccaCallStatusNoticeExtendedHandler = nullptr;
   }

   if(_ccaCallStatusNoticeHandler)
   {
      delete _ccaCallStatusNoticeHandler;
      _ccaCallStatusNoticeHandler = nullptr;
   }

   if(_ccaCellDeviceNameHandler)
   {
      delete _ccaCellDeviceNameHandler;
      _ccaCellDeviceNameHandler = nullptr;
   }

   if(_ccaCellNetworkStatusHandler)
   {
      delete _ccaCellNetworkStatusHandler;
      _ccaCellNetworkStatusHandler = nullptr;
   }

   if(_ccaCellSignalStrengthHandler)
   {
      delete _ccaCellSignalStrengthHandler;
      _ccaCellSignalStrengthHandler = nullptr;
   }

   if(_ccaCellStateOfChargeHandler)
   {
      delete _ccaCellStateOfChargeHandler;
      _ccaCellStateOfChargeHandler = nullptr;
   }

   if(_ccaDeviceRingtoneListHandler)
   {
      delete _ccaDeviceRingtoneListHandler;
      _ccaDeviceRingtoneListHandler = nullptr;
   }

   if(_ccaExternalVRStateHandler)
   {
      delete _ccaExternalVRStateHandler;
      _ccaExternalVRStateHandler = nullptr;
   }

   if(_ccaHandsFreeEnhancedCallControlHandler)
   {
      delete _ccaHandsFreeEnhancedCallControlHandler;
      _ccaHandsFreeEnhancedCallControlHandler = nullptr;
   }

   if(_ccaHFServiceHandler)
   {
      delete _ccaHFServiceHandler;
      _ccaHFServiceHandler = nullptr;
   }

   if(_ccaInComingCallEventHandler)
   {
      delete _ccaInComingCallEventHandler;
      _ccaInComingCallEventHandler = nullptr;
   }

   if(_ccaIncomingCallRejectedEventHandler)
   {
      delete _ccaIncomingCallRejectedEventHandler;
      _ccaIncomingCallRejectedEventHandler = nullptr;
   }

   if(_ccaMicrophoneMuteStateHandler)
   {
      delete _ccaMicrophoneMuteStateHandler;
      _ccaMicrophoneMuteStateHandler = nullptr;
   }

   if(_ccaNetworkOperatorHandler)
   {
      delete _ccaNetworkOperatorHandler;
      _ccaNetworkOperatorHandler = nullptr;
   }

   if(_ccaOutgoingCallFailedEventHandler)
   {
      delete _ccaOutgoingCallFailedEventHandler;
      _ccaOutgoingCallFailedEventHandler = nullptr;
   }

   if(_ccaRingtoneMuteStateHandler)
   {
      delete _ccaRingtoneMuteStateHandler;
      _ccaRingtoneMuteStateHandler = nullptr;
   }

   if(_ccaRingtonePlaybackStatusHandler)
   {
      delete _ccaRingtonePlaybackStatusHandler;
      _ccaRingtonePlaybackStatusHandler = nullptr;
   }

   if(_ccaSendDTMFFailedEventHandler)
   {
      delete _ccaSendDTMFFailedEventHandler;
      _ccaSendDTMFFailedEventHandler = nullptr;
   }

   if(_ccaSlcStatusHandler)
   {
      delete _ccaSlcStatusHandler;
      _ccaSlcStatusHandler = nullptr;
   }

   if(_ccaSuppressRingtoneOnOffListHandler)
   {
      delete _ccaSuppressRingtoneOnOffListHandler;
      _ccaSuppressRingtoneOnOffListHandler = nullptr;
   }

   if(_ccaSystemWideRingtonesListHandler)
   {
      delete _ccaSystemWideRingtonesListHandler;
      _ccaSystemWideRingtonesListHandler = nullptr;
   }

   if(_ccaVehicleAudioHandler)
   {
      delete _ccaVehicleAudioHandler;
      _ccaVehicleAudioHandler = nullptr;
   }

   if(_ccaVoiceRecognitionExtendedHandler)
   {
      delete _ccaVoiceRecognitionExtendedHandler;
      _ccaVoiceRecognitionExtendedHandler = nullptr;
   }

   if(_ccaVoiceRecognitionHandler)
   {
      delete _ccaVoiceRecognitionHandler;
      _ccaVoiceRecognitionHandler = nullptr;
   }

   if(_ccaWaitingModeFilePathHandler)
   {
      delete _ccaWaitingModeFilePathHandler;
      _ccaWaitingModeFilePathHandler = nullptr;
   }

   if(_ccaWaitingModeStateHandler)
   {
      delete _ccaWaitingModeStateHandler;
      _ccaWaitingModeStateHandler = nullptr;
   }

   // Deleting method handlers
   if(_ccaAcceptCallReqHandler)
   {
      delete _ccaAcceptCallReqHandler;
      _ccaAcceptCallReqHandler = nullptr;
   }

   if(_ccaActivateExternalVRReqHandler)
   {
      delete _ccaActivateExternalVRReqHandler;
      _ccaActivateExternalVRReqHandler = nullptr;
   }

   if(_ccaAutoWaitingModeOnOffReqHandler)
   {
      delete _ccaAutoWaitingModeOnOffReqHandler;
      _ccaAutoWaitingModeOnOffReqHandler = nullptr;
   }

   if(_ccaCancelOutgoingCallReqHandler)
   {
      delete _ccaCancelOutgoingCallReqHandler;
      _ccaCancelOutgoingCallReqHandler = nullptr;
   }

   if(_ccaConferenceInReqHandler)
   {
      delete _ccaConferenceInReqHandler;
      _ccaConferenceInReqHandler = nullptr;
   }

   if(_ccaDeActivateExternalVRReqHandler)
   {
      delete _ccaDeActivateExternalVRReqHandler;
      _ccaDeActivateExternalVRReqHandler = nullptr;
   }

   if(_ccaDialExtendedReqHandler)
   {
      delete _ccaDialExtendedReqHandler;
      _ccaDialExtendedReqHandler = nullptr;
   }

   if(_ccaDialReqHandler)
   {
      delete _ccaDialReqHandler;
      _ccaDialReqHandler = nullptr;
   }

   if(_ccaHangUpCallReqHandler)
   {
      delete _ccaHangUpCallReqHandler;
      _ccaHangUpCallReqHandler = nullptr;
   }

   if(_ccaHoldCallReqHandler)
   {
      delete _ccaHoldCallReqHandler;
      _ccaHoldCallReqHandler = nullptr;
   }

   if(_ccaIgnoreCallReqHandler)
   {
      delete _ccaIgnoreCallReqHandler;
      _ccaIgnoreCallReqHandler = nullptr;
   }

   if(_ccaPlayRingtoneReqHandler)
   {
      delete _ccaPlayRingtoneReqHandler;
      _ccaPlayRingtoneReqHandler = nullptr;
   }

   if(_ccaQueryTelephoneNumberReqHandler)
   {
      delete _ccaQueryTelephoneNumberReqHandler;
      _ccaQueryTelephoneNumberReqHandler = nullptr;
   }

   if(_ccaRedialReqHandler)
   {
      delete _ccaRedialReqHandler;
      _ccaRedialReqHandler = nullptr;
   }

   if(_ccaRejectCallReqHandler)
   {
      delete _ccaRejectCallReqHandler;
      _ccaRejectCallReqHandler = nullptr;
   }

   if(_ccaResumeCallReqHandler)
   {
      delete _ccaResumeCallReqHandler;
      _ccaResumeCallReqHandler = nullptr;
   }

   if(_ccaSendDTMFReqHandler)
   {
      delete _ccaSendDTMFReqHandler;
      _ccaSendDTMFReqHandler = nullptr;
   }

   if(_ccaSetActivePhoneReqHandler)
   {
      delete _ccaSetActivePhoneReqHandler;
      _ccaSetActivePhoneReqHandler = nullptr;
   }

   if(_ccaSetRingtoneReqHandler)
   {
      delete _ccaSetRingtoneReqHandler;
      _ccaSetRingtoneReqHandler = nullptr;
   }

   if(_ccaSpeedDialReqHandler)
   {
      delete _ccaSpeedDialReqHandler;
      _ccaSpeedDialReqHandler = nullptr;
   }

   if(_ccaSplitCallReqHandler)
   {
      delete _ccaSplitCallReqHandler;
      _ccaSplitCallReqHandler = nullptr;
   }

   if(_ccaStartStopWaitingModeReqHandler)
   {
      delete _ccaStartStopWaitingModeReqHandler;
      _ccaStartStopWaitingModeReqHandler = nullptr;
   }

   if(_ccaStopRingtoneReqHandler)
   {
      delete _ccaStopRingtoneReqHandler;
      _ccaStopRingtoneReqHandler = nullptr;
   }

   if(_ccaSuppressRingtoneOnOffReqHandler)
   {
      delete _ccaSuppressRingtoneOnOffReqHandler;
      _ccaSuppressRingtoneOnOffReqHandler = nullptr;
   }

   if(_ccaSwapCallReqHandler)
   {
      delete _ccaSwapCallReqHandler;
      _ccaSwapCallReqHandler = nullptr;
   }

   if(_ccaTransferCallToHandsetReqHandler)
   {
      delete _ccaTransferCallToHandsetReqHandler;
      _ccaTransferCallToHandsetReqHandler = nullptr;
   }

   if(_ccaTransferCallToVehicleReqHandler)
   {
      delete _ccaTransferCallToVehicleReqHandler;
      _ccaTransferCallToVehicleReqHandler = nullptr;
   }

   if(_ccaVoiceMailReqHandler)
   {
      delete _ccaVoiceMailReqHandler;
      _ccaVoiceMailReqHandler = nullptr;
   }
}
