/**
 * @file CcaCallStatusListHandler.cpp
 *
 * @swcomponent PhoneCallManager
 *
 * @brief This file contains the type definition of the CcaCallStatusListHandler class
 *
 * @copyright (C) 2016 Robert Bosch GmbH.
 *            The reproduction, distribution and utilization of this file as
 *            well as the communication of its contents to others without express
 *            authorization is prohibited. Offenders will be held liable for the
 *            payment of damages. All rights reserved in the event of the grant
 *            of a patent, utility model or design.
 *
 * @details
 *
 * @ingroup PmApp
 */

#include "CcaCallStatusListHandler.h"
#include "DeviceDetailsListHandler.h"
#include "PmAppTrace.h"

#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#define ETG_DEFAULT_TRACE_CLASS TR_CLASS_PM_SERVICE
#ifdef VARIANT_S_FTR_ENABLE_FW_ETG_USAGE
#include "trcGenProj/Header/CcaCallStatusListHandler.cpp.trc.h"
#define ETG_DEFAULT_TRACE_CLASS TR_CLASS_PM_SERVICE
#endif
#endif

using namespace com::bosch::pmapp;
using namespace pmcore;

std::vector<CcaCallStatus> CcaCallStatusListHandler::_ccaCallStatusList;

CcaCallStatusListHandler::CcaCallStatusListHandler() : _activeDeviceHandle(PM_DEVICEHANDLE_ZERO),
      _audioActiveStatus(false)
{
}

CcaCallStatusListHandler::~CcaCallStatusListHandler()
{
}

void CcaCallStatusListHandler::onCallsReportListChanged(const CallsReportList& callsReportList,
      CcaCallStatusList& ccaCallStatusList)
{
   ETG_TRACE_USR4(("CcaCallStatusListHandler::onCallsReportListChanged() entered"));

   if (!callsReportList.empty())
   {
      for (auto& mapIter : callsReportList)
      {
         BdAddress deviceAddress = mapIter.first;
         auto const ccaCallStatusIter = std::find_if(_ccaCallStatusList.begin(), _ccaCallStatusList.end(),
               [&deviceAddress](CcaCallStatus const& obj){return obj._deviceAddress == deviceAddress;});

         if (_ccaCallStatusList.end() == ccaCallStatusIter)
            continue;

         Multiparty multiparty = mapIter.second._multiparty;
         CallAttributesList callAttributesList = mapIter.second._callAttributesList;

         // Removing the ended calls and updating the existing calls
         for (CallCount callcount = 0; callcount < MAX_DEVICE_CALL_INSTANCES_COUNT_FOR_CCA; ++callcount)
         {
            TelephoneNumber telephoneNumber = (ccaCallStatusIter + callcount)->_telephoneNumber;

            if (CallAttributes() == ((ccaCallStatusIter + callcount)->_callAttributes))
               continue;

            auto iter = std::find_if(callAttributesList.begin(), callAttributesList.end(),
                  [&telephoneNumber](std::pair<TelephoneNumber, CallAttributes> const& obj)
                  {return obj.first == telephoneNumber;});

            if (callAttributesList.end() != iter)
            {
               // update call attributes
               (ccaCallStatusIter + callcount)->_callAttributes = iter->second;
               (ccaCallStatusIter + callcount)->_multiparty = multiparty;
            }
            else
            {
               // remove the ended call or simply clear the corresponding call related info
               (ccaCallStatusIter + callcount)->resetCallInfo();
            }
         }

         // Filling the newly added calls in the available slot
         for (auto& callAttributesListIter : callAttributesList)
         {
            TelephoneNumber telephoneNumber = callAttributesListIter.first;
            CallInstance callInstance = callAttributesListIter.second._instance;
            auto iter = std::find_if(ccaCallStatusIter,
                  ccaCallStatusIter + MAX_DEVICE_CALL_INSTANCES_COUNT_FOR_CCA,
                  [&callInstance](CcaCallStatus const& obj){return obj._callAttributes._instance == callInstance;});

            if (iter == ccaCallStatusIter + MAX_DEVICE_CALL_INSTANCES_COUNT_FOR_CCA)
            {
               // call not found. Hence filling the corresponding info in the free slot
               for (CallCount callcount = 0; callcount < MAX_DEVICE_CALL_INSTANCES_COUNT_FOR_CCA; ++callcount)
               {
                  if (CallAttributes() == ((ccaCallStatusIter + callcount)->_callAttributes))
                  {
                     (ccaCallStatusIter + callcount)->_telephoneNumber = telephoneNumber;
                     (ccaCallStatusIter + callcount)->_callAttributes = callAttributesListIter.second;
                     (ccaCallStatusIter + callcount)->_multiparty = multiparty;
                     break;
                  }
               }
            }
         }
      }

      printCcaCallStatusList();

      for (CallCount ccaCallStatusCallcount = 0; ccaCallStatusCallcount < _ccaCallStatusList.size();
            ccaCallStatusCallcount = static_cast<CallCount>(ccaCallStatusCallcount + MAX_DEVICE_CALL_INSTANCES_COUNT_FOR_CCA))
      {
         BdAddress deviceAddress = _ccaCallStatusList[ccaCallStatusCallcount]._deviceAddress;

         ETG_TRACE_USR4(("CcaCallStatusListHandler::onCallsReportListChanged() entered: deviceAddress: %s",
               deviceAddress.c_str()));

         auto callsReportListIter = std::find_if(callsReportList.begin(), callsReportList.end(),
               [&deviceAddress](std::pair<BdAddress, CallsReport> const& obj)
               {return obj.first == deviceAddress;});

         if (callsReportList.end() == callsReportListIter)
         {
            for (CallCount callcount = 0; callcount < MAX_DEVICE_CALL_INSTANCES_COUNT_FOR_CCA; ++callcount)
            {
               _ccaCallStatusList[ccaCallStatusCallcount+callcount].resetCallInfo();
            }
         }
         else
         {
            ETG_TRACE_USR4(("CcaCallStatusListHandler::onCallsReportListChanged(): callsReportListIter.size: %u",
                  callsReportListIter->second._callAttributesList.size()));
         }
      }

      ccaCallStatusList = _ccaCallStatusList;
   }
   else
   {
      // Reset all call instances
      for_each(_ccaCallStatusList.begin(), _ccaCallStatusList.end(), [](CcaCallStatus & obj){obj.resetCallInfo();});
   }

   if (_ccaCallStatusList.empty())
   {
      ETG_TRACE_USR4(("CcaCallStatusListHandler:: Empty callsReportList "));

      for (CallCount callCount = 0; callCount < MAX_DEVICE_CALL_INSTANCES_COUNT_FOR_CCA; ++callCount)
      {
         ccaCallStatusList.emplace_back(CcaCallStatus());
      }
   }
   printCcaCallStatusList();
}

void CcaCallStatusListHandler::onActivePassiveDeviceListChanged(
      const ActivePassiveDeviceListMap& activePassiveDeviceListMap, const DeviceHandle activeDeviceHandle)
{
   ETG_TRACE_USR4(("CcaCallStatusListHandler::onActivePassiveDeviceListChanged() entered"));

   _activeDeviceHandle = activeDeviceHandle;

   BdAddressList disconnectedDevices;

   // Removing the disconnected devices from the list- "_ccaCallStatusList"
   for (auto ccaCallStatus = _ccaCallStatusList.begin(); (ccaCallStatus != _ccaCallStatusList.end());
         ccaCallStatus += MAX_DEVICE_CALL_INSTANCES_COUNT_FOR_CCA)
   {
      auto iter = activePassiveDeviceListMap.find(ccaCallStatus->_deviceAddress);
      if (iter == activePassiveDeviceListMap.end())
      {
         disconnectedDevices.emplace_back(ccaCallStatus->_deviceAddress);
      }
   }

   for (auto& disconnectedDeviceAddress : disconnectedDevices)
   {
      // Removing all the other requests in the list received from the same device address
      _ccaCallStatusList.erase(std::remove_if(_ccaCallStatusList.begin(),
            _ccaCallStatusList.end(),
            [&disconnectedDeviceAddress](CcaCallStatus const& obj)
            {return (obj._deviceAddress == disconnectedDeviceAddress);}),
            _ccaCallStatusList.end());
   }

   // Adding the recently connected device(s)
   for (auto& connectedDevice : activePassiveDeviceListMap)
   {
      auto iter = std::find_if(_ccaCallStatusList.begin(), _ccaCallStatusList.end(),
            [&connectedDevice](CcaCallStatus const& obj)
            {return obj._deviceAddress == connectedDevice.first;});

      if (_ccaCallStatusList.end() != iter)
      {
         AudioActiveStatus audioActiveStatus = _audioActiveStatus ?
               (activeDeviceHandle == iter->_deviceHandle) : false;

         for (CallCount callCount = 0; callCount < MAX_DEVICE_CALL_INSTANCES_COUNT_FOR_CCA; ++callCount)
         {
            (iter + callCount)->_audioActiveStatus = audioActiveStatus;
         }
         continue;
      }

      DeviceHandle deviceHandle = PM_DEVICEHANDLE_ZERO;
      ::com::bosch::pmapp::DeviceDetailsListHandler::getInstance().getDeviceHandle(
            connectedDevice.first, deviceHandle);

      if (PM_DEVICEHANDLE_ZERO == deviceHandle)
      {
         // invalid device handle. Hence continue.
         continue;
      }

      CcaCallStatus ccaCallStatus(connectedDevice.first, deviceHandle);

      // Updating the Phone call audio status accordingly
      ccaCallStatus._audioActiveStatus = _audioActiveStatus ? (activeDeviceHandle == deviceHandle) : false;

      for (CallCount callInstancesCount = 0; callInstancesCount < MAX_DEVICE_CALL_INSTANCES_COUNT_FOR_CCA;
            ++callInstancesCount)
      {
         _ccaCallStatusList.emplace_back(ccaCallStatus);
      }
   }

   printCcaCallStatusList();
}

void CcaCallStatusListHandler::printCcaCallStatusList()
{
   ETG_TRACE_USR4(("printCcaCallStatusList() entered"));

   for (auto& iter : _ccaCallStatusList)
   {
      ETG_TRACE_USR4(("CcaCallStatusListHandler::_deviceAddress: %s", iter._deviceAddress.c_str()));
      ETG_TRACE_USR4(("CcaCallStatusListHandler::_deviceHandle: %u", iter._deviceHandle));

      ETG_TRACE_USR4(("CcaCallStatusListHandler::_instance: %u", iter._callAttributes._instance));
      ETG_TRACE_USR4(("CcaCallStatusListHandler::_state: %s", iter._callAttributes._state.c_str()));
      ETG_TRACE_USR4(("CcaCallStatusListHandler::_durationHr: %u", iter._callAttributes._durationHr));
      ETG_TRACE_USR4(("CcaCallStatusListHandler::_durationMin: %u", iter._callAttributes._durationMin));
      ETG_TRACE_USR4(("CcaCallStatusListHandler::_durationSec: %u", iter._callAttributes._durationSec));
      ETG_TRACE_USR4(("CcaCallStatusListHandler::_telephoneNumber: %s", iter._telephoneNumber.c_str()));
      ETG_TRACE_USR4(("CcaCallStatusListHandler::_mode: %u", iter._callAttributes._mode));
      ETG_TRACE_USR4(("CcaCallStatusListHandler::_type: %u", iter._callAttributes._type));
      ETG_TRACE_USR4(("CcaCallStatusListHandler::_direction: %u", iter._callAttributes._direction));

      ETG_TRACE_USR4(("CcaCallStatusListHandler::_multiparty: %u", iter._multiparty));
      ETG_TRACE_USR4(("CcaCallStatusListHandler::_audioActiveStatus: %u", iter._audioActiveStatus));
   }
}

void CcaCallStatusListHandler::convertCCACallInstance(const CCACallInstance ccaCallInstance, BdAddress& deviceAddress,
      CallInstance& pmcoreCallInstance)
{
   ETG_TRACE_USR4(("convertCCACallInstance() entered with ccaCallInstance: %u", ccaCallInstance));

   if (_ccaCallStatusList.size() <= ccaCallInstance)
   {
      pmcoreCallInstance = CALL_INSTANCE_DEFAULT;
      deviceAddress = "";
   }
   else
   {
      deviceAddress = _ccaCallStatusList[ccaCallInstance]._deviceAddress;
      pmcoreCallInstance = _ccaCallStatusList[ccaCallInstance]._callAttributes._instance;
   }

   printCcaCallStatusList();
}

void CcaCallStatusListHandler::getCCACallInstance(const BdAddress& deviceAddress,
      const CallInstance pmcoreCallInstance, CCACallInstance& ccaCallInstance)
{
   ETG_TRACE_USR4(("getCCACallInstance() entered with deviceAddress: %20s and pmcoreCallInstance: %u",
         deviceAddress.c_str(), pmcoreCallInstance));

   printCcaCallStatusList();

   for (ccaCallInstance = 0; ccaCallInstance < _ccaCallStatusList.size(); ++ccaCallInstance)
   {
      if ((deviceAddress == _ccaCallStatusList[ccaCallInstance]._deviceAddress) &&
            (pmcoreCallInstance == _ccaCallStatusList[ccaCallInstance]._callAttributes._instance))
      {
         break;
      }
   }

   if (_ccaCallStatusList.size() == ccaCallInstance)
   {
      ccaCallInstance = CCA_CALL_INSTANCE_DEFAULT;
   }
}

void CcaCallStatusListHandler::onAudioActiveStatusChanged(const AudioActiveStatus audioActiveStatus)
{
   ETG_TRACE_USR4(("onAudioActiveStatusChanged() entered with audioActiveStatus: %u", audioActiveStatus));

   _audioActiveStatus = audioActiveStatus;

   DeviceHandle activeDeviceHandle(_activeDeviceHandle);

   auto iter = std::find_if(_ccaCallStatusList.begin(), _ccaCallStatusList.end(),
         [&activeDeviceHandle](CcaCallStatus const& obj)
         {return obj._deviceHandle == activeDeviceHandle;});

   if (_ccaCallStatusList.end() != iter)
   {
      for (CallCount callCount = 0; callCount < MAX_DEVICE_CALL_INSTANCES_COUNT_FOR_CCA; ++callCount)
      {
         (iter + callCount)->_audioActiveStatus = _audioActiveStatus;
      }
   }

   printCcaCallStatusList();
}

pmcore::DeviceHandle CcaCallStatusListHandler::getActiveDeviceHandle()
{
   ETG_TRACE_USR4(("getActiveDeviceHandle() entered and _activeDeviceHandle: %u", _activeDeviceHandle));
   return _activeDeviceHandle;
}

pmcore::AudioActiveStatus CcaCallStatusListHandler::getAudioActiveStatus()
{
   ETG_TRACE_USR4(("getAudioActiveStatus() entered and _audioActiveStatus: %u", _audioActiveStatus));
   return _audioActiveStatus;
}
