/**
 * @file CcaCallStatusListHandler.h
 *
 * @swcomponent PhoneCallManager
 *
 * @brief This file contains the type definition of the CcaCallStatusListHandler class
 *
 * @copyright (C) 2016 Robert Bosch GmbH.
 *            The reproduction, distribution and utilization of this file as
 *            well as the communication of its contents to others without express
 *            authorization is prohibited. Offenders will be held liable for the
 *            payment of damages. All rights reserved in the event of the grant
 *            of a patent, utility model or design.
 *
 * @details
 *
 * @ingroup PmApp
 */

#ifndef CcaCallStatusListHandler_h
#define CcaCallStatusListHandler_h

#include "PmCoreIfTypes.h"
#include "PmSingleton.h"
#include "CallAttributes.h"
#include "CallsReport.h"
#include <vector>

#define MAX_DEVICE_CALL_INSTANCES_COUNT_FOR_CCA 0x02
#define CCA_CALL_INSTANCE_DEFAULT               0xFF

typedef unsigned short CCACallInstance;

class CcaCallStatus
{
public:
   CcaCallStatus() : _deviceAddress(""), _deviceHandle(PM_DEVICEHANDLE_ZERO), _telephoneNumber(""), _callAttributes(),
   _multiparty(false), _audioActiveStatus(false)
   {
   }

   CcaCallStatus(const pmcore::BdAddress deviceAddress, const pmcore::DeviceHandle deviceHandle) :
            _deviceAddress(deviceAddress), _deviceHandle(deviceHandle), _telephoneNumber(""),
            _callAttributes(com::bosch::pmapp::CallAttributes()),
            _multiparty(false), _audioActiveStatus(false)
   {
   }

   CcaCallStatus(const pmcore::BdAddress deviceAddress, const pmcore::DeviceHandle deviceHandle,
         pmcore::TelephoneNumber telephoneNumber,
         const com::bosch::pmapp::CallAttributes callAttributes, const pmcore::Multiparty multiparty,
         const pmcore::AudioActiveStatus audioActiveStatus) :
            _deviceAddress(deviceAddress), _deviceHandle(deviceHandle),
            _telephoneNumber(telephoneNumber), _callAttributes(callAttributes),
            _multiparty(multiparty), _audioActiveStatus(audioActiveStatus)
   {
   }

   CcaCallStatus(const CcaCallStatus& ref) :
      _deviceAddress(ref._deviceAddress), _deviceHandle(ref._deviceHandle),
      _telephoneNumber(ref._telephoneNumber), _callAttributes(ref._callAttributes),
      _multiparty(ref._multiparty), _audioActiveStatus(ref._audioActiveStatus)
   {
   }

   //coverity-fix:481955-Big parameter passed by value - Function parameter passed as by reference
   //and std::swap is not used
   CcaCallStatus& operator=(const CcaCallStatus& ref)
   {
      // Check for self assignment
      if(this != &ref)
      {
         _deviceAddress = ref._deviceAddress;
         _deviceHandle = ref._deviceHandle;
         _telephoneNumber = ref._telephoneNumber;
         _callAttributes = ref._callAttributes;
         _multiparty = ref._multiparty;
         _audioActiveStatus = ref._audioActiveStatus;
      }
      return *this;
   }

   bool operator==(const CcaCallStatus& rhs) const
   {
      return ((this->_deviceAddress == rhs._deviceAddress)
            && (this->_deviceHandle == rhs._deviceHandle)
            && (this->_telephoneNumber == rhs._telephoneNumber)
            && (this->_callAttributes == rhs._callAttributes)
            && (this->_multiparty == rhs._multiparty)
            && (this->_audioActiveStatus == rhs._audioActiveStatus));
   }

   bool operator!=(const CcaCallStatus& rhs) const
   {
      return (! operator==(rhs));
   }

   void resetCallInfo()
   {
      _telephoneNumber.clear();
      _callAttributes = com::bosch::pmapp::CallAttributes();
      _multiparty = false;
   }

   pmcore::BdAddress _deviceAddress;
   pmcore::DeviceHandle _deviceHandle;
   pmcore::TelephoneNumber _telephoneNumber;
   com::bosch::pmapp::CallAttributes _callAttributes;
   pmcore::Multiparty _multiparty;
   pmcore::AudioActiveStatus _audioActiveStatus;
};

typedef std::vector<CcaCallStatus> CcaCallStatusList;

class CcaCallStatusListHandler : public PmSingleton<CcaCallStatusListHandler>
{
public:
   /**
    * This function is used to maintain the connected list of devices
    *
    * @param[in]  activePassiveDeviceListMap - List of active and Passive devices
    *             activeDeviceHandle - the active device handle
    * @param[out]
    * @param[in,out]
    *
    * @return
    *
    */
   void onActivePassiveDeviceListChanged(const pmcore::ActivePassiveDeviceListMap& activePassiveDeviceListMap,
         const pmcore::DeviceHandle activeDeviceHandle);

   /**
    * This function is used to update the "_ccaCallStatusList"
    *
    * @param[in]  CallsReportList - Calls report list for each connected device
    * @param[out]  CcaCallStatus& CcaCallStatus - The ccaCallStatus is updated based on the callStatusListMap
    * @param[in,out]
    *
    * @return
    *
    */
   void onCallsReportListChanged(const com::bosch::pmapp::CallsReportList& callsReportList,
         CcaCallStatusList& ccaCallStatusList);

   /**
    * This function is used to convert CCA specific call instance to Pm Core specific call instance
    *
    * @param[in]  ccaCallInstance - CCA specific call instance
    *             deviceAddress - Device address to which the call instance belongs to.
    *             pmcoreCallInstance - Pm Core specific call instance
    * @param[out]  CcaCallStatus& CcaCallStatus - The ccaCallStatus is updated based on the callStatusListMap
    * @param[in,out]
    *
    * @return
    *
    */
   void convertCCACallInstance(const CCACallInstance ccaCallInstance, pmcore::BdAddress& deviceAddress,
         pmcore::CallInstance& pmcoreCallInstance);

   /**
    * This function is used to get CCA specific call instance from Pm Core specific call instance
    *
    * @param[in]  pmcoreCallInstance - pm core specific call instance
    *             deviceAddress - Device address to which the call instance belongs to.
    *             pmcoreCallInstance - Pm Core specific call instance
    * @param[out] ccaCallInstance - Corresponding CCA call instance
    * @param[in,out]
    *
    * @return
    *
    */
   void getCCACallInstance(const pmcore::BdAddress& deviceAddress, const pmcore::CallInstance pmcoreCallInstance,
         CCACallInstance& ccaCallInstance);

   /**
    * This function is called when the phone Call Audio status is changed
    * from PM Core.
    * The vehicle audio status corresponding to Active device is updated.
    *
    * @param[in]  audioActiveStatus - audio status
    * @param[out]
    * @param[in,out]
    *
    * @return
    *
    */
   void onAudioActiveStatusChanged(const pmcore::AudioActiveStatus audioActiveStatus);

   /**
    * This function is used to get the Active device handle
    *
    * @param[in]
    * @param[out]
    * @param[in,out]
    *
    * @return DeviceHandle - Active device handle
    *
    */
   pmcore::DeviceHandle getActiveDeviceHandle();

   /**
    * This function is used to get the Active audio status
    *
    * @param[in]
    * @param[out]
    * @param[in,out]
    *
    * @return AudioActiveStatus - audio Active status
    *
    */
   pmcore::AudioActiveStatus getAudioActiveStatus();

private:

   friend class PmSingleton<CcaCallStatusListHandler>;

   /**
    * Constructor of CcaCallStatusListHandler class
    *
    * @param[in]
    * @param[out]
    * @param[in,out]
    *
    * @return
    *
    */
   CcaCallStatusListHandler();

   /**
    * Destructor of CcaCallStatusListHandler class
    *
    * @param[in]
    * @param[out]
    * @param[in,out]
    *
    * @return
    *
    */
   ~CcaCallStatusListHandler();

   /**
    * This function is used to print the "_ccaCallStatusList"
    *
    * @param[in]
    * @param[out]
    * @param[in,out]
    *
    * @return
    *
    */
   void printCcaCallStatusList();

   // This vector stores the call status information corresponding to CCA
   static CcaCallStatusList _ccaCallStatusList;

   // This member is used to keep track of the Active Device Handle
   pmcore::DeviceHandle _activeDeviceHandle;

   // This member is used to keep track of the Audio Status
   pmcore::AudioActiveStatus _audioActiveStatus;
};

#endif // CcaCallStatusListHandler_h
