/**
 * @file CcaProperty.cpp
 *
 * @swcomponent PhoneCallManager
 *
 * @brief This file contains the definition of the CcaProperty class methods
 *
 * @copyright (C) 2016 Robert Bosch GmbH.
 *            The reproduction, distribution and utilization of this file as
 *            well as the communication of its contents to others without express
 *            authorization is prohibited. Offenders will be held liable for the
 *            payment of damages. All rights reserved in the event of the grant
 *            of a patent, utility model or design.
 *
 * @details
 *
 * @ingroup IpcWrapper
 */

#include "CcaProperty.h"
#include "PmAppTrace.h"

using namespace pmcore;

#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#define ETG_DEFAULT_TRACE_CLASS   TR_CLASS_PM_SERVICE
#ifdef VARIANT_S_FTR_ENABLE_FW_ETG_USAGE
#include "trcGenProj/Header/CcaProperty.cpp.trc.h"
#define ETG_DEFAULT_TRACE_CLASS   TR_CLASS_PM_SERVICE
#endif
#endif

CcaProperty::CcaProperty(ahl_tclBaseOneThreadService* pService)
: CcaFunction(pService)
{
   ETG_TRACE_USR4(("CcaProperty::CcaProperty entered"));
}

void CcaProperty::handleMessage(amt_tclServiceData* pInMsg)
{
   ETG_TRACE_USR4(("CcaProperty::handleMessage entered for CCA Message"));
   
   if (nullptr != pInMsg)
   {
      tU64 ccaToken = 0;
      CcaMsgHeader msgHeader;

      generateToken(pInMsg, ccaToken, msgHeader);

      switch (pInMsg->u8GetOpCode())
      {
         case AMT_C_U8_CCAMSG_OPCODE_PURESET: // Intentional fallthrough
         case AMT_C_U8_CCAMSG_OPCODE_SET:
         {
            if(ccaToken > 0)
            {
               onOpcodeSet(pInMsg, ccaToken);
            }
            else
            {
               sendErrorMessage(msgHeader, most_fi_tcl_e8_ErrorCode::FI_EN_NOTAVAILABLE);
            }
         }
         break;

         case AMT_C_U8_CCAMSG_OPCODE_GET:
         {
            if(ccaToken > 0)
            {
               onOpcodeGet(pInMsg, ccaToken);
            }
            else
            {
               sendErrorMessage(msgHeader, most_fi_tcl_e8_ErrorCode::FI_EN_NOTAVAILABLE);
            }
         }
         break;

         case AMT_C_U8_CCAMSG_OPCODE_UPREG:
         {
            if(ccaToken > 0)
            {
               if (addToNotificationTable(pInMsg))
               {
                  onOpcodeGet(pInMsg, ccaToken);
               }
               else
               {
                  sendErrorMessage(msgHeader, most_fi_tcl_e8_ErrorCode::FI_EN_NOTAVAILABLE);
               }
            }
            else
            {
               sendErrorMessage(msgHeader, most_fi_tcl_e8_ErrorCode::FI_EN_NOTAVAILABLE);
            }
         }
         break;

         case AMT_C_U8_CCAMSG_OPCODE_RELUPREG:
         {
            removeFromNotificationTable(pInMsg);
            removeEntryFromRequestMap(ccaToken);
         }
            break;

         case AMT_C_U8_CCAMSG_OPCODE_ERROR:
         {
            // TODO: Activate code again
            // _pService->vSendErrorMessage(
            // pInMsg->u16GetFunctionID(),
            // _pService->_u16InvalidOpCodeError);
         }
            break;

         default:
            break;
      }
   }
}

void CcaProperty::handleMessage(PmCoreResponseData* pCppResponse)
{
   ETG_TRACE_USR4(("CcaProperty::handleMessage entered for CPP response"));
   
   if(pCppResponse)
   {
      processMethodResponse(pCppResponse);
   }
   else
   {
      ETG_TRACE_ERR(("CcaProperty::handleMessage: cppResponse instance is NULL"));
   }
}

bool CcaProperty::addToNotificationTable(const amt_tclServiceData* pInMessage)
{
   ETG_TRACE_USR4(("CcaProperty::addToNotificationTable entered"));
   
   if (pInMessage != nullptr)
   {
      bool isAdded = _notificationTable.bAddNotification(
            pInMessage->u16GetFunctionID(),
            pInMessage->u16GetSourceAppID(),
            pInMessage->u16GetRegisterID(),
            TRUE,
            pInMessage->u16GetCmdCounter(),
            pInMessage->u16GetSourceSubID());

      return isAdded;
   }

   return false;
}

void CcaProperty::removeFromNotificationTable(const amt_tclServiceData* pInMessage)
{
   ETG_TRACE_USR4(("CcaProperty::removeFromNotificationTable entered"));
   
   if (pInMessage != nullptr)
   {
      (void)_notificationTable.bRemoveNotification(
            pInMessage->u16GetFunctionID(),
            pInMessage->u16GetSourceAppID(),
            pInMessage->u16GetRegisterID(),
            TRUE,
            pInMessage->u16GetCmdCounter(),
            pInMessage->u16GetSourceSubID());
   }
}

void CcaProperty::updateStatusToRegisteredClients(CcaMsgHeader& msgHeaderInfo, const fi_tclTypeBase& oOutData)
{
   ETG_TRACE_USR4(("CcaProperty::updateStatusToRegisteredClients entered"));

   ahl_tNotification* pNotify = _notificationTable.poGetNotificationList(msgHeaderInfo._u16Fid);

   for (; pNotify != nullptr; pNotify = pNotify->pNext)
   {
      msgHeaderInfo._u16DestAppID = pNotify->u16AppID;
      msgHeaderInfo._u16CmdCtr = pNotify->u16CmdCounter;
      msgHeaderInfo._u16RegisterId = pNotify->u16RegisterID;
      msgHeaderInfo._u8OpCode = AMT_C_U8_CCAMSG_OPCODE_STATUS;

      sendMessage(msgHeaderInfo, oOutData);
   }
}

void CcaProperty::updateOpcodeStatus(const tU64 ccaToken, const fi_tclTypeBase& oOutData, const tU16 functionId)
{
   ETG_TRACE_USR4(("CcaProperty::updateOpcodeStatus entered"));

   CcaMsgHeader msgHeaderInfo;

   bool entryFoundInReqMap = retrieveMessageHeader(ccaToken, msgHeaderInfo);

   if((AMT_C_U8_CCAMSG_OPCODE_GET == msgHeaderInfo._u8OpCode)
         || (AMT_C_U8_CCAMSG_OPCODE_UPREG == msgHeaderInfo._u8OpCode))
   {
      ETG_TRACE_USR4(("CcaProperty::update requested client"));
      msgHeaderInfo._u8OpCode = AMT_C_U8_CCAMSG_OPCODE_STATUS;
      // Respond only to the client who had requested a GET on that property
      sendMessage(msgHeaderInfo, oOutData);
   }
   else if((AMT_C_U8_CCAMSG_OPCODE_SET == msgHeaderInfo._u8OpCode)
		   || (AMT_C_U8_CCAMSG_OPCODE_PURESET == msgHeaderInfo._u8OpCode))
   {
      // Response is not needed for SET Property in CCA.
      // The flow will come here only for SET property Error where SET property's ERROR message
      // need to be updated to the client.
      sendMessage(msgHeaderInfo, oOutData);
   }
   else
   {
      // Notify all the clients who had registered for this property
      msgHeaderInfo._u16Fid = functionId;
      ETG_TRACE_USR4(("CcaProperty::update Function ID: %u to all the registered clients", msgHeaderInfo._u16Fid));
      updateStatusToRegisteredClients(msgHeaderInfo, oOutData);
   }

   // Remove the entry from the request map as response is sent above
   if(entryFoundInReqMap)
   {
      removeEntryFromRequestMap(ccaToken);
   }
}

bool CcaProperty::isOpcodeTypeGetorUpreg(const unsigned char& opcode)
{
   ETG_TRACE_USR4(("CcaProperty::isOpcodeTypeGetorUpreg entered"));

   bool isOpcodeGetorUpreg = ((AMT_C_U8_CCAMSG_OPCODE_GET == opcode)
         || (AMT_C_U8_CCAMSG_OPCODE_UPREG == opcode));

   ETG_TRACE_USR4(("isOpcodeGetorUpreg: %u", isOpcodeGetorUpreg));
   return isOpcodeGetorUpreg;
}

void CcaProperty::processMethodResponse(PmCoreResponseData* pmCoreResponseData)
{
   ETG_TRACE_USR4(("CcaProperty::processMethodResponse entered"));

   CcaMsgHeader msgHeaderInfo;

   bool entryFoundInReqMap = retrieveMessageHeader(pmCoreResponseData->_u64TokenId, msgHeaderInfo);

   if (entryFoundInReqMap)
   {
      if (isOpcodeTypeGetorUpreg(msgHeaderInfo._u8OpCode))
      {
         processOpcodeGetResponse(pmCoreResponseData);
      }
      else
      {
         ETG_TRACE_USR4(("Opcode type is SET"));

         if(PM_RESULT_OK == pmCoreResponseData->_pmResult._pmResultCode)
         {
            ETG_TRACE_USR4(("Opcode SET is successful. Change in property will be updated in the Property update"));
         }
         else
         {
            ETG_TRACE_USR4(("Opcode SET is unsuccessful."));
            processOpcodeSetError(pmCoreResponseData);
         }
         removeEntryFromRequestMap(pmCoreResponseData->_u64TokenId);
      }
   }
}

CcaProperty::~CcaProperty()
{
   ETG_TRACE_USR4(("CcaProperty::~CcaProperty entered"));
}
