/**
 * @file CcaInComingCallEventHandler.cpp
 *
 * @swcomponent PhoneCallManager
 *
 * @brief This file contains the definition of the CcaInComingCallEventHandler class methods
 *
 * @copyright (C) 2016 Robert Bosch GmbH.
 *            The reproduction, distribution and utilization of this file as
 *            well as the communication of its contents to others without express
 *            authorization is prohibited. Offenders will be held liable for the
 *            payment of damages. All rights reserved in the event of the grant
 *            of a patent, utility model or design.
 *
 * @details
 *
 * @ingroup IpcWrapper
 */

#include "CcaInComingCallEventHandler.h"
#include "PmAppTrace.h"

using namespace pmcore;

#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#define ETG_DEFAULT_TRACE_CLASS   TR_CLASS_PM_SERVICE
#ifdef VARIANT_S_FTR_ENABLE_FW_ETG_USAGE
#include "trcGenProj/Header/CcaInComingCallEventHandler.cpp.trc.h"
#define ETG_DEFAULT_TRACE_CLASS   TR_CLASS_PM_SERVICE
#endif
#endif

std::map<pmcore::BdAddress, IncomingCallAttributes> CcaInComingCallEventHandler::_incomingCallsList;

CcaInComingCallEventHandler::CcaInComingCallEventHandler(ahl_tclBaseOneThreadService* pAhlService)
: CcaProperty(pAhlService)
{
   ETG_TRACE_USR4(("CcaInComingCallEventHandler::CcaInComingCallEventHandler entered"));
}

CcaInComingCallEventHandler::~CcaInComingCallEventHandler()
{
   ETG_TRACE_USR4(("CcaInComingCallEventHandler::~CcaInComingCallEventHandler entered"));
   _incomingCallsList.clear();
}

void CcaInComingCallEventHandler::onOpcodeGet(amt_tclServiceData* pInMsg, const ActType act)
{
   ETG_TRACE_USR4(("CcaInComingCallEventHandler::onOpcodeGet entered"));
   (void) act;

   if (pInMsg)
   {
      IncomingCallAttributes incomingCallAttributes;
      BdAddress deviceAddress = "";
      ::com::bosch::pmapp::DeviceDetailsListHandler::getInstance().getActiveDeviceAddress(deviceAddress);

      auto iter = _incomingCallsList.find(deviceAddress);

      if (_incomingCallsList.end() != iter)
      {
         // Active device has an incoming call
         incomingCallAttributes = iter->second;
      }
      else
      {
         if (!_incomingCallsList.empty())
         {
            auto frontElementIter = _incomingCallsList.begin();

            deviceAddress = frontElementIter->first;
            incomingCallAttributes = frontElementIter->second;
         }
      }

      updateIncomingCallEvent(deviceAddress, incomingCallAttributes, pInMsg->u16GetFunctionID(), act);
   }
   else
   {
      removeEntryFromRequestMap(act);
   }
}

void CcaInComingCallEventHandler::onOpcodeSet(amt_tclServiceData* pInMsg, const ActType act)
{
   ETG_TRACE_ERR(("Received an invalid opcode - SET for MOST_TELFI_C_U16_INCOMINGCALLEVENT, FID: %u",
         pInMsg->u16GetFunctionID()));

   // Set is not possible, Hence removing the token from the table.
   removeEntryFromRequestMap(act);
}

void CcaInComingCallEventHandler::onPropertyUpdate(PropertyUpdate* propertyUpdate)
{
   ETG_TRACE_USR4(("CcaInComingCallEventHandler::onPropertyUpdate entered"));
   dispatchProperty(propertyUpdate);
}

void CcaInComingCallEventHandler::processOpcodeGetResponse(PmCoreResponseData* responseMsg)
{
   ETG_TRACE_USR4(("CcaInComingCallEventHandler::processOpcodeGetResponse entered"));
   dispatchProperty(responseMsg);
}

template<typename TPropertyType>
void CcaInComingCallEventHandler::dispatchProperty(TPropertyType* propertyUpdate)
{
   ETG_TRACE_USR4(("CcaInComingCallEventHandler::dispatchProperty entered"));

   CallsReportListPropertyUpdate* callsReportListPropertyUpdate =
         static_cast<CallsReportListPropertyUpdate*> (propertyUpdate);

   com::bosch::pmapp::CallsReportList callsReportList = callsReportListPropertyUpdate->_callsReportList;

   for (auto& callsReportIter : callsReportList)
   {
      BdAddress deviceAddress = callsReportIter.first;
      com::bosch::pmapp::CallsReport callsReport = callsReportIter.second;
      com::bosch::pmapp::CallAttributesList callAttributesList = callsReport._callAttributesList;

      TelephoneNumber incomingCallTelephoneNumber = "";
      com::bosch::pmapp::CallAttributes callAttributes;

      // Check whether the received "CallAttributesList" has an incoming call
      bool incomingCallPresent = getIncomingCallInfo(callAttributesList, incomingCallTelephoneNumber, callAttributes);

      // Check whether already updated
      auto incomingCallAttributesinListIter = _incomingCallsList.find(deviceAddress);

      if(incomingCallPresent)
      {
         // Incoming call present

         CCACallInstance ccaCallInstance = CCA_CALL_INSTANCE_DEFAULT;
         CcaCallStatusListHandler::getInstance().getCCACallInstance(deviceAddress,
               callAttributes._instance, ccaCallInstance);

         IncomingCallAttributes incomingCallAttributes(callAttributes, incomingCallTelephoneNumber, ccaCallInstance);

         if (_incomingCallsList.end() != incomingCallAttributesinListIter)
         {
            ETG_TRACE_USR4(("Incoming call already updated"));

            IncomingCallAttributes incomingCallAttributesinList = (incomingCallAttributesinListIter->second);

            if (incomingCallAttributesinList != incomingCallAttributes)
            {
               if (incomingCallAttributesinList._telephoneNumber == incomingCallTelephoneNumber)
               {
                  // Call attributes modified
                  // updating the modified call attributes
                  ETG_TRACE_USR4(("Incoming call already updated but the call attributes have be modified"));
                  _incomingCallsList.erase(incomingCallAttributesinListIter);
                  _incomingCallsList.emplace_hint(_incomingCallsList.end(), deviceAddress, incomingCallAttributes);
                  updateIncomingCallEvent(deviceAddress, incomingCallAttributes,
                        callsReportListPropertyUpdate->u16GetFunctionID());
               }
               else
               {
                  // New incoming call received before receiving the changed call status for the previous incoming call
                  // This might be an issue from the lower layers of PM App or Bt Stack.
                  // Hence updating idle incoming call event for the previous incoming call
                  // and then updating the active incoming call for the recent incoming call.
                  ETG_TRACE_USR4(("New incoming call received"));

                  IncomingCallAttributes idleIncomingCallAttributes(incomingCallAttributesinList);
                  // Setting the call state to Idle
                  idleIncomingCallAttributes._state = "Idle";

                  updateIncomingCallEvent(deviceAddress, idleIncomingCallAttributes,
                        callsReportListPropertyUpdate->u16GetFunctionID());

                  _incomingCallsList.erase(incomingCallAttributesinListIter);

                  _incomingCallsList.emplace_hint(_incomingCallsList.end(), deviceAddress, incomingCallAttributes);
                  updateIncomingCallEvent(deviceAddress, incomingCallAttributes,
                        callsReportListPropertyUpdate->u16GetFunctionID());
               }
            }
         }
         else
         {
            // update incoming call event

            _incomingCallsList.emplace_hint(_incomingCallsList.end(), deviceAddress, incomingCallAttributes);
            updateIncomingCallEvent(deviceAddress, incomingCallAttributes,
                  callsReportListPropertyUpdate->u16GetFunctionID());
         }
      }
      else
      {
         ETG_TRACE_USR4(("No Active incoming call present"));

         if (_incomingCallsList.end() != incomingCallAttributesinListIter)
         {
            // Need to update idle incoming call event
            IncomingCallAttributes idleIncomingCallAttributes = (incomingCallAttributesinListIter->second);
            // simply set state other than "Incoming" or "Waiting"
            idleIncomingCallAttributes._state = "Active";

            updateIncomingCallEvent(deviceAddress, idleIncomingCallAttributes,
                  callsReportListPropertyUpdate->u16GetFunctionID());

            _incomingCallsList.erase(incomingCallAttributesinListIter);
         }
      }
   }

   // Check and remove the incoming calls of the devices that are not present in the "callsReportList"
   for (auto incomingCallsListIter = _incomingCallsList.begin(); _incomingCallsList.end() != incomingCallsListIter; )
   {
      auto callsReportListIter = std::find_if(callsReportList.begin(), callsReportList.end(),
            [&incomingCallsListIter](std::pair<BdAddress, com::bosch::pmapp::CallsReport> const& obj)
            {return (obj.first == incomingCallsListIter->first);});

      if (callsReportList.end() == callsReportListIter)
      {
         ETG_TRACE_USR4(("Device itself is not present. Hence updating IDLE incoming call event"));

         IncomingCallAttributes incomingCallAttributes = incomingCallsListIter->second;
         incomingCallAttributes._state = "Idle";

         updateIncomingCallEvent(incomingCallsListIter->first, incomingCallAttributes,
               callsReportListPropertyUpdate->u16GetFunctionID());

         incomingCallsListIter = _incomingCallsList.erase(incomingCallsListIter);
      }
      else
         ++incomingCallsListIter;
   }
}

void CcaInComingCallEventHandler::processOpcodeSetError(PmCoreResponseData* pInMsg)
{
   (void) pInMsg;
}

bool CcaInComingCallEventHandler::getIncomingCallInfo(const com::bosch::pmapp::CallAttributesList& callAttributesList,
      TelephoneNumber& telephoneNumber, com::bosch::pmapp::CallAttributes& callAttributes)
{
   ETG_TRACE_USR4(("CcaInComingCallEventHandler::getIncomingCallInfo entered"));
   bool incomingCallPresent = false;
   telephoneNumber.clear();
   callAttributes = com::bosch::pmapp::CallAttributes();

   auto iter = std::find_if(callAttributesList.begin(), callAttributesList.end(),
         [](std::pair<TelephoneNumber, com::bosch::pmapp::CallAttributes> const& obj)
         {return ((obj.second._state == "Incoming") || (obj.second._state == "Waiting"));});

   if (callAttributesList.end() != iter)
   {
      telephoneNumber = iter->first;
      callAttributes = iter->second;
      incomingCallPresent = true;
   }

   return incomingCallPresent;
}

void CcaInComingCallEventHandler::updateIncomingCallEvent(const BdAddress& deviceAddress,
      const IncomingCallAttributes& incomingCallAttributes,
      const unsigned short int functionId, const tU64 ccaToken)
{
   ETG_TRACE_USR4(("CcaInComingCallEventHandler::updateIncomingCallEvent entered"));

   DeviceHandle deviceHandle = PM_DEVICEHANDLE_ZERO;
   ::com::bosch::pmapp::DeviceDetailsListHandler::getInstance().getDeviceHandle(deviceAddress, deviceHandle);

   most_telfi_tclMsgInComingCallEventStatus oIncomingCallEventStatus;
   oIncomingCallEventStatus.e8IncomingCallEventEnum.enType =
         (most_fi_tcl_e8_TelIncomingCallEventEnum::tenType::FI_EN_E8IDLE);
   oIncomingCallEventStatus.bCallWait = false;
   oIncomingCallEventStatus.u8DeviceHandle = PM_DEVICEHANDLE_ZERO;
   oIncomingCallEventStatus.u16CallInstance = CCA_CALL_INSTANCE_DEFAULT;

   if (PM_DEVICEHANDLE_ZERO != deviceHandle)
   {
      if ("Incoming" == incomingCallAttributes._state)
      {
         oIncomingCallEventStatus.e8IncomingCallEventEnum.enType =
               (most_fi_tcl_e8_TelIncomingCallEventEnum::tenType::FI_EN_E8RINGING);
      }
      else if ("Waiting" == incomingCallAttributes._state)
      {
         oIncomingCallEventStatus.e8IncomingCallEventEnum.enType =
               (most_fi_tcl_e8_TelIncomingCallEventEnum::tenType::FI_EN_E8RINGING);
         oIncomingCallEventStatus.bCallWait = true;
      }

      oIncomingCallEventStatus.u16CallInstance = incomingCallAttributes._ccaCallInstance;
      oIncomingCallEventStatus.u8DeviceHandle = deviceHandle;
      oIncomingCallEventStatus.sTelephoneNumber.bSet(incomingCallAttributes._telephoneNumber.c_str());
      oIncomingCallEventStatus.sFirstName.bSet(incomingCallAttributes._firstName.c_str());
      oIncomingCallEventStatus.sLastName.bSet(incomingCallAttributes._lastName.c_str());
   }

   ETG_TRACE_USR1((" Incoming Call Event:: %u",
         ETG_ENUM(TR_INCOMINGCALLEVENT, oIncomingCallEventStatus.e8IncomingCallEventEnum.enType)));
   ETG_TRACE_USR1(("oIncomingCallEventStatus.u16CallInstance: %d", oIncomingCallEventStatus.u16CallInstance));
   ETG_TRACE_USR1(("oIncomingCallEventStatus.u8DeviceHandle: %d", oIncomingCallEventStatus.u8DeviceHandle));
   ETG_TRACE_USR1(("oIncomingCallEventStatus.sTelephoneNumber: %s", oIncomingCallEventStatus.sTelephoneNumber.szGet()));
   ETG_TRACE_USR1(("oIncomingCallEventStatus.bCallWait: %d", oIncomingCallEventStatus.bCallWait));
   ETG_TRACE_USR1(("oIncomingCallEventStatus.sFirstName: %s", oIncomingCallEventStatus.sFirstName.szGet()));
   ETG_TRACE_USR1(("oIncomingCallEventStatus.sLastName: %s", oIncomingCallEventStatus.sLastName.szGet()));

   updateOpcodeStatus(ccaToken, oIncomingCallEventStatus, functionId);
   oIncomingCallEventStatus.vDestroy();
}
