/**
 * @file CcaDeviceRingtoneListHandler.cpp
 *
 * @swcomponent PhoneCallManager
 *
 * @brief This file contains the definition of the CcaDeviceRingtoneListHandler class methods
 *
 * @copyright (C) 2016 Robert Bosch GmbH.
 *            The reproduction, distribution and utilization of this file as
 *            well as the communication of its contents to others without express
 *            authorization is prohibited. Offenders will be held liable for the
 *            payment of damages. All rights reserved in the event of the grant
 *            of a patent, utility model or design.
 *
 * @details
 *
 * @ingroup IpcWrapper
 */

#include "CcaDeviceRingtoneListHandler.h"
#include "FileUtils.h"
#include "IpcWrapper.h"
#include "PmAppTrace.h"

using namespace pmcore;

#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#define ETG_DEFAULT_TRACE_CLASS   TR_CLASS_PM_SERVICE
#ifdef VARIANT_S_FTR_ENABLE_FW_ETG_USAGE
#include "trcGenProj/Header/CcaDeviceRingtoneListHandler.cpp.trc.h"
#define ETG_DEFAULT_TRACE_CLASS   TR_CLASS_PM_SERVICE
#endif
#endif

CcaDeviceRingtoneListHandler::CcaDeviceRingtoneListHandler(ahl_tclBaseOneThreadService* pAhlService)
: CcaProperty(pAhlService)
{
   ETG_TRACE_USR4(("CcaDeviceRingtoneListHandler::CcaDeviceRingtoneListHandler entered"));
}

void CcaDeviceRingtoneListHandler::onOpcodeGet(amt_tclServiceData* pInMsg, const ActType act)
{
   ETG_TRACE_USR4(("CcaDeviceRingtoneListHandler::onOpcodeGet entered"));
   (void) pInMsg;

   PmResult pmresult = pm_ipc_wrapper::IpcWrapper::getInstance().getDeviceRingtoneList(act);
   if(PM_RESULT_OK != pmresult._pmResultCode)
   {
      most_telfi_tclMsgDeviceRingtoneListStatus deviceRingtoneListStatus;
      updateOpcodeStatus(act, deviceRingtoneListStatus, PM_PROPERTYID_DEVICERINGTONELIST);
      deviceRingtoneListStatus.vDestroy();
   }
}

void CcaDeviceRingtoneListHandler::onOpcodeSet(amt_tclServiceData* pInMsg, const ActType act)
{
   ETG_TRACE_ERR(("Received an invalid opcode - SET for MOST_TELFI_C_U16_DEVICERINGTONELIST, FID: %u",
         pInMsg->u16GetFunctionID()));

   // Set is not possible, Hence removing the token from the table.
   removeEntryFromRequestMap(act);
}

void CcaDeviceRingtoneListHandler::onPropertyUpdate(PropertyUpdate* propertyUpdate)
{
   ETG_TRACE_USR4(("CcaDeviceRingtoneListHandler::onPropertyUpdate entered"));

   dispatchProperty(propertyUpdate);
}

void CcaDeviceRingtoneListHandler::processOpcodeGetResponse(PmCoreResponseData* responseMsg)
{
   ETG_TRACE_USR4(("CcaDeviceRingtoneListHandler::processOpcodeGetResponse entered"));

   dispatchProperty(responseMsg);
}

template<typename TPropertyType>
void CcaDeviceRingtoneListHandler::dispatchProperty(TPropertyType* propertyUpdate)
{
   ETG_TRACE_USR4(("CcaDeviceRingtoneListHandler::dispatchProperty entered"));

   DeviceRingtoneListPropertyUpdate* deviceRingtoneListPropertyUpdate =
         static_cast<DeviceRingtoneListPropertyUpdate*> (propertyUpdate);

   most_telfi_tclMsgDeviceRingtoneListStatus deviceRingtoneListStatus;
   most_fi_tcl_TelDeviceRingtoneListResultItem  deviceRingtoneListResultItem;

   DeviceRingtoneListType deviceRingtoneList =
         deviceRingtoneListPropertyUpdate->_deviceRingtoneList._deviceRingtoneList;

   for (auto& deviceRingtone : deviceRingtoneList)
   {
      deviceRingtoneListResultItem.sBdAddress.bSet(deviceRingtone._deviceAddress.c_str());
      deviceRingtoneListResultItem.u8RingtoneID = static_cast<tU8>(deviceRingtone._ringtoneId);
      std::string ringtoneName;
      (void)com::bosch::pmcommon::getFileNameFromFilePath(deviceRingtone._ringtoneName, ringtoneName);
      deviceRingtoneListResultItem.sRingtoneName.bSet(ringtoneName.c_str());

      switch(deviceRingtone._inbandRingingSupport)
      {
      case NOT_SUPPORTED:
    	  deviceRingtoneListResultItem.e8InbandRingingSupport.enType = most_fi_tcl_e8_TelInbandRingingSupport::FI_EN_E8NOT_SUPPORTED;
    	  break;
      case SUPPORTED_AND_ENABLED:
    	  deviceRingtoneListResultItem.e8InbandRingingSupport.enType = most_fi_tcl_e8_TelInbandRingingSupport::FI_EN_E8SUPPORTED_ENABLED;
    	  break;
      case SUPPORTED_AND_DISABLED:
    	  deviceRingtoneListResultItem.e8InbandRingingSupport.enType = most_fi_tcl_e8_TelInbandRingingSupport::FI_EN_E8SUPPORTED_DISABLED;
    	  break;
      default:
    	  deviceRingtoneListResultItem.e8InbandRingingSupport.enType = most_fi_tcl_e8_TelInbandRingingSupport::FI_EN_E8NOT_SUPPORTED;
    	  break;
      }

      deviceRingtoneListStatus.oDeviceRingtoneListResult.oItems.push_back(deviceRingtoneListResultItem);
   }

   updateOpcodeStatus(deviceRingtoneListPropertyUpdate->_u64TokenId, deviceRingtoneListStatus,
         deviceRingtoneListPropertyUpdate->u16GetFunctionID());

   deviceRingtoneListStatus.vDestroy();
}

void CcaDeviceRingtoneListHandler::processOpcodeSetError(PmCoreResponseData* pInMsg)
{
   (void) pInMsg;
}
