/**
 * @file CcaProperty.h
 *
 * @swcomponent PhoneCallManager
 *
 * @brief This file contains the definition of the CcaProperty class
 *
 * @copyright (C) 2016 Robert Bosch GmbH.
 *            The reproduction, distribution and utilization of this file as
 *            well as the communication of its contents to others without express
 *            authorization is prohibited. Offenders will be held liable for the
 *            payment of damages. All rights reserved in the event of the grant
 *            of a patent, utility model or design.
 *
 * @details
 *
 * @ingroup IpcWrapper
 */

#ifndef CcaProperty_h
#define CcaProperty_h

#include "CcaFunction.h"
#include <vector>

#include "ail_fw.h"
#include "fi_msgfw_fw.h"

/*
 * This class acts as a base class for all CCA property handlers
 */
class CcaProperty : public CcaFunction
{

public:

   /**
    * Performs opcode check for incoming message and invokes the message handler in the
    * derived class
    *
    * @param[in] amt_tclServiceData* pInMsg - Message in CCA format
    * @param[out]
    * @param[in,out]
    *
    * @return
    *
    */
   virtual void handleMessage(amt_tclServiceData* pInMsg) override;

   /**
    * Process Method response from CPP interfaces.
    *
    * @param[in]  PmCoreResponseData* - pointer to the response data from CPP layer
    * @param[out]
    * @param[in,out]
    *
    * @return
    *
    */
   virtual void handleMessage(PmCoreResponseData* pCppResponse) override;

   /**
    * Destructor of CcaProperty class
    *
    * @param[in]
    * @param[out]
    * @param[in,out]
    *
    * @return
    *
    */
   virtual ~CcaProperty();

   /**
    * Constructor of CcaProperty class
    *
    * @param[in] tU16 u16FunctionID:- Unique ID for every message(Method / Property) in the service
    *            ahl_tclBaseOneThreadService*:- Pointer to the CCA service base class
    * @param[out]
    * @param[in,out]
    *
    * @return
    *
    */
   CcaProperty(ahl_tclBaseOneThreadService* pService);

protected:

   /**
    * Method that would get invoked when received with an OPCODE - GET from the clients
    *
    * @param[in]  amt_tclServiceData* pInMessage - Message in CCA format
    *             pmcore::ActType act - Generated Token
    * @param[out]
    * @param[in,out]
    *
    * @return
    *
    */
   virtual void onOpcodeGet(amt_tclServiceData* pInMsg, const pmcore::ActType act) = 0;

   /**
    * Method that would get invoked when received with an OPCODE - SET from the clients
    *
    * @param[in]  amt_tclServiceData* pInMessage - Message in CCA format
    *             pmcore::ActType act - Generated Token
    * @param[out]
    * @param[in,out]
    *
    * @return
    *
    */
   virtual void onOpcodeSet(amt_tclServiceData* pInMsg, const pmcore::ActType act) = 0;

   /**
    * This method shall be overridden in the derived class property handlers,
    * where the clients shall be updated with the status.
    *
    * @param[in] PropertyUpdate* - pointer to the property update data
    * @param[out]
    * @param[in,out]
    *
    * @return
    *
    */
   virtual void onPropertyUpdate(PropertyUpdate* pPropertyUpdate) override = 0;

   /**
    * This method is used to process the response data from the core layer
    * and update the requested clients if it is Opcode Get.
    * If it is opcode SET, the error response alone is handled.
    *
    * @param[in]  PmCoreResponseData* - Pointer to the response data from CPP layer
    * @param[out]
    * @param[in,out]
    *
    * @return
    *
    */
   void processMethodResponse(PmCoreResponseData* pInMsg);

   /**
    * This method has to be overridden in the derived classes message handlers
    * to process the error response data from the core layer and update the requested
    * clients if needed
    *
    * @param[in]  PmCoreResponseData* - Pointer to the response data from CPP layer
    * @param[out]
    * @param[in,out]
    *
    * @return
    *
    */
   virtual void processOpcodeSetError(PmCoreResponseData* pInMsg) = 0;

   /**
    * Method that would get invoked when received with an OPCODE - UPREG from the clients
    *
    * @param[in] tU64 ccaToken - Token generated internally within IpcWrapper to map request / response between
    *            IPC wrapper and PmCore
    *            fi_tclTypeBase& oOutData - Message data to be sent
    *            tU16 functionId - Function ID of the property that needs to be updated to all the clients.
    *                              'functionId' will be used only if the change in property
    *                              needs to be updated to the clients.
    * @param[out]
    * @param[in,out]
    *
    * @return
    *
    */
   void updateOpcodeStatus(const tU64 ccaToken, const fi_tclTypeBase& oOutData, const tU16 functionId = 0x00);

   /**
    * Method that would get invoked in the derived class of each property
    * when received response from PM core for Get or when the status is updated.
    *
    * @param[in] PmCoreResponseData* responseMsg - cpp respone received from PM Core
    * @param[out]
    * @param[in,out]
    *
    * @return
    *
    */
   template<typename TPropertyType>
   void dispatchProperty(TPropertyType* propertyUpdate);

   /**
    * Method that would get invoked in the derived class of the property
    * when received response from PM core for status get.
    *
    * @param[in] PmCoreResponseData* responseMsg - cpp respone received from PM Core
    * @param[out]
    * @param[in,out]
    *
    * @return
    *
    */
   virtual void processOpcodeGetResponse(PmCoreResponseData* responseMsg) = 0;

   /**
    * This method is invoked to get the opcode type of the property.
    *
    * @param[in] unsigned char& opcode - opcode type received
    * @param[out] bool - true if the Opcode type is Get or UpReg, else false
    * @param[in,out]
    *
    * @return - Returns true if the Opcode type is Get, else False
    *
    */
   bool isOpcodeTypeGetorUpreg(const unsigned char& opcode);

private:

   /**
    * Method to add the clients that has registered for a PM CCA property to the notification table
    *
    * @param[in]  amt_tclServiceData* pInMessage - Message in CCA format
    * @param[out]
    * @param[in,out]
    *
    * @return
    *
    */
   bool addToNotificationTable(const amt_tclServiceData* pInMessage);

   /**
    * Method to remove the clients from the notification table when the clients has un registered for a property
    *
    * @param[in]  amt_tclServiceData* pInMessage - Message in CCA format
    * @param[out]
    * @param[in,out]
    *
    * @return
    *
    */
   void removeFromNotificationTable(const amt_tclServiceData* pInMessage);

   /**
    * Method to notify the property status to clients
    *
    * @param[in]  CcaMsgHeader& msgHeaderInfo - contains CCA message header information
    *             fi_tclTypeBase& oOutData - Message data to be sent
    * @param[out]
    * @param[in,out]
    *
    * @return
    *
    */
   void updateStatusToRegisteredClients(CcaMsgHeader& msgHeaderInfo, const fi_tclTypeBase& oOutData);

   ahl_tclNotificationTable _notificationTable; /**< The notification table that would have the entries of clients registered for a property*/

};

#endif // CcaProperty_h
