/**
 * @file CcaInComingCallEventHandler.h
 *
 * @swcomponent PhoneCallManager
 *
 * @brief This file contains the definition of the CcaInComingCallEventHandler class methods
 *
 * @copyright (C) 2016 Robert Bosch GmbH.
 *            The reproduction, distribution and utilization of this file as
 *            well as the communication of its contents to others without express
 *            authorization is prohibited. Offenders will be held liable for the
 *            payment of damages. All rights reserved in the event of the grant
 *            of a patent, utility model or design.
 *
 * @details
 *
 * @ingroup IpcWrapper
 */

#ifndef CcaInComingCallEventHandler_h
#define CcaInComingCallEventHandler_h

#include "CcaProperty.h"

class IncomingCallAttributes : public com::bosch::pmapp::CallAttributes
{
public:
   /**
    * Constructor of IncomingCallAttributes class
    *
    * @param[in]
    * @param[out]
    * @param[in,out]
    *
    * @return
    *
    */
   IncomingCallAttributes() : com::bosch::pmapp::CallAttributes(), _telephoneNumber(""),
   _ccaCallInstance(CCA_CALL_INSTANCE_DEFAULT)
   {

   }

   /**
    * Parameterized Constructor of IncomingCallAttributes class
    *
    * @param[in]
    * @param[out]
    * @param[in,out]
    *
    * @return
    *
    */
   IncomingCallAttributes(const com::bosch::pmapp::CallAttributes callAttributes,
         pmcore::TelephoneNumber telephoneNumber, CCACallInstance ccaCallInstance) :
            com::bosch::pmapp::CallAttributes(callAttributes), _telephoneNumber(telephoneNumber),
            _ccaCallInstance(ccaCallInstance)
   {

   }

   /**
    * Copy Constructor of IncomingCallAttributes class
    *
    * @param[in]
    * @param[out]
    * @param[in,out]
    *
    * @return
    *
    */
   IncomingCallAttributes(const IncomingCallAttributes& other) :
      com::bosch::pmapp::CallAttributes(other), _telephoneNumber(other._telephoneNumber),
      _ccaCallInstance(other._ccaCallInstance)
   {

   }

   /**
    * Destructor of IncomingCallAttributes class
    *
    * @param[in]
    * @param[out]
    * @param[in,out]
    *
    * @return
    *
    */
   ~IncomingCallAttributes()
   {

   }

   /**
    * Friend function used when copying the object.
    *
    * @param[in]
    * @param[out]
    * @param[in,out]
    *
    * @return
    *
    */
   friend void swap(IncomingCallAttributes& first, IncomingCallAttributes& second)
   {
      using std::swap;
      swap(first._telephoneNumber, second._telephoneNumber);
      swap(first._ccaCallInstance, second._ccaCallInstance);
   }

   /**
    * Overloading '=' operator
    *
    * @param[in]
    * @param[out]
    * @param[in,out]
    *
    * @return
    *
    */
   //lint -e{1720} suppress "assignment operator has non-const parameter", see copy-and-swap idiom
   IncomingCallAttributes& operator=(IncomingCallAttributes other)
   {
      com::bosch::pmapp::CallAttributes::operator=(other);
      swap(*this, other);
      return *this;
   }

   /**
    * Overloading '==' operator
    *
    * @param[in]
    * @param[out]
    * @param[in,out]
    *
    * @return
    *
    */
   bool operator==(const IncomingCallAttributes& rhs) const
   {
      return ((com::bosch::pmapp::CallAttributes::operator==(rhs))
            && (this->_telephoneNumber == rhs._telephoneNumber)
            && (this->_ccaCallInstance == rhs._ccaCallInstance));
   }

   /**
    * Overloading '!=' operator
    *
    * @param[in]
    * @param[out]
    * @param[in,out]
    *
    * @return
    *
    */
   bool operator!=(const IncomingCallAttributes& rhs) const
   {
      return (!(operator==(rhs)));
   }

   pmcore::TelephoneNumber _telephoneNumber;
   CCACallInstance _ccaCallInstance;
};

/*
 * Handler class for CCA property - MOST_TELFI_C_U16_INCOMINGCALLEVENT
 */
class CcaInComingCallEventHandler final: public CcaProperty
{

public:

   /**
    * Constructor of CcaInComingCallEventHandler class
    *
    * @param[in]
    * @param[out]
    * @param[in,out]
    *
    * @return
    *
    */
   CcaInComingCallEventHandler(ahl_tclBaseOneThreadService* pAhlService);

   /**
    * Destructor of CcaInComingCallEventHandler class
    *
    * @param[in]
    * @param[out]
    * @param[in,out]
    *
    * @return
    *
    */
   virtual ~CcaInComingCallEventHandler();

protected:

   /**
    * Actual processing of the CCA message takes place here and forward
    * request to IpcWrapper. From IpcWrapper request is posted to PM CORE CPP.
    * This method is invoked when client does a GET on PmApp CCA property
    *
    * @param[in] amt_tclServiceData* - Pointer to the CCA message data
    * @param[out]
    * @param[in,out]
    *
    * @return
    *
    */
   virtual void onOpcodeGet(amt_tclServiceData* pInMsg, const pmcore::ActType act) override;

   /**
    * Actual processing of the CCA message takes place here and forward
    * request to IpcWrapper. From IpcWrapper request is posted to PM CORE CPP
    * This method is invoked when client does a SET on PmApp CCA property
    *
    * @param[in] amt_tclServiceData* - Pointer to the CCA message data
    * @param[out]
    * @param[in,out]
    *
    * @return
    *
    */
   virtual void onOpcodeSet(amt_tclServiceData* pInMsg, const pmcore::ActType act) override;

   /**
    * PM shall update clients on the property updates received from CPP interfaces.
    *
    * @param[in] PropertyUpdate* - pointer to the property update data
    * @param[out]
    * @param[in,out]
    *
    * @return
    *
    */
   virtual void onPropertyUpdate(PropertyUpdate* pPropertyUpdate) override;

   /**
    * Overridden from the CcaProperty class
    *
    * @param[in]  PmCoreResponseData* - Pointer to the response data from CPP layer
    * @param[out]
    * @param[in,out]
    *
    * @return
    *
    */
   virtual void processOpcodeSetError(PmCoreResponseData* pInMsg) override;

   /**
    * Overridden from CcaProperty class
    *
    * @param[in]
    * @param[out]
    * @param[in,out]
    *
    * @return
    *
    */
   template<typename TPropertyType>
   void dispatchProperty(TPropertyType* propertyUpdate);

   /**
    * Overridden from CcaProperty class
    *
    * @param[in] PmCoreResponseData* responseMsg - cpp respone received from PM Core
    * @param[out]
    * @param[in,out]
    *
    * @return
    *
    */
   virtual void processOpcodeGetResponse(PmCoreResponseData* responseMsg) override;

private:

   /**
    * This function is used to get the incoming call from the given CallsReport of a device, if present.
    * And update "_incomingCallsList"
    *
    * @param[in] CallAttributesList& callAttributesList - received callAttributesList of a device
    * @param[out] pmcore::TelephoneNumber& telephoneNumber - telephoneNumber of the incoming call
    *             CallAttributesList& incomingCallAttributes - incoming call details
    * @param[in,out]
    *
    * @return     bool - true if incoming call is present
    */
   bool getIncomingCallInfo(const com::bosch::pmapp::CallAttributesList& callAttributesList,
         pmcore::TelephoneNumber& telephoneNumber,
         com::bosch::pmapp::CallAttributes& callAttributes);

   /**
    * This function is used to trigger the incoming call event to the clients
    *
    * @param[in]  BdAddress& deviceAddress - Device Address for which the incoming call is received
    *             IncomingCallAttributes& incomingCallAttributes - incomingCallAttributes
    *             const unsigned short int functionId - Function id
    *             const tU64 ccaToken - token generated
    * @param[out]
    * @param[in,out]
    *
    * @return
    */
   void updateIncomingCallEvent(const pmcore::BdAddress& deviceAddress,
         const IncomingCallAttributes& incomingCallAttributes,
         const unsigned short int functionId,
         const tU64 ccaToken = pmcore::PM_DEFAULT_ACT);

   // The incoming call number and the corresponding device address are recorded in this map
   static std::map<pmcore::BdAddress, IncomingCallAttributes> _incomingCallsList;
};

#endif // CcaInComingCallEventHandler_h
