/**
 * @file BmAppRequestIf.cpp
 *
 * @swcomponent PhoneCallManager
 *
 * @brief This file contains the definition of the BmAppRequestIf class
 *
 * @copyright (C) 2016 Robert Bosch GmbH.
 *            The reproduction, distribution and utilization of this file as
 *            well as the communication of its contents to others without express
 *            authorization is prohibited. Offenders will be held liable for the
 *            payment of damages. All rights reserved in the event of the grant
 *            of a patent, utility model or design.
 *
 * @details This file holds the class definition for BmAppRequestIf class
 *
 * @ingroup IpcWrapper
 */

#include "BmAppRequestIf.h"
#include "DeviceDetailsListHandler.h"

namespace com
{
namespace bosch
{
namespace pmapp
{

DEFINE_CLASS_LOGGER_AND_LEVEL("phone_call_mgr/clienthandlers", BmAppRequestIf, Info);

BmAppRequestIf::BmAppRequestIf()
:IBmAppRequestIf(false), _bmAppCallBackIf(nullptr), _hfpProfileActiveDevHandles()
{
   LOG_INFO("BmAppRequestIf::BmAppRequestIf() entered");

   createProxy("mostBtSetFiPort");
}

BmAppRequestIf::~BmAppRequestIf()
{
   destroyProxy();
   _hfpProfileActiveDevHandles.clear();
}

void BmAppRequestIf::createProxy(const std::string portName)
{
   LOG_INFO("BmAppRequestIf::createProxy entered");

   if (!(portName.empty()))
   {
      _btSetFIProxy = MOST_BTSet_FIProxy::createProxy(portName, *this);
   }
}

void BmAppRequestIf::destroyProxy()
{
   if (_btSetFIProxy.get())
   {
      _btSetFIProxy.reset();
   }
}

void BmAppRequestIf::setBmAppCallBackIf(IBmAppCallBackIf* bmAppCallBackIf)
{
   LOG_INFO("BmAppRequestIf::setIBmAppCallBackIf entered");
   if (bmAppCallBackIf)
   {
      _bmAppCallBackIf =  bmAppCallBackIf;
      _bmAppCallBackIf->setBtSetFIProxy(_btSetFIProxy);
   }
}

void BmAppRequestIf::onAvailable(const boost::shared_ptr<asf::core::Proxy>& proxy,
      const asf::core::ServiceStateChange& /*stateChange*/)
{
   LOG_INFO("BmAppRequestIf::onAvailable() entered");

   if ((_btSetFIProxy.get()) && (proxy == _btSetFIProxy))
   {
      registerProperties();
      _serviceAvailability = true;
   }
   else
   {
      LOG_ERROR("Undesirable proxy");
   }
}

void BmAppRequestIf::onUnavailable(const boost::shared_ptr<asf::core::Proxy>& proxy,
      const asf::core::ServiceStateChange& /*stateChange*/)
{
   LOG_INFO("BmAppRequestIf::onUnavailable() entered");

   if ((_btSetFIProxy.get()) && (proxy == _btSetFIProxy))
   {
      deregisterProperties();
      _serviceAvailability = false;
   }
   else
   {
      LOG_ERROR("Undesirable proxy");
   }
}

void BmAppRequestIf::registerProperties()
{
   LOG_INFO("BmAppRequestIf::registerProperties() entered");

   if (_bmAppCallBackIf)
   {
      // First Vehicle address should be registered. Only then DeviceListExt should be registered.
      // Because on device connected, the corresponding proxies should be registered which needs vehicle BT address.
      // Hence after receiving the Vehicle BT address, PM registers for DeviceListExt status.
      _btSetFIProxy->sendVehicleConfigurationUpReg(*_bmAppCallBackIf);
   }
}

void BmAppRequestIf::registerDeviceListExtendedProperty()
{
   LOG_INFO("BmAppRequestIf::registerDeviceListExtendedProperty() entered");

   if (_bmAppCallBackIf)
   {
      // First Vehicle address should be registered. Only then DeviceListExt should be registered.
      // Because on device connected, the corresponding proxies should be registered which needs vehicle BT address.
      // Hence after receiving the Vehicle BT address, PM registers for DeviceListExt status.
      _btSetFIProxy->sendDeviceListExtendedUpReg(*_bmAppCallBackIf);
   }
}

void BmAppRequestIf::deregisterProperties()
{
   LOG_INFO("BmAppRequestIf::deregisterProperties() entered");

   _btSetFIProxy->sendDeviceListExtendedRelUpRegAll();
   _btSetFIProxy->sendVehicleConfigurationRelUpRegAll();
}

void BmAppRequestIf::setPrimaryDevice(const pmcore::BdAddressList& activeDevicesList)
{
   LOG_INFO("setPrimaryDevice()");

   for (auto iter : activeDevicesList)
   {
      LOG_INFO("setPrimaryDevice():ActivedeviceAddress: %s", iter.c_str());
   }

   // As of now only one Active device is supported.
   pmcore::DeviceHandle deviceHandle = PM_DEVICEHANDLE_ZERO;

   if (!activeDevicesList.empty())
   {
      DeviceDetailsListHandler::getInstance().getDeviceHandle(activeDevicesList.front(), deviceHandle);
   }

   if (_serviceAvailability && (_btSetFIProxy.get()) && (_bmAppCallBackIf))
   {
      act_t act = _btSetFIProxy->sendSetPrimaryStart(*_bmAppCallBackIf, deviceHandle);

      if (DEFAULT_ACT == act)
      {
         LOG_INFO("sendSetPrimaryStart posting FAILED");
      }
   }
}

void BmAppRequestIf::setBTProfilesUsage(const uint8 deviceHandle, const bool profileUsageInfo,
      const BTProfileUsageType profileUsageTypeIn)
{
   LOG_INFO("setBTProfileUsage() deviceHandle : %u", deviceHandle);
   LOG_INFO("profileUsageInfo: %u and profileUsageTypeIn: %u", profileUsageInfo, profileUsageTypeIn);

   (void) profileUsageInfo;
   (void) profileUsageTypeIn;

   if (_serviceAvailability && (_btSetFIProxy.get()) && (_bmAppCallBackIf) && (deviceHandle != PM_DEVICEHANDLE_ZERO))
   {
      switch (profileUsageTypeIn)
      {
         case PROFILE_USAGE_GENERIC:
         {
            LOG_INFO("PROFILE_USAGE_GENERIC");
         }
         break;

         case PROFILE_USAGE_ACTIVE_CALL:
         {
            LOG_INFO("PROFILE_USAGE_ACTIVE_CALL");

            auto iter = std::find_if(_hfpProfileActiveDevHandles.begin(), _hfpProfileActiveDevHandles.end(),
                  [&deviceHandle](uint8 const& obj){return obj == deviceHandle;});

            bool updateStatus = false;

            if (profileUsageInfo)
            {
               if (_hfpProfileActiveDevHandles.end() == iter)
               {
                  _hfpProfileActiveDevHandles.emplace_back(deviceHandle);
                  updateStatus = true;
               }
            }
            else
            {
               if (_hfpProfileActiveDevHandles.end() != iter)
               {
                  _hfpProfileActiveDevHandles.erase(iter);
                  updateStatus = true;
               }
            }

            if (updateStatus)
            {
               // Set the BT Profile usage info(BTProfile, usage type and in-use)
               T_BTProfilesUsageInfo oBTProfUsageInfo;

               oBTProfUsageInfo.setE8BTProfile(T_e8_BTProfile::T_e8_BTProfile__e8BT_PROFILE_HFP);
               oBTProfUsageInfo.setE8BTProfileUsageType(T_e8_BTProfileUsageType__e8PROFILE_USAGE_ACTIVE_CALL);
               oBTProfUsageInfo.setBInUse(profileUsageInfo);

               sendBTProfilesUsage(deviceHandle, oBTProfUsageInfo);
            }
         }
         break;

         default:
            LOG_INFO("Bt ProfileUsageType: default");
      }
   }
}

void BmAppRequestIf::setBTProfilesUsage(const std::string deviceAddress, const bool profileUsageInfo,
      const BTProfileUsageType profileUsageType)
{
   LOG_INFO("setBTProfileUsage() deviceAddress : %s", deviceAddress.c_str());

   pmcore::DeviceHandle deviceHandle = PM_DEVICEHANDLE_ZERO;
   DeviceDetailsListHandler::getInstance().getDeviceHandle(deviceAddress, deviceHandle);

   setBTProfilesUsage(deviceHandle, profileUsageInfo, profileUsageType);
}

void BmAppRequestIf::sendBTProfilesUsage(const uint8 deviceHandle, const T_BTProfilesUsageInfo oBTProfUsageInfo)
{
   LOG_INFO("sendBTProfilesUsage() deviceHandle : %u", deviceHandle);
   LOG_INFO("sendBTProfilesUsage() oBTProfUsageInfo.getBInUse() : %u", oBTProfUsageInfo.getBInUse());

   T_BTProfilesUsageList oBTProfilesUsageList;
   T_BTProfilesUsageInfoList oBTProfUsageInfoList;

   // Set the device handle
   oBTProfUsageInfoList.setU8DeviceHandle(deviceHandle);
   // Set the BT Profile usage info
   oBTProfUsageInfoList.setOBTProfilesUsageInfo(oBTProfUsageInfo);

   oBTProfilesUsageList.push_back(oBTProfUsageInfoList);
   act_t act = _btSetFIProxy->sendSetBTProfilesUsageStart(*_bmAppCallBackIf, oBTProfilesUsageList);

   if (DEFAULT_ACT == act)
   {
      LOG_INFO("sendSetBTProfilesUsageStart posting FAILED");
   }
}

} // namespace pmapp
} // namespace bosch
} // namespace com
