/**
 * @file BmAppCallBackIf.cpp
 *
 * @swcomponent PhoneCallManager
 *
 * @brief This file contains the definition of the BmAppCallBackIf class
 *
 * @copyright (C) 2016 Robert Bosch GmbH.
 *            The reproduction, distribution and utilization of this file as
 *            well as the communication of its contents to others without express
 *            authorization is prohibited. Offenders will be held liable for the
 *            payment of damages. All rights reserved in the event of the grant
 *            of a patent, utility model or design.
 *
 * @details This file holds the class definition for BmAppCallBackIf class
 *
 * @ingroup IpcWrapper
 */

#include "BmAppCallBackIf.h"
#include "DeviceDetailsListHandler.h"
#include "IpcWrapper.h"
#include "FwBluetoothStringUtils.h"
#include "PhoneCallManager.h"

using namespace pm_ipc_wrapper;

namespace com
{
namespace bosch
{
namespace pmapp
{

DEFINE_CLASS_LOGGER_AND_LEVEL("phone_call_mgr/clienthandlers", BmAppCallBackIf, Info);

BmAppCallBackIf::BmAppCallBackIf()
{
   LOG_INFO("BmAppCallBackIf::BmAppCallBackIf() entered");
}

BmAppCallBackIf::~BmAppCallBackIf()
{
}

void BmAppCallBackIf::setBtSetFIProxy(MostBTSetFiProxy btSetFIProxy)
{
   LOG_INFO("BmAppCallBackIf::setBtSetFIProxy() entered");
   _btSetFIProxy = btSetFIProxy;
}

void BmAppCallBackIf::onDeviceListExtendedError(const MostBTSetFiProxy& proxy,
      const ::boost::shared_ptr< DeviceListExtendedError >& /*error*/)
{
   LOG_INFO("onDeviceListExtendedError() entered");

   if ((_btSetFIProxy.get()) && (proxy == _btSetFIProxy))
   {
      // As of now, Error handling is not done for property Get callbacks.
      LOG_ERROR("Error received for Property Get");
   }
   else
   {
      LOG_ERROR("Undesirable proxy received");
   }
}

void BmAppCallBackIf::onDeviceListExtendedStatus(const MostBTSetFiProxy& proxy,
      const ::boost::shared_ptr< DeviceListExtendedStatus >& status)
{
   LOG_INFO("onDeviceListExtendedStatus() entered");

   if ((_btSetFIProxy.get()) && (proxy == _btSetFIProxy))
   {
      const T_BTSetDeviceListChangeExtended& odeviceListChange = status->getODeviceListChange();
      ::most_BTSet_fi_types::T_e8_BTSetDeviceStatus deviceStatus = odeviceListChange.getE8DeviceStatus();

      if (::most_BTSet_fi_types::T_e8_BTSetDeviceStatus__e8DEVICE_DELETED == deviceStatus)
      {
         LOG_INFO("Device DELETED");

         pmcore::DeviceHandle deletedDeviceHandle = odeviceListChange.getU8DeviceHandle();

         pmcore::BdAddress deletedDeviceAddress = "";
          bool isDeviceRemoved = com::bosch::pmapp::DeviceDetailsListHandler::getInstance().removeDevice
                (deletedDeviceHandle, deletedDeviceAddress);

          if (isDeviceRemoved)
             IpcWrapper::getInstance().onDeviceDeletion(deletedDeviceAddress);
      }
      else
      {
         const T_BTSetDeviceListExtendedResult& odeviceListResult = status->getODeviceListExtendedResult();
         uint8 deviceListSize = status->getU8NumPairedDevices();

         for (uint8 pairedDevicesIndex = 0; pairedDevicesIndex != deviceListSize; ++pairedDevicesIndex)
         {
            const T_BTSetDeviceListExtendedResultItem& resultDevice = odeviceListResult[pairedDevicesIndex];

            DeviceDetails deviceDetails;

            T_e8_BTConnectionStatus hfpConnectionStatus = resultDevice.getODeviceProfileConnectionStatus().getE8HFP();

            bool processDeviceConnectionStateChange = true;

            DeviceDetails deviceDetailsFromList;
            com::bosch::pmapp::DeviceDetailsListHandler::getInstance().
                  getDeviceDetails(resultDevice.getSDeviceAddress(), deviceDetailsFromList);

            pmcore::DeviceIdentification deviceIdentificationExisting(deviceDetailsFromList._deviceIdentification);

            switch (hfpConnectionStatus)
            {
               case T_e8_BTConnectionStatus__e8STATUS_NOT_CONNECT:
               {
                  LOG_INFO(" HFP not connected ");
                  deviceDetails._deviceConnectionStatus = DEVICE_NOT_HFP_CONNECTED;

                  if (DEVICE_HFP_DISCONNECTED == deviceDetailsFromList._deviceConnectionStatus)
                  {
                     processDeviceConnectionStateChange = false;
                     com::bosch::pmapp::DeviceDetailsListHandler::getInstance().updateDeviceConnectionStatus(
                           deviceDetails._deviceAddress, DEVICE_NOT_HFP_CONNECTED);
                  }
               }
               break;

               case T_e8_BTConnectionStatus__e8STATUS_CONNECTED:
               {
                  LOG_INFO(" HFP connected ");
                  deviceDetails._deviceConnectionStatus = DEVICE_HFP_CONNECTED;
               }
               break;

               case T_e8_BTConnectionStatus__e8STATUS_DISCONNECTED:
               {
                  LOG_INFO(" HFP Disconnected ");
                  deviceDetails._deviceConnectionStatus = DEVICE_HFP_DISCONNECTED;

                  if (DEVICE_NOT_HFP_CONNECTED == deviceDetailsFromList._deviceConnectionStatus)
                  {
                     processDeviceConnectionStateChange = false;
                     com::bosch::pmapp::DeviceDetailsListHandler::getInstance().updateDeviceConnectionStatus(
                           deviceDetails._deviceAddress, DEVICE_HFP_DISCONNECTED);
                  }
               }
               break;

               case T_e8_BTConnectionStatus__e8STATUS_CONNECTING:
               {
                  LOG_INFO(" HFP Connecting ");
                  // Do nothing. Wait for next status.

                  processDeviceConnectionStateChange = false;
               }
               break;

               case T_e8_BTConnectionStatus__e8STATUS_DISCONNECTING:
               {
                  LOG_INFO(" HFP Disconnecting ");
                  // Do nothing. Wait for next status.

                  processDeviceConnectionStateChange = false;
               }
               break;

               default:
               {
                  LOG_INFO("Connection status: Default");
                  processDeviceConnectionStateChange = false;
               }
            }

            pmcore::DeviceIdentification deviceIdentificationReceived;
            T_BTSetDeviceIdentification deviceIdInfo = resultDevice.getODeviceInfo();
            deviceIdentificationReceived._vendorId = deviceIdInfo.getU16VendorID();
            deviceIdentificationReceived._vendorIdSource = deviceIdInfo.getU16VendorIDSource();

            bool updateDeviceIdentificationChange = (!(deviceIdentificationExisting == deviceIdentificationReceived)) ?
                  true : false;

            if (processDeviceConnectionStateChange)
            {
               deviceDetails._deviceAddress = resultDevice.getSDeviceAddress();

               if (DEVICE_HFP_DISCONNECTED == deviceDetails._deviceConnectionStatus)
               {
                  if (! (com::bosch::pmapp::DeviceDetailsListHandler::getInstance().updateDeviceConnectionStatus
                        (deviceDetails._deviceAddress, deviceDetails._deviceConnectionStatus)))
                  {
                     processDeviceConnectionStateChange = false;
                  }
               }
               else
               {
                  deviceDetails._deviceHandle = resultDevice.getU8DeviceHandle();
                  deviceDetails._deviceName = resultDevice.getSDeviceName();

                  // The below parameters of DeviceDetails may be received even after receiving the device pairing status.
                  // Hence these parameters shall be updated always.
                  deviceDetails._deviceIdentification = deviceIdentificationReceived;

                  com::bosch::pmapp::DeviceDetailsListHandler::getInstance().updateDeviceDetails(deviceDetails);
               }

               if ((deviceDetailsFromList._deviceConnectionStatus != deviceDetails._deviceConnectionStatus)
                     && processDeviceConnectionStateChange)
               {
                  IpcWrapper::getInstance().onDeviceConnectionStatusChanged(deviceDetails);

                  if (DEVICE_HFP_DISCONNECTED != deviceDetails._deviceConnectionStatus)
                  {
                     // Since the change is updated via Changed connection status(NOT_CONNECTED, CONNECTED) update
                     updateDeviceIdentificationChange = false;
                  }
               }
            }

            if (updateDeviceIdentificationChange)
            {
               com::bosch::pmapp::DeviceDetailsListHandler::getInstance().updateDeviceIdentification
                     (deviceDetails._deviceAddress, deviceIdentificationReceived);

               pmcore::BasicDeviceDetails basicDeviceDetails(resultDevice.getSDeviceAddress(),
                     resultDevice.getU8DeviceHandle(), deviceIdentificationReceived);

               IpcWrapper::getInstance().onDeviceIdentificationChanged(basicDeviceDetails);
            }

         }
      }
   }
   else
   {
      LOG_ERROR("Undesirable proxy received");
   }
}

void BmAppCallBackIf::onVehicleConfigurationError(const MostBTSetFiProxy& proxy,
      const ::boost::shared_ptr< VehicleConfigurationError >& /*error*/)
{
   LOG_INFO("onVehicleConfigurationError() entered");

   if ((_btSetFIProxy.get()) && (proxy == _btSetFIProxy))
   {
      // As of now, Error handling is not done for property Get callbacks.
      LOG_ERROR("Error received for Property Get");
   }
   else
   {
      LOG_ERROR("Undesirable proxy received");
   }
}

void BmAppCallBackIf::onVehicleConfigurationStatus(const MostBTSetFiProxy& proxy,
      const ::boost::shared_ptr< VehicleConfigurationStatus >& status)
{
   LOG_INFO("onVehicleConfigurationStatus() entered");

   if ((_btSetFIProxy.get()) && (proxy == _btSetFIProxy))
   {
      if (false == fw::isValidBdAddress(status->getSVehicleBTAddress(), false, false))
      {
         LOG_INFO("Received vehicle address is invalid");
      }
      else
      {
         pmcore::VehicleConfiguration vehicleConfiguration;
         vehicleConfiguration._vehicleBTAddress = status->getSVehicleBTAddress();
         T_BTSetDeviceIdentification deviceIdentification = status->getOVehicleIdentification();

         vehicleConfiguration._vendorIDSource = deviceIdentification.getU16VendorIDSource();
         vehicleConfiguration._vendorID = deviceIdentification.getU16VendorID();
         vehicleConfiguration._productID = deviceIdentification.getU16ProductID();
         vehicleConfiguration._productVersion = deviceIdentification.getU16ProductVersion();

         IpcWrapper::getInstance().setVehicleConfiguration(vehicleConfiguration);

         // Only after getting the vehicle address, it is possible to create dynamic object paths
         // for the HFP connecting devices.
         PmAppClientHandler* pmAppClientHandler = PhoneCallManager::getInstance().getPmAppClientHandler();
         if (pmAppClientHandler)
         {
            pmAppClientHandler->getBmAppRequestIf()->registerDeviceListExtendedProperty();
         }
      }
   }
   else
   {
      LOG_ERROR("Undesirable proxy received");
   }
}

void BmAppCallBackIf::onSetPrimaryError(const MostBTSetFiProxy& proxy,
      const ::boost::shared_ptr< SetPrimaryError >& error)
{
   LOG_INFO("onSetPrimaryError() entered");

   // TODO: Manipulation and Logical handling of errors need to be done in the upcoming releases.
   // Once those changes are done, logs printing the Error codes shall be removed
   // since they are already printed from the generated code of ASF.

   if ((_btSetFIProxy.get()) && (proxy == _btSetFIProxy))
   {
      if (error->hasCcaErrorCode())
      {
         SetPrimaryErrorStructure setPrimaryErrorStruct = error->getCcaErrorCode();

         ::fi_basetypes_most::T_e8_ErrorCode errorCode = setPrimaryErrorStruct.getE8ErrorCode();
         LOG_ERROR("CCA errorCode: %u", errorCode);
      }
      else if (error->hasSystemErrorCode())
      {
         ::asf::cca::CcaTypes::SystemErrorCode systemErrorCode = error->getSystemErrorCode();
         LOG_ERROR("systemErrorCode: %u", systemErrorCode);
      }
   }
   else
   {
      LOG_ERROR("Undesirable proxy received");
   }
}

void BmAppCallBackIf::onSetPrimaryResult(const MostBTSetFiProxy& proxy,
      const ::boost::shared_ptr< SetPrimaryResult >& result)
{
   LOG_INFO("onSetPrimaryResult() entered");

   if ((_btSetFIProxy.get()) && (proxy == _btSetFIProxy))
   {
      if (result->getBResult())
      {
         LOG_INFO("SetPrimaryDevice request Successfully processed");
      }
      else
      {
         LOG_INFO("SetPrimaryDevice Error");
         // TODO: Need to confirm whether retry is needed.
      }
   }
   else
   {
      LOG_ERROR("Undesirable proxy received");
   }
}

void BmAppCallBackIf::onSetBTProfilesUsageError(const MostBTSetFiProxy& proxy,
      const ::boost::shared_ptr< SetBTProfilesUsageError >& error)
{
   LOG_INFO("onSetBTProfilesUsageError() entered");

   // TODO: Manipulation and Logical handling of errors need to be done in the upcoming releases.
   // Once those changes are done, logs printing the Error codes shall be removed
   // since they are already printed from the generated code of ASF.

   if ((_btSetFIProxy.get()) && (proxy == _btSetFIProxy))
   {
      if (error->hasCcaErrorCode())
      {
         SetBTProfilesUsageErrorStructure setBTProfilesUsageErrorStruct = error->getCcaErrorCode();

         ::fi_basetypes_most::T_e8_ErrorCode errorCode = setBTProfilesUsageErrorStruct.getE8ErrorCode();
         LOG_ERROR("CCA errorCode: %u", errorCode);
      }
      else if (error->hasSystemErrorCode())
      {
         ::asf::cca::CcaTypes::SystemErrorCode systemErrorCode = error->getSystemErrorCode();
         LOG_ERROR("systemErrorCode: %u", systemErrorCode);
      }
   }
   else
   {
      LOG_ERROR("Undesirable proxy received");
   }
}

void BmAppCallBackIf::onSetBTProfilesUsageResult(const MostBTSetFiProxy& proxy,
      const ::boost::shared_ptr< SetBTProfilesUsageResult >& result)
{
   LOG_INFO("onSetBTProfilesUsageResult() entered");

   if ((_btSetFIProxy.get()) && (proxy == _btSetFIProxy))
   {
      if (result->getBResult())
      {
         LOG_INFO("SetBTProfilesUsage request Successfully processed");
      }
      else
      {
         LOG_INFO("SetBTProfilesUsage Error");
         // TODO: Need to confirm whether retry is needed.
      }
   }
   else
   {
      LOG_ERROR("Undesirable proxy received");
   }
}

} // namespace pmapp
} // namespace bosch
} // namespace com
