/**
 * @file PmVoiceRecHandler.cpp
 *
 * @swcomponent PhoneCallManager
 *
 * @brief This file contains the definition of the PmVoiceRecHandler class
 *
 * @copyright (C) 2016 Robert Bosch GmbH.
 *            The reproduction, distribution and utilization of this file as
 *            well as the communication of its contents to others without express
 *            authorization is prohibited. Offenders will be held liable for the
 *            payment of damages. All rights reserved in the event of the grant
 *            of a patent, utility model or design.
 *
 * @details This file provides the state machine interfaces of voice recognition
 *
 * @ingroup PmCore
 */

#include "PmVoiceRecHandler.h"
#include "PropertyUpdateNotifierToCore.h"
#include "PropertyDetails.h"
#include "PmAudioManagerWrapper.h"
#include "PmCoreMainController.h"
#include "DeviceInfoHandler.h"
#include "PmConfiguration.h"
#include "PmAppTrace.h"

#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#define ETG_DEFAULT_TRACE_CLASS TR_CLASS_PM_CORE
#ifdef VARIANT_S_FTR_ENABLE_FW_ETG_USAGE
#include "trcGenProj/Header/PmVoiceRecHandler.cpp.trc.h"
#define ETG_DEFAULT_TRACE_CLASS TR_CLASS_PM_CORE
#endif
#endif

namespace pmcore
{
   bool PmVoiceRecHandler::_temp = false;
   VRStatusEnumType PmVoiceRecHandler::_vrState = VR_SESSION_IDLE;

   PmVoiceRecHandler::PmVoiceRecHandler()
   {
      _deviceAddress = "";
      _smUserActionIf = nullptr;
      _smAMRequestIf = nullptr;
      _smSmHandlerIf = nullptr;
      _smTimerIf = nullptr;
      _smTraceObserverIf = nullptr;
      _extendVRSessionTimer = nullptr;
      _deActVRinProgress = false;
   }

   PmVoiceRecHandler::PmVoiceRecHandler(const BdAddress& deviceAddress)
   {
      _deviceAddress = deviceAddress;

      _smUserActionIf = new PmVoiceRecHandler::SmUserActionIf(*this);

      if(nullptr != _smUserActionIf)
      {
         setSCI_UserAction_OCB(_smUserActionIf);
      }

      _smAMRequestIf = new PmVoiceRecHandler::SmAMRequestIf(*this);

      if(nullptr != _smAMRequestIf)
      {
         setSCI_AMRequest_OCB(_smAMRequestIf);
      }

      _smSmHandlerIf = new PmVoiceRecHandler::SmSmHandlerIf(*this);

      if(nullptr != _smSmHandlerIf)
      {
         setSCI_SmHandler_OCB(_smSmHandlerIf);
      }

      _smTimerIf = new PmVoiceRecHandler::SmTimerIf(*this);

      if(nullptr != _smTimerIf)
      {
         setSCI_Timer_OCB(_smTimerIf);
      }

      _smTraceObserverIf = new PmVoiceRecHandler::SmTraceObserverIf<PmVoiceRecHandlerSmStates>(*this);

      if(nullptr != _smTraceObserverIf)
      {
         setTraceObserver(_smTraceObserverIf);
      }

      _deActVRinProgress = false;

      //initialize the state machine
      init();

      enter();

      _extendVRSessionTimer = new com::bosch::pmcommon::AsfTimer<PmVoiceRecHandler, std::string>();

      if(nullptr != _extendVRSessionTimer)
      {
         com::bosch::pmcommon::TimerData<std::string, ms_t, RepeatCount> timerData("ExtendVRSessionTimer",
               (ms_t)com::bosch::pmcommon::PmConfiguration::getInstance().getHoldTimeAfterEndingVRSession(),
               (RepeatCount)1);

         _extendVRSessionTimer->configure(this, timerData);
      }
   }

   PmVoiceRecHandler::~PmVoiceRecHandler()
   {
      if(nullptr != _smUserActionIf)
      {
         delete _smUserActionIf;
         _smUserActionIf = nullptr;
      }

      if(nullptr != _smAMRequestIf)
      {
         delete _smAMRequestIf;
         _smAMRequestIf = nullptr;
      }

      if(nullptr != _smSmHandlerIf)
      {
         delete _smSmHandlerIf;
         _smSmHandlerIf = nullptr;
      }

      if(nullptr != _smTimerIf)
      {
         delete _smTimerIf;
         _smTimerIf = nullptr;
      }

      if(nullptr != _smTraceObserverIf)
      {
         delete _smTraceObserverIf;
         _smTraceObserverIf = nullptr;
      }

      if(nullptr != _extendVRSessionTimer)
      {
         delete _extendVRSessionTimer;
         _extendVRSessionTimer = nullptr;
      }
   }

   PmResult PmVoiceRecHandler::startStopVoiceRecognitionRequest(const StartStop startStopVR)
   {
      PmResult pmResult(PM_RESULT_OK, "");

      if(startStopVR)
      {
         getSCI_UserAction()->raise_sTART_VR();
      }
      else
      {
         getSCI_UserAction()->raise_sTOP_VR();
      }

      return pmResult;
   }

   PmResult PmVoiceRecHandler::startStopExtVoiceRecognitionRequest(const StartStop startStopExtVR)
   {
      PmResult pmResult(PM_RESULT_OK, "");

      if(startStopExtVR)
      {
         getSCI_UserAction()->raise_sTART_EXT_VR();
      }
      else
      {
         getSCI_UserAction()->raise_sTOP_EXT_VR();
      }

      return pmResult;
   }

   void PmVoiceRecHandler::onAgVoiceRecStateUpdate(IN const VRStatus vrStatus)
   {
      ETG_TRACE_USR4(("PmVoiceRecHandler::onAgVoiceRecStateUpdate : %u", ETG_CENUM(VRStatusEnumType, vrStatus)));

      switch(vrStatus)
      {
         case VR_SESSION_IDLE:
         {
            getSCI_AgVRUpdate()->raise_vR_DEACTIVATED();
            _deActVRinProgress = false;
         }
         break;
         case VR_SESSION_ACTIVE:
         {
            getSCI_AgVRUpdate()->raise_vR_ACTIVATED();
         }
         break;
         default:
         {
            ETG_TRACE_ERR(("onAgVoiceRecStateUpdate Invalid Event"));
         }
         break;
      }
   }

   void PmVoiceRecHandler::onAudioManagerEventUpdate(IN const AudioManagerEventType amEvent)
   {
      ETG_TRACE_USR4(("PmVoiceRecHandler::onAudioManagerEventUpdate : %u", ETG_CENUM(AudioManagerEventType, amEvent)));

      switch(amEvent)
      {
         case VOICEREC_CHANNEL_GRANTED:
            getSCI_AMResponse()->raise_vOICEREC_CHANNEL_GRANTED();
            break;
         case VOICEREC_CHANNEL_DENIED:
            getSCI_AMResponse()->raise_vOICEREC_CHANNEL_DENIED(nullptr);
            break;
         case PLAY_AUDIO_SUCCESS:
            getSCI_AMResponse()->raise_pLAY_AUDIO_SUCCESS();
            break;
         case PLAY_AUDIO_FAILURE:
            getSCI_AMResponse()->raise_pLAY_AUDIO_FAILURE(nullptr);
            break;
         case PAUSE_AUDIO_SUCCESS:
            getSCI_AMResponse()->raise_pAUSE_AUDIO_SUCCESS();
            break;
         case PAUSE_AUDIO_FAILURE:
            getSCI_AMResponse()->raise_pAUSE_AUDIO_FAILURE(nullptr);
            break;
         case STOP_AUDIO_SUCCESS:
            getSCI_AMResponse()->raise_sTOP_AUDIO_SUCCESS();
            break;
         case STOP_AUDIO_FAILURE:
            getSCI_AMResponse()->raise_sTOP_AUDIO_FAILURE(nullptr);
            break;
         case VOICEREC_CHANNEL_RELEASED:
            getSCI_AMResponse()->raise_vOICEREC_CHANNEL_RELEASED_EXT_TRIGGER();
            break;
         default:
            ETG_TRACE_ERR(("onAudioManagerEventUpdate Invalid Event"));
            break;
      }
   }

   void PmVoiceRecHandler::onScoConnectionStatusChanged(IN const SCOStatus scoStatus)
   {
      ETG_TRACE_USR4(("PmVoiceRecHandler::onScoConnectionStatusChanged : %u", ETG_CENUM(SCOStatus, scoStatus)));

      switch(scoStatus)
      {
         case SCO_NOT_ESTABLISHED:
            getSCI_AgAudioUpdate()->raise_sCO_DISCONNECTED(SCO_NOT_ESTABLISHED);
            break;
         case SCO_NARROWBAND:
            getSCI_AgAudioUpdate()->raise_sCO_CONNECTED(SCO_NARROWBAND);
            break;
         case SCO_WIDEBAND:
            getSCI_AgAudioUpdate()->raise_sCO_CONNECTED(SCO_WIDEBAND);
            break;
         case SCO_NARROWBAND_SIRI:
            getSCI_AgAudioUpdate()->raise_sCO_CONNECTED(SCO_NARROWBAND_SIRI);
            break;
         case SCO_WIDEBAND_SIRI:
            getSCI_AgAudioUpdate()->raise_sCO_CONNECTED(SCO_WIDEBAND_SIRI);
            break;
         default:
            //error, something wrong
            ETG_TRACE_ERR(("PmVoiceRecHandler::onScoConnectionStatusChanged : Invalid ScoStatus"));
            getSCI_AgAudioUpdate()->raise_sCO_DISCONNECTED(SCO_NOT_ESTABLISHED);
            break;
      }
   }

   void PmVoiceRecHandler::onCallAdded()
   {
      ETG_TRACE_USR4(("PmVoiceRecHandler::onCallAdded() entered"));
      getSCI_AgVRUpdate()->raise_hFP_CALL_ADDED();
   }

   void PmVoiceRecHandler::switchToPassiveRequest(const pmaudiomanager::AudioChannel audioChannelToAcquire)
   {
      ETG_TRACE_USR4(("PmVoiceRecHandler::switchToPassiveRequest() entered"));
      getSCI_UserAction()->raise_sWITCH_TO_PASSIVE(audioChannelToAcquire);
   }

   void PmVoiceRecHandler::stopExtendedVR()
   {
      ETG_TRACE_USR4(("PmVoiceRecHandler::stopExtendedVR() entered"));
      getSCI_MediaPlayer()->raise_mEDIAREADYTOPLAY();
   }

   void PmVoiceRecHandler::startStopVoiceRecognitionError(const StartStop vrValue, PmResult& pmResult)
   {
      ETG_TRACE_USR4(("PmVoiceRecHandler::startStopVoiceRecognitionError() entered"));
      if(vrValue)
      {
         getSCI_AgVRUpdate()->raise_vR_ACTIVATION_FAILURE(&pmResult);
      }
      else
      {
         getSCI_AgVRUpdate()->raise_vR_DEACTIVATION_FAILURE(&pmResult);
      }
   }

   void PmVoiceRecHandler::switchToPassiveClientRequest()
   {
      ETG_TRACE_USR4(("PmVoiceRecHandler::switchToPassiveClientRequest() entered"));
      getSCI_UserAction()->raise_sWITCH_TO_PASSIVE_CLIENT_REQ();
   }

   void PmVoiceRecHandler::onDeviceDisconnection()
   {
      ETG_TRACE_USR4(("PmVoiceRecHandler::onDeviceDisconnection() entered"));
      getSCI_AgConnectionStatusUpdate()->raise_dEVICE_DISCONNECTED();
   }

   void PmVoiceRecHandler::SmUserActionIf::activateVR()
   {
      ETG_TRACE_USR4(("PmVoiceRecHandler::activateVR() entered"));

      evobtstackwrapper::EvoBtStackWrapper::getInstance().sendSetPropertyRequest(::ccdbusif::evolution::IF_HANDSFREE,
            _pmVoiceRecHandler.getDeviceAddress(), "VoiceRecognition", true);
   }

   void PmVoiceRecHandler::SmUserActionIf::deactivateVR()
   {
      ETG_TRACE_USR4(("PmVoiceRecHandler::deactivateVR() entered"));

      if (!_pmVoiceRecHandler._deActVRinProgress)
      {
         BTSResult btsResult = evobtstackwrapper::EvoBtStackWrapper::getInstance().sendSetPropertyRequest(
               ::ccdbusif::evolution::IF_HANDSFREE, _pmVoiceRecHandler.getDeviceAddress(), "VoiceRecognition", false);

         if (BTS_REQ_SUCCESS == btsResult._btsRequestResult)
         {
            _pmVoiceRecHandler._deActVRinProgress = true;
         }
      }
   }

   void PmVoiceRecHandler::SmUserActionIf::tempFunc(sc_boolean success)
   {
      ETG_TRACE_USR4(("PmVoiceRecHandler::tempFunc() entered"));
      _temp = success;
   }

   void PmVoiceRecHandler::SmAMRequestIf::requestAudioChannel()
   {
      ETG_TRACE_USR4(("PmVoiceRecHandler::requestAudioChannel() entered"));

      //Request voice recognition channel
      PmAudioManagerWrapper::getInstance().sendPrepareAudioRoute(_pmVoiceRecHandler.getDeviceAddress(),
            pmaudiomanager::AM_VOICERECOGNITION, VR_CONTROLLER);
   }

   void PmVoiceRecHandler::SmAMRequestIf::playHfAudio(SCOStatus scoStatus)
   {
      ETG_TRACE_USR4(("PmVoiceRecHandler::playHfAudio() entered"));

      //Requesting audio manager to play audio
      PmAudioManagerWrapper::getInstance().sendPlayAudio(_pmVoiceRecHandler.getDeviceAddress(), scoStatus);
   }

   void PmVoiceRecHandler::SmAMRequestIf::stopOrPauseHfAudio(AudioChannel audioChannelId)
   {
      ETG_TRACE_USR4(("stopOrPauseHfAudio() entered with audioChannelId: %u",
            ETG_CENUM(AudioChannel, audioChannelId)));

      switch (audioChannelId)
      {
         case AM_VOICERECOGNITION:
         {
            PmAudioManagerWrapper::getInstance().sendPauseAudio(_pmVoiceRecHandler.getDeviceAddress());
         }
         break;
         default:
         {
            PmAudioManagerWrapper::getInstance().sendStopAudio(_pmVoiceRecHandler.getDeviceAddress(), VR_CONTROLLER);
         }
      }
   }

   void PmVoiceRecHandler::SmAMRequestIf::stopAudio()
   {
      ETG_TRACE_USR4(("PmVoiceRecHandler::stopAudio() entered"));

      //Requesting audio manager to pause audio
      PmAudioManagerWrapper::getInstance().sendStopAudio(_pmVoiceRecHandler.getDeviceAddress(), VR_CONTROLLER);
   }

   void PmVoiceRecHandler::SmSmHandlerIf::updateVRState(VRStatusEnumType vrState)
   {
      ETG_TRACE_USR4(("PmVoiceRecHandler::updateVRState : %d", vrState));

      if (_vrState != vrState)
      {
         _vrState = vrState;
         PmCoreMainController::getInstance().getVRController().onVoiceRecognitionStatusChanged(
               _pmVoiceRecHandler.getDeviceAddress(), VoiceRecognitionStatus(vrState));
      }
   }

   void PmVoiceRecHandler::SmSmHandlerIf::updateVRStartError(void * pmResult)
   {
      ETG_TRACE_USR4(("PmVoiceRecHandler::updateVRStartError() entered"));

      // TODO: Need to update Error
      (void)(pmResult);
   }

   void PmVoiceRecHandler::SmSmHandlerIf::updateVRStopError(void * pmResult)
   {
      ETG_TRACE_USR4(("PmVoiceRecHandler::updateVRStopError() entered"));

      // TODO: Need to update Error
      (void)(pmResult);
      _pmVoiceRecHandler._deActVRinProgress = false;
   }

   void PmVoiceRecHandler::SmSmHandlerIf::updateVRInterruptedReason(VRInterruptedReason vrInterruptedReason)
   {
      ETG_TRACE_USR4(("PmVoiceRecHandler::updateVRInterruptedReason() entered"));

      // TODO: Need to update interrupted reason here
      (void)(vrInterruptedReason);
   }

   void PmVoiceRecHandler::SmSmHandlerIf::updateExternalVRState(ExternalVRStatusEnumType extVRState)
   {
      ETG_TRACE_USR4(("PmVoiceRecHandler::updateExternalVRState() entered"));
      (void)(extVRState);
   }

   void PmVoiceRecHandler::SmSmHandlerIf::updateExternalVRStartError(void * pmResult)
   {
      ETG_TRACE_USR4(("PmVoiceRecHandler::updateExternalVRStartError() entered"));
      (void)(pmResult);
   }

   void PmVoiceRecHandler::SmSmHandlerIf::updateExtVRInterruptedReason(VRInterruptedReason extVRInterruptedReason)
   {
      ETG_TRACE_USR4(("PmVoiceRecHandler::updateExtVRInterruptedReason() entered"));
      (void)(extVRInterruptedReason);
   }

   void PmVoiceRecHandler::SmSmHandlerIf::updateSwitchToPassiveError(void * pmResult)
   {
      ETG_TRACE_USR4(("PmVoiceRecHandler::updateSwitchToPassiveError() entered"));
      PmResult* pmResultFinal = static_cast<PmResult*> (pmResult);

      DeviceInfoHandler& deviceInfoHandler = PmCoreMainController::getInstance().getDeviceInfoHandler();
      deviceInfoHandler.processSwitchToPassiveResponse(_pmVoiceRecHandler.getDeviceAddress(), *pmResultFinal);
      _pmVoiceRecHandler._deActVRinProgress = false;
   }

   void PmVoiceRecHandler::SmSmHandlerIf::updateSwitchToPassiveNotHandledError()
   {
      ETG_TRACE_USR4(("PmVoiceRecHandler::updateSwitchToPassiveNotHandledError() entered"));

      // TODO: Once the issue(the event is handled in a state as well as in super state)
      // is fixed from ITEMIS team, the below lines should be uncommented and the temp fix must be removed.

      //      DeviceInfoHandler& deviceInfoHandler = PmCoreMainController::getInstance().getDeviceInfoHandler();
      //      deviceInfoHandler.processSwitchToPassiveResponse(_pmVoiceRecHandler.getDeviceAddress(),
      //            PmResult(PM_RESULT_ERR_GENERAL, ""));

      // Temp fix
      if (!_temp)
      {
         DeviceInfoHandler& deviceInfoHandler = PmCoreMainController::getInstance().getDeviceInfoHandler();
               deviceInfoHandler.processSwitchToPassiveResponse(_pmVoiceRecHandler.getDeviceAddress(),
                     PmResult(PM_RESULT_ERR_GENERAL, ""));
      }
   }

   void PmVoiceRecHandler::SmSmHandlerIf::updateswitchToPassiveResponse(sc_string msg)
   {
      ETG_TRACE_USR4(("PmVoiceRecHandler::updateswitchToPassiveResponse() entered with msg: %100s", msg));

      PmResult pmResult;
      bool switchToPassiveResponse = true;
      DeviceInfoHandler& deviceInfoHandler = PmCoreMainController::getInstance().getDeviceInfoHandler();
      std::string msgString(msg);

      if ("OK" == msgString)
      {
         switchToPassiveResponse = deviceInfoHandler.processSwitchToPassiveResponse(
               _pmVoiceRecHandler.getDeviceAddress(), pmResult);
      }
      else if ("Channel released due to Ext trigger" == msgString)
      {
         pmResult._pmResultCode = PM_RESULT_ERR_CHANNEL_REL_EXT_TRIGGER;
         switchToPassiveResponse = deviceInfoHandler.processSwitchToPassiveResponse(
               _pmVoiceRecHandler.getDeviceAddress(), pmResult);
      }
      else
      {
         pmResult._pmResultCode = PM_RESULT_ERR_GENERAL;
         switchToPassiveResponse = deviceInfoHandler.processSwitchToPassiveResponse(
               _pmVoiceRecHandler.getDeviceAddress(), pmResult);
      }

      if (!switchToPassiveResponse)
         PmAudioManagerWrapper::getInstance().sendStopAudio(_pmVoiceRecHandler.getDeviceAddress(), VR_CONTROLLER);
   }

   void PmVoiceRecHandler::SmSmHandlerIf::processVRStopped()
   {
      ETG_TRACE_USR4(("PmVoiceRecHandler::processVRStopped() entered"));

      DeviceInfoHandler& deviceInfoHandler = PmCoreMainController::getInstance().getDeviceInfoHandler();

      if (! deviceInfoHandler.isSCOEstablished(_pmVoiceRecHandler.getDeviceAddress()))
      {
         deviceInfoHandler.processVRRemoved(_pmVoiceRecHandler.getDeviceAddress());
      }
   }

   void PmVoiceRecHandler::SmSmHandlerIf::postGetCallsRequest2Btstack()
   {
      ETG_TRACE_USR4(("PmVoiceRecHandler::postGetCallsRequest2Btstack()::%20s",
            _pmVoiceRecHandler.getDeviceAddress().c_str()));

      PmCoreMainController::getInstance().getCallController().postGetCallsRequest2BtStack(
            _pmVoiceRecHandler.getDeviceAddress());
   }

   void PmVoiceRecHandler::SmTimerIf::stopExtendVRTimer()
   {
      ETG_TRACE_USR4(("PmVoiceRecHandler::stopExtendVRTimer() entered"));

      if((nullptr != _pmVoiceRecHandler._extendVRSessionTimer) && (true == _pmVoiceRecHandler._extendVRSessionTimer->isActive()))
      {
         ETG_TRACE_USR4(("PmVoiceRecHandler:: Stopping the timer"));

         _pmVoiceRecHandler._extendVRSessionTimer->stop();
      }
   }

   void PmVoiceRecHandler::SmTimerIf::startExtendVRTimer()
   {
      ETG_TRACE_USR4(("PmVoiceRecHandler::startExtendVRTimer() entered"));

      if(nullptr != _pmVoiceRecHandler._extendVRSessionTimer)
      {
         _pmVoiceRecHandler._extendVRSessionTimer->start();
      }
   }

   void PmVoiceRecHandler::timerElapsed(com::bosch::pmcommon::TimerData<std::string, ms_t, RepeatCount> data,
         com::bosch::pmcommon::AsfTimerCallbackData timerCallbackData)
   {
      ETG_TRACE_USR4(("PmVoiceRecHandler::timerElapsed() entered"));
      (void)(data);
      (void)(timerCallbackData);

      getSCI_Timer()->raise_eXTENDED_VRSESSION_TIMER_CALLBACK();
   }

   template<typename T>
   void PmVoiceRecHandler::SmTraceObserverIf<T>::stateEntered(T state)
   {
      ETG_TRACE_USR4(("PmVoiceRecHandler::stateEntered()::%u for deviceAddress: %20s",
            ETG_CENUM(PmVoiceRecHandlerSmStates, state), _pmVoiceRecHandler.getDeviceAddress().c_str()));
   }

   template<typename T>
   void PmVoiceRecHandler::SmTraceObserverIf<T>::stateExited(T state)
   {
      ETG_TRACE_USR4(("PmVoiceRecHandler::stateExited()::%u for deviceAddress: %20s",
            ETG_CENUM(PmVoiceRecHandlerSmStates, state), _pmVoiceRecHandler.getDeviceAddress().c_str()));
   }
}
