/**
 * @file YakinduSmVoiceRecIf.h
 *
 * @swcomponent PhoneCallManager
 *
 * @brief This file contains the definition of the YakinduSmVoiceRecIf class
 *
 * @copyright (C) 2016 Robert Bosch GmbH.
 *            The reproduction, distribution and utilization of this file as
 *            well as the communication of its contents to others without express
 *            authorization is prohibited. Offenders will be held liable for the
 *            payment of damages. All rights reserved in the event of the grant
 *            of a patent, utility model or design.
 *
 * @details This file provides the state machine interfaces of voice recognition
 *
 * @ingroup PmCore
 */

#ifndef YakinduSmVoiceRecIf_h
#define YakinduSmVoiceRecIf_h

#include "ISmVoiceRecIf.h"

namespace pmcore
{
   //class forward declaration
   class PmVoiceRecHandler;

   class YakinduSmVoiceRecIf final : public ISmVoiceRecIf
   {
   public:
      /**
       * Constructor of YakinduSmVoiceRecIf class
       *
       * @param[in]
       * @param[out]
       * @param[in,out]
       *
       * @return
       */
      YakinduSmVoiceRecIf();

      /**
       * Destructor of YakinduSmVoiceRecIf class
       *
       * @param[in]
       * @param[out]
       * @param[in,out]
       *
       * @return
       */
      ~YakinduSmVoiceRecIf();

      /**
       * This method is used to notify the device connection
       *
       * @param[in] deviceAddress - device address
       * @param[out]
       * @param[in,out]
       *
       * @return void
       */
      void onDeviceConnected(IN const BdAddress& deviceAddress);

      /**
       * This method is used to notify the device disconnection
       *
       * @param[in] deviceAddress - device address
       * @param[out]
       * @param[in,out]
       *
       * @return void
       */
      void onDeviceDisconnected(IN const BdAddress& deviceAddress);

      /**
       * Member function overridden from ISmVoiceRecIf class
       *
       * @param[in]
       * @param[out]
       * @param[in,out]
       *
       * @return PmResult - PmCore result message
       */
      PmResult startStopVoiceRecognitionRequest(IN const BdAddress& bdAddress, IN const StartStop startStopVR) override;

      /**
       * Member function overridden from ISmVoiceRecIf class
       *
       * @param[in]
       * @param[out]
       * @param[in,out]
       *
       * @return PmResult - PmCore result message
       */
      PmResult startStopExtVoiceRecognitionRequest(IN const BdAddress& bdAddress,
            IN const StartStop startStopExtVR) override;

      // **************************************************************************************************
      // Result calls
      /**
       * Member function overridden from ISmVoiceRecIf class
       *
       * @param[in]
       * @param[out]
       * @param[in,out]
       *
       * @return void
       */
      void onAgVoiceRecStateUpdate(IN const BdAddress& deviceAddress, IN const VRStatus vrStatus) override;

      /**
       * Member function overridden from ISmVoiceRecIf class
       *
       * @param[in]
       * @param[out]
       * @param[in,out]
       *
       * @return void
       */
      void onAudioManagerEventUpdate(IN const BdAddress& deviceAddress,
            IN const AudioManagerEventType amEvent) override;

      /**
       * Member function overridden from ISmVoiceRecIf class
       *
       * @param[in]
       * @param[out]
       * @param[in,out]
       *
       * @return void
       */
      void onScoConnectionStatusChanged(IN const BdAddress& deviceAddress, IN const SCOStatus scoStatus) override;

      /**
       * This method is used to notify VR SM about the received new call.
       *
       * @param[in] deviceAddress - device address
       * @param[in]
       * @param[out]
       * @param[in,out]
       *
       * @return void
       */
      void onCallAdded(IN const BdAddress& deviceAddress) override;

      /**
       * This method is used to switch a device to Passive
       *
       * @param[in] deviceAddress - device address
       * @param[in] audioChannelToAcquire - Audio channel to be acquired after stopping VR
       * @param[out]
       * @param[in,out]
       *
       * @return void
       */
      void switchToPassiveRequest(IN const BdAddress& deviceAddress,
            IN const pmaudiomanager::AudioChannel audioChannelToAcquire) override;

      /**
       * Overridden from "ISmVoiceRecIf"
       *
       * @param[in] deviceAddress - device address
       * @param[in]
       * @param[out]
       * @param[in,out]
       *
       * @return void
       */
      void switchToPassiveClientRequest(IN const BdAddress& deviceAddress) override;

      /**
       * Overridden from "ISmVoiceRecIf"
       *
       * @param[in] deviceAddress - device address
       * @param[in]
       * @param[out]
       * @param[in,out]
       *
       * @return void
       */
      void stopExtendedVR(IN const BdAddress& deviceAddress) override;

      /**
       * Overridden from "ISmVoiceRecIf"
       *
       * @param[in] deviceAddress - device address
       * @param[in] PmResult      - device address
       * @param[in] deviceAddress - device address
       * @param[out]
       * @param[in,out]
       *
       * @return void
       */
      void startStopVoiceRecognitionError(IN const BdAddress& deviceAddress, IN const StartStop vrValue,
            IN PmResult& pmResult) override;

   private:
      ::std::map<BdAddress, PmVoiceRecHandler*> _deviceToVoiceRecHandlerSmMap; //map between device address and pm voice rec handler
   };
}

#endif //YakinduSmVoiceCallIf_h
