/**
 * @file VRController.h
 *
 * @swcomponent PhoneCallManager
 *
 * @brief This file contains the definition of the VRController class
 *
 * @copyright (C) 2016 Robert Bosch GmbH.
 *            The reproduction, distribution and utilization of this file as
 *            well as the communication of its contents to others without express
 *            authorization is prohibited. Offenders will be held liable for the
 *            payment of damages. All rights reserved in the event of the grant
 *            of a patent, utility model or design.
 *
 * @details This file provides the VRController interfaces of PmCore
 *
 * @ingroup PmCore
 */

#ifndef VRController_h
#define VRController_h

#include "PropertyUpdateHandler.h"
#include "PmAudioManagerIfTypes.h"

namespace pmcore
{
   class ISmVoiceRecIf;

   // Response messages from PM Audio Manager
   class PmCoreIfMessage_PrepareAudioRouteResponse;
   class PmCoreIfMessage_PlayAudioResponse;
   class PmCoreIfMessage_StopAudioResponse;

   class VRController : public PropertyUpdateHandler
   {
   public:
      /**
       * Constructor of VRController class
       *
       * @param[in]
       * @param[out]
       * @param[in,out]
       *
       * @return
       */
      VRController();

      /**
       * Destructor of VRController class
       *
       * @param[in]
       * @param[out]
       * @param[in,out]
       *
       * @return
       */
      ~VRController();

      /**
       * Copy constructor of VRController class
       *
       * @param[in] ref - Reference to VRController
       * @param[out]
       * @param[in,out]
       *
       * @return
       */
      VRController(const VRController& ref);

      /**
       * operator= overloaded function
       *
       * @param[in]
       * @param[out]
       * @param[in,out]
       *
       * @return Returns the object of the VRController
       */
      VRController& operator=(const VRController& ref);

      /**
       * This method is used to subscribe interested properties in BT stack event notifier.
       *
       * @param[in]
       * @param[out]
       * @param[in,out]
       *
       * @return
       */
      void subscribeToBtStackEventNotifier();

      /**
       * This method is used to get property changes update from BT Stack
       * This method is overridden from PropertyUpdateHandler class
       *
       * @param[in] propertyId      - Id of the property or event
       * @param[in] propertyDetails - Details of the property or event
       * @param[out]
       * @param[in,out]
       *
       * @return
       */
      virtual void onPropertyUpdate(IN const PmCorePropertyAndEventId propertyId,
            IN std::shared_ptr<void> propertyDetails) override;

      // **************************************************************************************************
      // Request calls

      /**
       * This method is used to post the Get VoiceRecognition Status Request to the VR controller state machine
       *
       * @param[in] bdAddress  - BT address of the device
       * @param[in] act        - acknowledgment token
       * @param[out]
       * @param[in,out]
       *
       * @return void
       */
      void getVoiceRecognitionStatusRequest(IN const BdAddress& bdAddress, IN const ActType& act);

      /**
       * This method is used to post the Get External VoiceRecognition Status Request to the VR controller state machine
       *
       * @param[in] bdAddress  - BT address of the device
       * @param[in] act        - acknowledgment token
       * @param[out]
       * @param[in,out]
       *
       * @return void
       */
      void getExtVoiceRecognitionStatusRequest(IN const BdAddress& bdAddress, IN const ActType& act);

      /**
       * This method is used to notify the current VR status to the clients from VR Sm
       *
       * @param[in] bdAddress  - BT address of the device
       * @param[in] vrStatus   - Current VR Status of the device
       * @param[out]
       * @param[in,out]
       *
       * @return PmResult
       */
      PmResult onVoiceRecognitionStatusChanged(const BdAddress& bdAddress,
               const VoiceRecognitionStatus& vrStatus);

      /**
       * This method is used to notify the current External VR status to the clients from VR Sm
       *
       * @param[in] bdAddress    - BT address of the device
       * @param[in] extVRStatus  - Current External VR Status of the device
       * @param[out]
       * @param[in,out]
       *
       * @return PmResult
       */
      PmResult onExtVoiceRecognitionStatusChanged(const BdAddress& bdAddress,
               const ExtVoiceRecognitionStatus& extVRStatus);

      /**
       * This method is used to post the StartStop VoiceRecognition Request to the VR controller state machine
       *
       * @param[in] bdAddress    - BT address of the device
       * @param[in] startStopVR  - Flag for start stop VR
       * @param[out]
       * @param[in,out]
       *
       * @return PmResult - PmCore result message
       */
      PmResult startStopVoiceRecognitionRequest(IN const BdAddress& bdAddress, IN const StartStop startStopVR);

      /**
       * This method is used to post the StartStop External VoiceRecognition Request to the VR controller state machine
       *
       * @param[in] bdAddress    - BT address of the device
       * @param[in] startStopExtVR  - Flag for start stop VR
       * @param[out]
       * @param[in,out]
       *
       * @return PmResult - PmCore result message
       */
      PmResult startStopExtVoiceRecognitionRequest(IN const BdAddress& bdAddress, IN const StartStop startStopExtVR);

      /**
       * This method is used to post the Get Enhanced VoiceRecognition Feature Request to the VR controller
       *  state machine
       *
       * @param[in] bdAddress  - BT address of the device
       * @param[in] act        - acknowledgment token
       * @param[out]
       * @param[in,out]
       *
       * @return void
       */
      void getEnhancedVoiceRecognitionFeatureRequest(IN const BdAddress& bdAddress, IN const ActType& act);

      // **************************************************************************************************
      // Result calls

      /**
       * This method is used to post the VR Enable/Disable Result to the VR controller state machine
       *
       * @param[in] bdAddress  - BT address of the device
       * @param[in] vrValue - VR value sent in SetProperty request
       * @param[in] btsResult - BT stack result for the VR Enable/Disable Request
       * @param[out]
       * @param[in,out]
       *
       * @return PmResult - PmCore result message
       */
      PmResult startStopVoiceRecognitionResult(IN const BdAddress& bdAddress, IN const StartStop vrValue,
            IN const BTSResult& btsResult);

      /**
       * This method is used to post the Siri Enable Result to the VR controller state machine
       *
       * @param[in] bdAddress  - BT address of the device
       * @param[out] btsResult - BT stack result for the Siri Enable Request
       * @param[in,out]
       *
       * @return PmResult - PmCore result message
       */
      PmResult siriEnableResult(IN const BdAddress& bdAddress, IN const BTSResult& btsResult);

      /**
       * This method is used to post the Siri SetNR Result to the VR controller state machine
       *
       * @param[in] bdAddress  - BT address of the device
       * @param[out] btsResult - BT stack result for the Siri Set NR Request
       * @param[in,out]
       *
       * @return PmResult - PmCore result message
       */
      PmResult siriSetNRResult(IN const BdAddress& bdAddress, IN const BTSResult& btsResult);

      /**
       * This method is used to post the prepare audio route response to the VR Controller state machine
       *
       * @param[in] pmCoreIfMessage - shared ptr for PrepareAudioRouteResponse
       * @param[out]
       * @param[in,out]
       *
       * @return
       */
      void prepareAudioRouteResponse(IN std::shared_ptr<PmCoreIfMessage_PrepareAudioRouteResponse> pmCoreIfMessage);

      /**
       * This method is used to post the play Audio response to the VR Controller state machine
       *
       * @param[in] pmCoreIfMessage - shared ptr for PlayAudioResponse
       * @param[out]
       * @param[in,out]
       *
       * @return
       */
      void playAudioResponse(IN std::shared_ptr<PmCoreIfMessage_PlayAudioResponse> pmCoreIfMessage);

      /**
       * This method is used to post the stop Audio response to the VR Controller state machine
       *
       * @param[in] pmCoreIfMessage - shared ptr for StopAudioResponse
       * @param[out]
       * @param[in,out]
       *
       * @return
       */
      void stopAudioResponse(IN std::shared_ptr<PmCoreIfMessage_StopAudioResponse> pmCoreIfMessage);

      /**
       * This method is used to know whether the VR status of the device is IDLE or not
       *
       * @param[in] bdAddress  - BT address of the device
       * @param[in]
       * @param[out]
       * @param[in,out]
       *
       * @return true- if VR status is IDLE, else false
       */
      bool isVRStatusIdle(const BdAddress& bdAddress);

      /**
       * This method is used to know whether the External VR status of the device is IDLE or not
       *
       * @param[in] bdAddress  - BT address of the device
       * @param[in]
       * @param[out]
       * @param[in,out]
       *
       * @return true- if External VR status is IDLE, else false
       */
      bool isExternalVRStatusIdle(const BdAddress& bdAddress);

      /**
       * This method is used to switch the device to Passive
       * If VR is in Active state, then the VR should be stopped in this device and the VR channel should be paused.
       *
       * @param[in] bdAddress       - BT address of the device
       * @param[in] audioChannelToAcquire - The audio channel to be acquired after stopping VR
       * @param[out]
       * @param[in,out]
       *
       * @return void
       */
      void switchToPassive(const BdAddress& deviceAddress, const pmaudiomanager::AudioChannel audioChannelToAcquire);

      /**
       * This method is used to switch the device to Passive based on client request
       * If VR is in Active state, then the VR should be stopped in this device and
       * the acquired VR channel should be released.
       *
       * @param[in] bdAddress       - BT address of the device
       * @param[in]
       * @param[out]
       * @param[in,out]
       *
       * @return void
       */
      void switchToPassiveClientRequest(const BdAddress& deviceAddress);

      /**
       * This method is used to Stop extended vr on receiving media ready to play status
       *
       * @param[in] bdAddress       - BT address of the device
       * @param[in]
       * @param[out]
       * @param[in,out]
       *
       * @return void
       */
      void stopExtendedVR(const BdAddress& deviceAddress);

   private:
      PropertyIdList       _propertyIdList;     /**< List of property Id's */
      ISmVoiceRecIf*       _smVoiceRecIf;       /**< pointer for voice recognition sm interface */

      // This map is used to maintain the VR state of each connected device
      static std::map<pmcore::BdAddress, pmcore::VoiceRecognitionStatus> _deviceVRStatus;

      // This map is used to maintain the External VR state of each connected device
      static std::map<pmcore::BdAddress, pmcore::ExtVoiceRecognitionStatus> _deviceExtVRStatus;

      // This map is used to maintain the enhanced feature support for each device
      static std::map<pmcore::BdAddress, pmcore::EnhancedVoiceRecognitionFeature> _enhancedVRSupportedDevices;
   };
}

#endif // VRController_h
