/**
 * @file YakinduSmRingtoneIf.cpp
 *
 * @swcomponent PhoneCallManager
 *
 * @brief This file contains the definition of the YakinduSmRingtoneIf class
 *
 * @copyright (C) 2016 Robert Bosch GmbH.
 *            The reproduction, distribution and utilization of this file as
 *            well as the communication of its contents to others without express
 *            authorization is prohibited. Offenders will be held liable for the
 *            payment of damages. All rights reserved in the event of the grant
 *            of a patent, utility model or design.
 *
 * @details This file provides the state machine interfaces of ringtone
 *
 * @ingroup PmCore
 */

#include "YakinduSmRingtoneIf.h"
#include "PmCoreMainController.h"
#include "PmRingtoneHandler.h"
#include "PmCoreMainController.h"
#include "PmAppTrace.h"

#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#define ETG_DEFAULT_TRACE_CLASS TR_CLASS_PM_CORE
#ifdef VARIANT_S_FTR_ENABLE_FW_ETG_USAGE
#include "trcGenProj/Header/YakinduSmRingtoneIf.cpp.trc.h"
#define ETG_DEFAULT_TRACE_CLASS TR_CLASS_PM_CORE
#endif
#endif

namespace pmcore
{
   YakinduSmRingtoneIf::YakinduSmRingtoneIf()
   {
      ETG_TRACE_USR4(("YakinduSmRingtoneIf::YakinduSmRingtoneIf"));

      _deviceToRTHandlerSmMap.emplace_hint(_deviceToRTHandlerSmMap.end(), DEVICE_ADDRESS_ALL,
            new PmRingtoneHandler(DEVICE_ADDRESS_ALL));
   }

   YakinduSmRingtoneIf::~YakinduSmRingtoneIf()
   {
      for (auto &iter : _deviceToRTHandlerSmMap)
      {
         if (iter.second)
         {
            delete iter.second;
         }
      }
      _deviceToRTHandlerSmMap.clear();
   }

   void YakinduSmRingtoneIf::onAudioManagerEventUpdate(IN const BdAddress& deviceAddress,
         IN const AudioManagerEventType amEvent)
   {
      ETG_TRACE_USR4(("YakinduSmRingtoneIf::onAudioManagerEventUpdate"));

      auto it = _deviceToRTHandlerSmMap.find(deviceAddress);
      if(it != _deviceToRTHandlerSmMap.end())
      {
         it->second->onAudioManagerEventUpdate(amEvent);
      }
   }

   void YakinduSmRingtoneIf::playRingtoneRequest(IN std::shared_ptr<PmCoreIfMessage_PlayRingtoneRequest> pmCoreIfMessage)
   {
      ETG_TRACE_USR4(("YakinduSmRingtoneIf::playRingtoneRequest"));

      auto it = _deviceToRTHandlerSmMap.find(pmCoreIfMessage->getBdAddress());
      if(it != _deviceToRTHandlerSmMap.end())
      {
         it->second->playRingtoneRequest(pmCoreIfMessage);
      }
   }

   void YakinduSmRingtoneIf::switchToWaitForStopAudioResponse(
         IN std::shared_ptr<PmCoreIfMessage_PlayRingtoneRequest> pmCoreIfMessage)
   {
      ETG_TRACE_USR4(("YakinduSmRingtoneIf::switchToWaitForStopAudioResponse"));

      auto it = _deviceToRTHandlerSmMap.find(pmCoreIfMessage->getBdAddress());
      if(it != _deviceToRTHandlerSmMap.end())
      {
         it->second->switchToWaitForStopAudioResponse(pmCoreIfMessage);
      }
   }

   void YakinduSmRingtoneIf::switchToWaitForPlayAudioResponse(
         IN std::shared_ptr<PmCoreIfMessage_PlayRingtoneRequest> pmCoreIfMessage)
   {
      ETG_TRACE_USR4(("YakinduSmRingtoneIf::switchToWaitForPlayAudioResponse"));

      auto it = _deviceToRTHandlerSmMap.find(pmCoreIfMessage->getBdAddress());
      if(it != _deviceToRTHandlerSmMap.end())
      {
         it->second->switchToWaitForPlayAudioResponse(pmCoreIfMessage);
      }
   }

   void YakinduSmRingtoneIf::switchToWaitForChannelAcquisition(
         IN std::shared_ptr<PmCoreIfMessage_PlayRingtoneRequest> pmCoreIfMessage)
   {
      ETG_TRACE_USR4(("YakinduSmRingtoneIf::switchToWaitForChannelAcquisition"));

      auto it = _deviceToRTHandlerSmMap.find(pmCoreIfMessage->getBdAddress());
      if(it != _deviceToRTHandlerSmMap.end())
      {
         it->second->switchToWaitForChannelAcquisition(pmCoreIfMessage);
      }
   }

   void YakinduSmRingtoneIf::pauseRingtoneRequest(IN const BdAddress deviceAddress)
   {
      ETG_TRACE_USR4(("YakinduSmRingtoneIf::pauseRingtoneRequest"));

      auto it = _deviceToRTHandlerSmMap.find(deviceAddress);
      if(it != _deviceToRTHandlerSmMap.end())
      {
         it->second->pauseRingtoneRequest();
      }
   }

   void YakinduSmRingtoneIf::stopRingtoneRequest(IN std::shared_ptr<PmCoreIfMessage_StopRingtoneRequest> pmCoreIfMessage)
   {
      ETG_TRACE_USR4(("YakinduSmRingtoneIf::stopRingtoneRequest"));

      auto it = _deviceToRTHandlerSmMap.find(pmCoreIfMessage->getBdAddress());
      if(it != _deviceToRTHandlerSmMap.end())
      {
         it->second->stopRingtoneRequest(pmCoreIfMessage);
      }
   }

   void YakinduSmRingtoneIf::setRingtoneMuteStateRequest(IN const BdAddress& bdAddress,
         IN const MuteState ringtoneMuteState)
   {
      ETG_TRACE_USR4(("YakinduSmRingtoneIf::setRingtoneMuteStateRequest"));

      auto it = _deviceToRTHandlerSmMap.find(bdAddress);
      if(it != _deviceToRTHandlerSmMap.end())
      {
         it->second->setRingtoneMuteStateRequest(bdAddress, ringtoneMuteState);
      }
   }

   void YakinduSmRingtoneIf::switchToActiveRequest(IN const BdAddress& bdAddress)
   {
      ETG_TRACE_USR4(("YakinduSmRingtoneIf::switchToActiveRequest"));

      auto it = _deviceToRTHandlerSmMap.find(bdAddress);
      if(it != _deviceToRTHandlerSmMap.end())
      {
         it->second->switchToActiveRequest();
      }
   }

   void YakinduSmRingtoneIf::processSourceSwitch(IN const BdAddress& bdAddress)
   {
      ETG_TRACE_USR4(("YakinduSmRingtoneIf::processSourceSwitch entered with bdAddress: %20s",
            bdAddress.c_str()));

      auto it = _deviceToRTHandlerSmMap.find(bdAddress);
      if(it != _deviceToRTHandlerSmMap.end())
      {
         it->second->processSourceSwitch();
      }
   }

   void YakinduSmRingtoneIf::onDeviceConnected(IN const BdAddress& deviceAddress)
   {
      ETG_TRACE_USR4(("YakinduSmRingtoneIf::onDeviceConnected"));

      auto it = _deviceToRTHandlerSmMap.find(deviceAddress);

      if(it == _deviceToRTHandlerSmMap.end())
      {
         _deviceToRTHandlerSmMap.emplace_hint(_deviceToRTHandlerSmMap.end(), deviceAddress,
               new PmRingtoneHandler(deviceAddress));
      }
      else
      {
         ETG_TRACE_USR4(("SM already exists for device %20s", deviceAddress.c_str()));
      }
   }

   void YakinduSmRingtoneIf::onDeviceDisconnected(IN const BdAddress& deviceAddress)
   {
      ETG_TRACE_USR4(("YakinduSmRingtoneIf::onDeviceDisconnected"));

      auto it = _deviceToRTHandlerSmMap.find(deviceAddress);
      if(it != _deviceToRTHandlerSmMap.end())
      {
         if(it->second)
            delete it->second;

         _deviceToRTHandlerSmMap.erase(it);
      }
   }

}
