/**
 * @file RingtoneController.cpp
 *
 * @swcomponent PhoneCallManager
 *
 * @brief This file contains the definition of the RingtoneController class
 *
 * @copyright (C) 2016 Robert Bosch GmbH.
 *            The reproduction, distribution and utilization of this file as
 *            well as the communication of its contents to others without express
 *            authorization is prohibited. Offenders will be held liable for the
 *            payment of damages. All rights reserved in the event of the grant
 *            of a patent, utility model or design.
 *
 * @details This file provides the RingtoneController interfaces of PmCore.
 *
 * @ingroup PmCore
 */

#include "RingtoneController.h"
#include "PmCoreMainController.h"
#include "PropertyUpdateNotifierToCore.h"
#include "PmCoreIfMessageRequest.h"
#include "PmCoreIfMessageResult.h"
#include "PropertyDetails.h"
#include "EvoBtStackWrapper.h"
#include "YakinduSmRingtoneIf.h"
#include "PmAppTrace.h"

#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#define ETG_DEFAULT_TRACE_CLASS TR_CLASS_PM_CORE
#ifdef VARIANT_S_FTR_ENABLE_FW_ETG_USAGE
#include "trcGenProj/Header/RingtoneController.cpp.trc.h"
#define ETG_DEFAULT_TRACE_CLASS TR_CLASS_PM_CORE
#endif
#endif

using namespace com::bosch::pmcommon;

namespace pmcore
{
   RingtoneController::RingtoneController() :
      _propertyIdList(),
      _ringtoneMuteStateListMap(),
      _ringtonePlaybackSMState(RT_SM_NOT_PLAYING)
   {
      ETG_TRACE_USR1(("RingtoneController"));

      _smRingtoneIf = new YakinduSmRingtoneIf;

      subscribeToBtStackEventNotifier();
   }

   RingtoneController::~RingtoneController()
   {
      ETG_TRACE_USR1(("~RingtoneController"));

      PropertyUpdateNotifierToCore::getInstance().detachControllerInNotifierList(_propertyIdList, this);
      _propertyIdList.clear();
      _ringtoneMuteStateListMap.clear();

      if(nullptr != _smRingtoneIf)
      {
         delete _smRingtoneIf;
         _smRingtoneIf = nullptr;
      }
   }

   void RingtoneController::subscribeToBtStackEventNotifier()
   {
      ETG_TRACE_USR1(("subscribeToBtStackEventNotifier"));

      // Events from PM Audiomanager
      _propertyIdList.push_back(AM_RINGTONE_CHANNEL_ACQUISITION_SUCCESS);
      _propertyIdList.push_back(AM_RINGTONE_CHANNEL_ACQUISITION_FAILURE);
      _propertyIdList.push_back(AM_PHONE_CHANNEL_ACQUISITION_SUCCESS);
      _propertyIdList.push_back(AM_PHONE_CHANNEL_STOPAUDIO_SUCCESS);
      _propertyIdList.push_back(AM_VR_CHANNEL_ACQUISITION_SUCCESS);
      _propertyIdList.push_back(AM_WAITINGMODE_CHANNEL_ACQUISITION_SUCCESS);
      _propertyIdList.push_back(AM_RINGTONE_CHANNEL_PLAYAUDIO_SUCCESS);
      _propertyIdList.push_back(AM_RINGTONE_CHANNEL_PLAYAUDIO_FAILURE);
      _propertyIdList.push_back(AM_RINGTONE_CHANNEL_PAUSEAUDIO_SUCCESS);
      _propertyIdList.push_back(AM_RINGTONE_CHANNEL_PAUSEAUDIO_FAILURE);
      _propertyIdList.push_back(AM_RINGTONE_CHANNEL_STOPAUDIO_SUCCESS);
      _propertyIdList.push_back(AM_RINGTONE_CHANNEL_STOPAUDIO_FAILURE);
      _propertyIdList.push_back(AM_RINGTONE_CHANNEL_RELEASED);

      _propertyIdList.push_back(ON_DEVICE_CONNECTED);
      _propertyIdList.push_back(ON_DEVICE_DISCONNECTED);

      PropertyUpdateNotifierToCore::getInstance().attachControllerToNotifierList(_propertyIdList, this);
   }

   void RingtoneController::onPropertyUpdate(IN const PmCorePropertyAndEventId propertyId,
         IN std::shared_ptr<void> propertyDetails)
   {
      ETG_TRACE_USR4(("RingtoneController::onPropertyUpdate propertyId : %d",
            ETG_CENUM(PmCorePropertyAndEventId, propertyId)));

      if(nullptr == propertyDetails)
      {
         ETG_TRACE_ERR(("RingtoneController::onPropertyUpdate with empty details"));
         return;
      }

      switch(propertyId)
      {
         case AM_RINGTONE_CHANNEL_ACQUISITION_SUCCESS:
         {
            std::shared_ptr<PropertyDetails<AmErrorMessage>> property =
                  std::static_pointer_cast<PropertyDetails<AmErrorMessage>>(propertyDetails);

            ETG_TRACE_USR4(("AM_RINGTONE_CHANNEL_ACQUISITION_SUCCESS- DeviceAddress: %s",
                  property->getBdAddress().c_str()));

            _smRingtoneIf->onAudioManagerEventUpdate(property->getBdAddress(), RINGTONE_CHANNEL_GRANTED);
         }
         break;
         // Intentional fall-through
         case AM_PHONE_CHANNEL_ACQUISITION_SUCCESS:
         case AM_VR_CHANNEL_ACQUISITION_SUCCESS:
         case AM_WAITINGMODE_CHANNEL_ACQUISITION_SUCCESS:
         {
            std::shared_ptr<PropertyDetails<AmErrorMessage>> property =
                  std::static_pointer_cast<PropertyDetails<AmErrorMessage>>(propertyDetails);

            BdAddress deviceAddress = property->getBdAddress();

            ETG_TRACE_USR4(("AM_PHONE_CHANNEL_ACQUISITION_SUCCESS or "
                  "AM_VR_CHANNEL_ACQUISITION_SUCCESS or AM_WAITINGMODE_CHANNEL_ACQUISITION_SUCCESS- DeviceAddress: %s",
                  deviceAddress.c_str()));

            processSourceSwitch(deviceAddress);
         }
         break;
         case AM_RINGTONE_CHANNEL_ACQUISITION_FAILURE:
         {
            std::shared_ptr<PropertyDetails<AmErrorMessage>> property =
                  std::static_pointer_cast<PropertyDetails<AmErrorMessage>>(propertyDetails);

            ETG_TRACE_USR4(("AM_RINGTONE_CHANNEL_ACQUISITION_FAILURE- DeviceAddress: %s",
                  property->getBdAddress().c_str()));

            _smRingtoneIf->onAudioManagerEventUpdate(property->getBdAddress(), RINGTONE_CHANNEL_DENIED);
         }
         break;
         case AM_RINGTONE_CHANNEL_PLAYAUDIO_SUCCESS:
         {
            std::shared_ptr<PropertyDetails<AmErrorMessage>> property =
                  std::static_pointer_cast<PropertyDetails<AmErrorMessage>>(propertyDetails);

            ETG_TRACE_USR4(("AM_RINGTONE_CHANNEL_PLAYAUDIO_SUCCESS- DeviceAddress: %s",
                  property->getBdAddress().c_str()));

            _smRingtoneIf->onAudioManagerEventUpdate(property->getBdAddress(), PLAY_AUDIO_SUCCESS);
         }
         break;
         case AM_RINGTONE_CHANNEL_PLAYAUDIO_FAILURE:
         {
            std::shared_ptr<PropertyDetails<AmErrorMessage>> property =
                  std::static_pointer_cast<PropertyDetails<AmErrorMessage>>(propertyDetails);

            ETG_TRACE_USR4(("AM_RINGTONE_CHANNEL_PLAYAUDIO_FAILURE- DeviceAddress: %s",
                  property->getBdAddress().c_str()));

            _smRingtoneIf->onAudioManagerEventUpdate(property->getBdAddress(), PLAY_AUDIO_FAILURE);
         }
         break;
         case AM_RINGTONE_CHANNEL_PAUSEAUDIO_SUCCESS:
         {
            std::shared_ptr<PropertyDetails<AmErrorMessage>> property =
                  std::static_pointer_cast<PropertyDetails<AmErrorMessage>>(propertyDetails);

            ETG_TRACE_USR4(("AM_RINGTONE_CHANNEL_PAUSEAUDIO_SUCCESS- DeviceAddress: %s",
                  property->getBdAddress().c_str()));

            _smRingtoneIf->onAudioManagerEventUpdate(property->getBdAddress(), PAUSE_AUDIO_SUCCESS);
         }
         break;
         case AM_RINGTONE_CHANNEL_PAUSEAUDIO_FAILURE:
         {
            std::shared_ptr<PropertyDetails<AmErrorMessage>> property =
                  std::static_pointer_cast<PropertyDetails<AmErrorMessage>>(propertyDetails);

            ETG_TRACE_USR4(("AM_RINGTONE_CHANNEL_PAUSEAUDIO_FAILURE- DeviceAddress: %s",
                  property->getBdAddress().c_str()));

            _smRingtoneIf->onAudioManagerEventUpdate(property->getBdAddress(), PAUSE_AUDIO_FAILURE);
         }
         break;
         case AM_PHONE_CHANNEL_STOPAUDIO_SUCCESS:
         {
            std::shared_ptr<PropertyDetails<AmErrorMessage>> property =
                  std::static_pointer_cast<PropertyDetails<AmErrorMessage>>(propertyDetails);

            ETG_TRACE_USR4(("AM_PHONE_CHANNEL_STOPAUDIO_SUCCESS- DeviceAddress: %s",
                  property->getBdAddress().c_str()));

            _smRingtoneIf->onAudioManagerEventUpdate(property->getBdAddress(), STOP_AUDIO_SUCCESS);
         }
         break;
         case AM_RINGTONE_CHANNEL_STOPAUDIO_SUCCESS:
         {
            std::shared_ptr<PropertyDetails<AmErrorMessage>> property =
                  std::static_pointer_cast<PropertyDetails<AmErrorMessage>>(propertyDetails);

            ETG_TRACE_USR4(("AM_RINGTONE_CHANNEL_STOPAUDIO_SUCCESS- DeviceAddress: %s",
                  property->getBdAddress().c_str()));

            _smRingtoneIf->onAudioManagerEventUpdate(property->getBdAddress(), STOP_AUDIO_SUCCESS);
         }
         break;
         case AM_RINGTONE_CHANNEL_STOPAUDIO_FAILURE:
         {

            std::shared_ptr<PropertyDetails<AmErrorMessage>> property =
                  std::static_pointer_cast<PropertyDetails<AmErrorMessage>>(propertyDetails);

            ETG_TRACE_USR4(("AM_RINGTONE_CHANNEL_STOPAUDIO_FAILURE- DeviceAddress: %s",
                  property->getBdAddress().c_str()));

            _smRingtoneIf->onAudioManagerEventUpdate(property->getBdAddress(), STOP_AUDIO_FAILURE);

         }
         break;
         case AM_RINGTONE_CHANNEL_RELEASED:
         {
            std::shared_ptr<PropertyDetails<AmErrorMessage>> property =
                  std::static_pointer_cast<PropertyDetails<AmErrorMessage>>(propertyDetails);

            ETG_TRACE_USR4(("RINGTONE_CHANNEL_RELEASED- DeviceAddress: %s",
                  property->getBdAddress().c_str()));

            _smRingtoneIf->onAudioManagerEventUpdate(property->getBdAddress(), RINGTONE_CHANNEL_RELEASED);
         }
         break;

         case ON_DEVICE_CONNECTED:
         {
             std::shared_ptr<PropertyDetails<BasicDeviceDetails>> property =
                   std::static_pointer_cast<PropertyDetails<BasicDeviceDetails>>(propertyDetails);

             ETG_TRACE_USR4(("onPropertyUpdate deviceAddress : %s", property->getMessage()._deviceAddress.c_str()));
             static_cast<YakinduSmRingtoneIf*>(_smRingtoneIf)->onDeviceConnected(property->getMessage()._deviceAddress);
         }
         break;

         case ON_DEVICE_DISCONNECTED:
         {
             std::shared_ptr<PropertyDetails<BdAddress>> property =
                   std::static_pointer_cast<PropertyDetails<BdAddress>>(propertyDetails);

             BdAddress deviceAddress = property->getMessage();

             ETG_TRACE_USR4(("onPropertyUpdate deviceAddress : %s", property->getMessage().c_str()));

             static_cast<YakinduSmRingtoneIf*>(_smRingtoneIf)->onDeviceDisconnected(property->getMessage());
         }
         break;

         default:
            ETG_TRACE_ERR(("RingtoneController::Invalid propertyId"));
            break;
      }
   }

   // Request calls
   void RingtoneController::playRingtoneRequest(IN std::shared_ptr<PmCoreIfMessage_PlayRingtoneRequest> pmCoreIfMessage)
   {
      ETG_TRACE_USR1(("playRingtoneRequest() entered"));

      BdAddress deviceAddress = pmCoreIfMessage->getBdAddress();
      bool processRequest = true;
      DeviceInfoHandler& deviceInfoHandler = PmCoreMainController::getInstance().getDeviceInfoHandler();

      // The vehicle RingtonePlaybackState is noted so that based on that state,
      // the RT SM is started for the incoming call
      RingtonePlaybackStateSMType vehicleRingtonePlaybackSMState = _ringtonePlaybackSMState;

      if (DEVICE_ADDRESS_ALL == deviceAddress)
      {
         if (SWITCHED_TO_ACTIVE != deviceInfoHandler.acquireFreeActiveDeviceSlot(deviceAddress, ActiveSwitchingRequest()))
         {
            processRequest = false;
            pmCoreIfMessage->setPmResult(PmResult(PM_RESULT_ERR_HFP, "Call or VR in progress"));
         }
      }
      else
      {
         DeviceRole deviceRole = DEVICEROLE_DEFAULT;
         deviceInfoHandler.getDeviceRole(deviceAddress, deviceRole);
         if (DEVICEROLE_ACTIVE == deviceRole)
         {
            // Need to switch the RT SM corresponding to DEVICE_ADDRESS_ALL to the IDLE state
            _smRingtoneIf->pauseRingtoneRequest(DEVICE_ADDRESS_ALL);
            PmAudioManagerWrapper::getInstance().swapDeviceAddress(DEVICE_ADDRESS_ALL, deviceAddress, CALL_CONTROLLER);
         }
      }

      if (processRequest)
      {
         // TODO: Check if there is any other possibility to communicate between different state machines
         switch (vehicleRingtonePlaybackSMState)
         {
            case RT_SM_STOPPING:
               _smRingtoneIf->switchToWaitForStopAudioResponse(pmCoreIfMessage);
               break;
            case RT_SM_STARTING_TO_PLAY:
               _smRingtoneIf->switchToWaitForPlayAudioResponse(pmCoreIfMessage);
               break;
            case RT_SM_PREPARING_CH:
               _smRingtoneIf->switchToWaitForChannelAcquisition(pmCoreIfMessage);
               break;
            default:
               _smRingtoneIf->playRingtoneRequest(pmCoreIfMessage);
         }
      }
   }

   void RingtoneController::stopRingtoneRequest(IN std::shared_ptr<PmCoreIfMessage_StopRingtoneRequest> pmCoreIfMessage)
   {
      ETG_TRACE_USR1(("stopRingtoneRequest() entered"));

      _smRingtoneIf->stopRingtoneRequest(pmCoreIfMessage);
   }

   PmResult RingtoneController::setRingtoneMuteStateRequest(IN const BdAddress& bdAddress,
         IN const MuteState ringtoneMuteState, IN const ActType act)
   {
      ETG_TRACE_USR1(("setRingtoneMuteStateRequest ringtoneMuteState :%d", ringtoneMuteState));
      PmResult pmResult(PM_RESULT_ERR_GENERAL, "");

      auto it = _ringtoneMuteStateListMap.find(bdAddress);
      if(it == _ringtoneMuteStateListMap.end())
      {
         ETG_TRACE_USR4(("RingtoneController::setRingtoneMuteStateRequest Inserting new element into a map"));
         //Inserting new element into a map with RingtoneMuteState as FALSE
         //Actual value will be updated after MUTE/UNMUTE success
         _ringtoneMuteStateListMap.insert(std::pair<BdAddress, RingtoneMuteState>(bdAddress,false));
      }

      if(nullptr != _smRingtoneIf)
      {
         _smRingtoneIf->setRingtoneMuteStateRequest(bdAddress, ringtoneMuteState);
         pmResult._pmResultCode = PM_RESULT_OK;
      }

      // Response to client's request
      PmCoreMainController::getInstance().getPmCoreCallbackIfWrapper().doSetRingtoneMuteStateResponse(pmResult,
            bdAddress, act);

      return pmResult;
   }

   void RingtoneController::updateRingtoneMuteState(IN const BdAddress& bdAddress,IN const MuteState ringtoneMuteState)
   {
      ETG_TRACE_USR1(("updateRingtoneMuteState() entered"));

     auto it = _ringtoneMuteStateListMap.find(bdAddress);

      if(it != _ringtoneMuteStateListMap.end())
      {
         if(ringtoneMuteState != it->second._ringtoneMuteState)
         {
            it->second._ringtoneMuteState = ringtoneMuteState;
            PmCoreMainController::getInstance().getPmCoreCallbackIfWrapper().doOnRingtoneMuteStateChanged(
                  bdAddress, it->second);
         }
      }
      else
      {
         ETG_TRACE_ERR(("RingtoneController::Map element is not present"));
      }
   }

   void RingtoneController::getRingtoneMuteStateRequest(IN const BdAddress& bdAddress, IN const ActType act)
   {
      ETG_TRACE_USR1(("getRingtoneMuteStateRequest() entered"));

      PmResult pmResult(PM_RESULT_ERR_DEVICE_NOT_EXIST, "");
      RingtoneMuteState ringtoneMuteState;
      auto it = _ringtoneMuteStateListMap.find(bdAddress);

      if(it != _ringtoneMuteStateListMap.end())
      {
         ringtoneMuteState = it->second;
         pmResult._pmResultCode = PM_RESULT_OK;
      }

      PmCoreMainController::getInstance().getPmCoreCallbackIfWrapper().doGetRingtoneMuteStateResponse(pmResult,
            bdAddress, ringtoneMuteState, act);
   }

   void RingtoneController::switchToActiveRequest(IN const BdAddress& bdAddress)
   {
      ETG_TRACE_USR1(("RingtoneController::switchToActiveRequest() entered"));
      _smRingtoneIf->switchToActiveRequest(bdAddress);
   }

   void RingtoneController::processSourceSwitch(IN const BdAddress& bdAddress)
   {
      ETG_TRACE_USR1(("RingtoneController::processSourceSwitch() entered with deviceAddress: %20s",
            bdAddress.c_str()));
      _smRingtoneIf->processSourceSwitch(bdAddress);
   }

   void RingtoneController::getRingtonePlaybackStatusRequest(IN const ActType act)
   {
      ETG_TRACE_USR1(("RingtoneController::getRingtonePlaybackStatusRequest() entered"));

      // Response to client's request
      PmCoreMainController::getInstance().getPmCoreCallbackIfWrapper().doGetRingtonePlaybackStatusResponse(
            PmResult(), RingtonePlaybackState(getRingtonePlaybackStateFromSMType(_ringtonePlaybackSMState)), act);
   }

   void RingtoneController::onRingtonePlaybackStateChanged(
         const RingtonePlaybackStateSMType ringtonePlaybackStateSMType)
   {
      ETG_TRACE_USR1(("RingtoneController::onRingtonePlaybackStateChanged() entered"));

      _ringtonePlaybackSMState = ringtonePlaybackStateSMType;

      // State is updated as RT_STARTING on status update- "RT_SM_PREPARING_CH" itself. Hence ignoring the status update
      // "RT_STARTING_TO_PLAY" from SCT
      if (RT_SM_STARTING_TO_PLAY != _ringtonePlaybackSMState)
      {
         PmCoreMainController::getInstance().getPmCoreCallbackIfWrapper().doOnRingtonePlaybackStatusChanged(
               RingtonePlaybackState(getRingtonePlaybackStateFromSMType(ringtonePlaybackStateSMType)));
      }
   }

   RingtonePlaybackStateType RingtoneController::getRingtonePlaybackStateFromSMType(
         const RingtonePlaybackStateSMType ringtonePlaybackStateSMType)
   {
      ETG_TRACE_USR4(("RingtoneController::getRingtonePlaybackStateFromSMType() entered"));

      RingtonePlaybackStateType state;

      switch (ringtonePlaybackStateSMType)
      {
         case RT_SM_PLAYING:
            state = RT_PLAYING;
            break;
         case RT_SM_STARTING_TO_PLAY:
         case RT_SM_PREPARING_CH:
            state = RT_STARTING;
            break;
         case RT_SM_STOPPING:
            state = RT_STOPPING;
            break;
         default:
            state = RT_NOT_PLAYING;
      }

      return state;
   }

} // namespace pmcore
