/**
 * @file YakinduSmRingtoneIf.h
 *
 * @swcomponent PhoneCallManager
 *
 * @brief This file contains the definition of the YakinduSmRingtoneIf class
 *
 * @copyright (C) 2016 Robert Bosch GmbH.
 *            The reproduction, distribution and utilization of this file as
 *            well as the communication of its contents to others without express
 *            authorization is prohibited. Offenders will be held liable for the
 *            payment of damages. All rights reserved in the event of the grant
 *            of a patent, utility model or design.
 *
 * @details This file provides the state machine interfaces of voice call
 *
 * @ingroup PmCore
 */

#ifndef YakinduSmRingtoneIf_h
#define YakinduSmRingtoneIf_h

#include "ISmRingtoneIf.h"

namespace pmcore
{
   //class forward declaration
   class PmRingtoneHandler;

   class YakinduSmRingtoneIf final : public ISmRingtoneIf
   {
   public:
      /**
       * Constructor of YakinduSmRingtoneIf class
       *
       * @param[in]
       * @param[out]
       * @param[in,out]
       *
       * @return
       */
      YakinduSmRingtoneIf();

      /**
       * Destructor of YakinduSmRingtoneIf class
       *
       * @param[in]
       * @param[out]
       * @param[in,out]
       *
       * @return
       */
      ~YakinduSmRingtoneIf();

      /**
       * This method is used to post the AudioManager update to the Ringtone state machine
       *
       * @param[in] deviceAddress - device address
       * @param[in] amEvent - audio manager event update
       * @param[out]
       * @param[in,out]
       *
       * @return void
       */
      void onAudioManagerEventUpdate(IN const BdAddress& deviceAddress,
            IN const AudioManagerEventType amEvent) override;

      /**
       * Member function overridden from ISmRingtoneIf class
       *
       * @param[in]
       * @param[out]
       * @param[in,out]
       *
       * @return void
       */
      void playRingtoneRequest(IN std::shared_ptr<PmCoreIfMessage_PlayRingtoneRequest> pmCoreIfMessage) override;

      /**
       * Member function overridden from ISmRingtoneIf class
       *
       * @param[in]
       * @param[out]
       * @param[in,out]
       *
       * @return void
       */
      void switchToWaitForStopAudioResponse(
            IN std::shared_ptr<PmCoreIfMessage_PlayRingtoneRequest> pmCoreIfMessage) override;

      /**
       * Member function overridden from ISmRingtoneIf class
       *
       * @param[in]
       * @param[out]
       * @param[in,out]
       *
       * @return void
       */
      virtual void switchToWaitForPlayAudioResponse(
            IN std::shared_ptr<PmCoreIfMessage_PlayRingtoneRequest> pmCoreIfMessage) override;

      /**
       * Member function overridden from ISmRingtoneIf class
       *
       * @param[in]
       * @param[out]
       * @param[in,out]
       *
       * @return void
       */
      virtual void switchToWaitForChannelAcquisition(
            IN std::shared_ptr<PmCoreIfMessage_PlayRingtoneRequest> pmCoreIfMessage) override;

      /**
       * Member function overridden from ISmRingtoneIf class
       *
       * @param[in]
       * @param[out]
       * @param[in,out]
       *
       * @return void
       */
      void pauseRingtoneRequest(IN const BdAddress deviceAddress) override;

      /**
       * Member function overridden from ISmRingtoneIf class
       *
       * @param[in]
       * @param[out]
       * @param[in,out]
       *
       * @return void
       */
      void stopRingtoneRequest(IN std::shared_ptr<PmCoreIfMessage_StopRingtoneRequest> pmCoreIfMessage) override;

      /**
       * Member function overridden from ISmRingtoneIf class
       *
       * @param[in]
       * @param[out]
       * @param[in,out]
       *
       * @return void
       */
      virtual void setRingtoneMuteStateRequest(IN const BdAddress& bdAddress,
            IN const MuteState ringtoneMuteState) override;

      /**
       * This method is used to post the switch to Active request to the SM
       *
       * @param[in] bdAddress - Device address of the connected device
       * @param[out]
       * @param[in,out]
       *
       * @return void
       */
      virtual void switchToActiveRequest(IN const BdAddress& bdAddress) override;

      /**
       * This method is used to inform the SM about the acquisition of a channel other then Ring-tone channel
       * for the given device address
       *
       * @param[in] bdAddress - BT address of the device
       * @param[out]
       * @param[in,out]
       *
       * @return void
       */
      virtual void processSourceSwitch(IN const BdAddress& bdAddress) override;

      /**
       * This method is used to notify the device connection
       *
       * @param[in] deviceAddress - device address
       * @param[out]
       * @param[in,out]
       *
       * @return void
       */
      void onDeviceConnected(IN const BdAddress& deviceAddress);

      /**
       * This method is used to notify the device disconnection
       *
       * @param[in] deviceAddress - device address
       * @param[out]
       * @param[in,out]
       *
       * @return void
       */
      void onDeviceDisconnected(IN const BdAddress& deviceAddress);

   private:
      ::std::map<BdAddress, PmRingtoneHandler*> _deviceToRTHandlerSmMap; //map between device address and RT handler
   };
}

#endif //YakinduSmRingtoneIf_h
