/**
 * @file RingtoneController.h
 *
 * @swcomponent PhoneCallManager
 *
 * @brief This file contains the definition of the RingtoneController class
 *
 * @copyright (C) 2016 Robert Bosch GmbH.
 *            The reproduction, distribution and utilization of this file as
 *            well as the communication of its contents to others without express
 *            authorization is prohibited. Offenders will be held liable for the
 *            payment of damages. All rights reserved in the event of the grant
 *            of a patent, utility model or design.
 *
 * @details This file provides the RingtoneController interfaces of PmCore
 *
 * @ingroup PmCore
 */

#ifndef RingtoneController_h
#define RingtoneController_h

#include "ISmRingtoneIf.h"
#include "PmCoreIfTypes.h"
#include "PropertyUpdateHandler.h"
#include "PmAudioManagerWrapper.h"

namespace pmcore
{
   class ISmVoiceCallIf;
   class PmCoreIfMessage_PlayRingtoneRequest;
   class PmCoreIfMessage_StopRingtoneRequest;

   class RingtoneController : public PropertyUpdateHandler
   {
   public:
      /**
       * Constructor of RingtoneController class
       *
       * @param[in]
       * @param[out]
       * @param[in,out]
       *
       * @return
       */
      RingtoneController();

      /**
       * Destructor of RingtoneController class
       *
       * @param[in]
       * @param[out]
       * @param[in,out]
       *
       * @return
       */
      ~RingtoneController();

      /**
       * Copy constructor of RingtoneController class
       *
       * @param[in] ref - Reference to RingtoneController
       * @param[out]
       * @param[in,out]
       *
       * @return
       *
       */
      RingtoneController(const RingtoneController& ref);

      /**
       * operator= overloaded function
       *
       * @param[in]
       * @param[out]
       * @param[in,out]
       *
       * @return Returns the object of the RingtoneController
       *
       */
      RingtoneController& operator=(const RingtoneController& ref);

      /**
       * This method is used to subscribe interested properties in BT stack event notifier.
       *
       * @param[in]
       * @param[out]
       * @param[in,out]
       *
       * @return
       */
      void subscribeToBtStackEventNotifier();

      /**
       * This method is used to get property changes update from BT Stack
       * This method is overridden from PropertyUpdateHandler class
       *
       * @param[in] propertyId      - Id of the property or event
       * @param[in] propertyDetails - Details of the property or event
       * @param[out]
       * @param[in,out]
       *
       * @return void
       */
      virtual void onPropertyUpdate(IN const PmCorePropertyAndEventId propertyId,
            IN std::shared_ptr<void> propertyDetails) override;

      // **************************************************************************************************
      // Request calls

      /**
       * This method is used to post the PlayRingTone Request to the ringtone handler state machine
       *
       * @param[in] pmCoreIfMessage - shared ptr for PlayRingTone
       * @param[out]
       * @param[in,out]
       *
       * @return PmResult - PmCore result message
       */
      void playRingtoneRequest(IN std::shared_ptr<PmCoreIfMessage_PlayRingtoneRequest> pmCoreIfMessage);

      /**
       * This method is used to post the Stop ringtone Request to the ringtone handler state machine
       *
       * @param[in] pmCoreIfMessage - shared ptr for StopRingtone
       * @param[out]
       * @param[in,out]
       *
       * @return void
       */
      void stopRingtoneRequest(IN std::shared_ptr<PmCoreIfMessage_StopRingtoneRequest> pmCoreIfMessage);

      /**
       * This method is used to post the set Ringtone Mute state request to the Ringtone handler state machine
       *
       * @param[in] bdAddress         - BdAddress of the device
       * @param[in] ringtoneMuteState - flag for MuteState
       * @param[in] act               - acknowledgement token
       * @param[out]
       * @param[in,out]
       *
       * @return void
       */
      PmResult setRingtoneMuteStateRequest(IN const BdAddress& bdAddress,
            IN const MuteState ringtoneMuteState, IN const ActType act);

      /**
       * This method is used to update the Ringtone mute state to Ringtone Handler
       * @param[in] bdAddress - BT Address of the device
       * @param[in] ringtoneMuteState - flag for MuteState
       * @param[out]
       * @param[in,out]
       *
       * @return void
       */
      void updateRingtoneMuteState(IN const BdAddress& bdAddress, IN const MuteState ringtoneMuteState);

      /**
       * This method is used to post the Get Ringtone MuteState Request to the Ringtone Handler state machine
       *
       * @param[in] bdAddress - BT address of the device
       * @param[in] act       - acknowledgment token
       * @param[out]
       * @param[in,out]
       *
       * @return void
       */
      void getRingtoneMuteStateRequest(IN const BdAddress& bdAddress, IN const ActType act);

      /**
       * This method is used to post the Switch to Active request to the SM
       *
       * @param[in] bdAddress - BT address of the device
       * @param[in] act       - acknowledgment token
       * @param[out]
       * @param[in,out]
       *
       * @return void
       */
      void switchToActiveRequest(IN const BdAddress& bdAddress);

      /**
       * This method is used to get the vehicle Ring-tone by the clients
       *
       * @param[in] act       - acknowledgment token
       * @param[out]
       * @param[in,out]
       *
       * @return void
       */
      void getRingtonePlaybackStatusRequest(IN const ActType act);

      /**
       * This method is used to get the vehicle Ring-tone by the clients
       *
       * @param[in] ringtonePlaybackState  - ref to RingtonePlaybackState class
       * @param[out]
       * @param[in,out]
       *
       * @return void
       */
      void onRingtonePlaybackStateChanged(const RingtonePlaybackStateSMType ringtonePlaybackStateSMType);

      // **************************************************************************************************
      // Result calls

   private:
      /**
       * This method is used to inform the SM about the acquisition of a channel other than Ring-tone channel for the
       * given device address
       *
       * @param[in] bdAddress - BT address of the device
       * @param[out]
       * @param[in,out]
       *
       * @return void
       */
      void processSourceSwitch(IN const BdAddress& bdAddress);

      /**
       * This method is used to get the RingtonePlaybackStateType from RingtonePlaybackStateSMType
       *
       * @param[in] RingtonePlaybackStateSMType - RingtonePlaybackState SM Type
       * @param[out] RingtonePlaybackStateType - RingtonePlaybackState Interface type
       * @param[in,out]
       *
       * @return void
       */
      RingtonePlaybackStateType getRingtonePlaybackStateFromSMType(
               const RingtonePlaybackStateSMType ringtonePlaybackStateSMType);

      PropertyIdList                          _propertyIdList;           /**< list of property id's */
      ISmRingtoneIf*                          _smRingtoneIf;             /**< pointer for Ringtone sm interface */
      std::map<BdAddress, RingtoneMuteState>  _ringtoneMuteStateListMap; /**< Ringtone MuteState list */
      RingtonePlaybackStateSMType _ringtonePlaybackSMState; /**< Ringtone playback state type*/
   };
}

#endif // RingtoneController_h

