/**
 * @file PmRingtoneHandler.h
 *
 * @swcomponent PhoneCallManager
 *
 * @brief This file contains the definition of the PmRingtoneHandler class
 *
 * @copyright (C) 2016 Robert Bosch GmbH.
 *            The reproduction, distribution and utilization of this file as
 *            well as the communication of its contents to others without express
 *            authorization is prohibited. Offenders will be held liable for the
 *            payment of damages. All rights reserved in the event of the grant
 *            of a patent, utility model or design.
 *
 * @details This file provides the state machine interfaces of Ringtone Handler
 *
 * @ingroup PmCore
 */

#ifndef PmRingtoneHandler_h
#define PmRingtoneHandler_h

#include "EvoBtStackWrapper.h"
#include "PmCoreIfMessageResult.h"
#include "../src-gen/PmRingtoneHandlerSm.h"

namespace pmcore
{
   class PmRingtoneHandler final: public PmRingtoneHandlerSm
   {
   public:
      /**
       * Constructor of PmRingtoneHandler class
       *
       * @param[in] deviceAddress - device address
       * @param[out]
       * @param[in,out]
       *
       * @return
       */
      PmRingtoneHandler();

      /**
       * Parameterized Constructor of PmRingtoneHandler class
       *
       * @param[in] deviceAddress - device address
       * @param[out]
       * @param[in,out]
       *
       * @return
       */
      PmRingtoneHandler(const BdAddress& deviceAddress);

      /**
       * Destructor of PmRingtoneHandler class
       *
       * @param[in]
       * @param[out]
       * @param[in,out]
       *
       * @return
       */
      ~PmRingtoneHandler();

      /**
       * This method is used to post the AudioManager Event Updates  to the ringtone handler state machine
       *
       * @param[in] amEvent - AudioManager Event
       * @param[out]
       * @param[in,out]
       *
       * @return void
       */
      void onAudioManagerEventUpdate(IN const AudioManagerEventType amEvent);

      /**
       * This method is used to post the PlayRingTone Request to the ringtone handler state machine
       *
       * @param[in] pmCoreIfMessage - shared ptr for PlayRingTone
       * @param[out]
       * @param[in,out]
       *
       * @return PmResult - PmCore result message
       */
      void playRingtoneRequest(IN std::shared_ptr<PmCoreIfMessage_PlayRingtoneRequest> pmCoreIfMessage);

      /**
       * This method is used to inform the SM that it has to wait for Stop Audio response
       * and then should start playing ringtone
       *
       * @param[in] pmCoreIfMessage - shared ptr for PlayRingTone
       * @param[out]
       * @param[in,out]
       *
       * @return PmResult - PmCore result message
       */
      void switchToWaitForStopAudioResponse(
            IN std::shared_ptr<PmCoreIfMessage_PlayRingtoneRequest> pmCoreIfMessage);

      /**
       * This method informs SM to wait for Play audio response to replay the RT
       *
       * @param[in]
       * @param[out]
       * @param[in,out]
       *
       * @return void
       */
      virtual void switchToWaitForPlayAudioResponse(
            IN std::shared_ptr<PmCoreIfMessage_PlayRingtoneRequest> pmCoreIfMessage);

      /**
       * This method is used to process the Play Ringtone request after Ring-tone channel acquisition
       *
       * @param[in]
       * @param[out]
       * @param[in,out]
       *
       * @return void
       */
      virtual void switchToWaitForChannelAcquisition(
            IN std::shared_ptr<PmCoreIfMessage_PlayRingtoneRequest> pmCoreIfMessage);

      /**
       * This method is used to post the pause Request to the ringtone handler state machine
       *
       * @param[in]
       * @param[out]
       * @param[in,out]
       *
       * @return PmResult - PmCore result message
       */
      void pauseRingtoneRequest();

      /**
       * This method is used to post the Stop ringtone Request to the ringtone handler state machine
       *
       * @param[in] pmCoreIfMessage - shared ptr for StopRingtone
       * @param[out]
       * @param[in,out]
       *
       * @return void
       */
      void stopRingtoneRequest(IN std::shared_ptr<PmCoreIfMessage_StopRingtoneRequest> pmCoreIfMessage);

      /**
       * This method is used to post the set ringtone Request to the ringtone handler state machine
       *
       * @param[in] bdAddress         - device BT address
       * @param[in] ringtoneMuteState - flag for ringtone mute state.
       * @param[out]
       * @param[in,out]
       *
       * @return void
       */
      void setRingtoneMuteStateRequest(IN const BdAddress& bdAddress, IN const MuteState ringtoneMuteState);

      /**
       * This method is used to post the switch to Active request to the SM
       *
       * @param[in]
       * @param[out]
       * @param[in,out]
       *
       * @return void
       */
      void switchToActiveRequest();

      /**
       * This method is used to inform the SM acquisition of a channel other than Ring-tone channel
       * for the given device address
       *
       * @param[in]
       * @param[out]
       * @param[in,out]
       *
       * @return void
       */
      void processSourceSwitch();

      // **************************************************************************************************
      // Result calls

   private:

      /**
       * This function is used to get the deviceAddress of the SM instance
       *
       * @param[in]
       * @param[out]
       * @param[in,out]
       *
       * @return BdAddress - device address
       */
      const BdAddress& getDeviceAddress() { return _deviceAddress; }

      /**
       * SmUserActionIf class inherits the functions of SCI_UserAction_OCB class
       *
       * @param[in]
       * @param[out]
       * @param[in,out]
       *
       * @return
       */
      class SmUserActionIf final: public SCI_UserAction_OCB
      {
      public:
         SmUserActionIf(PmRingtoneHandler& PmRingtoneHandler): _pmRingtoneHandler(PmRingtoneHandler){}

         ~SmUserActionIf(){}

         virtual void stopRingtone(pmcore::PmCoreIfMessage_StopRingtoneRequest * msg) override;

      private:
         PmRingtoneHandler&    _pmRingtoneHandler;
      };

      class SmSmHandlerIf final: public SCI_SmHandler_OCB
      {
      public:
         SmSmHandlerIf(PmRingtoneHandler& PmRingtoneHandler): _pmRingtoneHandler(PmRingtoneHandler){}

         ~SmSmHandlerIf(){}

         virtual void sendPlayResponse(PmResultMessage result) override;

         virtual void sendPauseResponse(PmResultMessage result) override;

         virtual void updateRingtoneStatus(RingtonePlaybackStateSMType smState) override;

      private:
         PmRingtoneHandler&    _pmRingtoneHandler;
      };

      class SmAudioManagerRequestIf final: public SCI_AudioManagerRequest_OCB
      {
      public:
         SmAudioManagerRequestIf(PmRingtoneHandler& PmRingtoneHandler): _pmRingtoneHandler(PmRingtoneHandler){}

         ~SmAudioManagerRequestIf(){}

         virtual void requestRingtoneAudio() override;

         virtual void playRingtone() override;

         virtual void pauseRingtone() override;

         virtual void resumeRingtone() override;

      private:
         PmRingtoneHandler&    _pmRingtoneHandler;
      };

      /**
       * SmTraceObserverIf class inherits the functions of TraceObserver class
       *
       * @param[in]
       * @param[out]
       * @param[in,out]
       *
       * @return
       */
      template<typename T>
      class SmTraceObserverIf final: public ysc::TraceObserver<T>
      {
      public:
         SmTraceObserverIf(PmRingtoneHandler& pmRingtoneHandler):
            _pmRingtoneHandler(pmRingtoneHandler){}

         virtual ~SmTraceObserverIf(){}

         virtual void stateEntered(T state) override;

         virtual void stateExited(T state) override;

      private:
         PmRingtoneHandler&    _pmRingtoneHandler;
      };

      BdAddress                        _deviceAddress;                           /**< BT address of the device */
      SmUserActionIf*                  _smUserActionIf;                          /**< pointer for UserAction interface */
      SmSmHandlerIf*                   _smSmHandlerIf;                           /**< pointer for SmHandler interface */
      SmAudioManagerRequestIf*         _smAudioManagerRequestIf;                 /**< pointer for AudioManger Request interface */
      std::vector<PmCoreIfMessage_PlayRingtoneRequest> _playRingtoneRequestList; /**< List of PlayRingtoneRequest*/
      SmTraceObserverIf<PmRingtoneHandlerSmStates>* _smTraceObserverIf; /**< pointer for TraceObserver interface */
   };
}

#endif //YakinduSmVoiceCallIf_h
