/**
 * @file RVController.cpp
 *
 * @swcomponent PhoneCallManager
 *
 * @brief This file contains the definition of the RVController class
 *
 * @copyright (C) 2016 Robert Bosch GmbH.
 *            The reproduction, distribution and utilization of this file as
 *            well as the communication of its contents to others without express
 *            authorization is prohibited. Offenders will be held liable for the
 *            payment of damages. All rights reserved in the event of the grant
 *            of a patent, utility model or design.
 *
 * @details This file provides the RVController interfaces of PmCore.
 *
 * @ingroup PmCore
 */

#include "RVController.h"
#include "PmCoreMainController.h"
#include "PropertyUpdateNotifierToCore.h"
#include "PropertyDetails.h"
#include "YakinduSmVolumeControlIf.h"
#include "PmConfiguration.h"
#include "PmAppTrace.h"

#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#define ETG_DEFAULT_TRACE_CLASS TR_CLASS_PM_CORE
#ifdef VARIANT_S_FTR_ENABLE_FW_ETG_USAGE
#include "trcGenProj/Header/RVController.cpp.trc.h"
#define ETG_DEFAULT_TRACE_CLASS TR_CLASS_PM_CORE
#endif
#endif

using namespace com::bosch::pmcommon;

namespace pmcore
{
   RVController::RVController() :
      _propertyIdList()
   {
      ETG_TRACE_USR1(("RVController"));

      _smVolumeControlIf = new YakinduSmVolumeControlIf;

      subscribeToBtStackEventNotifier();
   }

   RVController::~RVController()
   {
      ETG_TRACE_USR1(("~RVController"));

      _asfTimerInstanceMap.clear();

      PropertyUpdateNotifierToCore::getInstance().detachControllerInNotifierList(_propertyIdList, this);
      _propertyIdList.clear();

      if(nullptr != _smVolumeControlIf)
      {
         delete _smVolumeControlIf;
         _smVolumeControlIf = nullptr;
      }
   }

   void RVController::subscribeToBtStackEventNotifier()
   {
      ETG_TRACE_USR1(("subscribeToBtStackEventNotifier"));

      // Pushing the interested properties and events
      _propertyIdList.push_back(ON_DEVICE_CONNECTED);
      _propertyIdList.push_back(ON_DEVICE_DISCONNECTED);
      _propertyIdList.push_back(AM_UPDATE_AUDIO_VOLUME);
      _propertyIdList.push_back(BTS_UPDATE_SPEAKER_VOLUME);

      bool rvcEnabled = com::bosch::pmcommon::PmConfiguration::getInstance().getRemoteVolumeControl();
      if(rvcEnabled)
      {
          PropertyUpdateNotifierToCore::getInstance().attachControllerToNotifierList(_propertyIdList, this);
      }
   }

   void RVController::onPropertyUpdate(IN const PmCorePropertyAndEventId propertyId,
         IN std::shared_ptr<void> propertyDetails)
   {
      ETG_TRACE_USR4(("RVController::onPropertyUpdate propertyId : %d",
            ETG_CENUM(PmCorePropertyAndEventId, propertyId)));

      if(nullptr == propertyDetails)
      {
         ETG_TRACE_ERR(("RVController::onPropertyUpdate with empty details"));
         return;
      }

      switch(propertyId)
      {
         case ON_DEVICE_CONNECTED:
         {
            std::shared_ptr<PropertyDetails<BasicDeviceDetails>> property =
                  std::static_pointer_cast<PropertyDetails<BasicDeviceDetails>>(propertyDetails);

            ETG_TRACE_USR4(("onPropertyUpdate deviceAddress : %s", property->getMessage()._deviceAddress.c_str()));
            ETG_TRACE_USR4(("onPropertyUpdate deviceHandle : %u", property->getMessage()._deviceHandle));

            static_cast<YakinduSmVolumeControlIf*>(_smVolumeControlIf)->onDeviceConnected(property->getMessage()._deviceAddress,
                  property->getMessage()._deviceHandle);
         }
         break;
         case ON_DEVICE_DISCONNECTED:
         {
            std::shared_ptr<PropertyDetails<BdAddress>> property =
                  std::static_pointer_cast<PropertyDetails<BdAddress>>(propertyDetails);

            BdAddress deviceAddress = property->getMessage();
            ETG_TRACE_USR4(("onPropertyUpdate deviceAddress : %s", property->getMessage().c_str()));

            static_cast<YakinduSmVolumeControlIf*>(_smVolumeControlIf)->onDeviceDisconnected(property->getMessage());
         }
         break;
         case AM_UPDATE_AUDIO_VOLUME:
         {
            std::shared_ptr<PropertyDetails<pmaudiomanager::AmPropertyType2VolumeMap>> property =
                  std::static_pointer_cast<PropertyDetails<pmaudiomanager::AmPropertyType2VolumeMap>>(propertyDetails);

            //Here device address is empty intentionally, since AM does not provide device address
            ETG_TRACE_USR4(("onPropertyUpdate deviceAddress : %s", property->getBdAddress().c_str()));

            BdAddress bdAddress("");
            pmaudiomanager::AmPropertyType2VolumeMap amVolumeLevelListMap = property->getMessage();

            pmaudiomanager::GeniviAMPropertyType gamVoiceRecSourceID = com::bosch::pmcommon::PmConfiguration::getInstance().getGAMVoiceRecSourceID();
            pmaudiomanager::GeniviAMPropertyType gamDefaultPhoneSourceXID = com::bosch::pmcommon::PmConfiguration::getInstance().getGAMDefaultPhoneSourceXID();
            pmaudiomanager::GeniviAMPropertyType gamDefaultPhoneSourceOffsetID = com::bosch::pmcommon::PmConfiguration::getInstance().getGAMDefaultPhoneSourceOffsetID();
            pmaudiomanager::GeniviAMVolumeLevel minAMVolumeLevel = com::bosch::pmcommon::PmConfiguration::getInstance().getMinAMVolumeLevel();
            pmaudiomanager::GeniviAMVolumeLevel maxAMVolumeLevel = com::bosch::pmcommon::PmConfiguration::getInstance().getMaxAMVolumeLevel();

            for (auto it = amVolumeLevelListMap.begin(); it != amVolumeLevelListMap.end(); it++)
            {
               //components/audio/ai_audio/components/AudioManager/include/audiomanagertypes.h
               //Phone related volume settings starts from 97 to 106 which is mapped to device handle 2 to 11.
               //MSP_VOLUME_PHONE_1 = 97 and so on
               if(it->first == gamVoiceRecSourceID)
               {
                  if((it->second >= minAMVolumeLevel) && (it->second <= maxAMVolumeLevel))
                  {
                     ETG_TRACE_USR4(("amVolumeLevelListMap DeviceHandle:%d, VolumeLevel:%d",
                           it->first, it->second));
                     BdAddressList activeDevicesList;
                     VRController& vRController = PmCoreMainController::getInstance().getVRController();
                     DeviceInfoHandler& deviceInfoHandler = PmCoreMainController::getInstance().getDeviceInfoHandler();

                     deviceInfoHandler.getActiveDevicesList(activeDevicesList);
                     for(auto it = activeDevicesList.begin(); it != activeDevicesList.end(); it++)
                     {
                        if(false == vRController.isVRStatusIdle(*it))
                        {
                           bdAddress = *it;
                           break;
                        }
                        else
                        {
                           ETG_TRACE_USR4(("RVController::VR session is not active for the device address"));
                        }
                     }
                  }
               }
               else if (it->first >= gamDefaultPhoneSourceXID)
               {
                  if((it->second >= minAMVolumeLevel) && (it->second <= maxAMVolumeLevel))
                  {
                     ETG_TRACE_USR4(("amVolumeLevelListMap DeviceHandle:%d, VolumeLevel:%d",
                           it->first, it->second));
                     bdAddress = static_cast<YakinduSmVolumeControlIf*>(_smVolumeControlIf)->getBdAddressFromDeviceHandle(
                           static_cast<unsigned char>(it->first - gamDefaultPhoneSourceOffsetID));
                  }
               }

               if(0 != bdAddress.compare(""))
               {
                  static_cast<YakinduSmVolumeControlIf*>(_smVolumeControlIf)->onVolumeUpdateFromAm(bdAddress,
                        static_cast<unsigned char>(it->second));
               }
            }
         }
         break;
         case BTS_UPDATE_SPEAKER_VOLUME:
         {
            std::shared_ptr<PropertyDetails<VolumeLevel>> property =
                  std::static_pointer_cast<PropertyDetails<VolumeLevel>>(propertyDetails);

            ETG_TRACE_USR4(("onPropertyUpdate deviceAddress : %s", property->getBdAddress().c_str()));
            ETG_TRACE_USR4(("onPropertyUpdate Volume : %d", property->getMessage()));

            static_cast<YakinduSmVolumeControlIf*>(_smVolumeControlIf)->onVolumeUpdateFromAg(property->getBdAddress(),
                  property->getMessage());
         }
         break;
         default:
            ETG_TRACE_ERR(("RVController::Invalid propertyId"));
            break;
      }
   }

   void RVController::startAsfTimer(std::string token, ms_t duration, RepeatCount repeatCount)
   {
      ETG_TRACE_USR4(("RVController::startAsfTimer Token: %s",token.c_str()));

      com::bosch::pmcommon::TimerData<std::string /*Token*/, ms_t /*Duration*/, RepeatCount> timerdata(
            token,(unsigned int)duration,(unsigned int)repeatCount);
      auto it = _asfTimerInstanceMap.find(token);
      if(it == _asfTimerInstanceMap.end())
      {
         ETG_TRACE_USR4(("starting the new timer"));

         com::bosch::pmcommon::ITimer<RVController,std::string,ms_t,RepeatCount>* asfTimer = new AsfTimer<RVController,std::string>();
         //Inserting Token and asfTimer instance pair
         _asfTimerInstanceMap.emplace_hint(_asfTimerInstanceMap.end(),token, asfTimer);
         asfTimer->start(this,timerdata);
      }
      else
      {
         ETG_TRACE_USR4(("Timer instance found in _asfTimerInstanceMap"));

         if(it->second->isStopped())
         {
            ETG_TRACE_USR4(("starting the old timer"));
            //Timer instance for the Token is already available.
            //So reusing the same
            it->second->start(this,timerdata);
         }
      }
   }

   void RVController::stopAsfTimer(std::string token)
   {
      ETG_TRACE_USR4(("RVController::stopAsfTimer Token: %s",token.c_str()));

      auto it = _asfTimerInstanceMap.find(token);

      if(it != _asfTimerInstanceMap.end())
      {
         if(true == it->second->isActive())
         {
            ETG_TRACE_USR4(("Stopping the active timer"));

            //Stop timer
            it->second->stop();
         }
      }
   }

   void RVController::deleteAsfTimer(std::string token)
   {
      ETG_TRACE_USR4(("RVController::deleteAsfTimer Token: %s",token.c_str()));

      auto it = _asfTimerInstanceMap.find(token);

      if(it != _asfTimerInstanceMap.end())
      {
         ETG_TRACE_USR4(("RVController::deleteAsfTimer isActive :%d",it->second->isActive()));
         if(false == it->second->isActive())
         {
            ETG_TRACE_USR4(("Deleting the stopped timer"));

            //Deleting the timer instance
            delete it->second;

            //Erasing the timer instance
            _asfTimerInstanceMap.erase(it);
            ETG_TRACE_USR4(("RVController::_asfTimerInstanceMap.size() :%d",_asfTimerInstanceMap.size()));
         }
      }
   }

   void RVController::timerElapsed(com::bosch::pmcommon::TimerData<std::string,ms_t,RepeatCount> data,
         com::bosch::pmcommon::AsfTimerCallbackData timerCallbackData)
   {
      ETG_TRACE_USR4(("RVController::timerCallbackData._repeatCount : %d", timerCallbackData._repeatCount));
      ETG_TRACE_USR4(("RVController::_timerInfo : %s", data._timerInfo.c_str()));

      static_cast<YakinduSmVolumeControlIf*>(_smVolumeControlIf)->onTimerEventUpdate(data._timerInfo.substr(
            DEVICE_ADDRESS_START,DEVICE_ADDRESS_LENGTH));
   }

} // namespace pmcore
