/**
 * @file YakinduSmVolumeControlIf.h
 *
 * @swcomponent PhoneCallManager
 *
 * @brief This file contains the definition of the YakinduSmVolumeControlIf class
 *
 * @copyright (C) 2016 Robert Bosch GmbH.
 *            The reproduction, distribution and utilization of this file as
 *            well as the communication of its contents to others without express
 *            authorization is prohibited. Offenders will be held liable for the
 *            payment of damages. All rights reserved in the event of the grant
 *            of a patent, utility model or design.
 *
 * @details This file provides the state machine interfaces of Remote Volume control
 *
 * @ingroup PmCore
 */

#ifndef YakinduSmVolumeControlIf_h
#define YakinduSmVolumeControlIf_h

#include "ISmVolumeControlIf.h"

namespace pmcore
{
   //class forward declaration
   class RVCHandler;

   class YakinduSmVolumeControlIf final : public ISmVolumeControlIf
   {
   public:
      /**
       * Constructor of YakinduSmVolumeControlIf class
       *
       * @param[in]
       * @param[out]
       * @param[in,out]
       *
       * @return
       */
      YakinduSmVolumeControlIf();

      /**
       * Destructor of YakinduSmVolumeControlIf class
       *
       * @param[in]
       * @param[out]
       * @param[in,out]
       *
       * @return
       */
      ~YakinduSmVolumeControlIf();

      /**
       * This method is used to get the device address from device handle
       *
       * @param[in] deviceHandle - device handle
       * @param[out]
       * @param[in,out]
       *
       * @return BdAddress - Device Address
       */
      BdAddress getBdAddressFromDeviceHandle(IN const DeviceHandle& deviceHandle);

      /**
       * This method is used to notify the device connection
       *
       * @param[in] deviceAddress - device address
       * @param[out]
       * @param[in,out]
       *
       * @return void
       */
      void onDeviceConnected(IN const BdAddress& deviceAddress, IN const DeviceHandle& deviceHandle);

      /**
       * This method is used to notify the device disconnection
       *
       * @param[in] deviceAddress - device address
       * @param[out]
       * @param[in,out]
       *
       * @return void
       */
      void onDeviceDisconnected(IN const BdAddress& deviceAddress);

      /**
       * This method is used to post the volume update from Audio to the Remote volume state machine
       *
       * @param[in] deviceAddress - Address of BT device
       * @param[in] volume - volume level
       * @param[out]
       * @param[in,out]
       *
       * @return void
       */
      void onVolumeUpdateFromAm(IN const BdAddress& deviceAddress, IN const VolumeLevel& volume) override;

      /**
       * This method is used to post the volume update from AG to the Remote volume state machine
       *
       * @param[in] deviceAddress - Address of BT device
       * @param[in] volume - volume level
       * @param[out]
       * @param[in,out]
       *
       * @return void
       */
      void onVolumeUpdateFromAg(IN const BdAddress& deviceAddress, IN const VolumeLevel& volume) override;

      /**
       * This method is used to post the volume update from AG to the Remote volume state machine
       *
       * @param[in] deviceAddress - Address of BT device
       * @param[out]
       * @param[in,out]
       *
       * @return void
       */
      void onTimerEventUpdate(IN const BdAddress& deviceAddress) override;

   private:
      ::std::map<BdAddress, RVCHandler*>   _deviceToRVCHandlerSmMap;   //map between device address and RVC handler
      ::std::map<DeviceHandle, BdAddress>  _deviceIdToDevAddrMap;      //map between device Id and device Address
   };
}

#endif //YakinduSmVolumeControlIf_h
