/**
 * @file PropertyUpdateNotifierToCore.cpp
 *
 * @swcomponent PhoneCallManager
 *
 * @brief This file contains the definition of the PropertyUpdateNotifierToCore class
 *
 * @copyright (C) 2016 Robert Bosch GmbH.
 *            The reproduction, distribution and utilization of this file as
 *            well as the communication of its contents to others without express
 *            authorization is prohibited. Offenders will be held liable for the
 *            payment of damages. All rights reserved in the event of the grant
 *            of a patent, utility model or design.
 *
 * @details This class holds the responsibility of updating the controllers  CallController, VRController,
 * DeviceInfoHandler on the property / event update received from stack based on the subscription done by the
 * controllers. Additionally, it also caters the information on device connected, device disconnected information
 * too to the respective controllers.
 *
 * @ingroup PmCore
 */

#include "PropertyUpdateNotifierToCore.h"
#include "PmAppTrace.h"

#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#define ETG_DEFAULT_TRACE_CLASS TR_CLASS_PM_CORE
#ifdef VARIANT_S_FTR_ENABLE_FW_ETG_USAGE
#include "trcGenProj/Header/PropertyUpdateNotifierToCore.cpp.trc.h"
#define ETG_DEFAULT_TRACE_CLASS TR_CLASS_PM_CORE
#endif
#endif

namespace pmcore
{
   PropertyUpdateNotifierToCore::PropertyUpdateNotifierToCore()
   {
      ETG_TRACE_USR1(("PropertyUpdateNotifierToCore"));
   }

   PropertyUpdateNotifierToCore::~PropertyUpdateNotifierToCore()
   {
      ETG_TRACE_USR1(("~PropertyUpdateNotifierToCore"));
      _propertyToControllersMap.clear();
   }

   void PropertyUpdateNotifierToCore::attachControllerToNotifierList(const PropertyIdList& propertyIdList,
            const PropertyUpdateHandler* propertyHandler)
   {
      ETG_TRACE_USR1(("attachControllerToNotifierList"));

      if(nullptr != propertyHandler)
      {
         PropertyIdList::const_iterator itIdList;
         PropertyToControllersMap::iterator itMap;

         // Iterate the list of properties to Subscribe
         for (itIdList = propertyIdList.begin(); itIdList != propertyIdList.end(); ++itIdList)
         {
            ETG_TRACE_USR4(("PropertyId :%d ", ETG_CENUM(PmCorePropertyAndEventId, *itIdList)));

            //Finding the given property in the _propertyToControllersMap
            itMap = _propertyToControllersMap.find(*itIdList);

            if (itMap != _propertyToControllersMap.end())
            {
               // Check if the controller is already added for this property.
               if (find(itMap->second.begin(), itMap->second.end(), propertyHandler) == itMap->second.end())
               {
                  ETG_TRACE_USR4(("Adding controller to the property : %d",
                        ETG_CENUM(PmCorePropertyAndEventId, *itIdList)));
                  itMap->second.push_back(const_cast <PropertyUpdateHandler*> (propertyHandler));
               }
               else
               {
                  ETG_TRACE_USR4(("This controller is already added for this property"));
               }
            }
            else
            {
               ETG_TRACE_USR4(("Inserting New PropertyId,controllersList pair into Map"));
               ControllerList controllersList;
               controllersList.push_back(const_cast <PropertyUpdateHandler*> (propertyHandler));
               _propertyToControllersMap.insert(std::pair<PmCorePropertyAndEventId, ControllerList>(
                     static_cast<PmCorePropertyAndEventId> (*itIdList), controllersList));

            }
         }
         printPropertyToControllersMap();
      }
   }

   void PropertyUpdateNotifierToCore::detachControllerInNotifierList(const PropertyIdList& propertyIdList,
         const PropertyUpdateHandler* propertyHandler)
   {
      ETG_TRACE_USR1(("detachControllerInNotifierList"));

      if(nullptr != propertyHandler)
      {
         PropertyIdList::const_iterator itIdList;
         PropertyToControllersMap::iterator itMap;
         ControllerList::iterator itVector;

         // Iterate the list of properties to UnSubscribe
         for (itIdList = propertyIdList.begin(); itIdList != propertyIdList.end(); ++itIdList)
         {
            ETG_TRACE_USR4(("PropertyId :%d ", ETG_CENUM(PmCorePropertyAndEventId, *itIdList)));

            //Finding the given property in the _propertyToControllersMap
            itMap = _propertyToControllersMap.find(*itIdList);

            if (itMap != _propertyToControllersMap.end())
            {
               // Check if the controller is present for this property.
               itVector = find(itMap->second.begin(), itMap->second.end(), propertyHandler);

               if (itVector != itMap->second.end())
               {
                  // Removing controller for that property
                  itMap->second.erase(itVector);

                  // Remove from map entry if no controller is registered for that property
                  if(0 == itMap->second.size())
                  {
                     _propertyToControllersMap.erase(itMap->first);
                  }
               }
               else
               {
                  ETG_TRACE_USR4(("This controller is not found for this property"));
               }
            }
            else
            {
               ETG_TRACE_USR4(("This property is not found in _propertyToControllersMap"));
            }
         }

         printPropertyToControllersMap();
      }
   }

   void PropertyUpdateNotifierToCore::printPropertyToControllersMap()
   {
      ETG_TRACE_USR1(("printPropertyToControllersMap"));
      std::map<PmCorePropertyAndEventId, ControllerList>::iterator itMap;
      ControllerList::iterator itVector;

      for (itMap = _propertyToControllersMap.begin(); itMap != _propertyToControllersMap.end(); ++itMap)
      {
         ETG_TRACE_USR4(("PropertyId :%d ",ETG_CENUM(PmCorePropertyAndEventId, itMap->first)));

         for (itVector = itMap->second.begin() ; itVector != itMap->second.end(); ++itVector )
         {
            ETG_TRACE_USR4(("Property controller Instance = %p",  *itVector));
         }
      }
   }

} // namespace pmcore
