/**
 * @file PmSettings.cpp
 *
 * @swcomponent PhoneCallManager
 *
 * @brief This file contains the type definition of the PmSettings class
 *
 * @copyright (C) 2016 Robert Bosch GmbH.
 *            The reproduction, distribution and utilization of this file as
 *            well as the communication of its contents to others without express
 *            authorization is prohibited. Offenders will be held liable for the
 *            payment of damages. All rights reserved in the event of the grant
 *            of a patent, utility model or design.
 *
 * @details
 *
 * @ingroup PmCore
 */

#include "PmCoreRequestIf.h"
#include "PmSettings.h"
#include "PmCoreMainController.h"
#include "PropertyUpdateNotifierToCore.h"
#include "PropertyDetails.h"
#include "DbHandler.h"
#include "FileUtils.h"
#include <dirent.h>
#include "PmAppTrace.h"

#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#define ETG_DEFAULT_TRACE_CLASS TR_CLASS_PM_CORE
#ifdef VARIANT_S_FTR_ENABLE_FW_ETG_USAGE
#include "trcGenProj/Header/PmSettings.cpp.trc.h"
#define ETG_DEFAULT_TRACE_CLASS TR_CLASS_PM_CORE
#endif
#endif

namespace pmcore
{
   #define  PM_RINGTONES_DIR  "/var/opt/bosch/static/connectivity/tones"

   PmSettings::PmSettings() :
         _propertyIdList(),
         _systemWideRingtonesList(),
         _deviceRingtoneList(),
         _suppressRingtoneOnOffList(),
         _autoWaitingModeOnOffList()
   {
      ETG_TRACE_USR4(("PmSettings() entered"));
      subscribeToBtStackEventNotifier();

      //Initialize DbHandler
      DbHandler::getInstance();
   }

   PmSettings::~PmSettings()
   {
      ETG_TRACE_USR4(("~PmSettings() entered"));

      PropertyUpdateNotifierToCore::getInstance().detachControllerInNotifierList(_propertyIdList, this);
      _propertyIdList.clear();
      _systemWideRingtonesList._systemRingtonesList.clear();
      _deviceRingtoneList._deviceRingtoneList.clear();
      _suppressRingtoneOnOffList._suppressRingtoneOnOffList.clear();
      _autoWaitingModeOnOffList._autoWaitingModeOnOffList.clear();
   }

   void PmSettings::subscribeToBtStackEventNotifier()
   {
      ETG_TRACE_USR1(("PmSettings::subscribeToBtStackEventNotifier() entered"));

      // Pushing the interested properties and events
      _propertyIdList.push_back(ON_DEVICE_PAIRED);
      _propertyIdList.push_back(ON_DEVICE_DELETED);
      _propertyIdList.push_back(BTS_UPDATE_INBAND_RINGING);

      PropertyUpdateNotifierToCore::getInstance().attachControllerToNotifierList(_propertyIdList, this);
   }

   void PmSettings::onDevicePaired(const DeviceHandle deviceHandle, const BdAddress& deviceAddress)
   {
      ETG_TRACE_USR4(("PmSettings::onDevicePaired() entered"));

      if(true == DbHandler::getInstance().addDeviceInDb(deviceHandle, deviceAddress))
      {
         ETG_TRACE_USR4(("onDevicePaired() device added to database"));

         updateDeviceRingtoneList();
         updateSuppressRingtoneOnOffList();
         updateAutoWaitingModeOnOffList();
      }
      else
      {
         ETG_TRACE_ERR(("onDevicePaired() error while adding device to database"));
      }
   }

   void PmSettings::onDeviceDeleted(const BdAddress& deviceAddress)
   {
      ETG_TRACE_USR4(("PmSettings::onDeviceDeleted() entered"));

      if(true == DbHandler::getInstance().removeDeviceFromDb(deviceAddress))
      {
         ETG_TRACE_USR4(("onDeviceDeleted() device removed database"));

         updateDeviceRingtoneList();
         updateSuppressRingtoneOnOffList();
         updateAutoWaitingModeOnOffList();
      }
      else
      {
         ETG_TRACE_ERR(("onDevicePaired() error while removing device from database"));
      }
   }

   void PmSettings::onPropertyUpdate(IN const PmCorePropertyAndEventId propertyId,
         IN std::shared_ptr<void> propertyDetails)
   {
      ETG_TRACE_USR4(("PmSettings::onPropertyUpdate propertyId : %d",
            ETG_CENUM(PmCorePropertyAndEventId, propertyId)));

      if(nullptr == propertyDetails)
      {
         ETG_TRACE_ERR(("PmSettings::onPropertyUpdate with empty details"));
         return;
      }

      switch(propertyId)
      {
         case ON_DEVICE_PAIRED:
         {
            std::shared_ptr<PropertyDetails<BasicDeviceDetails>> property =
                  std::static_pointer_cast<PropertyDetails<BasicDeviceDetails>>(propertyDetails);

            BdAddress deviceAddress = property->getMessage()._deviceAddress;

            ETG_TRACE_USR4(("onPropertyUpdate deviceAddress : %s", deviceAddress.c_str()));
            ETG_TRACE_USR4(("onPropertyUpdate deviceHandle : %u", property->getMessage()._deviceHandle));

            onDevicePaired(property->getMessage()._deviceHandle, deviceAddress);
         }
         break;

         case ON_DEVICE_DELETED:
         {
            std::shared_ptr<PropertyDetails<BdAddress>> property =
                  std::static_pointer_cast<PropertyDetails<BdAddress>>(propertyDetails);

            ETG_TRACE_USR4(("onPropertyUpdate deviceAddress : %s", property->getMessage().c_str()));

            onDeviceDeleted(property->getMessage());
         }
         break;

         case BTS_UPDATE_INBAND_RINGING:
         {
            std::shared_ptr<PropertyDetails<bool>> property =
                  std::static_pointer_cast<PropertyDetails<bool>>(propertyDetails);

            BdAddress deviceAddress = property->getBdAddress();
            bool isInbandRingingEnabled = property->getMessage();

            ETG_TRACE_USR4(("onPropertyUpdate BdAddress     : %s", deviceAddress.c_str()));
            ETG_TRACE_USR4(("onPropertyUpdate InbandRinging : %u", isInbandRingingEnabled));

            DeviceRingtoneListType& deviceRingtoneList = _deviceRingtoneList._deviceRingtoneList;

            auto deviceRingtoneListIter = std::find_if(deviceRingtoneList.begin(), deviceRingtoneList.end(),
                  [&deviceAddress](DeviceRingtone const& obj){return obj._deviceAddress == deviceAddress;});

            if (deviceRingtoneList.end() != deviceRingtoneListIter)
            {
               RingtoneId ringtoneId = RINGTONE_ID_DEFAULT;
               if(true == DbHandler::getInstance().getRingtoneId(deviceAddress, ringtoneId))
               {
                  if ((RINGTONE_ID_DEFAULT == ringtoneId) && (true == isInbandRingingEnabled))
                  {
                     deviceRingtoneListIter->_ringtoneId = RINGTONE_ID_INBAND;
                     deviceRingtoneListIter->_ringtoneName = RINGTONE_NAME_INBAND;
                  }
                  else if ((RINGTONE_ID_INBAND == deviceRingtoneListIter->_ringtoneId) && (false == isInbandRingingEnabled))
                  {
                     deviceRingtoneListIter->_ringtoneId = RINGTONE_ID_SYSTEM_DEFAULT;

                     if(RINGTONE_ID_DEFAULT != _systemWideRingtonesList._activeSystemRingtoneId)
                     {
                        deviceRingtoneListIter->_ringtoneId = _systemWideRingtonesList._activeSystemRingtoneId;
                     }

                     RingtoneName ringtoneName;
                     (void)getRingtoneName(deviceRingtoneListIter->_ringtoneId, ringtoneName);
                     deviceRingtoneListIter->_ringtoneName = ringtoneName;
                  }
               }
               if(true == PmCoreMainController::getInstance().getDeviceInfoHandler().isFeatureSupported(
            		   deviceAddress, "inband_ringtone"))
               {
            	   deviceRingtoneListIter->_inbandRingingSupport =
            			   isInbandRingingEnabled ? SUPPORTED_AND_ENABLED : SUPPORTED_AND_DISABLED;
               }
               else
               {
            	   deviceRingtoneListIter->_inbandRingingSupport = NOT_SUPPORTED;
               }

               PmCoreMainController::getInstance().getPmCoreCallbackIfWrapper().doOnDeviceRingtoneListChanged(
                     _deviceRingtoneList);
            }
         }
         break;

         default:
            ETG_TRACE_ERR(("PmSettings::Invalid propertyId"));
            break;
      }
   }

   void PmSettings::loadSettingsFromDb()
   {
      ETG_TRACE_USR4(("PmSettings::loadSettingsFromDb() entered"));

      if(true == DbHandler::getInstance().openDatabase())
      {
         ETG_TRACE_USR4(("loadSettingsFromDb() database open success"));

         //clear all system and device related settings.
         _systemWideRingtonesList._systemRingtonesList.clear();
         _deviceRingtoneList._deviceRingtoneList.clear();
         _suppressRingtoneOnOffList._suppressRingtoneOnOffList.clear();
         _autoWaitingModeOnOffList._autoWaitingModeOnOffList.clear();

         //Update all system and device related settings to clients.
         ReadSystemWideRingtonesFromDir();
         updateSystemWideRingtoneList();
         updateDeviceRingtoneList();
         updateSuppressRingtoneOnOffList();
         updateAutoWaitingModeOnOffList();
      }
      else
      {
         ETG_TRACE_ERR(("loadSettingsFromDb() database open failed"));
      }
   }

   void PmSettings::saveSettingsToDb()
   {
      ETG_TRACE_USR4(("PmSettings::saveSettingsToDb() entered"));

      DbHandler::getInstance().closeDatabase();
   }

   PmResult PmSettings::restoreDefaultSetting(IN const BdAddress& deviceAddress, IN const ActType act)
   {
      ETG_TRACE_USR4(("PmSettings::restoreDefaultSetting() DeviceAddress : %s", deviceAddress.c_str()));

      PmResult pmResult(PM_RESULT_ERR_GENERAL, "");

      if(true == DbHandler::getInstance().restoreDefaultSetting(deviceAddress))
      {
         if((0 == deviceAddress.compare(DEVICE_ADDRESS_ALL)) || (0 == deviceAddress.compare(DEVICE_ADDRESS_DELETE_ALL)))
         {
            updateSystemWideRingtoneList();
         }

         updateDeviceRingtoneList();
         updateSuppressRingtoneOnOffList();
         updateAutoWaitingModeOnOffList();

         pmResult._pmResultCode = PM_RESULT_OK;
      }

      PmCoreMainController::getInstance().getPmCoreCallbackIfWrapper().doRestoreDefaultSettingResponse(pmResult, act);

      return pmResult;
   }

   PmResult PmSettings::setRingtone(IN const BdAddress& deviceAddress, IN const RingtoneId ringtoneId,
         IN const ActType act)
   {
      PmResult pmResult(PM_RESULT_ERR_GENERAL, "");

      RingtoneId ringtoneIdLocal = ringtoneId;

      if(RINGTONE_ID_INBAND == ringtoneId)
      {
          ringtoneIdLocal = RINGTONE_ID_DEFAULT;
      }

      if((true == DbHandler::getInstance().setRingtoneId(deviceAddress, ringtoneIdLocal)))
      {
         if(0 == deviceAddress.compare(DEVICE_ADDRESS_ALL))
         {
            updateSystemWideRingtoneList();
         }

         updateDeviceRingtoneList();

         pmResult._pmResultCode = PM_RESULT_OK;
      }

      PmCoreMainController::getInstance().getPmCoreCallbackIfWrapper().doSetRingtoneResponse(pmResult, act);

      return pmResult;
   }

   void PmSettings::getSystemRingtonesList(IN const ActType act)
   {
      PmResult pmResult(PM_RESULT_OK, "");

      PmCoreMainController::getInstance().getPmCoreCallbackIfWrapper().doGetSystemWideRingtonesListResponse(
            pmResult, _systemWideRingtonesList, act);
   }

   void PmSettings::getDeviceRingtoneList(IN const ActType act)
   {
      PmResult pmResult(PM_RESULT_OK, "");

      PmCoreMainController::getInstance().getPmCoreCallbackIfWrapper().doGetDeviceRingtoneListResponse(
            pmResult, _deviceRingtoneList, act);
   }

   PmResult PmSettings::suppressRingtoneOnOff(IN const BdAddress& deviceAddress,
         IN SuppressRingtoneState suppressRingtoneState, IN const ActType act)
   {
      PmResult pmResult(PM_RESULT_ERR_GENERAL, "");

      if(true == DbHandler::getInstance().setSuppressRingtoneSetting(deviceAddress, suppressRingtoneState))
      {
         updateSuppressRingtoneOnOffList();

         pmResult._pmResultCode = PM_RESULT_OK;
      }

      PmCoreMainController::getInstance().getPmCoreCallbackIfWrapper().doSuppressRingtoneOnOffResponse(pmResult, act);

      return pmResult;
   }

   void PmSettings::getSuppressRingtoneOnOffList(IN const ActType act)
   {
      PmResult pmResult(PM_RESULT_OK, "");

      PmCoreMainController::getInstance().getPmCoreCallbackIfWrapper().doGetSuppressRingtoneOnOffListResponse(
            pmResult, _suppressRingtoneOnOffList, act);
   }

   PmResult PmSettings::autoWaitingModeOnOff(IN const BdAddress& deviceAddress,
         IN AutoWaitingModeState autoWaitingModeState, IN const ActType act)
   {
      PmResult pmResult(PM_RESULT_ERR_GENERAL, "");

      if(true == DbHandler::getInstance().setAutoWaitingModeSetting(deviceAddress, autoWaitingModeState))
      {
         updateAutoWaitingModeOnOffList();

         pmResult._pmResultCode = PM_RESULT_OK;
      }

      PmCoreMainController::getInstance().getPmCoreCallbackIfWrapper().doAutoWaitingModeOnOffResponse(pmResult, act);

      return pmResult;
   }

   void PmSettings::getAutoWaitingModeOnOffList(IN const ActType act)
   {
      PmResult pmResult(PM_RESULT_OK, "");

      PmCoreMainController::getInstance().getPmCoreCallbackIfWrapper().doGetAutoWaitingModeOnOffListResponse(
            pmResult, _autoWaitingModeOnOffList, act);
   }

   PmResult PmSettings::getRingtoneName(IN const BdAddress& deviceAddress, OUT RingtoneName& ringtoneName)
   {
      ETG_TRACE_USR4(("PmSettings::getRingtoneName for DeviceAddress: %s", deviceAddress.c_str()));

      PmResult pmResult(PM_RESULT_ERR_GENERAL, "");

      if(0 == deviceAddress.compare(DEVICE_ADDRESS_ALL))
      {
         RingtoneId ringtoneId = _systemWideRingtonesList._activeSystemRingtoneId;

         if(RINGTONE_ID_DEFAULT == ringtoneId)
         {
            ringtoneId = RINGTONE_ID_SYSTEM_DEFAULT;
         }

         for(auto systemRingtone : _systemWideRingtonesList._systemRingtonesList)
         {
            if(systemRingtone._ringtoneId == ringtoneId)
            {
               ringtoneName = systemRingtone._ringtoneName;
               pmResult._pmResultCode = PM_RESULT_OK;
               break;
            }
         }
      }
      else
      {
         for(auto& deviceRingtone : _deviceRingtoneList._deviceRingtoneList)
         {
            if(0 == deviceRingtone._deviceAddress.compare(deviceAddress))
            {
               ringtoneName = deviceRingtone._ringtoneName;
               pmResult._pmResultCode = PM_RESULT_OK;
               break;
            }
         }
      }

      ETG_TRACE_USR4(("PmSettings::getRingtoneName: %s", ringtoneName.c_str()));

      return pmResult;
   }

   PmResult PmSettings::getRingtoneName(IN RingtoneId ringtoneId, OUT RingtoneName& ringtoneName)
   {
      ETG_TRACE_USR4(("PmSettings::getRingtoneName for RingtoneId: %d", ringtoneId));

      PmResult pmResult(PM_RESULT_ERR_GENERAL, "");

      if((RINGTONE_ID_DEFAULT == ringtoneId) || (RINGTONE_ID_SYSTEM_MAX == ringtoneId))
      {
         ringtoneId = RINGTONE_ID_SYSTEM_DEFAULT;
      }

      for(auto it : _systemWideRingtonesList._systemRingtonesList)
      {
         if(ringtoneId == it._ringtoneId)
         {
            ringtoneName = it._ringtoneName;
            pmResult._pmResultCode = PM_RESULT_OK;
            break;
         }
      }

      ETG_TRACE_USR4(("PmSettings::getRingtoneName: %s", ringtoneName.c_str()));

      return pmResult;
   }

   PmResult PmSettings::getSuppressRingtoneSetting(IN const BdAddress& deviceAddress,
         OUT SuppressRingtoneState& suppressRingtoneState)
   {
      PmResult pmResult(PM_RESULT_ERR_GENERAL, "");

      if(true == DbHandler::getInstance().getSuppressRingtoneSetting(deviceAddress, suppressRingtoneState))
      {
         pmResult._pmResultCode = PM_RESULT_OK;
      }

      return pmResult;
   }

   PmResult PmSettings::getAutoWaitingModeSetting(IN const BdAddress& deviceAddress,
         OUT AutoWaitingModeState& autoWaitingModeState)
   {
      PmResult pmResult(PM_RESULT_ERR_GENERAL, "");

      if(true == DbHandler::getInstance().getAutoWaitingModeSetting(deviceAddress, autoWaitingModeState))
      {
         pmResult._pmResultCode = PM_RESULT_OK;
      }

      return pmResult;
   }

   void PmSettings::ReadSystemWideRingtonesFromDir()
   {
      ETG_TRACE_USR4(("ReadSystemWideRingtonesFromDir() entered"));

      //Get the available ringtones in the system path ("/var/opt/bosch/static/connectivity/tones") if not done already
      DIR* dirp = nullptr;

      dirp = opendir(PM_RINGTONES_DIR);

      if (nullptr != dirp)
      {
         struct dirent* directory;
         RingtoneId ringtoneId = RINGTONE_ID_SYSTEM_DEFAULT;

         while ((directory = readdir(dirp)) != nullptr)
         {
            std::string fileName(directory->d_name);

            ETG_TRACE_USR4(("ReadSystemWideRingtonesFromDir() Searched file/directory in the given path: %s",
                  fileName.c_str()));

            if((true != com::bosch::pmcommon::isDirectory(fileName)) && (std::string::npos != fileName.find(".mp3")))
            {
               ETG_TRACE_USR4(("ReadSystemWideRingtonesFromDir() Ringtone found in the given path: %s",
                     fileName.c_str()));

               SystemRingtones systemRingtones;
               systemRingtones._ringtoneId = ringtoneId++;
               systemRingtones._ringtoneName = PM_RINGTONES_DIR "/" + fileName; // full file name including path

               ETG_TRACE_USR4(("ReadSystemWideRingtonesFromDir() assigned RingtoneId: %d",
                     systemRingtones._ringtoneId));
               ETG_TRACE_USR4(("ReadSystemWideRingtonesFromDir() received RingtoneName: %s",
                     systemRingtones._ringtoneName.c_str()));

               try
               {
                  _systemWideRingtonesList._systemRingtonesList.push_back(systemRingtones);
               }
               catch(...)
               {
                  closedir(dirp);
                  return;
               }
            }
         }

         closedir(dirp);
      }
   }

   void PmSettings::updateSystemWideRingtoneList()
   {
      ETG_TRACE_USR4(("updateSystemWideRingtoneList() entered"));

      //Prepare SystemWideRingtones List
      //Get the active System Wide Ringtone Id from Database
      RingtoneId ringtoneId = RINGTONE_ID_DEFAULT;

      if(true == DbHandler::getInstance().getRingtoneId(DEVICE_ADDRESS_ALL, ringtoneId))
      {
         if(RINGTONE_ID_DEFAULT == ringtoneId)
         {
            ringtoneId = RINGTONE_ID_INBAND;
         }
         _systemWideRingtonesList._activeSystemRingtoneId = ringtoneId;

         ETG_TRACE_USR4(("updateSystemWideRingtoneList() ActiveSystemRingtoneId : %d",
               _systemWideRingtonesList._activeSystemRingtoneId));

         PmCoreMainController::getInstance().getPmCoreCallbackIfWrapper().doOnSystemWideRingtonesListChanged(
               _systemWideRingtonesList);
      }
      else
      {
         ETG_TRACE_ERR(("updateSystemWideRingtoneList() getting ActiveSystemRingtoneId failed"));
      }
   }

   void PmSettings::updateDeviceRingtoneList()
   {
      ETG_TRACE_USR4(("updateDeviceRingtoneList() entered"));

      DeviceRingtoneListType deviceRingtoneList;

      if(true == DbHandler::getInstance().getDeviceRingtoneIdList(deviceRingtoneList))
      {
         ETG_TRACE_USR4(("updateDeviceRingtoneList() getting update from database success"));

         //clear the list first and then update to the latest
         _deviceRingtoneList._deviceRingtoneList.clear();
         _deviceRingtoneList._deviceRingtoneList = deviceRingtoneList;

         // Update Ringtone names
         for(auto& deviceRingtone : _deviceRingtoneList._deviceRingtoneList)
         {
            deviceRingtone._inbandRingingSupport = NOT_SUPPORTED;

            if(PmCoreMainController::getInstance().getCallController().isInbandRinging(deviceRingtone._deviceAddress))
            {
               ETG_TRACE_USR4(("updateDeviceRingtoneList() device (%s) supports InBand Ringing",
                     deviceRingtone._deviceAddress.c_str()));

               if(RINGTONE_ID_DEFAULT == deviceRingtone._ringtoneId)
               {
                  deviceRingtone._ringtoneId = RINGTONE_ID_INBAND;
               }
               deviceRingtone._inbandRingingSupport = SUPPORTED_AND_ENABLED;
            }
            else
            {
               ETG_TRACE_USR4(("updateDeviceRingtoneList() device (%s) not supports InBand Ringing",
                     deviceRingtone._deviceAddress.c_str()));

               if(RINGTONE_ID_DEFAULT == deviceRingtone._ringtoneId)
               {
                  deviceRingtone._ringtoneId = RINGTONE_ID_SYSTEM_DEFAULT;
               }

               if(true == PmCoreMainController::getInstance().getDeviceInfoHandler().isFeatureSupported(
                     deviceRingtone._deviceAddress, "inband_ringtone"))
               {
            	   deviceRingtone._inbandRingingSupport = SUPPORTED_AND_DISABLED;
               }
            }

            if(RINGTONE_ID_INBAND == deviceRingtone._ringtoneId)
            {
               deviceRingtone._ringtoneName = RINGTONE_NAME_INBAND;
            }
            else
            {
               for(auto& systemRingtones : _systemWideRingtonesList._systemRingtonesList)
               {
                  if(deviceRingtone._ringtoneId == systemRingtones._ringtoneId)
                  {
                     deviceRingtone._ringtoneName = systemRingtones._ringtoneName;
                     break;
                  }
               }
            }

            ETG_TRACE_USR4(("updateDeviceRingtoneList() RingtoneId: %d", deviceRingtone._ringtoneId));
            ETG_TRACE_USR4(("updateDeviceRingtoneList() RingtoneName: %s", deviceRingtone._ringtoneName.c_str()));
            ETG_TRACE_USR4(("updateDeviceRingtoneList() InbandRingingSupport: %u", deviceRingtone._inbandRingingSupport));
         }

         PmCoreMainController::getInstance().getPmCoreCallbackIfWrapper().doOnDeviceRingtoneListChanged(
               _deviceRingtoneList);
      }
      else
      {
         ETG_TRACE_ERR(("updateDeviceRingtoneList() getting update from database failed"));
      }
   }

   void PmSettings::updateSuppressRingtoneOnOffList()
   {
      ETG_TRACE_USR4(("updateSuppressRingtoneOnOffList() entered"));

      SuppressRingtoneOnOffListMap suppressRingtoneOnOffList;

      if(true == DbHandler::getInstance().getSuppressRingtoneSettingList(suppressRingtoneOnOffList))
      {
         ETG_TRACE_USR4(("updateSuppressRingtoneOnOffList() getting update from database success"));

         //clear the list first and then update to the latest
         _suppressRingtoneOnOffList._suppressRingtoneOnOffList.clear();
         _suppressRingtoneOnOffList._suppressRingtoneOnOffList = suppressRingtoneOnOffList;

         PmCoreMainController::getInstance().getPmCoreCallbackIfWrapper().doOnSuppressRingtoneOnOffListChanged(
                     _suppressRingtoneOnOffList);
      }
      else
      {
         ETG_TRACE_ERR(("updateSuppressRingtoneOnOffList() getting update from database failed"));
      }
   }

   void PmSettings::updateAutoWaitingModeOnOffList()
   {
      ETG_TRACE_USR4(("updateAutoWaitingModeOnOffList() entered"));

      AutoWaitingModeOnOffListMap autoWaitingModeOnOffList;

      if(true == DbHandler::getInstance().getAutoWaitingModeSettingList(autoWaitingModeOnOffList))
      {
         ETG_TRACE_USR4(("updateAutoWaitingModeOnOffList() getting update from database success"));

         //clear the list first and then update to the latest
         _autoWaitingModeOnOffList._autoWaitingModeOnOffList.clear();
         _autoWaitingModeOnOffList._autoWaitingModeOnOffList = autoWaitingModeOnOffList;

         PmCoreMainController::getInstance().getPmCoreCallbackIfWrapper().doOnAutoWaitingModeOnOffListChanged(
               _autoWaitingModeOnOffList);
      }
      else
      {
         ETG_TRACE_ERR(("updateAutoWaitingModeOnOffList() getting update from database failed"));
      }
   }
}
