/**
 * @file PmSettings.h
 *
 * @swcomponent PhoneCallManager
 *
 * @brief This file contains the definition of the PmSettings class
 *
 * @copyright (C) 2016 Robert Bosch GmbH.
 *            The reproduction, distribution and utilization of this file as
 *            well as the communication of its contents to others without express
 *            authorization is prohibited. Offenders will be held liable for the
 *            payment of damages. All rights reserved in the event of the grant
 *            of a patent, utility model or design.
 *
 * @details This file handles the pm related settings and updating the same
 *          to clients if any changes in the same
 *
 * @ingroup PmCore
 */

#ifndef PmSettings_h
#define PmSettings_h

#include "PmCoreIfTypes.h"
#include "PropertyUpdateHandler.h"

namespace pmcore
{
   class PmSettings : public PropertyUpdateHandler
   {
   public:
      /**
       * Constructor of PmSettings class
       *
       * @param[in]
       * @param[out]
       * @param[in,out]
       *
       * @return
       */
      PmSettings();

      /**
       * Destructor of PmSettings class
       *
       * @param[in]
       * @param[out]
       * @param[in,out]
       *
       * @return
       */
      ~PmSettings();

      /**
       * This method is used to subscribe interested properties in BT stack event notifier.
       *
       * @param[in]
       * @param[out]
       * @param[in,out]
       *
       * @return void
       */
      void subscribeToBtStackEventNotifier();

      /**
       * This method is used to inform the device paired
       *
       * @param[in] deviceHandle - device handle
       * @param[in] deviceAddress - BT address of the device
       * @param[out]
       * @param[in,out]
       *
       * @return void
       */
      void onDevicePaired(IN const DeviceHandle deviceHandle, IN const BdAddress& deviceAddress);

      /**
       * This method is used to inform the device deletion
       *
       * @param[in] deviceAddress - BT address of the device
       * @param[out]
       * @param[in,out]
       *
       * @return void
       */
      void onDeviceDeleted(IN const BdAddress& deviceAddress);

      /**
       * This method is used to get property changes update from BT Stack
       * This method is overridden from PropertyUpdateHandler class
       *
       * @param[in] propertyId      - Id of the property or event
       * @param[in] propertyDetails - Details of the property or event
       * @param[out]
       * @param[in,out]
       *
       * @return void
       */
      virtual void onPropertyUpdate(IN const PmCorePropertyAndEventId propertyId,
            IN std::shared_ptr<void> propertyDetails) override;

      /**
       * This method is used to load the system wide and device specific settings from database
       *
       * @param[in]
       * @param[out]
       * @param[in,out]
       *
       * @return void
       */
      void loadSettingsFromDb();

      /**
       * This method is used to close the database when system shutdown happens
       *
       * @param[in]
       * @param[out]
       * @param[in,out]
       *
       * @return void
       */
      void saveSettingsToDb();

      /**
       * This method is used to restore the settings (both system wide and device specific) to default
       *
       * @param[in] deviceAddress - BT address of the device
       * @param[in] act - acknowledgment token
       * @param[out]
       * @param[in,out]
       *
       * @return PmResult
       */
      PmResult restoreDefaultSetting(IN const BdAddress& deviceAddress, IN const ActType act);

      /**
       * This method is used to set the ringtone for both system wide and device specific settings
       *
       * @param[in] deviceAddress - BT address of the device
       * @param[in] ringtoneId - ringtone id
       * @param[in] act - acknowledgment token
       * @param[out]
       * @param[in,out]
       *
       * @return PmResult
       */
      PmResult setRingtone(IN const BdAddress& deviceAddress, IN const RingtoneId ringtoneId, IN const ActType act);

      /**
       * This method is used to get the system wide ringtones list
       *
       * @param[in] act - acknowledgment token
       * @param[out]
       * @param[in,out]
       *
       * @return void
       */
      void getSystemRingtonesList(IN const ActType act);

      /**
       * This method is used to get the device ringtone list
       *
       * @param[in] act - acknowledgment token
       * @param[out]
       * @param[in,out]
       *
       * @return void
       */
      void getDeviceRingtoneList(IN const ActType act);

      /**
       * This method is used to set the suppress ringtone for both system wide and device specific settings
       *
       * @param[in] deviceAddress - BT address of the device
       * @param[in] suppressRingtoneState - suppress ringtone state
       * @param[in] act - acknowledgment token
       * @param[out]
       * @param[in,out]
       *
       * @return PmResult
       */
      PmResult suppressRingtoneOnOff(IN const BdAddress& deviceAddress,
            IN SuppressRingtoneState suppressRingtoneState, IN const ActType act);

      /**
       * This method is used to get the suppress ringtone on off list
       *
       * @param[in] act - acknowledgment token
       * @param[out]
       * @param[in,out]
       *
       * @return void
       */
      void getSuppressRingtoneOnOffList(IN const ActType act);

      /**
       * This method is used to set the auto waiting mode for both system wide and device specific settings
       *
       * @param[in] deviceAddress - BT address of the device
       * @param[in] autoWaitingModeState - auto waiting mode state
       * @param[in] act - acknowledgment token
       * @param[out]
       * @param[in,out]
       *
       * @return PmResult
       */
      PmResult autoWaitingModeOnOff(IN const BdAddress& deviceAddress,
            IN AutoWaitingModeState autoWaitingModeState, IN const ActType act);

      /**
       * This method is used to get the auto waiting mode on off list
       *
       * @param[in] act - acknowledgment token
       * @param[out]
       * @param[in,out]
       *
       * @return void
       */
      void getAutoWaitingModeOnOffList(IN const ActType act);

      /**
       * This method is used to get the ringtone name for given device address
       *
       * @param[in] deviceAddress - BT address of the device
       * @param[out] ringtoneName - ringtone name
       * @param[in,out]
       *
       * @return PmResult
       */
      PmResult getRingtoneName(IN const BdAddress& deviceAddress, OUT RingtoneName& ringtoneName);

      /**
       * This method is used to get the ringtone name for given ringtone id
       *
       * @param[in] deviceAddress - BT address of the device
       * @param[out] ringtoneId - ringtone id
       * @param[in,out]
       *
       * @return PmResult
       */
      PmResult getRingtoneName(IN RingtoneId ringtoneId, OUT RingtoneName& ringtoneName);

      /**
       * This method is used to get the suppress ringtone setting for given device address
       *
       * @param[in] deviceAddress - BT address of the device
       * @param[out] suppressRingtoneState - suppress ringtone state
       * @param[in,out]
       *
       * @return PmResult
       */
      PmResult getSuppressRingtoneSetting(IN const BdAddress& deviceAddress,
            OUT SuppressRingtoneState& suppressRingtoneState);

      /**
       * This method is used to get the auto waiting mode setting for given device address
       *
       * @param[in] deviceAddress - BT address of the device
       * @param[out] autoWaitingModeState - auto waiting mode state
       * @param[in,out]
       *
       * @return PmResult
       */
      PmResult getAutoWaitingModeSetting(IN const BdAddress& deviceAddress,
            OUT AutoWaitingModeState& autoWaitingModeState);

   private:
      /**
       * This method is used to read the system wide ringtones from directory
       *
       * @param[in]
       * @param[out]
       * @param[in,out]
       *
       * @return void
       */
      void ReadSystemWideRingtonesFromDir();

      /**
       * This method is used to update the system wide ringtones list
       *
       * @param[in]
       * @param[out]
       * @param[in,out]
       *
       * @return void
       */
      void updateSystemWideRingtoneList();

      /**
       * This method is used to update the device ringtone list
       *
       * @param[in]
       * @param[out]
       * @param[in,out]
       *
       * @return void
       */
      void updateDeviceRingtoneList();

      /**
       * This method is used to update the suppress ringtone on off list
       *
       * @param[in]
       * @param[out]
       * @param[in,out]
       *
       * @return void
       */
      void updateSuppressRingtoneOnOffList();

      /**
       * This method is used to update the auto waiting mode on off list
       *
       * @param[in]
       * @param[out]
       * @param[in,out]
       *
       * @return void
       */
      void updateAutoWaitingModeOnOffList();

      PropertyIdList             _propertyIdList;            /**< List of property Id's */
      SystemWideRingtonesList    _systemWideRingtonesList;   /**< List of System Wide Ringtones */
      DeviceRingtoneList         _deviceRingtoneList;        /**< List of Device Ringtone */
      SuppressRingtoneOnOffList  _suppressRingtoneOnOffList; /**< Suppress Ringtone OnOff List */
      AutoWaitingModeOnOffList   _autoWaitingModeOnOffList;  /**< Auto Waiting Mode OnOff List */
   };

} // namespace pmcore

#endif // PmSettings_h
