/**
 * @file PmCoreMainController.cpp
 *
 * @swcomponent PhoneCallManager
 *
 * @brief This file contains the definition of the PmCoreMainController class
 *
 * @copyright (C) 2016 Robert Bosch GmbH.
 *            The reproduction, distribution and utilization of this file as
 *            well as the communication of its contents to others without express
 *            authorization is prohibited. Offenders will be held liable for the
 *            payment of damages. All rights reserved in the event of the grant
 *            of a patent, utility model or design.
 *
 * @details This file handles the PmCoreIfMessages from all RequestIf interfaces.
 *
 * @ingroup PmCore
 */

#include "PmCoreMainController.h"
#include "PmCoreIfMessage.h"
#include "PmCoreIfMessageRequest.h"
#include "PmCoreIfMessageResult.h"
#include "PmCoreIfMessageCreator.h"
#include "FwBluetoothStringUtils.h"
#include "PropertyUpdateNotifierToCore.h"
#include "PropertyDetails.h"
#include "EvoBtStackWrapper.h"
#include "PmAppTrace.h"

#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#define ETG_DEFAULT_TRACE_CLASS TR_CLASS_PM_CORE
#ifdef VARIANT_S_FTR_ENABLE_FW_ETG_USAGE
#include "trcGenProj/Header/PmCoreMainController.cpp.trc.h"
#define ETG_DEFAULT_TRACE_CLASS TR_CLASS_PM_CORE
#endif
#endif

namespace pmcore
{
   PmCoreMainController::PmCoreMainController():
         _pmCoreCallbackIfWrapper(nullptr)
   {
      ETG_TRACE_USR1(("PmCoreMainController"));
   }

   PmCoreMainController::~PmCoreMainController()
   {
      ETG_TRACE_USR1(("~PmCoreMainController"));

      if (nullptr != _pmCoreCallbackIfWrapper)
      {
         delete _pmCoreCallbackIfWrapper;
         _pmCoreCallbackIfWrapper = nullptr;
      }
   }

   PmResult PmCoreMainController::registerCallback(IPmCoreCallbackIf* pmCoreCallbackIf)
   {
      ETG_TRACE_USR1(("registerCallback: pmCoreCallbackIf = 0x%p", (void *) pmCoreCallbackIf));
      PmResult pmResult(PM_RESULT_OK, "");

      if(nullptr != pmCoreCallbackIf)
      {
         _pmCoreCallbackIfWrapper = new PmCoreCallbackIfWrapper(pmCoreCallbackIf);

         if(nullptr == _pmCoreCallbackIfWrapper)
         {
            ETG_TRACE_FATAL(("registerCallback: _pmCoreCallbackIfWrapper is 0"));
            pmResult._pmResultCode = PM_RESULT_ERR_GENERAL;
         }
         else
         {
            ETG_TRACE_USR1(("registering PmCoreCallbackIfWrapper: _pmCoreCallbackIfWrapper = 0x%p",
                  (void *) _pmCoreCallbackIfWrapper));
         }
      }
      else
      {
         ETG_TRACE_FATAL(("registerCallback: pmCoreCallbackIf is 0"));
         pmResult._pmResultCode = PM_RESULT_ERR_GENERAL;
      }

      return pmResult;
   }

   PmCoreCallbackIfWrapper& PmCoreMainController::getPmCoreCallbackIfWrapper(void)
   {
      return *(_pmCoreCallbackIfWrapper);
   }

   PmSettings& PmCoreMainController::getPmSettings(void)
   {
      return _pmSettings;
   }

   DeviceInfoHandler& PmCoreMainController::getDeviceInfoHandler(void)
   {
      return _deviceInfoHandler;
   }

   CallController& PmCoreMainController::getCallController(void)
   {
      return _callController;
   }

   VRController& PmCoreMainController::getVRController(void)
   {
      return _vrController;
   }

   RingtoneController& PmCoreMainController::getRingtoneController(void)
   {
      return _ringtoneController;
   }

   RVController& PmCoreMainController::getRVController(void)
   {
      return _rVController;
   }

   void PmCoreMainController::handlePmCoreIfMessage(IN std::shared_ptr<PmCoreIfMessage>& pmCoreIfMessage)
   {
      ETG_TRACE_USR1(("handlePmCoreIfMessage: pmCoreIfMessage = 0x%p", pmCoreIfMessage.get()));

      if (nullptr == pmCoreIfMessage)
      {
         ETG_TRACE_FATAL(("handlePmCoreIfMessage: pmCoreIfMessage is null"));
         return;
      }

      PmCoreIfMsgId msgId = pmCoreIfMessage->getMessageId();

      switch(msgId)
      {
         case PM_CORE_IF_MSG_ID_SET_SYSTEM_STATE:
            (void) handlePmCoreIfMessage_SetSystemState(
                  std::static_pointer_cast<PmCoreIfMessage_SetSystemState>(pmCoreIfMessage));
            break;
         case PM_CORE_IF_MSG_ID_SET_VEHICLE_CONFIGURATION:
            (void) handlePmCoreIfMessage_SetVehicleConfiguration(
                  std::static_pointer_cast<PmCoreIfMessage_SetVehicleConfiguration>(pmCoreIfMessage));
            break;
         case PM_CORE_IF_MSG_ID_ON_DEVICE_PAIRED:
            (void) handlePmCoreIfMessage_OnDevicePaired(
                  std::static_pointer_cast<PmCoreIfMessage_OnDevicePaired>(pmCoreIfMessage));
            break;
         case PM_CORE_IF_MSG_ID_ON_DEVICE_CONNECTED:
            (void) handlePmCoreIfMessage_OnDeviceConnected(
                  std::static_pointer_cast<PmCoreIfMessage_OnDeviceConnected>(pmCoreIfMessage));
            break;
         case PM_CORE_IF_MSG_ID_ON_DEVICE_DISCONNECTED:
            (void) handlePmCoreIfMessage_OnDeviceDisconnected(
                  std::static_pointer_cast<PmCoreIfMessage_OnDeviceDisconnected>(pmCoreIfMessage));
            break;
         case PM_CORE_IF_MSG_ID_ON_DEVICE_DELETED:
            (void) handlePmCoreIfMessage_OnDeviceDeleted(
                  std::static_pointer_cast<PmCoreIfMessage_OnDeviceDeleted>(pmCoreIfMessage));
            break;
         case PM_CORE_IF_MSG_ID_GET_SYSTEM_WIDE_RINGTONE_LIST_REQUEST:
            (void) handlePmCoreIfMessage_GetSystemWideRingtoneListRequest(
                  std::static_pointer_cast<PmCoreIfMessage_GetSystemWideRingtoneListRequest>(pmCoreIfMessage));
            break;
         case PM_CORE_IF_MSG_ID_GET_DEVICE_RINGTONE_LIST_REQUEST:
            (void) handlePmCoreIfMessage_GetDeviceRingtoneListRequest(
                  std::static_pointer_cast<PmCoreIfMessage_GetDeviceRingtoneListRequest>(pmCoreIfMessage));
            break;
         case PM_CORE_IF_MSG_ID_SET_RINGTONE_REQUEST:
            (void) handlePmCoreIfMessage_SetRingtoneRequest(
                  std::static_pointer_cast<PmCoreIfMessage_SetRingtoneRequest>(pmCoreIfMessage));
            break;
         case PM_CORE_IF_MSG_ID_PLAY_RINGTONE_REQUEST:
            (void) handlePmCoreIfMessage_PlayRingtoneRequest(
                  std::static_pointer_cast<PmCoreIfMessage_PlayRingtoneRequest>(pmCoreIfMessage));
            break;
         case PM_CORE_IF_MSG_ID_STOP_RINGTONE_REQUEST:
            (void) handlePmCoreIfMessage_StopRingtoneRequest(
                  std::static_pointer_cast<PmCoreIfMessage_StopRingtoneRequest>(pmCoreIfMessage));
            break;
         case PM_CORE_IF_MSG_ID_GET_SUPPRESS_RINGTONE_ON_OFF_LIST_REQUEST:
            (void) handlePmCoreIfMessage_GetSuppressRingtoneOnOffListRequest(
                  std::static_pointer_cast<PmCoreIfMessage_GetSuppressRingtoneOnOffListRequest>(pmCoreIfMessage));
            break;
         case PM_CORE_IF_MSG_ID_SUPPRESS_RINGTONE_ON_OFF_REQUEST:
            (void) handlePmCoreIfMessage_SuppressRingtoneOnOffRequest(
                  std::static_pointer_cast<PmCoreIfMessage_SuppressRingtoneOnOffRequest>(pmCoreIfMessage));
            break;
         case PM_CORE_IF_MSG_ID_GET_AUTO_WAITING_MODE_ON_OFF_LIST_REQUEST:
            (void) handlePmCoreIfMessage_GetAutoWaitingModeOnOffListRequest(
                  std::static_pointer_cast<PmCoreIfMessage_GetAutoWaitingModeOnOffListRequest>(pmCoreIfMessage));
            break;
         case PM_CORE_IF_MSG_ID_AUTO_WAITING_MODE_ON_OFF_REQUEST:
            (void) handlePmCoreIfMessage_AutoWaitingModeOnOffRequest(
                  std::static_pointer_cast<PmCoreIfMessage_AutoWaitingModeOnOffRequest>(pmCoreIfMessage));
            break;
         case PM_CORE_IF_MSG_ID_SET_WAITING_MODE_FILE_PATH_REQUEST:
            (void) handlePmCoreIfMessage_SetWaitingModeFilePathRequest(
                  std::static_pointer_cast<PmCoreIfMessage_SetWaitingModeFilePathRequest>(pmCoreIfMessage));
            break;
         case PM_CORE_IF_MSG_ID_GET_WAITING_MODE_FILE_PATH_REQUEST:
            (void) handlePmCoreIfMessage_GetWaitingModeFilePathRequest(
                  std::static_pointer_cast<PmCoreIfMessage_GetWaitingModeFilePathRequest>(pmCoreIfMessage));
            break;
         case PM_CORE_IF_MSG_ID_GET_PHONE_CALL_AUDIO_ACTIVE_REQUEST:
            (void) handlePmCoreIfMessage_GetPhoneCallAudioActiveRequest(
                  std::static_pointer_cast<PmCoreIfMessage_GetPhoneCallAudioActiveRequest>(pmCoreIfMessage));
            break;
         case PM_CORE_IF_MSG_ID_GET_ACTIVE_PASSIVE_DEVICE_LIST_REQUEST:
            (void) handlePmCoreIfMessage_GetActivePassiveDeviceListRequest(
                  std::static_pointer_cast<PmCoreIfMessage_GetActivePassiveDeviceListRequest>(pmCoreIfMessage));
            break;
         case PM_CORE_IF_MSG_ID_SET_ACTIVE_PHONE_REQUEST:
            (void) handlePmCoreIfMessage_SetActivePhoneRequest(
                  std::static_pointer_cast<PmCoreIfMessage_SetActivePhoneRequest>(pmCoreIfMessage));
            break;
         case PM_CORE_IF_MSG_ID_RESTORE_DEFAULT_SETTING_REQUEST:
            (void) handlePmCoreIfMessage_RestoreDefaultSettingRequest(
                  std::static_pointer_cast<PmCoreIfMessage_RestoreDefaultSettingRequest>(pmCoreIfMessage));
            break;
         case PM_CORE_IF_MSG_ID_GET_SUPPORTED_FEATURES_REQUEST:
            (void) handlePmCoreIfMessage_GetSupportedFeaturesRequest(
                  std::static_pointer_cast<PmCoreIfMessage_GetSupportedFeaturesRequest>(pmCoreIfMessage));
            break;
         case PM_CORE_IF_MSG_ID_GET_NETWORK_STATUS_REQUEST:
            (void) handlePmCoreIfMessage_GetNetworkStatusRequest(
                  std::static_pointer_cast<PmCoreIfMessage_GetNetworkStatusRequest>(pmCoreIfMessage));
            break;
         case PM_CORE_IF_MSG_ID_GET_SIGNAL_STRENGTH_REQUEST:
            (void) handlePmCoreIfMessage_GetSignalStrengthRequest(
                  std::static_pointer_cast<PmCoreIfMessage_GetSignalStrengthRequest>(pmCoreIfMessage));
            break;
         case PM_CORE_IF_MSG_ID_GET_BATTERY_CHARGE_LEVEL_REQUEST:
            (void) handlePmCoreIfMessage_GetBatteryChargeLevelRequest(
                  std::static_pointer_cast<PmCoreIfMessage_GetBatteryChargeLevelRequest>(pmCoreIfMessage));
            break;
         case PM_CORE_IF_MSG_ID_GET_NETWORK_OPERATOR_REQUEST:
            (void) handlePmCoreIfMessage_GetNetworkOperatorRequest(
                  std::static_pointer_cast<PmCoreIfMessage_GetNetworkOperatorRequest>(pmCoreIfMessage));
            break;
         case PM_CORE_IF_MSG_ID_VOICE_MAIL_LIST_REQUEST:
            (void) handlePmCoreIfMessage_VoiceMailListRequest(
                  std::static_pointer_cast<PmCoreIfMessage_VoiceMailListRequest>(pmCoreIfMessage));
            break;
         case PM_CORE_IF_MSG_ID_GET_CALL_STATUS_LIST_REQUEST:
            (void) handlePmCoreIfMessage_GetCallStatusListRequest(
                  std::static_pointer_cast<PmCoreIfMessage_GetCallStatusListRequest>(pmCoreIfMessage));
            break;
         case PM_CORE_IF_MSG_ID_GET_MICROPHONE_MUTE_STATE_REQUEST:
            (void) handlePmCoreIfMessage_GetMicrophoneMuteStateRequest(
                  std::static_pointer_cast<PmCoreIfMessage_GetMicrophoneMuteStateRequest>(pmCoreIfMessage));
            break;
         case PM_CORE_IF_MSG_ID_SET_MICROPHONE_MUTE_STATE_REQUEST:
            (void) handlePmCoreIfMessage_SetMicrophoneMuteStateRequest(
                  std::static_pointer_cast<PmCoreIfMessage_SetMicrophoneMuteStateRequest>(pmCoreIfMessage));
            break;
         case PM_CORE_IF_MSG_ID_MERGE_CALLS_REQUEST:
            (void) handlePmCoreIfMessage_MergeCallsRequest(
                  std::static_pointer_cast<PmCoreIfMessage_MergeCallsRequest>(pmCoreIfMessage));
            break;
         case PM_CORE_IF_MSG_ID_SPLIT_CALLS_REQUEST:
            (void) handlePmCoreIfMessage_SplitCallsRequest(
                  std::static_pointer_cast<PmCoreIfMessage_SplitCallsRequest>(pmCoreIfMessage));
            break;
         case PM_CORE_IF_MSG_ID_HANGUP_CALLS_REQUEST:
            (void) handlePmCoreIfMessage_HangupCallsRequest(
                  std::static_pointer_cast<PmCoreIfMessage_HangupCallsRequest>(pmCoreIfMessage));
            break;
         case PM_CORE_IF_MSG_ID_REDIAL_REQUEST:
            (void) handlePmCoreIfMessage_RedialRequest(
                  std::static_pointer_cast<PmCoreIfMessage_RedialRequest>(pmCoreIfMessage));
            break;
         case PM_CORE_IF_MSG_ID_GET_SCO_CONNECTION_REQUEST:
            (void) handlePmCoreIfMessage_GetSCOConnectionRequest(
                  std::static_pointer_cast<PmCoreIfMessage_GetSCOConnectionRequest>(pmCoreIfMessage));
            break;
         case PM_CORE_IF_MSG_ID_GET_RINGTONE_MUTE_STATE_REQUEST:
            (void) handlePmCoreIfMessage_GetRingtoneMuteStateRequest(
                  std::static_pointer_cast<PmCoreIfMessage_GetRingtoneMuteStateRequest>(pmCoreIfMessage));
            break;
         case PM_CORE_IF_MSG_ID_SET_RINGTONE_MUTE_STATE_REQUEST:
            (void) handlePmCoreIfMessage_SetRingtoneMuteStateRequest(
                  std::static_pointer_cast<PmCoreIfMessage_SetRingtoneMuteStateRequest>(pmCoreIfMessage));
            break;
         case PM_CORE_IF_MSG_ID_GET_WAITING_MODE_STATE_REQUEST:
            (void) handlePmCoreIfMessage_GetWaitingModeStateRequest(
                  std::static_pointer_cast<PmCoreIfMessage_GetWaitingModeStateRequest>(pmCoreIfMessage));
            break;
         case PM_CORE_IF_MSG_ID_GET_RINGTONE_PLAYBACK_STATUS_REQUEST:
            (void) handlePmCoreIfMessage_GetRingtonePlaybackStatusRequest(
                  std::static_pointer_cast<PmCoreIfMessage_GetRingtonePlaybackStatusRequest>(pmCoreIfMessage));
            break;
         case PM_CORE_IF_MSG_ID_START_STOP_WAITING_MODE_REQUEST:
            (void) handlePmCoreIfMessage_StartStopWaitingModeRequest(
                  std::static_pointer_cast<PmCoreIfMessage_StartStopWaitingModeRequest>(pmCoreIfMessage));
            break;
         case PM_CORE_IF_MSG_ID_DIAL_REQUEST:
            (void) handlePmCoreIfMessage_DialRequest(
                  std::static_pointer_cast<PmCoreIfMessage_DialRequest>(pmCoreIfMessage));
            break;
         case PM_CORE_IF_MSG_ID_ACCEPT_CALL_REQUEST:
            (void) handlePmCoreIfMessage_AcceptCallRequest(
                  std::static_pointer_cast<PmCoreIfMessage_AcceptCallRequest>(pmCoreIfMessage));
            break;
         case PM_CORE_IF_MSG_ID_SWAP_CALL_REQUEST:
            (void) handlePmCoreIfMessage_SwapCallRequest(
                  std::static_pointer_cast<PmCoreIfMessage_SwapCallRequest>(pmCoreIfMessage));
            break;
         case PM_CORE_IF_MSG_ID_SPEED_DIAL_REQUEST:
            (void) handlePmCoreIfMessage_SpeedDialRequest(
                  std::static_pointer_cast<PmCoreIfMessage_SpeedDialRequest>(pmCoreIfMessage));
            break;
         case PM_CORE_IF_MSG_ID_SEND_DTMF_REQUEST:
            (void) handlePmCoreIfMessage_SendDTMFRequest(
                  std::static_pointer_cast<PmCoreIfMessage_SendDTMFRequest>(pmCoreIfMessage));
            break;
         case PM_CORE_IF_MSG_ID_TRANSFER_AUDIO_REQUEST:
            (void) handlePmCoreIfMessage_TransferAudioRequest(
                  std::static_pointer_cast<PmCoreIfMessage_TransferAudioRequest>(pmCoreIfMessage));
            break;
         case PM_CORE_IF_MSG_ID_GET_VOICE_RECOGNITION_STATUS_REQUEST:
            (void) handlePmCoreIfMessage_GetVoiceRecognitionStatusRequest(
                  std::static_pointer_cast<PmCoreIfMessage_GetVoiceRecognitionStatusRequest>(pmCoreIfMessage));
            break;
         case PM_CORE_IF_MSG_ID_GET_EXTERNAL_VOICE_RECOGNITION_STATUS_REQUEST:
            (void) handlePmCoreIfMessage_GetExtVoiceRecognitionStatusRequest(
                  std::static_pointer_cast<PmCoreIfMessage_GetExtVoiceRecognitionStatusRequest>(pmCoreIfMessage));
            break;
         case PM_CORE_IF_MSG_ID_START_STOP_VOICE_RECOGNITION_REQUEST:
            (void) handlePmCoreIfMessage_StartStopVoiceRecognitionRequest(
                  std::static_pointer_cast<PmCoreIfMessage_StartStopVoiceRecognitionRequest>(pmCoreIfMessage));
            break;
         case PM_CORE_IF_MSG_ID_START_STOP_EXTERNAL_VOICE_RECOGNITION_REQUEST:
            (void) handlePmCoreIfMessage_StartStopExternalVoiceRecognitionRequest(
                  std::static_pointer_cast<PmCoreIfMessage_StartStopExternalVoiceRecognitionRequest>(pmCoreIfMessage));
            break;
         case PM_CORE_IF_MSG_ID_GET_ENHANCED_VOICE_RECOGNITION_FEATURE_REQUEST:
            (void) handlePmCoreIfMessage_GetEnhancedVoiceRecognitionFeatureRequest(
                  std::static_pointer_cast<PmCoreIfMessage_GetEnhancedVoiceRecognitionFeatureRequest>(
                        pmCoreIfMessage));
            break;
         case PM_CORE_IF_MSG_ID_STOP_EXTENDED_VOICE_RECOGNITION_REQUEST:
            (void) handlePmCoreIfMessage_stopExtendedVRRequest(
                  std::static_pointer_cast<PmCoreIfMessage_stopExtendedVRRequest>(pmCoreIfMessage));
            break;
         case PM_CORE_IF_MSG_ID_REQUEST_PHONE_NUMBER_RESULT:
            (void) handlePmCoreIfMessage_RequestPhoneNumberResult(
                  std::static_pointer_cast<PmCoreIfMessage_RequestPhoneNumberResult>(pmCoreIfMessage));
            break;
         case PM_CORE_IF_MSG_ID_GET_SUBSCRIBER_RESULT:
            (void) handlePmCoreIfMessage_GetSubscriberResult(
                  std::static_pointer_cast<PmCoreIfMessage_GetSubscriberResult>(pmCoreIfMessage));
            break;
         case PM_CORE_IF_MSG_ID_SEND_CUSTOM_AT_COMMAND_RESULT:
            (void) handlePmCoreIfMessage_SendCustomATCommandResult(
                  std::static_pointer_cast<PmCoreIfMessage_SendCustomATCommandResult>(pmCoreIfMessage));
            break;
         case PM_CORE_IF_MSG_ID_HF_UPDATE_INDICATORS_VALUE_RESULT:
            (void) handlePmCoreIfMessage_HFUpdateIndicatorsValueResult(
                  std::static_pointer_cast<PmCoreIfMessage_HFUpdateIndicatorsValueResult>(pmCoreIfMessage));
            break;
         case PM_CORE_IF_MSG_ID_GET_MODEMS_RESULT:
            (void) handlePmCoreIfMessage_GetModemsResult(
                  std::static_pointer_cast<PmCoreIfMessage_GetModemsResult>(pmCoreIfMessage));
            break;
         case PM_CORE_IF_MSG_ID_ACCEPT_SCO_CONNECT_RESULT:
            (void) handlePmCoreIfMessage_AcceptSCOConnectResult(
                  std::static_pointer_cast<PmCoreIfMessage_AcceptSCOConnectResult>(pmCoreIfMessage));
            break;
         case PM_CORE_IF_MSG_ID_SIRI_ENABLE_RESULT:
            (void) handlePmCoreIfMessage_SiriEnableResult(
                  std::static_pointer_cast<PmCoreIfMessage_SiriEnableResult>(pmCoreIfMessage));
            break;
         case PM_CORE_IF_MSG_ID_SIRI_SET_NR_RESULT:
            (void) handlePmCoreIfMessage_SiriSetNRResult(
                  std::static_pointer_cast<PmCoreIfMessage_SiriSetNRResult>(pmCoreIfMessage));
            break;
         case PM_CORE_IF_MSG_ID_GET_CALLS_RESULT:
            (void) handlePmCoreIfMessage_GetCallsResult(
                  std::static_pointer_cast<PmCoreIfMessage_GetCallsResult>(pmCoreIfMessage));
            break;
         case PM_CORE_IF_MSG_ID_DIAL_RESULT:
            (void) handlePmCoreIfMessage_DialResult(
                  std::static_pointer_cast<PmCoreIfMessage_DialResult>(pmCoreIfMessage));
            break;
         case PM_CORE_IF_MSG_ID_SPEED_DIAL_RESULT:
            (void) handlePmCoreIfMessage_SpeedDialResult(
                  std::static_pointer_cast<PmCoreIfMessage_SpeedDialResult>(pmCoreIfMessage));
            break;
         case PM_CORE_IF_MSG_ID_REDIAL_RESULT:
            (void) handlePmCoreIfMessage_RedialResult(
                  std::static_pointer_cast<PmCoreIfMessage_RedialResult>(pmCoreIfMessage));
            break;
         case PM_CORE_IF_MSG_ID_SWAP_CALL_RESULT:
            (void) handlePmCoreIfMessage_SwapCallResult(
                  std::static_pointer_cast<PmCoreIfMessage_SwapCallResult>(pmCoreIfMessage));
            break;
         case PM_CORE_IF_MSG_ID_RELEASE_AND_ACCEPT_RESULT:
            (void) handlePmCoreIfMessage_ReleaseAndAcceptResult(
                  std::static_pointer_cast<PmCoreIfMessage_ReleaseAndAcceptResult>(pmCoreIfMessage));
            break;
         case PM_CORE_IF_MSG_ID_RELEASE_AND_SWAP_RESULT:
            (void) handlePmCoreIfMessage_ReleaseAndSwapResult(
                  std::static_pointer_cast<PmCoreIfMessage_ReleaseAndSwapResult>(pmCoreIfMessage));
            break;
         case PM_CORE_IF_MSG_ID_HOLD_AND_ACCEPT_RESULT:
            (void) handlePmCoreIfMessage_HoldAndAcceptResult(
                  std::static_pointer_cast<PmCoreIfMessage_HoldAndAcceptResult>(pmCoreIfMessage));
            break;
         case PM_CORE_IF_MSG_ID_HANGUP_ALL_RESULT:
            (void) handlePmCoreIfMessage_HangupAllResult(
                  std::static_pointer_cast<PmCoreIfMessage_HangupAllResult>(pmCoreIfMessage));
            break;
         case PM_CORE_IF_MSG_ID_SPLIT_CALLS_RESULT:
            (void) handlePmCoreIfMessage_SplitCallsResult(
                  std::static_pointer_cast<PmCoreIfMessage_SplitCallsResult>(pmCoreIfMessage));
            break;
         case PM_CORE_IF_MSG_ID_MERGE_CALLS_RESULT:
            (void) handlePmCoreIfMessage_MergeCallsResult(
                  std::static_pointer_cast<PmCoreIfMessage_MergeCallsResult>(pmCoreIfMessage));
            break;
         case PM_CORE_IF_MSG_ID_HANGUP_MULTIPARTY_RESULT:
            (void) handlePmCoreIfMessage_HangupMultipartyResult(
                  std::static_pointer_cast<PmCoreIfMessage_HangupMultipartyResult>(pmCoreIfMessage));
            break;
         case PM_CORE_IF_MSG_ID_SEND_DTMF_RESULT:
            (void) handlePmCoreIfMessage_SendDTMFResult(
                  std::static_pointer_cast<PmCoreIfMessage_SendDTMFResult>(pmCoreIfMessage));
            break;
         case PM_CORE_IF_MSG_ID_HANGUP_CALL_RESULT:
            (void) handlePmCoreIfMessage_HangupCallResult(
                  std::static_pointer_cast<PmCoreIfMessage_HangupCallResult>(pmCoreIfMessage));
            break;
         case PM_CORE_IF_MSG_ID_ACCEPT_CALL_RESULT:
            (void) handlePmCoreIfMessage_AcceptCallResult(
                  std::static_pointer_cast<PmCoreIfMessage_AcceptCallResult>(pmCoreIfMessage));
            break;
         case PM_CORE_IF_MSG_ID_HOLD_INCOMING_CALL_RESULT:
            (void) handlePmCoreIfMessage_HoldIncomingCallResult(
                  std::static_pointer_cast<PmCoreIfMessage_HoldIncomingCallResult>(pmCoreIfMessage));
            break;
         case PM_CORE_IF_MSG_ID_TRANSFER_AUDIO_RESULT:
            (void) handlePmCoreIfMessage_TransferAudioResult(
                  IN std::static_pointer_cast<PmCoreIfMessage_TransferAudioResult>(pmCoreIfMessage));
            break;
         case PM_CORE_IF_MSG_ID_START_STOP_VOICE_RECOGNITION_RESULT:
            (void) handlePmCoreIfMessage_StartStopVoiceRecognitionResult(
                  IN std::static_pointer_cast<PmCoreIfMessage_StartStopVoiceRecognitionResult>(pmCoreIfMessage));
            break;

            // Messages from PM_Audiomanager
         case PM_CORE_IF_MSG_ID_PREPARE_AUDIO_ROUTE_RESPONSE:
            (void) handlePmCoreIfMessage_PrepareAudioRouteResponse(
                  std::static_pointer_cast<PmCoreIfMessage_PrepareAudioRouteResponse>(pmCoreIfMessage));
            break;
         case PM_CORE_IF_MSG_ID_PLAY_AUDIO_RESPONSE:
            (void) handlePmCoreIfMessage_PlayAudioResponse(
                  std::static_pointer_cast<PmCoreIfMessage_PlayAudioResponse>(pmCoreIfMessage));
            break;
         case PM_CORE_IF_MSG_ID_STOP_AUDIO_RESPONSE:
            (void) handlePmCoreIfMessage_StopAudioResponse(
                  std::static_pointer_cast<PmCoreIfMessage_StopAudioResponse>(pmCoreIfMessage));
            break;
         case PM_CORE_IF_MSG_ID_SET_MIC_MUTE_STATE_RESPONSE:
            (void) handlePmCoreIfMessage_SetMicMuteStateResponse(
                  std::static_pointer_cast<PmCoreIfMessage_SetMicMuteStateResponse>(pmCoreIfMessage));
            break;
         case PM_CORE_IF_MSG_ID_PAUSE_AUDIO_RESPONSE:
            (void) handlePmCoreIfMessage_PauseAudioResponse(
                  std::static_pointer_cast<PmCoreIfMessage_PauseAudioResponse>(pmCoreIfMessage));
            break;

         default:
            ETG_TRACE_ERR(("handlePmCoreIfMessage: PmCoreIfMsgId is not handled"));
            break;
      }
   }

   void PmCoreMainController::handlePmCoreIfMessage_SetSystemState(
         std::shared_ptr<PmCoreIfMessage_SetSystemState> pmCoreIfMessage)
   {
      ETG_TRACE_USR1(("handlePmCoreIfMessage_SetSystemState"));

      switch(pmCoreIfMessage->getSystemState())
      {
         case PM_SPM_SYSTEM_ON:
            _pmSettings.loadSettingsFromDb();
            break;
         case PM_SPM_SYSTEM_STANDBY:
            //Do Nothing
            break;
         case PM_SPM_SYSTEM_OFF:
            _pmSettings.saveSettingsToDb();
            break;
         default:
            ETG_TRACE_ERR(("setSystemState: invalid system state"));
            break;
      }
   }

   void PmCoreMainController::handlePmCoreIfMessage_SetVehicleConfiguration(
         std::shared_ptr<PmCoreIfMessage_SetVehicleConfiguration> pmCoreIfMessage)
   {
      ETG_TRACE_USR1(("handlePmCoreIfMessage_SetVehicleConfiguration"));

      //Sending Set Vehicle Configuration to BtStackWrapper
      evobtstackwrapper::EvoBtStackWrapper::getInstance().setVehicleConfiguration(pmCoreIfMessage->getVehicleConfig());
   }

   void PmCoreMainController::handlePmCoreIfMessage_OnDevicePaired(
         std::shared_ptr<PmCoreIfMessage_OnDevicePaired> pmCoreIfMessage)
   {
      ETG_TRACE_USR1(("handlePmCoreIfMessage_OnDevicePaired"));

      // Process device paired.
      PmCorePropertyAndEventId propertyId(ON_DEVICE_PAIRED);
      std::shared_ptr<PropertyDetails<BasicDeviceDetails>> propertyDetails = std::make_shared
            <PropertyDetails<BasicDeviceDetails>>("", BasicDeviceDetails(pmCoreIfMessage->getPairedDeviceInfo()));

      // Sending notification to interested controllers
      PropertyUpdateNotifierToCore::getInstance().notifyControllers(propertyId, propertyDetails);
   }

   void PmCoreMainController::handlePmCoreIfMessage_OnDeviceConnected(
         std::shared_ptr<PmCoreIfMessage_OnDeviceConnected> pmCoreIfMessage)
   {
      ETG_TRACE_USR1(("handlePmCoreIfMessage_OnDeviceConnected"));

      // Process device connected.
      PmCorePropertyAndEventId propertyId(ON_DEVICE_CONNECTED);
      std::shared_ptr<PropertyDetails<BasicDeviceDetails>> propertyDetails = std::make_shared
            <PropertyDetails<BasicDeviceDetails>>("", BasicDeviceDetails(pmCoreIfMessage->getConnectedDeviceInfo()));

      // Sending notification to interested controllers
      PropertyUpdateNotifierToCore::getInstance().notifyControllers(propertyId, propertyDetails);

      //Sending Device connected to BtStackWrapper
      evobtstackwrapper::EvoBtStackWrapper::getInstance().onDeviceConnected(
            pmCoreIfMessage->getConnectedDeviceInfo()._deviceAddress);
   }

   void PmCoreMainController::handlePmCoreIfMessage_OnDeviceDisconnected(
         std::shared_ptr<PmCoreIfMessage_OnDeviceDisconnected> pmCoreIfMessage)
   {
      ETG_TRACE_USR1(("handlePmCoreIfMessage_OnDeviceDisconnected"));

      // Process device disconnected.
      PmCorePropertyAndEventId propertyId(ON_DEVICE_DISCONNECTED);
      std::shared_ptr<PropertyDetails<BdAddress>> propertyDetails = std::make_shared
            <PropertyDetails<BdAddress>>("", BdAddress(pmCoreIfMessage->getBdAddress()));

      // Sending notification to interested controllers
      PropertyUpdateNotifierToCore::getInstance().notifyControllers(propertyId, propertyDetails);

      //Sending Device disconnected to BtStackWrapper
      evobtstackwrapper::EvoBtStackWrapper::getInstance().onDeviceDisconnected(pmCoreIfMessage->getBdAddress());
   }

   void PmCoreMainController::handlePmCoreIfMessage_OnDeviceDeleted(
         std::shared_ptr<PmCoreIfMessage_OnDeviceDeleted> pmCoreIfMessage)
   {
      ETG_TRACE_USR1(("handlePmCoreIfMessage_OnDeviceDeleted"));

      // Process device deleted.
      PmCorePropertyAndEventId propertyId(ON_DEVICE_DELETED);
      std::shared_ptr<PropertyDetails<BdAddress>> propertyDetails = std::make_shared
            <PropertyDetails<BdAddress>>("", BdAddress(pmCoreIfMessage->getBdAddress()));

      // Sending notification to interested controllers
      PropertyUpdateNotifierToCore::getInstance().notifyControllers(propertyId, propertyDetails);
   }

   void PmCoreMainController::handlePmCoreIfMessage_GetSystemWideRingtoneListRequest(
         std::shared_ptr<PmCoreIfMessage_GetSystemWideRingtoneListRequest> pmCoreIfMessage)
   {
      ETG_TRACE_USR1(("handlePmCoreIfMessage_GetSystemWideRingtoneListRequest"));

      //Posting request to corresponding controller
      _pmSettings.getSystemRingtonesList(pmCoreIfMessage->getAct());
   }

   void PmCoreMainController::handlePmCoreIfMessage_GetDeviceRingtoneListRequest(
         std::shared_ptr<PmCoreIfMessage_GetDeviceRingtoneListRequest> pmCoreIfMessage)
   {
      ETG_TRACE_USR1(("handlePmCoreIfMessage_GetDeviceRingtoneListRequest"));

      //Posting request to corresponding controller
      _pmSettings.getDeviceRingtoneList(pmCoreIfMessage->getAct());
   }

   void PmCoreMainController::handlePmCoreIfMessage_SetRingtoneRequest(
         std::shared_ptr<PmCoreIfMessage_SetRingtoneRequest> pmCoreIfMessage)
   {
      ETG_TRACE_USR1(("handlePmCoreIfMessage_SetRingtoneRequest"));

      //Posting request to corresponding controller
      PmResult pmResult = _pmSettings.setRingtone(pmCoreIfMessage->getBdAddress(), pmCoreIfMessage->getRingtoneId(),
            pmCoreIfMessage->getAct());

      if(PM_RESULT_OK != pmResult._pmResultCode)
      {
         ETG_TRACE_USR4(("sending SetRingtone request failed"));
      }
   }

   void PmCoreMainController::handlePmCoreIfMessage_PlayRingtoneRequest(
         std::shared_ptr<PmCoreIfMessage_PlayRingtoneRequest> pmCoreIfMessage)
   {
      ETG_TRACE_USR1(("handlePmCoreIfMessage_PlayRingtoneRequest"));

      //Posting request to corresponding controller
      _ringtoneController.playRingtoneRequest(pmCoreIfMessage);

      if(PM_RESULT_OK != pmCoreIfMessage->getPmResult()._pmResultCode)
      {
         ETG_TRACE_USR4(("sending playRingtone request failed"));
      }

      // Response to client's request
      PmCoreMainController::getInstance().getPmCoreCallbackIfWrapper().doPlayRingtoneResponse(
            pmCoreIfMessage->getPmResult(), pmCoreIfMessage->getAct());
   }

   void PmCoreMainController::handlePmCoreIfMessage_StopRingtoneRequest(
         std::shared_ptr<PmCoreIfMessage_StopRingtoneRequest> pmCoreIfMessage)
   {
      ETG_TRACE_USR1(("handlePmCoreIfMessage_StopRingtoneRequest"));

      //Posting request to corresponding controller
      _ringtoneController.stopRingtoneRequest(pmCoreIfMessage);

      if(PM_RESULT_OK != pmCoreIfMessage->getPmResult()._pmResultCode)
      {
         ETG_TRACE_USR4(("sending StopRingtone request failed"));
      }

      // Response to client's request
      PmCoreMainController::getInstance().getPmCoreCallbackIfWrapper().doStopRingtoneResponse(
            pmCoreIfMessage->getPmResult(), pmCoreIfMessage->getAct());
   }

   void PmCoreMainController::handlePmCoreIfMessage_GetSuppressRingtoneOnOffListRequest(
         std::shared_ptr<PmCoreIfMessage_GetSuppressRingtoneOnOffListRequest> pmCoreIfMessage)
   {
      ETG_TRACE_USR1(("handlePmCoreIfMessage_GetSuppressRingtoneOnOffListRequest"));

      //Posting request to corresponding controller
      _pmSettings.getSuppressRingtoneOnOffList(pmCoreIfMessage->getAct());
   }

   void PmCoreMainController::handlePmCoreIfMessage_SuppressRingtoneOnOffRequest(
         std::shared_ptr<PmCoreIfMessage_SuppressRingtoneOnOffRequest> pmCoreIfMessage)
   {
      ETG_TRACE_USR1(("handlePmCoreIfMessage_SuppressRingtoneOnOffRequest"));

      //Posting request to corresponding controller
      PmResult pmResult = _pmSettings.suppressRingtoneOnOff(pmCoreIfMessage->getBdAddress(),
            pmCoreIfMessage->getSuppressRingtoneState(),pmCoreIfMessage->getAct());

      if(PM_RESULT_OK != pmResult._pmResultCode)
      {
         ETG_TRACE_USR4(("sending SuppressRingtoneOnOff request failed"));
      }
   }

   void PmCoreMainController::handlePmCoreIfMessage_GetAutoWaitingModeOnOffListRequest(
         std::shared_ptr<PmCoreIfMessage_GetAutoWaitingModeOnOffListRequest> pmCoreIfMessage)
   {
      ETG_TRACE_USR1(("handlePmCoreIfMessage_GetAutoWaitingModeOnOffListRequest"));

      //Posting request to corresponding controller
      _pmSettings.getAutoWaitingModeOnOffList(pmCoreIfMessage->getAct());
   }

   void PmCoreMainController::handlePmCoreIfMessage_AutoWaitingModeOnOffRequest(
         std::shared_ptr<PmCoreIfMessage_AutoWaitingModeOnOffRequest> pmCoreIfMessage)
   {
      ETG_TRACE_USR1(("handlePmCoreIfMessage_AutoWaitingModeOnOffRequest"));

      //Posting request to corresponding controller
      PmResult pmResult = _pmSettings.autoWaitingModeOnOff(pmCoreIfMessage->getBdAddress(),
            pmCoreIfMessage->getAutoWaitingModeState(),pmCoreIfMessage->getAct());

      if(PM_RESULT_OK != pmResult._pmResultCode)
      {
         ETG_TRACE_USR4(("sending AutoWaitingModeOnOff request failed"));
      }
   }

   void PmCoreMainController::handlePmCoreIfMessage_SetWaitingModeFilePathRequest(
         std::shared_ptr<PmCoreIfMessage_SetWaitingModeFilePathRequest> pmCoreIfMessage)
   {
      ETG_TRACE_USR1(("handlePmCoreIfMessage_SetWaitingModeFilePathRequest"));

      //Posting request to corresponding controller
      PmResult pmResult = _callController.setWaitingModeFilePathRequest(pmCoreIfMessage->getWaitingModeFilePath(),
            pmCoreIfMessage->getAct());

      if(PM_RESULT_OK != pmResult._pmResultCode)
      {
         ETG_TRACE_USR4(("sending Set WaitingMode FilePath request failed"));
      }
   }

   void PmCoreMainController::handlePmCoreIfMessage_GetWaitingModeFilePathRequest(
         std::shared_ptr<PmCoreIfMessage_GetWaitingModeFilePathRequest> pmCoreIfMessage)
   {
      ETG_TRACE_USR1(("handlePmCoreIfMessage_GetWaitingModeFilePathRequest"));

      WaitingModeFilePath waitingModeFilePath;
      PmResult pmResult(PM_RESULT_OK, "");

      //Posting request to corresponding controller
      _callController.getWaitingModeFilePath(waitingModeFilePath);

      PmCoreMainController::getInstance().getPmCoreCallbackIfWrapper().doGetWaitingModeFilePathResponse(pmResult,
           waitingModeFilePath, pmCoreIfMessage->getAct());
   }

   void PmCoreMainController::handlePmCoreIfMessage_GetPhoneCallAudioActiveRequest(
         std::shared_ptr<PmCoreIfMessage_GetPhoneCallAudioActiveRequest> pmCoreIfMessage)
   {
      ETG_TRACE_USR1(("handlePmCoreIfMessage_GetPhoneCallAudioActiveRequest"));

      //Posting request to corresponding controller
      _callController.getPhoneCallAudioActiveRequest(pmCoreIfMessage->getAct());
   }

   void PmCoreMainController::handlePmCoreIfMessage_GetActivePassiveDeviceListRequest(
         std::shared_ptr<PmCoreIfMessage_GetActivePassiveDeviceListRequest> pmCoreIfMessage)
   {
      ETG_TRACE_USR1(("handlePmCoreIfMessage_GetActivePassiveDeviceListRequest"));

      //Posting request to corresponding controller
      //request will be responded and will be success always, no need to respond from here
      _deviceInfoHandler.getActivePassiveDeviceList(pmCoreIfMessage->getAct());
   }

   void PmCoreMainController::handlePmCoreIfMessage_SetActivePhoneRequest(
         std::shared_ptr<PmCoreIfMessage_SetActivePhoneRequest> pmCoreIfMessage)
   {
      ETG_TRACE_USR1(("handlePmCoreIfMessage_SetActivePhoneRequest"));
      PmResult pmResult(PM_RESULT_OK, "");

      BdAddressList deviceAddressList = pmCoreIfMessage->getBdAddressList();
      BdAddressList::iterator it = deviceAddressList.begin();

      ActivePassiveDeviceListMap activePassiveDevicelist;

      for( ; (it != deviceAddressList.end()); it++)
      {
         activePassiveDevicelist.insert(std::pair<BdAddress, DeviceRole>(*it, DEVICEROLE_ACTIVE));
      }

      //Posting request to corresponding controller
      pmResult = _deviceInfoHandler.setDeviceRoleRequest(activePassiveDevicelist);

      // Response to client's request
      PmCoreMainController::getInstance().getPmCoreCallbackIfWrapper().doSetActivePhoneResponse(pmResult,
            pmCoreIfMessage->getAct());
   }

   void PmCoreMainController::handlePmCoreIfMessage_RestoreDefaultSettingRequest(
         std::shared_ptr<PmCoreIfMessage_RestoreDefaultSettingRequest> pmCoreIfMessage)
   {
      ETG_TRACE_USR1(("handlePmCoreIfMessage_RestoreDefaultSettingRequest"));

      //Posting request to corresponding controller
      _pmSettings.restoreDefaultSetting(pmCoreIfMessage->getBdAddress(), pmCoreIfMessage->getAct());
   }

   void PmCoreMainController::handlePmCoreIfMessage_GetSupportedFeaturesRequest(
         std::shared_ptr<PmCoreIfMessage_GetSupportedFeaturesRequest> pmCoreIfMessage)
   {
      ETG_TRACE_USR1(("handlePmCoreIfMessage_GetSupportedFeaturesRequest"));

      //Posting request to corresponding controller
      _deviceInfoHandler.getSupportedFeatures(pmCoreIfMessage->getBdAddress(), pmCoreIfMessage->getAct());
   }

   void PmCoreMainController::handlePmCoreIfMessage_GetNetworkStatusRequest(
         std::shared_ptr<PmCoreIfMessage_GetNetworkStatusRequest> pmCoreIfMessage)
   {
      ETG_TRACE_USR1(("handlePmCoreIfMessage_GetNetworkStatusRequest"));

      //Posting request to corresponding controller
      _deviceInfoHandler.getNetworkStatus(pmCoreIfMessage->getBdAddress(), pmCoreIfMessage->getAct());
   }

   void PmCoreMainController::handlePmCoreIfMessage_GetSignalStrengthRequest(
         std::shared_ptr<PmCoreIfMessage_GetSignalStrengthRequest> pmCoreIfMessage)
   {
      ETG_TRACE_USR1(("handlePmCoreIfMessage_GetSignalStrengthRequest"));

      //Posting request to corresponding controller
      _deviceInfoHandler.getSignalStrength(pmCoreIfMessage->getBdAddress(), pmCoreIfMessage->getAct());
   }

   void PmCoreMainController::handlePmCoreIfMessage_GetBatteryChargeLevelRequest(
         std::shared_ptr<PmCoreIfMessage_GetBatteryChargeLevelRequest> pmCoreIfMessage)
   {
      ETG_TRACE_USR1(("handlePmCoreIfMessage_GetBatteryChargeLevelRequest"));

      //Posting request to corresponding controller
      _deviceInfoHandler.getBatteryChargeLevel(pmCoreIfMessage->getBdAddress(), pmCoreIfMessage->getAct());
   }

   void PmCoreMainController::handlePmCoreIfMessage_GetNetworkOperatorRequest(
         std::shared_ptr<PmCoreIfMessage_GetNetworkOperatorRequest> pmCoreIfMessage)
   {
      ETG_TRACE_USR1(("handlePmCoreIfMessage_GetNetworkOperatorRequest"));

      //Posting request to corresponding controller
      _deviceInfoHandler.getNetworkOperator(pmCoreIfMessage->getBdAddress(), pmCoreIfMessage->getAct());
   }

   void PmCoreMainController::handlePmCoreIfMessage_VoiceMailListRequest(
         std::shared_ptr<PmCoreIfMessage_VoiceMailListRequest> pmCoreIfMessage)
   {
      ETG_TRACE_USR1(("handlePmCoreIfMessage_VoiceMailListRequest"));

      //Posting request to corresponding controller
      _deviceInfoHandler.getVoiceMailList(pmCoreIfMessage->getBdAddress(), pmCoreIfMessage->getAct());
   }

   void PmCoreMainController::handlePmCoreIfMessage_GetCallStatusListRequest(
         std::shared_ptr<PmCoreIfMessage_GetCallStatusListRequest> pmCoreIfMessage)
   {
      ETG_TRACE_USR1(("handlePmCoreIfMessage_GetCallStatusListRequest"));

      //Posting request to corresponding controller
      _callController.getCallStatusListRequest(pmCoreIfMessage->getAct());
   }

   void PmCoreMainController::handlePmCoreIfMessage_GetMicrophoneMuteStateRequest(
         std::shared_ptr<PmCoreIfMessage_GetMicrophoneMuteStateRequest> pmCoreIfMessage)
   {
      ETG_TRACE_USR1(("handlePmCoreIfMessage_GetMicrophoneMuteStateRequest"));

      //Posting request to corresponding controller
      _callController.getMicrophoneMuteStateRequest(pmCoreIfMessage->getAct());
   }

   void PmCoreMainController::handlePmCoreIfMessage_SetMicrophoneMuteStateRequest(
         std::shared_ptr<PmCoreIfMessage_SetMicrophoneMuteStateRequest> pmCoreIfMessage)
   {
      ETG_TRACE_USR1(("handlePmCoreIfMessage_SetMicrophoneMuteStateRequest"));
      PmResult pmResult(PM_RESULT_OK, "");

      //Posting request to corresponding controller
      pmResult = _callController.setMicrophoneMuteStateRequest(pmCoreIfMessage->getMuteState());

      if(PM_RESULT_OK != pmResult._pmResultCode)
      {
         ETG_TRACE_USR4(("setting MicrophoneMuteState request failed"));
      }

      // Response to client's request
      _pmCoreCallbackIfWrapper->doSetMicrophoneMuteStateResponse(pmResult, pmCoreIfMessage->getAct());
   }

   void PmCoreMainController::handlePmCoreIfMessage_MergeCallsRequest(
         std::shared_ptr<PmCoreIfMessage_MergeCallsRequest> pmCoreIfMessage)
   {
      ETG_TRACE_USR1(("handlePmCoreIfMessage_MergeCallsRequest"));

      //Posting request to corresponding controller
      _callController.mergeCallsRequest(pmCoreIfMessage);

      if(PM_RESULT_OK != pmCoreIfMessage->getPmResult()._pmResultCode)
      {
         ETG_TRACE_USR4(("sending MergeCalls request failed"));
      }

      // Response to client's request
      _pmCoreCallbackIfWrapper->doMergeCallsResponse(pmCoreIfMessage->getPmResult(), pmCoreIfMessage->getAct());
   }

   void PmCoreMainController::handlePmCoreIfMessage_SplitCallsRequest(
         std::shared_ptr<PmCoreIfMessage_SplitCallsRequest> pmCoreIfMessage)
   {
      ETG_TRACE_USR1(("handlePmCoreIfMessage_SplitCallsRequest"));

      //Posting request to corresponding controller
      _callController.splitCallsRequest(pmCoreIfMessage);

      if(PM_RESULT_OK != pmCoreIfMessage->getPmResult()._pmResultCode)
      {
         ETG_TRACE_USR4(("sending SplitCalls request failed"));
      }

      // Response to client's request
      _pmCoreCallbackIfWrapper->doSplitCallsResponse(pmCoreIfMessage->getPmResult(), pmCoreIfMessage->getAct());
   }

   void PmCoreMainController::handlePmCoreIfMessage_HangupCallsRequest(
         std::shared_ptr<PmCoreIfMessage_HangupCallsRequest> pmCoreIfMessage)
   {
      ETG_TRACE_USR1(("handlePmCoreIfMessage_HangupCallsRequest"));

      //Posting request to corresponding controller
      _callController.hangupCallsRequest(pmCoreIfMessage);

      if(PM_RESULT_OK != pmCoreIfMessage->getPmResult()._pmResultCode)
      {
         ETG_TRACE_USR4(("sending HangupCalls request failed"));
      }

      // Response to client's request
      _pmCoreCallbackIfWrapper->doHangupCallsResponse(pmCoreIfMessage->getPmResult(), pmCoreIfMessage->getAct());
   }

   void PmCoreMainController::handlePmCoreIfMessage_RedialRequest(
         std::shared_ptr<PmCoreIfMessage_RedialRequest> pmCoreIfMessage)
   {
      ETG_TRACE_USR1(("handlePmCoreIfMessage_RedialRequest"));

      //Posting request to corresponding controller
      _callController.redialRequest(pmCoreIfMessage);

      if(PM_RESULT_OK != pmCoreIfMessage->getPmResult()._pmResultCode)
      {
         ETG_TRACE_USR4(("sending Redial request failed"));

         CallInstance callInstance = CALL_INSTANCE_DEFAULT;

         //Error response received, send response to IpcWrapper.
         //Success response will be sent on receiving RedialResult from BtStack.
         _pmCoreCallbackIfWrapper->doRedialResponse(pmCoreIfMessage->getPmResult(), pmCoreIfMessage->getBdAddress(),
               callInstance, pmCoreIfMessage->getAct());
      }
   }

   void PmCoreMainController::handlePmCoreIfMessage_GetSCOConnectionRequest(
         std::shared_ptr<PmCoreIfMessage_GetSCOConnectionRequest> pmCoreIfMessage)
   {
      ETG_TRACE_USR1(("handlePmCoreIfMessage_GetSCOConnectionRequest"));

      //Posting request to corresponding controller
      _deviceInfoHandler.getSCOConnectionRequest(pmCoreIfMessage->getBdAddress(), pmCoreIfMessage->getAct());
   }

   void PmCoreMainController::handlePmCoreIfMessage_GetRingtoneMuteStateRequest(
         std::shared_ptr<PmCoreIfMessage_GetRingtoneMuteStateRequest> pmCoreIfMessage)
   {
      ETG_TRACE_USR1(("handlePmCoreIfMessage_GetRingtoneMuteStateRequest"));

      //Posting request to corresponding controller
      _ringtoneController.getRingtoneMuteStateRequest(pmCoreIfMessage->getBdAddress(), pmCoreIfMessage->getAct());
   }

   void PmCoreMainController::handlePmCoreIfMessage_SetRingtoneMuteStateRequest(
         std::shared_ptr<PmCoreIfMessage_SetRingtoneMuteStateRequest> pmCoreIfMessage)
   {
      ETG_TRACE_USR1(("handlePmCoreIfMessage_SetRingtoneMuteStateRequest"));
      PmResult pmResult(PM_RESULT_OK, "");

      //Posting request to corresponding controller
      pmResult = _ringtoneController.setRingtoneMuteStateRequest(pmCoreIfMessage->getBdAddress(),
            pmCoreIfMessage->getMuteState(), pmCoreIfMessage->getAct());

      if(PM_RESULT_OK != pmResult._pmResultCode)
      {
         ETG_TRACE_USR4(("setting RingtoneMuteState request failed"));
      }
   }

   void PmCoreMainController::handlePmCoreIfMessage_GetWaitingModeStateRequest(
         std::shared_ptr<PmCoreIfMessage_GetWaitingModeStateRequest> pmCoreIfMessage)
   {
      ETG_TRACE_USR1(("handlePmCoreIfMessage_GetWaitingModeStateRequest"));

      //Posting request to corresponding controller
      _callController.getWaitingModeStateRequest(pmCoreIfMessage->getBdAddress(), pmCoreIfMessage->getAct());
   }

   void PmCoreMainController::handlePmCoreIfMessage_GetRingtonePlaybackStatusRequest(
         std::shared_ptr<PmCoreIfMessage_GetRingtonePlaybackStatusRequest> pmCoreIfMessage)
   {
      ETG_TRACE_USR1(("handlePmCoreIfMessage_GetRingtonePlaybackStatusRequest"));

      //Posting request to corresponding controller
      _ringtoneController.getRingtonePlaybackStatusRequest(pmCoreIfMessage->getAct());
   }

   void PmCoreMainController::handlePmCoreIfMessage_StartStopWaitingModeRequest(
         std::shared_ptr<PmCoreIfMessage_StartStopWaitingModeRequest> pmCoreIfMessage)
   {
      ETG_TRACE_USR1(("handlePmCoreIfMessage_StartStopWaitingModeRequest"));

      BdAddress deviceAddress = pmCoreIfMessage->getBdAddress();
      PmResult pmResult(PM_RESULT_OK, "");

      ActiveSwitchingRequest activeSwitchingRequest(deviceAddress, STARTING_WAITING_MODE,
            PendingRequestInfo(pmCoreIfMessage->getAct()), CALL_CONTROLLER);

      bool postRequestToController = false;

      if (CANNOT_BE_MADE_ACTIVE != _deviceInfoHandler.acquireFreeActiveDeviceSlot(deviceAddress,
            activeSwitchingRequest))
      {
         ETG_TRACE_USR1(("Handling Active device request for Waitingmode"));
         postRequestToController = true;
      }
      else
      {
         ETG_TRACE_USR1(("Handling Passive device request for Waitingmode"));

         DeviceRoleSwitchResultEnum acquireActiveDeviceSlotResult =
               _deviceInfoHandler.acquireActiveDeviceSlot(activeSwitchingRequest);

         // IMPORTANT: If Stop waiting mode is received for Passive device, Error is sent from here
         if ((pmCoreIfMessage->getStartStopWaitingMode() == false) ||
               (CANNOT_BE_MADE_ACTIVE == acquireActiveDeviceSlotResult))
         {
            // Error response
            pmResult._pmResultCode = PM_RESULT_ERR_GENERAL;
         }
         else if (SWITCHED_TO_ACTIVE == acquireActiveDeviceSlotResult)
         {
            postRequestToController = true;
         }
      }

      if (postRequestToController)
      {
         //Posting request to corresponding controller
         _callController.startStopWaitingModeRequest(pmCoreIfMessage);

         if(PM_RESULT_OK != pmCoreIfMessage->getPmResult()._pmResultCode)
         {
            ETG_TRACE_USR4(("sending StartStopWaitingMode request failed"));
         }

         pmResult = pmCoreIfMessage->getPmResult();
      }

      _pmCoreCallbackIfWrapper->doStartStopWaitingModeResponse(pmResult, deviceAddress,
            pmCoreIfMessage->getAct());
   }

   void PmCoreMainController::handlePmCoreIfMessage_DialRequest(
         std::shared_ptr<PmCoreIfMessage_DialRequest> pmCoreIfMessage)
   {
      ETG_TRACE_USR1(("handlePmCoreIfMessage_DialRequest"));

      //Posting request to corresponding controller
      _callController.dialRequest(pmCoreIfMessage);

      if(PM_RESULT_OK != pmCoreIfMessage->getPmResult()._pmResultCode)
      {
         ETG_TRACE_USR4(("sending Dial request failed"));

         CallInstance callInstance = CALL_INSTANCE_DEFAULT;

         //Error response received, send response to IpcWrapper.
         //Success response will be sent on receiving DialResult from BtStack.
         _pmCoreCallbackIfWrapper->doDialResponse(pmCoreIfMessage->getPmResult(), pmCoreIfMessage->getBdAddress(),
               callInstance, pmCoreIfMessage->getAct());
      }
   }

   void PmCoreMainController::handlePmCoreIfMessage_AcceptCallRequest(
         std::shared_ptr<PmCoreIfMessage_AcceptCallRequest> pmCoreIfMessage)
   {
      ETG_TRACE_USR1(("handlePmCoreIfMessage_AcceptCallRequest"));

      BdAddress deviceAddress = pmCoreIfMessage->getBdAddress();

      ActiveSwitchingRequest activeSwitchingRequest(deviceAddress, ACCEPTING_CALL,
            PendingRequestInfo(pmCoreIfMessage->getCallInstance(), pmCoreIfMessage->getAcceptOperation(),
                  pmCoreIfMessage->getAct()), CALL_CONTROLLER);

      bool postAcceptCallToSM = false;
      bool setAcceptOperation = false;

      DeviceRoleSwitchResultEnum deviceRoleSwitchResult =
            _deviceInfoHandler.acquireFreeActiveDeviceSlot(deviceAddress, activeSwitchingRequest);

      // "acquireFreeActiveDeviceSlot" is added to handle a special scenario.
      // Consider active VR in Active device and an incoming call in the Passive device.
      // On accepting the Passive device incoming call, HMI will stop VR in the current Active device and then
      // will accept the Passive device call.

      if (ALREADY_ACTIVE == deviceRoleSwitchResult)
      {
         postAcceptCallToSM = true;
      }
      else if (SWITCHED_TO_ACTIVE == deviceRoleSwitchResult)
      {
         postAcceptCallToSM = setAcceptOperation = true;
      }
      else
      {
         ETG_TRACE_USR1(("Handling Passive device request for Accepting call"));

         PmResult pmResult(PM_RESULT_OK, "");

         DeviceRoleSwitchResultEnum acquireActiveDeviceSlotResult =
                        _deviceInfoHandler.acquireActiveDeviceSlot(activeSwitchingRequest);

         if (CANNOT_BE_MADE_ACTIVE == acquireActiveDeviceSlotResult)
         {
            // Error response
            pmResult._pmResultCode = PM_RESULT_ERR_GENERAL;

            _pmCoreCallbackIfWrapper->doAcceptCallResponse(pmResult, deviceAddress,
                  pmCoreIfMessage->getCallInstance(), pmCoreIfMessage->getAct());
         }
         else if (SWITCHED_TO_ACTIVE == acquireActiveDeviceSlotResult)
         {
            postAcceptCallToSM = setAcceptOperation = true;
         }
      }

      if (setAcceptOperation)
      {
         switch (pmCoreIfMessage->getAcceptOperation())
         {
            case ACCEPT_CALLS:
               pmCoreIfMessage->setAcceptOperation(ACCEPT_CALLS_BY_ACQUIRING_CHANNEL);
               break;
            case RELEASE_AND_ACCEPT:
               pmCoreIfMessage->setAcceptOperation(RELEASE_AND_ACCEPT_BY_ACQUIRING_CHANNEL);
               break;
            case HOLD_AND_ACCEPT:
               pmCoreIfMessage->setAcceptOperation(HOLD_AND_ACCEPT_BY_ACQUIRING_CHANNEL);
               break;
            default:
               pmCoreIfMessage->setAcceptOperation(ACCEPT_CALLS_BY_ACQUIRING_CHANNEL);
         }
      }

      if (postAcceptCallToSM)
      {
         //Posting request to corresponding controller
         _callController.acceptCallRequest(pmCoreIfMessage);

         if(PM_RESULT_OK != pmCoreIfMessage->getPmResult()._pmResultCode)
         {
            ETG_TRACE_USR4(("sending AcceptCall request failed"));
         }

         // Response to client's request
         _pmCoreCallbackIfWrapper->doAcceptCallResponse(pmCoreIfMessage->getPmResult(),
               pmCoreIfMessage->getBdAddress(), pmCoreIfMessage->getCallInstance(), pmCoreIfMessage->getAct());
      }
   }

   void PmCoreMainController::handlePmCoreIfMessage_SwapCallRequest(
         std::shared_ptr<PmCoreIfMessage_SwapCallRequest> pmCoreIfMessage)
   {
      ETG_TRACE_USR1(("handlePmCoreIfMessage_SwapCallRequest"));

      //Posting request to corresponding controller
      _callController.swapCallRequest(pmCoreIfMessage);

      if(PM_RESULT_OK != pmCoreIfMessage->getPmResult()._pmResultCode)
      {
         ETG_TRACE_USR4(("sending SwapCall request failed"));
      }

      // Response to client's request
      _pmCoreCallbackIfWrapper->doSwapCallResponse(pmCoreIfMessage->getPmResult(), pmCoreIfMessage->getBdAddress(),
            pmCoreIfMessage->getAct());
   }

   void PmCoreMainController::handlePmCoreIfMessage_SpeedDialRequest(
         std::shared_ptr<PmCoreIfMessage_SpeedDialRequest> pmCoreIfMessage)
   {
      ETG_TRACE_USR1(("handlePmCoreIfMessage_SpeedDialRequest"));

      //Posting request to corresponding controller
      _callController.speedDialRequest(pmCoreIfMessage);

      if(PM_RESULT_OK != pmCoreIfMessage->getPmResult()._pmResultCode)
      {
         ETG_TRACE_USR4(("sending SpeedDial request failed"));

         CallInstance callInstance = CALL_INSTANCE_DEFAULT;

         //Error response received, send response to IpcWrapper.
         //Success response will be sent on receiving SpeedDialResult from BtStack.
         _pmCoreCallbackIfWrapper->doSpeedDialResponse(pmCoreIfMessage->getPmResult(), pmCoreIfMessage->getBdAddress(),
               callInstance, pmCoreIfMessage->getAct());
      }
   }

   void PmCoreMainController::handlePmCoreIfMessage_SendDTMFRequest(
         std::shared_ptr<PmCoreIfMessage_SendDTMFRequest> pmCoreIfMessage)
   {
      ETG_TRACE_USR1(("handlePmCoreIfMessage_SendDTMFRequest"));
      PmResult pmResult(PM_RESULT_OK, "");

      //Posting request to corresponding controller
      pmResult = _callController.sendDTMFRequest(pmCoreIfMessage);

      if(PM_RESULT_OK != pmResult._pmResultCode)
      {
         ETG_TRACE_USR4(("handlePmCoreIfMessage_SendDTMFRequest: sending result to controller failed"));
      }

      // Response to client's request
      _pmCoreCallbackIfWrapper->doSendDTMFResponse(pmResult, pmCoreIfMessage->getBdAddress(),
            pmCoreIfMessage->getAct());
   }

   void PmCoreMainController::handlePmCoreIfMessage_TransferAudioRequest(
         std::shared_ptr<PmCoreIfMessage_TransferAudioRequest> pmCoreIfMessage)
   {
      ETG_TRACE_USR1(("handlePmCoreIfMessage_TransferAudioRequest"));

      //Posting request to corresponding controller
      _callController.transferAudioRequest(pmCoreIfMessage);

      if(PM_RESULT_OK != pmCoreIfMessage->getPmResult()._pmResultCode)
      {
         ETG_TRACE_USR4(("sending TransferAudio request failed"));
      }

      // Response to client's request
      _pmCoreCallbackIfWrapper->doTransferAudioResponse(pmCoreIfMessage->getPmResult(),
            pmCoreIfMessage->getBdAddress(), pmCoreIfMessage->getAct());
   }

   void PmCoreMainController::handlePmCoreIfMessage_GetVoiceRecognitionStatusRequest(
         std::shared_ptr<PmCoreIfMessage_GetVoiceRecognitionStatusRequest> pmCoreIfMessage)
   {
      ETG_TRACE_USR1(("handlePmCoreIfMessage_GetVoiceRecognitionStatusRequest"));

      //Posting request to corresponding controller
      _vrController.getVoiceRecognitionStatusRequest(pmCoreIfMessage->getBdAddress(), pmCoreIfMessage->getAct());
   }

   void PmCoreMainController::handlePmCoreIfMessage_GetExtVoiceRecognitionStatusRequest(
         std::shared_ptr<PmCoreIfMessage_GetExtVoiceRecognitionStatusRequest> pmCoreIfMessage)
   {
      ETG_TRACE_USR1(("handlePmCoreIfMessage_GetExtVoiceRecognitionStatusRequest"));

      //Posting request to corresponding controller
      _vrController.getExtVoiceRecognitionStatusRequest(pmCoreIfMessage->getBdAddress(), pmCoreIfMessage->getAct());
   }

   void PmCoreMainController::handlePmCoreIfMessage_StartStopVoiceRecognitionRequest(
         std::shared_ptr<PmCoreIfMessage_StartStopVoiceRecognitionRequest> pmCoreIfMessage)
   {
      ETG_TRACE_USR1(("handlePmCoreIfMessage_StartStopVoiceRecognitionRequest"));
      PmResult pmResult(PM_RESULT_OK, "");

      BdAddress deviceAddress = pmCoreIfMessage->getBdAddress();

      ActiveSwitchingRequest activeSwitchingRequest(deviceAddress, ACTIVATING_VR,
            PendingRequestInfo(pmCoreIfMessage->getAct()), VR_CONTROLLER);

      bool postRequestToController = false;

      // "acquireFreeActiveDeviceSlot" is added to handle a special scenario.
      // Consider active VR in Active device and an incoming call in the Passive device.
      // On accepting the Passive device incoming call, HMI will stop VR in the current Active device and then
      // will accept the Passive device call.
      if (CANNOT_BE_MADE_ACTIVE != _deviceInfoHandler.acquireFreeActiveDeviceSlot(deviceAddress,
            activeSwitchingRequest))
      {
         ETG_TRACE_USR1(("Handling Active device request"));
         postRequestToController = true;
      }
      else
      {
         ETG_TRACE_USR1(("Handling Passive device request for Starting VR"));

         DeviceRoleSwitchResultEnum acquireActiveDeviceSlotResult =
                        _deviceInfoHandler.acquireActiveDeviceSlot(activeSwitchingRequest);

         if (CANNOT_BE_MADE_ACTIVE == acquireActiveDeviceSlotResult)
         {
            // Error response
            pmResult._pmResultCode = PM_RESULT_ERR_GENERAL;
         }
         else if (SWITCHED_TO_ACTIVE == acquireActiveDeviceSlotResult)
         {
            postRequestToController = true;
         }
      }

      if (postRequestToController)
      {
         //Posting request to corresponding controller
         pmResult = _vrController.startStopVoiceRecognitionRequest(pmCoreIfMessage->getBdAddress(),
               pmCoreIfMessage->getStartStopVR());

         if(PM_RESULT_OK != pmResult._pmResultCode)
         {
            ETG_TRACE_USR4(("handlePmCoreIfMessage_StartStopVoiceRecognitionRequest: sending result to controller "
                  "failed"));
         }
      }
      // Response to client's request
      _pmCoreCallbackIfWrapper->doStartStopVoiceRecognitionResponse(pmResult, deviceAddress,
            pmCoreIfMessage->getAct());
   }

   void PmCoreMainController::handlePmCoreIfMessage_StartStopExternalVoiceRecognitionRequest(
         std::shared_ptr<PmCoreIfMessage_StartStopExternalVoiceRecognitionRequest> pmCoreIfMessage)
   {
      ETG_TRACE_USR1(("handlePmCoreIfMessage_StartStopExternalVoiceRecognitionRequest"));
      PmResult pmResult(PM_RESULT_OK, "");

      BdAddress deviceAddress = pmCoreIfMessage->getBdAddress();
      StartStop startStopExtVR = pmCoreIfMessage->getStartStopExtVR();

      ActiveSwitchingRequest activeSwitchingRequest(deviceAddress, ACTIVATING_VR,
            PendingRequestInfo(pmCoreIfMessage->getAct()), VR_CONTROLLER);

      bool postRequestToController = false;

      // "acquireFreeActiveDeviceSlot" is added to handle a special scenario.
      // Consider active VR in Active device and an incoming call in the Passive device.
      // On accepting the Passive device incoming call, HMI will stop VR in the current Active device and then
      // will accept the Passive device call.

      if (startStopExtVR)
      {
         if (CANNOT_BE_MADE_ACTIVE != _deviceInfoHandler.acquireFreeActiveDeviceSlot(deviceAddress,
               activeSwitchingRequest))
         {
            ETG_TRACE_USR1(("Handling Active device request"));
            postRequestToController = true;
         }
         else
         {
            ETG_TRACE_USR1(("Handling Passive device request for Starting VR"));

            DeviceRoleSwitchResultEnum acquireActiveDeviceSlotResult =
                  _deviceInfoHandler.acquireActiveDeviceSlot(activeSwitchingRequest);

            if (CANNOT_BE_MADE_ACTIVE == acquireActiveDeviceSlotResult)
            {
               // Error response
               pmResult._pmResultCode = PM_RESULT_ERR_GENERAL;
            }
            else if (SWITCHED_TO_ACTIVE == acquireActiveDeviceSlotResult)
            {
               postRequestToController = true;
            }
         }

         if (postRequestToController)
         {
            //Posting request to corresponding controller
            pmResult = _vrController.startStopExtVoiceRecognitionRequest(pmCoreIfMessage->getBdAddress(),
                  startStopExtVR);

            if(PM_RESULT_OK != pmResult._pmResultCode)
            {
               ETG_TRACE_USR4(("startStopExtVoiceRecognitionRequest: sending result to controller failed"));
            }
         }

         // Response to client's request
         _pmCoreCallbackIfWrapper->doActivateExternalVRResponse(pmResult, deviceAddress,
               pmCoreIfMessage->getAct());
      }
      else
      {
         //Posting request to corresponding controller
         pmResult = _vrController.startStopExtVoiceRecognitionRequest(pmCoreIfMessage->getBdAddress(),
               startStopExtVR);

         // Response to client's request
         _pmCoreCallbackIfWrapper->doDeActivateExternalVRResponse(pmResult, deviceAddress,
               pmCoreIfMessage->getAct());
      }
   }

   void PmCoreMainController::handlePmCoreIfMessage_stopExtendedVRRequest(
         std::shared_ptr<PmCoreIfMessage_stopExtendedVRRequest> pmCoreIfMessage)
   {
      ETG_TRACE_USR1(("handlePmCoreIfMessage_stopExtendedVRRequest"));

      //Posting request to corresponding controller
      _vrController.stopExtendedVR(pmCoreIfMessage->getBdAddress());
   }

   void PmCoreMainController::handlePmCoreIfMessage_GetEnhancedVoiceRecognitionFeatureRequest(
         std::shared_ptr<PmCoreIfMessage_GetEnhancedVoiceRecognitionFeatureRequest> pmCoreIfMessage)
   {
      ETG_TRACE_USR1(("handlePmCoreIfMessage_GetEnhancedVoiceRecognitionFeatureRequest"));

      //Posting request to corresponding controller
      _vrController.getEnhancedVoiceRecognitionFeatureRequest(pmCoreIfMessage->getBdAddress(),
            pmCoreIfMessage->getAct());
   }

   void PmCoreMainController::handlePmCoreIfMessage_RequestPhoneNumberResult(
         std::shared_ptr<PmCoreIfMessage_RequestPhoneNumberResult> pmCoreIfMessage)
   {
      ETG_TRACE_USR1(("handlePmCoreIfMessage_RequestPhoneNumberResult"));
      PmResult pmResult(PM_RESULT_OK, "");

      // Posting result to corresponding controller
      _callController.requestPhoneNumberResult(pmCoreIfMessage->getBTSResult(), pmCoreIfMessage->getBdAddress(),
            pmCoreIfMessage->getTelephoneNumber(), pmCoreIfMessage->getAct());

      if(PM_RESULT_OK != pmResult._pmResultCode)
      {
         ETG_TRACE_USR4(("handlePmCoreIfMessage_RequestPhoneNumberResult: sending result to controller failed"));
      }
   }

   void PmCoreMainController::handlePmCoreIfMessage_GetSubscriberResult(
         std::shared_ptr<PmCoreIfMessage_GetSubscriberResult> pmCoreIfMessage)
   {
      ETG_TRACE_USR1(("handlePmCoreIfMessage_GetSubscriberResult"));

      // Posting result to corresponding controller
      _deviceInfoHandler.getVoiceMailListResponse(pmCoreIfMessage);
   }

   void PmCoreMainController::handlePmCoreIfMessage_SendCustomATCommandResult(
         std::shared_ptr<PmCoreIfMessage_SendCustomATCommandResult> pmCoreIfMessage)
   {
      ETG_TRACE_USR1(("handlePmCoreIfMessage_SendCustomATCommandResult"));
      PmResult pmResult(PM_RESULT_OK, "");

      (void)pmCoreIfMessage; //To remove gen4 warning
      //TODO:Posting result to corresponding controller

      if(PM_RESULT_OK != pmResult._pmResultCode)
      {
         ETG_TRACE_USR4(("handlePmCoreIfMessage_SendCustomATCommandResult: sending result to controller failed"));
      }
   }

   void PmCoreMainController::handlePmCoreIfMessage_HFUpdateIndicatorsValueResult(
         std::shared_ptr<PmCoreIfMessage_HFUpdateIndicatorsValueResult> pmCoreIfMessage)
   {
      ETG_TRACE_USR1(("handlePmCoreIfMessage_HFUpdateIndicatorsValueResult"));
      PmResult pmResult(PM_RESULT_OK, "");

      (void)pmCoreIfMessage; //To remove gen4 warning
      //TODO:Posting result to corresponding controller

      if(PM_RESULT_OK != pmResult._pmResultCode)
      {
         ETG_TRACE_USR4(("handlePmCoreIfMessage_HFUpdateIndicatorsValueResult: sending result to controller failed"));
      }
   }

   void PmCoreMainController::handlePmCoreIfMessage_GetModemsResult(
         std::shared_ptr<PmCoreIfMessage_GetModemsResult> pmCoreIfMessage)
   {
      ETG_TRACE_USR1(("handlePmCoreIfMessage_GetModemsResult"));
      PmResult pmResult(PM_RESULT_OK, "");

      (void)pmCoreIfMessage; //To remove gen4 warning
      //TODO:Posting result to corresponding controller

      if(PM_RESULT_OK != pmResult._pmResultCode)
      {
         ETG_TRACE_USR4(("handlePmCoreIfMessage_GetModemsResult: sending result to controller failed"));
      }
   }

   void PmCoreMainController::handlePmCoreIfMessage_AcceptSCOConnectResult(
         std::shared_ptr<PmCoreIfMessage_AcceptSCOConnectResult> pmCoreIfMessage)
   {
      ETG_TRACE_USR1(("handlePmCoreIfMessage_AcceptSCOConnectResult"));

      PmResult pmResult(PM_RESULT_OK, "");

      //TODO: To be discussed
      // Some requests are posted from different controllers (i.e AcceptSCO can be called from
      // either callController or VRController).
      // In such case, how to post the result message to both the controllers.

      // Posting result to corresponding controller
      _callController.acceptSCOConnectResult(pmCoreIfMessage);

      if(PM_RESULT_OK != pmResult._pmResultCode)
      {
         ETG_TRACE_USR4(("handlePmCoreIfMessage_AcceptSCOConnectResult: sending result to controller failed"));
      }
   }

   void PmCoreMainController::handlePmCoreIfMessage_SiriEnableResult(
         std::shared_ptr<PmCoreIfMessage_SiriEnableResult> pmCoreIfMessage)
   {
      ETG_TRACE_USR1(("handlePmCoreIfMessage_SiriEnableResult"));
      PmResult pmResult(PM_RESULT_OK, "");

      // Posting result to corresponding controller
      pmResult = _vrController.siriEnableResult(pmCoreIfMessage->getBdAddress(), pmCoreIfMessage->getBTSResult());
      if(PM_RESULT_OK != pmResult._pmResultCode)
      {
         ETG_TRACE_USR4(("handlePmCoreIfMessage_SiriEnableResult: sending result to controller failed"));
      }
   }

   void PmCoreMainController::handlePmCoreIfMessage_SiriSetNRResult(
         std::shared_ptr<PmCoreIfMessage_SiriSetNRResult> pmCoreIfMessage)
   {
      ETG_TRACE_USR1(("handlePmCoreIfMessage_SiriSetNRResult"));
      PmResult pmResult(PM_RESULT_OK, "");

      // Posting result to corresponding controller
      pmResult = _vrController.siriSetNRResult(pmCoreIfMessage->getBdAddress(), pmCoreIfMessage->getBTSResult());
      if(PM_RESULT_OK != pmResult._pmResultCode)
      {
         ETG_TRACE_USR4(("handlePmCoreIfMessage_SiriSetNRResult: sending result to controller failed"));
      }
   }

   void PmCoreMainController::handlePmCoreIfMessage_GetCallsResult(
         std::shared_ptr<PmCoreIfMessage_GetCallsResult> pmCoreIfMessage)
   {
      ETG_TRACE_USR1(("handlePmCoreIfMessage_GetCallsResult"));

      // Posting result to corresponding controller
      _callController.getCallsResult(pmCoreIfMessage);
   }

   void PmCoreMainController::handlePmCoreIfMessage_DialResult(
         std::shared_ptr<PmCoreIfMessage_DialResult> pmCoreIfMessage)
   {
      ETG_TRACE_USR1(("handlePmCoreIfMessage_DialResult"));

      // Posting result to corresponding controller
      _callController.dialResult(pmCoreIfMessage);
   }

   void PmCoreMainController::handlePmCoreIfMessage_SpeedDialResult(
         std::shared_ptr<PmCoreIfMessage_SpeedDialResult> pmCoreIfMessage)
   {
      ETG_TRACE_USR1(("handlePmCoreIfMessage_SpeedDialResult"));

      // Posting result to corresponding controller
      _callController.speedDialResult(pmCoreIfMessage);
   }

   void PmCoreMainController::handlePmCoreIfMessage_RedialResult(
         std::shared_ptr<PmCoreIfMessage_RedialResult> pmCoreIfMessage)
   {
      ETG_TRACE_USR1(("handlePmCoreIfMessage_RedialResult"));

      // Posting result to corresponding controller
      _callController.redialResult(pmCoreIfMessage);
   }

   void PmCoreMainController::handlePmCoreIfMessage_SwapCallResult(
         std::shared_ptr<PmCoreIfMessage_SwapCallResult> pmCoreIfMessage)
   {
      ETG_TRACE_USR1(("handlePmCoreIfMessage_SwapCallResult"));

      // Posting result to corresponding controller
      _callController.swapCallResult(pmCoreIfMessage);
   }

   void PmCoreMainController::handlePmCoreIfMessage_ReleaseAndAcceptResult(
         std::shared_ptr<PmCoreIfMessage_ReleaseAndAcceptResult> pmCoreIfMessage)
   {
      ETG_TRACE_USR1(("handlePmCoreIfMessage_ReleaseAndAcceptResult"));

      // Posting result to corresponding controller
      _callController.releaseAndAcceptResult(pmCoreIfMessage);
   }

   void PmCoreMainController::handlePmCoreIfMessage_ReleaseAndSwapResult(
         std::shared_ptr<PmCoreIfMessage_ReleaseAndSwapResult> pmCoreIfMessage)
   {
      ETG_TRACE_USR1(("handlePmCoreIfMessage_ReleaseAndSwapResult"));

      // Posting result to corresponding controller
      _callController.releaseAndSwapResult(pmCoreIfMessage);
   }

   void PmCoreMainController::handlePmCoreIfMessage_HoldAndAcceptResult(
         std::shared_ptr<PmCoreIfMessage_HoldAndAcceptResult> pmCoreIfMessage)
   {
      ETG_TRACE_USR1(("handlePmCoreIfMessage_HoldAndAcceptResult"));

      // Posting result to corresponding controller
      _callController.holdAndAcceptResult(pmCoreIfMessage);
   }

   void PmCoreMainController::handlePmCoreIfMessage_HangupAllResult(
         std::shared_ptr<PmCoreIfMessage_HangupAllResult> pmCoreIfMessage)
   {
      ETG_TRACE_USR1(("handlePmCoreIfMessage_HangupAllResult"));

      // Posting result to corresponding controller
      _callController.hangupAllResult(pmCoreIfMessage);
   }

   void PmCoreMainController::handlePmCoreIfMessage_SplitCallsResult(
         std::shared_ptr<PmCoreIfMessage_SplitCallsResult> pmCoreIfMessage)
   {
      ETG_TRACE_USR1(("handlePmCoreIfMessage_SplitCallsResult"));

      // Posting result to corresponding controller
      _callController.splitCallsResult(pmCoreIfMessage);
   }

   void PmCoreMainController::handlePmCoreIfMessage_MergeCallsResult(
         std::shared_ptr<PmCoreIfMessage_MergeCallsResult> pmCoreIfMessage)
   {
      ETG_TRACE_USR1(("handlePmCoreIfMessage_MergeCallsResult"));

      // Posting result to corresponding controller
      _callController.mergeCallsResult(pmCoreIfMessage);
   }

   void PmCoreMainController::handlePmCoreIfMessage_HangupMultipartyResult(
         std::shared_ptr<PmCoreIfMessage_HangupMultipartyResult> pmCoreIfMessage)
   {
      ETG_TRACE_USR1(("handlePmCoreIfMessage_HangupMultipartyResult"));

      // Posting result to corresponding controller
      _callController.hangupMultipartyResult(pmCoreIfMessage);
   }

   void PmCoreMainController::handlePmCoreIfMessage_SendDTMFResult(
         std::shared_ptr<PmCoreIfMessage_SendDTMFResult> pmCoreIfMessage)
   {
      ETG_TRACE_USR1(("handlePmCoreIfMessage_SendDTMFResult"));

      // Posting result to corresponding controller
      _callController.sendDTMFResult(pmCoreIfMessage);
   }

   void PmCoreMainController::handlePmCoreIfMessage_HangupCallResult(
         std::shared_ptr<PmCoreIfMessage_HangupCallResult> pmCoreIfMessage)
   {
      ETG_TRACE_USR1(("handlePmCoreIfMessage_HangupCallResult"));

      // Posting result to corresponding controller
      _callController.hangupCallResult(pmCoreIfMessage);
   }

   void PmCoreMainController::handlePmCoreIfMessage_AcceptCallResult(
         std::shared_ptr<PmCoreIfMessage_AcceptCallResult> pmCoreIfMessage)
   {
      ETG_TRACE_USR1(("handlePmCoreIfMessage_AcceptCallResult"));

      // Posting result to corresponding controller
      _callController.acceptCallResult(pmCoreIfMessage);
   }

   void PmCoreMainController::handlePmCoreIfMessage_HoldIncomingCallResult(
         std::shared_ptr<PmCoreIfMessage_HoldIncomingCallResult> pmCoreIfMessage)
   {
      ETG_TRACE_USR1(("handlePmCoreIfMessage_HoldIncomingCallResult"));

      // Posting result to corresponding controller
      _callController.holdIncomingCallResult(pmCoreIfMessage);
   }

   void PmCoreMainController::handlePmCoreIfMessage_TransferAudioResult(
         std::shared_ptr<PmCoreIfMessage_TransferAudioResult> pmCoreIfMessage)
   {
      ETG_TRACE_USR1(("handlePmCoreIfMessage_TransferAudioResult"));

      // Posting result to corresponding controller
      _callController.transferAudioResult(pmCoreIfMessage);
   }

   void PmCoreMainController::handlePmCoreIfMessage_StartStopVoiceRecognitionResult(
         std::shared_ptr<PmCoreIfMessage_StartStopVoiceRecognitionResult> pmCoreIfMessage)
   {
      ETG_TRACE_USR1(("handlePmCoreIfMessage_StartStopVoiceRecognitionResult"));

      // Posting result to corresponding controller
      _vrController.startStopVoiceRecognitionResult(pmCoreIfMessage->getBdAddress(), pmCoreIfMessage->getVrValue(),
            pmCoreIfMessage->getBTSResult());
   }

   void PmCoreMainController::handlePmCoreIfMessage_PrepareAudioRouteResponse(
         IN std::shared_ptr<PmCoreIfMessage_PrepareAudioRouteResponse> pmCoreIfMessage)
   {
      ETG_TRACE_USR4(("handlePmCoreIfMessage_PrepareAudioRouteResponse"));

      if (CALL_CONTROLLER == pmCoreIfMessage->getControllerOrigin())
      {
         _callController.prepareAudioRouteResponse(pmCoreIfMessage);
      }
      else if (VR_CONTROLLER == pmCoreIfMessage->getControllerOrigin())
      {
         _vrController.prepareAudioRouteResponse(pmCoreIfMessage);
      }
      else
      {
         // Origin is Main Controller. The response need to be handled here.
      }
   }

   void PmCoreMainController::handlePmCoreIfMessage_PlayAudioResponse(
               IN std::shared_ptr<PmCoreIfMessage_PlayAudioResponse> pmCoreIfMessage)
   {
      ETG_TRACE_USR4(("handlePmCoreIfMessage_PlayAudioResponse"));

      if (CALL_CONTROLLER == pmCoreIfMessage->getControllerOrigin())
      {
         _callController.playAudioResponse(pmCoreIfMessage);
      }
      else if (VR_CONTROLLER == pmCoreIfMessage->getControllerOrigin())
      {
         _vrController.playAudioResponse(pmCoreIfMessage);
      }
      else
      {
         // Origin is Main Controller. The response need to be handled here.
      }
   }

   void PmCoreMainController::handlePmCoreIfMessage_StopAudioResponse(
               IN std::shared_ptr<PmCoreIfMessage_StopAudioResponse> pmCoreIfMessage)
   {
      ETG_TRACE_USR4(("handlePmCoreIfMessage_StopAudioResponse"));

      if (CALL_CONTROLLER == pmCoreIfMessage->getControllerOrigin())
      {
         _callController.stopAudioResponse(pmCoreIfMessage);
      }
      else if (VR_CONTROLLER == pmCoreIfMessage->getControllerOrigin())
      {
         _vrController.stopAudioResponse(pmCoreIfMessage);
      }
      else
      {
         // Origin is Main Controller. The response need to be handled here.
      }
   }

   void PmCoreMainController::handlePmCoreIfMessage_SetMicMuteStateResponse(
               IN std::shared_ptr<PmCoreIfMessage_SetMicMuteStateResponse> pmCoreIfMessage)
   {
      ETG_TRACE_USR4(("handlePmCoreIfMessage_SetMicMuteStateResponse"));

      if (CALL_CONTROLLER == pmCoreIfMessage->getControllerOrigin())
      {
         _callController.setMicMuteStateResponse(pmCoreIfMessage);
      }
      else if (MAIN_CONTROLLER == pmCoreIfMessage->getControllerOrigin())
      {
         // Origin is Main Controller. The response need to be handled here.
      }
   }

   void PmCoreMainController::handlePmCoreIfMessage_PauseAudioResponse(
               IN std::shared_ptr<PmCoreIfMessage_PauseAudioResponse> pmCoreIfMessage)
   {
      ETG_TRACE_USR4(("handlePmCoreIfMessage_PauseAudioResponse"));

      if (CALL_CONTROLLER == pmCoreIfMessage->getControllerOrigin())
      {
         _callController.pauseAudioResponse(pmCoreIfMessage);
      }
      else if (MAIN_CONTROLLER == pmCoreIfMessage->getControllerOrigin())
      {
         // Origin is Main Controller. The response need to be handled here.
      }
   }

   void PmCoreMainController::postResponseForRequest(IN const std::string& operation, IN const PmResult pmResult,
         IN const BdAddress deviceAddress, IN const CallInstance callInstance,
         IN const ActType act)
   {
      ETG_TRACE_USR4(("postResponseForRequest"));

      if (operation == "AcceptCall")
      {
         _pmCoreCallbackIfWrapper->doAcceptCallResponse(pmResult, deviceAddress, callInstance, act);
      }
      else if (operation == "StartWaitingMode")
      {
         _pmCoreCallbackIfWrapper->doStartStopWaitingModeResponse(pmResult, deviceAddress, act);
      }
      else if (operation == "StartVoiceRecognition")
      {
         _pmCoreCallbackIfWrapper->doStartStopVoiceRecognitionResponse(pmResult, deviceAddress, act);
      }
   }
} // namespace pmcore
