/**
 * @file PmCoreRequestIf.cpp
 *
 * @swcomponent PhoneCallManager
 *
 * @brief This file contains the definition of the PmInterface class
 *
 * @copyright (C) 2016 Robert Bosch GmbH.
 *            The reproduction, distribution and utilization of this file as
 *            well as the communication of its contents to others without express
 *            authorization is prohibited. Offenders will be held liable for the
 *            payment of damages. All rights reserved in the event of the grant
 *            of a patent, utility model or design.
 *
 * @details This file provides the CPP request interfaces of PM Core.
 *
 * @ingroup PmCore
 */

#include "PmCoreRequestIf.h"
#include "PmCoreIfMessageCreator.h"
#include "PmCoreMainController.h"
#include "EvoBtStackWrapper.h"
#include "FwBluetoothStringUtils.h"
#include "PropertyUpdateNotifierToCore.h"
#include "PmAudioManagerWrapper.h"
#include "ahl_fw.h"
#include "PmAppTrace.h"

#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#define ETG_DEFAULT_TRACE_CLASS TR_CLASS_PM_CORE
#ifdef VARIANT_S_FTR_ENABLE_FW_ETG_USAGE
#include "trcGenProj/Header/PmCoreRequestIf.cpp.trc.h"
#define ETG_DEFAULT_TRACE_CLASS TR_CLASS_PM_CORE
#endif
#endif

namespace pmcore
{
   PmInterface::PmInterface()
   {
      ETG_TRACE_USR1(("PmInterface"));
   }

   PmInterface::~PmInterface()
   {
      ETG_TRACE_USR1(("~PmInterface"));
   }

   PmResult PmInterface::registerPmCoreCallbackIf(IPmCoreCallbackIf* pmCoreCallbackIf)
   {
      ETG_TRACE_USR1(("registerPmCoreCallbackIf: pmCoreCallbackIf = 0x%p", (void *) pmCoreCallbackIf));

      PmResult pmResult(PM_RESULT_OK, "");

      pmResult = PmCoreMainController::getInstance().registerCallback(pmCoreCallbackIf);

      if (PM_RESULT_OK == pmResult._pmResultCode)
      {
         ETG_TRACE_USR1(("registerPmCoreCallbackIf: pmResult._pmResultCode = %d", pmResult._pmResultCode));
      }

      return pmResult;
   }

   PmResult PmInterface::initializePmCore(ahl_tclBaseOneThreadApp* mainApplication)
   {
      ETG_TRACE_USR1(("initializePmCore"));

      PmResult pmResult(PM_RESULT_OK, "");

      //TODO: When this method is called to be checked
      //TODO sequence needs to be implemented.
      // All singleton classes are instantiated here.
      PropertyUpdateNotifierToCore::getInstance();
      PmCoreMainController::getInstance();
      evobtstackwrapper::EvoBtStackWrapper::getInstance();

      // Instantiation of PM AudioManager wrapper and preparing the PM Audio-manager for further communications.
      pmaudiomanager::AmResult amResult;
      amResult = PmAudioManagerWrapper::getInstance().preparePmAudioManager(mainApplication);

      if (pmaudiomanager::AM_RESULT_OK != amResult._amResultCode)
      {
         pmResult._pmResultCode = PM_RESULT_ERR_GENERAL;
         pmResult._pmResultMessage = amResult._amResultMessage;
      }

      return pmResult;
   }

   PmResult PmInterface::setSystemState(IN const SystemState systemState)
   {
      PmResult pmResult(PM_RESULT_OK, "");

      std::shared_ptr<PmCoreIfMessage> pmCoreIfMessage = getNewPmCoreIfMessage_SetSystemState(systemState);
      pmCoreIfMessage->traceMessage();
      PmCoreMainController::getInstance().handlePmCoreIfMessage(pmCoreIfMessage);

      return pmResult;
   }

   PmResult PmInterface::setVehicleConfiguration(IN const VehicleConfiguration& vehicleConfig)
   {
      PmResult pmResult(PM_RESULT_OK, "");

      // device address validity check
      if (false == fw::isValidBdAddress(vehicleConfig._vehicleBTAddress, false, false))
      {
         ETG_TRACE_ERR(("setVehicleConfiguration: invalid bdAddress"));
         pmResult._pmResultCode = PM_RESULT_ERR_INVALID_PARAMETER;
      }
      else
      {
         std::shared_ptr<PmCoreIfMessage> pmCoreIfMessage = getNewPmCoreIfMessage_SetVehicleConfiguration(vehicleConfig);
         pmCoreIfMessage->traceMessage();
         PmCoreMainController::getInstance().handlePmCoreIfMessage(pmCoreIfMessage);
      }

      return pmResult;
   }

   PmResult PmInterface::onDevicePaired(IN const BasicDeviceDetails& basicDeviceDetails)
   {
      ETG_TRACE_USR4(("onDevicePaired"));
      
      PmResult pmResult(PM_RESULT_OK, "");

      // device address validity check
      if (false == fw::isValidBdAddress(basicDeviceDetails._deviceAddress, false, false))
      {
         ETG_TRACE_ERR(("onDevicePaired: invalid bdAddress"));
         pmResult._pmResultCode = PM_RESULT_ERR_INVALID_PARAMETER;
      }
      else
      {
         std::shared_ptr<PmCoreIfMessage> pmCoreIfMessage = getNewPmCoreIfMessage_OnDevicePaired(basicDeviceDetails);
         pmCoreIfMessage->traceMessage();
         PmCoreMainController::getInstance().handlePmCoreIfMessage(pmCoreIfMessage);
      }

      return pmResult;
   }

   PmResult PmInterface::onDeviceConnected(IN const BasicDeviceDetails& basicDeviceDetails)
   {
      ETG_TRACE_USR4(("onDeviceConnected"));

      PmResult pmResult(PM_RESULT_OK, "");

      // device address validity check
      if (false == fw::isValidBdAddress(basicDeviceDetails._deviceAddress, false, false))
      {
         ETG_TRACE_ERR(("onDeviceConnected: invalid bdAddress"));
         pmResult._pmResultCode = PM_RESULT_ERR_INVALID_PARAMETER;
      }
      else
      {
         std::shared_ptr<PmCoreIfMessage> pmCoreIfMessage = getNewPmCoreIfMessage_OnDeviceConnected(basicDeviceDetails);
         pmCoreIfMessage->traceMessage();
         PmCoreMainController::getInstance().handlePmCoreIfMessage(pmCoreIfMessage);
      }

      return pmResult;
   }

   PmResult PmInterface::onDeviceDisconnected(IN const BdAddress& bdAddress)
   {
      ETG_TRACE_USR4(("onDeviceDisconnected"));

      PmResult pmResult(PM_RESULT_OK, "");

      // device address validity check
      if (false == fw::isValidBdAddress(bdAddress, false, false))
      {
         ETG_TRACE_ERR(("onDeviceDisconnected: invalid bdAddress"));
         pmResult._pmResultCode = PM_RESULT_ERR_INVALID_PARAMETER;
      }
      else
      {
         std::shared_ptr<PmCoreIfMessage> pmCoreIfMessage = getNewPmCoreIfMessage_OnDeviceDisconnected(bdAddress);
         pmCoreIfMessage->traceMessage();
         PmCoreMainController::getInstance().handlePmCoreIfMessage(pmCoreIfMessage);
      }

      return pmResult;
   }

   PmResult PmInterface::onDeviceDeleted(IN const BdAddress& bdAddress)
   {
      ETG_TRACE_USR4(("onDeviceDeleted"));

      PmResult pmResult(PM_RESULT_OK, "");

      // device address validity check
      if ((false == fw::isValidBdAddress(bdAddress, false, false)) && (0 != bdAddress.compare(DEVICE_ADDRESS_ALL)))
      {
         ETG_TRACE_ERR(("onDeviceDeleted: invalid bdAddress"));
         pmResult._pmResultCode = PM_RESULT_ERR_INVALID_PARAMETER;
      }
      else
      {
         std::shared_ptr<PmCoreIfMessage> pmCoreIfMessage = getNewPmCoreIfMessage_OnDeviceDeleted(bdAddress);
         pmCoreIfMessage->traceMessage();
         PmCoreMainController::getInstance().handlePmCoreIfMessage(pmCoreIfMessage);
      }

      return pmResult;
   }

   PmResult PmInterface::onDeviceIdentificationUpdated(IN const BasicDeviceDetails& basicDeviceDetails)
   {
      ETG_TRACE_USR4(("onDeviceIdentificationUpdated"));

      PmResult pmResult(PM_RESULT_OK, "");

      //TODO sequence needs to be implemented.
      (void)basicDeviceDetails; //To remove gen4 warning

      return pmResult;
   }

   PmResult PmInterface::getSystemWideRingtonesList(IN const ActType act)
   {
      ETG_TRACE_USR4(("getSystemWideRingtonesList"));

      PmResult pmResult(PM_RESULT_OK, "");

      std::shared_ptr<PmCoreIfMessage> pmCoreIfMessage = getNewPmCoreIfMessage_GetSystemWideRingtoneListRequest(act);
      pmCoreIfMessage->traceMessage();
      PmCoreMainController::getInstance().handlePmCoreIfMessage(pmCoreIfMessage);

      return pmResult;
   }

   PmResult PmInterface::getDeviceRingtoneList(IN const ActType act)
   {
      ETG_TRACE_USR4(("getDeviceRingtoneList"));

      PmResult pmResult(PM_RESULT_OK, "");

      std::shared_ptr<PmCoreIfMessage> pmCoreIfMessage = getNewPmCoreIfMessage_GetDeviceRingtoneListRequest(act);
      pmCoreIfMessage->traceMessage();
      PmCoreMainController::getInstance().handlePmCoreIfMessage(pmCoreIfMessage);

      return pmResult;
   }

   PmResult PmInterface::setRingtone(IN const BdAddress& bdAddress, IN const RingtoneId ringtoneId,
         IN const ActType act)
   {
      ETG_TRACE_USR4(("setRingtone"));

      PmResult pmResult(PM_RESULT_OK, "");

      // device address validity check
      if ((false == fw::isValidBdAddress(bdAddress, false, false)) && (0 != bdAddress.compare(DEVICE_ADDRESS_ALL)))
      {
         ETG_TRACE_ERR(("setRingtone: invalid bdAddress"));
         pmResult._pmResultCode = PM_RESULT_ERR_INVALID_PARAMETER;
      }
      else
      {
         std::shared_ptr<PmCoreIfMessage> pmCoreIfMessage = getNewPmCoreIfMessage_SetRingtoneRequest(
               bdAddress, ringtoneId, act);
         pmCoreIfMessage->traceMessage();
         PmCoreMainController::getInstance().handlePmCoreIfMessage(pmCoreIfMessage);
      }

      return pmResult;
   }

   PmResult PmInterface::playRingtone(IN const RingtoneId ringtoneId, IN const PlayCount playCount,
         IN const ActType act)
   {
      ETG_TRACE_USR4(("playRingtone"));

      PmResult pmResult(PM_RESULT_OK, "");

      RingtoneId ringtoneIdLocal = ringtoneId;

      //TODO Need to recheck implementation of ringtone playback when active system ringtone id is RINGTONE_ID_INBAND
      if(RINGTONE_ID_INBAND == ringtoneId)
      {
          ETG_TRACE_USR4(("Playback for Inband Ringtone not Supported. Playing the System Default Ringtone."));
          ringtoneIdLocal = RINGTONE_ID_SYSTEM_DEFAULT;
      }

      std::shared_ptr<PmCoreIfMessage> pmCoreIfMessage = getNewPmCoreIfMessage_PlayRingtoneRequest(
            DEVICE_ADDRESS_ALL, ringtoneIdLocal, playCount, act, PM_CORE_IF_MSG_ORIGIN_CLIENT);
      pmCoreIfMessage->traceMessage();
      PmCoreMainController::getInstance().handlePmCoreIfMessage(pmCoreIfMessage);

      return pmResult;
   }

   PmResult PmInterface::stopRingtone(IN const ActType act)
   {
      ETG_TRACE_USR4(("stopRingtone"));

      PmResult pmResult(PM_RESULT_OK, "");

      std::shared_ptr<PmCoreIfMessage> pmCoreIfMessage = getNewPmCoreIfMessage_StopRingtoneRequest(DEVICE_ADDRESS_ALL,
            act, PM_CORE_IF_MSG_ORIGIN_CLIENT);
      pmCoreIfMessage->traceMessage();
      PmCoreMainController::getInstance().handlePmCoreIfMessage(pmCoreIfMessage);

      return pmResult;
   }

   PmResult PmInterface::getSuppressRingtoneOnOffList(IN const ActType act)
   {
      PmResult pmResult(PM_RESULT_OK, "");

      std::shared_ptr<PmCoreIfMessage> pmCoreIfMessage = getNewPmCoreIfMessage_GetSuppressRingtoneOnOffListRequest(
            act);
      pmCoreIfMessage->traceMessage();
      PmCoreMainController::getInstance().handlePmCoreIfMessage(pmCoreIfMessage);

      return pmResult;
   }

   PmResult PmInterface::suppressRingtoneOnOff(IN const BdAddress& bdAddress,
         IN const SuppressRingtoneState suppressRingtoneState, IN const ActType act)
   {
      PmResult pmResult(PM_RESULT_OK, "");

      // device address validity check
      if ((false == fw::isValidBdAddress(bdAddress, false, false)) && (0 != bdAddress.compare(DEVICE_ADDRESS_ALL)))
      {
         ETG_TRACE_ERR(("suppressRingtoneOnOff: invalid bdAddress"));
         pmResult._pmResultCode = PM_RESULT_ERR_INVALID_PARAMETER;
      }
      else
      {
         std::shared_ptr<PmCoreIfMessage> pmCoreIfMessage = getNewPmCoreIfMessage_SuppressRingtoneOnOffRequest(
               bdAddress, suppressRingtoneState, act);
         pmCoreIfMessage->traceMessage();
         PmCoreMainController::getInstance().handlePmCoreIfMessage(pmCoreIfMessage);
      }

      return pmResult;
   }

   PmResult PmInterface::getAutoWaitingModeOnOffList(IN const ActType act)
   {
      PmResult pmResult(PM_RESULT_OK, "");

      std::shared_ptr<PmCoreIfMessage> pmCoreIfMessage = getNewPmCoreIfMessage_GetAutoWaitingModeOnOffListRequest(
            act);
      pmCoreIfMessage->traceMessage();
      PmCoreMainController::getInstance().handlePmCoreIfMessage(pmCoreIfMessage);

      return pmResult;
   }

   PmResult PmInterface::autoWaitingModeOnOff(IN const BdAddress& bdAddress,
         IN const AutoWaitingModeState autoWaitingModeState, IN const ActType act)
   {
      PmResult pmResult(PM_RESULT_OK, "");

      // device address validity check
      if ((false == fw::isValidBdAddress(bdAddress, false, false)) && (0 != bdAddress.compare(DEVICE_ADDRESS_ALL)))
      {
         ETG_TRACE_ERR(("autoWaitingModeOnOff: invalid bdAddress"));
         pmResult._pmResultCode = PM_RESULT_ERR_INVALID_PARAMETER;
      }
      else
      {
         std::shared_ptr<PmCoreIfMessage> pmCoreIfMessage = getNewPmCoreIfMessage_AutoWaitingModeOnOffRequest(
               bdAddress, autoWaitingModeState, act);
         pmCoreIfMessage->traceMessage();
         PmCoreMainController::getInstance().handlePmCoreIfMessage(pmCoreIfMessage);
      }

      return pmResult;
   }

   PmResult PmInterface::setWaitingModeFilePath(IN const WaitingModeFilePath& waitingModeFilePath, IN const ActType act)
   {
      PmResult pmResult(PM_RESULT_OK, "");

      std::shared_ptr<PmCoreIfMessage> pmCoreIfMessage = getNewPmCoreIfMessage_SetWaitingModeFilePathRequest(
            waitingModeFilePath, act);
      pmCoreIfMessage->traceMessage();
      PmCoreMainController::getInstance().handlePmCoreIfMessage(pmCoreIfMessage);

      return pmResult;
   }

   PmResult PmInterface::getWaitingModeFilePath(IN const ActType act)
   {
      PmResult pmResult(PM_RESULT_OK, "");

      std::shared_ptr<PmCoreIfMessage> pmCoreIfMessage = getNewPmCoreIfMessage_GetWaitingModeFilePathRequest(act);
      pmCoreIfMessage->traceMessage();
      PmCoreMainController::getInstance().handlePmCoreIfMessage(pmCoreIfMessage);

      return pmResult;
   }

   PmResult PmInterface::getPhoneCallAudioActive(IN const ActType act)
   {
      PmResult pmResult(PM_RESULT_OK, "");

      std::shared_ptr<PmCoreIfMessage> pmCoreIfMessage = getNewPmCoreIfMessage_GetPhoneCallAudioActiveRequest(act);
      pmCoreIfMessage->traceMessage();
      PmCoreMainController::getInstance().handlePmCoreIfMessage(pmCoreIfMessage);

      return pmResult;
   }

   PmResult PmInterface::getActivePassiveDeviceList(IN const ActType act)
   {
      PmResult pmResult(PM_RESULT_OK, "");

      std::shared_ptr<PmCoreIfMessage> pmCoreIfMessage = getNewPmCoreIfMessage_GetActivePassiveDeviceListRequest(
            act);
      pmCoreIfMessage->traceMessage();
      PmCoreMainController::getInstance().handlePmCoreIfMessage(pmCoreIfMessage);

      return pmResult;
   }

   PmResult PmInterface::setActivePhone(IN const BdAddressList& deviceAddressList, IN const ActType act)
   {
      PmResult pmResult(PM_RESULT_OK, "");

      BdAddressList::const_iterator it;
      for (it = deviceAddressList.begin(); it != deviceAddressList.end(); ++it)
      {
         // device address validity check
         if (false == fw::isValidBdAddress(*it, false, false))
         {
            ETG_TRACE_ERR(("setActivePhone: invalid bdAddress"));
            pmResult._pmResultCode = PM_RESULT_ERR_INVALID_PARAMETER;
         }
      }

      if(PM_RESULT_OK == pmResult._pmResultCode)
      {
         std::shared_ptr<PmCoreIfMessage> pmCoreIfMessage = getNewPmCoreIfMessage_SetActivePhoneRequest(
               deviceAddressList, act);
         pmCoreIfMessage->traceMessage();
         PmCoreMainController::getInstance().handlePmCoreIfMessage(pmCoreIfMessage);
      }

      return pmResult;
   }

   PmResult PmInterface::restoreDefaultSetting(IN const BdAddress& bdAddress, IN const ActType act)
   {
      PmResult pmResult(PM_RESULT_OK, "");

      // device address validity check
      if ((false == fw::isValidBdAddress(bdAddress, false, false)) && (0 != bdAddress.compare(DEVICE_ADDRESS_ALL) &&
            (0 != bdAddress.compare(DEVICE_ADDRESS_DELETE_ALL))))
      {
         ETG_TRACE_ERR(("restoreDefaultSetting: invalid bdAddress"));
         pmResult._pmResultCode = PM_RESULT_ERR_INVALID_PARAMETER;
      }
      else
      {
         std::shared_ptr<PmCoreIfMessage> pmCoreIfMessage = getNewPmCoreIfMessage_RestoreDefaultSettingRequest(
               bdAddress, act);
         pmCoreIfMessage->traceMessage();
         PmCoreMainController::getInstance().handlePmCoreIfMessage(pmCoreIfMessage);
      }

      return pmResult;
   }

   PmResult PmInterface::getSupportedFeatures(IN const BdAddress& bdAddress, IN const ActType act)
   {
      PmResult pmResult(PM_RESULT_OK, "");

      // device address validity check
      if (false == fw::isValidBdAddress(bdAddress, false, false))
      {
         ETG_TRACE_ERR(("getSupportedFeatures: invalid bdAddress"));
         pmResult._pmResultCode = PM_RESULT_ERR_INVALID_PARAMETER;
      }
      else
      {
         std::shared_ptr<PmCoreIfMessage> pmCoreIfMessage = getNewPmCoreIfMessage_GetSupportedFeaturesRequest(
               bdAddress, act);
         pmCoreIfMessage->traceMessage();
         PmCoreMainController::getInstance().handlePmCoreIfMessage(pmCoreIfMessage);
      }

      return pmResult;
   }

   PmResult PmInterface::getNetworkStatus(IN const BdAddress& bdAddress, IN const ActType act)
   {
      PmResult pmResult(PM_RESULT_OK, "");

      // device address validity check
      if (false == fw::isValidBdAddress(bdAddress, false, false))
      {
         ETG_TRACE_ERR(("getNetworkStatus: invalid bdAddress"));
         pmResult._pmResultCode = PM_RESULT_ERR_INVALID_PARAMETER;
      }
      else
      {
         std::shared_ptr<PmCoreIfMessage> pmCoreIfMessage = getNewPmCoreIfMessage_GetNetworkStatusRequest(
               bdAddress, act);
         pmCoreIfMessage->traceMessage();
         PmCoreMainController::getInstance().handlePmCoreIfMessage(pmCoreIfMessage);
      }

      return pmResult;
   }

   PmResult PmInterface::getSignalStrength(IN const BdAddress& bdAddress, IN const ActType act)
   {
      PmResult pmResult(PM_RESULT_OK, "");

      // device address validity check
      if (false == fw::isValidBdAddress(bdAddress, false, false))
      {
         ETG_TRACE_ERR(("getSignalStrength: invalid bdAddress"));
         pmResult._pmResultCode = PM_RESULT_ERR_INVALID_PARAMETER;
      }
      else
      {
         std::shared_ptr<PmCoreIfMessage> pmCoreIfMessage = getNewPmCoreIfMessage_GetSignalStrengthRequest(
               bdAddress, act);
         pmCoreIfMessage->traceMessage();
         PmCoreMainController::getInstance().handlePmCoreIfMessage(pmCoreIfMessage);
      }

      return pmResult;
   }

   PmResult PmInterface::getBatteryChargeLevel(IN const BdAddress& bdAddress, IN const ActType act)
   {
      PmResult pmResult(PM_RESULT_OK, "");

      // device address validity check
      if (false == fw::isValidBdAddress(bdAddress, false, false))
      {
         ETG_TRACE_ERR(("getBatteryChargeLevel: invalid bdAddress"));
         pmResult._pmResultCode = PM_RESULT_ERR_INVALID_PARAMETER;
      }
      else
      {
         std::shared_ptr<PmCoreIfMessage> pmCoreIfMessage = getNewPmCoreIfMessage_GetBatteryChargeLevelRequest(
               bdAddress, act);
         pmCoreIfMessage->traceMessage();
         PmCoreMainController::getInstance().handlePmCoreIfMessage(pmCoreIfMessage);
      }

      return pmResult;
   }

   PmResult PmInterface::getNetworkOperator(IN const BdAddress& bdAddress, IN const ActType act)
   {
      PmResult pmResult(PM_RESULT_OK, "");

      // device address validity check
      if (false == fw::isValidBdAddress(bdAddress, false, false))
      {
         ETG_TRACE_ERR(("getNetworkOperator: invalid bdAddress"));
         pmResult._pmResultCode = PM_RESULT_ERR_INVALID_PARAMETER;
      }
      else
      {
         std::shared_ptr<PmCoreIfMessage> pmCoreIfMessage = getNewPmCoreIfMessage_GetNetworkOperatorRequest(
               bdAddress, act);
         pmCoreIfMessage->traceMessage();
         PmCoreMainController::getInstance().handlePmCoreIfMessage(pmCoreIfMessage);
      }

      return pmResult;
   }

   PmResult PmInterface::voiceMailList(IN const BdAddress& bdAddress, IN const ActType act)
   {
      PmResult pmResult(PM_RESULT_OK, "");

      // device address validity check
      if (false == fw::isValidBdAddress(bdAddress, false, false))
      {
         ETG_TRACE_ERR(("voiceMailList: invalid bdAddress"));
         pmResult._pmResultCode = PM_RESULT_ERR_INVALID_PARAMETER;
      }
      else
      {
         std::shared_ptr<PmCoreIfMessage> pmCoreIfMessage = getNewPmCoreIfMessage_VoiceMailListRequest(
               bdAddress, act);
         pmCoreIfMessage->traceMessage();
         PmCoreMainController::getInstance().handlePmCoreIfMessage(pmCoreIfMessage);
      }

      return pmResult;
   }

   PmResult PmInterface::getCallStatusList(IN const ActType act)
   {
      PmResult pmResult(PM_RESULT_OK, "");

      std::shared_ptr<PmCoreIfMessage> pmCoreIfMessage = getNewPmCoreIfMessage_GetCallStatusListRequest(
            act);
      pmCoreIfMessage->traceMessage();
      PmCoreMainController::getInstance().handlePmCoreIfMessage(pmCoreIfMessage);

      return pmResult;
   }

   PmResult PmInterface::getMicrophoneMuteState(IN const ActType act)
   {
      PmResult pmResult(PM_RESULT_OK, "");

      std::shared_ptr<PmCoreIfMessage> pmCoreIfMessage = getNewPmCoreIfMessage_GetMicrophoneMuteStateRequest(
            act);
      pmCoreIfMessage->traceMessage();
      PmCoreMainController::getInstance().handlePmCoreIfMessage(pmCoreIfMessage);

      return pmResult;
   }

   PmResult PmInterface::setMicrophoneMuteState(IN const MuteState microphoneMuteState, IN const ActType act)
   {
      PmResult pmResult(PM_RESULT_OK, "");

      std::shared_ptr<PmCoreIfMessage> pmCoreIfMessage = getNewPmCoreIfMessage_SetMicrophoneMuteStateRequest(
            microphoneMuteState, act);
      pmCoreIfMessage->traceMessage();
      PmCoreMainController::getInstance().handlePmCoreIfMessage(pmCoreIfMessage);

      return pmResult;
   }

   PmResult PmInterface::mergeCalls(IN const DeviceCallInstanceListMap& mergeCallList, IN const ActType act)
   {
      PmResult pmResult(PM_RESULT_OK, "");

      DeviceCallInstanceListMap::const_iterator it;
      for (it = mergeCallList.begin(); it != mergeCallList.end(); it++)
      {
         // device address validity check
         if (true == fw::isValidBdAddress(it->first, false, false))
         {
            //TODO check if logic need to be changed when at least one device address is not valid case
            std::shared_ptr<PmCoreIfMessage> pmCoreIfMessage = getNewPmCoreIfMessage_MergeCallsRequest(it->first,
                  it->second, act);
            pmCoreIfMessage->traceMessage();
            PmCoreMainController::getInstance().handlePmCoreIfMessage(pmCoreIfMessage);
         }
         else
         {
            ETG_TRACE_ERR(("mergeCalls: invalid bdAddress"));
            pmResult._pmResultCode = PM_RESULT_ERR_INVALID_PARAMETER;
         }
      }

      return pmResult;
   }

   PmResult PmInterface::splitCalls(IN const DeviceCallInstanceListMap& splitCallList, IN const ActType act)
   {
      PmResult pmResult(PM_RESULT_OK, "");

      DeviceCallInstanceListMap::const_iterator it;
      for (it = splitCallList.begin(); it != splitCallList.end(); it++)
      {
         // device address validity check
         if (true == fw::isValidBdAddress(it->first, false, false))
         {
            //TODO check if logic need to be changed when at least one device address is not valid case
            std::shared_ptr<PmCoreIfMessage> pmCoreIfMessage = getNewPmCoreIfMessage_SplitCallsRequest(
                  it->first, it->second, act);
            pmCoreIfMessage->traceMessage();
            PmCoreMainController::getInstance().handlePmCoreIfMessage(pmCoreIfMessage);
         }
         else
         {
            ETG_TRACE_ERR(("splitCalls: invalid bdAddress"));
            pmResult._pmResultCode = PM_RESULT_ERR_INVALID_PARAMETER;
         }
      }

      return pmResult;
   }

   PmResult PmInterface::hangupCalls(IN const DeviceCallInstanceListMap& hangupCallList, IN const ActType act)
   {
      PmResult pmResult(PM_RESULT_OK, "");

      DeviceCallInstanceListMap::const_iterator it;
      for (it = hangupCallList.begin(); it != hangupCallList.end(); it++)
      {
         // device address validity check
         if (true == fw::isValidBdAddress(it->first, false, false))
         {
            //TODO check if logic need to be changed when at least one device address is not valid case
            std::shared_ptr<PmCoreIfMessage> pmCoreIfMessage = getNewPmCoreIfMessage_HangupCallsRequest(
                  it->first, it->second, act);
            pmCoreIfMessage->traceMessage();
            PmCoreMainController::getInstance().handlePmCoreIfMessage(pmCoreIfMessage);
         }
         else
         {
            ETG_TRACE_ERR(("hangupCalls: invalid bdAddress"));
            pmResult._pmResultCode = PM_RESULT_ERR_INVALID_PARAMETER;
         }
      }

      return pmResult;
   }

   PmResult PmInterface::redial(IN const BdAddress& bdAddress, IN const HideCallerId hideCallerId,
         IN const ActType act)
   {
      PmResult pmResult(PM_RESULT_OK, "");

      // device address validity check
      if (false == fw::isValidBdAddress(bdAddress, false, false))
      {
         ETG_TRACE_ERR(("redial: invalid bdAddress"));
         pmResult._pmResultCode = PM_RESULT_ERR_INVALID_PARAMETER;
      }
      else
      {
         std::shared_ptr<PmCoreIfMessage> pmCoreIfMessage = getNewPmCoreIfMessage_RedialRequest(
               bdAddress, hideCallerId, act);
         pmCoreIfMessage->traceMessage();
         PmCoreMainController::getInstance().handlePmCoreIfMessage(pmCoreIfMessage);
      }

      return pmResult;
   }

   PmResult PmInterface::getSCOConnection(IN const BdAddress& bdAddress, IN const ActType act)
   {
      PmResult pmResult(PM_RESULT_OK, "");

      // device address validity check
      if (false == fw::isValidBdAddress(bdAddress, false, false))
      {
         ETG_TRACE_ERR(("getSCOConnection: invalid bdAddress"));
         pmResult._pmResultCode = PM_RESULT_ERR_INVALID_PARAMETER;
      }
      else
      {
         std::shared_ptr<PmCoreIfMessage> pmCoreIfMessage = getNewPmCoreIfMessage_GetSCOConnectionRequest(
               bdAddress, act);
         pmCoreIfMessage->traceMessage();
         PmCoreMainController::getInstance().handlePmCoreIfMessage(pmCoreIfMessage);
      }

      return pmResult;
   }

   PmResult PmInterface::getRingtoneMuteState(IN const BdAddress& bdAddress, IN const ActType act)
   {
      PmResult pmResult(PM_RESULT_OK, "");

      // device address validity check
      if (false == fw::isValidBdAddress(bdAddress, false, false))
      {
         ETG_TRACE_ERR(("getRingtoneMuteState: invalid bdAddress"));
         pmResult._pmResultCode = PM_RESULT_ERR_INVALID_PARAMETER;
      }
      else
      {
         std::shared_ptr<PmCoreIfMessage> pmCoreIfMessage = getNewPmCoreIfMessage_GetRingtoneMuteStateRequest(
               bdAddress, act);
         pmCoreIfMessage->traceMessage();
         PmCoreMainController::getInstance().handlePmCoreIfMessage(pmCoreIfMessage);
      }

      return pmResult;
   }

   PmResult PmInterface::setRingtoneMuteState(IN const BdAddress& bdAddress, IN const MuteState ringtoneMuteState,
         IN const ActType act)
   {
      PmResult pmResult(PM_RESULT_OK, "");

      // device address validity check
      if (false == fw::isValidBdAddress(bdAddress, false, false))
      {
         ETG_TRACE_ERR(("setRingtoneMuteState: invalid bdAddress"));
         pmResult._pmResultCode = PM_RESULT_ERR_INVALID_PARAMETER;
      }
      else
      {
         std::shared_ptr<PmCoreIfMessage> pmCoreIfMessage = getNewPmCoreIfMessage_SetRingtoneMuteStateRequest(
               bdAddress, ringtoneMuteState, act);
         pmCoreIfMessage->traceMessage();
         PmCoreMainController::getInstance().handlePmCoreIfMessage(pmCoreIfMessage);
      }

      return pmResult;
   }

   PmResult PmInterface::getWaitingModeState(IN const BdAddress& bdAddress, IN const ActType act)
   {
      PmResult pmResult(PM_RESULT_OK, "");

      // device address validity check
      if (false == fw::isValidBdAddress(bdAddress, false, false))
      {
         ETG_TRACE_ERR(("getWaitingModeState: invalid bdAddress"));
         pmResult._pmResultCode = PM_RESULT_ERR_INVALID_PARAMETER;
      }
      else
      {
         std::shared_ptr<PmCoreIfMessage> pmCoreIfMessage = getNewPmCoreIfMessage_GetWaitingModeStateRequest(
               bdAddress, act);
         pmCoreIfMessage->traceMessage();
         PmCoreMainController::getInstance().handlePmCoreIfMessage(pmCoreIfMessage);
      }

      return pmResult;
   }

   PmResult PmInterface::getRingtonePlaybackStatus(IN const ActType act)
   {
      PmResult pmResult(PM_RESULT_OK, "");

      std::shared_ptr<PmCoreIfMessage> pmCoreIfMessage = getNewPmCoreIfMessage_GetRingtonePlaybackStatusRequest(act);
      pmCoreIfMessage->traceMessage();
      PmCoreMainController::getInstance().handlePmCoreIfMessage(pmCoreIfMessage);

      return pmResult;
   }

   PmResult PmInterface::startStopWaitingMode(IN const BdAddress& bdAddress, IN const StartStop startStopWaitingMode,
         IN const ActType act)
   {
      PmResult pmResult(PM_RESULT_OK, "");

      // device address validity check
      if (false == fw::isValidBdAddress(bdAddress, false, false))
      {
         ETG_TRACE_ERR(("startStopWaitingMode: invalid bdAddress"));
         pmResult._pmResultCode = PM_RESULT_ERR_INVALID_PARAMETER;
      }
      else
      {
         std::shared_ptr<PmCoreIfMessage> pmCoreIfMessage = getNewPmCoreIfMessage_StartStopWaitingModeRequest(
               bdAddress, startStopWaitingMode, act);
         pmCoreIfMessage->traceMessage();
         PmCoreMainController::getInstance().handlePmCoreIfMessage(pmCoreIfMessage);
      }

      return pmResult;
   }

   PmResult PmInterface::dial(IN const BdAddress& bdAddress, IN const TelephoneNumber& telephoneNumber,
         IN const HideCallerId hideCallerId, IN const ActType act)
   {
      PmResult pmResult(PM_RESULT_OK, "");

      // device address validity check
      if (false == fw::isValidBdAddress(bdAddress, false, false))
      {
         ETG_TRACE_ERR(("dial: invalid bdAddress (%s)", bdAddress.c_str()));
         pmResult._pmResultCode = PM_RESULT_ERR_INVALID_PARAMETER;
      }
      else
      {
         std::shared_ptr<PmCoreIfMessage> pmCoreIfMessage = getNewPmCoreIfMessage_DialRequest(bdAddress,
               telephoneNumber, hideCallerId, act);
         pmCoreIfMessage->traceMessage();
         PmCoreMainController::getInstance().handlePmCoreIfMessage(pmCoreIfMessage);
      }

      return pmResult;
   }

   PmResult PmInterface::acceptCall(IN const BdAddress& bdAddress, IN const CallInstance callInstance,
         IN const AcceptOperation operation, IN const ActType act)
   {
      PmResult pmResult(PM_RESULT_OK, "");

      // device address validity check
      if (false == fw::isValidBdAddress(bdAddress, false, false))
      {
         ETG_TRACE_ERR(("acceptCall: invalid bdAddress"));
         pmResult._pmResultCode = PM_RESULT_ERR_INVALID_PARAMETER;
      }
      else
      {
         std::shared_ptr<PmCoreIfMessage> pmCoreIfMessage = getNewPmCoreIfMessage_AcceptCallRequest(
               bdAddress, callInstance, operation, act);
         pmCoreIfMessage->traceMessage();
         PmCoreMainController::getInstance().handlePmCoreIfMessage(pmCoreIfMessage);
      }

      return pmResult;
   }

   PmResult PmInterface::swapCall(IN const BdAddress& bdAddress, IN const SwapOperation operation,
         IN const ActType act)
   {
      PmResult pmResult(PM_RESULT_OK, "");

      // device address validity check
      if (false == fw::isValidBdAddress(bdAddress, false, false))
      {
         ETG_TRACE_ERR(("swapCall: invalid bdAddress"));
         pmResult._pmResultCode = PM_RESULT_ERR_INVALID_PARAMETER;
      }
      else
      {
         std::shared_ptr<PmCoreIfMessage> pmCoreIfMessage = getNewPmCoreIfMessage_SwapCallRequest(
               bdAddress, operation, act);
         pmCoreIfMessage->traceMessage();
         PmCoreMainController::getInstance().handlePmCoreIfMessage(pmCoreIfMessage);
      }

      return pmResult;
   }

   PmResult PmInterface::speedDial(IN const BdAddress& bdAddress, IN const MemDialIndex memDialIndex,
         IN const HideCallerId hideCallerId, IN const ActType act)
   {
      PmResult pmResult(PM_RESULT_OK, "");

      // device address validity check
      if (false == fw::isValidBdAddress(bdAddress, false, false))
      {
         ETG_TRACE_ERR(("speedDial: invalid bdAddress"));
         pmResult._pmResultCode = PM_RESULT_ERR_INVALID_PARAMETER;
      }
      else
      {
         std::shared_ptr<PmCoreIfMessage> pmCoreIfMessage = getNewPmCoreIfMessage_SpeedDialRequest(
               bdAddress, memDialIndex, hideCallerId, act);
         pmCoreIfMessage->traceMessage();
         PmCoreMainController::getInstance().handlePmCoreIfMessage(pmCoreIfMessage);
      }

      return pmResult;
   }

   PmResult PmInterface::sendDTMF(IN const BdAddress& bdAddress, IN const DTMFTones& tones, IN const ActType act)
   {
      PmResult pmResult(PM_RESULT_OK, "");

      // device address validity check
      if (false == fw::isValidBdAddress(bdAddress, false, false))
      {
         ETG_TRACE_ERR(("sendDTMF: invalid bdAddress"));
         pmResult._pmResultCode = PM_RESULT_ERR_INVALID_PARAMETER;
      }
      else
      {
         std::shared_ptr<PmCoreIfMessage> pmCoreIfMessage = getNewPmCoreIfMessage_SendDTMFRequest(
               bdAddress, tones, act);
         pmCoreIfMessage->traceMessage();
         PmCoreMainController::getInstance().handlePmCoreIfMessage(pmCoreIfMessage);
      }

      return pmResult;
   }

   PmResult PmInterface::transferAudio(IN const BdAddress& bdAddress, IN const AudioDirection audioDirection,
         IN const ActType act)
   {
      PmResult pmResult(PM_RESULT_OK, "");

      // device address validity check
      if (false == fw::isValidBdAddress(bdAddress, false, false))
      {
         ETG_TRACE_ERR(("transferAudio: invalid bdAddress"));
         pmResult._pmResultCode = PM_RESULT_ERR_INVALID_PARAMETER;
      }
      else
      {
         std::shared_ptr<PmCoreIfMessage> pmCoreIfMessage = getNewPmCoreIfMessage_TransferAudioRequest(
               bdAddress, audioDirection, act);
         pmCoreIfMessage->traceMessage();
         PmCoreMainController::getInstance().handlePmCoreIfMessage(pmCoreIfMessage);
      }

      return pmResult;
   }

   PmResult PmInterface::getVoiceRecognitionStatus(IN const BdAddress& bdAddress, IN const ActType act)
   {
      PmResult pmResult(PM_RESULT_OK, "");

      // device address validity check
      if (false == fw::isValidBdAddress(bdAddress, false, false))
      {
         ETG_TRACE_ERR(("getVoiceRecognitionStatus: invalid bdAddress"));
         pmResult._pmResultCode = PM_RESULT_ERR_INVALID_PARAMETER;
      }
      else
      {
         std::shared_ptr<PmCoreIfMessage> pmCoreIfMessage = getNewPmCoreIfMessage_GetVoiceRecognitionStatusRequest(
               bdAddress, act);
         pmCoreIfMessage->traceMessage();
         PmCoreMainController::getInstance().handlePmCoreIfMessage(pmCoreIfMessage);
      }

      return pmResult;
   }

   PmResult PmInterface::startStopVoiceRecognition(IN const BdAddress& bdAddress, IN const StartStop startStopVR,
         IN const ActType act)
   {
      PmResult pmResult(PM_RESULT_OK, "");

      // device address validity check
      if (false == fw::isValidBdAddress(bdAddress, false, false))
      {
         ETG_TRACE_ERR(("startStopVoiceRecognition: invalid bdAddress"));
         pmResult._pmResultCode = PM_RESULT_ERR_INVALID_PARAMETER;
      }
      else
      {
         std::shared_ptr<PmCoreIfMessage> pmCoreIfMessage = getNewPmCoreIfMessage_StartStopVoiceRecognitionRequest(
               bdAddress, startStopVR, act);
         pmCoreIfMessage->traceMessage();
         PmCoreMainController::getInstance().handlePmCoreIfMessage(pmCoreIfMessage);
      }

      return pmResult;
   }

   PmResult PmInterface::getEnhancedVoiceRecognitionFeature(IN const BdAddress& bdAddress, IN const ActType act)
   {
      PmResult pmResult(PM_RESULT_OK, "");

      // device address validity check
      if (false == fw::isValidBdAddress(bdAddress, false, false))
      {
         ETG_TRACE_ERR(("getEnhancedVoiceRecognitionFeature: invalid bdAddress"));
         pmResult._pmResultCode = PM_RESULT_ERR_INVALID_PARAMETER;
      }
      else
      {
         std::shared_ptr<PmCoreIfMessage> pmCoreIfMessage =
               getNewPmCoreIfMessage_GetEnhancedVoiceRecognitionFeatureRequest(bdAddress, act);
         pmCoreIfMessage->traceMessage();
         PmCoreMainController::getInstance().handlePmCoreIfMessage(pmCoreIfMessage);
      }

      return pmResult;
   }

   PmResult PmInterface::getExternalVRState(IN const BdAddress& deviceAddress, IN const ActType act)
   {
      PmResult pmResult(PM_RESULT_OK, "");
      (void)(act);

      // device address validity check
      if (false == fw::isValidBdAddress(deviceAddress, false, false))
      {
         ETG_TRACE_ERR(("getExternalVRState: invalid deviceAddress"));
         pmResult._pmResultCode = PM_RESULT_ERR_INVALID_PARAMETER;
      }
      else
      {
         std::shared_ptr<PmCoreIfMessage> pmCoreIfMessage = getNewPmCoreIfMessage_GetExtVoiceRecognitionStatusRequest(
               deviceAddress, act);
         pmCoreIfMessage->traceMessage();
         PmCoreMainController::getInstance().handlePmCoreIfMessage(pmCoreIfMessage);
      }

      return pmResult;
   }

   PmResult PmInterface::activateExternalVR(IN const BdAddress& deviceAddress, IN const ActType act)
   {
      PmResult pmResult(PM_RESULT_OK, "");
      (void)(act);

      // device address validity check
      if (false == fw::isValidBdAddress(deviceAddress, false, false))
      {
         ETG_TRACE_ERR(("activateExternalVR: invalid deviceAddress"));
         pmResult._pmResultCode = PM_RESULT_ERR_INVALID_PARAMETER;
      }
      else
      {
         std::shared_ptr<PmCoreIfMessage> pmCoreIfMessage =
               getNewPmCoreIfMessage_StartStopExternalVoiceRecognitionRequest(deviceAddress, true, act);
         pmCoreIfMessage->traceMessage();
         PmCoreMainController::getInstance().handlePmCoreIfMessage(pmCoreIfMessage);
      }

      return pmResult;
   }

   PmResult PmInterface::deActivateExternalVR(IN const BdAddress& deviceAddress, IN const ActType act)
   {
      PmResult pmResult(PM_RESULT_OK, "");
      (void)(act);

      // device address validity check
      if (false == fw::isValidBdAddress(deviceAddress, false, false))
      {
         ETG_TRACE_ERR(("deActivateExternalVR: invalid deviceAddress"));
         pmResult._pmResultCode = PM_RESULT_ERR_INVALID_PARAMETER;
      }
      else
      {
         std::shared_ptr<PmCoreIfMessage> pmCoreIfMessage =
               getNewPmCoreIfMessage_StartStopExternalVoiceRecognitionRequest(deviceAddress, false, act);
         pmCoreIfMessage->traceMessage();
         PmCoreMainController::getInstance().handlePmCoreIfMessage(pmCoreIfMessage);
      }

      return pmResult;
   }

   PmResult PmInterface::stopExtendedVR(IN const BdAddress& deviceAddress, IN const ActType act)
   {
      PmResult pmResult(PM_RESULT_OK, "");
      (void)(act);

      // device address validity check
      if (false == fw::isValidBdAddress(deviceAddress, false, false))
      {
         ETG_TRACE_ERR(("stopExtendedVR: invalid deviceAddress %s", deviceAddress));
         pmResult._pmResultCode = PM_RESULT_ERR_INVALID_PARAMETER;
      }
      else
      {
         std::shared_ptr<PmCoreIfMessage> pmCoreIfMessage =
               getNewPmCoreIfMessage_stopExtendedVRRequest(deviceAddress, act);
         pmCoreIfMessage->traceMessage();
         PmCoreMainController::getInstance().handlePmCoreIfMessage(pmCoreIfMessage);
      }

      return pmResult;
   }
}
