/**
 * @file PmInterfaceTypesInternal.h
 *
 * @swcomponent PhoneCallManager
 *
 * @brief This file contains the internal type definitions of PmInterface
 *
 * @copyright (C) 2016 Robert Bosch GmbH.
 *            The reproduction, distribution and utilization of this file as
 *            well as the communication of its contents to others without express
 *            authorization is prohibited. Offenders will be held liable for the
 *            payment of damages. All rights reserved in the event of the grant
 *            of a patent, utility model or design.
 *
 * @details PmInterface type definitions are defined here.
 *
 * @ingroup PmCore
 */

#ifndef PmInterfaceTypesInternal_h
#define PmInterfaceTypesInternal_h

#include <string>
#include <vector>
#include <map>
#include <memory>
#include <algorithm>

namespace pmcore
{
/**
 * macros to mark parameters data flow direction
 */
#ifndef IN
#define IN /**< direction marker for input parameter */
#endif
#ifndef OUT
#define OUT /**< direction marker for output parameter */
#endif
#ifndef INOUT
#define INOUT /**< direction marker for input/output parameter */
#endif

   #define DEVICE_ADDRESS_ALL           "ALL"

   #define DEVICE_ADDRESS_DELETE_ALL    "DELETE_ALL"

   #define SCO_TIMER_ID                 "SCO_TIMER"

   #define SIGNAL_STRENGTH_DEFAULT      0xFF

   #define BATTERY_CHARGE_LEVEL_DEFAULT 0xFF

   #define CALL_INSTANCE_DEFAULT        0xFF

   #define CALL_DURATION_DEFAULT        0x00

   #define TIMER_REPEAT_COUNT_DEFAULT   0x00

   #define RINGTONE_ID_DEFAULT          -1

   #define RINGTONE_ID_INBAND           0x01

   #define RINGTONE_NAME_INBAND         "Bluetooth Device Ringtone"

   #define RINGTONE_ID_SYSTEM_DEFAULT   0x02

   #define RINGTONE_ID_SYSTEM_MAX       0xFF

   #define CALL_COUNT_DEFAULT           0x02

   #define PM_DEVICEHANDLE_ZERO (unsigned char)0x00

   #define MAX_ACTIVE_DEVICES_DEFAULT   0x01

   #define DEFAULT_FILE_PLAY_COUNT      0xFF

   #define NUMBER_OF_SECONDS_PER_DAY    (24 * 60 * 60)

   #define NUMBER_OF_SECONDS_PER_HOUR   (60 * 60)

   #define NUMBER_OF_SECONDS_PER_MINUTE 60

   #define INBAND_ACCEPT_WAIT_TIME      2

   #define CONTAINER_SIZE_ONE           0x01

   #define TIMER_REPEAT_COUNT_ONE       0x01

   #define ONE_SECOND_IN_MILLISECONDS   1000

   #define DEFAULT_VEHICLE_ID  0

   constexpr int VALUE_ONE = 1;

   /**
    * When a client such as a HMI is requesting to play a vehicle ringtone, then
    * this macro is used as a device address parameter.
    */
   #define CLIENT_REQUEST "CLIENT_REQUEST"

   // Apple device
   #define PM_APPLE_USB_IMPLEMENTER_VENDOR_ID   0x05AC
   #define PM_USB_IMPLEMENTER_FORUM_ID_SOURCE   0x0002
   #define PM_APPLE_BLUETOOTH_SIG_VENDOR_ID     0x004C
   #define PM_BLUETOOTH_SIG_ID_SOURCE           0x0001

   #define COUNT_ZERO                  0x00
   #define ENABLE_SIRI_NR              0x01
   #define ENABLED_SIRI_NR             0x01
   #define DISABLED_SIRI_NR            0x00

   //RVC defines
   constexpr int MAX_HFP_SCO_VOICE_CALL_VOLUME = 100;
   constexpr int MIN_HFP_SCO_VOICE_CALL_VOLUME = 0;

   constexpr int GAM_SINK_ID = 1;

   constexpr int DEVICE_ADDRESS_START  = 0;
   constexpr int DEVICE_ADDRESS_LENGTH = 12;

   /**
    * Default value for asynchronous completion token (ACT).
    */
   enum
   {
      PM_DEFAULT_ACT = 0
   };

   /**
    * PmInterface result code
    */
   typedef enum
   {
      PM_RESULT_OK = 0,
      PM_RESULT_ERR_GENERAL,
      PM_RESULT_ERR_DEVICE_NOT_EXIST,
      PM_RESULT_ERR_INVALID_PARAMETER,
      PM_RESULT_ERR_NOT_IMPLEMENTED,
      PM_RESULT_ERR_FAILED,
      PM_RESULT_ERR_TIMEOUT,
      PM_RESULT_ERR_CHANNEL_REL_EXT_TRIGGER,
      PM_RESULT_ERR_USER_PREF_CHANGED_FOR_DEVICE_ROLE,
      PM_RESULT_ERR_INVALID_DEVICE_ROLE,
      PM_RESULT_ERR_HFP
   } PmResultCode;

   /**
    * SPM System State
    */
   typedef enum
   {
      PM_SPM_SYSTEM_OFF = 0UL,
      PM_SPM_SYSTEM_ON = 1UL,
      PM_SPM_SYSTEM_STANDBY = 2UL
   } SystemState;

   /**
    * Possible DeviceRole states
    */
   typedef enum
   {
      DEVICEROLE_PASSIVE = 0x00u,
      DEVICEROLE_ACTIVE  = 0x01u,
      DEVICEROLE_DEFAULT = 0xFFu
   } DeviceRoleEnumType;

   typedef enum
   {
      CALL_STATE_IDLE                   = 0x00u,
      CALL_STATE_INCOMING               = 0x01u,
      CALL_STATE_INCOMING_INBAND        = 0x02u,
      CALL_STATE_ALERTING               = 0x03u,
      CALL_STATE_DIALING                = 0x04u,
      CALL_STATE_HELD                   = 0x05u,
      CALL_STATE_ACTIVE                 = 0x06u,
      CALL_STATE_DISCONNECTING          = 0x07u,
      CALL_STATE_DISCONNECTED           = 0x08u,
      CALL_STATE_CALL_X_DIALING         = 0x09u,
      CALL_STATE_CALL_X_ALERTING        = 0x0Au,
      CALL_STATE_CALL_X_WAITING         = 0x0Bu,
      CALL_STATE_CALLS_SWAPPABLE        = 0x0Cu,
      CALL_STATE_CALLS_CONFERENCE       = 0x0Du,
      CALL_STATE_ALL_CALLS_DISCONNECTED = 0x0Eu,
      CALL_STATE_UNKNOWN                = 0xFFu
   }CallStateEnumType;

   typedef enum
   {
      PHONE_CHANNEL_GRANTED       = 0x00u,
      PHONE_CHANNEL_DENIED        = 0x01u,
      RINGTONE_CHANNEL_GRANTED    = 0x02u,
      RINGTONE_CHANNEL_DENIED     = 0x03u,
      WAITINGMODE_CHANNEL_GRANTED = 0x04u,
      WAITINGMODE_CHANNEL_DENIED  = 0x05u,
      VOICEREC_CHANNEL_GRANTED    = 0x06u,
      VOICEREC_CHANNEL_DENIED     = 0x07u,
      PLAY_AUDIO_SUCCESS          = 0x08u,
      PLAY_AUDIO_FAILURE          = 0x09u,
      PAUSE_AUDIO_SUCCESS         = 0x0Au,
      PAUSE_AUDIO_FAILURE         = 0x0Bu,
      STOP_AUDIO_SUCCESS          = 0x0Cu,
      STOP_AUDIO_FAILURE          = 0x0Du,
      PHONE_CHANNEL_RELEASED      = 0x0Eu,
      RINGTONE_CHANNEL_RELEASED   = 0x0Fu,
      WAITINGMODE_CHANNEL_RELEASED= 0x10u,
      VOICEREC_CHANNEL_RELEASED   = 0x11u
   }AudioManagerEventType;

   typedef enum
   {
      NETWORK_STATUS_UNKNOWN      = 0x00u,
      NETWORK_STATUS_UNREGISTERED = 0x01u,
      NETWORK_STATUS_REGISTERED   = 0x02u,
      NETWORK_STATUS_SEARCHING    = 0x03u,
      NETWORK_STATUS_DENIED       = 0x04u,
      NETWORK_STATUS_ROAMING      = 0x05u,
      NETWORK_STATUS_DEFAULT      = 0xFFu
   } NetworkStatusEnumType;

   typedef enum
   {
      CALL_MODE_UNKNOWN = 0x00u,
      CALL_MODE_VOICE   = 0x01u,
      CALL_MODE_DATA    = 0x02u,
      CALL_MODE_FAX     = 0x03u
   } CallModeEnumType;

   typedef enum
   {
      NUMBER_TYPE_UNKNOWN       = 0x00u,
      NUMBER_TYPE_INTERNATIONAL = 0x01u,
      NUMBER_TYPE_NATIONAL      = 0x02u,
      NUMBER_TYPE_NETWORK       = 0x03u,
      NUMBER_TYPE_DEDICATED     = 0x04u
   } NumberTypeEnumType;

   typedef enum
   {
      CALL_DIRECTION_UNKNOWN  = 0x00u,
      CALL_DIRECTION_INCOMING = 0x01u,
      CALL_DIRECTION_OUTGOING = 0xFFu
   } CallDirectionEnumType;

   typedef enum
   {
      SHOW_CALLERID    = 0x00u,
      HIDE_CALLERID    = 0x01u,
      DEFAULT_CALLERID = 0xFFu
   } HideCallerIdEnumType;

   typedef enum
   {
      SCO_NOT_ESTABLISHED  = 0x00u,
      SCO_PAUSED,
      SCO_NARROWBAND,
      SCO_WIDEBAND,
      SCO_NARROWBAND_SIRI,
      SCO_WIDEBAND_SIRI,
      SCO_DEFAULT          = 0xFFu
   } SCOStatusEnumType;

   typedef enum
   {
      NOT_SUPPORTED          = 0x00u,
      SUPPORTED_AND_ENABLED  = 0x01u,
      SUPPORTED_AND_DISABLED = 0x02u
   } InbandRingingSupport;

   typedef enum
   {
      WAITING_MODE_IDLE       = 0x00u,
      WAITING_MODE_ACTIVE     = 0x01u,
      WAITING_MODE_STARTING   = 0x02u,
      WAITING_MODE_STOPPING   = 0x03u
   } WaitingModeStateEnumType;

   typedef enum
   {
      ACCEPT_CALLS       = 0x00u,
      RELEASE_AND_ACCEPT,
      HOLD_AND_ACCEPT,
      ACCEPT_CALLS_BY_ACQUIRING_CHANNEL,
      RELEASE_AND_ACCEPT_BY_ACQUIRING_CHANNEL,
      HOLD_AND_ACCEPT_BY_ACQUIRING_CHANNEL
   } AcceptOperationEnumType;

   typedef enum
   {
      SWAP_CALLS       = 0x00u,
      RELEASE_AND_SWAP = 0x01u
   } SwapOperationEnumType;

   typedef enum
   {
      AUDIO_DIRECTION_HANDSFREE    = 0x00u,
      AUDIO_DIRECTION_AUDIOGATEWAY = 0x01u
   } AudioDirectionEnumType;

   typedef enum
   {
      AUDIO_PATH_IDLE                        = 0x00u,
      AUDIO_PATH_CALL                        = 0x01u,
      AUDIO_PATH_IN_BAND_RING                = 0x02u,
      AUDIO_PATH_MODULE_INTERNAL_RING        = 0x03u,
      AUDIO_PATH_VOICE_RECOGNITION           = 0x05u,
      AUDIO_PATH_PENDING_FOR_INBAND_RINGTONE = 0x06u,
      AUDIO_PATH_UNKNOWN                     = 0xFFu,
   } AudioPathEnumType;

   typedef enum
   {
      VR_SESSION_IDLE   = 0x00u,
      VR_SESSION_ACTIVE,
      VR_SESSION_STARTING,
      VR_SESSION_ENDING
   } VRStatusEnumType;

   typedef enum
   {
      EXT_VR_SESSION_IDLE  = 0x00u,
      EXT_VR_SESSION_ACTIVE,
      EXT_VR_SESSION_READY
   } ExternalVRStatusEnumType;

   typedef enum
   {
      VR_INTERRUPT_NO_ERROR = 0x00u,
      HFP_CALL,
      AUDIO_CHANNEL_RELEASED
   } VRInterruptedReason;

   typedef enum
   {
      ENHANCED_VR_ENABLED  = 0x00u,
      ENHANCED_VR_DISABLED = 0x01u,
      ENHANCED_VR_DEFAULT  = 0x02u
   } EnhancedVRAvailabilityEnumType;

   /**
    * Type definition for PmCoreIfMsgId.
    */
   typedef enum
   {
      PM_CORE_IF_MSG_ID_START = 0x00u,

      //Generic methods
      PM_CORE_IF_MSG_ID_SET_SYSTEM_STATE,
      PM_CORE_IF_MSG_ID_SET_VEHICLE_CONFIGURATION,
      PM_CORE_IF_MSG_ID_ON_DEVICE_PAIRED,
      PM_CORE_IF_MSG_ID_ON_DEVICE_CONNECTED,
      PM_CORE_IF_MSG_ID_ON_DEVICE_DISCONNECTED,
      PM_CORE_IF_MSG_ID_ON_DEVICE_DELETED,

      //Methods request
      PM_CORE_IF_MSG_ID_GET_SYSTEM_WIDE_RINGTONE_LIST_REQUEST,
      PM_CORE_IF_MSG_ID_GET_DEVICE_RINGTONE_LIST_REQUEST,
      PM_CORE_IF_MSG_ID_SET_RINGTONE_REQUEST,
      PM_CORE_IF_MSG_ID_PLAY_RINGTONE_REQUEST,
      PM_CORE_IF_MSG_ID_STOP_RINGTONE_REQUEST,
      PM_CORE_IF_MSG_ID_GET_SUPPRESS_RINGTONE_ON_OFF_LIST_REQUEST,
      PM_CORE_IF_MSG_ID_SUPPRESS_RINGTONE_ON_OFF_REQUEST,
      PM_CORE_IF_MSG_ID_GET_AUTO_WAITING_MODE_ON_OFF_LIST_REQUEST,
      PM_CORE_IF_MSG_ID_AUTO_WAITING_MODE_ON_OFF_REQUEST,
      PM_CORE_IF_MSG_ID_SET_WAITING_MODE_FILE_PATH_REQUEST,
      PM_CORE_IF_MSG_ID_GET_WAITING_MODE_FILE_PATH_REQUEST,
      PM_CORE_IF_MSG_ID_GET_PHONE_CALL_AUDIO_ACTIVE_REQUEST,
      PM_CORE_IF_MSG_ID_GET_ACTIVE_PASSIVE_DEVICE_LIST_REQUEST,
      PM_CORE_IF_MSG_ID_SET_ACTIVE_PHONE_REQUEST,
      PM_CORE_IF_MSG_ID_RESTORE_DEFAULT_SETTING_REQUEST,
      PM_CORE_IF_MSG_ID_GET_SUPPORTED_FEATURES_REQUEST,
      PM_CORE_IF_MSG_ID_GET_NETWORK_STATUS_REQUEST,
      PM_CORE_IF_MSG_ID_GET_SIGNAL_STRENGTH_REQUEST,
      PM_CORE_IF_MSG_ID_GET_BATTERY_CHARGE_LEVEL_REQUEST,
      PM_CORE_IF_MSG_ID_GET_NETWORK_OPERATOR_REQUEST,
      PM_CORE_IF_MSG_ID_VOICE_MAIL_LIST_REQUEST,
      PM_CORE_IF_MSG_ID_GET_CALL_STATUS_LIST_REQUEST,
      PM_CORE_IF_MSG_ID_GET_MICROPHONE_MUTE_STATE_REQUEST,
      PM_CORE_IF_MSG_ID_SET_MICROPHONE_MUTE_STATE_REQUEST,
      PM_CORE_IF_MSG_ID_MERGE_CALLS_REQUEST,
      PM_CORE_IF_MSG_ID_SPLIT_CALLS_REQUEST,
      PM_CORE_IF_MSG_ID_HANGUP_CALLS_REQUEST,
      PM_CORE_IF_MSG_ID_REDIAL_REQUEST,
      PM_CORE_IF_MSG_ID_GET_SCO_CONNECTION_REQUEST,
      PM_CORE_IF_MSG_ID_GET_RINGTONE_MUTE_STATE_REQUEST,
      PM_CORE_IF_MSG_ID_SET_RINGTONE_MUTE_STATE_REQUEST,
      PM_CORE_IF_MSG_ID_GET_WAITING_MODE_STATE_REQUEST,
      PM_CORE_IF_MSG_ID_GET_RINGTONE_PLAYBACK_STATUS_REQUEST,
      PM_CORE_IF_MSG_ID_START_STOP_WAITING_MODE_REQUEST,
      PM_CORE_IF_MSG_ID_DIAL_REQUEST,
      PM_CORE_IF_MSG_ID_ACCEPT_CALL_REQUEST,
      PM_CORE_IF_MSG_ID_SWAP_CALL_REQUEST,
      PM_CORE_IF_MSG_ID_SPEED_DIAL_REQUEST,
      PM_CORE_IF_MSG_ID_SEND_DTMF_REQUEST,
      PM_CORE_IF_MSG_ID_TRANSFER_AUDIO_REQUEST,
      PM_CORE_IF_MSG_ID_GET_VOICE_RECOGNITION_STATUS_REQUEST,
      PM_CORE_IF_MSG_ID_START_STOP_VOICE_RECOGNITION_REQUEST,
      PM_CORE_IF_MSG_ID_GET_ENHANCED_VOICE_RECOGNITION_FEATURE_REQUEST,
      PM_CORE_IF_MSG_ID_GET_EXTERNAL_VOICE_RECOGNITION_STATUS_REQUEST,
      PM_CORE_IF_MSG_ID_START_STOP_EXTERNAL_VOICE_RECOGNITION_REQUEST,
      PM_CORE_IF_MSG_ID_STOP_EXTENDED_VOICE_RECOGNITION_REQUEST,

      //Methods result
      PM_CORE_IF_MSG_ID_REQUEST_PHONE_NUMBER_RESULT,
      PM_CORE_IF_MSG_ID_GET_SUBSCRIBER_RESULT,
      PM_CORE_IF_MSG_ID_SEND_CUSTOM_AT_COMMAND_RESULT,
      PM_CORE_IF_MSG_ID_HF_UPDATE_INDICATORS_VALUE_RESULT,
      PM_CORE_IF_MSG_ID_GET_MODEMS_RESULT,
      PM_CORE_IF_MSG_ID_ACCEPT_SCO_CONNECT_RESULT,
      PM_CORE_IF_MSG_ID_SIRI_ENABLE_RESULT,
      PM_CORE_IF_MSG_ID_SIRI_SET_NR_RESULT,
      PM_CORE_IF_MSG_ID_GET_CALLS_RESULT,
      PM_CORE_IF_MSG_ID_DIAL_RESULT,
      PM_CORE_IF_MSG_ID_SPEED_DIAL_RESULT,
      PM_CORE_IF_MSG_ID_REDIAL_RESULT,
      PM_CORE_IF_MSG_ID_SWAP_CALL_RESULT,
      PM_CORE_IF_MSG_ID_RELEASE_AND_ACCEPT_RESULT,
      PM_CORE_IF_MSG_ID_RELEASE_AND_SWAP_RESULT,
      PM_CORE_IF_MSG_ID_HOLD_AND_ACCEPT_RESULT,
      PM_CORE_IF_MSG_ID_HANGUP_ALL_RESULT,
      PM_CORE_IF_MSG_ID_SPLIT_CALLS_RESULT,
      PM_CORE_IF_MSG_ID_MERGE_CALLS_RESULT,
      PM_CORE_IF_MSG_ID_HANGUP_MULTIPARTY_RESULT,
      PM_CORE_IF_MSG_ID_SEND_DTMF_RESULT,
      PM_CORE_IF_MSG_ID_HANGUP_CALL_RESULT,
      PM_CORE_IF_MSG_ID_ACCEPT_CALL_RESULT,
      PM_CORE_IF_MSG_ID_HOLD_INCOMING_CALL_RESULT,
      PM_CORE_IF_MSG_ID_TRANSFER_AUDIO_RESULT,
      PM_CORE_IF_MSG_ID_START_STOP_VOICE_RECOGNITION_RESULT,

      // Response from PM Audiomanager
      PM_CORE_IF_MSG_ID_PREPARE_AUDIO_ROUTE_RESPONSE,
      PM_CORE_IF_MSG_ID_PLAY_AUDIO_RESPONSE,
      PM_CORE_IF_MSG_ID_STOP_AUDIO_RESPONSE,
      PM_CORE_IF_MSG_ID_SET_MIC_MUTE_STATE_RESPONSE,
      PM_CORE_IF_MSG_ID_PAUSE_AUDIO_RESPONSE,
      PM_CORE_IF_MSG_ID_GET_AUDIO_SINK_VOLUME_RESPONSE,
      PM_CORE_IF_MSG_ID_SET_AUDIO_SINK_VOLUME_RESPONSE,

      PM_CORE_IF_MSG_ID_END

   } PmCoreIfMsgId;

   /**
    * Type definition for PmCoreIfMsgOrigin.
    */
   typedef enum
   {
      PM_CORE_IF_MSG_ORIGIN_INTERNAL = 0x00u,
      PM_CORE_IF_MSG_ORIGIN_CLIENT = 0x01u,
      PM_CORE_IF_MSG_ORIGIN_PM_AUDIOMANAGER_IF = 0x02u,
      PM_CORE_IF_MSG_ORIGIN_BT_STACK_IF = 0xFFu
   } PmCoreIfMsgOrigin;

   /**
    * Type definition for common result of a request.
    */
   typedef enum
   {
      BTS_REQ_SUCCESS = 0x00u, /**< request was successful */
      BTS_REQ_FAILED,          /**< request/operation failed */

      BTS_REQ_LAST             /**< end marker, must be the last one */
   }BTSRequestResult;

   typedef enum
   {
      ON_DEVICE_PAIRED,                          //IpcWrapper Start
      ON_DEVICE_CONNECTED,
      ON_DEVICE_DISCONNECTED,
      ON_DEVICE_DELETED,                         //IpcWrapper End

      BTS_UPDATE_PROXY_SERVICE_AVAILABILITY,     //Proxy ServiceAvailability Status from BtStack

      BTS_UPDATE_SUPPORTED_FEATURES,             //DeviceInfoHandler Start
      BTS_UPDATE_CHLD_FEATURE,
      BTS_UPDATE_INBAND_RINGING,
      BTS_UPDATE_BATTERY_CHARGE_LEVEL,
      BTS_UPDATE_NETWORK_STATUS,
      BTS_UPDATE_SIGNAL_STRENGTH,
      BTS_UPDATE_CURRENT_OPERATOR_NAME,
      BTS_UPDATE_MOBILE_COUNTRY_CODE,
      BTS_UPDATE_EMERGENCY_NUMBERS,
      BTS_UPDATE_END_DEVICE_INFO_HANDLER,        // DeviceInfoHandler End

      BTS_UPDATE_AUDIO,                          // CallController Start
      BTS_UPDATE_SCO_TYPE,
      BTS_UPDATE_AUDIO_PATH,
      BTS_UPDATE_ALL_CALL_STATUS,
      BTS_UPDATE_AUDIO_INDICATOR,
      BTS_UPDATE_VOICE_CALL_STATUS,
      BTS_UPDATE_MICROPHONE_VOLUME,
      BTS_UPDATE_END_CALL_CONTROLLER,            // CallController End

      BTS_UPDATE_SIRI_SUPPORTED_TYPE,            // VRController Start
      BTS_UPDATE_VOICE_RECOGNITION_STATUS,
      BTS_UPDATE_ECHO_CANCELING_NOISE_REDUCTION,
      BTS_UPDATE_SIRI_STATUS,
      BTS_UPDATE_NOISE_REDUCTION,
      BTS_UPDATE_EYES_FREE_MODE,
      BTS_UPDATE_END_VR_CONTROLLER,              // VRController End

      BTS_SIGNAL_SCO_CONNECT_REQUEST,            // Signal Start
      BTS_SIGNAL_VOICE_CALL_ADDED,
      BTS_SIGNAL_VOICE_CALL_REMOVED,             // Signal End

      BTS_SIGNAL_VOICE_CALL_ADDED_END_VR,

      // Notification signal from PM Audiomanager
      // Acquisition
      AM_RINGTONE_CHANNEL_ACQUISITION_SUCCESS,
      AM_RINGTONE_CHANNEL_ACQUISITION_FAILURE,
      AM_PHONE_CHANNEL_ACQUISITION_SUCCESS,
      AM_PHONE_CHANNEL_ACQUISITION_FAILURE,
      AM_WAITINGMODE_CHANNEL_ACQUISITION_SUCCESS,
      AM_WAITINGMODE_CHANNEL_ACQUISITION_FAILURE,
      AM_VR_CHANNEL_ACQUISITION_SUCCESS,
      AM_VR_CHANNEL_ACQUISITION_FAILURE,

      // PlayAudio
      AM_RINGTONE_CHANNEL_PLAYAUDIO_SUCCESS,
      AM_RINGTONE_CHANNEL_PLAYAUDIO_FAILURE,
      AM_PHONE_CHANNEL_PLAYAUDIO_SUCCESS,
      AM_PHONE_CHANNEL_PLAYAUDIO_FAILURE,
      AM_WAITINGMODE_CHANNEL_PLAYAUDIO_SUCCESS,
      AM_WAITINGMODE_CHANNEL_PLAYAUDIO_FAILURE,
      AM_VR_CHANNEL_PLAYAUDIO_SUCCESS,
      AM_VR_CHANNEL_PLAYAUDIO_FAILURE,

      // StopAudio
      AM_RINGTONE_CHANNEL_STOPAUDIO_SUCCESS,
      AM_RINGTONE_CHANNEL_STOPAUDIO_FAILURE,
      AM_PHONE_CHANNEL_STOPAUDIO_SUCCESS,
      AM_PHONE_CHANNEL_STOPAUDIO_FAILURE,
      AM_WAITINGMODE_CHANNEL_STOPAUDIO_SUCCESS,
      AM_WAITINGMODE_CHANNEL_STOPAUDIO_FAILURE,
      AM_VR_CHANNEL_STOPAUDIO_SUCCESS,
      AM_VR_CHANNEL_STOPAUDIO_FAILURE,

      // PauseAudio
      AM_RINGTONE_CHANNEL_PAUSEAUDIO_SUCCESS,
      AM_RINGTONE_CHANNEL_PAUSEAUDIO_FAILURE,
      AM_PHONE_CHANNEL_PAUSEAUDIO_SUCCESS,
      AM_PHONE_CHANNEL_PAUSEAUDIO_FAILURE,
      AM_WAITINGMODE_CHANNEL_PAUSEAUDIO_SUCCESS,
      AM_WAITINGMODE_CHANNEL_PAUSEAUDIO_FAILURE,
      AM_VR_CHANNEL_PAUSEAUDIO_SUCCESS,
      AM_VR_CHANNEL_PAUSEAUDIO_FAILURE,

      AM_PHONE_CHANNEL_RELEASED,
      AM_RINGTONE_CHANNEL_RELEASED,
      AM_WAITINGMODE_CHANNEL_RELEASED,

      // MicMute
      AM_SET_MIC_MUTE_SUCCESS,
      AM_SET_MIC_MUTE_FAILURE,

      AM_SET_MIC_UNMUTE_SUCCESS,
      AM_SET_MIC_UNMUTE_FAILURE,

      //RVC
      AM_UPDATE_AUDIO_VOLUME,
      BTS_UPDATE_SPEAKER_VOLUME,

      PROPERTY_AND_EVENTID_LAST                    // Used as a Default
   }PmCorePropertyAndEventId;

   typedef enum
   {
      VOICE_CALL_LINEIDENTIFICATION,
      VOICE_CALL_NAME,
      VOICE_CALL_MULTIPARTY,
      VOICE_CALL_STATE,
      VOICE_CALL_EMERGENCY,
      VOICE_CALL_DIRECTION,
      VOICE_CALL_MODE,
      VOICE_CALL_TYPE,
   }VoiceCallPropertyId;

   typedef enum
   {
      PHONE_AUDIO_CHANNEL_IDLE           = 0x00u,

      PHONE_CHANNEL_REQUESTED,
      PHONE_CHANNEL_ALLOCATED,
      PHONE_CHANNEL_NOT_ALLOCATED,
      PHONE_START_STREAMING_REQUESTED,
      PHONE_START_STREAMING_SUCCESS,
      PHONE_START_STREAMING_FAILED,
      PHONE_STOP_STREAMING_REQUESTED,
      PHONE_STOP_STREAMING_SUCCESS,
      PHONE_STOP_STREAMING_FAILED,
      PHONE_PAUSE_STREAMING_REQUESTED,
      PHONE_PAUSE_STREAMING_SUCCESS,
      PHONE_PAUSE_STREAMING_FAILED,

      RINGTONE_CHANNEL_REQUESTED,
      RINGTONE_CHANNEL_ALLOCATED,
      RINGTONE_CHANNEL_NOT_ALLOCATED,
      RINGTONE_START_STREAMING_REQUESTED,
      RINGTONE_START_STREAMING_SUCCESS,
      RINGTONE_START_STREAMING_FAILED,
      RINGTONE_STOP_STREAMING_REQUESTED,
      RINGTONE_STOP_STREAMING_SUCCESS,
      RINGTONE_STOP_STREAMING_FAILED,
      RINGTONE_PAUSE_STREAMING_REQUESTED,
      RINGTONE_PAUSE_STREAMING_SUCCESS,
      RINGTONE_PAUSE_STREAMING_FAILED,

      WAITINGMODE_CHANNEL_REQUESTED,
      WAITINGMODE_CHANNEL_ALLOCATED,
      WAITINGMODE_CHANNEL_NOT_ALLOCATED,
      WAITINGMODE_START_STREAMING_REQUESTED,
      WAITINGMODE_START_STREAMING_SUCCESS,
      WAITINGMODE_START_STREAMING_FAILED,
      WAITINGMODE_STOP_STREAMING_REQUESTED,
      WAITINGMODE_STOP_STREAMING_SUCCESS,
      WAITINGMODE_STOP_STREAMING_FAILED,
      WAITINGMODE_PAUSE_STREAMING_REQUESTED,
      WAITINGMODE_PAUSE_STREAMING_SUCCESS,
      WAITINGMODE_PAUSE_STREAMING_FAILED

   }PmCorePhoneAudioState;

   typedef enum
   {
      NO_AUDIO_PROCESS_PENDING       = 0x00u,
      WAIT_FOR_PAUSE_AUDIO_RESPONSE  = 0x01u,
      STOP_AUDIO_NOT_CALLED          = 0x02u,
      WAIT_FOR_STOP_AUDIO_RESPONSE   = 0x03u

   } StopAudioStateEnumType;

   typedef enum
   {
      DIAL_CALL_ERROR              = 0x00u,
      SPEED_DIAL_ERROR,
      ACCEPT_CALL_ERROR,
      HOLD_AND_ACCEPT_ERROR,
      RELEASE_AND_ACCEPT_ERROR,
      RELEASE_AND_SWAP_ERROR,
      SWAP_CALL_ERROR,
      CONFERENCE_CALL_ERROR,
      SPLIT_CALL_ERROR,
      HANGUP_MULTIPARTY_ERROR,
      TRANSFER_CALL_ERROR,
      TRANSFER_CALL_ERROR_FOR_SWITCH_TO_PASSIVE,
      HANGUP_CALL_ERROR,
      HANGUP_ALL_CALL_ERROR,

      RINGTONE_CHANNEL_DENIED_ERROR,
      PHONE_CHANNEL_DENIED_FOR_ACCEPT_ERROR,
      PHONE_CHANNEL_DENIED_FOR_RINGTONE_ERROR,
      PHONE_CHANNEL_DENIED_FOR_TRANSFER_AUDIO_ERROR,
      ECNR_FAILED_FOR_TRANSFER_AUDIO_ERROR,
      WAITINGMODE_CHANNEL_DENIED_ERROR,
      ACCEPT_CALL_FAILED_TO_START_WAITINGMODE_ERROR,
      START_WAITINGMODE_FAILED_ERROR,
      STOP_WAITINGMODE_FAILED_ERROR
   } SmErrorEnumType;

   // This enum is used to update the Phone sub state value.
   typedef enum
   {
      CALL_PRESENT_WITHOUT_SCO,
      CALL_PRESENT_WITH_SCO,
      INCOMING_CALL_PRESENT,
      CALL_REMOVED
   } CallExistenceState;

   // This enum is used to know about the Auto reject reason.
   typedef enum
   {
      REJECT_REASON_UNKNOWN,
      WAITING_MODE_IN_PROGRESS,
      THIRD_INCOMING_CALL
   } AutoRejectRequestReason;

   /*
    * Origin of the controller
    */
   typedef enum
   {
      MAIN_CONTROLLER = 0UL, // The client request to play Ringtone will be triggered from Main Controller(Default)
      CALL_CONTROLLER,
      VR_CONTROLLER,
      RV_CONTROLLER
   }ControllerOriginEnum;

   typedef enum
   {
      INCOMING_CALL_REJECTED,
      OUTGOING_CALL_FAILED,
      SPEED_DIAL_FAILED,
      ACCEPT_CALL_FAILED,
      SWAP_CALL_FAILED,
      CONFERENCE_CALL_FAILED,
      SPLIT_CALL_FAILED,
      HANGUP_CALL_FAILED,
      TRANSFER_CALL_AUDIO_TO_AG_FAILED,
      TRANSFER_CALL_AUDIO_TO_HF_FAILED,
      SEND_DTMF_FAILED
   }NotificationEventName;

   typedef enum
   {
      RT_SM_NOT_PLAYING,
      RT_SM_PREPARING_CH,
      RT_SM_STARTING_TO_PLAY,
      RT_SM_PLAYING,
      RT_SM_STOPPING
   }RingtonePlaybackStateSMType;

   typedef enum
   {
      RT_NOT_PLAYING,
      RT_STARTING,
      RT_PLAYING,
      RT_STOPPING
   }RingtonePlaybackStateType;

   typedef ControllerOriginEnum ControllerType;

   typedef bool          SuppressRingtoneState;

   typedef bool          AutoWaitingModeState;

   typedef bool          AudioActiveStatus;

   typedef bool          Multiparty;

   typedef bool          Emergency;

   typedef bool          MuteState;

   typedef bool          StartStop;

   typedef bool          RVCUpdate;

   typedef std::string   PmResultMessage;

   typedef std::string   BdAddress;

   typedef std::string   DeviceName;

   typedef std::string   RingtoneName;

   typedef std::string   FilePath;

   typedef std::string   ContactName;

   typedef std::string   FirstName;

   typedef std::string   LastName;

   typedef std::string   TelephoneNumber;

   typedef std::string   Reason;

   typedef std::string   FriendlyNameType;

   typedef std::string   NetworkOperatorType;

   typedef std::string   AmErrorMessage;

   /**
    * <Possible call states>
    *
    * Active - The call is active.
    * Held - The call is on hold.
    * Dialing - The call is being dialed.
    * Alerting - The remote party is being alerted.
    * Incoming - Incoming call in progress.
    * Waiting - Call is waiting.
    * Idle - The call is ended or already it is in Idle state
    */
   typedef std::string   CallState;

   typedef std::string   DTMFTones;

   typedef std::string   EnhancedVRFeature;

   typedef std::string   MobileCountryCode; //TODO: is req?

   typedef std::string   ErrorName;

   typedef std::string   ErrorMessage;

   typedef unsigned char DeviceHandle;

   typedef unsigned char DeviceRole;

   typedef unsigned char NetworkStatusType;

   typedef unsigned char SignalStrengthType;

   typedef unsigned char BatteryChargeLevelType;

   typedef unsigned char CallInstance;

   typedef unsigned char CallCount;

   typedef unsigned char Duration;

   typedef unsigned char CallMode;

   typedef unsigned char NumberType;

   typedef unsigned char CallDirection;

   typedef unsigned char HideCallerId;

   typedef unsigned char SCOStatus;

   typedef unsigned char WaitingModeStateType;

   typedef unsigned char AcceptOperation;

   typedef unsigned char SwapOperation;

   typedef unsigned char MemDialIndex;

   typedef unsigned char AudioDirection;

   typedef unsigned char AudioPath;

   typedef unsigned char VolumeLevel;

   typedef unsigned char VolumeType;

   typedef unsigned char VRStatus;

   typedef unsigned char ExtVRStatus;

   typedef unsigned char EnhancedVRAvailability;

   typedef unsigned char PlayCount;

   typedef unsigned char DevicesCount;

   typedef unsigned short            ScoTimeout;

   typedef unsigned short            SiriType;

   typedef int                       WaitTime;

   typedef int                       RvcTimeout;

   typedef signed short int          RingtoneId;

   typedef std::vector<BdAddress>    BdAddressList;

   typedef unsigned short VehicleDetails;

   /*
    * List of possible supported features-
    *
    *-------------------------------------------------------------------------
    * Features received from Interface- "Handsfree" and Property - "Features"
    * (These are dependent on +BRSF command from AG)
    * -------------------------------------------------------------------------
    * 3way : 3 way calling
    * ecnr : ECNR function
    * voice_recognition : Voice Recognition is supported
    * inband_ringtone : Inband Ringtone is supported
    * attach_voice_tag : Attaching a phone number to a voice tag
    * reject_incoming_call : Ability to reject an incoming call
    * enhanced_call_status : Support to query the list of current calls in Audio Gateway
    * enhanced_call_control : Ability to unmerging a conference call or ending an individual call within a conference call
    * extended_error_codes : Extended Audio Gateway error
    * codec_negotiation : Codec negotiation between HF and AG
    *
    *-----------------------------------------------------------------------------
    * Features received from Interface- "Handsfree" and Property - "ChldFeature"
    * (These are dependent on +CHLD command from AG)
    * -----------------------------------------------------------------------------
    * EndTargetCall : End target call support
    * SplitMergeCalls : Split the Merged Calls
    *
    */
   typedef std::vector<std::string>  SupportedFeaturesType;

   typedef std::vector<std::string>  ChldFeaturesType; //TODO: is req?

   typedef std::vector<CallInstance> CallInstanceList;

   typedef std::vector<PmCorePropertyAndEventId> PropertyIdList;   /**< List of property and signal Id's */

   typedef std::map<BdAddress, SuppressRingtoneState> SuppressRingtoneOnOffListMap;

   typedef std::map<BdAddress, AutoWaitingModeState>  AutoWaitingModeOnOffListMap;

   typedef std::map<BdAddress, DeviceRole>            ActivePassiveDeviceListMap;

   typedef std::map<BdAddress, CallInstanceList>      DeviceCallInstanceListMap;

   typedef std::map<DeviceHandle, VolumeLevel>        AmVolumeLevelListMap;

   typedef void* voidptr;

   /**
    * Asynchronous Completion Token type
    */
   typedef unsigned long long int ActType;

   /**
    * struct NotificationEvent
    */
   typedef struct
   {
      NotificationEventName  _eventName;
      Reason                 _reason;
      BdAddress              _bdAddress;
      TelephoneNumber        _telephoneNumber;
      CallInstanceList       _callInstanceList;
   } NotificationEvent;

   /**
    * struct VoiceCallProperty
    */
   typedef struct
   {
      CallInstance         _callInstance;
      VoiceCallPropertyId  _propertyId;
      void*                _value;
   } VoiceCallProperty;

   /**
    * struct NotificationEvent
    */
   typedef struct
   {
      ActType            _actType;
      CallInstanceList   _callInstanceList;
   } PmcoreRequestDetails;
}

#endif // PmInterfaceTypesInternal_h
