/**
 * @file PmCoreRequestIf.h
 *
 * @swcomponent PhoneCallManager
 *
 * @brief This file contains the definition of the PmInterface class
 *
 * @copyright (C) 2016 Robert Bosch GmbH.
 *            The reproduction, distribution and utilization of this file as
 *            well as the communication of its contents to others without express
 *            authorization is prohibited. Offenders will be held liable for the
 *            payment of damages. All rights reserved in the event of the grant
 *            of a patent, utility model or design.
 *
 * @details This file provides the CPP request interfaces of PmCore.
 *
 * @ingroup PmCore
 */

#ifndef PmCoreRequestIf_h
#define PmCoreRequestIf_h

#include "PmSingleton.h"
#include "IPmCoreRequestIf.h"

namespace pmcore
{
   class PmInterface final : public IPmCoreRequestIf, public PmSingleton <PmInterface>
   {
   public:
      /**
       * Member function Overridden from IPmCoreRequestIf class
       *
       * @param[in]
       * @param[out]
       * @param[in,out]
       *
       * @return PmResult (PmResultCode, PmResultMessage)
       */
      PmResult registerPmCoreCallbackIf(IPmCoreCallbackIf* pmCoreCallbackIf) override;

      /**
       * Member function Overridden from IPmCoreRequestIf class
       *
       * @param[in]
       * @param[out]
       * @param[in,out]
       *
       * @return PmResult (PmResultCode, PmResultMessage)
       */
      PmResult initializePmCore(ahl_tclBaseOneThreadApp* mainApplication) override;

      /**
       * Member function Overridden from IPmCoreRequestIf class
       *
       * @param[in]
       * @param[out]
       * @param[in,out]
       *
       * @return PmResult (PmResultCode, PmResultMessage)
       */
      PmResult setSystemState(IN const SystemState systemState) override;

      /**
       * Member function Overridden from IPmCoreRequestIf class
       *
       * @param[in]
       * @param[out]
       * @param[in,out]
       *
       * @return PmResult (PmResultCode, PmResultMessage)
       */
      PmResult setVehicleConfiguration(IN const VehicleConfiguration& vehicleConfig) override;

      /**
       * Member function Overridden from IPmCoreRequestIf class
       *
       * @param[in]
       * @param[out]
       * @param[in,out]
       *
       * @return PmResult (PmResultCode, PmResultMessage)
       */
      PmResult onDevicePaired(IN const BasicDeviceDetails& basicDeviceDetails) override;

      /**
       * Member function Overridden from IPmCoreRequestIf class
       *
       * @param[in]
       * @param[out]
       * @param[in,out]
       *
       * @return PmResult (PmResultCode, PmResultMessage)
       */
      PmResult onDeviceConnected(IN const BasicDeviceDetails& basicDeviceDetails) override;

      /**
       * Member function Overridden from IPmCoreRequestIf class
       *
       * @param[in]
       * @param[out]
       * @param[in,out]
       *
       * @return PmResult (PmResultCode, PmResultMessage)
       */
      PmResult onDeviceDisconnected(IN const BdAddress& deviceAddress) override;

      /**
       * Member function Overridden from IPmCoreRequestIf class
       *
       * @param[in]
       * @param[out]
       * @param[in,out]
       *
       * @return PmResult (PmResultCode, PmResultMessage)
       */
      PmResult onDeviceDeleted(IN const BdAddress& deviceAddress) override;

      /**
       * Member function Overridden from IPmCoreRequestIf class
       *
       * @param[in] BasicDeviceDetails& - reference to BasicDeviceDetails class
       * @param[out]
       * @param[in,out]
       *
       * @return PmResult (PmResultCode, PmResultMessage)
       */
      PmResult onDeviceIdentificationUpdated(IN const pmcore::BasicDeviceDetails& deviceDetails) override;

      /**
       * Member function Overridden from IPmCoreRequestIf class
       *
       * @param[in]
       * @param[out]
       * @param[in,out]
       *
       * @return PmResult (PmResultCode, PmResultMessage)
       */
      PmResult getSystemWideRingtonesList(IN const ActType act = 0u) override;

      /**
       * Member function Overridden from IPmCoreRequestIf class
       *
       * @param[in]
       * @param[out]
       * @param[in,out]
       *
       * @return PmResult (PmResultCode, PmResultMessage)
       */
      PmResult getDeviceRingtoneList(IN const ActType act = 0u) override;

      /**
       * Member function Overridden from IPmCoreRequestIf class
       *
       * @param[in]
       * @param[out]
       * @param[in,out]
       *
       * @return PmResult (PmResultCode, PmResultMessage)
       */
      PmResult setRingtone(IN const BdAddress& bdAddress, IN const RingtoneId ringtoneId,
            IN const ActType act = 0u) override;

      /**
       * Member function Overridden from IPmCoreRequestIf class
       *
       * @param[in]
       * @param[out]
       * @param[in,out]
       *
       * @return PmResult (PmResultCode, PmResultMessage)
       */
      PmResult playRingtone(IN const RingtoneId ringtoneId, IN const PlayCount playCount,
            IN const ActType act = 0u) override;

      /**
       * Member function Overridden from IPmCoreRequestIf class
       *
       * @param[in]
       * @param[out]
       * @param[in,out]
       *
       * @return PmResult (PmResultCode, PmResultMessage)
       */
      PmResult stopRingtone(IN const ActType act = 0u) override;

      /**
       * Member function Overridden from IPmCoreRequestIf class
       *
       * @param[in]
       * @param[out]
       * @param[in,out]
       *
       * @return PmResult (PmResultCode, PmResultMessage)
       */
      PmResult getSuppressRingtoneOnOffList(IN const ActType act = 0u) override;

      /**
       * Member function Overridden from IPmCoreRequestIf class
       *
       * @param[in]
       * @param[out]
       * @param[in,out]
       *
       * @return PmResult (PmResultCode, PmResultMessage)
       */
      PmResult suppressRingtoneOnOff(IN const BdAddress& bdAddress, IN const SuppressRingtoneState suppressRingtoneState,
            IN const ActType act = 0u) override;

      /**
       * Member function Overridden from IPmCoreRequestIf class
       *
       * @param[in]
       * @param[out]
       * @param[in,out]
       *
       * @return PmResult (PmResultCode, PmResultMessage)
       */
      PmResult getAutoWaitingModeOnOffList(IN const ActType act = 0u) override;

      /**
       * Member function Overridden from IPmCoreRequestIf class
       *
       * @param[in]
       * @param[out]
       * @param[in,out]
       *
       * @return PmResult (PmResultCode, PmResultMessage)
       */
      PmResult autoWaitingModeOnOff(IN const BdAddress& bdAddress, IN const AutoWaitingModeState autoWaitingModeState,
            IN const ActType act = 0u) override;

      /**
       * Member function Overridden from IPmCoreRequestIf class
       *
       * @param[in]
       * @param[out]
       * @param[in,out]
       *
       * @return PmResult (PmResultCode, PmResultMessage)
       */
      PmResult setWaitingModeFilePath(IN const WaitingModeFilePath& waitingModeFilePath,
            IN const ActType act = 0u) override;

      /**
       * Member function Overridden from IPmCoreRequestIf class
       *
       * @param[in]
       * @param[out]
       * @param[in,out]
       *
       * @return PmResult (PmResultCode, PmResultMessage)
       */
      PmResult getWaitingModeFilePath(IN const ActType act = 0u) override;

      /**
       * Member function Overridden from IPmCoreRequestIf class
       *
       * @param[in]
       * @param[out]
       * @param[in,out]
       *
       * @return PmResult (PmResultCode, PmResultMessage)
       */
      PmResult getPhoneCallAudioActive(IN const ActType act = 0u) override;

      /**
       * Member function Overridden from IPmCoreRequestIf class
       *
       * @param[in]
       * @param[out]
       * @param[in,out]
       *
       * @return PmResult (PmResultCode, PmResultMessage)
       */
      PmResult getActivePassiveDeviceList(IN const ActType act = 0u) override;

      /**
       * Member function Overridden from IPmCoreRequestIf class
       *
       * @param[in]
       * @param[out]
       * @param[in,out]
       *
       * @return PmResult (PmResultCode, PmResultMessage)
       */
      PmResult setActivePhone(IN const BdAddressList& deviceAddressList, IN const ActType act = 0u) override;

      /**
       * Member function Overridden from IPmCoreRequestIf class
       *
       * @param[in]
       * @param[out]
       * @param[in,out]
       *
       * @return PmResult (PmResultCode, PmResultMessage)
       */
      PmResult restoreDefaultSetting(IN const BdAddress& deviceAddress, IN const ActType act = 0u) override;

      /**
       * Member function Overridden from IPmCoreRequestIf class
       *
       * @param[in]
       * @param[out]
       * @param[in,out]
       *
       * @return PmResult (PmResultCode, PmResultMessage)
       */
      PmResult getSupportedFeatures(IN const BdAddress& deviceAddress, IN const ActType act = 0u) override;

      /**
       * Member function Overridden from IPmCoreRequestIf class
       *
       * @param[in]
       * @param[out]
       * @param[in,out]
       *
       * @return PmResult (PmResultCode, PmResultMessage)
       */
      PmResult getNetworkStatus(IN const BdAddress& deviceAddress, IN const ActType act = 0u) override;

      /**
       * Member function Overridden from IPmCoreRequestIf class
       *
       * @param[in]
       * @param[out]
       * @param[in,out]
       *
       * @return PmResult (PmResultCode, PmResultMessage)
       */
      PmResult getSignalStrength(IN const BdAddress& deviceAddress, IN const ActType act = 0u) override;

      /**
       * Member function Overridden from IPmCoreRequestIf class
       *
       * @param[in]
       * @param[out]
       * @param[in,out]
       *
       * @return PmResult (PmResultCode, PmResultMessage)
       */
      PmResult getBatteryChargeLevel(IN const BdAddress& deviceAddress, IN const ActType act = 0u) override;

      /**
       * Member function Overridden from IPmCoreRequestIf class
       *
       * @param[in]
       * @param[out]
       * @param[in,out]
       *
       * @return PmResult (PmResultCode, PmResultMessage)
       */
      PmResult getNetworkOperator(IN const BdAddress& deviceAddress, IN const ActType act = 0u) override;

      /**
       * Member function Overridden from IPmCoreRequestIf class
       *
       * @param[in]
       * @param[out]
       * @param[in,out]
       *
       * @return PmResult (PmResultCode, PmResultMessage)
       */
      PmResult voiceMailList(IN const BdAddress& deviceAddress, IN const ActType act = 0u) override;

      /**
       * Member function Overridden from IPmCoreRequestIf class
       *
       * @param[in]
       * @param[out]
       * @param[in,out]
       *
       * @return PmResult (PmResultCode, PmResultMessage)
       */
      PmResult getCallStatusList(IN const ActType act = 0u) override;

      /**
       * Member function Overridden from IPmCoreRequestIf class
       *
       * @param[in]
       * @param[out]
       * @param[in,out]
       *
       * @return PmResult (PmResultCode, PmResultMessage)
       */
      PmResult getMicrophoneMuteState(IN const ActType act = 0u) override;

      /**
       * Member function Overridden from IPmCoreRequestIf class
       *
       * @param[in]
       * @param[out]
       * @param[in,out]
       *
       * @return PmResult (PmResultCode, PmResultMessage)
       */
      PmResult setMicrophoneMuteState(IN const MuteState microphoneMuteState, IN const ActType act = 0u) override;

      /**
       * Member function Overridden from IPmCoreRequestIf class
       *
       * @param[in]
       * @param[out]
       * @param[in,out]
       *
       * @return PmResult (PmResultCode, PmResultMessage)
       */
      PmResult mergeCalls(IN const DeviceCallInstanceListMap& mergeCallList, IN const ActType act = 0u) override;

      /**
       * Member function Overridden from IPmCoreRequestIf class
       *
       * @param[in]
       * @param[out]
       * @param[in,out]
       *
       * @return PmResult (PmResultCode, PmResultMessage)
       */
      PmResult splitCalls(IN const DeviceCallInstanceListMap& splitCallList, IN const ActType act = 0u) override;

      /**
       * Member function Overridden from IPmCoreRequestIf class
       *
       * @param[in]
       * @param[out]
       * @param[in,out]
       *
       * @return PmResult (PmResultCode, PmResultMessage)
       */
      PmResult hangupCalls(IN const DeviceCallInstanceListMap& hangupCallList, IN const ActType act = 0u) override;

      /**
       * Member function Overridden from IPmCoreRequestIf class
       *
       * @param[in]
       * @param[out]
       * @param[in,out]
       *
       * @return PmResult (PmResultCode, PmResultMessage)
       */
      PmResult redial(IN const BdAddress& deviceAddress, IN const HideCallerId hideCallerId,
            IN const ActType act = 0u) override;

      /**
       * Member function Overridden from IPmCoreRequestIf class
       *
       * @param[in]
       * @param[out]
       * @param[in,out]
       *
       * @return PmResult (PmResultCode, PmResultMessage)
       */
      PmResult getSCOConnection(IN const BdAddress& deviceAddress, IN const ActType act = 0u) override;

      /**
       * Member function Overridden from IPmCoreRequestIf class
       *
       * @param[in]
       * @param[out]
       * @param[in,out]
       *
       * @return PmResult (PmResultCode, PmResultMessage)
       */
      PmResult getRingtoneMuteState(IN const BdAddress& deviceAddress, IN const ActType act = 0u) override;

      /**
       * Member function Overridden from IPmCoreRequestIf class
       *
       * @param[in]
       * @param[out]
       * @param[in,out]
       *
       * @return PmResult (PmResultCode, PmResultMessage)
       */
      PmResult setRingtoneMuteState(IN const BdAddress& deviceAddress, IN const MuteState ringtoneMuteState,
            IN const ActType act = 0u) override;

      /**
       * Member function Overridden from IPmCoreRequestIf class
       *
       * @param[in]
       * @param[out]
       * @param[in,out]
       *
       * @return PmResult (PmResultCode, PmResultMessage)
       */
      PmResult getWaitingModeState(IN const BdAddress& deviceAddress, IN const ActType act = 0u) override;

      /**
       * Member function Overridden from IPmCoreRequestIf class
       *
       * @param[in]
       * @param[out]
       * @param[in,out]
       *
       * @return PmResult (PmResultCode, PmResultMessage)
       */
      PmResult getRingtonePlaybackStatus(IN const ActType act = 0u) override;

      /**
       * Member function Overridden from IPmCoreRequestIf class
       *
       * @param[in]
       * @param[out]
       * @param[in,out]
       *
       * @return PmResult (PmResultCode, PmResultMessage)
       */
      PmResult startStopWaitingMode(IN const BdAddress& deviceAddress, IN const StartStop startStopWaitingMode,
            IN const ActType act = 0u) override;

      /**
       * Member function Overridden from IPmCoreRequestIf class
       *
       * @param[in]
       * @param[out]
       * @param[in,out]
       *
       * @return PmResult (PmResultCode, PmResultMessage)
       */
      PmResult dial(IN const BdAddress& deviceAddress, IN const TelephoneNumber& telephoneNumber,
            IN const HideCallerId hideCallerId, IN const ActType act = 0u) override;

      /**
       * Member function Overridden from IPmCoreRequestIf class
       *
       * @param[in]
       * @param[out]
       * @param[in,out]
       *
       * @return PmResult (PmResultCode, PmResultMessage)
       */
      PmResult acceptCall(IN const BdAddress& deviceAddress, IN const CallInstance callInstance,
            IN const AcceptOperation operation, IN const ActType act = 0u) override;

      /**
       * Member function Overridden from IPmCoreRequestIf class
       *
       * @param[in]
       * @param[out]
       * @param[in,out]
       *
       * @return PmResult (PmResultCode, PmResultMessage)
       */
      PmResult swapCall(IN const BdAddress& deviceAddress, IN const SwapOperation operation,
            IN const ActType act = 0u) override;

      /**
       * Member function Overridden from IPmCoreRequestIf class
       *
       * @param[in]
       * @param[out]
       * @param[in,out]
       *
       * @return PmResult (PmResultCode, PmResultMessage)
       */
      PmResult speedDial(IN const BdAddress& deviceAddress, IN const MemDialIndex memDialIndex,
            IN const HideCallerId hideCallerId, IN const ActType act = 0u) override;

      /**
       * Member function Overridden from IPmCoreRequestIf class
       *
       * @param[in]
       * @param[out]
       * @param[in,out]
       *
       * @return PmResult (PmResultCode, PmResultMessage)
       */
      PmResult sendDTMF(IN const BdAddress& deviceAddress, IN const DTMFTones& tones, IN const ActType act = 0u) override;

      /**
       * Member function Overridden from IPmCoreRequestIf class
       *
       * @param[in]
       * @param[out]
       * @param[in,out]
       *
       * @return PmResult (PmResultCode, PmResultMessage)
       */
      PmResult transferAudio(IN const BdAddress& deviceAddress, IN const AudioDirection audioDirection,
            IN const ActType act = 0u) override;

      /**
       * Member function Overridden from IPmCoreRequestIf class
       *
       * @param[in]
       * @param[out]
       * @param[in,out]
       *
       * @return PmResult (PmResultCode, PmResultMessage)
       */
      PmResult getVoiceRecognitionStatus(IN const BdAddress& deviceAddress, IN const ActType act = 0u) override;

      /**
       * Member function Overridden from IPmCoreRequestIf class
       *
       * @param[in]
       * @param[out]
       * @param[in,out]
       *
       * @return PmResult (PmResultCode, PmResultMessage)
       */
      PmResult startStopVoiceRecognition(IN const BdAddress& deviceAddress, IN const StartStop startStopVR,
            IN const ActType act = 0u) override;

      /**
       * Member function Overridden from IPmCoreRequestIf class
       *
       * @param[in]
       * @param[out]
       * @param[in,out]
       *
       * @return PmResult (PmResultCode, PmResultMessage)
       */
      PmResult getEnhancedVoiceRecognitionFeature(IN const BdAddress& deviceAddress, IN const ActType act = 0u) override;

      /**
       * Member function Overridden from IPmCoreRequestIf class
       *
       * @param[in]
       * @param[out]
       * @param[in,out]
       *
       * @return PmResult (PmResultCode, PmResultMessage)
       */
      PmResult getExternalVRState(IN const BdAddress& deviceAddress, IN const ActType act = 0u) override;

      /**
       * Member function Overridden from IPmCoreRequestIf class
       *
       * @param[in]
       * @param[out]
       * @param[in,out]
       *
       * @return PmResult (PmResultCode, PmResultMessage)
       */
      PmResult activateExternalVR(IN const BdAddress& deviceAddress, IN const ActType act = 0u) override;

      /**
       * Member function Overridden from IPmCoreRequestIf class
       *
       * @param[in]
       * @param[out]
       * @param[in,out]
       *
       * @return PmResult (PmResultCode, PmResultMessage)
       */
      PmResult deActivateExternalVR(IN const BdAddress& deviceAddress, IN const ActType act = 0u) override;

      /**
       * This method is used to stop the extended voice recognition session in the AG for the given device address.
       *
       * @param[in] BdAddress& - device address
       * @param[in] ActType - asynchronous completion token
       * @param[out]
       * @param[in,out]
       *
       * @return PmResult (PmResultCode, PmResultMessage)
       */
      virtual PmResult stopExtendedVR(IN const BdAddress& deviceAddress, IN const ActType act = 0u) override;

   private:
      friend class PmSingleton<PmInterface>;

      /**
       * Constructor of PmInterface class
       *
       * @param[in]
       * @param[out]
       * @param[in,out]
       *
       * @return
       */
      PmInterface();

      /**
       * Destructor of PmInterface class
       *
       * @param[in]
       * @param[out]
       * @param[in,out]
       *
       * @return
       */
      virtual ~PmInterface();
   };
}

#endif // PmCoreRequestIf_h
