/**
 * @file PmCoreCallbackIfWrapper.h
 *
 * @swcomponent PhoneCallManager
 *
 * @brief This file contains the definition of the PmCoreCallbackIfWrapper class
 *
 * @copyright (C) 2016 Robert Bosch GmbH.
 *            The reproduction, distribution and utilization of this file as
 *            well as the communication of its contents to others without express
 *            authorization is prohibited. Offenders will be held liable for the
 *            payment of damages. All rights reserved in the event of the grant
 *            of a patent, utility model or design.
 *
 * @details
 *
 * @ingroup PmCore
 */

#ifndef PmCoreCallbackIfWrapper_h
#define PmCoreCallbackIfWrapper_h

#include "PmCoreIfTypes.h"

namespace pmcore
{

   /**
    * forward declaration
    */
   class IPmCoreCallbackIf;

   /**
    * PmCoreCallbackIfWrapper class definition
    */
   class PmCoreCallbackIfWrapper final
   {
   public:

      /**
       * Constructor of the interface class PmCoreCallbackIfWrapper
       *
       * @param[in]
       * @param[out]
       * @param[in,out]
       *
       * @return
       */
      PmCoreCallbackIfWrapper(IPmCoreCallbackIf* pmCoreCallbackIf);

      /**
       * virtual destructor of the interface class PmCoreCallbackIfWrapper
       *
       * @param[in]
       * @param[out]
       * @param[in,out]
       *
       * @return
       */
      virtual ~PmCoreCallbackIfWrapper();

      /**
       * This method provides the system wide ringtone list with the currently selected ringtone.
       *
       * @param[in] PmResult& - reference to PmResult class
       * @param[in] SystemWideRingtonesList& - reference to SystemWideRingtonesList class
       * @param[in] ActType - asynchronous completion token
       * @param[out]
       * @param[in,out]
       *
       * @return void
       */
      void doGetSystemWideRingtonesListResponse(IN const PmResult& pmResult,
            IN const SystemWideRingtonesList& systemWideRingtonesList, IN const ActType act = 0u) const;

      /**
       * This method is used to notify the changes in SystemWideRingtonesList.
       *
       * @param[in] SystemWideRingtonesList& - reference to SystemWideRingtonesList class
       * @param[out]
       * @param[in,out]
       *
       * @return void
       */
      void doOnSystemWideRingtonesListChanged(IN const SystemWideRingtonesList& systemWideRingtonesList) const;

      /**
       * This interface provides a list of paired devices with its currently selected ringtone.
       *
       * @param[in] PmResult& - reference to PmResult class
       * @param[in] DeviceRingtoneList& - reference to DeviceRingtoneList class
       * @param[in] ActType - asynchronous completion token
       * @param[out]
       * @param[in,out]
       *
       * @return void
       */
      void doGetDeviceRingtoneListResponse(IN const PmResult& pmResult,
            IN const DeviceRingtoneList& deviceRingtoneList, IN const ActType act = 0u) const;

      /**
       * This method is used to notify the changes in SystemWideRingtonesList.
       *
       * @param[in] DeviceRingtoneList& - reference to DeviceRingtoneList class
       * @param[out]
       * @param[in,out]
       *
       * @return void
       */
      void doOnDeviceRingtoneListChanged(IN const DeviceRingtoneList& deviceRingtoneList) const;

      /**
       * Response for the method setRingtone.
       *
       * @param[in] PmResult& - reference to PmResult class
       * @param[in] ActType - asynchronous completion token
       * @param[out]
       * @param[in,out]
       *
       * @return void
       */
      void doSetRingtoneResponse(IN const PmResult& pmResult, IN const ActType act = 0u) const;

      /**
       * Response for the method playRingtone.
       *
       * @param[in] PmResult& - reference to PmResult class
       * @param[in] ActType - asynchronous completion token
       * @param[out]
       * @param[in,out]
       *
       * @return void
       */
      void doPlayRingtoneResponse(IN const PmResult& pmResult, IN const ActType act = 0u) const;

      /**
       * Response for the method stopRingtone.
       *
       * @param[in] PmResult& - reference to PmResult class
       * @param[in] ActType - asynchronous completion token
       * @param[out]
       * @param[in,out]
       *
       * @return void
       */
      void doStopRingtoneResponse(IN const PmResult& pmResult, IN const ActType act = 0u) const;

      /**
       * This method provides the suppress ringtone setting state for all paired devices.
       *
       * @param[in] PmResult& - reference to PmResult class
       * @param[in] SuppressRingtoneOnOffList& - reference to SuppressRingtoneOnOffList class
       * @param[in] ActType - asynchronous completion token
       * @param[out]
       * @param[in,out]
       *
       * @return void
       */
      void doGetSuppressRingtoneOnOffListResponse(IN const PmResult& pmResult,
            IN const SuppressRingtoneOnOffList& suppressRingtoneOnOffList, IN const ActType act = 0u) const;

      /**
       * This method is used to notify the changes in SuppressRingtoneOnOffList.
       *
       * @param[in] SuppressRingtoneOnOffList& - reference to SuppressRingtoneOnOffList class
       * @param[out]
       * @param[in,out]
       *
       * @return void
       */
      void doOnSuppressRingtoneOnOffListChanged(IN const SuppressRingtoneOnOffList& suppressRingtoneOnOffList) const;

      /**
       * Response for the method suppressRingtoneOnOff.
       *
       * @param[in] PmResult& - reference to PmResult class
       * @param[in] ActType - asynchronous completion token
       * @param[out]
       * @param[in,out]
       *
       * @return void
       */
      void doSuppressRingtoneOnOffResponse(IN const PmResult& pmResult, IN const ActType act = 0u) const;

      /**
       * This method provides the auto waiting mode setting state for all paired devices.
       *
       * @param[in] PmResult& - reference to PmResult class
       * @param[in] AutoWaitingModeOnOffList& - reference to AutoWaitingModeOnOffList class
       * @param[in] ActType - asynchronous completion token
       * @param[out]
       * @param[in,out]
       *
       * @return void
       */
      void doGetAutoWaitingModeOnOffListResponse(IN const PmResult& pmResult,
            IN const AutoWaitingModeOnOffList& autoWaitingModeOnOffList, IN const ActType act = 0u) const;

      /**
       *  This method is used to notify the changes in AutoWaitingModeOnOffList.
       *
       * @param[in] AutoWaitingModeOnOffList& - reference to AutoWaitingModeOnOffList class
       * @param[out]
       * @param[in,out]
       *
       * @return void
       */
      void doOnAutoWaitingModeOnOffListChanged(IN AutoWaitingModeOnOffList& autoWaitingModeOnOffList) const;

      /**
       * Response for the method AutoWaitingModeOnOff.
       *
       * @param[in] PmResult& - reference to PmResult class
       * @param[in] ActType - asynchronous completion token
       * @param[out]
       * @param[in,out]
       *
       * @return void
       */
      void doAutoWaitingModeOnOffResponse(IN const PmResult& pmResult, IN const ActType act = 0u) const;

      /**
       * This method sets the filepath for WaitingMode NarrowBand and WideBand audio files.
       *
       * @param[in] PmResult& - reference to PmResult class
       * @param[in] ActType - asynchronous completion token
       * @param[out]
       * @param[in,out]
       *
       * @return void
       */
      void doSetWaitingModeFilePathResponse(IN const PmResult& pmResult, IN const ActType act = 0u) const;

      /**
       * This method provides the filepath for WaitingMode NarrowBand and WideBand audio files.
       *
       * @param[in] PmResult& - reference to PmResult class
       * @param[in] WaitingModeFilePath& - reference to WaitingModeFilePath class
       * @param[in] ActType - asynchronous completion token
       * @param[out]
       * @param[in,out]
       *
       * @return void
       */
      void doGetWaitingModeFilePathResponse(IN const PmResult& pmResult,
            IN const WaitingModeFilePath& waitingModeFilePath, IN const ActType act = 0u) const;

      /**
       * This method is used to notify the changes in WaitingModeFilePath.
       *
       * @param[in] WaitingModeFilePath& - reference to WaitingModeFilePath class
       * @param[out]
       * @param[in,out]
       *
       * @return void
       */
      void doOnWaitingModeFilePathChanged(IN const WaitingModeFilePath& waitingModeFilePath) const;

      /**
       * This method provides the state of phone call audio streaming.
       * The value will be set as TRUE when:
       * * SCO is established between AG and HF AND
       * * Audio channel is available AND
       * * ECNR session is started.
       * In all other cases the Status will be reported as FALSE.
       *
       * @param[in] PmResult& - reference to PmResult class
       * @param[in] PhoneCallAudioActive& - reference to PhoneCallAudioActive class
       * @param[in] ActType - asynchronous completion token
       * @param[out]
       * @param[in,out]
       *
       * @return void
       */
      void doGetPhoneCallAudioActiveResponse(IN const PmResult& pmResult,
            IN const PhoneCallAudioActive& phoneCallAudioActive, IN const ActType act = 0u) const;

      /**
       * This method is used to notify the changes in PhoneCallAudioActive.
       *
       * @param[in] PhoneCallAudioActive& - reference to PhoneCallAudioActive class
       * @param[out]
       * @param[in,out]
       *
       * @return void
       */
      void doOnPhoneCallAudioActiveChanged(IN const PhoneCallAudioActive& phoneCallAudioActive) const;

      /**
       * This method provides the list of devices with current role (active / passive)
       *
       * @param[in] PmResult& - reference to PmResult class
       * @param[in] ActivePassiveDeviceList& - reference to ActivePassiveDeviceList class
       * @param[in] ActType - asynchronous completion token
       * @param[out]
       * @param[in,out]
       *
       * @return void
       */
      void doGetActivePassiveDeviceListResponse(IN const PmResult& pmResult,
            IN const ActivePassiveDeviceList& activePassiveDeviceList, IN const ActType act = 0u) const;

      /**
       * This method is used to notify the changes in ActivePassiveDeviceList.
       *
       * @param[in] ActivePassiveDeviceList& - reference to ActivePassiveDeviceList class
       * @param[out]
       * @param[in,out]
       *
       * @return void
       */
      void doOnActivePassiveDeviceListChanged(IN const ActivePassiveDeviceList& activePassiveDeviceList) const;

      /**
       * Response for the method setActivePhone.
       *
       * @param[in] PmResult& - reference to PmResult class
       * @param[in] ActType - asynchronous completion token
       * @param[out]
       * @param[in,out]
       *
       * @return void
       */
      void doSetActivePhoneResponse(IN const PmResult& pmResult, IN const ActType act = 0u) const;

      /**
       * Response for the method restoreDefaultSetting.
       *
       * @param[in] PmResult& - reference to PmResult class
       * @param[in] ActType - asynchronous completion token
       * @param[out]
       * @param[in,out]
       *
       * @return void
       */
      void doRestoreDefaultSettingResponse(IN const PmResult& pmResult, IN const ActType act = 0u) const;

      /**
       * This method is used to indicate the different phone call manager related events.
       * Examples Events:
       * IncomingCallRejected
       * OutgoingCallFailed
       *
       * @param[in] NotificationEvent& - reference to NotificationEvent class
       * @param[out]
       * @param[in,out]
       *
       * @return void
       */
      void doOnNotificationEvent(IN const NotificationEvent& notificationEvent) const;

      /**
       * This method provides the supported features of the connected device.
       *
       * @param[in] PmResult& - reference to PmResult class
       * @param[in] BdAddress& - device address
       * @param[in] SupportedFeatures& - reference to SupportedFeatures class
       * @param[in] ActType - asynchronous completion token
       * @param[out]
       * @param[in,out]
       *
       * @return void
       */
      void doGetSupportedFeaturesResponse(IN const PmResult& pmResult, IN const BdAddress& deviceAddress,
            IN const SupportedFeatures& supportedFeatures, IN const ActType act = 0u) const;

      /**
       * This method is used to notify the changes in SupportedFeatures of AudioGateway.
       *
       * @param[in] BdAddress& - device address
       * @param[in] SupportedFeatures& - reference to SupportedFeatures class
       * @param[out]
       * @param[in,out]
       *
       * @return void
       */
      void doOnSupportedFeaturesChanged(IN const BdAddress& deviceAddress, IN const SupportedFeatures& supportedFeatures) const;

      /**
       * This method provides the network status of the connected device.
       *
       * @param[in] PmResult& - reference to PmResult class
       * @param[in] BdAddress& - device address
       * @param[in] NetworkStatus& - reference to NetworkStatus class
       * @param[in] ActType - asynchronous completion token
       * @param[out]
       * @param[in,out]
       *
       * @return void
       */
      void doGetNetworkStatusResponse(IN const PmResult& pmResult, IN const BdAddress& deviceAddress,
            IN const NetworkStatus& networkStatus, IN const ActType act = 0u) const;

      /**
       * This method is used to notify the changes in NetworkStatus of AudioGateway.
       *
       * @param[in] BdAddress& - device address
       * @param[in] NetworkStatus& - reference to NetworkStatus class
       * @param[out]
       * @param[in,out]
       *
       * @return void
       */
      void doOnNetworkStatusChanged(IN const BdAddress& deviceAddress, IN const NetworkStatus& networkStatus) const;

      /**
       * This method provides the signal strength of the connected device.
       *
       * @param[in] PmResult& - reference to PmResult class
       * @param[in] BdAddress& - device address
       * @param[in] SignalStrength& - reference to SignalStrength class
       * @param[in] ActType - asynchronous completion token
       * @param[out]
       * @param[in,out]
       *
       * @return void
       */
      void doGetSignalStrengthResponse(IN const PmResult& pmResult, IN const BdAddress& deviceAddress,
            IN const SignalStrength& signalStrength, IN const ActType act = 0u) const;

      /**
       * This method is used to notify the changes in SignalStrength of AudioGateway.
       *
       * @param[in] BdAddress& - device address
       * @param[in] SignalStrength& - reference to SignalStrength class
       * @param[out]
       * @param[in,out]
       *
       * @return void
       */
      void doOnSignalStrengthChanged(IN const BdAddress& deviceAddress, IN const SignalStrength& signalStrength) const;

      /**
       * This method provides the battery charge level of the connected device.
       *
       * @param[in] PmResult& - reference to PmResult class
       * @param[in] BdAddress& - device address
       * @param[in] BatteryChargeLevel& - reference to BatteryChargeLevel class
       * @param[in] ActType - asynchronous completion token
       * @param[out]
       * @param[in,out]
       *
       * @return void
       */
      void doGetBatteryChargeLevelResponse(IN const PmResult& pmResult, IN const BdAddress& deviceAddress,
            IN const BatteryChargeLevel& batteryChargeLevel, IN const ActType act = 0u) const;

      /**
       * This method is used to notify the changes in BatteryChargeLevel of AudioGateway.
       *
       * @param[in] BdAddress& - device address
       * @param[in] BatteryChargeLevel& - reference to BatteryChargeLevel class
       * @param[out]
       * @param[in,out]
       *
       * @return void
       */
      void doOnBatteryChargeLevelChanged(IN const BdAddress& deviceAddress,
            IN const BatteryChargeLevel& batteryChargeLevel) const;

      /**
       * This method provides the network operator of the connected device.
       *
       * @param[in] PmResult& - reference to PmResult class
       * @param[in] BdAddress& - device address
       * @param[in] NetworkOperator& - reference to NetworkOperator class
       * @param[in] ActType - asynchronous completion token
       * @param[out]
       * @param[in,out]
       *
       * @return void
       */
      void doGetNetworkOperatorResponse(IN const PmResult& pmResult, IN const BdAddress& deviceAddress,
            IN const NetworkOperator& networkOperator, IN const ActType act = 0u) const;

      /**
       * This method is used to notify the changes in NetworkOperator of AudioGateway.
       *
       * @param[in] BdAddress& - device address
       * @param[in] NetworkOperator& - reference to NetworkOperator class
       * @param[out]
       * @param[in,out]
       *
       * @return void
       */
      void doOnNetworkOperatorChanged(IN const BdAddress& deviceAddress, IN const NetworkOperator& networkOperator) const;

      /**
       * Response for the method VoiceMailList.
       *
       * @param[in] PmResult& - reference to PmResult class
       * @param[in] BdAddress& - device address
       * @param[in] VoiceMailList& - reference to VoiceMailList class
       * @param[in] ActType - asynchronous completion token
       * @param[out]
       * @param[in,out]
       *
       * @return void
       */
      void doVoiceMailListResponse(IN const PmResult& pmResult, IN const BdAddress& deviceAddress,
            IN const VoiceMailList& voiceMailList, IN const ActType act = 0u) const;

      /**
       * This method is used to get the call status of the connected devices.
       *
       * @param[in] PmResult& - reference to PmResult class
       * @param[in] CallStatusList& - reference to CallStatusList class
       * @param[in] ActType - asynchronous completion token
       * @param[out]
       * @param[in,out]
       *
       * @return void
       */
      void doGetCallStatusListResponse(IN const PmResult& pmResult,
            IN const CallStatusList& callStatusList, IN const ActType act = 0u) const;

      /**
       * This method is used to notify the changes in CallStatusList.
       *
       * @param[in] CallStatusList& - reference to CallStatusList class
       * @param[out]
       * @param[in,out]
       *
       * @return void
       */
      void doOnCallStatusListChanged(IN const CallStatusList& callStatusList) const;

      /**
       * This method is used to get the state of Microphone whether it is in mute or not.
       *
       * @param[in] PmResult& - reference to PmResult class
       * @param[in] MicrophoneMuteState& - reference to MicrophoneMuteState class
       * @param[in] ActType - asynchronous completion token
       * @param[out]
       * @param[in,out]
       *
       * @return void
       */
      void doGetMicrophoneMuteStateResponse(IN const PmResult& pmResult,
            IN const MicrophoneMuteState& microphoneMuteState, IN const ActType act = 0u) const;

      /**
       * This method is used to notify the changes in MicrophoneMuteState.
       *
       * @param[in] MicrophoneMuteState& - reference to MicrophoneMuteState class
       * @param[out]
       * @param[in,out]
       *
       * @return void
       */
      void doOnMicrophoneMuteStateChanged(IN const MicrophoneMuteState& microphoneMuteState) const;

      /**
       * Response for the setMicrophoneMuteState request.
       *
       * @param[in] PmResult& - reference to PmResult class
       * @param[in] ActType - asynchronous completion token
       * @param[out]
       * @param[in,out]
       *
       * @return void
       */
      void doSetMicrophoneMuteStateResponse(IN const PmResult& pmResult, IN const ActType act = 0u) const;

      /**
       * Response for the MergeCalls request for the device.
       *
       * @param[in] PmResult& - reference to PmResult class
       * @param[in] ActType - asynchronous completion token
       * @param[out]
       * @param[in,out]
       *
       * @return void
       */
      void doMergeCallsResponse(IN const PmResult& pmResult, IN const ActType act = 0u) const;

      /**
       * Response for the SplitCalls request for the device.
       *
       * @param[in] PmResult& - reference to PmResult class
       * @param[in] ActType - asynchronous completion token
       * @param[out]
       * @param[in,out]
       *
       * @return
       */
      void doSplitCallsResponse(IN const PmResult& pmResult, IN const ActType act = 0u) const;

      /**
       * Response for the HangupCalls request for the device.
       *
       * @param[in] PmResult& - reference to PmResult class
       * @param[in] ActType - asynchronous completion token
       * @param[out]
       * @param[in,out]
       *
       * @return void
       */
      void doHangupCallsResponse(IN const PmResult& pmResult, IN const ActType act = 0u) const;

      /**
       * Response for the Redial request for the device.
       *
       * @param[in] PmResult& - reference to PmResult class
       * @param[in] BdAddress& - device address
       * @param[in] CallInstance& - call instance
       * @param[in] ActType - asynchronous completion token
       * @param[out]
       * @param[in,out]
       *
       * @return void
       */
      void doRedialResponse(IN const PmResult& pmResult, IN const BdAddress& deviceAddress, IN const CallInstance callInstance,
            IN const ActType act = 0u) const;

      /**
       * This method is used to get the current status of SCO establishment between AG and HF.
       *
       * @param[in] PmResult& - reference to PmResult class
       * @param[in] BdAddress& - device address
       * @param[in] SCOConnection& - reference to SCOConnection class
       * @param[in] ActType - asynchronous completion token
       * @param[out]
       * @param[in,out]
       *
       * @return void
       */
      void doGetSCOConnectionResponse(IN const PmResult& pmResult, IN const BdAddress& deviceAddress,
            IN const SCOConnection& scoConnection, IN const ActType act = 0u) const;

      /**
       * This method is used to notify the changes in SCO Connection.
       *
       * @param[in] BdAddress& - device address
       * @param[in] SCOConnection& - reference to SCOConnection class
       * @param[out]
       * @param[in,out]
       *
       * @return void
       */
      void doOnSCOConnectionChanged(IN const BdAddress& deviceAddress, IN const SCOConnection& scoConnection) const;

      /**
       * This method is used to get the playing ringtone mute state.
       *
       * @param[in] PmResult& - reference to PmResult class
       * @param[in] BdAddress& - device address
       * @param[in] RingtoneMuteState& - reference to RingtoneMuteState class
       * @param[in] ActType - asynchronous completion token
       * @param[out]
       * @param[in,out]
       *
       * @return void
       */
      void doGetRingtoneMuteStateResponse(IN const PmResult& pmResult, IN const BdAddress& deviceAddress,
            IN const RingtoneMuteState& ringtoneMuteState, IN const ActType act = 0u) const;

      /**
       * This method is used to notify the changes in RingtoneMuteState of the device.
       *
       * @param[in] BdAddress& - device address
       * @param[in] RingtoneMuteState& - reference to RingtoneMuteState class
       * @param[out]
       * @param[in,out]
       *
       * @return void
       */
      void doOnRingtoneMuteStateChanged(IN const BdAddress& deviceAddress, IN const RingtoneMuteState& ringtoneMuteState) const;

      /**
       * Response for the setRingtoneMuteState request.
       *
       * @param[in] PmResult& - reference to PmResult class
       * @param[in] BdAddress& - device address
       * @param[in] ActType - asynchronous completion token
       * @param[out]
       * @param[in,out]
       *
       * @return void
       */
      void doSetRingtoneMuteStateResponse(IN const PmResult& pmResult, IN const BdAddress& deviceAddress,
            IN const ActType act = 0u) const;

      /**
       * This method is used to get the  current waiting mode state.
       *
       * @param[in] PmResult& - reference to PmResult class
       * @param[in] BdAddress& - device address
       * @param[in] WaitingModeState& - reference to WaitingModeState class
       * @param[in] ActType - asynchronous completion token
       * @param[out]
       * @param[in,out]
       *
       * @return void
       */
      void doGetWaitingModeStateResponse(IN const PmResult& pmResult, IN const BdAddress& deviceAddress,
            IN const WaitingModeState& waitingModeState, IN const ActType act = 0u) const;

      /**
       * This method is used to notify the changes in WaitingModeState of the device.
       *
       * @param[in] BdAddress& - device address
       * @param[in] WaitingModeState& - reference to WaitingModeState class
       * @param[out]
       * @param[in,out]
       *
       * @return void
       */
      void doOnWaitingModeStateChanged(IN const BdAddress& deviceAddress, IN const WaitingModeState& waitingModeState) const;

      /**
       * This method is used to get the  current playback status of the vehicle ringtone
       *
       * @param[in] PmResult& - reference to PmResult class
       * @param[in] ringtonePlaybackState& - reference to RingtonePlaybackState class
       * @param[in] ActType - asynchronous completion token
       * @param[out]
       * @param[in,out]
       *
       * @return void
       */
      void doGetRingtonePlaybackStatusResponse(IN const pmcore::PmResult& pmResult,
            IN const pmcore::RingtonePlaybackState& ringtonePlaybackState, IN const pmcore::ActType act = 0u) const;

      /**
       * This method is used to notify the changes in ringtonePlaybackState of the vehicle ringtone.
       *
       * @param[in] ringtonePlaybackState& - reference to RingtonePlaybackState class
       * @param[out]
       * @param[in,out]
       *
       * @return void
       */
      void doOnRingtonePlaybackStatusChanged(
            IN const pmcore::RingtonePlaybackState& ringtonePlaybackState) const;

      /**
       * Response for the startStopWaitingMode request.
       *
       * @param[in] PmResult& - reference to PmResult class
       * @param[in] BdAddress& - device address
       * @param[in] ActType - asynchronous completion token
       * @param[out]
       * @param[in,out]
       *
       * @return void
       */
      void doStartStopWaitingModeResponse(IN const PmResult& pmResult, IN const BdAddress& deviceAddress,
            IN const ActType act = 0u) const;

      /**
       * Response for the dial request.
       *
       * @param[in] PmResult& - reference to PmResult class
       * @param[in] BdAddress& - device address
       * @param[in] CallInstance - call instance
       * @param[in] ActType - asynchronous completion token
       * @param[out]
       * @param[in,out]
       *
       * @return void
       */
      void doDialResponse(IN const PmResult& pmResult, IN const BdAddress& deviceAddress,
            IN const CallInstance callInstance, IN const ActType act = 0u) const;

      /**
       * Response for the acceptCall request.
       *
       * @param[in] PmResult& - reference to PmResult class
       * @param[in] BdAddress& - device address
       * @param[in] CallInstance - call instance
       * @param[in] ActType - asynchronous completion token
       * @param[out]
       * @param[in,out]
       *
       * @return void
       */
      void doAcceptCallResponse(IN const PmResult& pmResult, IN const BdAddress& deviceAddress,
            IN const CallInstance callInstance, IN const ActType act = 0u) const;

      /**
       * Response for the swapCall request.
       *
       * @param[in] PmResult& - reference to PmResult class
       * @param[in] BdAddress& - device address
       * @param[in] ActType - asynchronous completion token
       * @param[out]
       * @param[in,out]
       *
       * @return void
       */
      void doSwapCallResponse(IN const PmResult& pmResult, IN const BdAddress& deviceAddress,
            IN const ActType act = 0u) const;

      /**
       * Response for the speedDial request.
       *
       * @param[in] PmResult& - reference to PmResult class
       * @param[in] BdAddress& - device address
       * @param[in] CallInstance - call instance
       * @param[in] ActType - asynchronous completion token
       * @param[out]
       * @param[in,out]
       *
       * @return void
       */
      void doSpeedDialResponse(IN const PmResult& pmResult, IN const BdAddress& deviceAddress,
            IN const CallInstance callInstance, IN const ActType act = 0u) const;

      /**
       * Response for the SendDTMF request for the device.
       *
       * @param[in] PmResult& - reference to PmResult class
       * @param[in] BdAddress& - device address
       * @param[in] ActType - asynchronous completion token
       * @param[out]
       * @param[in,out]
       *
       * @return void
       */
      void doSendDTMFResponse(IN const PmResult& pmResult, IN const BdAddress& deviceAddress, IN const ActType act = 0u) const;

      /**
       * Response for the TransferAudio request for the device.
       *
       * @param[in] PmResult& - reference to PmResult class
       * @param[in] BdAddress& - device address
       * @param[in] ActType - asynchronous completion token
       * @param[out]
       * @param[in,out]
       *
       * @return void
       */
      void doTransferAudioResponse(IN const PmResult& pmResult, IN const BdAddress& deviceAddress,
            IN const ActType act = 0u) const;

      /**
       * This method provides the status of the Voice Recognition session.
       *
       * @param[in] PmResult& - reference to PmResult class
       * @param[in] BdAddress& - device address
       * @param[in] VoiceRecognitionStatus& - reference to VoiceRecognitionStatus class
       * @param[in] ActType - asynchronous completion token
       * @param[out]
       * @param[in,out]
       *
       * @return void
       */
      void doGetVoiceRecognitionStatusResponse(IN const PmResult& pmResult, IN const BdAddress& deviceAddress,
            IN const VoiceRecognitionStatus& voiceRecognitionStatus, IN const ActType act = 0u) const;

      /**
       * This method is used to notify the status of Voice Recognition.
       *
       * @param[in] BdAddress& - device address
       * @param[in] VoiceRecognitionStatus& - reference to VoiceRecognitionStatus class
       * @param[out]
       * @param[in,out]
       *
       * @return void
       */
      void doOnVoiceRecognitionStatusChanged(IN const BdAddress& deviceAddress,
            IN const VoiceRecognitionStatus& voiceRecognitionStatus) const;

      /**
       * Response for the startStopVoiceRecognition request.
       * 
       * @param[in] PmResult& - reference to PmResult class
       * @param[in] BdAddress& - Reference to Device address class
       * @param[in] ActType - asynchronous completion token
       * @param[out]
       * @param[in,out]
       *
       * @return void
       */
      void doStartStopVoiceRecognitionResponse(IN const PmResult& pmResult, IN const BdAddress& deviceAddress,
             IN const ActType act = 0u) const;

      /**
       * This method provides the VoiceRecognition feature available in the AG and current availability state of the same.
       *
       * @param[in] PmResult& - reference to PmResult class
       * @param[in] BdAddress& - device address
       * @param[in] EnhancedVoiceRecognitionFeature& - reference to EnhancedVoiceRecognitionFeature class
       * @param[in] ActType - asynchronous completion token
       * @param[out]
       * @param[in,out]
       *
       * @return void
       */
      void doGetEnhancedVoiceRecognitionFeatureResponse(IN const PmResult& pmResult, IN const BdAddress& deviceAddress,
            IN const EnhancedVoiceRecognitionFeature& enhancedVoiceRecognitionFeature, IN const ActType act = 0u) const;

      /**
       * This method is used to notify the changes in the availability of the Enhanced VR feature
       * (SIRI in iPhone, Google Voice in Android phones, etc..) in the AudioGateway.
       *
       * @param[in] BdAddress& - device address
       * @param[in] EnhancedVoiceRecognitionFeature& - reference to EnhancedVoiceRecognitionFeature class
       * @param[out]
       * @param[in,out]
       *
       * @return void
       */
      void doOnEnhancedVoiceRecognitionFeatureChanged(IN const BdAddress& deviceAddress,
            IN const EnhancedVoiceRecognitionFeature& enhancedVoiceRecognitionFeature) const;

      /**
       * This method provides the external VR state for the requested device
       *
       * @param[in] pmcore::PmResult& - reference to pmcore::PmResult class
       * @param[in] pmcore::BdAddress& - device address
       * @param[in] pmcore::ExtVoiceRecognitionStatus& - external VR status
       * @param[in] pmcore::ActType - asynchronous completion token
       * @param[out]
       * @param[in,out]
       *
       * @return void
       */
      void doGetExternalVRStateResponse(IN const pmcore::PmResult& pmResult,
            IN const pmcore::BdAddress& deviceAddress,
            IN const pmcore::ExtVoiceRecognitionStatus& extVoiceRecognitionStatus,
            IN const pmcore::ActType act = 0u) const;

      /**
       * This method is used to notify the changes in external VR state
       *
       * @param[in] pmcore::BdAddress& - device address
       * @param[in] pmcore::ExtVoiceRecognitionStatus& - external VR status
       * @param[out]
       * @param[in,out]
       *
       * @return void
       */
      void doOnExternalVRStateChanged(IN const pmcore::BdAddress& deviceAddress,
            IN const pmcore::ExtVoiceRecognitionStatus& extVoiceRecognitionStatus) const;

      /**
       * Response for the activateExternalVR request.
       * @param[in] PmResult& - reference to PmResult class
       * @param[in] BdAddress& deviceAddress
       * @param[in] ActType - asynchronous completion token
       * @param[out]
       * @param[in,out]
       *
       * @return void
       */
      virtual void doActivateExternalVRResponse(IN const PmResult& pmResult, IN const BdAddress& deviceAddress,
            IN const ActType act = 0u) const;

      /**
       * Response for the deActivateExternalVR request.
       * @param[in] PmResult& - reference to PmResult class
       * @param[in] BdAddress& deviceAddress
       * @param[in] ActType - asynchronous completion token
       * @param[out]
       * @param[in,out]
       *
       * @return void
       */
      virtual void doDeActivateExternalVRResponse(IN const PmResult& pmResult, IN const BdAddress& deviceAddress,
            IN const ActType act = 0u) const;

      /**
       * This method is used to notify the Call added or call added with SCO established status to the clients
       * so that the Phone sub state property is set
       *
       * @param[in]  deviceAddres - Device address for which the call is added
       *             callExistenceState - call existence status
       * @param[in]
       * @param[out]
       * @param[in,out]
       *
       * @return void
       */
      void doOnPmStateChanged(IN const BdAddress& deviceAddres, IN const CallExistenceState
            callExistenceState) const;

   private:

      IPmCoreCallbackIf* _pmCoreCallbackIf; /**< pointer to IPmCoreCallbackIf class */
   };
}

#endif // PmCoreCallbackIfWrapper_h
