/**
 * @file IPmCoreRequestIf.h
 *
 * @swcomponent PhoneCallManager
 *
 * @brief This file contains the definition of the IPmCoreRequestIf class
 *
 * @copyright (C) 2016 Robert Bosch GmbH.
 *            The reproduction, distribution and utilization of this file as
 *            well as the communication of its contents to others without express
 *            authorization is prohibited. Offenders will be held liable for the
 *            payment of damages. All rights reserved in the event of the grant
 *            of a patent, utility model or design.
 *
 * @details This file provides the CPP request interfaces of PmCore.
 *
 * @ingroup PmCore
 */

#ifndef IPmCoreRequestIf_h
#define IPmCoreRequestIf_h

#include "PmCoreIfTypes.h"

//class forward declaration
class ahl_tclBaseOneThreadApp;

namespace pmcore
{
   /**
    * forward declaration
    */
   class IPmCoreCallbackIf;

   /**
    * IPmCoreRequestIf class definition
    */
   class IPmCoreRequestIf
   {
   public:

      /**
       * Destructor of IPmCoreRequestIf class
       *
       * @param[in]
       * @param[out]
       * @param[in,out]
       *
       * @return
       */
      virtual ~IPmCoreRequestIf() {}

      /**
       * Function to register the Callback wrapper component with PmInterface
       * This function must be called before creating the PmCore handlers.
       *
       * @param[in] IPmCoreCallbackIf* - pointer to callback wrapper class
       * @param[out]
       * @param[in,out]
       *
       * @return PmResult (PmResultCode, PmResultMessage)
       */
      virtual PmResult registerPmCoreCallbackIf(IPmCoreCallbackIf* pmCoreCallbackIf) = 0;

      /**
       * Function to create all internal PM Core components.
       *
       * @param[in] mainApplication - pointer of CCA PmApplication
       * @param[out]
       * @param[in,out]
       *
       * @return PmResult (PmResultCode, PmResultMessage)
       */
      virtual PmResult initializePmCore(ahl_tclBaseOneThreadApp* mainApplication) = 0;

      /**
       * This method is used to set the SPM system state to PM Core.
       *
       * @param[in] SystemState& - system state
       * @param[out]
       * @param[in,out]
       *
       * @return PmResult (PmResultCode, PmResultMessage)
       */
      virtual PmResult setSystemState(IN const SystemState systemState) = 0;

      /**
       * This method provides the BT address of the vehicle to PM Core.
       *
       * @param[in] VehicleConfiguration& - vehicle configurations
       * @param[out]
       * @param[in,out]
       *
       * @return PmResult (PmResultCode, PmResultMessage)
       */
      virtual PmResult setVehicleConfiguration(IN const pmcore::VehicleConfiguration& vehicleConfig) = 0;

      /**
       * This method updates the device paired status to PM Core.
       *
       * @param[in] BasicDeviceDetails& - reference to BasicDeviceDetails class
       * @param[out]
       * @param[in,out]
       *
       * @return PmResult (PmResultCode, PmResultMessage)
       */
      virtual PmResult onDevicePaired(IN const pmcore::BasicDeviceDetails& deviceDetails) = 0;

      /**
       * This method updates the device connected status to PM Core.
       *
       * @param[in] BasicDeviceDetails& - reference to BasicDeviceDetails class
       * @param[out]
       * @param[in,out]
       *
       * @return PmResult (PmResultCode, PmResultMessage)
       */
      virtual PmResult onDeviceConnected(IN const pmcore::BasicDeviceDetails& deviceDetails) = 0;

      /**
       * This method updates the device disconnected status to PM Core.
       *
       * @param[in] BdAddress& - device address
       * @param[out]
       * @param[in,out]
       *
       * @return PmResult (PmResultCode, PmResultMessage)
       */
      virtual PmResult onDeviceDisconnected(IN const BdAddress& deviceAddress) = 0;

      /**
       * This interface updates the device deleted status to PM Core.
       *
       * @param[in] BdAddress& - device address
       * @param[out]
       * @param[in,out]
       *
       * @return PmResult (PmResultCode, PmResultMessage)
       */
      virtual PmResult onDeviceDeleted(IN const BdAddress& deviceAddress) = 0;

      /**
       * This interface updates the device identification detail to PM Core if it was not updated in the recently
       * sent Connection status ie., onDevicePaired() or onDeviceConnected().
       *
       * @param[in] BasicDeviceDetails& - reference to BasicDeviceDetails class
       * @param[out]
       * @param[in,out]
       *
       * @return PmResult (PmResultCode, PmResultMessage)
       */
      virtual PmResult onDeviceIdentificationUpdated(IN const pmcore::BasicDeviceDetails& deviceDetails) = 0;

      /**
       * This method provides the system wide ringtone list with the currently selected ringtone.
       *
       * @param[in] ActType - asynchronous completion token
       * @param[out]
       * @param[in,out]
       *
       * @return PmResult (PmResultCode, PmResultMessage)
       */
      virtual PmResult getSystemWideRingtonesList(IN const ActType act = 0u) = 0;

      /**
       * This interface provides a list of paired devices with its currently selected ringtone.
       *
       * @param[in] ActType - asynchronous completion token
       * @param[out]
       * @param[in,out]
       *
       * @return PmResult (PmResultCode, PmResultMessage)
       */
      virtual PmResult getDeviceRingtoneList(IN const ActType act = 0u) = 0;

      /**
       * This method is used to set the active ringtone for system wide or device specific.
       * If the DeviceAddress string is ALL, then the received RingtoneID is set for all the paired devices treating it as a system wide ringtone.
       * The setting will be applicable also for the devices that are paired after the API is invoked.
       * The updated system wide ringtone will be available in the onSystemWideRingtonesListChanged method.
       * The updated device specific ringtone will be available in the onDeviceRingtoneListChanged method.
       * Note: - The setting will be retained persistently in PmCore Database.
       *
       * @param[in] BdAddress& - "ALL" or device address
       * @param[in] RingtoneId - ringtone id
       * @param[in] ActType - asynchronous completion token
       * @param[out]
       * @param[in,out]
       *
       * @return PmResult (PmResultCode, PmResultMessage)
       */
      virtual PmResult setRingtone(IN const BdAddress& bdAddress, IN const RingtoneId ringtoneId,
            IN const ActType act = 0u) = 0;

      /**
       * Member function Overridden from IPmCoreRequestIf class
       *
       * @param[in] RingtoneId - ringtone id
       * @param[in] PlayCount  - Play count of ringtone
       * @param[in] ActType    - asynchronous completion token
       * @param[out]
       * @param[in,out]
       *
       * @return PmResult (PmResultCode, PmResultMessage)
       */
      virtual PmResult playRingtone(IN const RingtoneId ringtoneId, IN const PlayCount playCount,
            IN const ActType act = 0u) = 0;

      /**
       * Member function Overridden from IPmCoreRequestIf class
       *
       * @param[in] ActType - asynchronous completion token
       * @param[out]
       * @param[in,out]
       *
       * @return PmResult (PmResultCode, PmResultMessage)
       */
      virtual PmResult stopRingtone(IN const ActType act = 0u) = 0;

      /**
       * This method provides the suppress ringtone setting state for all paired devices.
       *
       * @param[in] ActType - asynchronous completion token
       * @param[out]
       * @param[in,out]
       *
       * @return PmResult (PmResultCode, PmResultMessage)
       */
      virtual PmResult getSuppressRingtoneOnOffList(IN const ActType act = 0u) = 0;

      /**
       * This method provides the ability to set the ringtone on or off in the vehicle when an incoming call is received.
       * If the DeviceAddress string is ALL, then the received setting is set for all the paired devices treating it as a system wide setting.
       * The same will be applicable also for the devices that are paired after the API is invoked.
       * The updated changes will be available in the onSuppressRingtoneOnOffListChanged method.
       * Note: - The setting will be retained persistently in PmCore DataBase.
       *
       * @param[in] BdAddress& - "ALL" or device address
       * @param[in] SuppressRingtoneState - suppress ringtone state
       * @param[in] ActType - asynchronous completion token
       * @param[out]
       * @param[in,out]
       *
       * @return PmResult (PmResultCode, PmResultMessage)
       */
      virtual PmResult suppressRingtoneOnOff(IN const BdAddress& bdAddress, IN const SuppressRingtoneState suppressRingtoneState, IN const ActType act = 0u) = 0;

      /**
       * This method provides the auto waiting mode setting state for all paired devices.
       *
       * @param[in] ActType - asynchronous completion token
       * @param[out]
       * @param[in,out]
       *
       * @return PmResult (PmResultCode, PmResultMessage)
       */
      virtual PmResult getAutoWaitingModeOnOffList(IN const ActType act = 0u) = 0;

      /**
       * This method provides the ability to enable or disable the auto waiting mode when an incoming call is received.
       * When the setting is enabled, an incoming call will be automatically put on waiting mode and the waiting mode tone will be heard in the far end.
       * If the DeviceAddress string is ALL, then the setting is set for all the paired devices treating it as a system wide setting.
       * The setting will be applicable for the devices that are paired after the API is invoked.
       * The updated changes will be available in the onAutoWaitingModeOnOffListChanged method.
       * Note: - The setting will be retained persistently in PmCore DataBase
       *
       * @param[in] BdAddress& - "ALL" or device address
       * @param[in] AutoWaitingModeState - auto waiting state
       * @param[in] ActType - asynchronous completion token
       * @param[out]
       * @param[in,out]
       *
       * @return PmResult (PmResultCode, PmResultMessage)
       */
      virtual PmResult autoWaitingModeOnOff(IN const BdAddress& bdAddress, IN const AutoWaitingModeState autoWaitingModeState, IN const ActType act = 0u) = 0;

      /**
       * This method sets the filepath for WaitingMode NarrowBand and WideBand audio files.
       *
       * @param[in] WaitingModeFilePath& - reference to WaitingModeFilePath class
       * @param[in] ActType - asynchronous completion token
       * @param[out]
       * @param[in,out]
       *
       * @return PmResult (PmResultCode, PmResultMessage)
       */
      virtual PmResult setWaitingModeFilePath(IN const WaitingModeFilePath& waitingModeFilePath, IN const ActType act = 0u) = 0;

      /**
       * This method provides the filepath for WaitingMode NarrowBand and WideBand audio files.
       *
       * @param[in] ActType - asynchronous completion token
       * @param[out]
       * @param[in,out]
       *
       * @return PmResult (PmResultCode, PmResultMessage)
       */
      virtual PmResult getWaitingModeFilePath(IN const ActType act = 0u) = 0;

      /**
       * This method provides the state of phone call audio streaming.
       * The value will be set as TRUE when:
       * * SCO is established between AG and HF AND
       * * Audio channel is available AND
       * * ECNR session is started.
       * In all other cases the Status will be reported as FALSE.
       *
       * @param[in] ActType - asynchronous completion token
       * @param[out]
       * @param[in,out]
       *
       * @return PmResult (PmResultCode, PmResultMessage)
       */
      virtual PmResult getPhoneCallAudioActive(IN const ActType act = 0u) = 0;

      /**
       * This method provides the list of devices with current role (active / passive)
       *
       * @param[in] ActType - asynchronous completion token
       * @param[out]
       * @param[in,out]
       *
       * @return PmResult (PmResultCode, PmResultMessage)
       */
      virtual PmResult getActivePassiveDeviceList(IN const ActType act = 0u) = 0;

      /**
       * This method provides the ability to set a device as an active device.
       * After successful switching, current role of the device will be notified via onActivePassiveDeviceListChanged method.
       * Note: - This API is meant to be invoked only for HFP connected devices.
       *
       * @param[in] BdAddressList& - list of device addresses
       * @param[in] ActType - asynchronous completion token
       * @param[out]
       * @param[in,out]
       *
       * @return PmResult (PmResultCode, PmResultMessage)
       */
      virtual PmResult setActivePhone(IN const BdAddressList& deviceAddressList, IN const ActType act = 0u) = 0;

      /**
       * This method is used to restore the settings - AutoWaitingMode, SuppressRingtone, DeviceRingtone to its default values for the given device address.
       * If the DeviceAddress string is ALL, then the settings for all the devices will be reset to default and also system settings will be reset to default.
       *
       * @param[in] BdAddress& - device address
       * @param[in] ActType - asynchronous completion token
       * @param[out]
       * @param[in,out]
       *
       * @return PmResult (PmResultCode, PmResultMessage)
       */
      virtual PmResult restoreDefaultSetting(IN const BdAddress& deviceAddress, IN const ActType act = 0u) = 0;

      /**
       * This method provides the supported features of the connected device.
       *
       * @param[in] BdAddress& - device address
       * @param[in] ActType - asynchronous completion token
       * @param[out]
       * @param[in,out]
       *
       * @return PmResult (PmResultCode, PmResultMessage)
       */
      virtual PmResult getSupportedFeatures(IN const BdAddress& deviceAddress, IN const ActType act = 0u) = 0;

      /**
       * This method provides the network status of the connected device.
       *
       * @param[in] BdAddress& - device address
       * @param[in] ActType - asynchronous completion token
       * @param[out]
       * @param[in,out]
       *
       * @return PmResult (PmResultCode, PmResultMessage)
       */
      virtual PmResult getNetworkStatus(IN const BdAddress& deviceAddress, IN const ActType act = 0u) = 0;

      /**
       * This method provides the signal strength of the connected device.
       *
       * @param[in] BdAddress& - device address
       * @param[in] ActType - asynchronous completion token
       * @param[out]
       * @param[in,out]
       *
       * @return PmResult (PmResultCode, PmResultMessage)
       */
      virtual PmResult getSignalStrength(IN const BdAddress& deviceAddress, IN const ActType act = 0u) = 0;

      /**
       * This method provides the battery charge level of the connected device.
       *
       * @param[in] BdAddress& - device address
       * @param[in] ActType - asynchronous completion token
       * @param[out]
       * @param[in,out]
       *
       * @return PmResult (PmResultCode, PmResultMessage)
       */
      virtual PmResult getBatteryChargeLevel(IN const BdAddress& deviceAddress, IN const ActType act = 0u) = 0;

      /**
       * This method provides the network operator of the connected device.
       *
       * @param[in] BdAddress& - device address
       * @param[in] ActType - asynchronous completion token
       * @param[out]
       * @param[in,out]
       *
       * @return PmResult (PmResultCode, PmResultMessage)
       */
      virtual PmResult getNetworkOperator(IN const BdAddress& deviceAddress, IN const ActType act = 0u) = 0;

      /**
       * This method is used to get the "Voice Mail" numbers of the connected device.
       *
       * @param[in] BdAddress& - device address
       * @param[in] ActType - asynchronous completion token
       * @param[out]
       * @param[in,out]
       *
       * @return PmResult (PmResultCode, PmResultMessage)
       */
      virtual PmResult voiceMailList(IN const BdAddress& deviceAddress, IN const ActType act = 0u) = 0;

      /**
       * This method is used to get the call status of the connected devices.
       *
       * @param[in] ActType - asynchronous completion token
       * @param[out]
       * @param[in,out]
       *
       * @return PmResult (PmResultCode, PmResultMessage)
       */
      virtual PmResult getCallStatusList(IN const ActType act = 0u) = 0;

      /**
       * This method is used to get the state of Microphone whether it is in mute or not.
       *
       * @param[in] ActType - asynchronous completion token
       * @param[out]
       * @param[in,out]
       *
       * @return PmResult (PmResultCode, PmResultMessage)
       */
      virtual PmResult getMicrophoneMuteState(IN const ActType act = 0u) = 0;

      /**
       * This method is used to mute/unmute the microphone of the target using the interfaces of ECNR engine.
       * After successful switching, change will be notified via onMicrophoneMuteStateChanged method.
       *
       * @param[in] MuteState - mute or unmute
       * @param[in] ActType - asynchronous completion token
       * @param[out]
       * @param[in,out]
       *
       * @return PmResult (PmResultCode, PmResultMessage)
       */
      virtual PmResult setMicrophoneMuteState(IN const MuteState microphoneMuteState, IN const ActType act = 0u) = 0;

      /**
       * Joins or conferences in all active and held calls of a single or multiple HFP connected devices together into a multi-party call.
       * Returns the new list of calls participating in the multiparty call.
       * The change in call states will be notified by the onCallStatusListChanged method.
       *
       * @param[in] DeviceCallInstanceListMap& - device address and call instances map
       * @param[in] ActType - asynchronous completion token
       * @param[out]
       * @param[in,out]
       *
       * @return PmResult (PmResultCode, PmResultMessage)
       */
      virtual PmResult mergeCalls(IN const DeviceCallInstanceListMap& mergeCallList, IN const ActType act = 0u) = 0;

      /**
       * Places the Conference call on hold and makes desired call active. Returns the new list of calls participating in the multiparty call.
       * The change in call states will be notified by the onCallStatusListChanged method.
       *
       * @param[in] DeviceCallInstanceListMap& - device address and call instances map
       * @param[in] ActType - asynchronous completion token
       * @param[out]
       * @param[in,out]
       *
       * @return PmResult (PmResultCode, PmResultMessage)
       */
      virtual PmResult splitCalls(IN const DeviceCallInstanceListMap& splitCallList, IN const ActType act = 0u) = 0;

      /**
       * Releases all calls including waiting calls and multi party calls.
       * Note: All calls in the device will be disconnected for the given DeviceAddress.
       * If DeviceAddress is ALL then all the calls in all the connected devices will be disconnected.
       * The change in call states will be notified by the onCallStatusListChanged method.
       *
       * @param[in] DeviceCallInstanceListMap& - device address and call instances map
       * @param[in] ActType - asynchronous completion token
       * @param[out]
       * @param[in,out]
       *
       * @return PmResult (PmResultCode, PmResultMessage)
       */
      virtual PmResult hangupCalls(IN const DeviceCallInstanceListMap& hangupCallList, IN const ActType act = 0u) = 0;

      /**
       * Initiates a new outgoing call that was dialed before for the given device address.
       * If device address is "SYSTEM", then outgoing call will be initiated from the active phone for the last dialed number in the vehicle.
       * The change in call states will be notified by the onCallStatusListChanged method.
       *
       * @param[in] BdAddress& - device address
       * @param[in] HideCallerId - hide caller id
       * @param[in] ActType - asynchronous completion token
       * @param[out]
       * @param[in,out]
       *
       * @return PmResult (PmResultCode, PmResultMessage)
       */
      virtual PmResult redial(IN const BdAddress& deviceAddress, IN const HideCallerId hideCallerId, IN const ActType act = 0u) = 0;

      /**
       * This method is used to get the current status of SCO establishment between AG and HF.
       *
       * @param[in] BdAddress& - device address
       * @param[in] ActType - asynchronous completion token
       * @param[out]
       * @param[in,out]
       *
       * @return PmResult (PmResultCode, PmResultMessage)
       */
      virtual PmResult getSCOConnection(IN const BdAddress& deviceAddress, IN const ActType act = 0u) = 0;

      /**
       * This method is used to get the playing ringtone mute state.
       *
       * @param[in] BdAddress& - device address
       * @param[in] ActType - asynchronous completion token
       * @param[out]
       * @param[in,out]
       *
       * @return PmResult (PmResultCode, PmResultMessage)
       */
      virtual PmResult getRingtoneMuteState(IN const BdAddress& deviceAddress, IN const ActType act = 0u) = 0;

      /**
       * This method is used to set the playing ringtone mute state.
       * Clients can get final state via onRingtoneMuteStateChanged method.
       *
       * @param[in] BdAddress& - device address
       * @param[in] MuteState - mute / unmute
       * @param[in] ActType - asynchronous completion token
       * @param[out]
       * @param[in,out]
       *
       * @return PmResult (PmResultCode, PmResultMessage)
       */
      virtual PmResult setRingtoneMuteState(IN const BdAddress& deviceAddress, IN const MuteState ringtoneMuteState, IN const ActType act = 0u) = 0;

      /**
       * This method is used to get the  current waiting mode state.
       *
       * @param[in] BdAddress& - device address
       * @param[in] ActType - asynchronous completion token
       * @param[out]
       * @param[in,out]
       *
       * @return PmResult (PmResultCode, PmResultMessage)
       */
      virtual PmResult getWaitingModeState(IN const BdAddress& deviceAddress, IN const ActType act = 0u) = 0;

      /**
       * This method is used to get the ringtone playback status
       *
       * @param[in]
       * @param[out]
       * @param[in,out]
       *
       * @return PmResult (PmResultCode, PmResultMessage)
       */
      virtual PmResult getRingtonePlaybackStatus(IN const ActType act = 0u) = 0;

      /**
       * Start or stop the waiting mode. Waiting mode status is updated by the onWaitingModeStateChanged method.
       *
       * @param[in] BdAddress& - device address
       * @param[in] StartStop - start / stop the waiting mode
       * @param[in] ActType - asynchronous completion token
       * @param[out]
       * @param[in,out]
       *
       * @return PmResult (PmResultCode, PmResultMessage)
       */
      virtual PmResult startStopWaitingMode(IN const BdAddress& deviceAddress, IN const StartStop startStopWaitingMode, IN const ActType act = 0u) = 0;

      /**
       * Initiates a new outgoing call for the specified device.
       * The call state will be notified by the onCallStatusListChanged method.
       *
       * @param[in] BdAddress& - device address
       * @param[in] TelephoneNumber - telephone number
       * @param[in] HideCallerId - hide caller id
       * @param[in] ActType - asynchronous completion token
       * @param[out]
       * @param[in,out]
       *
       * @return PmResult (PmResultCode, PmResultMessage)
       */
      virtual PmResult dial(IN const BdAddress& deviceAddress, IN const TelephoneNumber& telephoneNumber, IN const HideCallerId hideCallerId, IN const ActType act = 0u) = 0;

      /**
       * This method could be invoked to:
       * * Answer an Incoming call OR
       * * Release currently active call (0 or more) and answer the currently waiting call OR
       * * Put the current call on hold and answer the currently waiting call
       * The call state will be notified by the onCallStatusListChanged method.
       *
       * @param[in] BdAddress& - device address
       * @param[in] CallInstance - call instance
       * @param[in] AcceptOperation - accept operation
       * @param[in] ActType - asynchronous completion token
       * @param[out]
       * @param[in,out]
       *
       * @return PmResult (PmResultCode, PmResultMessage)
       */
      virtual PmResult acceptCall(IN const BdAddress& deviceAddress, IN const CallInstance callInstance, IN const AcceptOperation operation = ACCEPT_CALLS, IN const ActType act = 0u) = 0;

      /**
       * This method could be invoked to:
       * * Place all the calls (0 or more including calls in a multi-party conversation) that are Active to Held, and all calls (0 or more) that are Held to Active OR
       * * Release the currently active call (0 or more) and activate any currently held calls
       * The call state will be notified by the onCallStatusListChanged method.
       *
       * @param[in] BdAddress& - device address
       * @param[in] SwapOperation - swap operation
       * @param[in] ActType - asynchronous completion token
       * @param[out]
       * @param[in,out]
       *
       * @return PmResult (PmResultCode, PmResultMessage)
       */
      virtual PmResult swapCall(IN const BdAddress& deviceAddress, IN const SwapOperation operation, IN const ActType act = 0u) = 0;

      /**
       * Initiates a new outgoing call by number index.
       * Note: If an active call (single or multiparty) exists, then it is automatically put on hold if the dial procedure is successful.
       *
       * @param[in] BdAddress& - device address
       * @param[in] MemDialIndex - speed dial index
       * @param[in] HideCallerId - hide caller id
       * @param[in] ActType - asynchronous completion token
       * @param[out]
       * @param[in,out]
       *
       * @return PmResult (PmResultCode, PmResultMessage)
       */
      virtual PmResult speedDial(IN const BdAddress& deviceAddress, IN const MemDialIndex memDialIndex, IN const HideCallerId hideCallerId, IN const ActType act = 0u) = 0;

      /**
       * Sends DTMF tones to the network. The tones have a fixed duration.
       *
       * @param[in] BdAddress& - device address
       * @param[in] DTMFTones& - tones
       * @param[in] ActType - asynchronous completion token
       * @param[out]
       * @param[in,out]
       *
       * @return PmResult (PmResultCode, PmResultMessage)
       */
      virtual PmResult sendDTMF(IN const BdAddress& deviceAddress, IN const DTMFTones& tones, IN const ActType act = 0u) = 0;

      /**
       * This method transfers the audio to AG or HF.
       *
       * @param[in] BdAddress& - device address
       * @param[in] AudioDirection - AG or HF
       * @param[in] ActType - asynchronous completion token
       * @param[out]
       * @param[in,out]
       *
       * @return PmResult (PmResultCode, PmResultMessage)
       */
      virtual PmResult transferAudio(IN const BdAddress& deviceAddress, IN const AudioDirection audioDirection,
            IN const ActType act = 0u) = 0;

      /**
       * This method provides the status of the Voice Recognition session.
       *
       * @param[in] BdAddress& - device address
       * @param[in] ActType - asynchronous completion token
       * @param[out]
       * @param[in,out]
       *
       * @return PmResult (PmResultCode, PmResultMessage)
       */
      virtual PmResult getVoiceRecognitionStatus(IN const BdAddress& deviceAddress, IN const ActType act = 0u) = 0;

      /**
       * This method is used to activate and deactivate the voice recognition session in the AG for the given device address.
       * The change in VoiceRecognition status will be notified via onVoiceRecognitionStatusChanged method.
       *
       * @param[in] BdAddress& - device address
       * @param[in] StartStop - start or stop voice recognition
       * @param[in] ActType - asynchronous completion token
       * @param[out]
       * @param[in,out]
       *
       * @return PmResult (PmResultCode, PmResultMessage)
       */
      virtual PmResult startStopVoiceRecognition(IN const BdAddress& deviceAddress, IN const StartStop startStopVR, IN const ActType act = 0u) = 0;

      /**
       * This method provides the VoiceRecognition feature available in the AG and current availability state of the same.
       *
       * @param[in] BdAddress& - device address
       * @param[in] ActType - asynchronous completion token
       * @param[out]
       * @param[in,out]
       *
       * @return PmResult (PmResultCode, PmResultMessage)
       */
      virtual PmResult getEnhancedVoiceRecognitionFeature(IN const BdAddress& deviceAddress, IN const ActType act = 0u) = 0;

      /**
       * This method provides the external VR status in the AG.
       *
       * @param[in] BdAddress& - device address
       * @param[in] ActType - asynchronous completion token
       * @param[out]
       * @param[in,out]
       *
       * @return PmResult (PmResultCode, PmResultMessage)
       */
      virtual PmResult getExternalVRState(IN const BdAddress& deviceAddress, IN const ActType act = 0u) = 0;

      /**
       * This method is used to activate the external voice recognition session in the AG for the given device address.
       *
       * @param[in] BdAddress& - device address
       * @param[in] ActType - asynchronous completion token
       * @param[out]
       * @param[in,out]
       *
       * @return PmResult (PmResultCode, PmResultMessage)
       */
      virtual PmResult activateExternalVR(IN const BdAddress& deviceAddress, IN const ActType act = 0u) = 0;

      /**
       * This method is used to deactivate the external voice recognition session in the AG for the given device address.
       *
       * @param[in] BdAddress& - device address
       * @param[in] ActType - asynchronous completion token
       * @param[out]
       * @param[in,out]
       *
       * @return PmResult (PmResultCode, PmResultMessage)
       */
      virtual PmResult deActivateExternalVR(IN const BdAddress& deviceAddress, IN const ActType act = 0u) = 0;

      /**
       * This method is used to stop the extended voice recognition session in the AG for the given device address.
       *
       * @param[in] BdAddress& - device address
       * @param[in] ActType - asynchronous completion token
       * @param[out]
       * @param[in,out]
       *
       * @return PmResult (PmResultCode, PmResultMessage)
       */
      virtual PmResult stopExtendedVR(IN const BdAddress& deviceAddress, IN const ActType act = 0u) = 0;
   };
}

#endif //IPmCoreRequestIf_h
