/**
 * @file IPmCoreCallbackIf.h
 *
 * @swcomponent PhoneCallManager
 *
 * @brief This file contains the definition of the IPmCoreCallbackIf class
 *
 * @copyright (C) 2016 Robert Bosch GmbH.
 *            The reproduction, distribution and utilization of this file as
 *            well as the communication of its contents to others without express
 *            authorization is prohibited. Offenders will be held liable for the
 *            payment of damages. All rights reserved in the event of the grant
 *            of a patent, utility model or design.
 *
 * @details This file provides the CPP response interfaces of PmCore.
 *
 * @ingroup PmCore
 */

#ifndef IPmCoreCallbackIf_h
#define IPmCoreCallbackIf_h

#include "PmCoreIfTypes.h"

namespace pmcore
{
   /**
    * IPmCoreCallbackIf class definition
    */
   class IPmCoreCallbackIf
   {
   public:

      /**
       * virtual constructor of the interface class IPmCoreCallbackIf
       *
       * @param[in] ActType - asynchronous completion token
       * @param[out]
       * @param[in,out]
       *
       * @return
       */
      IPmCoreCallbackIf()
      {
      }

      /**
       * virtual destructor of the interface class IPmCoreCallbackIf
       *
       * @param[in] ActType - asynchronous completion token
       * @param[out]
       * @param[in,out]
       *
       * @return
       */
      virtual ~IPmCoreCallbackIf()
      {
      }

      /**
       * This method provides the system wide ringtone list with the currently selected ringtone.
       *
       * @param[in] PmResult& - reference to PmResult class
       * @param[in] SystemWideRingtonesList& - reference to SystemWideRingtonesList class
       * @param[in] ActType - asynchronous completion token
       * @param[out]
       * @param[in,out]
       *
       * @return void
       */
      virtual void getSystemWideRingtonesListResponse(IN const PmResult& pmResult,
            IN const SystemWideRingtonesList& systemWideRingtonesList, IN const ActType act = 0u) = 0;

      /**
       * This method is used to notify the changes in SystemWideRingtonesList.
       *
       * @param[in] SystemWideRingtonesList& - reference to SystemWideRingtonesList class
       * @param[out]
       * @param[in,out]
       *
       * @return void
       */
      virtual void onSystemWideRingtonesListChanged(IN const SystemWideRingtonesList& systemWideRingtonesList) = 0;

      /**
       * This interface provides a list of paired devices with its currently selected ringtone.
       *
       * @param[in] PmResult& - reference to PmResult class
       * @param[in] DeviceRingtoneList& - reference to DeviceRingtoneList class
       * @param[in] ActType - asynchronous completion token
       * @param[out]
       * @param[in,out]
       *
       * @return void
       */
      virtual void getDeviceRingtoneListResponse(IN const PmResult& pmResult,
            IN const DeviceRingtoneList& deviceRingtoneList, IN const ActType act = 0u) = 0;

      /**
       * This method is used to notify the changes in SystemWideRingtonesList.
       *
       * @param[in] DeviceRingtoneList& - reference to DeviceRingtoneList class
       * @param[out]
       * @param[in,out]
       *
       * @return void
       */
      virtual void onDeviceRingtoneListChanged(IN const DeviceRingtoneList& deviceRingtoneList) = 0;

      /**
       * Response for the method setRingtone.
       *
       * @param[in] PmResult& - reference to PmResult class
       * @param[in] ActType - asynchronous completion token
       * @param[out]
       * @param[in,out]
       *
       * @return void
       */
      virtual void setRingtoneResponse(IN const PmResult& pmResult, IN const ActType act = 0u) = 0;

      /**
       * Response for the method playRingtone.
       *
       * @param[in] PmResult& - reference to PmResult class
       * @param[in] ActType - asynchronous completion token
       * @param[out]
       * @param[in,out]
       *
       * @return void
       */
      virtual void playRingtoneResponse(IN const PmResult& pmResult, IN const ActType act = 0u) = 0;

      /**
       * Response for the method stopRingtone.
       *
       * @param[in] PmResult& - reference to PmResult class
       * @param[in] ActType - asynchronous completion token
       * @param[out]
       * @param[in,out]
       *
       * @return void
       */
      virtual void stopRingtoneResponse(IN const PmResult& pmResult, IN const ActType act = 0u) = 0;

      /**
       * This method provides the suppress ringtone setting state for all paired devices.
       *
       * @param[in] PmResult& - reference to PmResult class
       * @param[in] SuppressRingtoneOnOffList& - reference to SuppressRingtoneOnOffList class
       * @param[in] ActType - asynchronous completion token
       * @param[out]
       * @param[in,out]
       *
       * @return void
       */
      virtual void getSuppressRingtoneOnOffListResponse(IN const PmResult& pmResult,
            IN const SuppressRingtoneOnOffList& suppressRingtoneOnOffList, IN const ActType act = 0u) = 0;

      /**
       * This method is used to notify the changes in SuppressRingtoneOnOffList.
       *
       * @param[in] SuppressRingtoneOnOffList& - reference to SuppressRingtoneOnOffList class
       * @param[out]
       * @param[in,out]
       *
       * @return void
       */
      virtual void onSuppressRingtoneOnOffListChanged(IN const SuppressRingtoneOnOffList& suppressRingtoneOnOffList) = 0;

      /**
       * Response for the method suppressRingtoneOnOff.
       *
       * @param[in] PmResult& - reference to PmResult class
       * @param[in] ActType - asynchronous completion token
       * @param[out]
       * @param[in,out]
       *
       * @return void
       */
      virtual void suppressRingtoneOnOffResponse(IN const PmResult& pmResult, IN const ActType act = 0u) = 0;

      /**
       * This method provides the auto waiting mode setting state for all paired devices.
       *
       * @param[in] PmResult& - reference to PmResult class
       * @param[in] AutoWaitingModeOnOffList& - reference to AutoWaitingModeOnOffList class
       * @param[in] ActType - asynchronous completion token
       * @param[out]
       * @param[in,out]
       *
       * @return void
       */
      virtual void getAutoWaitingModeOnOffListResponse(IN const PmResult& pmResult,
            IN const AutoWaitingModeOnOffList& autoWaitingModeOnOffList, IN const ActType act = 0u) = 0;

      /**
       *  This method is used to notify the changes in AutoWaitingModeOnOffList.
       *
       * @param[in] AutoWaitingModeOnOffList& - reference to AutoWaitingModeOnOffList class
       * @param[out]
       * @param[in,out]
       *
       * @return void
       */
      virtual void onAutoWaitingModeOnOffListChanged(IN AutoWaitingModeOnOffList& autoWaitingModeOnOffList) = 0;

      /**
       * Response for the method AutoWaitingModeOnOff.
       *
       * @param[in] PmResult& - reference to PmResult class
       * @param[in] ActType - asynchronous completion token
       * @param[out]
       * @param[in,out]
       *
       * @return void
       */
      virtual void autoWaitingModeOnOffResponse(IN const PmResult& pmResult, IN const ActType act = 0u) = 0;

      /**
       * This method sets the filepath for WaitingMode NarrowBand and WideBand audio files.
       *
       * @param[in] PmResult& - reference to PmResult class
       * @param[in] ActType - asynchronous completion token
       * @param[out]
       * @param[in,out]
       *
       * @return void
       */
      virtual void setWaitingModeFilePathResponse(IN const PmResult& pmResult, IN const ActType act = 0u) = 0;

      /**
       * This method provides the filepath for WaitingMode NarrowBand and WideBand audio files.
       *
       * @param[in] PmResult& - reference to PmResult class
       * @param[in] WaitingModeFilePath& - reference to WaitingModeFilePath class
       * @param[in] ActType - asynchronous completion token
       * @param[out]
       * @param[in,out]
       *
       * @return void
       */
      virtual void getWaitingModeFilePathResponse(IN const PmResult& pmResult,
            IN const WaitingModeFilePath& waitingModeFilePath, IN const ActType act = 0u) = 0;

      /**
       * This method is used to notify the changes in WaitingModeFilePath.
       *
       * @param[in] WaitingModeFilePath& - reference to WaitingModeFilePath class
       * @param[out]
       * @param[in,out]
       *
       * @return void
       */
      virtual void onWaitingModeFilePathChanged(IN const WaitingModeFilePath& waitingModeFilePath) = 0;

      /**
       * This method provides the state of phone call audio streaming.
       * The value will be set as TRUE when:
       * * SCO is established between AG and HF AND
       * * Audio channel is available AND
       * * ECNR session is started.
       * In all other cases the Status will be reported as FALSE.
       *
       * @param[in] PmResult& - reference to PmResult class
       * @param[in] PhoneCallAudioActive& - reference to PhoneCallAudioActive class
       * @param[in] ActType - asynchronous completion token
       * @param[out]
       * @param[in,out]
       *
       * @return void
       */
      virtual void getPhoneCallAudioActiveResponse(IN const PmResult& pmResult,
            IN const PhoneCallAudioActive& phoneCallAudioActive, IN const ActType act = 0u) = 0;

      /**
       * This method is used to notify the changes in PhoneCallAudioActive.
       *
       * @param[in] PhoneCallAudioActive& - reference to PhoneCallAudioActive class
       * @param[out]
       * @param[in,out]
       *
       * @return void
       */
      virtual void onPhoneCallAudioActiveChanged(IN const PhoneCallAudioActive& phoneCallAudioActive) = 0;

      /**
       * This method provides the list of devices with current role (active / passive)
       *
       * @param[in] PmResult& - reference to PmResult class
       * @param[in] ActivePassiveDeviceList& - reference to ActivePassiveDeviceList class
       * @param[in] ActType - asynchronous completion token
       * @param[out]
       * @param[in,out]
       *
       * @return void
       */
      virtual void getActivePassiveDeviceListResponse(IN const PmResult& pmResult,
            IN const ActivePassiveDeviceList& activePassiveDeviceList, IN const ActType act = 0u) = 0;

      /**
       * This method is used to notify the changes in ActivePassiveDeviceList.
       *
       * @param[in] ActivePassiveDeviceList& - reference to ActivePassiveDeviceList class
       * @param[out]
       * @param[in,out]
       *
       * @return void
       */
      virtual void onActivePassiveDeviceListChanged(IN const ActivePassiveDeviceList& activePassiveDeviceList) = 0;

      /**
       * Response for the method setActivePhone.
       *
       * @param[in] PmResult& - reference to PmResult class
       * @param[in] ActType - asynchronous completion token
       * @param[out]
       * @param[in,out]
       *
       * @return void
       */
      virtual void setActivePhoneResponse(IN const PmResult& pmResult, IN const ActType act = 0u) = 0;

      /**
       * Response for the method restoreDefaultSetting.
       *
       * @param[in] PmResult& - reference to PmResult class
       * @param[in] ActType - asynchronous completion token
       * @param[out]
       * @param[in,out]
       *
       * @return void
       */
      virtual void restoreDefaultSettingResponse(IN const PmResult& pmResult, IN const ActType act = 0u) = 0;

      /**
       * This method is used to indicate the different phone call manager related events.
       * Examples Events:
       * IncomingCallRejected
       * OutgoingCallFailed
       *
       * @param[in] NotificationEvent& - reference to NotificationEvent class
       * @param[out]
       * @param[in,out]
       *
       * @return void
       */
      virtual void onNotificationEvent(IN const NotificationEvent& notificationEvent) = 0;

      /**
       * This method provides the supported features of the connected device.
       *
       * @param[in] PmResult& - reference to PmResult class
       * @param[in] BdAddress& - device address
       * @param[in] SupportedFeatures& - reference to SupportedFeatures class
       * @param[in] ActType - asynchronous completion token
       * @param[out]
       * @param[in,out]
       *
       * @return void
       */
      virtual void getSupportedFeaturesResponse(IN const PmResult& pmResult, IN const BdAddress& deviceAddress,
            IN const SupportedFeatures& supportedFeatures, IN const ActType act = 0u) = 0;

      /**
       * This method is used to notify the changes in SupportedFeatures of AudioGateway.
       *
       * @param[in] BdAddress& - device address
       * @param[in] SupportedFeatures& - reference to SupportedFeatures class
       * @param[out]
       * @param[in,out]
       *
       * @return void
       */
      virtual void onSupportedFeaturesChanged(IN const BdAddress& deviceAddress,
            IN const SupportedFeatures& supportedFeatures) = 0;

      /**
       * This method provides the network status of the connected device.
       *
       * @param[in] PmResult& - reference to PmResult class
       * @param[in] BdAddress& - device address
       * @param[in] NetworkStatus& - reference to NetworkStatus class
       * @param[in] ActType - asynchronous completion token
       * @param[out]
       * @param[in,out]
       *
       * @return void
       */
      virtual void getNetworkStatusResponse(IN const PmResult& pmResult, IN const BdAddress& deviceAddress,
            IN const NetworkStatus& networkStatus, IN const ActType act = 0u) = 0;

      /**
       * This method is used to notify the changes in NetworkStatus of AudioGateway.
       *
       * @param[in] BdAddress& - device address
       * @param[in] NetworkStatus& - reference to NetworkStatus class
       * @param[out]
       * @param[in,out]
       *
       * @return void
       */
      virtual void onNetworkStatusChanged(IN const BdAddress& deviceAddress, IN const NetworkStatus& networkStatus) = 0;

      /**
       * This method provides the signal strength of the connected device.
       *
       * @param[in] PmResult& - reference to PmResult class
       * @param[in] BdAddress& - device address
       * @param[in] SignalStrength& - reference to SignalStrength class
       * @param[in] ActType - asynchronous completion token
       * @param[out]
       * @param[in,out]
       *
       * @return void
       */
      virtual void getSignalStrengthResponse(IN const PmResult& pmResult, IN const BdAddress& deviceAddress,
            IN const SignalStrength& signalStrength, IN const ActType act = 0u) = 0;

      /**
       * This method is used to notify the changes in SignalStrength of AudioGateway.
       *
       * @param[in] BdAddress& - device address
       * @param[in] SignalStrength& - reference to SignalStrength class
       * @param[out]
       * @param[in,out]
       *
       * @return void
       */
      virtual void onSignalStrengthChanged(IN const BdAddress& deviceAddress,
            IN const SignalStrength& signalStrength) = 0;

      /**
       * This method provides the battery charge level of the connected device.
       *
       * @param[in] PmResult& - reference to PmResult class
       * @param[in] BdAddress& - device address
       * @param[in] BatteryChargeLevel& - reference to BatteryChargeLevel class
       * @param[in] ActType - asynchronous completion token
       * @param[out]
       * @param[in,out]
       *
       * @return void
       */
      virtual void getBatteryChargeLevelResponse(IN const PmResult& pmResult, IN const BdAddress& deviceAddress,
            IN const BatteryChargeLevel& batteryChargeLevel, IN const ActType act = 0u) = 0;

      /**
       * This method is used to notify the changes in BatteryChargeLevel of AudioGateway.
       *
       * @param[in] BdAddress& - device address
       * @param[in] BatteryChargeLevel& - reference to BatteryChargeLevel class
       * @param[out]
       * @param[in,out]
       *
       * @return void
       */
      virtual void onBatteryChargeLevelChanged(IN const BdAddress& deviceAddress,
            IN const BatteryChargeLevel& batteryChargeLevel) = 0;

      /**
       * This method provides the network operator of the connected device.
       *
       * @param[in] PmResult& - reference to PmResult class
       * @param[in] BdAddress& - device address
       * @param[in] NetworkOperator& - reference to NetworkOperator class
       * @param[in] ActType - asynchronous completion token
       * @param[out]
       * @param[in,out]
       *
       * @return void
       */
      virtual void getNetworkOperatorResponse(IN const PmResult& pmResult, IN const BdAddress& deviceAddress,
            IN const NetworkOperator& networkOperator, IN const ActType act = 0u) = 0;

      /**
       * This method is used to notify the changes in NetworkOperator of AudioGateway.
       *
       * @param[in] BdAddress& - device address
       * @param[in] NetworkOperator& - reference to NetworkOperator class
       * @param[out]
       * @param[in,out]
       *
       * @return void
       */
      virtual void onNetworkOperatorChanged(IN const BdAddress& deviceAddress,
            IN const NetworkOperator& networkOperator) = 0;

      /**
       * Response for the method VoiceMailList.
       *
       * @param[in] PmResult& - reference to PmResult class
       * @param[in] BdAddress& - device address
       * @param[in] VoiceMailList& - reference to VoiceMailList class
       * @param[in] ActType - asynchronous completion token
       * @param[out]
       * @param[in,out]
       *
       * @return void
       */
      virtual void voiceMailListResponse(IN const PmResult& pmResult, IN const BdAddress& deviceAddress,
            IN const VoiceMailList& voiceMailList, IN const ActType act = 0u) = 0;

      /**
       * This method is used to get the call status of the connected devices.
       *
       * @param[in] PmResult& - reference to PmResult class
       * @param[in] CallStatusList& - reference to CallStatusList class
       * @param[in] ActType - asynchronous completion token
       * @param[out]
       * @param[in,out]
       *
       * @return void
       */
      virtual void getCallStatusListResponse(IN const PmResult& pmResult, IN const CallStatusList& callStatusList,
            IN const ActType act = 0u) = 0;

      /**
       * This method is used to notify the changes in CallStatusList.
       *
       * @param[in] CallStatusList& - reference to CallStatusList class
       * @param[out]
       * @param[in,out]
       *
       * @return void
       */
      virtual void onCallStatusListChanged(IN const CallStatusList& callStatusList) = 0;

      /**
       * This method is used to get the state of Microphone whether it is in mute or not.
       *
       * @param[in] PmResult& - reference to PmResult class
       * @param[in] MicrophoneMuteState& - reference to MicrophoneMuteState class
       * @param[in] ActType - asynchronous completion token
       * @param[out]
       * @param[in,out]
       *
       * @return void
       */
      virtual void getMicrophoneMuteStateResponse(IN const PmResult& pmResult,
            IN const MicrophoneMuteState& microphoneMuteState, IN const ActType act = 0u) = 0;

      /**
       * This method is used to notify the changes in MicrophoneMuteState.
       *
       * @param[in] MicrophoneMuteState& - reference to MicrophoneMuteState class
       * @param[out]
       * @param[in,out]
       *
       * @return void
       */
      virtual void onMicrophoneMuteStateChanged(IN const MicrophoneMuteState& microphoneMuteState) = 0;

      /**
       * Response for the setMicrophoneMuteState request.
       *
       * @param[in] PmResult& - reference to PmResult class
       * @param[in] ActType - asynchronous completion token
       * @param[out]
       * @param[in,out]
       *
       * @return void
       */
      virtual void setMicrophoneMuteStateResponse(IN const PmResult& pmResult, IN const ActType act = 0u) = 0;

      /**
       * Response for the MergeCalls request for the device.
       *
       * @param[in] PmResult& - reference to PmResult class
       * @param[in] ActType - asynchronous completion token
       * @param[out]
       * @param[in,out]
       *
       * @return void
       */
      virtual void mergeCallsResponse(IN const PmResult& pmResult, IN const ActType act = 0u) = 0;

      /**
       * Response for the SplitCalls request for the device.
       *
       * @param[in] PmResult& - reference to PmResult class
       * @param[in] ActType - asynchronous completion token
       * @param[out]
       * @param[in,out]
       *
       * @return
       */
      virtual void splitCallsResponse(IN const PmResult& pmResult, IN const ActType act = 0u) = 0;

      /**
       * Response for the HangupCalls request for the device.
       *
       * @param[in] PmResult& - reference to PmResult class
       * @param[in] ActType - asynchronous completion token
       * @param[out]
       * @param[in,out]
       *
       * @return void
       */
      virtual void hangupCallsResponse(IN const PmResult& pmResult, IN const ActType act = 0u) = 0;

      /**
       * Response for the Redial request for the device.
       *
       * @param[in] PmResult& - reference to PmResult class
       * @param[in] BdAddress& - device address
       * @param[in] CallInstance& - call instance
       * @param[in] ActType - asynchronous completion token
       * @param[out]
       * @param[in,out]
       *
       * @return void
       */
      virtual void redialResponse(IN const PmResult& pmResult, IN const BdAddress& deviceAddress,
            IN const CallInstance callInstance, IN const ActType act = 0u) = 0;

      /**
       * This method is used to get the current status of SCO establishment between AG and HF.
       *
       * @param[in] PmResult& - reference to PmResult class
       * @param[in] BdAddress& - device address
       * @param[in] SCOConnection& - reference to SCOConnection class
       * @param[in] ActType - asynchronous completion token
       * @param[out]
       * @param[in,out]
       *
       * @return void
       */
      virtual void getSCOConnectionResponse(IN const PmResult& pmResult, IN const BdAddress& deviceAddress,
            IN const SCOConnection& scoConnection, IN const ActType act = 0u) = 0;

      /**
       * This method is used to notify the changes in SCO Connection.
       *
       * @param[in] BdAddress& - device address
       * @param[in] SCOConnection& - reference to SCOConnection class
       * @param[out]
       * @param[in,out]
       *
       * @return void
       */
      virtual void onSCOConnectionChanged(IN const BdAddress& deviceAddress, IN const SCOConnection& scoConnection) = 0;

      /**
       * This method is used to get the playing ringtone mute state.
       *
       * @param[in] PmResult& - reference to PmResult class
       * @param[in] BdAddress& - device address
       * @param[in] RingtoneMuteState& - reference to RingtoneMuteState class
       * @param[in] ActType - asynchronous completion token
       * @param[out]
       * @param[in,out]
       *
       * @return void
       */
      virtual void getRingtoneMuteStateResponse(IN const PmResult& pmResult, IN const BdAddress& deviceAddress,
            IN const RingtoneMuteState& ringtoneMuteState, IN const ActType act = 0u) = 0;

      /**
       * This method is used to notify the changes in RingtoneMuteState of the device.
       *
       * @param[in] BdAddress& - device address
       * @param[in] RingtoneMuteState& - reference to RingtoneMuteState class
       * @param[out]
       * @param[in,out]
       *
       * @return void
       */
      virtual void onRingtoneMuteStateChanged(IN const BdAddress& deviceAddress,
            IN const RingtoneMuteState& ringtoneMuteState) = 0;

      /**
       * Response for the setRingtoneMuteState request.
       *
       * @param[in] PmResult& - reference to PmResult class
       * @param[in] BdAddress& - device address
       * @param[in] ActType - asynchronous completion token
       * @param[out]
       * @param[in,out]
       *
       * @return void
       */
      virtual void setRingtoneMuteStateResponse(IN const PmResult& pmResult, IN const BdAddress& deviceAddress,
            IN const ActType act = 0u) = 0;

      /**
       * This method is used to get the  current waiting mode state.
       *
       * @param[in] PmResult& - reference to PmResult class
       * @param[in] BdAddress& - device address
       * @param[in] WaitingModeState& - reference to WaitingModeState class
       * @param[in] ActType - asynchronous completion token
       * @param[out]
       * @param[in,out]
       *
       * @return void
       */
      virtual void getWaitingModeStateResponse(IN const PmResult& pmResult, IN const BdAddress& deviceAddress,
            IN const WaitingModeState& waitingModeState, IN const ActType act = 0u) = 0;

      /**
       * This method is used to notify the changes in WaitingModeState of the device.
       *
       * @param[in] BdAddress& - device address
       * @param[in] WaitingModeState& - reference to WaitingModeState class
       * @param[out]
       * @param[in,out]
       *
       * @return void
       */
      virtual void onWaitingModeStateChanged(IN const BdAddress& deviceAddress,
            IN const WaitingModeState& waitingModeState) = 0;

      /**
       * This method is used to get the  current playback status of the vehicle ringtone
       *
       * @param[in] PmResult& - reference to PmResult class
       * @param[in] ringtonePlaybackState& - reference to RingtonePlaybackState class
       * @param[in] ActType - asynchronous completion token
       * @param[out]
       * @param[in,out]
       *
       * @return void
       */
      virtual void getRingtonePlaybackStatusResponse(IN const PmResult& pmResult,
            IN const RingtonePlaybackState& ringtonePlaybackState, IN const ActType act = 0u) = 0;

      /**
       * This method is used to notify the changes in ringtonePlaybackState of the vehicle ringtone.
       *
       * @param[in] ringtonePlaybackState& - reference to RingtonePlaybackState class
       * @param[out]
       * @param[in,out]
       *
       * @return void
       */
      virtual void onRingtonePlaybackStatusChanged(IN const RingtonePlaybackState& ringtonePlaybackState) = 0;

      /**
       * Response for the startStopWaitingMode request.
       *
       * @param[in] PmResult& - reference to PmResult class
       * @param[in] BdAddress& - device address
       * @param[in] ActType - asynchronous completion token
       * @param[out]
       * @param[in,out]
       *
       * @return void
       */
      virtual void startStopWaitingModeResponse(IN const PmResult& pmResult, IN const BdAddress& deviceAddress,
            IN const ActType act = 0u) = 0;

      /**
       * Response for the dial request.
       *
       * @param[in] PmResult& - reference to PmResult class
       * @param[in] BdAddress& - device address
       * @param[in] CallInstance - call instance
       * @param[in] ActType - asynchronous completion token
       * @param[out]
       * @param[in,out]
       *
       * @return void
       */
      virtual void dialResponse(IN const PmResult& pmResult, IN const BdAddress& deviceAddress,
            IN const CallInstance callInstance, IN const ActType act = 0u) = 0;

      /**
       * Response for the acceptCall request.
       *
       * @param[in] PmResult& - reference to PmResult class
       * @param[in] BdAddress& - device address
       * @param[in] CallInstance - call instance
       * @param[in] ActType - asynchronous completion token
       * @param[out]
       * @param[in,out]
       *
       * @return void
       */
      virtual void acceptCallResponse(IN const PmResult& pmResult, IN const BdAddress& deviceAddress,
            IN const CallInstance callInstance, IN const ActType act = 0u) = 0;

      /**
       * Response for the swapCall request.
       *
       * @param[in] PmResult& - reference to PmResult class
       * @param[in] BdAddress& - device address
       * @param[in] ActType - asynchronous completion token
       * @param[out]
       * @param[in,out]
       *
       * @return void
       */
      virtual void swapCallResponse(IN const PmResult& pmResult, IN const BdAddress& deviceAddress,
            IN const ActType act = 0u) = 0;

      /**
       * Response for the speedDial request.
       *
       * @param[in] PmResult& - reference to PmResult class
       * @param[in] BdAddress& - device address
       * @param[in] CallInstance - call instance
       * @param[in] ActType - asynchronous completion token
       * @param[out]
       * @param[in,out]
       *
       * @return void
       */
      virtual void speedDialResponse(IN const PmResult& pmResult, IN const BdAddress& deviceAddress,
            IN const CallInstance callInstance, IN const ActType act = 0u) = 0;

      /**
       * Response for the SendDTMF request for the device.
       *
       * @param[in] PmResult& - reference to PmResult class
       * @param[in] BdAddress& - device address
       * @param[in] ActType - asynchronous completion token
       * @param[out]
       * @param[in,out]
       *
       * @return void
       */
      virtual void sendDTMFResponse(IN const PmResult& pmResult, IN const BdAddress& deviceAddress,
            IN const ActType act = 0u) = 0;

      /**
       * Response for the TransferAudio request for the device.
       *
       * @param[in] PmResult& - reference to PmResult class
       * @param[in] BdAddress& - device address
       * @param[in] ActType - asynchronous completion token
       * @param[out]
       * @param[in,out]
       *
       * @return void
       */
      virtual void transferAudioResponse(IN const PmResult& pmResult, IN const BdAddress& deviceAddress,
            IN const ActType act = 0u) = 0;

      /**
       * This method provides the status of the Voice Recognition session.
       *
       * @param[in] PmResult& - reference to PmResult class
       * @param[in] BdAddress& - device address
       * @param[in] VoiceRecognitionStatus& - reference to VoiceRecognitionStatus class
       * @param[in] ActType - asynchronous completion token
       * @param[out]
       * @param[in,out]
       *
       * @return void
       */
      virtual void getVoiceRecognitionStatusResponse(IN const PmResult& pmResult, IN const BdAddress& deviceAddress,
            IN const VoiceRecognitionStatus& voiceRecognitionStatus, IN const ActType act = 0u) = 0;

      /**
       * This method is used to notify the status of Voice Recognition.
       *
       * @param[in] BdAddress& - device address
       * @param[in] VoiceRecognitionStatus& - reference to VoiceRecognitionStatus class
       * @param[out]
       * @param[in,out]
       *
       * @return void
       */
      virtual void onVoiceRecognitionStatusChanged(IN const BdAddress& deviceAddress,
            IN const VoiceRecognitionStatus& voiceRecognitionStatus) = 0;

      /**
       * Response for the startStopVoiceRecognition request.
       * @param[in] PmResult& - reference to PmResult class
       * @param[in] BdAddress& deviceAddress
       * @param[in] ActType - asynchronous completion token
       * @param[out]
       * @param[in,out]
       *
       * @return void
       */
      virtual void startStopVoiceRecognitionResponse(IN const PmResult& pmResult, IN const BdAddress& deviceAddress,
            IN const ActType act = 0u) = 0;

      /**
       * This method provides the VoiceRecognition feature available in the AG and current availability state of the same.
       *
       * @param[in] PmResult& - reference to PmResult class
       * @param[in] BdAddress& - device address
       * @param[in] EnhancedVoiceRecognitionFeature& - reference to EnhancedVoiceRecognitionFeature class
       * @param[in] ActType - asynchronous completion token
       * @param[out]
       * @param[in,out]
       *
       * @return void
       */
      virtual void getEnhancedVoiceRecognitionFeatureResponse(IN const PmResult& pmResult, IN const BdAddress& deviceAddress,
            IN const EnhancedVoiceRecognitionFeature& enhancedVoiceRecognitionFeature, IN const ActType act = 0u) = 0;

      /**
       * This method is used to notify the changes in the availability of the Enhanced VR feature
       * (SIRI in iPhone, Google Voice in Android phones, etc..) in the AudioGateway.
       *
       * @param[in] BdAddress& - device address
       * @param[in] EnhancedVoiceRecognitionFeature& - reference to EnhancedVoiceRecognitionFeature class
       * @param[out]
       * @param[in,out]
       *
       * @return void
       */
      virtual void onEnhancedVoiceRecognitionFeatureChanged(IN const BdAddress& deviceAddress,
            IN const EnhancedVoiceRecognitionFeature& enhancedVoiceRecognitionFeature) = 0;

      /**
       * This method provides the external VR state for the requested device
       *
       * @param[in] pmcore::PmResult& - reference to pmcore::PmResult class
       * @param[in] pmcore::BdAddress& - device address
       * @param[in] pmcore::ExtVoiceRecognitionStatus& - external VR status
       * @param[in] pmcore::ActType - asynchronous completion token
       * @param[out]
       * @param[in,out]
       *
       * @return void
       */
      virtual void getExternalVRStateResponse(IN const pmcore::PmResult& pmResult,
            IN const pmcore::BdAddress& deviceAddress,
            IN const pmcore::ExtVoiceRecognitionStatus& extVoiceRecognitionStatus,
            IN const pmcore::ActType act = 0u) = 0;

      /**
       * This method is used to notify the changes in external VR state
       *
       * @param[in] pmcore::BdAddress& - device address
       * @param[in] pmcore::ExtVoiceRecognitionStatus& - external VR status
       * @param[out]
       * @param[in,out]
       *
       * @return void
       */
      virtual void onExternalVRStateChanged(IN const pmcore::BdAddress& deviceAddress,
            IN const pmcore::ExtVoiceRecognitionStatus& extVoiceRecognitionStatus) = 0;

      /**
       * Response for the activateExternalVR request.
       * @param[in] PmResult& - reference to PmResult class
       * @param[in] BdAddress& deviceAddress
       * @param[in] ActType - asynchronous completion token
       * @param[out]
       * @param[in,out]
       *
       * @return void
       */
      virtual void activateExternalVRResponse(IN const PmResult& pmResult, IN const BdAddress& deviceAddress,
            IN const ActType act = 0u) = 0;

      /**
       * Response for the deActivateExternalVR request.
       * @param[in] PmResult& - reference to PmResult class
       * @param[in] BdAddress& deviceAddress
       * @param[in] ActType - asynchronous completion token
       * @param[out]
       * @param[in,out]
       *
       * @return void
       */
      virtual void deActivateExternalVRResponse(IN const PmResult& pmResult, IN const BdAddress& deviceAddress,
            IN const ActType act = 0u) = 0;

      /**
       * This method is used to notify call existence status to the clients so that Phone sub state can be updated
       *
       *TODO: This functionality should be handled in IPC wrapper and this should be removed in future.
       *
       * @param[in] BdAddress& - device address
       * @param[in] callExistenceState - call existence state
       * @param[out]
       * @param[in,out]
       *
       * @return void
       */
      virtual void onPmStateChanged(IN const BdAddress& deviceAddres, IN const CallExistenceState
            callExistenceState) = 0;
   };
}

#endif // IPmCoreCallbackIf_h
