/**
 * @file DeviceInfo.cpp
 *
 * @swcomponent PhoneCallManager
 *
 * @brief This file contains the type definition of the DeviceInfo class
 *
 * @copyright (C) 2016 Robert Bosch GmbH.
 *            The reproduction, distribution and utilization of this file as
 *            well as the communication of its contents to others without express
 *            authorization is prohibited. Offenders will be held liable for the
 *            payment of damages. All rights reserved in the event of the grant
 *            of a patent, utility model or design.
 *
 * @details
 *
 * @ingroup PmCore
 */

#include "DeviceInfo.h"
#include "PmCoreMainController.h"
#include "PmAppTrace.h"

#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#define ETG_DEFAULT_TRACE_CLASS TR_CLASS_PM_CORE
#ifdef VARIANT_S_FTR_ENABLE_FW_ETG_USAGE
#include "trcGenProj/Header/DeviceInfo.cpp.trc.h"
#define ETG_DEFAULT_TRACE_CLASS TR_CLASS_PM_CORE
#endif
#endif

namespace pmcore
{
   DeviceInfo::DeviceInfo() :
         _deviceHandle(PM_DEVICEHANDLE_ZERO),
         _deviceAddress(""),
         _voiceMailList(),
         _deviceRole(DEVICEROLE_DEFAULT),
         _isAppleDevice(false)
   {
   }

   DeviceInfo::DeviceInfo(const DeviceHandle deviceHandle, const BdAddress& deviceAddress) :
         _deviceHandle(deviceHandle),
         _deviceAddress(deviceAddress),
         _voiceMailList(),
         _deviceRole(DEVICEROLE_DEFAULT),
         _isAppleDevice(false)
   {
      ETG_TRACE_USR1(("DeviceInfo: DeviceHandle: %d, DeviceAddress : %s", deviceHandle, deviceAddress.c_str()));
   }

   DeviceInfo::~DeviceInfo()
   {
      ETG_TRACE_USR1(("~DeviceInfo"));
   }

   void DeviceInfo::getSupportedFeatures(SupportedFeatures& supportedFeatures)
   {
      ETG_TRACE_USR1(("getSupportedFeatures"));

      supportedFeatures = _supportedFeatures;
   }

   void DeviceInfo::setSupportedFeatures(const SupportedFeatures& supportedFeatures)
   {
      ETG_TRACE_USR1(("setSupportedFeatures"));

      if(supportedFeatures != _supportedFeatures)
      {
         _supportedFeatures = supportedFeatures;

         onSupportedFeaturesChanged();
      }
   }

   void DeviceInfo::onSupportedFeaturesChanged()
   {
      ETG_TRACE_USR1(("onSupportedFeaturesChanged"));

      SupportedFeatures supportedFeatures;
      getSupportedFeatures(supportedFeatures);

      PmCoreMainController::getInstance().getPmCoreCallbackIfWrapper().doOnSupportedFeaturesChanged(
            _deviceAddress, supportedFeatures);
   }

   void DeviceInfo::getNetworkStatus(NetworkStatus& networkStatus)
   {
      ETG_TRACE_USR1(("getNetworkStatus"));

      networkStatus = _networkStatus;
   }

   void DeviceInfo::setNetworkStatus(const NetworkStatus& networkStatus)
   {
      ETG_TRACE_USR1(("setNetworkStatus"));

      if(networkStatus != _networkStatus)
      {
         _networkStatus = networkStatus;

         onNetworkStatusChanged();
      }
   }

   void DeviceInfo::onNetworkStatusChanged()
   {
      ETG_TRACE_USR1(("onNetworkStatusChanged"));

      NetworkStatus networkStatus;
      getNetworkStatus(networkStatus);

      PmCoreMainController::getInstance().getPmCoreCallbackIfWrapper().doOnNetworkStatusChanged(
            _deviceAddress, networkStatus);
   }

   void DeviceInfo::getSignalStrength(SignalStrength& signalStrength)
   {
      ETG_TRACE_USR1(("getSignalStrength"));

      signalStrength = _signalStrength;
   }

   void DeviceInfo::setSignalStrength(const SignalStrength& signalStrength)
   {
      ETG_TRACE_USR1(("setSignalStrength"));

      if(signalStrength != _signalStrength)
      {
         _signalStrength = signalStrength;

         onSignalStrengthChanged();
      }
   }

   void DeviceInfo::onSignalStrengthChanged()
   {
      ETG_TRACE_USR1(("onSignalStrengthChanged"));

      SignalStrength signalStrength;
      getSignalStrength(signalStrength);

      PmCoreMainController::getInstance().getPmCoreCallbackIfWrapper().doOnSignalStrengthChanged(
            _deviceAddress, signalStrength);

   }

   void DeviceInfo::getBatteryChargeLevel(BatteryChargeLevel& batteryChargeLevel)
   {
      ETG_TRACE_USR1(("getBatteryChargeLevel"));

      batteryChargeLevel = _batteryChargeLevel;
   }

   void DeviceInfo::setBatteryChargeLevel(const BatteryChargeLevel& batteryChargeLevel)
   {
      ETG_TRACE_USR1(("setBatteryChargeLevel"));

      if(batteryChargeLevel != _batteryChargeLevel)
      {
         _batteryChargeLevel = batteryChargeLevel;

         onBatteryChargeLevelChanged();
      }
   }

   void DeviceInfo::onBatteryChargeLevelChanged()
   {
      ETG_TRACE_USR1(("onBatteryChargeLevelChanged"));

      BatteryChargeLevel batteryChargeLevel;
      getBatteryChargeLevel(batteryChargeLevel);

      PmCoreMainController::getInstance().getPmCoreCallbackIfWrapper().doOnBatteryChargeLevelChanged(
            _deviceAddress, batteryChargeLevel);

   }

   void DeviceInfo::getNetworkOperator(NetworkOperator& networkOpertor)
   {
      ETG_TRACE_USR1(("getNetworkOperator"));

      networkOpertor = _networkOperator;
   }

   void DeviceInfo::setNetworkOperator(const NetworkOperator& networkOpertor)
   {
      ETG_TRACE_USR1(("setNetworkOperator"));

      if(networkOpertor != _networkOperator)
      {
         _networkOperator = networkOpertor;

         onNetworkOperatorChanged();
      }
   }

   void DeviceInfo::onNetworkOperatorChanged()
   {
      ETG_TRACE_USR1(("onNetworkOperatorChanged"));

      NetworkOperator networkOpertor;
      getNetworkOperator(networkOpertor);

      PmCoreMainController::getInstance().getPmCoreCallbackIfWrapper().doOnNetworkOperatorChanged(
            _deviceAddress, networkOpertor);

   }

   void DeviceInfo::getDeviceRole(DeviceRole& deviceRole)
   {
      ETG_TRACE_USR1(("getDeviceRole"));

      deviceRole = _deviceRole;
   }

   void DeviceInfo::setDeviceRole(const DeviceRole& deviceRole)
   {
      ETG_TRACE_USR1(("setDeviceRole"));

      if(deviceRole != _deviceRole)
      {
         _deviceRole = deviceRole;
      }
   }

   bool DeviceInfo::isAppleDevice()
   {
      ETG_TRACE_USR1(("isAppleDevice: %d", _isAppleDevice));

      return _isAppleDevice;
   }

   void DeviceInfo::setAppleDevice(IN const bool isAppleDevice)
   {
      ETG_TRACE_USR1(("setAppleDevice: %d", isAppleDevice));

      if(isAppleDevice != _isAppleDevice)
      {
         _isAppleDevice = isAppleDevice;
      }
   }

   void DeviceInfo::getVoiceMailList(VoiceMailList& voiceMailList)
   {
      ETG_TRACE_USR1(("getVoiceMailList"));

      voiceMailList = _voiceMailList;
   }

   void DeviceInfo::setVoiceMailList(const VoiceMailList& voiceMailList)
   {
      ETG_TRACE_USR1(("setVoiceMailList"));

      if(voiceMailList != _voiceMailList)
      {
         _voiceMailList = voiceMailList;
      }
   }
}
