/**
 * @file DeviceInfoHandler.h
 *
 * @swcomponent PhoneCallManager
 *
 * @brief This file contains the definition of the DeviceInfoHandler class
 *
 * @copyright (C) 2016 Robert Bosch GmbH.
 *            The reproduction, distribution and utilization of this file as
 *            well as the communication of its contents to others without express
 *            authorization is prohibited. Offenders will be held liable for the
 *            payment of damages. All rights reserved in the event of the grant
 *            of a patent, utility model or design.
 *
 * @details This file handles the device related information and updating the same
 *          to clients if any changes in the same
 *
 * @ingroup PmCore
 */

#ifndef DeviceInfoHandler_h
#define DeviceInfoHandler_h

#include "PmCoreIfTypes.h"
#include "DeviceInfo.h"
#include "PropertyUpdateHandler.h"

   typedef enum
   {
      SET_ACTIVE_REQUEST_FROM_CLIENT,
      TEMP_ROLE_SWITCH,
      ACCEPTING_CALL,
      ACTIVATING_VR,
      STARTING_WAITING_MODE,
      TEMP_ROLE_SWITCH_AND_THEN_ACCEPTING_CALL,
      TEMP_ROLE_SWITCH_AND_THEN_ACTIVATING_VR,
      TEMP_ROLE_SWITCH_AND_THEN_STARTING_WAITING_MODE
   } SwitchingReason;

   typedef enum
   {
      CANNOT_BE_MADE_ACTIVE,
      ALREADY_ACTIVE,
      SWITCHED_TO_ACTIVE,
      SWITCHING_TO_ACTIVE_IN_PROGRESS
   } DeviceRoleSwitchResultEnum;

namespace pmcore
{

   class PendingRequestInfo
   {
   public:
      PendingRequestInfo() : _callInstance(CALL_INSTANCE_DEFAULT), _acceptOperation(ACCEPT_CALLS),
      _act(PM_DEFAULT_ACT)
      {

      }

      PendingRequestInfo(const ActType act) :
         _callInstance(CALL_INSTANCE_DEFAULT), _acceptOperation(ACCEPT_CALLS), _act(act)
      {

      }

      PendingRequestInfo(const CallInstance callInstance, const AcceptOperation acceptOperation, const ActType act) :
         _callInstance(callInstance), _acceptOperation(acceptOperation), _act(act)
      {

      }

      PendingRequestInfo(const PendingRequestInfo& ref) :
         _callInstance(ref._callInstance), _acceptOperation(ref._acceptOperation), _act(ref._act)
      {

      }

      friend void swap(PendingRequestInfo& first, PendingRequestInfo& second)
      {
         using std::swap;
         swap(first._callInstance, second._callInstance);
         swap(first._acceptOperation, second._acceptOperation);
         swap(first._act, second._act);
      }

      //lint -e{1529} suppress "assignment operator not first checking for assignment to this", see copy-and-swap idiom
      PendingRequestInfo& operator=(PendingRequestInfo ref)
      {
         swap(*this, ref);
         return *this;
      }

      bool operator==(const PendingRequestInfo& rhs) const
      {
         return ((this->_callInstance == rhs._callInstance)
               && (this->_acceptOperation == rhs._acceptOperation)
               && (this->_act == rhs._act));
      }

      bool operator!=(const PendingRequestInfo& rhs) const
      {
         return (!operator==(rhs));
      }

      CallInstance _callInstance;
      AcceptOperation _acceptOperation;
      ActType _act;
   };

   class ActiveSwitchingRequest
   {
   public:
      ActiveSwitchingRequest() : _deviceAddress(""), _switchingReason(SET_ACTIVE_REQUEST_FROM_CLIENT),
      _pendingRequestInfo(),
      _passiveSwitchingControllerType(CALL_CONTROLLER), _prevPassiveSwitchingControllerType(CALL_CONTROLLER)
      {

      }

      ActiveSwitchingRequest(const BdAddress deviceAddress, const SwitchingReason switchingReason,
            const PendingRequestInfo pendingRequestInfo, const ControllerType switchingControllerType) :
         _deviceAddress(deviceAddress), _switchingReason(switchingReason), _pendingRequestInfo(pendingRequestInfo),
         _passiveSwitchingControllerType(switchingControllerType),
         _prevPassiveSwitchingControllerType(CALL_CONTROLLER)
      {

      }

      ActiveSwitchingRequest(const ActiveSwitchingRequest& ref) :
         _deviceAddress(ref._deviceAddress), _switchingReason(ref._switchingReason),
         _pendingRequestInfo(ref._pendingRequestInfo),
         _passiveSwitchingControllerType(ref._passiveSwitchingControllerType),
         _prevPassiveSwitchingControllerType(ref._prevPassiveSwitchingControllerType)
      {

      }

      friend void swap(ActiveSwitchingRequest& first, ActiveSwitchingRequest& second)
      {
         using std::swap;
         swap(first._deviceAddress, second._deviceAddress);
         swap(first._switchingReason, second._switchingReason);
         swap(first._pendingRequestInfo, second._pendingRequestInfo);
         swap(first._passiveSwitchingControllerType, second._passiveSwitchingControllerType);
         swap(first._prevPassiveSwitchingControllerType, second._prevPassiveSwitchingControllerType);
      }

      //lint -e{1529} suppress "assignment operator not first checking for assignment to this", see copy-and-swap idiom
      ActiveSwitchingRequest& operator=(ActiveSwitchingRequest ref)
      {
         swap(*this, ref);
         return *this;
      }

      bool operator==(const ActiveSwitchingRequest& rhs) const
      {
         return ((this->_deviceAddress == rhs._deviceAddress)
               && (this->_switchingReason == rhs._switchingReason)
               && (this->_pendingRequestInfo == rhs._pendingRequestInfo)
               && (this->_passiveSwitchingControllerType == rhs._passiveSwitchingControllerType)
               && (this->_prevPassiveSwitchingControllerType == rhs._prevPassiveSwitchingControllerType));
      }

      bool operator!=(const ActiveSwitchingRequest& rhs) const
      {
         return (!operator==(rhs));
      }

      pmcore::BdAddress _deviceAddress;
      SwitchingReason _switchingReason;
      PendingRequestInfo _pendingRequestInfo;
      ControllerType _passiveSwitchingControllerType;

      // This parameter is useful when switchingReason is "USING_AVAILABLE_SLOT" and receiving accept call request from
      // passive device or VR activation request. Now if the transfer audio or pause audio or VR deactivation in
      // passive device fails, then the corresponding error is sent and the previous SwitchingControllerType to be
      // reminded
      ControllerType _prevPassiveSwitchingControllerType;
   };


   //class forward declaration
   class PmCoreIfMessage_GetSubscriberResult;

   class DeviceInfoHandler : public PropertyUpdateHandler
   {
   public:
      /**
       * Constructor of DeviceInfoHandler class
       *
       * @param[in]
       * @param[out]
       * @param[in,out]
       *
       * @return
       */
      DeviceInfoHandler();

      /**
       * Destructor of DeviceInfoHandler class
       *
       * @param[in]
       * @param[out]
       * @param[in,out]
       *
       * @return
       */
      ~DeviceInfoHandler();

      /**
       * This method is used to subscribe interested properties in BT stack event notifier.
       *
       * @param[in]
       * @param[out]
       * @param[in,out]
       *
       * @return
       */
      void subscribeToBtStackEventNotifier();

      /**
       * This method is used to inform the device connection
       *
       * @param[in] deviceHandle - device handle
       * @param[in] deviceAddress - BT address of the device
       * @param[out]
       * @param[in,out]
       *
       * @return void
       */
      void onDeviceConnected(IN const DeviceHandle deviceHandle, IN const BdAddress& deviceAddress);

      /**
       * This method is used to inform the device disconnection
       *
       * @param[in] deviceAddress - BT address of the device
       * @param[out]
       * @param[in,out]
       *
       * @return void
       */
      void onDeviceDisconnected(IN const BdAddress& deviceAddress);

      /**
       * This method is used to get property changes update from BT Stack
       * This method is overridden from PropertyUpdateHandler class
       *
       * @param[in] propertyId      - Id of the property or event
       * @param[in] propertyDetails - Details of the property or event
       * @param[out]
       * @param[in,out]
       *
       * @return
       */
      virtual void onPropertyUpdate(IN const PmCorePropertyAndEventId propertyId,
            IN std::shared_ptr<void> propertyDetails) override;

      /**
       * This method is used to check the given feature is supported or not
       *
       * @param[in] deviceAddress - BT address of the device
       * @param[in] feature - feature name
       * @param[out]
       * @param[in,out]
       *
       * @return bool
       */
      bool isFeatureSupported(IN const BdAddress& deviceAddress, IN const std::string& feature);

      /**
       * This method is used to get the Supported features
       *
       * @param[in] deviceAddress - BT address of the device
       * @param[in] act - asynchronous completion token
       * @param[out]
       * @param[in,out]
       *
       * @return void
       */
      void getSupportedFeatures(IN const BdAddress& deviceAddress, IN const ActType act);

      /**
       * This method is used to update the Supported features to Device Info
       *
       * @param[in] deviceAddress - BT address of the device
       * @param[in] supportedFeatures - supported features
       * @param[out]
       * @param[in,out]
       *
       * @return PmResult
       */
      void onSupportedFeaturesSignal(IN const BdAddress& deviceAddress, IN const SupportedFeatures& supportedFeatures);

      /**
       * This method is used to get the Network status
       *
       * @param[in] deviceAddress - BT address of the device
       * @param[in] act - asynchronous completion token
       * @param[out]
       * @param[in,out]
       *
       * @return void
       */
      void getNetworkStatus(IN const BdAddress& deviceAddress, IN const ActType act);

      /**
       * This method is used to update the Network status to clients
       *
       * @param[in] deviceAddress - BT address of the device
       * @param[in] networkStatus - network status
       * @param[out]
       * @param[in,out]
       *
       * @return void
       */
      void onNetworkStatusSignal(IN const BdAddress& deviceAddress, IN const NetworkStatus& networkStatus);

      /**
       * This method is used to get the Signal strength
       *
       * @param[in] deviceAddress - BT address of the device
       * @param[in] act - asynchronous completion token
       * @param[out]
       * @param[in,out]
       *
       * @return void
       */
      void getSignalStrength(IN const BdAddress& deviceAddress, IN const ActType act);

      /**
       * This method is used to update the Signal strength to clients
       *
       * @param[in] deviceAddress - BT address of the device
       * @param[in] signalStrength - signal strength
       * @param[out]
       * @param[in,out]
       *
       * @return void
       */
      void onSignalStrengthSignal(IN const BdAddress& deviceAddress, IN const SignalStrength& signalStrength);

      /**
       * This method is used to get the Battery charge level
       *
       * @param[in] deviceAddress - BT address of the device
       * @param[in] act - asynchronous completion token
       * @param[out]
       * @param[in,out]
       *
       * @return void
       */
      void getBatteryChargeLevel(IN const BdAddress& deviceAddress, IN const ActType act);

      /**
       * This method is used to update the Battery charge level to clients
       *
       * @param[in] deviceAddress - BT address of the device
       * @param[in] batteryChargeLevel - battery charge level
       * @param[out]
       * @param[in,out]
       *
       * @return void
       */
      void onBatteryChargeLevelSignal(IN const BdAddress& deviceAddress,
            IN const BatteryChargeLevel& batteryChargeLevel);

      /**
       * This method is used to get the Network operator
       *
       * @param[in] deviceAddress - BT address of the device
       * @param[in] act - asynchronous completion token
       * @param[out]
       * @param[in,out]
       *
       * @return void
       */
      void getNetworkOperator(IN const BdAddress& deviceAddress, IN const ActType act);

      /**
       * This method is used to update the Network operator to clients
       *
       * @param[in] deviceAddress - BT address of the device
       * @param[in] networkOperator - network operator
       * @param[out]
       * @param[in,out]
       *
       * @return void
       */
      void onNetworkOperatorSignal(IN const BdAddress& deviceAddress, IN const NetworkOperator& networkOpertaor);

      /**
       * This method is used to identify the Apple device from device address
       *
       * @param[in] deviceAddress - BT address of the device
       * @param[out]
       * @param[in,out]
       *
       * @return bool
       */
      bool isAppleDevice(IN const BdAddress& deviceAddress);

      /**
       * This method is used to set the Apple device
       *
       * @param[in] deviceAddress - BT address of the device
       * @param[in] isAppleDevice - true if Apple device
       * @param[out]
       * @param[in,out]
       *
       * @return void
       */
      void setAppleDevice(IN const BdAddress& deviceAddress, IN const bool isAppleDevice = false);

      /**
       * This method is used to get the Device Role
       *
       * @param[in] deviceAddress - BT address of the device
       * @param[in] act - asynchronous completion token
       * @param[out] deviceRole - Role of the BT Device (Active/Passive)
       * @param[in,out]
       *
       * @return PmResult
       */
      PmResult getDeviceRole(IN const BdAddress& deviceAddress, OUT DeviceRole& deviceRole,
            IN const ActType act = PM_DEFAULT_ACT);

      /**
       * This method is used to set the Device Role
       *
       * @param[in] setActivePassiveDeviceList - list of devices with roles to be set
       * @param[out]
       * @param[in,out]
       *
       * @return PmResult
       */
      PmResult setDeviceRoleRequest(IN ActivePassiveDeviceListMap& setActivePassiveDeviceList);

      /**
       * This method is used to acquire the free active device slot if available and revert back the change
       * once the received deviceAddress turns to be Idle again.
       * This function is expected to be called when
       * 1. VR is activated in the Passive device
       * 2. Passive device receives an incoming call
       * 3. Dial is made from Passive device
       *
       * @param[in]  deviceAddress - device which needs to be set as Active temporarily
       *             activeSwitchingRequest - the active switching info should be provided here
       * @param[out]
       * @param[in,out]
       *
       * @return DeviceRoleSwitchResultEnum
       */
      DeviceRoleSwitchResultEnum acquireFreeActiveDeviceSlot(const BdAddress& deviceAddress,
            const ActiveSwitchingRequest activeSwitchingRequest);

      /**
       * This method is used to clear the requested element from _devicesToBeMadePassive
       *
       * @param[in]  deviceAddress - device which needs to be set as Active temporarily
       * @param[out]
       * @param[in,out]
       *
       * @return
       */
      void clearElementFromDeviceToBeMadePassiveMap(const BdAddress& deviceAddress);

      /**
       * This method is used to acquire an Active device slot for the requested device.
       * This method will release SCO in an Active device and makes it a Passive and then provides
       * the Active device for the requested device.
       *
       * @param[in]  activeSwitchingRequest - the active switching info should be provided here
       * @param[out]
       * @param[in,out]
       *
       * @return true- Request processing started; false- the request is not processed
       */
      DeviceRoleSwitchResultEnum acquireActiveDeviceSlot(const ActiveSwitchingRequest activeSwitchingRequest);

      /**
       * This method is a contrary to acquireFreeActiveDeviceSlot().
       * The device which was made Active via acquireFreeActiveDeviceSlot()
       * should be again made Passive via processVRRemoved() when VR is stopped
       *
       * @param[in] deviceAddress - device which needs to be set as Passive
       * @param[out]
       * @param[in,out]
       *
       * @return
       */
      void processVRRemoved(const BdAddress& deviceAddress);

      /**
       * This method is a contrary to acquireFreeActiveDeviceSlot().
       * The device which was made Active via acquireFreeActiveDeviceSlot()
       * should be again made Passive via processVRRemoved() when all calls are removed
       *
       * @param[in] deviceAddress - device which needs to be set as Passive
       * @param[out]
       * @param[in,out]
       *
       * @return
       */
      void processAllCallsRemoved(const BdAddress& deviceAddress);

      /**
       * This method is used to get the Active device list
       *
       * @param[in]
       * @param[out] activeDeviceList - active device list
       * @param[in,out]
       *
       * @return void
       */
      void getActiveDevicesList(OUT BdAddressList& activeDeviceList);

      /**
       * This method is used to get the Active/Passive device list
       *
       * @param[in] act - asynchronous completion token
       * @param[out]
       * @param[in,out]
       *
       * @return void
       */
      void getActivePassiveDeviceList(IN const ActType act);

      /**
       * VoiceMailList response from stack, this method is used to set the device voice mail list (internal purpose)
       * and the same is forwarded to requested client.
       *
       * @param[in] pmCoreIfMessage - shared ptr for GetSubscriberResult
       * @param[out]
       * @param[in,out]
       *
       * @return void
       */
      void getVoiceMailListResponse(std::shared_ptr<PmCoreIfMessage_GetSubscriberResult> pmCoreIfMessage);

      /**
       * This method is used to get the device voice mail list
       *
       * @param[in] deviceAddress - BT address of the device
       * @param[in] act - asynchronous completion token
       * @param[out]
       * @param[in,out]
       *
       * @return void
       */
      void getVoiceMailList(IN const BdAddress& deviceAddress, IN const ActType act);

      /**
       * This method is used to get the device info list
       *
       * @param[in]
       * @param[out] deviceInfoList - device info list
       * @param[in,out]
       *
       * @return PmResult
       */
      PmResult getDeviceInfoList(OUT DeviceInfoList& deviceInfoList);

      /**
       * This method is used to post the Get SCOConnection Request
       *
       * @param[in] bdAddress - BT address of the device
       * @param[in] act       - acknowledgment token
       * @param[out]
       * @param[in,out]
       *
       * @return void
       */
      void getSCOConnectionRequest(IN const BdAddress& bdAddress, IN const ActType act);

      /**
       * This method is used to process the response from Sm after processing the "SwitchToPassive" request
       *
       * @param[in]  deviceAddress - Address of the device from which the "SwitchToPassive" response is received
       *             PMResult - Result of the operation
       *             erasePair - true- erase the iter found.
       *                False- Dont erase iter found. Will be cleared in the calling function
       * @param[out]
       * @param[in,out]
       *
       * @return true- if the event is successfully processed. Else if pause audio is done,
       * stop audio should be triggered, so that no channel is held unnecessarily.
       */
      bool processSwitchToPassiveResponse(const BdAddress deviceAddress, const PmResult pmResult,
            const bool erasePair = true);

      /**
       * This method is used to check the SCO connection status of a device
       *
       * @param[in] bdAddress        - BT address of the device
       * @param[in]
       * @param[out]
       * @param[in,out]
       *
       * @return true- if SCO is established, false- if SCO is not established
       */
      bool isSCOEstablished(const BdAddress& deviceAddress);

      /**
       * This method is used to check whether the device is in Idle state ie., there are no calls, no VR and
       * no External VR in the device
       *
       * @param[in] deviceAddress        - BT address of the device
       * @param[in]
       * @param[out]
       * @param[in,out]
       *
       * @return true- if the device is in Idle state, else false
       */
      bool isDeviceInIdleState(const BdAddress& deviceAddress);

   private:
      /**
       * This method is used to update the properties with default values when device disconnection happens
       *
       * @param[in] deviceAddress - BT address of the device
       * @param[out]
       * @param[in,out]
       *
       * @return void
       */
      void updatePropertiesWithDefaultValues(const BdAddress& deviceAddress);

      /**
       * This method is used to add the device to active passive device list
       *
       * @param[in] deviceAddress - BT address of the device
       * @param[out]
       * @param[in,out]
       *
       * @return bool
       */
      bool addDeviceToActivePassiveDeviceList(IN const BdAddress& deviceAddress);

      /**
       * This method is used to remove the device from active passive device list
       *
       * @param[in] deviceAddress - BT address of the device
       * @param[out]
       * @param[in,out]
       *
       * @return bool
       */
      bool removeDeviceFromActivePassiveDeviceList(IN const BdAddress& deviceAddress);

      /**
       * This method is used to update the Active/Passive device list to clients
       *
       * @param[in]
       * @param[out]
       * @param[in,out]
       *
       * @return void
       */
      void updateActivePassiveDeviceList();

      /**
       * This method is used to clear the active passive device list
       *
       * @param[in]
       * @param[out]
       * @param[in,out]
       *
       * @return void
       */
      void clearActivePassiveDeviceList();

      /**
       * This method is used to get the number active devices
       *
       * @param[in]
       * @param[out]
       * @param[in,out]
       *
       * @return uint8_t - number of active devices
       */
      DevicesCount getNumberOfActiveDevices() const;

      /**
       * This method is used to add the device to device info list
       *
       * @param[in] deviceHandle - device handle
       * @param[in] deviceAddress - BT address of the device
       * @param[out]
       * @param[in,out]
       *
       * @return bool
       */
      bool addDeviceToDeviceInfoList(IN const DeviceHandle deviceHandle, IN const BdAddress& deviceAddress);

      /**
       * This method is used to remove the device from device info list
       *
       * @param[in] deviceAddress - BT address of the device
       * @param[out]
       * @param[in,out]
       *
       * @return bool
       */
      bool removeDeviceFromDeviceInfoList(IN const BdAddress& deviceAddress);

      /**
       * This method is used to clear the device info list
       *
       * @param[in]
       * @param[out]
       * @param[in,out]
       *
       * @return void
       */
      void clearDeviceInfoList();

      /**
       * This method is used to
       * 1. set _activePassiveDeviceList
       * 2. udpate the clients about the change in _activePassiveDeviceList
       * 3. Set the PmAudioManager Wrapper Active devices list
       *
       * @param[in]  setActivePassiveDeviceList - list of devices with roles to be set
       * @param[out]
       * @param[in,out]
       *
       * @return PmResult
       */
      PmResult setDeviceRole(IN ActivePassiveDeviceListMap& setActivePassiveDeviceList, bool requestFromClient = false);

      /**
       * This method is used to process the Active switching ie., the requested aceept call in Passive device or the
       * VR Actuvation in passive device after the Active to Passive device switching is processed.
       *
       * @param[in]  devicetoPassive        - this is the device which should be made Passive
       *             activeSwitchingRequest - info needed for switching to Active device
       * @param[out]
       * @param[in,out]
       *
       * @return
       */
      void processActiveSwitching(const BdAddress devicetoPassive,
            const ActiveSwitchingRequest activeSwitchingRequest);

      /**
       * This method is used to print the "_activePassiveDeviceList"
       *
       * @param[in]
       * @param[out]
       * @param[in,out]
       *
       * @return
       */
      void printActivePassiveDeviceList();

      /**
       * This method is used to know whether the switching reason is for Accept call or Waiting mode
       *
       * @param[in]  switchingReason - Switching reason
       * @param[out]
       * @param[in,out]
       *
       * @return  true- if the switching reason is to Accept call or Start Waiting mode in Passive device. Else false
       */
      bool isSwitchReasonAcceptCallOrStartWaitMode(const SwitchingReason switchingReason);

      DeviceInfoList             _deviceInfoList;          /**< device info list */
      ActivePassiveDeviceListMap _activePassiveDeviceList; /**< active passive device list map */
      PropertyIdList             _propertyIdList;          /**< List of property Id's */

      // SCO connection status of each device is maintained in this map
      std::map<BdAddress, SCOStatus> _deviceSCOStatus;

      // When the key "device address" turns to Idle state, that device is made Passive and the corresponding value
      // device address is made to be Active and the corresponding action is taken
      std::map<BdAddress, ActiveSwitchingRequest> _devicesToBeMadePassive;
   };

} // namespace pmcore

#endif // DeviceInfoHandler_h
