/**
 * @file DeviceInfo.h
 *
 * @swcomponent PhoneCallManager
 *
 * @brief This file contains the definition of the DeviceInfo class
 *
 * @copyright (C) 2016 Robert Bosch GmbH.
 *            The reproduction, distribution and utilization of this file as
 *            well as the communication of its contents to others without express
 *            authorization is prohibited. Offenders will be held liable for the
 *            payment of damages. All rights reserved in the event of the grant
 *            of a patent, utility model or design.
 *
 * @details This file handles the device related information and updating the same
 *          to clients if any changes in the same
 *
 * @ingroup PmCore
 */

#ifndef DeviceInfo_h
#define DeviceInfo_h

#include "PmCoreIfTypes.h"

namespace pmcore
{
   class DeviceInfo
   {
   public:
      /**
       * Parameterized Constructor of DeviceInfo class
       *
       * @param[in] deviceHandle - device handle of the BT device
       * @param[in] deviceAddress - BT address of the device
       * @param[out]
       * @param[in,out]
       *
       * @return
       */
      DeviceInfo(IN const DeviceHandle deviceHandle, IN const BdAddress& deviceAddress);

      /**
       * Destructor of DeviceInfo class
       *
       * @param[in]
       * @param[out]
       * @param[in,out]
       *
       * @return
       */
      ~DeviceInfo();

      /**
       * This method is used to get the Supported features
       *
       * @param[in]
       * @param[out] supportedFeatures - supported feature of the BT device
       * @param[in,out]
       *
       * @return void
       */
      void getSupportedFeatures(OUT SupportedFeatures& supportedFeatures);

      /**
       * This method is used to set the Supported features
       *
       * @param[in] supportedFeatures - supported feature of the BT device
       * @param[out]
       * @param[in,out]
       *
       * @return void
       */
      void setSupportedFeatures(IN const SupportedFeatures& supportedFeatures);

      /**
       * This method is used to update the Supported features to clients
       *
       * @param[in]
       * @param[out]
       * @param[in,out]
       *
       * @return void
       */
      void onSupportedFeaturesChanged();

      /**
       * This method is used to get the Network status
       *
       * @param[in]
       * @param[out] networkStatus - Network status of the BT device
       * @param[in,out]
       *
       * @return void
       */
      void getNetworkStatus(OUT NetworkStatus& networkStatus);

      /**
       * This method is used to set the Network status
       *
       * @param[in] networkStatus - Network status of the BT device
       * @param[out]
       * @param[in,out]
       *
       * @return void
       */
      void setNetworkStatus(IN const NetworkStatus& networkStatus);

      /**
       * This method is used to update the Network status to clients
       *
       * @param[in]
       * @param[out]
       * @param[in,out]
       *
       * @return void
       */
      void onNetworkStatusChanged();

      /**
       * This method is used to get the Signal strength
       *
       * @param[in]
       * @param[out] signalStrength - Signal strength of the BT device
       * @param[in,out]
       *
       * @return void
       */
      void getSignalStrength(OUT SignalStrength& signalStrength);

      /**
       * This method is used to set the Signal strength
       *
       * @param[in] signalStrength - Signal strength of the BT device
       * @param[out]
       * @param[in,out]
       *
       * @return void
       */
      void setSignalStrength(IN const SignalStrength& signalStrength);

      /**
       * This method is used to update the Signal strength to clients
       *
       * @param[in]
       * @param[out]
       * @param[in,out]
       *
       * @return void
       */
      void onSignalStrengthChanged();

      /**
       * This method is used to get the Battery charge level
       *
       * @param[in]
       * @param[out] batteryChargeLevel - Battery charge level of the BT Device
       * @param[in,out]
       *
       * @return void
       */
      void getBatteryChargeLevel(OUT BatteryChargeLevel& batteryChargeLevel);

      /**
       * This method is used to set the Battery charge level
       *
       * @param[in] batteryChargeLevel - Battery charge level of the BT Device
       * @param[out]
       * @param[in,out]
       *
       * @return void
       */
      void setBatteryChargeLevel(IN const BatteryChargeLevel& batteryChargeLevel);

      /**
       * This method is used to update the Battery charge level to clients
       *
       * @param[in]
       * @param[out]
       * @param[in,out]
       *
       * @return void
       */
      void onBatteryChargeLevelChanged();

      /**
       * This method is used to get the Network operator
       *
       * @param[in]
       * @param[out] networkOpertor - Network operator of the BT Device
       * @param[in,out]
       *
       * @return void
       */
      void getNetworkOperator(OUT NetworkOperator& networkOpertor);

      /**
       * This method is used to set the Network operator
       *
       * @param[in] networkOpertor - Network operator of the BT Device
       * @param[out]
       * @param[in,out]
       *
       * @return void
       */
      void setNetworkOperator(IN const NetworkOperator& networkOpertor);

      /**
       * This method is used to update the Network operator to clients
       *
       * @param[in]
       * @param[out]
       * @param[in,out]
       *
       * @return void
       */
      void onNetworkOperatorChanged();

      /**
       * This method is used to get the Device Role
       *
       * @param[in]
       * @param[out] deviceRole - Role of the BT Device (Active/Passive)
       * @param[in,out]
       *
       * @return void
       */
      void getDeviceRole(OUT DeviceRole& deviceRole);

      /**
       * This method is used to set the Device Role
       *
       * @param[in] deviceRole - Role of the BT Device
       * @param[out]
       * @param[in,out]
       *
       * @return void
       */
      void setDeviceRole(IN const DeviceRole& deviceRole);

      /**
       * This method is used to identify the Apple device
       *
       * @param[in]
       * @param[out]
       * @param[in,out]
       *
       * @return bool
       */
      bool isAppleDevice();

      /**
       * This method is used to set the Apple device
       *
       * @param[in] isAppleDevice - true if Apple device
       * @param[out]
       * @param[in,out]
       *
       * @return void
       */
      void setAppleDevice(IN const bool isAppleDevice = false);

      /**
       * This method is used to get the VoiceMailList
       *
       * @param[in]
       * @param[out] voiceMailList - VoiceMailList of the BT Device
       * @param[in,out]
       *
       * @return void
       */
      void getVoiceMailList(OUT VoiceMailList& voiceMailList);

      /**
       * This method is used to set the VoiceMailList
       *
       * @param[in] voiceMailList - VoiceMailList of the BT Device
       * @param[out]
       * @param[in,out]
       *
       * @return void
       */
      void setVoiceMailList(IN const VoiceMailList& voiceMailList);

   private:
      /**
       * Constructor of DeviceInfo class
       * Note: Intentionally added in private, since _deviceHandle and _deviceAddress to be known at the time of
       *       creation of this class instance. Use parameterized constructor instead of this.
       *
       * @param[in]
       * @param[out]
       * @param[in,out]
       *
       * @return
       */
      DeviceInfo();

      DeviceHandle       _deviceHandle;       /**< device handle */
      BdAddress          _deviceAddress;      /**< device address */
      SupportedFeatures  _supportedFeatures;  /**< instance of SupportedFeatures class */
      NetworkStatus      _networkStatus;      /**< instance of NetworkStatus class */
      SignalStrength     _signalStrength;     /**< instance of SignalStrength class */
      BatteryChargeLevel _batteryChargeLevel; /**< instance of BatteryChargeLevel class */
      NetworkOperator    _networkOperator;    /**< instance of NetworkOperator class */
      VoiceMailList      _voiceMailList;      /**< voice mail list */
      DeviceRole         _deviceRole;         /**< device role*/
      bool               _isAppleDevice;      /**< boolean to identify Apple device */
   };

   typedef std::map<BdAddress, DeviceInfo> DeviceInfoList;

} // namespace pmcore

#endif // DeviceInfo_h
