/**
 * @file Database.cpp
 *
 * @swcomponent PhoneCallManager
 *
 * @brief This file contains the definition of the Database class
 *
 * @copyright (C) 2016 Robert Bosch GmbH.
 *            The reproduction, distribution and utilization of this file as
 *            well as the communication of its contents to others without express
 *            authorization is prohibited. Offenders will be held liable for the
 *            payment of damages. All rights reserved in the event of the grant
 *            of a patent, utility model or design.
 *
 * @details This file handles the database related operations
 *
 * @ingroup PmCore
 */

#include "Database.h"
#include <time.h>
#include <errno.h>
#include <unistd.h>
#include <vector>
#include "PmAppTrace.h"

#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#define ETG_DEFAULT_TRACE_CLASS TR_CLASS_PM_CORE
#ifdef VARIANT_S_FTR_ENABLE_FW_ETG_USAGE
#include "trcGenProj/Header/Database.cpp.trc.h"
#define ETG_DEFAULT_TRACE_CLASS TR_CLASS_PM_CORE
#endif
#endif

namespace pmcore
{
   Database::Database()
   {
      _dbHandle = nullptr;
      _sqlStatement = nullptr;
      _queryResult = SQLITE_ERROR;

      // accept URI's as filenames
      (void)sqlite3_config(SQLITE_CONFIG_URI, 1);
   }

   Database::~Database()
   {
      _dbHandle = nullptr;
      _sqlStatement = nullptr;
      _queryResult = SQLITE_ERROR;
   }

   bool Database::columnSelected()
   {
      bool isColumnSelected = false;

      if(SQLITE_ROW == _queryResult)
      {
         isColumnSelected = true;
      }

      return isColumnSelected;
   }

   void Database::getInt(const int column, int& value)
   {
      ETG_TRACE_USR1(("Database::getInt() column : %d", column));

      Locker locker(&_dbAccessLock);

      if(nullptr != _sqlStatement)
      {
         int columnType = sqlite3_column_type(_sqlStatement, column);

         if(SQLITE_INTEGER == columnType)
         {
            value = sqlite3_column_int(_sqlStatement, column);

            ETG_TRACE_USR1(("Database::getInt() value : %d", value));
         }
         else
         {
            ETG_TRACE_ERR(("Database::getInt() Invalid data format requested, actual format is %d", columnType));
         }
      }
      else
      {
         ETG_TRACE_ERR(("Database::getInt() Invalid SqlStatement"));
      }
   }

   void Database::getString(const int column, std::string& value)
   {
      ETG_TRACE_USR1(("Database::getString() column : %d", column));

      Locker locker(&_dbAccessLock);

      if(nullptr != _sqlStatement)
      {
         int columnType = sqlite3_column_type(_sqlStatement, column);

         if(SQLITE_TEXT == columnType)
         {
            value = std::string(reinterpret_cast<const char*>(sqlite3_column_text(_sqlStatement, column)));

            ETG_TRACE_USR1(("Database::getString() value : %s", value.c_str()));
         }
         else
         {
            ETG_TRACE_ERR(("Database::getString() Invalid data format requested, actual format is %d", columnType));
         }
      }
      else
      {
         ETG_TRACE_ERR(("Database::getString() Invalid SqlStatement"));
      }
   }

   void Database::getErrMsg(std::string& value)
   {
      Locker locker(&_dbAccessLock);

      if(nullptr != _dbHandle)
      {
         value = sqlite3_errmsg(_dbHandle);

         ETG_TRACE_USR1(("Database::getErrMsg() value : %s", value.c_str()));
      }
      else
      {
         ETG_TRACE_ERR(("Database::getErrMsg() Invalid DbHandle"));
      }
   }

   bool Database::open(const std::string& fileName)
   {
      ETG_TRACE_USR1(("Database::open() fileName : %s", fileName.c_str()));

      Locker locker(&_dbAccessLock);

      bool retValue = false;
      _queryResult = SQLITE_ERROR;

      if(false == fileName.empty())
      {
         _queryResult = sqlite3_open_v2(fileName.c_str(), &_dbHandle,
               SQLITE_OPEN_READWRITE | SQLITE_OPEN_CREATE, nullptr);

         if((SQLITE_OK == _queryResult) && (nullptr != _dbHandle))
         {
            ETG_TRACE_USR1(("Database::open() success, Received DbHandle : 0x%p", _dbHandle));

            // register busy handler
            _queryResult = sqlite3_busy_handler(_dbHandle, busyCallback, _dbHandle);

            if(SQLITE_OK == _queryResult)
            {
               ETG_TRACE_ERR(("Database::open(): busy handler registration success"));

               retValue = configure();
            }
            else
            {
               ETG_TRACE_ERR(("Database::open(): busy handler registration failed: %d", _queryResult));
            }
         }
         else
         {
            ETG_TRACE_ERR(("Database::open() failed result : %d, errmsg : %s",
                  _queryResult, sqlite3_errstr(_queryResult)));
         }
      }
      else
      {
         ETG_TRACE_ERR(("Database::open() Invalid FileName"));
      }

      return retValue;
   }

   bool Database::close()
   {
      ETG_TRACE_USR1(("Database::close() entered"));

      Locker locker(&_dbAccessLock);

      bool retValue = false;
      _queryResult = SQLITE_ERROR;

      if(nullptr != _dbHandle)
      {
         _queryResult = sqlite3_close_v2(_dbHandle);

         if(SQLITE_OK == _queryResult)
         {
            retValue = true;
            _dbHandle = nullptr;

            ETG_TRACE_USR1(("Database::close() success"));
         }
         else
         {
            ETG_TRACE_ERR(("Database::close() failed result : %d, errmsg : %s",
                  _queryResult, sqlite3_errstr(_queryResult)));
            ETG_TRACE_ERR(("Database::close() failed errmsg : %s", sqlite3_errmsg(_dbHandle)));
         }
      }
      else
      {
         ETG_TRACE_ERR(("Database::close() Invalid DbHandle"));
      }

      return retValue;
   }

   void Database::attach(const std::string& db, const std::string& alias)
   {
      ETG_TRACE_USR1(("Database::attach() db : %50s, alias : %50s", db.c_str(), alias.c_str()));

      //TODO: implement
   }

   void Database::detach()
   {
      ETG_TRACE_USR1(("Database::detach() entered"));

      //TODO: implement
   }

   int Database::beginTransaction()
   {
      ETG_TRACE_USR1(("Database::beginTransaction() entered"));

      Locker locker(&_dbAccessLock);

      _queryResult = SQLITE_ERROR;

      std::string query("BEGIN TRANSACTION;");

      if(nullptr != _dbHandle)
      {
         char* errmsg = nullptr;

         _queryResult = sqlite3_exec(_dbHandle, query.c_str(), nullptr, nullptr, &errmsg);

         if(SQLITE_OK == _queryResult)
         {
            ETG_TRACE_USR1(("Database::beginTransaction() success"));
         }
         else
         {
            ETG_TRACE_ERR(("Database::beginTransaction() failed result : %d, errmsg : %s",
                  _queryResult, sqlite3_errstr(_queryResult)));
            ETG_TRACE_ERR(("Database::beginTransaction() failed errmsg : %s", errmsg));

            sqlite3_free(errmsg);
         }
      }
      else
      {
         ETG_TRACE_ERR(("Database::beginTransaction() Invalid DbHandle (or) Empty Query"));
      }

      return _queryResult;
   }

   int Database::endTransaction()
   {
      ETG_TRACE_USR1(("Database::endTransaction() entered"));

      Locker locker(&_dbAccessLock);

      _queryResult = SQLITE_ERROR;

      std::string query("COMMIT TRANSACTION;");

      if(nullptr != _dbHandle)
      {
         char* errmsg = nullptr;

         _queryResult = sqlite3_exec(_dbHandle, query.c_str(), nullptr, nullptr, &errmsg);

         if(SQLITE_OK == _queryResult)
         {
            ETG_TRACE_USR1(("Database::endTransaction() success"));
         }
         else
         {
            ETG_TRACE_ERR(("Database::endTransaction() failed result : %d, errmsg : %s",
                  _queryResult, sqlite3_errstr(_queryResult)));
            ETG_TRACE_ERR(("Database::endTransaction() failed errmsg : %s", errmsg));

            sqlite3_free(errmsg);
         }
      }
      else
      {
         ETG_TRACE_ERR(("Database::endTransaction() Invalid DbHandle (or) Empty Query"));
      }

      return _queryResult;
   }

   int Database::prepare(std::string& query)
   {
      ETG_TRACE_USR4(("Database::prepare() query : %s", query.c_str()));

      Locker locker(&_dbAccessLock);

      _queryResult = SQLITE_ERROR;

      if((nullptr != _dbHandle) && (false == query.empty()))
      {
         _queryResult = sqlite3_prepare_v2(_dbHandle, query.c_str(), -1, &_sqlStatement, nullptr);

         if(SQLITE_OK == _queryResult)
         {
            ETG_TRACE_USR1(("Database::prepare() success"));
         }
         else
         {
            ETG_TRACE_ERR(("Database::prepare() failed result : %d, errmsg : %s",
                  _queryResult, sqlite3_errstr(_queryResult)));
            ETG_TRACE_ERR(("Database::prepare() failed errmsg : %s", sqlite3_errmsg(_dbHandle)));
         }
      }
      else
      {
         ETG_TRACE_ERR(("Database::prepare() Invalid DbHandle (or) Empty Query"));
      }

      return _queryResult;
   }

   int Database::bind(const int column, const int value)
   {
      ETG_TRACE_USR1(("Database::bind() column : %d, value : %d", column, value));

      Locker locker(&_dbAccessLock);

      _queryResult = SQLITE_ERROR;

      if(nullptr != _sqlStatement)
      {
         _queryResult = sqlite3_bind_int(_sqlStatement, column, value);

         if(SQLITE_OK == _queryResult)
         {
            ETG_TRACE_USR1(("Database::bind() success"));
         }
         else
         {
            ETG_TRACE_ERR(("Database::bind() failed result : %d, errmsg : %s",
                  _queryResult, sqlite3_errstr(_queryResult)));
         }
      }
      else
      {
         ETG_TRACE_ERR(("Database::bind() Invalid SqlStatement"));
      }

      return _queryResult;
   }

   int Database::bind(const int column, const std::string& value)
   {
      ETG_TRACE_USR1(("Database::bind() column : %d, value : %s", column, value.c_str()));

      Locker locker(&_dbAccessLock);

      _queryResult = SQLITE_ERROR;

      if(nullptr != _sqlStatement)
      {
         _queryResult = sqlite3_bind_text(_sqlStatement, column, value.c_str(), -1, nullptr);

         if(SQLITE_OK == _queryResult)
         {
            ETG_TRACE_USR1(("Database::bind() success"));
         }
         else
         {
            ETG_TRACE_ERR(("Database::bind() failed result : %d, errmsg : %s",
                  _queryResult, sqlite3_errstr(_queryResult)));
         }
      }
      else
      {
         ETG_TRACE_ERR(("Database::bind() Invalid SqlStatement"));
      }

      return _queryResult;
   }

   int Database::step()
   {
      ETG_TRACE_USR1(("Database::step() entered"));

      Locker locker(&_dbAccessLock);

      _queryResult = SQLITE_ERROR;

      if(nullptr != _sqlStatement)
      {
         _queryResult = sqlite3_step(_sqlStatement);

         if(SQLITE_DONE == _queryResult)
         {
            ETG_TRACE_USR1(("Database::step() finished successfully"));

            //reset the prepared statement object back to its initial state.
            sqlite3_reset(_sqlStatement);
         }
         else if(SQLITE_ROW == _queryResult)
         {
            ETG_TRACE_USR1(("Database::step() data received"));
         }
         else
         {
            ETG_TRACE_ERR(("Database::step() failed result : %d, errmsg : %s",
                  _queryResult, sqlite3_errstr(_queryResult)));
         }
      }
      else
      {
         ETG_TRACE_ERR(("Database::step() Invalid SqlStatement"));
      }

      return _queryResult;
   }

   int Database::finalize()
   {
      ETG_TRACE_USR1(("Database::finalize() entered"));

      Locker locker(&_dbAccessLock);

      _queryResult = SQLITE_ERROR;

      if(nullptr != _sqlStatement)
      {
         _queryResult = sqlite3_finalize(_sqlStatement);

         if(SQLITE_OK == _queryResult)
         {
            ETG_TRACE_USR1(("Database::finalize() success"));

            _sqlStatement = nullptr;
         }
         else
         {
            ETG_TRACE_ERR(("Database::finalize() failed result : %d, errmsg : %s",
                  _queryResult, sqlite3_errstr(_queryResult)));
         }
      }
      else
      {
         ETG_TRACE_ERR(("Database::finalize() Invalid SqlStatement"));
      }

      return _queryResult;
   }

   int Database::exec(std::string& query)
   {
      ETG_TRACE_USR1(("Database::exec() query : %s", query.c_str()));

      Locker locker(&_dbAccessLock);

      _queryResult = SQLITE_ERROR;

      if((nullptr != _dbHandle) && (false == query.empty()))
      {
         char* errmsg = nullptr;

         _queryResult = sqlite3_exec(_dbHandle, query.c_str(), nullptr, nullptr, &errmsg);

         if(SQLITE_OK == _queryResult)
         {
            ETG_TRACE_USR1(("Database::exec() success"));
         }
         else
         {
            ETG_TRACE_ERR(("Database::exec() failed result : %d, errmsg : %s",
                  _queryResult, sqlite3_errstr(_queryResult)));
            ETG_TRACE_ERR(("Database::exec() failed errmsg : %s", errmsg)); //this equal to sqlite3_errmsg(_dbHandle);

            sqlite3_free(errmsg);
         }
      }
      else
      {
         ETG_TRACE_ERR(("Database::exec() Invalid DbHandle (or) Empty Query"));
      }

      return _queryResult;
   }

   bool Database::configure()
   {
      ETG_TRACE_USR1(("Database::configure() entered"));

      bool result = false;
      std::vector<std::string> connectionConfiguration;

      connectionConfiguration.push_back(std::string("PRAGMA FOREIGN_KEYS = OFF;"));
      connectionConfiguration.push_back(std::string("PRAGMA JOURNAL_MODE = OFF;"));
      connectionConfiguration.push_back(std::string("PRAGMA LOCKING_MODE = NORMAL;"));
      connectionConfiguration.push_back(std::string("PRAGMA SYNCHRONOUS = OFF;")); // NORMAL
      connectionConfiguration.push_back(std::string("PRAGMA AUTOMATIC_INDEX = OFF;"));
      connectionConfiguration.push_back(std::string("PRAGMA SECURE_DELETE = OFF;"));
      connectionConfiguration.push_back(std::string("PRAGMA TEMP_STORE = MEMORY;"));
      connectionConfiguration.push_back(std::string("PRAGMA ENCODING = \"UTF-8\";"));
      connectionConfiguration.push_back(std::string("PRAGMA CACHE_SIZE = 10000;"));

      auto it = connectionConfiguration.begin();

      // Execute the database initialization script
      while (connectionConfiguration.end() != it)
      {
         // Execute SQL query
         _queryResult = exec(*it);

         // Check for success
         if (SQLITE_OK == _queryResult)
         {
            result = true;
            it++;
         }
         else
         {
            result = false;
            break;
         }
      }

      return result;
   }

   int Database::busyCallback(void* userData, int times)
   {
      ETG_TRACE_USR4(("busyCallback() entered"));

      long sleepTime = times * 10000;
      sqlite3 *dbHandle = (sqlite3 *)userData;

      //TODO to be checked the shutdown sequence
      // ask a global variable if this busy should end because of shutdown
      //if (mShutDown) {
      //   ETG_TRACE_USR2(("DB busy, but asked for shutdown"));
      //   mShutDown++;
      //   return 0;
      //}

      // limit the sleep time to max shutdown delay time
      long maxShutdownDelay = 1000000L;

      if (sleepTime > maxShutdownDelay)
      {
         sleepTime = maxShutdownDelay;
      }

      if (!(times % 10))
      {
         // print only every 10th time
         ETG_TRACE_USR2(("DB busy, waiting for %d us: %s", (int)sleepTime, sqlite3_errmsg(dbHandle)));
      }

      // sleep a while
      usleep((useconds_t)sleepTime);

      return 1; // = 1: continue trying
   }

} // namespace pmcommon
