/**
 * @file YakinduSmVoiceCallIf.cpp
 *
 * @swcomponent PhoneCallManager
 *
 * @brief This file contains the definition of the YakinduSmVoiceCallIf class
 *
 * @copyright (C) 2016 Robert Bosch GmbH.
 *            The reproduction, distribution and utilization of this file as
 *            well as the communication of its contents to others without express
 *            authorization is prohibited. Offenders will be held liable for the
 *            payment of damages. All rights reserved in the event of the grant
 *            of a patent, utility model or design.
 *
 * @details This file provides the state machine interfaces of voice call
 *
 * @ingroup PmCore
 */

#include "YakinduSmVoiceCallIf.h"
#include "PmCallHandler.h"
#include "PmAppTrace.h"

#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#define ETG_DEFAULT_TRACE_CLASS TR_CLASS_PM_CORE
#ifdef VARIANT_S_FTR_ENABLE_FW_ETG_USAGE
#include "trcGenProj/Header/YakinduSmVoiceCallIf.cpp.trc.h"
#define ETG_DEFAULT_TRACE_CLASS TR_CLASS_PM_CORE
#endif
#endif

namespace pmcore
{
   YakinduSmVoiceCallIf::YakinduSmVoiceCallIf():_deviceToCallHandlerSmMap()
   {
   }

   YakinduSmVoiceCallIf::~YakinduSmVoiceCallIf()
   {
      _deviceToCallHandlerSmMap.clear();
   }

   void YakinduSmVoiceCallIf::onDeviceConnected(IN const BdAddress& deviceAddress)
   {
      ETG_TRACE_USR4(("YakinduSmVoiceCallIf::onDeviceConnected"));

      ::std::map<BdAddress, PmCallHandler*>::iterator it = _deviceToCallHandlerSmMap.find(deviceAddress);

      if(it == _deviceToCallHandlerSmMap.end())
      {
         PmCallHandler* pmCallHandler = new PmCallHandler(deviceAddress);
         if(nullptr != pmCallHandler)
         {
            _deviceToCallHandlerSmMap.insert(std::pair<BdAddress, PmCallHandler*>(deviceAddress, pmCallHandler));
         }
      }
      else
      {
         ETG_TRACE_USR4(("SM already exists for device %s", deviceAddress.c_str()));
      }
   }

   void YakinduSmVoiceCallIf::onDeviceDisconnected(IN const BdAddress& deviceAddress)
   {
      ETG_TRACE_USR4(("YakinduSmVoiceCallIf::onDeviceDisconnected"));

      std::map<BdAddress, PmCallHandler*>::iterator it = _deviceToCallHandlerSmMap.find(deviceAddress);
      if(it != _deviceToCallHandlerSmMap.end())
      {
         delete it->second;
         _deviceToCallHandlerSmMap.erase(it);
      }
   }

   void YakinduSmVoiceCallIf::setCallCount(IN const BdAddress& deviceAddress, IN const CallCount callCount)
   {
      std::map<BdAddress, PmCallHandler*>::iterator it = _deviceToCallHandlerSmMap.find(deviceAddress);
      if(it != _deviceToCallHandlerSmMap.end())
      {
         it->second->setCallCount(callCount);
      }
   }

   void YakinduSmVoiceCallIf::onAgCallStateUpdate(IN const BdAddress& deviceAddress,
         IN const CallStateEnumType callState)
   {
      std::map<BdAddress, PmCallHandler*>::iterator it = _deviceToCallHandlerSmMap.find(deviceAddress);
      if(it != _deviceToCallHandlerSmMap.end())
      {
         it->second->onAgCallStateUpdate(callState);
      }
   }

   void YakinduSmVoiceCallIf::onAudioManagerEventUpdate(IN const BdAddress& deviceAddress,
         IN const AudioManagerEventType amEvent)
   {
      std::map<BdAddress, PmCallHandler*>::iterator it = _deviceToCallHandlerSmMap.find(deviceAddress);
      if(it != _deviceToCallHandlerSmMap.end())
      {
         it->second->onAudioManagerEventUpdate(amEvent);
      }
   }

   void YakinduSmVoiceCallIf::onScoConnectionStatusChanged(IN const BdAddress& deviceAddress, IN const SCOStatus scoStatus)
   {
      std::map<BdAddress, PmCallHandler*>::iterator it = _deviceToCallHandlerSmMap.find(deviceAddress);
      if(it != _deviceToCallHandlerSmMap.end())
      {
         it->second->onScoConnectionStatusChanged(scoStatus);
      }
   }

   void YakinduSmVoiceCallIf::switchToPassiveRequest(IN const BdAddress& deviceAddress)
   {
      std::map<BdAddress, PmCallHandler*>::iterator it = _deviceToCallHandlerSmMap.find(deviceAddress);
      if(it != _deviceToCallHandlerSmMap.end())
      {
         it->second->switchToPassiveRequest();
      }
   }

   void YakinduSmVoiceCallIf::switchToPassiveClientRequest(IN const BdAddress& deviceAddress)
   {
      std::map<BdAddress, PmCallHandler*>::iterator it = _deviceToCallHandlerSmMap.find(deviceAddress);
      if(it != _deviceToCallHandlerSmMap.end())
      {
         // TODO: Need to establish a new interface to release channel
         it->second->switchToPassiveRequest();
      }
   }

   void YakinduSmVoiceCallIf::switchToActiveRequest(IN const BdAddress& deviceAddress)
   {
      std::map<BdAddress, PmCallHandler*>::iterator it = _deviceToCallHandlerSmMap.find(deviceAddress);
      if(it != _deviceToCallHandlerSmMap.end())
      {
         it->second->switchToActiveRequest();
      }
   }

   void YakinduSmVoiceCallIf::autoRejectCallRequest(IN const BdAddress& deviceAddress,
         IN const AutoRejectCall autoRejectCall)
   {
      std::map<BdAddress, PmCallHandler*>::iterator it = _deviceToCallHandlerSmMap.find(deviceAddress);
      if(it != _deviceToCallHandlerSmMap.end())
      {
         it->second->autoRejectCallRequest(autoRejectCall);
      }
   }

   void YakinduSmVoiceCallIf::onCallRemoved(IN const BdAddress& deviceAddress, IN const CallInstance callInstance)
   {
      std::map<BdAddress, PmCallHandler*>::iterator it = _deviceToCallHandlerSmMap.find(deviceAddress);
      if(it != _deviceToCallHandlerSmMap.end())
      {
         it->second->onCallRemoved(callInstance);
      }
   }
   void YakinduSmVoiceCallIf::onTimerEventUpdate(IN const BdAddress& deviceAddress)
   {
      ETG_TRACE_USR4(("YakinduSmVoiceCallIf::onTimerEventUpdate"));
      std::map<BdAddress, PmCallHandler*>::iterator it = _deviceToCallHandlerSmMap.find(deviceAddress);
      if(it != _deviceToCallHandlerSmMap.end())
      {
         it->second->onTimerEventUpdate();
      }
   }

   void YakinduSmVoiceCallIf::mergeCallsRequest(IN std::shared_ptr<PmCoreIfMessage_MergeCallsRequest> pmCoreIfMessage)
   {
      std::map<BdAddress, PmCallHandler*>::iterator it = _deviceToCallHandlerSmMap.find(pmCoreIfMessage->getBdAddress());
      if(it != _deviceToCallHandlerSmMap.end())
      {
         it->second->mergeCallsRequest(pmCoreIfMessage);
      }
   }

   void YakinduSmVoiceCallIf::splitCallsRequest(IN std::shared_ptr<PmCoreIfMessage_SplitCallsRequest> pmCoreIfMessage)
   {
      std::map<BdAddress, PmCallHandler*>::iterator it = _deviceToCallHandlerSmMap.find(pmCoreIfMessage->getBdAddress());
      if(it != _deviceToCallHandlerSmMap.end())
      {
         it->second->splitCallsRequest(pmCoreIfMessage);
      }
   }

   void YakinduSmVoiceCallIf::hangupCallsRequest(IN std::shared_ptr<PmCoreIfMessage_HangupCallsRequest> pmCoreIfMessage)
   {
      ETG_TRACE_USR1(("YakinduSmVoiceCallIf::hangupCallsRequest"));

      std::map<BdAddress, PmCallHandler*>::iterator it = _deviceToCallHandlerSmMap.find(pmCoreIfMessage->getBdAddress());
      if(it != _deviceToCallHandlerSmMap.end())
      {
         it->second->hangupCallsRequest(pmCoreIfMessage);
      }
   }

   void YakinduSmVoiceCallIf::redialRequest(IN std::shared_ptr<PmCoreIfMessage_RedialRequest> pmCoreIfMessage)
   {
      std::map<BdAddress, PmCallHandler*>::iterator it = _deviceToCallHandlerSmMap.find(pmCoreIfMessage->getBdAddress());
      if(it != _deviceToCallHandlerSmMap.end())
      {
         it->second->redialRequest(pmCoreIfMessage);
      }
   }

   void YakinduSmVoiceCallIf::startStopWaitingModeRequest(IN std::shared_ptr<PmCoreIfMessage_StartStopWaitingModeRequest> pmCoreIfMessage)
   {
      std::map<BdAddress, PmCallHandler*>::iterator it = _deviceToCallHandlerSmMap.find(pmCoreIfMessage->getBdAddress());
      if(it != _deviceToCallHandlerSmMap.end())
      {
         it->second->startStopWaitingModeRequest(pmCoreIfMessage);
      }
   }

   void YakinduSmVoiceCallIf::dialRequest(std::shared_ptr<PmCoreIfMessage_DialRequest> pmCoreIfMessage)
   {
      std::map<BdAddress, PmCallHandler*>::iterator it = _deviceToCallHandlerSmMap.find(pmCoreIfMessage->getBdAddress());
      if(it != _deviceToCallHandlerSmMap.end())
      {
         it->second->dialRequest(pmCoreIfMessage);
      }
   }

   void YakinduSmVoiceCallIf::acceptCallRequest(IN std::shared_ptr<PmCoreIfMessage_AcceptCallRequest> pmCoreIfMessage)
   {
      ETG_TRACE_USR4(("YakinduSmVoiceCallIf::acceptCallRequest BdAddress:%s",pmCoreIfMessage->getBdAddress().c_str()));

      std::map<BdAddress, PmCallHandler*>::iterator it = _deviceToCallHandlerSmMap.find(pmCoreIfMessage->getBdAddress());
      if(it != _deviceToCallHandlerSmMap.end())
      {
         it->second->acceptCallRequest(pmCoreIfMessage);
      }
   }

   void YakinduSmVoiceCallIf::swapCallRequest(IN std::shared_ptr<PmCoreIfMessage_SwapCallRequest> pmCoreIfMessage)
   {
      std::map<BdAddress, PmCallHandler*>::iterator it = _deviceToCallHandlerSmMap.find(pmCoreIfMessage->getBdAddress());
      if(it != _deviceToCallHandlerSmMap.end())
      {
         it->second->swapCallRequest(pmCoreIfMessage);
      }
   }

   void YakinduSmVoiceCallIf::speedDialRequest(IN std::shared_ptr<PmCoreIfMessage_SpeedDialRequest> pmCoreIfMessage)
   {
      std::map<BdAddress, PmCallHandler*>::iterator it = _deviceToCallHandlerSmMap.find(pmCoreIfMessage->getBdAddress());
      if(it != _deviceToCallHandlerSmMap.end())
      {
         it->second->speedDialRequest(pmCoreIfMessage);
      }
   }

   void YakinduSmVoiceCallIf::transferAudioRequest(IN std::shared_ptr<PmCoreIfMessage_TransferAudioRequest> pmCoreIfMessage)
   {
      std::map<BdAddress, PmCallHandler*>::iterator it = _deviceToCallHandlerSmMap.find(pmCoreIfMessage->getBdAddress());
      if(it != _deviceToCallHandlerSmMap.end())
      {
         it->second->transferAudioRequest(pmCoreIfMessage);
      }
   }

   void YakinduSmVoiceCallIf::transferAudioRequest(IN const BdAddress& deviceAddress, IN const std::string& audio)
   {
      std::map<BdAddress, PmCallHandler*>::iterator it = _deviceToCallHandlerSmMap.find(deviceAddress);
      if(it != _deviceToCallHandlerSmMap.end())
      {
         it->second->transferAudioRequest(audio);
      }
   }

   void YakinduSmVoiceCallIf::acceptSCOConnectResult(IN std::shared_ptr<PmCoreIfMessage_AcceptSCOConnectResult> pmCoreIfMessage)
   {
      std::map<BdAddress, PmCallHandler*>::iterator it = _deviceToCallHandlerSmMap.find(pmCoreIfMessage->getBdAddress());
      if(it != _deviceToCallHandlerSmMap.end())
      {
         it->second->acceptSCOConnectResult(pmCoreIfMessage);
      }
   }

   void YakinduSmVoiceCallIf::dialResult(IN std::shared_ptr<PmCoreIfMessage_DialResult> pmCoreIfMessage)
   {
      std::map<BdAddress, PmCallHandler*>::iterator it = _deviceToCallHandlerSmMap.find(pmCoreIfMessage->getBdAddress());
      if(it != _deviceToCallHandlerSmMap.end())
      {
         it->second->dialResult(pmCoreIfMessage);
      }
   }

   void YakinduSmVoiceCallIf::speedDialResult(IN std::shared_ptr<PmCoreIfMessage_SpeedDialResult> pmCoreIfMessage)
   {
      std::map<BdAddress, PmCallHandler*>::iterator it = _deviceToCallHandlerSmMap.find(pmCoreIfMessage->getBdAddress());
      if(it != _deviceToCallHandlerSmMap.end())
      {
         it->second->speedDialResult(pmCoreIfMessage);
      }
   }

   void YakinduSmVoiceCallIf::redialResult(IN std::shared_ptr<PmCoreIfMessage_RedialResult> pmCoreIfMessage)
   {
      std::map<BdAddress, PmCallHandler*>::iterator it = _deviceToCallHandlerSmMap.find(pmCoreIfMessage->getBdAddress());
      if(it != _deviceToCallHandlerSmMap.end())
      {
         it->second->redialResult(pmCoreIfMessage);
      }
   }

   void YakinduSmVoiceCallIf::swapCallResult(IN std::shared_ptr<PmCoreIfMessage_SwapCallResult> pmCoreIfMessage)
   {
      std::map<BdAddress, PmCallHandler*>::iterator it = _deviceToCallHandlerSmMap.find(pmCoreIfMessage->getBdAddress());
      if(it != _deviceToCallHandlerSmMap.end())
      {
         it->second->swapCallResult(pmCoreIfMessage);
      }
   }

   void YakinduSmVoiceCallIf::releaseAndAcceptResult(IN std::shared_ptr<PmCoreIfMessage_ReleaseAndAcceptResult> pmCoreIfMessage)
   {
      std::map<BdAddress, PmCallHandler*>::iterator it = _deviceToCallHandlerSmMap.find(pmCoreIfMessage->getBdAddress());
      if(it != _deviceToCallHandlerSmMap.end())
      {
         it->second->releaseAndAcceptResult(pmCoreIfMessage);
      }
   }

   void YakinduSmVoiceCallIf::releaseAndSwapResult(IN std::shared_ptr<PmCoreIfMessage_ReleaseAndSwapResult> pmCoreIfMessage)
   {
      std::map<BdAddress, PmCallHandler*>::iterator it = _deviceToCallHandlerSmMap.find(pmCoreIfMessage->getBdAddress());
      if(it != _deviceToCallHandlerSmMap.end())
      {
         it->second->releaseAndSwapResult(pmCoreIfMessage);
      }
   }

   void YakinduSmVoiceCallIf::holdAndAcceptResult(IN std::shared_ptr<PmCoreIfMessage_HoldAndAcceptResult> pmCoreIfMessage)
   {
      std::map<BdAddress, PmCallHandler*>::iterator it = _deviceToCallHandlerSmMap.find(pmCoreIfMessage->getBdAddress());
      if(it != _deviceToCallHandlerSmMap.end())
      {
         it->second->holdAndAcceptResult(pmCoreIfMessage);
      }
   }

   void YakinduSmVoiceCallIf::hangupAllResult(IN std::shared_ptr<PmCoreIfMessage_HangupAllResult> pmCoreIfMessage)
   {
      std::map<BdAddress, PmCallHandler*>::iterator it = _deviceToCallHandlerSmMap.find(pmCoreIfMessage->getBdAddress());
      if(it != _deviceToCallHandlerSmMap.end())
      {
         it->second->hangupAllResult(pmCoreIfMessage);
      }
   }

   void YakinduSmVoiceCallIf::splitCallsResult(IN std::shared_ptr<PmCoreIfMessage_SplitCallsResult> pmCoreIfMessage)
   {
      std::map<BdAddress, PmCallHandler*>::iterator it = _deviceToCallHandlerSmMap.find(pmCoreIfMessage->getBdAddress());
      if(it != _deviceToCallHandlerSmMap.end())
      {
         it->second->splitCallsResult(pmCoreIfMessage);
      }
   }

   void YakinduSmVoiceCallIf::mergeCallsResult(IN std::shared_ptr<PmCoreIfMessage_MergeCallsResult> pmCoreIfMessage)
   {
      std::map<BdAddress, PmCallHandler*>::iterator it = _deviceToCallHandlerSmMap.find(pmCoreIfMessage->getBdAddress());
      if(it != _deviceToCallHandlerSmMap.end())
      {
         it->second->mergeCallsResult(pmCoreIfMessage);
      }
   }

   void YakinduSmVoiceCallIf::hangupMultipartyResult(IN std::shared_ptr<PmCoreIfMessage_HangupMultipartyResult> pmCoreIfMessage)
   {
      std::map<BdAddress, PmCallHandler*>::iterator it = _deviceToCallHandlerSmMap.find(pmCoreIfMessage->getBdAddress());
      if(it != _deviceToCallHandlerSmMap.end())
      {
         it->second->hangupMultipartyResult(pmCoreIfMessage);
      }
   }

   void YakinduSmVoiceCallIf::hangupCallResult(IN std::shared_ptr<PmCoreIfMessage_HangupCallResult> pmCoreIfMessage)
   {
      std::map<BdAddress, PmCallHandler*>::iterator it = _deviceToCallHandlerSmMap.find(pmCoreIfMessage->getBdAddress());
      if(it != _deviceToCallHandlerSmMap.end())
      {
         it->second->hangupCallResult(pmCoreIfMessage);
      }
   }

   void YakinduSmVoiceCallIf::acceptCallResult(IN std::shared_ptr<PmCoreIfMessage_AcceptCallResult> pmCoreIfMessage)
   {
      std::map<BdAddress, PmCallHandler*>::iterator it = _deviceToCallHandlerSmMap.find(pmCoreIfMessage->getBdAddress());
      if(it != _deviceToCallHandlerSmMap.end())
      {
         it->second->acceptCallResult(pmCoreIfMessage);
      }
   }

   void YakinduSmVoiceCallIf::holdIncomingCallResult(IN std::shared_ptr<PmCoreIfMessage_HoldIncomingCallResult> pmCoreIfMessage)
   {
      std::map<BdAddress, PmCallHandler*>::iterator it = _deviceToCallHandlerSmMap.find(pmCoreIfMessage->getBdAddress());
      if(it != _deviceToCallHandlerSmMap.end())
      {
         it->second->holdIncomingCallResult(pmCoreIfMessage);
      }
   }

   void YakinduSmVoiceCallIf::transferAudioResult(IN std::shared_ptr<PmCoreIfMessage_TransferAudioResult> pmCoreIfMessage)
   {
      std::map<BdAddress, PmCallHandler*>::iterator it = _deviceToCallHandlerSmMap.find(pmCoreIfMessage->getBdAddress());
      if(it != _deviceToCallHandlerSmMap.end())
      {
         it->second->transferAudioResult(pmCoreIfMessage);
      }
   }

}
