/**
 * @file PmCallHandler.cpp
 *
 * @swcomponent PhoneCallManager
 *
 * @brief This file contains the definition of the PmCallHandler class
 *
 * @copyright (C) 2016 Robert Bosch GmbH.
 *            The reproduction, distribution and utilization of this file as
 *            well as the communication of its contents to others without express
 *            authorization is prohibited. Offenders will be held liable for the
 *            payment of damages. All rights reserved in the event of the grant
 *            of a patent, utility model or design.
 *
 * @details This file provides the state machine interfaces of voice call
 *
 * @ingroup PmCore
 */

#include "PmCallHandler.h"
#include "PropertyUpdateNotifierToCore.h"
#include "PmCoreIfMessageCreator.h"
#include "PropertyDetails.h"
#include "PmAudioManagerWrapper.h"
#include "PmCoreMainController.h"
#include "DeviceInfoHandler.h"
#include "PmConfiguration.h"
#include "PmAppTrace.h"

#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#define ETG_DEFAULT_TRACE_CLASS TR_CLASS_PM_CORE
#ifdef VARIANT_S_FTR_ENABLE_FW_ETG_USAGE
#include "trcGenProj/Header/PmCallHandler.cpp.trc.h"
#define ETG_DEFAULT_TRACE_CLASS TR_CLASS_PM_CORE
#endif
#endif

namespace pmcore
{
   PmCallHandler::PmCallHandler()
   {
      _deviceAddress = "";
      _smUserActionIf = nullptr;
      _smAgCallStateUpdateIf = nullptr;
      _smSmHandlerIf = nullptr;
      _smAudioManagerRequestIf = nullptr;
      _smAgResponseIf = nullptr;
      _smTimerUpdateIf = nullptr;
      _smTraceObserverIf = nullptr;
      _autoRejectCall = AutoRejectCall();
   }

   PmCallHandler::PmCallHandler(const BdAddress& deviceAddress)
   {
      _deviceAddress = deviceAddress;

      _smUserActionIf = new PmCallHandler::SmUserActionIf(*this);

      if(nullptr != _smUserActionIf)
      {
         setSCI_UserAction_OCB(_smUserActionIf);
      }

      _smAgCallStateUpdateIf = new PmCallHandler::SmAgCallStateUpdateIf(*this);

      if(nullptr != _smAgCallStateUpdateIf)
      {
         setSCI_AgCallStateUpdate_OCB(_smAgCallStateUpdateIf);
      }

      _smSmHandlerIf = new PmCallHandler::SmSmHandlerIf(*this);

      if(nullptr != _smSmHandlerIf)
      {
         setSCI_SmHandler_OCB(_smSmHandlerIf);
      }

      _smAudioManagerRequestIf = new PmCallHandler::SmAudioManagerRequestIf(*this);

      if(nullptr != _smAudioManagerRequestIf)
      {
         setSCI_AudioManagerRequest_OCB(_smAudioManagerRequestIf);
      }

      _smAgResponseIf = new PmCallHandler::SmAgResponseIf(*this);

      if(nullptr != _smAgResponseIf)
      {
         setSCI_AgResponse_OCB(_smAgResponseIf);
      }

      _smTimerUpdateIf = new PmCallHandler::SmTimerUpdateIf(*this);

      if(nullptr != _smTimerUpdateIf)
      {
         setSCI_TimerUpdate_OCB(_smTimerUpdateIf);
      }

      _smTraceObserverIf = new PmCallHandler::SmTraceObserverIf<PmCallStateHandlerSmStates>(*this);

      if(nullptr != _smTraceObserverIf)
      {
         setTraceObserver(_smTraceObserverIf);
      }

      _autoRejectCall = AutoRejectCall();

      //initialize the state machine
      init();

      enter();
   }

   PmCallHandler::~PmCallHandler()
   {
      if(nullptr != _smUserActionIf)
      {
         delete _smUserActionIf;
         _smUserActionIf = nullptr;
      }

      if(nullptr != _smAgCallStateUpdateIf)
      {
         delete _smAgCallStateUpdateIf;
         _smAgCallStateUpdateIf = nullptr;
      }

      if(nullptr != _smSmHandlerIf)
      {
         delete _smSmHandlerIf;
         _smSmHandlerIf = nullptr;
      }

      if(nullptr != _smAudioManagerRequestIf)
      {
         delete _smAudioManagerRequestIf;
         _smAudioManagerRequestIf = nullptr;
      }

      if(nullptr != _smAgResponseIf)
      {
         delete _smAgResponseIf;
         _smAgResponseIf = nullptr;
      }

      if(nullptr != _smTimerUpdateIf)
      {
         delete _smTimerUpdateIf;
         _smTimerUpdateIf = nullptr;
      }

      if(nullptr != _smTraceObserverIf)
      {
         delete _smTraceObserverIf;
         _smTraceObserverIf = nullptr;
      }
   }

   void PmCallHandler::setCallCount(IN const CallCount callCount)
   {
      ETG_TRACE_USR4(("setCallCount::set_callCount : %d", callCount));

      getSCI_AgCallStateUpdate()->set_callCount(callCount);
   }

   void PmCallHandler::onAgCallStateUpdate(IN const CallStateEnumType callState)
   {
      ETG_TRACE_USR4(("PmCallHandler::onAgCallStateUpdate : %u", ETG_CENUM(CallStateEnumType, callState)));

      switch(callState)
      {
         case CALL_STATE_IDLE:
            getSCI_AgCallStateUpdate()->raise_iDLE();
            break;
         case CALL_STATE_INCOMING:
            getSCI_AgCallStateUpdate()->raise_iNCOMING();
            break;
         case CALL_STATE_INCOMING_INBAND:
            getSCI_AgCallStateUpdate()->raise_iNCOMING_INBAND();
            break;
         case CALL_STATE_DIALING:
            getSCI_AgCallStateUpdate()->raise_dIALING();
            break;
         case CALL_STATE_HELD:
            getSCI_AgCallStateUpdate()->raise_hELD();
            break;
         case CALL_STATE_ACTIVE:
            getSCI_AgCallStateUpdate()->raise_aCTIVE();
            break;
         case CALL_STATE_DISCONNECTING:
            getSCI_AgCallStateUpdate()->raise_cALL_DISCONNECTING();
            break;
         case CALL_STATE_DISCONNECTED:
            getSCI_AgCallStateUpdate()->raise_cALL_DISCONNECTED();
            break;
         case CALL_STATE_CALL_X_DIALING:
            getSCI_AgCallStateUpdate()->raise_cALL_X_DIALING();
            break;
         case CALL_STATE_CALL_X_WAITING:
            getSCI_AgCallStateUpdate()->raise_cALL_X_WAITING();
            break;
         case CALL_STATE_CALLS_SWAPPABLE:
            getSCI_AgCallStateUpdate()->raise_cALLS_SWAPPABLE();
            break;
         case CALL_STATE_CALLS_CONFERENCE:
            getSCI_AgCallStateUpdate()->raise_cALLS_CONFERENCE();
            break;
         case CALL_STATE_ALL_CALLS_DISCONNECTED:
            getSCI_AgCallStateUpdate()->raise_aLL_CALLS_DISCONNECTED();
            break;
         default:
            getSCI_AgCallStateUpdate()->raise_iDLE();
            break;
      }
   }

   void PmCallHandler::onAudioManagerEventUpdate(IN const AudioManagerEventType amEvent)
   {
      ETG_TRACE_USR4(("PmCallHandler::onAudioManagerEventUpdate : %u", ETG_CENUM(AudioManagerEventType, amEvent)));

      switch(amEvent)
      {
         case PHONE_CHANNEL_GRANTED:
            getSCI_AudioManagerRequest()->set_phoneAudioState(PHONE_CHANNEL_ALLOCATED);
            getSCI_AudioManagerResponse()->raise_pHONE_CHANNEL_GRANTED();
            break;
         case PHONE_CHANNEL_DENIED:
            getSCI_AudioManagerRequest()->set_phoneAudioState(PHONE_CHANNEL_NOT_ALLOCATED);
            getSCI_AudioManagerResponse()->raise_pHONE_CHANNEL_DENIED();
            break;
         case RINGTONE_CHANNEL_GRANTED:
            getSCI_AudioManagerRequest()->set_phoneAudioState(RINGTONE_CHANNEL_ALLOCATED);
            getSCI_AudioManagerResponse()->raise_rINGTONE_CHANNEL_GRANTED();
            break;
         case RINGTONE_CHANNEL_DENIED:
            getSCI_AudioManagerRequest()->set_phoneAudioState(RINGTONE_CHANNEL_NOT_ALLOCATED);
            getSCI_AudioManagerResponse()->raise_rINGTONE_CHANNEL_DENIED();
            break;
         case WAITINGMODE_CHANNEL_GRANTED:
            getSCI_AudioManagerRequest()->set_phoneAudioState(WAITINGMODE_CHANNEL_ALLOCATED);
            getSCI_AudioManagerResponse()->raise_wAITINGMODE_CHANNEL_GRANTED();
            break;
         case WAITINGMODE_CHANNEL_DENIED:
            getSCI_AudioManagerRequest()->set_phoneAudioState(WAITINGMODE_CHANNEL_NOT_ALLOCATED);
            getSCI_AudioManagerResponse()->raise_wAITINGMODE_CHANNEL_DENIED();
            break;
         case PLAY_AUDIO_SUCCESS:
            getSCI_AudioManagerRequest()->set_phoneAudioState(PHONE_START_STREAMING_SUCCESS);
            getSCI_AudioManagerResponse()->raise_pLAY_AUDIO_SUCCESS();
            break;
         case PLAY_AUDIO_FAILURE:
            getSCI_AudioManagerResponse()->raise_pLAY_AUDIO_FAILURE();
            break;
         case PAUSE_AUDIO_SUCCESS:
            getSCI_AudioManagerRequest()->set_phoneAudioState(PHONE_PAUSE_STREAMING_SUCCESS);
            getSCI_AudioManagerResponse()->raise_pAUSE_AUDIO_SUCCESS();
            break;
         case PAUSE_AUDIO_FAILURE:
            getSCI_AudioManagerResponse()->raise_pAUSE_AUDIO_FAILURE();
            break;
         case STOP_AUDIO_SUCCESS:
            getSCI_AudioManagerRequest()->set_phoneAudioState(PHONE_STOP_STREAMING_SUCCESS);
            getSCI_AudioManagerResponse()->raise_sTOP_AUDIO_SUCCESS();
            break;
         case STOP_AUDIO_FAILURE:
            getSCI_AudioManagerResponse()->raise_sTOP_AUDIO_FAILURE();
            break;
         case PHONE_CHANNEL_RELEASED:
            getSCI_AudioManagerRequest()->set_phoneAudioState(PHONE_AUDIO_CHANNEL_IDLE);
            getSCI_AudioManagerResponse()->raise_pHONE_CHANNEL_RELEASED();
            break;
         case RINGTONE_CHANNEL_RELEASED:
            getSCI_AudioManagerRequest()->set_phoneAudioState(PHONE_AUDIO_CHANNEL_IDLE);
            getSCI_AudioManagerResponse()->raise_rINGTONE_CHANNEL_RELEASED();
            break;
         case WAITINGMODE_CHANNEL_RELEASED:
            getSCI_AudioManagerRequest()->set_phoneAudioState(PHONE_AUDIO_CHANNEL_IDLE);
            getSCI_AudioManagerResponse()->raise_wAITINGMODE_CHANNEL_RELEASED();
            break;
         default:
            ETG_TRACE_ERR(("onAudioManagerEventUpdate Invalid Event"));
            break;
      }
   }

   void PmCallHandler::onScoConnectionStatusChanged(IN const SCOStatus scoStatus)
   {
      ETG_TRACE_USR4(("PmCallHandler::onScoConnectionStatusChanged : %u",
            ETG_CENUM(SCOStatusEnumType, scoStatus)));

      // The below if condition avoids duplication of event posting to the State chart.

      if (scoStatus != getSCI_AgAudioUpdate()->get_scoStatus())
      {
         // Sometimes on device reconnection with an active call, call status is first received and
         // then SCO status is received as "SCO_NOT_ESTABLISHED" (in call controller) due to GetProperties
         // posted to the stack during device connection before receiving actual SCO connection.
         // To block this, the below condition is added.
         if (!((SCO_PAUSED == scoStatus) && (SCO_NOT_ESTABLISHED == getSCI_AgAudioUpdate()->get_scoStatus())))
         {
            getSCI_AgAudioUpdate()->set_scoStatus(scoStatus);

            if(SCO_NOT_ESTABLISHED == scoStatus)
            {
               getSCI_AgAudioUpdate()->raise_sCO_DISCONNECTED(SCO_NOT_ESTABLISHED);
            }
            else if(SCO_PAUSED == scoStatus)
            {
               getSCI_AgAudioUpdate()->raise_sCO_PAUSED(SCO_PAUSED);
            }
            else if(SCO_NARROWBAND == scoStatus)
            {
               getSCI_AgAudioUpdate()->raise_sCO_CONNECTED(SCO_NARROWBAND);
            }
            else if(SCO_WIDEBAND == scoStatus)
            {
               getSCI_AgAudioUpdate()->raise_sCO_CONNECTED(SCO_WIDEBAND);
            }
            else
            {
               //error, something wrong
            }
         }
      }
   }

   void PmCallHandler::onTimerEventUpdate()
   {
      ETG_TRACE_USR4(("PmCallHandler::onTimerEventUpdate() entered"));

      getSCI_TimerUpdate()->raise_tIMER_CALLBACK();
   }

   void PmCallHandler::mergeCallsRequest(IN std::shared_ptr<PmCoreIfMessage_MergeCallsRequest> pmCoreIfMessage)
   {
      getSCI_UserAction()->raise_cONFERENCE_CALL(pmCoreIfMessage.get());
   }

   void PmCallHandler::splitCallsRequest(IN std::shared_ptr<PmCoreIfMessage_SplitCallsRequest> pmCoreIfMessage)
   {
      getSCI_UserAction()->raise_sPLIT_CALLS(pmCoreIfMessage.get());
   }

   void PmCallHandler::hangupCallsRequest(IN std::shared_ptr<PmCoreIfMessage_HangupCallsRequest> pmCoreIfMessage)
   {
      //Handling of Hangup all calls (CallInstance - 0xFF)
      CallInstanceList callInstanceList = pmCoreIfMessage->getCallInstanceList();

      auto it = callInstanceList.begin();

      if(it != callInstanceList.end())
      {
         _cancelOutgoingCallRequest.setBdAddress(pmCoreIfMessage->getBdAddress());
         _cancelOutgoingCallRequest.setCallInstanceList(callInstanceList);
         _cancelOutgoingCallRequest.setAct(pmCoreIfMessage->getAct());

         if(CALL_INSTANCE_DEFAULT == *it)
         {
            getSCI_UserAction()->raise_hANGUP_ALL_CALLS(pmCoreIfMessage.get());
         }
         else
         {
            getSCI_UserAction()->raise_hANGUP_CALLS(pmCoreIfMessage.get());
         }

         _actAndCallInstanceListMap.insert(std::pair<ActType,CallInstanceList>(pmCoreIfMessage->getAct(),
               pmCoreIfMessage->getCallInstanceList()));
      }
   }

   void PmCallHandler::redialRequest(IN std::shared_ptr<PmCoreIfMessage_RedialRequest> pmCoreIfMessage)
   {
      getSCI_UserAction()->raise_rEDIAL(pmCoreIfMessage.get());
   }

   void PmCallHandler::startStopWaitingModeRequest(
         IN std::shared_ptr<PmCoreIfMessage_StartStopWaitingModeRequest> pmCoreIfMessage)
   {
      PmResult pmResult(PM_RESULT_OK, "");

      if(pmCoreIfMessage->getStartStopWaitingMode())
      {
         getSCI_UserAction()->raise_sTART_WAITING_MODE();
      }
      else
      {
         getSCI_UserAction()->raise_sTOP_WAITING_MODE();
      }

      pmCoreIfMessage->setPmResult(pmResult);
   }

   void PmCallHandler::dialRequest(std::shared_ptr<PmCoreIfMessage_DialRequest> pmCoreIfMessage)
   {
      getSCI_UserAction()->raise_dIAL_CALL(pmCoreIfMessage.get());
   }

   void PmCallHandler::acceptCallRequest(std::shared_ptr<PmCoreIfMessage_AcceptCallRequest> pmCoreIfMessage)
   {
      ETG_TRACE_USR4(("PmCallHandler::acceptCallRequest() entered"));

      bool setAcceptCallRequest = true;

      if((_acceptCallRequest.getBdAddress() == pmCoreIfMessage->getBdAddress()) &&
            (_acceptCallRequest.getCallInstance() == pmCoreIfMessage->getCallInstance()))
      {
         if((_acceptCallRequest.getAct() == pmCoreIfMessage->getAct()))
         {
            ETG_TRACE_USR4(("Accept Request received again from same client with the same command counter"));
         }
         else
         {
            ETG_TRACE_USR4(("Accept Request received again with different command counter or from another client"));
         }

         setAcceptCallRequest = false;
      }


      if(true == setAcceptCallRequest)
      {
         PmCoreIfMessage_AcceptCallRequest msg(pmCoreIfMessage->getBdAddress(), pmCoreIfMessage->getCallInstance(),
               pmCoreIfMessage->getAcceptOperation(), pmCoreIfMessage->getAct(), pmCoreIfMessage->getOrigin());

         _acceptCallRequest = msg;

         CallInstanceList callInstanceList;
         callInstanceList.push_back(pmCoreIfMessage->getCallInstance());
         _actAndCallInstanceListMap.emplace_hint(_actAndCallInstanceListMap.end(),
               pmCoreIfMessage->getAct(),callInstanceList);

         AcceptOperation acceptOperation = pmCoreIfMessage->getAcceptOperation();

         if ((ACCEPT_CALLS_BY_ACQUIRING_CHANNEL == acceptOperation) ||
               (RELEASE_AND_ACCEPT_BY_ACQUIRING_CHANNEL == acceptOperation) ||
               (HOLD_AND_ACCEPT_BY_ACQUIRING_CHANNEL == acceptOperation))
         {
            getSCI_UserAction()->raise_aCCEPT_CALL_BY_ACQUIRING_CH(pmCoreIfMessage.get());
         }
         else
         {
            getSCI_UserAction()->raise_aCCEPT_CALL(pmCoreIfMessage.get());
         }
      }

      //Send success response to requested client
      PmResult pmResult(PM_RESULT_OK, "");
      pmCoreIfMessage->setPmResult(pmResult);
   }

   void PmCallHandler::swapCallRequest(IN std::shared_ptr<PmCoreIfMessage_SwapCallRequest> pmCoreIfMessage)
   {
      ETG_TRACE_USR4(("PmCallHandler::swapCallRequest Entered"));
      if (CALL_INSTANCE_DEFAULT != _autoRejectCall._instance)
      {
         // Scenario: Auto reject for an incoming call is in progress during 3 way call scenario and swap call
         // request is received from client (HMI) and assume the request is posted to BT stack.
         // Issue: At this time, the BT stack misbehaves and ends the calls in 3 way call scenario and accepts
         // the call for which Hangup was posted.
         // Solution: Hence swap call is blocked during this scenario
         ETG_TRACE_USR4(("Auto reject call is in progress. Hence this swap call is not processed"));

         // TODO: Need to update error for this SWAP call request
      }
      else
      {
         getSCI_UserAction()->raise_sWAP_CALLS(pmCoreIfMessage.get());
         CallInstanceList callInstanceList; //Dummy CallInstance List
         _actAndCallInstanceListMap.insert(std::pair<ActType,CallInstanceList>(pmCoreIfMessage->getAct(),callInstanceList));
      }
   }

   void PmCallHandler::speedDialRequest(IN std::shared_ptr<PmCoreIfMessage_SpeedDialRequest> pmCoreIfMessage)
   {
      getSCI_UserAction()->raise_sPEED_DIAL(pmCoreIfMessage.get());
   }

   void PmCallHandler::transferAudioRequest(IN std::shared_ptr<PmCoreIfMessage_TransferAudioRequest> pmCoreIfMessage)
   {
      if(AUDIO_DIRECTION_AUDIOGATEWAY == pmCoreIfMessage->getAudioDirection())
      {
         getSCI_UserAction()->raise_tRANSFER_AUDIO_TO_AG(pmCoreIfMessage.get());
      }
      else
      {
         if(AUDIO_DIRECTION_HANDSFREE == pmCoreIfMessage->getAudioDirection())
         {
            getSCI_UserAction()->raise_tRANSFER_AUDIO_TO_HF(pmCoreIfMessage.get());
         }
      }
   }

   void PmCallHandler::transferAudioRequest(IN const std::string& audio)
   {
      ETG_TRACE_USR4(("PmCallHandler::transferAudioRequest"));

      std::string audioType(audio);

      BTSResult btsResult = evobtstackwrapper::EvoBtStackWrapper::getInstance().sendSetPropertyRequest(
            ::ccdbusif::evolution::IF_HANDSFREE, getDeviceAddress(), "Audio", audioType);

      if (BTS_REQ_FAILED == btsResult._btsRequestResult)
      {
         ETG_TRACE_USR4(("PmCallHandler::transferAudioRequest: Transfer Audio Failure"));
      }
   }

   void PmCallHandler::switchToPassiveRequest()
   {
      getSCI_SmHandler()->raise_sWITCH_TO_PASSIVE();
   }

   void PmCallHandler::switchToActiveRequest()
   {
      ETG_TRACE_USR4(("PmCallHandler::switchToActiveRequest"));
      getSCI_SmHandler()->raise_sWITCH_TO_ACTIVE();
   }

   void PmCallHandler::autoRejectCallRequest(const AutoRejectCall autoRejectCall)
   {
      ETG_TRACE_USR4(("PmCallHandler::autoRejectCallRequest"));

      // Even if posting "sendHangupRequest" is failed, the below member variable is set.
      // So that if Swap call is called when Hangup call posting has failed, that will result in an unexpected behaviour.
      // Check comments in SwapCallRequest for a detailed explaination on the same.
      _autoRejectCall = autoRejectCall;

      //TODO : logic needs to be checked if hangup request is failed
      BTSResult btsResult = evobtstackwrapper::EvoBtStackWrapper::getInstance().sendHangupRequest(_deviceAddress,
            autoRejectCall._instance, PM_DEFAULT_ACT);

      if(BTS_REQ_FAILED == btsResult._btsRequestResult)
      {
         ETG_TRACE_USR4(("Posting sendHangupRequest Failed"));
      }
   }

   void PmCallHandler::onCallRemoved(IN const CallInstance callInstance)
   {
      ETG_TRACE_USR4(("PmCallHandler::onCallRemoved"));

      if (callInstance == _autoRejectCall._instance)
      {
         NotificationEvent notificationEvent;
         notificationEvent._bdAddress = getDeviceAddress();
         notificationEvent._eventName = INCOMING_CALL_REJECTED;

         if (WAITING_MODE_IN_PROGRESS == _autoRejectCall._autoRejectRequestReason)
         {
            notificationEvent._reason = "Waiting mode state is Active";
         }
         else if (THIRD_INCOMING_CALL == _autoRejectCall._autoRejectRequestReason)
         {
            notificationEvent._reason = "Three-way call is active";
         }

         notificationEvent._telephoneNumber = _autoRejectCall._telephoneNumber;

         PmCoreMainController::getInstance().getPmCoreCallbackIfWrapper().doOnNotificationEvent(notificationEvent);

         _autoRejectCall = AutoRejectCall();
      }
   }

   void PmCallHandler::acceptSCOConnectResult(
         IN std::shared_ptr<PmCoreIfMessage_AcceptSCOConnectResult> pmCoreIfMessage)
   {
      (void)(pmCoreIfMessage);
   }

   void PmCallHandler::dialResult(std::shared_ptr<PmCoreIfMessage_DialResult> pmCoreIfMessage)
   {
      ETG_TRACE_USR4(("PmCallHandler::%20s::dialResult::%d", getDeviceAddress().c_str(),
            ETG_CENUM(BTSRequestResult, pmCoreIfMessage->getBTSResult()._btsRequestResult)));

      if(BTS_REQ_SUCCESS == pmCoreIfMessage->getBTSResult()._btsRequestResult)
      {
         getSCI_AgResponse()->raise_dIAL_OK();
      }
      else if(BTS_REQ_FAILED == pmCoreIfMessage->getBTSResult()._btsRequestResult)
      {
         getSCI_AgResponse()->raise_dIAL_ERROR(pmCoreIfMessage.get());
      }
   }

   void PmCallHandler::speedDialResult(IN std::shared_ptr<PmCoreIfMessage_SpeedDialResult> pmCoreIfMessage)
   {
      ETG_TRACE_USR4(("PmCallHandler::%20s::speedDialResult::%d", getDeviceAddress().c_str(),
            ETG_CENUM(BTSRequestResult, pmCoreIfMessage->getBTSResult()._btsRequestResult)));

      if(BTS_REQ_SUCCESS == pmCoreIfMessage->getBTSResult()._btsRequestResult)
      {
         getSCI_AgResponse()->raise_sPEED_DIAL_OK();
      }
      else if(BTS_REQ_FAILED == pmCoreIfMessage->getBTSResult()._btsRequestResult)
      {
         getSCI_AgResponse()->raise_sPEED_DIAL_ERROR(pmCoreIfMessage.get());
      }
   }

   void PmCallHandler::redialResult(IN std::shared_ptr<PmCoreIfMessage_RedialResult> pmCoreIfMessage)
   {
      ETG_TRACE_USR4(("PmCallHandler::%20s::redialResult::%d", getDeviceAddress().c_str(),
            ETG_CENUM(BTSRequestResult, pmCoreIfMessage->getBTSResult()._btsRequestResult)));

      if(BTS_REQ_SUCCESS == pmCoreIfMessage->getBTSResult()._btsRequestResult)
      {
         getSCI_AgResponse()->raise_rEDIAL_OK();
      }
      else if(BTS_REQ_FAILED == pmCoreIfMessage->getBTSResult()._btsRequestResult)
      {
         getSCI_AgResponse()->raise_rEDIAL_ERROR(pmCoreIfMessage.get());
      }
   }

   void PmCallHandler::swapCallResult(IN std::shared_ptr<PmCoreIfMessage_SwapCallResult> pmCoreIfMessage)
   {

      ETG_TRACE_USR4(("PmCallHandler::%20s::swapCallResult::%d", getDeviceAddress().c_str(),
            ETG_CENUM(BTSRequestResult, pmCoreIfMessage->getBTSResult()._btsRequestResult)));

      if(BTS_REQ_SUCCESS == pmCoreIfMessage->getBTSResult()._btsRequestResult)
      {
         getSCI_AgResponse()->raise_sWAP_OK();
         auto it = _actAndCallInstanceListMap.find(pmCoreIfMessage->getAct());
         if (_actAndCallInstanceListMap.end() != it)
            _actAndCallInstanceListMap.erase(it);
      }
      else if(BTS_REQ_FAILED == pmCoreIfMessage->getBTSResult()._btsRequestResult)
      {
         processErrorResponseFromStack(pmCoreIfMessage.get(), SWAP_CALL_ERROR);
      }
   }

   void PmCallHandler::releaseAndAcceptResult(
         IN std::shared_ptr<PmCoreIfMessage_ReleaseAndAcceptResult> pmCoreIfMessage)
   {
      ETG_TRACE_USR4(("PmCallHandler::%20s::releaseAndAcceptResult::%d", getDeviceAddress().c_str(),
            ETG_CENUM(BTSRequestResult, pmCoreIfMessage->getBTSResult()._btsRequestResult)));

      if(BTS_REQ_SUCCESS == pmCoreIfMessage->getBTSResult()._btsRequestResult)
      {
         getSCI_AgResponse()->raise_rELEASE_AND_ACCEPT_OK();
         auto it = _actAndCallInstanceListMap.find(pmCoreIfMessage->getAct());
         if (_actAndCallInstanceListMap.end() != it)
            _actAndCallInstanceListMap.erase(it);
      }
      else if(BTS_REQ_FAILED == pmCoreIfMessage->getBTSResult()._btsRequestResult)
      {
         getSCI_AgResponse()->raise_rELEASE_AND_ACCEPT_ERROR(pmCoreIfMessage.get());
      }
   }

   void PmCallHandler::releaseAndSwapResult(IN std::shared_ptr<PmCoreIfMessage_ReleaseAndSwapResult> pmCoreIfMessage)
   {
      ETG_TRACE_USR4(("PmCallHandler::%20s::releaseAndSwapResult::%d", getDeviceAddress().c_str(),
            ETG_CENUM(BTSRequestResult, pmCoreIfMessage->getBTSResult()._btsRequestResult)));

      if(BTS_REQ_SUCCESS == pmCoreIfMessage->getBTSResult()._btsRequestResult)
      {
         getSCI_AgResponse()->raise_rELEASE_AND_ACCEPT_OK();
         auto it = _actAndCallInstanceListMap.find(pmCoreIfMessage->getAct());
         if (_actAndCallInstanceListMap.end() != it)
            _actAndCallInstanceListMap.erase(it);
      }
      else if(BTS_REQ_FAILED == pmCoreIfMessage->getBTSResult()._btsRequestResult)
      {
         getSCI_AgResponse()->raise_rELEASE_AND_ACCEPT_ERROR(pmCoreIfMessage.get());
      }
   }

   void PmCallHandler::holdAndAcceptResult(IN std::shared_ptr<PmCoreIfMessage_HoldAndAcceptResult> pmCoreIfMessage)
   {
      ETG_TRACE_USR4(("PmCallHandler::%20s::holdAndAcceptResult::%d", getDeviceAddress().c_str(),
            ETG_CENUM(BTSRequestResult, pmCoreIfMessage->getBTSResult()._btsRequestResult)));

      if(BTS_REQ_SUCCESS == pmCoreIfMessage->getBTSResult()._btsRequestResult)
      {
         getSCI_AgResponse()->raise_hOLD_AND_ACCEPT_OK();
         auto it = _actAndCallInstanceListMap.find(pmCoreIfMessage->getAct());
         if (_actAndCallInstanceListMap.end() != it)
            _actAndCallInstanceListMap.erase(it);
      }
      else if(BTS_REQ_FAILED == pmCoreIfMessage->getBTSResult()._btsRequestResult)
      {
         getSCI_AgResponse()->raise_hOLD_AND_ACCEPT_ERROR(pmCoreIfMessage.get());
      }
   }

   void PmCallHandler::hangupAllResult(IN std::shared_ptr<PmCoreIfMessage_HangupAllResult> pmCoreIfMessage)
   {
      ETG_TRACE_USR4(("PmCallHandler::%20s::hangupAllResult::%d", getDeviceAddress().c_str(),
            ETG_CENUM(BTSRequestResult, pmCoreIfMessage->getBTSResult()._btsRequestResult)));

      if(BTS_REQ_SUCCESS == pmCoreIfMessage->getBTSResult()._btsRequestResult)
      {
         getSCI_AgResponse()->raise_hANGUP_ALL_OK();
         auto it = _actAndCallInstanceListMap.find(pmCoreIfMessage->getAct());
         if (_actAndCallInstanceListMap.end() != it)
            _actAndCallInstanceListMap.erase(it);
      }
      else if(BTS_REQ_FAILED == pmCoreIfMessage->getBTSResult()._btsRequestResult)
      {
         processErrorResponseFromStack(pmCoreIfMessage.get(), HANGUP_ALL_CALL_ERROR);
      }
   }

   void PmCallHandler::splitCallsResult(IN std::shared_ptr<PmCoreIfMessage_SplitCallsResult> pmCoreIfMessage)
   {
      ETG_TRACE_USR4(("PmCallHandler::%20s::splitCallsResult::%d", getDeviceAddress().c_str(),
            ETG_CENUM(BTSRequestResult, pmCoreIfMessage->getBTSResult()._btsRequestResult)));

      if(BTS_REQ_SUCCESS == pmCoreIfMessage->getBTSResult()._btsRequestResult)
      {
         getSCI_AgResponse()->raise_sPLIT_OK();
         auto it = _actAndCallInstanceListMap.find(pmCoreIfMessage->getAct());
         if (_actAndCallInstanceListMap.end() != it)
            _actAndCallInstanceListMap.erase(it);
      }
      else if(BTS_REQ_FAILED == pmCoreIfMessage->getBTSResult()._btsRequestResult)
      {
         processErrorResponseFromStack(pmCoreIfMessage.get(), SPLIT_CALL_ERROR);
      }
   }

   void PmCallHandler::mergeCallsResult(IN std::shared_ptr<PmCoreIfMessage_MergeCallsResult> pmCoreIfMessage)
   {
      ETG_TRACE_USR4(("PmCallHandler::%20s::mergeCallsResult::%d", getDeviceAddress().c_str(),
            ETG_CENUM(BTSRequestResult, pmCoreIfMessage->getBTSResult()._btsRequestResult)));

      if(BTS_REQ_SUCCESS == pmCoreIfMessage->getBTSResult()._btsRequestResult)
      {
         getSCI_AgResponse()->raise_cONF_OK();
         auto it = _actAndCallInstanceListMap.find(pmCoreIfMessage->getAct());
         if (_actAndCallInstanceListMap.end() != it)
            _actAndCallInstanceListMap.erase(it);
      }
      else if(BTS_REQ_FAILED == pmCoreIfMessage->getBTSResult()._btsRequestResult)
      {
         processErrorResponseFromStack(pmCoreIfMessage.get(), CONFERENCE_CALL_ERROR);
      }
   }

   void PmCallHandler::hangupMultipartyResult(
         IN std::shared_ptr<PmCoreIfMessage_HangupMultipartyResult> pmCoreIfMessage)
   {
      ETG_TRACE_USR4(("PmCallHandler::%20s::hangupMultipartyResult::%d", getDeviceAddress().c_str(),
            ETG_CENUM(BTSRequestResult, pmCoreIfMessage->getBTSResult()._btsRequestResult)));

      if(BTS_REQ_SUCCESS == pmCoreIfMessage->getBTSResult()._btsRequestResult)
      {
         getSCI_AgResponse()->raise_hANGUP_MULTIPARTY_OK();
         auto it = _actAndCallInstanceListMap.find(pmCoreIfMessage->getAct());
         if (_actAndCallInstanceListMap.end() != it)
            _actAndCallInstanceListMap.erase(it);
      }
      else if(BTS_REQ_FAILED == pmCoreIfMessage->getBTSResult()._btsRequestResult)
      {
         processErrorResponseFromStack(pmCoreIfMessage.get(), HANGUP_MULTIPARTY_ERROR);
      }
   }

   void PmCallHandler::hangupCallResult(std::shared_ptr<PmCoreIfMessage_HangupCallResult> pmCoreIfMessage)
   {
      ETG_TRACE_USR4(("PmCallHandler::%20s::hangupCallResult::%d", getDeviceAddress().c_str(),
            ETG_CENUM(BTSRequestResult, pmCoreIfMessage->getBTSResult()._btsRequestResult)));

      if(BTS_REQ_SUCCESS == pmCoreIfMessage->getBTSResult()._btsRequestResult)
      {
         if ((CALL_INSTANCE_DEFAULT != _autoRejectCall._instance) &&
               (_autoRejectCall._instance == pmCoreIfMessage->getCallInstance()))
         {
            ETG_TRACE_USR4(("AutoReject:Hangup call response is success"));
         }
         else
         {
            getSCI_AgResponse()->raise_hANGUP_ALL_OK();
            auto it = _actAndCallInstanceListMap.find(pmCoreIfMessage->getAct());
            if (_actAndCallInstanceListMap.end() != it)
               _actAndCallInstanceListMap.erase(it);
         }
      }
      else if(BTS_REQ_FAILED == pmCoreIfMessage->getBTSResult()._btsRequestResult)
      {
         processErrorResponseFromStack(pmCoreIfMessage.get(), HANGUP_CALL_ERROR);
      }
   }

   void PmCallHandler::acceptCallResult(IN std::shared_ptr<PmCoreIfMessage_AcceptCallResult> pmCoreIfMessage)
   {
      ETG_TRACE_USR4(("PmCallHandler::%20s::acceptCallResult::%d", getDeviceAddress().c_str(),
            ETG_CENUM(BTSRequestResult, pmCoreIfMessage->getBTSResult()._btsRequestResult)));

      if(BTS_REQ_SUCCESS == pmCoreIfMessage->getBTSResult()._btsRequestResult)
      {
         getSCI_AgResponse()->raise_aCCEPT_OK();
         auto it = _actAndCallInstanceListMap.find(pmCoreIfMessage->getAct());
         if (_actAndCallInstanceListMap.end() != it)
            _actAndCallInstanceListMap.erase(it);
      }
      else if(BTS_REQ_FAILED == pmCoreIfMessage->getBTSResult()._btsRequestResult)
      {
         getSCI_AgResponse()->raise_aCCEPT_ERROR(pmCoreIfMessage.get());
      }
   }

   void PmCallHandler::holdIncomingCallResult(
         IN std::shared_ptr<PmCoreIfMessage_HoldIncomingCallResult> pmCoreIfMessage)
   {
      (void)(pmCoreIfMessage);
   }

   void PmCallHandler::transferAudioResult(
         IN std::shared_ptr<PmCoreIfMessage_TransferAudioResult> pmCoreIfMessage)
   {
      ETG_TRACE_USR4(("PmCallHandler::%20s::transferAudioResult::%d", getDeviceAddress().c_str(),
            ETG_CENUM(BTSRequestResult, pmCoreIfMessage->getBTSResult()._btsRequestResult)));

      if(BTS_REQ_SUCCESS == pmCoreIfMessage->getBTSResult()._btsRequestResult)
      {
         getSCI_AgResponse()->raise_tRANSFER_OK();
      }
      else if(BTS_REQ_FAILED == pmCoreIfMessage->getBTSResult()._btsRequestResult)
      {
         getSCI_AgResponse()->raise_tRANSFER_ERROR(pmCoreIfMessage.get());
      }
   }

   void PmCallHandler::processErrorResponseFromStack(PmCoreIfMessage * msg, SmErrorEnumType errorInfo)
   {
      ETG_TRACE_USR4(("PmCallHandler::%20s::processErrorResponseFromStack", getDeviceAddress().c_str()));

      switch(errorInfo)
      {
         case SWAP_CALL_ERROR:
         {
            auto it = _actAndCallInstanceListMap.find(((PmCoreIfMessage_SwapCallResult*)msg)->getAct());
            if(it != _actAndCallInstanceListMap.end())
            {
               NotificationEvent notificationEvent;
               notificationEvent._bdAddress = getDeviceAddress();
               notificationEvent._eventName = SWAP_CALL_FAILED;
               notificationEvent._reason = ((PmCoreIfMessage_SwapCallResult*)msg)->getBTSResult()._errorMessage;
               notificationEvent._callInstanceList = it->second;
               PmCoreMainController::getInstance().getPmCoreCallbackIfWrapper().doOnNotificationEvent(
                     notificationEvent);
               _actAndCallInstanceListMap.erase(it);
            }
         }
         break;
         case CONFERENCE_CALL_ERROR:
         {
            auto it = _actAndCallInstanceListMap.find(((PmCoreIfMessage_MergeCallsResult*)msg)->getAct());
            if(it != _actAndCallInstanceListMap.end())
            {
               NotificationEvent notificationEvent;
               notificationEvent._bdAddress = getDeviceAddress();
               notificationEvent._eventName = CONFERENCE_CALL_FAILED;
               notificationEvent._reason = ((PmCoreIfMessage_MergeCallsResult*)msg)->getBTSResult()._errorMessage;
               notificationEvent._callInstanceList = it->second;
               PmCoreMainController::getInstance().getPmCoreCallbackIfWrapper().doOnNotificationEvent(
                     notificationEvent);
               _actAndCallInstanceListMap.erase(it);
            }
         }
         break;
         case SPLIT_CALL_ERROR:
         {
            auto it = _actAndCallInstanceListMap.find(((PmCoreIfMessage_SplitCallsResult*)msg)->getAct());
            if(it != _actAndCallInstanceListMap.end())
            {
               NotificationEvent notificationEvent;
               notificationEvent._bdAddress = getDeviceAddress();
               notificationEvent._eventName = SPLIT_CALL_FAILED;
               notificationEvent._reason = ((PmCoreIfMessage_SplitCallsResult*)msg)->getBTSResult()._errorMessage;
               notificationEvent._callInstanceList = it->second;
               PmCoreMainController::getInstance().getPmCoreCallbackIfWrapper().doOnNotificationEvent(
                     notificationEvent);
               _actAndCallInstanceListMap.erase(it);
            }
         }
         break;
         case HANGUP_MULTIPARTY_ERROR:
         {
            auto it = _actAndCallInstanceListMap.find(((PmCoreIfMessage_HangupMultipartyResult*)msg)->getAct());
            if(it != _actAndCallInstanceListMap.end())
            {
               NotificationEvent notificationEvent;
               notificationEvent._bdAddress = getDeviceAddress();
               notificationEvent._eventName = HANGUP_CALL_FAILED;
               notificationEvent._reason = ((PmCoreIfMessage_HangupMultipartyResult*)msg)->getBTSResult()._errorMessage;
               notificationEvent._callInstanceList = it->second;
               PmCoreMainController::getInstance().getPmCoreCallbackIfWrapper().doOnNotificationEvent(
                     notificationEvent);
               _actAndCallInstanceListMap.erase(it);
            }
         }
         break;

         case HANGUP_CALL_ERROR:
         {
            //During Hangup request from client CallController sets the corresponding call status as "Disconnecting"
            //If the same Hangup request failed, then we need to set the original call status same as previous
            //By calling GetCallsRequest CallController will be receiving latest call status from stack.
            evobtstackwrapper::EvoBtStackWrapper::getInstance().sendGetCallsRequest(getDeviceAddress(),
                  PM_DEFAULT_ACT);

            auto it = _actAndCallInstanceListMap.find(((PmCoreIfMessage_HangupCallResult*)msg)->getAct());
            if(it != _actAndCallInstanceListMap.end())
            {
               NotificationEvent notificationEvent;
               notificationEvent._bdAddress = getDeviceAddress();
               notificationEvent._eventName = HANGUP_CALL_FAILED;
               notificationEvent._reason = ((PmCoreIfMessage_HangupCallResult*)msg)->getBTSResult()._errorMessage;
               notificationEvent._callInstanceList = it->second;
               PmCoreMainController::getInstance().getPmCoreCallbackIfWrapper().doOnNotificationEvent(
                     notificationEvent);
               _actAndCallInstanceListMap.erase(it);
            }

         }
         break;

         case HANGUP_ALL_CALL_ERROR:
         {
            //During Hangup request from client CallController sets the corresponding call status as "Disconnecting"
            //If the same Hangup request failed, then we need to set the original call status same as previous
            //By calling GetCallsRequest CallController will be receiving latest call status from stack.
            evobtstackwrapper::EvoBtStackWrapper::getInstance().sendGetCallsRequest(
                  getDeviceAddress(), PM_DEFAULT_ACT);

            auto it = _actAndCallInstanceListMap.find(((PmCoreIfMessage_HangupAllResult*)msg)->getAct());
            if(it != _actAndCallInstanceListMap.end())
            {
               NotificationEvent notificationEvent;
               notificationEvent._bdAddress = getDeviceAddress();
               notificationEvent._eventName = HANGUP_CALL_FAILED;
               notificationEvent._reason = ((PmCoreIfMessage_HangupAllResult*)msg)->getBTSResult()._errorMessage;
               notificationEvent._callInstanceList = it->second;
               PmCoreMainController::getInstance().getPmCoreCallbackIfWrapper().doOnNotificationEvent(
                     notificationEvent);
               _actAndCallInstanceListMap.erase(it);
            }

         }
         break;

         default:
            ETG_TRACE_USR4(("processErrorResponseFromStack: Default case"));
      }
   }

   void PmCallHandler::SmUserActionIf::dial(PmCoreIfMessage_DialRequest * msg)
   {
      ETG_TRACE_USR4(("PmCallHandler::dial() entered"));

      PmResult pmResult(PM_RESULT_OK, "");
      BTSResult btsResult;

      btsResult = evobtstackwrapper::EvoBtStackWrapper::getInstance().sendDialRequest(msg->getBdAddress(),
            msg->getTelephoneNumber(), msg->getHideCallerId(), msg->getAct());

      convertBTSResultToPmResult(btsResult, pmResult);

      msg->setPmResult(pmResult);
   }

   void PmCallHandler::SmUserActionIf::acquirePhoneAudio(PmCoreIfMessage_AcceptCallRequest* msg)
   {
      ETG_TRACE_USR4(("PmCallHandler::acquirePhoneAudio() entered"));

      if (msg)
      {
         PmResult pmResult(PM_RESULT_ERR_GENERAL, "");
         BTSResult btsResult(BTS_REQ_FAILED, ::ccdbusif::evolution::ERROR_UNKNOWN, "");

         AcceptOperation acceptOperation = msg->getAcceptOperation();

         if ((ACCEPT_CALLS == acceptOperation) || (ACCEPT_CALLS_BY_ACQUIRING_CHANNEL == acceptOperation) ||
               (RELEASE_AND_ACCEPT_BY_ACQUIRING_CHANNEL == acceptOperation) ||
               (HOLD_AND_ACCEPT_BY_ACQUIRING_CHANNEL == acceptOperation))
         {
            ETG_TRACE_USR4(("PmCallHandler::requestPhoneAudio()::%20s",_pmCallHandler.getDeviceAddress().c_str()));

            _pmCallHandler.getSCI_AudioManagerRequest()->set_phoneAudioState(PHONE_CHANNEL_REQUESTED);

            PmAudioManagerWrapper::getInstance().sendPrepareAudioRoute(_pmCallHandler.getDeviceAddress(),
                  pmaudiomanager::AM_PHONEAUDIO, CALL_CONTROLLER);

            pmResult._pmResultCode = PM_RESULT_OK;
         }
         else if(RELEASE_AND_ACCEPT == acceptOperation)
         {
            btsResult = evobtstackwrapper::EvoBtStackWrapper::getInstance().sendReleaseAndAnswerRequest(
                  msg->getBdAddress(), msg->getAct());

            convertBTSResultToPmResult(btsResult, pmResult);
         }
         else if(HOLD_AND_ACCEPT == acceptOperation)
         {
            btsResult = evobtstackwrapper::EvoBtStackWrapper::getInstance().sendHoldAndAnswerRequest(
                  msg->getBdAddress(), msg->getAct());

            convertBTSResultToPmResult(btsResult, pmResult);
         }
         else
         {
            //error, something wrong
         }

         if((RELEASE_AND_ACCEPT == acceptOperation) || (HOLD_AND_ACCEPT == acceptOperation))
         {
            CallInstanceList callInstanceList;
            callInstanceList.push_back(msg->getCallInstance());
            _pmCallHandler._actAndCallInstanceListMap.insert(std::pair<ActType,CallInstanceList>(msg->getAct(),callInstanceList));
         }

         msg->setPmResult(pmResult);
      }
   }

   void PmCallHandler::SmUserActionIf::speedDial(PmCoreIfMessage_SpeedDialRequest * msg)
   {
      ETG_TRACE_USR4(("PmCallHandler::speedDial() entered"));

      PmResult pmResult(PM_RESULT_OK, "");
      BTSResult btsResult;

      btsResult = evobtstackwrapper::EvoBtStackWrapper::getInstance().sendMemDialRequest(msg->getBdAddress(),
            msg->getMemDialIndex(), msg->getHideCallerId(), msg->getAct());

      convertBTSResultToPmResult(btsResult, pmResult);

      msg->setPmResult(pmResult);
   }

   void PmCallHandler::SmUserActionIf::reDial(pmcore::PmCoreIfMessage_RedialRequest * msg)
   {
      ETG_TRACE_USR4(("PmCallHandler::reDial() entered"));

      PmResult pmResult(PM_RESULT_OK, "");
      BTSResult btsResult;

      btsResult = evobtstackwrapper::EvoBtStackWrapper::getInstance().sendRedialRequest(msg->getBdAddress(),
            msg->getHideCallerId(), msg->getAct());

      convertBTSResultToPmResult(btsResult, pmResult);

      msg->setPmResult(pmResult);
   }

   void PmCallHandler::SmUserActionIf::hangupCalls(PmCoreIfMessage_HangupCallsRequest * msg)
   {
      ETG_TRACE_USR4(("PmCallHandler::hangupCalls() entered"));

      PmResult pmResult(PM_RESULT_OK, "");
      BTSResult btsResult;

      CallInstanceList::const_iterator it;

      for(it = msg->getCallInstanceList().begin(); it != msg->getCallInstanceList().end(); it++)
      {
         //TODO logic needs to be checked if any of the hangup request is failed
         btsResult = evobtstackwrapper::EvoBtStackWrapper::getInstance().sendHangupRequest(msg->getBdAddress(),
               *it, msg->getAct());
      }

      convertBTSResultToPmResult(btsResult, pmResult);

      msg->setPmResult(pmResult);
   }

   void PmCallHandler::SmUserActionIf::hangupAllCalls(PmCoreIfMessage_HangupCallsRequest * msg)
   {
      ETG_TRACE_USR4(("PmCallHandler::hangupAllCalls() entered"));

      PmResult pmResult(PM_RESULT_OK, "");
      BTSResult btsResult;

      btsResult = evobtstackwrapper::EvoBtStackWrapper::getInstance().sendHangupAllRequest(msg->getBdAddress(),
            msg->getAct());

      convertBTSResultToPmResult(btsResult, pmResult);

      msg->setPmResult(pmResult);
   }

   void PmCallHandler::SmUserActionIf::swapCalls(PmCoreIfMessage_SwapCallRequest * msg)
   {
      ETG_TRACE_USR4(("PmCallHandler::swapCalls() entered"));

      PmResult pmResult(PM_RESULT_OK, "");
      BTSResult btsResult(BTS_REQ_FAILED, ::ccdbusif::evolution::ERROR_UNKNOWN, "");

      if(SWAP_CALLS == msg->getSwapOperation())
      {
         btsResult = evobtstackwrapper::EvoBtStackWrapper::getInstance().sendSwapCallsRequest(
               msg->getBdAddress(), msg->getAct());
      }
      else if(RELEASE_AND_SWAP == msg->getSwapOperation())
      {
         btsResult = evobtstackwrapper::EvoBtStackWrapper::getInstance().sendReleaseAndSwapRequest(
               msg->getBdAddress(), msg->getAct());
      }
      else
      {
         //error, something wrong
      }

      convertBTSResultToPmResult(btsResult, pmResult);

      msg->setPmResult(pmResult);
   }

   void PmCallHandler::SmUserActionIf::splitCalls(PmCoreIfMessage_SplitCallsRequest * msg)
   {
      ETG_TRACE_USR4(("PmCallHandler::splitCalls() entered"));

      PmResult pmResult(PM_RESULT_OK, "");
      BTSResult btsResult(BTS_REQ_FAILED, ::ccdbusif::evolution::ERROR_UNKNOWN, "");

      CallInstanceList::const_iterator it;

      for(it = msg->getCallInstanceList().begin(); it != msg->getCallInstanceList().end(); it++)
      {
         btsResult = evobtstackwrapper::EvoBtStackWrapper::getInstance().sendPrivateChatRequest(
               msg->getBdAddress(), *it, msg->getAct());

         _pmCallHandler._actAndCallInstanceListMap.insert(std::pair<ActType,CallInstanceList>(msg->getAct(),msg->getCallInstanceList()));
      }

      convertBTSResultToPmResult(btsResult, pmResult);

      msg->setPmResult(pmResult);
   }

   void PmCallHandler::SmUserActionIf::conference(PmCoreIfMessage_MergeCallsRequest * msg)
   {
      ETG_TRACE_USR4(("PmCallHandler::conference() entered"));

      PmResult pmResult(PM_RESULT_OK, "");
      BTSResult btsResult;

      btsResult = evobtstackwrapper::EvoBtStackWrapper::getInstance().sendCreateMultipartyRequest(
            msg->getBdAddress(), msg->getAct());

      _pmCallHandler._actAndCallInstanceListMap.insert(std::pair<ActType,CallInstanceList>(msg->getAct(),msg->getCallInstanceList()));


      convertBTSResultToPmResult(btsResult, pmResult);

      msg->setPmResult(pmResult);
   }

   void PmCallHandler::SmUserActionIf::transferAudio(PmCoreIfMessage_TransferAudioRequest * msg)
   {
      ETG_TRACE_USR4(("PmCallHandler::transferAudio() entered"));

      PmResult pmResult(PM_RESULT_OK, "");
      BTSResult btsResult(BTS_REQ_FAILED, ::ccdbusif::evolution::ERROR_UNKNOWN, "");
      std::string audio;

      if(AUDIO_DIRECTION_HANDSFREE == msg->getAudioDirection())
      {
         audio = "HF";
         btsResult = evobtstackwrapper::EvoBtStackWrapper::getInstance().sendSetPropertyRequest(
               ::ccdbusif::evolution::IF_HANDSFREE, msg->getBdAddress(), "Audio", audio);
      }
      else if(AUDIO_DIRECTION_AUDIOGATEWAY == msg->getAudioDirection())
      {
         audio = "AG";
         btsResult = evobtstackwrapper::EvoBtStackWrapper::getInstance().sendSetPropertyRequest(
               ::ccdbusif::evolution::IF_HANDSFREE, msg->getBdAddress(), "Audio", audio);
      }
      else
      {
         //error, something wrong
      }

      convertBTSResultToPmResult(btsResult, pmResult);

      msg->setPmResult(pmResult);

      if (BTS_REQ_SUCCESS != btsResult._btsRequestResult)
      {
         ETG_TRACE_USR4(("Posting transfer audio request failed"));

         NotificationEvent notificationEvent;
         notificationEvent._bdAddress = _pmCallHandler.getDeviceAddress();
         notificationEvent._eventName = TRANSFER_CALL_AUDIO_TO_AG_FAILED;
         if (AUDIO_DIRECTION_HANDSFREE == msg->getAudioDirection())
            notificationEvent._eventName = TRANSFER_CALL_AUDIO_TO_HF_FAILED;

         PmCoreMainController::getInstance().getPmCoreCallbackIfWrapper().doOnNotificationEvent(
               notificationEvent);
      }
   }

   void PmCallHandler::SmUserActionIf::muteRingtone(PmCoreIfMessage_SetRingtoneMuteStateRequest * msg)
   {
      (void)(msg);
   }

   void PmCallHandler::SmUserActionIf::postAcceptCall()
   {
      ETG_TRACE_USR4(("PmCallHandler::postAcceptCall()::%20s",_pmCallHandler.getDeviceAddress().c_str()));

      PmResult pmResult(PM_RESULT_OK, "");
      BTSResult btsResult(BTS_REQ_FAILED, ::ccdbusif::evolution::ERROR_UNKNOWN, "");

      AcceptOperation acceptOperation = _pmCallHandler._acceptCallRequest.getAcceptOperation();

      if ((ACCEPT_CALLS == acceptOperation) || (ACCEPT_CALLS_BY_ACQUIRING_CHANNEL == acceptOperation))
      {
         ETG_TRACE_USR4(("SmSmHandlerIf::sendAnswerRequest::CallInstance: %u",
               _pmCallHandler._acceptCallRequest.getCallInstance()));

         btsResult = evobtstackwrapper::EvoBtStackWrapper::getInstance().sendAnswerRequest(
               _pmCallHandler._acceptCallRequest.getBdAddress(),
               _pmCallHandler._acceptCallRequest.getCallInstance(), _pmCallHandler._acceptCallRequest.getAct());
      }
      else if ((RELEASE_AND_ACCEPT == acceptOperation) ||
            (RELEASE_AND_ACCEPT_BY_ACQUIRING_CHANNEL == acceptOperation))
      {
         btsResult = evobtstackwrapper::EvoBtStackWrapper::getInstance().sendReleaseAndAnswerRequest(
               _pmCallHandler._acceptCallRequest.getBdAddress(),
               _pmCallHandler._acceptCallRequest.getAct());
      }
      else if ((HOLD_AND_ACCEPT == acceptOperation) || (HOLD_AND_ACCEPT_BY_ACQUIRING_CHANNEL == acceptOperation))
      {
         btsResult = evobtstackwrapper::EvoBtStackWrapper::getInstance().sendHoldAndAnswerRequest(
               _pmCallHandler._acceptCallRequest.getBdAddress(),
               _pmCallHandler._acceptCallRequest.getAct());
      }
      else
      {
         //error, something wrong
      }

      _pmCallHandler._acceptCallRequest = PmCoreIfMessage_AcceptCallRequest();
   }

   void PmCallHandler::SmUserActionIf::postHoldAndAcceptCall()
   {
      ETG_TRACE_USR4(("PmCallHandler::postHoldAndAcceptCall()::%20s",_pmCallHandler.getDeviceAddress().c_str()));

      PmResult pmResult(PM_RESULT_OK, "");
      BTSResult btsResult(BTS_REQ_FAILED, ::ccdbusif::evolution::ERROR_UNKNOWN, "");

      btsResult = evobtstackwrapper::EvoBtStackWrapper::getInstance().sendHoldAndAnswerRequest(
            _pmCallHandler._acceptCallRequest.getBdAddress(),
            _pmCallHandler._acceptCallRequest.getAct());

      _pmCallHandler._acceptCallRequest = PmCoreIfMessage_AcceptCallRequest();
   }

   void PmCallHandler::SmUserActionIf::postCancelOutgoing()
   {
      ETG_TRACE_USR4(("PmCallHandler::postCancelOutgoing() entered"));

      PmResult pmResult(PM_RESULT_OK, "");
      BTSResult btsResult(BTS_REQ_FAILED, ::ccdbusif::evolution::ERROR_UNKNOWN, "");

      CallInstanceList::const_iterator it;

      for(it = _pmCallHandler._cancelOutgoingCallRequest.getCallInstanceList().begin();
            it != _pmCallHandler._cancelOutgoingCallRequest.getCallInstanceList().end(); it++)
      {
         btsResult = evobtstackwrapper::EvoBtStackWrapper::getInstance().sendHangupRequest(
               _pmCallHandler._cancelOutgoingCallRequest.getBdAddress(), *it,
               _pmCallHandler._cancelOutgoingCallRequest.getAct());
      }

      //convertBTSResultToPmResult(btsResult, pmResult);

      //msg.setPmResult(pmResult);
   }

   void PmCallHandler::SmUserActionIf::setTransferAudioToHfToSuccess(
         pmcore::PmCoreIfMessage_TransferAudioRequest * msg)
   {
      ETG_TRACE_USR4(("PmCallHandler::setTransferAudioToHfToSuccess() entered"));
      msg->setPmResult(PmResult());
   }

   void PmCallHandler::SmSmHandlerIf::postSwitchToPassiveResponse(PmResultMessage result)
   {
      ETG_TRACE_USR4(("PmCallHandler::%20s::postSwitchToPassiveResponse()",_pmCallHandler.getDeviceAddress().c_str()));

      BTSResult btsResult(BTS_REQ_FAILED, ::ccdbusif::evolution::ERROR_UNKNOWN, "");

      DeviceInfoHandler& deviceInfoHandler = PmCoreMainController::getInstance().getDeviceInfoHandler();

      PmResultCode pmResultCode = (0 == result.compare("OK")) ? PM_RESULT_OK : PM_RESULT_ERR_GENERAL;

      // TODO: Check the flow here. If device is not present, then "processSwitchToPassiveResponse" will return false.
      // Hence in that scenario, the stopAudio should be done. Since the channel should be HELD at any cost.
      if (!deviceInfoHandler.processSwitchToPassiveResponse(_pmCallHandler.getDeviceAddress(),
            PmResult(pmResultCode, result)))
      {
         PmAudioManagerWrapper::getInstance().sendStopAudio(_pmCallHandler.getDeviceAddress(), CALL_CONTROLLER);
      }
   }

   void PmCallHandler::SmAgCallStateUpdateIf::waitingCall()
   {

   }

   void PmCallHandler::SmSmHandlerIf::stopSm()
   {

   }

   void PmCallHandler::SmSmHandlerIf::enterIdle()
   {

   }

   void PmCallHandler::SmSmHandlerIf::updatePhoneCallAudioActive(sc_boolean audioActive)
   {
      ETG_TRACE_USR4(("PmCallHandler::%20s::updatePhoneCallAudioActive::%d",_pmCallHandler.getDeviceAddress().c_str(), audioActive));

      PmCoreMainController::getInstance().getCallController().onCallAudioActiveChanged(
            _pmCallHandler.getDeviceAddress(), audioActive);
   }

   void PmCallHandler::SmSmHandlerIf::updateCallStatus()
   {
      ETG_TRACE_USR4(("PmCallHandler::%20s::updateCallStatus()",_pmCallHandler.getDeviceAddress().c_str()));

      PmCoreMainController::getInstance().getCallController().onCallStatusListChanged();
   }

   void PmCallHandler::SmSmHandlerIf::updateVehicleAudioStastus()
   {

   }

   void PmCallHandler::SmSmHandlerIf::callDisconnected()
   {

   }

   void PmCallHandler::SmSmHandlerIf::updateWaitingModeState(WaitingModeStateEnumType waitingModeState)
   {
      ETG_TRACE_USR4(("PmCallHandler::%20s::updateWaitingModeState()",_pmCallHandler.getDeviceAddress().c_str()));

      PmCoreMainController::getInstance().getCallController().updateWaitingModeState(_pmCallHandler._deviceAddress, waitingModeState);
   }

   void PmCallHandler::SmSmHandlerIf::acceptWaitingModeCall()
   {
      ETG_TRACE_USR4(("PmCallHandler::%20s::acceptWaitingModeCall()",_pmCallHandler.getDeviceAddress().c_str()));
      PmCoreMainController::getInstance().getCallController().acceptWaitingModeCall(_pmCallHandler._deviceAddress);
   }

   void PmCallHandler::SmSmHandlerIf::holdAndAcceptWaitingModeCall()
   {
      ETG_TRACE_USR4(("PmCallHandler::%20s::holdAndAcceptWaitingModeCall()",_pmCallHandler.getDeviceAddress().c_str()));
      PmCoreMainController::getInstance().getCallController().holdAndAcceptWaitingModeCall(_pmCallHandler._deviceAddress);
   }

   void PmCallHandler::SmUserActionIf::postTransferAudio(sc_string audio)
   {
      ETG_TRACE_USR4(("PmCallHandler::%20s::postTransferAudio()",_pmCallHandler.getDeviceAddress().c_str()));

      std::string audioType(audio);

      BTSResult btsResult = evobtstackwrapper::EvoBtStackWrapper::getInstance().sendSetPropertyRequest(
            ::ccdbusif::evolution::IF_HANDSFREE, _pmCallHandler.getDeviceAddress(), "Audio", audioType);

      if (BTS_REQ_FAILED == btsResult._btsRequestResult)
      {
         AudioDirectionEnumType audioDirection =
               ("AG" == audioType) ? AUDIO_DIRECTION_AUDIOGATEWAY : AUDIO_DIRECTION_HANDSFREE;

         std::shared_ptr<PmCoreIfMessage_TransferAudioResult> pmCoreIfMessage =
               getNewPmCoreIfMessage_TransferAudioResult(btsResult, _pmCallHandler.getDeviceAddress(),
                     audioDirection, PM_DEFAULT_ACT);

         _pmCallHandler.getSCI_AgResponse()->raise_tRANSFER_ERROR(pmCoreIfMessage.get());
      }
   }

   void PmCallHandler::SmSmHandlerIf::traceMessage(sc_string trace)
   {
      ETG_TRACE_USR4(("PmCallHandler::%20s::traceMessage:%50s", _pmCallHandler.getDeviceAddress().c_str(), trace));
   }

   void PmCallHandler::SmSmHandlerIf::generateCallStateEvent()
   {
      ETG_TRACE_USR4(("PmCallHandler::%20s::generateCallStateEvent",_pmCallHandler.getDeviceAddress().c_str()));

      //Generate the Callstate Event for stateMachine
      PmCoreMainController::getInstance().getCallController().generateAgCallStateEventToSM(_pmCallHandler.getDeviceAddress());
   }

   void PmCallHandler::SmSmHandlerIf::updateErrorStatus(SmErrorEnumType errorInfo)
   {
      ETG_TRACE_USR4(("PmCallHandler::%20s::updateErrorStatus::%u",
            _pmCallHandler.getDeviceAddress().c_str(), ETG_CENUM(SmErrorEnumType, errorInfo)));

      switch(errorInfo)
      {
         case RINGTONE_CHANNEL_DENIED_ERROR:
         {

         }
         break;
         case PHONE_CHANNEL_DENIED_FOR_ACCEPT_ERROR:
         {
            if(!_pmCallHandler._acceptCallRequest.getBdAddress().empty())
            {
               auto it = _pmCallHandler._actAndCallInstanceListMap.find(_pmCallHandler._acceptCallRequest.getAct());
               if(it != _pmCallHandler._actAndCallInstanceListMap.end())
               {
                  NotificationEvent notificationEvent;
                  notificationEvent._bdAddress = _pmCallHandler.getDeviceAddress();
                  notificationEvent._eventName = ACCEPT_CALL_FAILED;
                  notificationEvent._reason = "Phone Channel Acquisition Failure";
                  notificationEvent._callInstanceList = it->second;
                  PmCoreMainController::getInstance().getPmCoreCallbackIfWrapper().doOnNotificationEvent(
                        notificationEvent);
                  _pmCallHandler._actAndCallInstanceListMap.erase(it);
               }
               _pmCallHandler._acceptCallRequest = PmCoreIfMessage_AcceptCallRequest();
            }
         }
         break;
         case PHONE_CHANNEL_DENIED_FOR_RINGTONE_ERROR:
         {

         }
         break;
         case PHONE_CHANNEL_DENIED_FOR_TRANSFER_AUDIO_ERROR:
         {
            NotificationEvent notificationEvent;
            notificationEvent._bdAddress = _pmCallHandler.getDeviceAddress();
            notificationEvent._eventName = TRANSFER_CALL_AUDIO_TO_HF_FAILED;
            notificationEvent._reason = "Phone Channel Acquisition Failure";
            PmCoreMainController::getInstance().getPmCoreCallbackIfWrapper().doOnNotificationEvent(
                  notificationEvent);
         }
         break;
         case ECNR_FAILED_FOR_TRANSFER_AUDIO_ERROR:
         {
            NotificationEvent notificationEvent;
            notificationEvent._bdAddress = _pmCallHandler.getDeviceAddress();
            notificationEvent._eventName = TRANSFER_CALL_AUDIO_TO_HF_FAILED;
            notificationEvent._reason = "ECNR Failure";
            PmCoreMainController::getInstance().getPmCoreCallbackIfWrapper().doOnNotificationEvent(
                  notificationEvent);
         }
         break;
         case WAITINGMODE_CHANNEL_DENIED_ERROR:
         {
            // As of now, this error is not handled.
            // Change in state will notify clients that Waiting mode is unable to be started.
         }
         break;
         case START_WAITINGMODE_FAILED_ERROR:
         {
            // As of now, this error is not handled.
            // Change in state will notify clients that Waiting mode is unable to be started.
         }
         break;
         case STOP_WAITINGMODE_FAILED_ERROR:
         {
            // As of now, this error is not handled.
            // Change in state will notify clients that Waiting mode is unable to be stopped.
         }
         break;
         case ACCEPT_CALL_FAILED_TO_START_WAITINGMODE_ERROR:
         {
            // As of now, this error is not handled.
            // Change in state will notify clients that Waiting mode is unable to be started.
         }
         break;
         default:
            ETG_TRACE_USR4(("PmCallHandler::Unknown Error"));
      }
   }
   void PmCallHandler::SmSmHandlerIf::updateCallErrorStatus(PmCoreIfMessage * msg, SmErrorEnumType errorInfo)
   {
      ETG_TRACE_USR4(("PmCallHandler::%20s::updateCallErrorStatus::%u",
            _pmCallHandler.getDeviceAddress().c_str(), ETG_CENUM(SmErrorEnumType, errorInfo)));

      switch(errorInfo)
      {
         case DIAL_CALL_ERROR:
         {
            NotificationEvent notificationEvent;
            notificationEvent._bdAddress = ((PmCoreIfMessage_DialResult*)msg)->getBdAddress();
            notificationEvent._eventName = OUTGOING_CALL_FAILED;
            notificationEvent._reason = ((PmCoreIfMessage_DialResult*)msg)->getBTSResult()._errorMessage;
            notificationEvent._telephoneNumber = ((PmCoreIfMessage_DialResult*)msg)->getTelephoneNumber();

            PmCoreMainController::getInstance().getPmCoreCallbackIfWrapper().doOnNotificationEvent(notificationEvent);
         }
         break;
         case SPEED_DIAL_ERROR:
         {
            NotificationEvent notificationEvent;
            notificationEvent._bdAddress = ((PmCoreIfMessage_SpeedDialResult*)msg)->getBdAddress();
            notificationEvent._eventName = SPEED_DIAL_FAILED;
            notificationEvent._reason = ((PmCoreIfMessage_SpeedDialResult*)msg)->getBTSResult()._errorMessage;
            notificationEvent._telephoneNumber = "";

            PmCoreMainController::getInstance().getPmCoreCallbackIfWrapper().doOnNotificationEvent(notificationEvent);
         }
         break;
         case ACCEPT_CALL_ERROR:
         {
            auto it = _pmCallHandler._actAndCallInstanceListMap.find(((PmCoreIfMessage_AcceptCallResult*)msg)->getAct());
            if(it != _pmCallHandler._actAndCallInstanceListMap.end())
            {
               NotificationEvent notificationEvent;
               notificationEvent._bdAddress = _pmCallHandler.getDeviceAddress();
               notificationEvent._eventName = ACCEPT_CALL_FAILED;
               notificationEvent._reason = ((PmCoreIfMessage_AcceptCallResult*)msg)->getBTSResult()._errorMessage;
               notificationEvent._callInstanceList = it->second;
               PmCoreMainController::getInstance().getPmCoreCallbackIfWrapper().doOnNotificationEvent(
                     notificationEvent);
               _pmCallHandler._actAndCallInstanceListMap.erase(it);
            }
         }
         break;
         case HOLD_AND_ACCEPT_ERROR:
         {
            auto it = _pmCallHandler._actAndCallInstanceListMap.find(((PmCoreIfMessage_HoldAndAcceptResult*)msg)->getAct());
            if(it != _pmCallHandler._actAndCallInstanceListMap.end())
            {
               NotificationEvent notificationEvent;
               notificationEvent._bdAddress = _pmCallHandler.getDeviceAddress();
               notificationEvent._eventName = ACCEPT_CALL_FAILED;
               notificationEvent._reason = ((PmCoreIfMessage_HoldAndAcceptResult*)msg)->getBTSResult()._errorMessage;
               notificationEvent._callInstanceList = it->second;
               PmCoreMainController::getInstance().getPmCoreCallbackIfWrapper().doOnNotificationEvent(
                     notificationEvent);
               _pmCallHandler._actAndCallInstanceListMap.erase(it);
            }
         }
         break;
         case RELEASE_AND_ACCEPT_ERROR:
         {
            auto it = _pmCallHandler._actAndCallInstanceListMap.find(((PmCoreIfMessage_ReleaseAndAcceptResult*)msg)->getAct());
            if(it != _pmCallHandler._actAndCallInstanceListMap.end())
            {
               NotificationEvent notificationEvent;
               notificationEvent._bdAddress = _pmCallHandler.getDeviceAddress();
               notificationEvent._eventName = ACCEPT_CALL_FAILED;
               notificationEvent._reason = ((PmCoreIfMessage_ReleaseAndAcceptResult*)msg)->getBTSResult()._errorMessage;
               notificationEvent._callInstanceList = it->second;
               PmCoreMainController::getInstance().getPmCoreCallbackIfWrapper().doOnNotificationEvent(
                     notificationEvent);
               _pmCallHandler._actAndCallInstanceListMap.erase(it);
            }
         }
         break;
         case RELEASE_AND_SWAP_ERROR:
         {
            auto it = _pmCallHandler._actAndCallInstanceListMap.find(((PmCoreIfMessage_ReleaseAndSwapResult*)msg)->getAct());
            if(it != _pmCallHandler._actAndCallInstanceListMap.end())
            {
               NotificationEvent notificationEvent;
               notificationEvent._bdAddress = _pmCallHandler.getDeviceAddress();
               notificationEvent._eventName = SWAP_CALL_FAILED;
               notificationEvent._reason = ((PmCoreIfMessage_ReleaseAndSwapResult*)msg)->getBTSResult()._errorMessage;
               notificationEvent._callInstanceList = it->second;
               PmCoreMainController::getInstance().getPmCoreCallbackIfWrapper().doOnNotificationEvent(
                     notificationEvent);
               _pmCallHandler._actAndCallInstanceListMap.erase(it);
            }
         }
         break;

         case TRANSFER_CALL_ERROR:
         {
            NotificationEvent notificationEvent;
            notificationEvent._bdAddress = _pmCallHandler.getDeviceAddress();
            notificationEvent._eventName = TRANSFER_CALL_AUDIO_TO_AG_FAILED;
            if (AUDIO_DIRECTION_HANDSFREE == ((PmCoreIfMessage_TransferAudioResult*)msg)->getAudioDirection())
               notificationEvent._eventName = TRANSFER_CALL_AUDIO_TO_HF_FAILED;

            PmCoreMainController::getInstance().getPmCoreCallbackIfWrapper().doOnNotificationEvent(
                  notificationEvent);
         }
         break;

         default:
            ETG_TRACE_USR4(("PmCallHandler::Unknown Error"));

      }
   }

   void PmCallHandler::SmSmHandlerIf::handleSCODisconnection()
   {
      ETG_TRACE_USR4(("PmCallHandler::handleSCODisconnection()::%20s", _pmCallHandler.getDeviceAddress().c_str()));
      PmCoreMainController::getInstance().getCallController().processSCODisconnection(_pmCallHandler.getDeviceAddress());
   }

   void PmCallHandler::SmSmHandlerIf::startSCOTimer()
   {
      ETG_TRACE_USR4(("PmCallHandler::startSCOTimer()::%20s", _pmCallHandler.getDeviceAddress().c_str()));

      ScoTimeout scoDisconnectTimeout = com::bosch::pmcommon::PmConfiguration::getInstance().getHoldTimeAfterSCODisconnection();

      std::string token = _pmCallHandler.getDeviceAddress() + "/" + SCO_TIMER_ID;
      PmCoreMainController::getInstance().getCallController().startAsfTimer(token,
            scoDisconnectTimeout, TIMER_REPEAT_COUNT_ONE);
   }

   void PmCallHandler::SmSmHandlerIf::stopSCOTimer()
   {
      ETG_TRACE_USR4(("PmCallHandler::stopSCOTimer()::%20s", _pmCallHandler.getDeviceAddress().c_str()));
      std::string token = _pmCallHandler.getDeviceAddress() + "/" + SCO_TIMER_ID;

      PmCoreMainController::getInstance().getCallController().stopAsfTimer(token);
   }

   void PmCallHandler::SmSmHandlerIf::acceptCallFinished()
   {
      ETG_TRACE_USR4(("PmCallHandler::acceptCallFinished()::%20s",_pmCallHandler.getDeviceAddress().c_str()));
      _pmCallHandler._acceptCallRequest = PmCoreIfMessage_AcceptCallRequest();
   }

   void PmCallHandler::SmSmHandlerIf::handleSwitchToActiveInOutbandRinging()
   {
      ETG_TRACE_USR4(("PmCallHandler::handleSwitchToActiveInOutbandRinging()::%20s",
            _pmCallHandler.getDeviceAddress().c_str()));

      PmCoreMainController::getInstance().getRingtoneController().
            switchToActiveRequest(_pmCallHandler.getDeviceAddress());
   }

   void PmCallHandler::SmSmHandlerIf::processAllCallsRemoved()
   {
      ETG_TRACE_USR4(("PmCallHandler::processAllCallsRemoved()::%20s", _pmCallHandler.getDeviceAddress().c_str()));

      PmCoreMainController::getInstance().getDeviceInfoHandler().processAllCallsRemoved(
            _pmCallHandler.getDeviceAddress());
   }

   void PmCallHandler::SmAudioManagerRequestIf::requestPhoneAudio()
   {
      ETG_TRACE_USR4(("PmCallHandler::requestPhoneAudio()::%20s", _pmCallHandler.getDeviceAddress().c_str()));

      _pmCallHandler.getSCI_AudioManagerRequest()->set_phoneAudioState(PHONE_CHANNEL_REQUESTED);

      PmAudioManagerWrapper::getInstance().swapDeviceAddress(DEVICE_ADDRESS_ALL, _pmCallHandler.getDeviceAddress(),
            CALL_CONTROLLER);

      //Request phone channel to play the phone ring tone
      PmAudioManagerWrapper::getInstance().sendPrepareAudioRoute(_pmCallHandler.getDeviceAddress(),
            pmaudiomanager::AM_PHONEAUDIO, CALL_CONTROLLER);
   }

   void PmCallHandler::SmAudioManagerRequestIf::requestWaitingModeAudio()
   {
      ETG_TRACE_USR4(("PmCallHandler::requestWaitingModeAudio()::%20s",_pmCallHandler.getDeviceAddress().c_str()));

      _pmCallHandler.getSCI_AudioManagerRequest()->set_phoneAudioState(WAITINGMODE_CHANNEL_REQUESTED);

      //Request WaitingMode channel to play the vehicle ring tone
      PmAudioManagerWrapper::getInstance().sendPrepareAudioRoute(_pmCallHandler.getDeviceAddress(),
            pmaudiomanager::AM_WAITINGMODE, CALL_CONTROLLER);
   }

   void PmCallHandler::SmAudioManagerRequestIf::playHfAudio(SCOStatus scoStatus)
   {
      ETG_TRACE_USR4(("PmCallHandler::playHfAudio()::%20s",_pmCallHandler.getDeviceAddress().c_str()));

      _pmCallHandler.getSCI_AudioManagerRequest()->set_phoneAudioState(PHONE_START_STREAMING_REQUESTED);

      //Requesting audio manager to play audio
      PmAudioManagerWrapper::getInstance().sendPlayAudio(_pmCallHandler.getDeviceAddress(), scoStatus);
   }

   void PmCallHandler::SmAudioManagerRequestIf::pauseHfAudio()
   {
      ETG_TRACE_USR4(("PmCallHandler::pauseHfAudio()::%20s",_pmCallHandler.getDeviceAddress().c_str()));

      _pmCallHandler.getSCI_AudioManagerRequest()->set_phoneAudioState(PHONE_PAUSE_STREAMING_REQUESTED);

      //Requesting audio manager to pause audio
      PmAudioManagerWrapper::getInstance().sendPauseAudio(_pmCallHandler.getDeviceAddress());
   }

   void PmCallHandler::SmAudioManagerRequestIf::playRingtone()
   {
      ETG_TRACE_USR4(("PmCallHandler::playRingtone()::%20s", _pmCallHandler.getDeviceAddress().c_str()));

      _pmCallHandler.getSCI_AudioManagerRequest()->set_phoneAudioState(RINGTONE_CHANNEL_REQUESTED);

      std::shared_ptr<PmCoreIfMessage> pmCoreIfMessage = getNewPmCoreIfMessage_PlayRingtoneRequest(
            _pmCallHandler.getDeviceAddress(), 0, DEFAULT_FILE_PLAY_COUNT, 0, PM_CORE_IF_MSG_ORIGIN_INTERNAL);
      pmCoreIfMessage->traceMessage();
      PmCoreMainController::getInstance().handlePmCoreIfMessage(pmCoreIfMessage);
   }

   void PmCallHandler::SmAudioManagerRequestIf::playWaitingModeTone(SCOStatus scoStatus)
   {
      ETG_TRACE_USR4(("PmCallHandler::playWaitingModeTone()::%20s", _pmCallHandler.getDeviceAddress().c_str()));

      WaitingModeFilePath  waitingModeFilePath;
      PmCoreMainController::getInstance().getCallController().getWaitingModeFilePath(waitingModeFilePath);

      FilePath filePath = waitingModeFilePath._filePathWB;

      if(SCO_NARROWBAND == scoStatus)
      {
         filePath = waitingModeFilePath._filePathNB;
      }

      //Requesting audio manager to play audio
      PmAudioManagerWrapper::getInstance().sendPlayAudio(_pmCallHandler.getDeviceAddress(),
            filePath, CALL_CONTROLLER, DEFAULT_FILE_PLAY_COUNT, WAITINGMODE_TONE);
   }

   void PmCallHandler::SmAudioManagerRequestIf::stopAudio()
   {
      ETG_TRACE_USR4(("PmCallHandler::stopAudio()::%20s",_pmCallHandler.getDeviceAddress().c_str()));

      _pmCallHandler.getSCI_AudioManagerRequest()->set_phoneAudioState(PHONE_STOP_STREAMING_REQUESTED);

      //Requesting audio manager to pause audio
      PmAudioManagerWrapper::getInstance().sendStopAudio(_pmCallHandler.getDeviceAddress(), CALL_CONTROLLER);
   }

   void PmCallHandler::SmSmHandlerIf::printPhoneAudioState(PmCorePhoneAudioState audioState)
   {
      ETG_TRACE_USR4(("PmCallHandler::printPhoneAudioState()::%20s",_pmCallHandler.getDeviceAddress().c_str()));
      ETG_TRACE_USR4(("audioState: %u", ETG_CENUM(PmCorePhoneAudioState, audioState)));
   }

   StopAudioStateEnumType PmCallHandler::SmSmHandlerIf::isStopAudioRequested()
   {
      ETG_TRACE_USR4(("PmCallHandler::isStopAudioRequested()::%20s",_pmCallHandler.getDeviceAddress().c_str()));

      PmCorePhoneAudioState audioState = _pmCallHandler.getSCI_AudioManagerRequest()->get_phoneAudioState();

      if((PHONE_CHANNEL_REQUESTED == audioState) || (RINGTONE_CHANNEL_REQUESTED == audioState) || (PHONE_CHANNEL_ALLOCATED == audioState)
            || (RINGTONE_CHANNEL_ALLOCATED == audioState) || (PHONE_START_STREAMING_REQUESTED  == audioState)
            || (RINGTONE_START_STREAMING_REQUESTED == audioState)|| (PHONE_START_STREAMING_SUCCESS == audioState)
            || (RINGTONE_START_STREAMING_SUCCESS == audioState) || (WAITINGMODE_CHANNEL_REQUESTED == audioState)
            || (WAITINGMODE_CHANNEL_ALLOCATED == audioState) || (WAITINGMODE_START_STREAMING_REQUESTED  == audioState)
            || (WAITINGMODE_START_STREAMING_SUCCESS  == audioState) || (PHONE_PAUSE_STREAMING_SUCCESS  == audioState)
            || (RINGTONE_PAUSE_STREAMING_SUCCESS  == audioState) || (WAITINGMODE_PAUSE_STREAMING_SUCCESS  == audioState)
            || (RINGTONE_CHANNEL_NOT_ALLOCATED  == audioState) || (PHONE_CHANNEL_NOT_ALLOCATED  == audioState)
            || (WAITINGMODE_CHANNEL_NOT_ALLOCATED  == audioState))
      {
         return STOP_AUDIO_NOT_CALLED;
      }
      else if ((PHONE_STOP_STREAMING_REQUESTED == audioState)
            || (RINGTONE_STOP_STREAMING_REQUESTED == audioState)
            || (WAITINGMODE_STOP_STREAMING_REQUESTED == audioState))
      {
         return WAIT_FOR_STOP_AUDIO_RESPONSE;
      }
      else if((PHONE_PAUSE_STREAMING_REQUESTED == audioState) || (RINGTONE_PAUSE_STREAMING_REQUESTED == audioState) ||
            (WAITINGMODE_PAUSE_STREAMING_REQUESTED == audioState))
      {
         return WAIT_FOR_PAUSE_AUDIO_RESPONSE;
      }
      else
      {
         return NO_AUDIO_PROCESS_PENDING;
      }
   }

   sc_boolean PmCallHandler::SmSmHandlerIf::isChannelDenied()
   {
      ETG_TRACE_USR4(("PmCallHandler::isChannelDenied()::%20s",_pmCallHandler.getDeviceAddress().c_str()));

      PmCorePhoneAudioState audioState = _pmCallHandler.getSCI_AudioManagerRequest()->get_phoneAudioState();
      bool channelDenied = false;

      if((RINGTONE_CHANNEL_NOT_ALLOCATED  == audioState) || (PHONE_CHANNEL_NOT_ALLOCATED  == audioState)
            || (WAITINGMODE_CHANNEL_NOT_ALLOCATED  == audioState))
      {
         channelDenied = true;
      }

      return channelDenied;
   }

   sc_boolean PmCallHandler::SmSmHandlerIf::isRingtoneSuppressed()
   {
      ETG_TRACE_USR4(("PmCallHandler::isRingtoneSuppressed()::%20s",_pmCallHandler.getDeviceAddress().c_str()));

      SuppressRingtoneState suppressRingtoneState = false;

      PmCoreMainController::getInstance().getPmSettings().getSuppressRingtoneSetting(_pmCallHandler.getDeviceAddress(),
            suppressRingtoneState);

      return suppressRingtoneState;
   }

   sc_boolean PmCallHandler::SmSmHandlerIf::isInbandRinging()
   {
      ETG_TRACE_USR4(("PmCallHandler::isInbandRinging()::%20s",_pmCallHandler.getDeviceAddress().c_str()));

      return PmCoreMainController::getInstance().getCallController().isInbandRinging(
            _pmCallHandler.getDeviceAddress());
   }

   void PmCallHandler::SmSmHandlerIf::setAcceptCallParam(pmcore::PmCoreIfMessage_AcceptCallRequest * msg)
   {
      ETG_TRACE_USR4(("PmCallHandler::setAcceptCallParam()::%20s",_pmCallHandler.getDeviceAddress().c_str()));
      _acceptCallParam = *msg;
   }

   pmcore::PmCoreIfMessage_AcceptCallRequest * PmCallHandler::SmSmHandlerIf::getAcceptCallParam()
   {
      ETG_TRACE_USR4(("PmCallHandler::getAcceptCallParam()::%20s",_pmCallHandler.getDeviceAddress().c_str()));
      return &_acceptCallParam;
   }

   void PmCallHandler::SmAgResponseIf::acceptError()
   {

   }

   void PmCallHandler::SmAgResponseIf::dialError()
   {

   }

   void PmCallHandler::SmAgResponseIf::swapError()
   {

   }

   void PmCallHandler::SmAgResponseIf::hangupError()
   {

   }

   void PmCallHandler::SmAgResponseIf::transferError()
   {

   }

   void PmCallHandler::SmTimerUpdateIf::handleTimerCallback()
   {

   }

   template<typename T>
   void PmCallHandler::SmTraceObserverIf<T>::stateEntered(T state)
   {
      ETG_TRACE_USR4(("PmCallHandler::BDA::%20s::stateEntered()::%u", _pmCallHandler._deviceAddress.c_str(),
            ETG_CENUM(PmCallStateHandlerSmStates,state)));
   }

   template<typename T>
   void PmCallHandler::SmTraceObserverIf<T>::stateExited(T state)
   {
      ETG_TRACE_USR4(("PmCallHandler::BDA::%20s::stateExited()::%u", _pmCallHandler._deviceAddress.c_str(),
            ETG_CENUM(PmCallStateHandlerSmStates,state)));
   }

}
