/**
 * @file YakinduSmVoiceCallIf.h
 *
 * @swcomponent PhoneCallManager
 *
 * @brief This file contains the definition of the YakinduSmVoiceCallIf class
 *
 * @copyright (C) 2016 Robert Bosch GmbH.
 *            The reproduction, distribution and utilization of this file as
 *            well as the communication of its contents to others without express
 *            authorization is prohibited. Offenders will be held liable for the
 *            payment of damages. All rights reserved in the event of the grant
 *            of a patent, utility model or design.
 *
 * @details This file provides the state machine interfaces of voice call
 *
 * @ingroup PmCore
 */

#ifndef YakinduSmVoiceCallIf_h
#define YakinduSmVoiceCallIf_h

#include "ISmVoiceCallIf.h"

namespace pmcore
{
   //class forward declaration
   class PmCallHandler;

   class YakinduSmVoiceCallIf final : public ISmVoiceCallIf
   {
   public:
      /**
       * Constructor of YakinduSmVoiceCallIf class
       *
       * @param[in]
       * @param[out]
       * @param[in,out]
       *
       * @return
       */
      YakinduSmVoiceCallIf();

      /**
       * Destructor of YakinduSmVoiceCallIf class
       *
       * @param[in]
       * @param[out]
       * @param[in,out]
       *
       * @return
       */
      ~YakinduSmVoiceCallIf();

      /**
       * This method is used to notify the device connection
       *
       * @param[in] deviceAddress - device address
       * @param[out]
       * @param[in,out]
       *
       * @return void
       */
      void onDeviceConnected(IN const BdAddress& deviceAddress);

      /**
       * This method is used to notify the device disconnection
       *
       * @param[in] deviceAddress - device address
       * @param[out]
       * @param[in,out]
       *
       * @return void
       */
      void onDeviceDisconnected(IN const BdAddress& deviceAddress);

      /**
       * This method is used to set the call count to the voice call state machine
       *
       * @param[in] deviceAddress - device address
       * @param[in] callCount - call count
       * @param[out]
       * @param[in,out]
       *
       * @return void
       */
      void setCallCount(IN const BdAddress& deviceAddress, IN const CallCount callCount) override;

      /**
       * Member function overridden from ISmVoiceCallIf class
       *
       * @param[in]
       * @param[out]
       * @param[in,out]
       *
       * @return void
       */
      void onAgCallStateUpdate(IN const BdAddress& deviceAddress, IN const CallStateEnumType callState) override;

      /**
       * Member function overridden from ISmVoiceCallIf class
       *
       * @param[in]
       * @param[out]
       * @param[in,out]
       *
       * @return void
       */
      void onAudioManagerEventUpdate(IN const BdAddress& deviceAddress,
            IN const AudioManagerEventType amEvent) override;

      /**
       * Member function overridden from ISmVoiceCallIf class
       *
       * @param[in]
       * @param[out]
       * @param[in,out]
       *
       * @return void
       */
      void onScoConnectionStatusChanged(IN const BdAddress& deviceAddress, IN const SCOStatus scoStatus) override;

      /**
       * Member function overridden from ISmVoiceCallIf class
       *
       * @param[in] deviceAddress - device address
       * @param[in]
       * @param[out]
       * @param[in,out]
       *
       * @return void
       */
      void switchToPassiveRequest(IN const BdAddress& deviceAddress) override;

      /**
       * This method is used to switch the device to Passive
       * If the SCO is established for this device, it should be relinquished and the Phone channel should be released
       * if acquired. Because the device is no more an Active device to hold the Phone channel
       *
       * @param[in] bdAddress       - BT address of the device
       * @param[in]
       * @param[out]
       * @param[in,out]
       *
       * @return void
       */
      void switchToPassiveClientRequest(const BdAddress& deviceAddress) override;

      /**
       * Member function overridden from ISmVoiceCallIf class
       *
       * @param[in] deviceAddress - device address
       * @param[in]
       * @param[out]
       * @param[in,out]
       *
       * @return void
       */
      void switchToActiveRequest(IN const BdAddress& deviceAddress) override;

      /**
       * Member function overridden from ISmVoiceCallIf class
       *
       * @param[in] deviceAddress - device address
       * @param[in] autoRejectRequestReason - Auto rejecting reason
       * @param[out]
       * @param[in,out]
       *
       * @return void
       */
      void autoRejectCallRequest(IN const BdAddress& deviceAddress, IN const AutoRejectCall autoRejectCall) override;

      /**
       * Member function overridden from ISmVoiceCallIf class
       *
       * @param[in] deviceAddress - device address
       * @param[in] callInstance - callInstance of the removed call
       * @param[out]
       * @param[in,out]
       *
       * @return void
       */
      void onCallRemoved(IN const BdAddress& deviceAddress, IN const CallInstance callInstance) override;

      /**
       * Member function overridden from ISmVoiceCallIf class
       *
       * @param[in]
       * @param[out]
       * @param[in,out]
       *
       * @return void
       */
      void onTimerEventUpdate(IN const BdAddress& deviceAddress);

      // Request calls
      /**
       * Member function overridden from ISmVoiceCallIf class
       *
       * @param[in]
       * @param[out]
       * @param[in,out]
       *
       * @return void
       */
      void mergeCallsRequest(IN std::shared_ptr<PmCoreIfMessage_MergeCallsRequest> pmCoreIfMessage) override;

      /**
       * Member function overridden from ISmVoiceCallIf class
       *
       * @param[in]
       * @param[out]
       * @param[in,out]
       *
       * @return void
       */
      void splitCallsRequest(IN std::shared_ptr<PmCoreIfMessage_SplitCallsRequest> pmCoreIfMessage) override;

      /**
       * Member function overridden from ISmVoiceCallIf class
       *
       * @param[in]
       * @param[out]
       * @param[in,out]
       *
       * @return void
       */
      void hangupCallsRequest(IN std::shared_ptr<PmCoreIfMessage_HangupCallsRequest> pmCoreIfMessage) override;

      /**
       * Member function overridden from ISmVoiceCallIf class
       *
       * @param[in]
       * @param[out]
       * @param[in,out]
       *
       * @return void
       */
      void redialRequest(IN std::shared_ptr<PmCoreIfMessage_RedialRequest> pmCoreIfMessage) override;

      /**
       * Member function overridden from ISmVoiceCallIf class
       *
       * @param[in]
       * @param[out]
       * @param[in,out]
       *
       * @return void
       */
      void startStopWaitingModeRequest(IN std::shared_ptr<PmCoreIfMessage_StartStopWaitingModeRequest> pmCoreIfMessage) override;

      /**
       * Member function overridden from ISmVoiceCallIf class
       *
       * @param[in]
       * @param[out]
       * @param[in,out]
       *
       * @return void
       */
      void dialRequest(IN std::shared_ptr<PmCoreIfMessage_DialRequest> pmCoreIfMessage) override;

      /**
       * Member function overridden from ISmVoiceCallIf class
       *
       * @param[in]
       * @param[out]
       * @param[in,out]
       *
       * @return void
       */
      void acceptCallRequest(IN std::shared_ptr<PmCoreIfMessage_AcceptCallRequest> pmCoreIfMessage) override;

      /**
       * Member function overridden from ISmVoiceCallIf class
       *
       * @param[in]
       * @param[out]
       * @param[in,out]
       *
       * @return void
       */
      void swapCallRequest(IN std::shared_ptr<PmCoreIfMessage_SwapCallRequest> pmCoreIfMessage) override;

      /**
       * Member function overridden from ISmVoiceCallIf class
       *
       * @param[in]
       * @param[out]
       * @param[in,out]
       *
       * @return void
       */
      void speedDialRequest(IN std::shared_ptr<PmCoreIfMessage_SpeedDialRequest> pmCoreIfMessage) override;

      /**
       * Member function overridden from ISmVoiceCallIf class
       *
       * @param[in]
       * @param[out]
       * @param[in,out]
       *
       * @return void
       */
      void transferAudioRequest(IN std::shared_ptr<PmCoreIfMessage_TransferAudioRequest> pmCoreIfMessage) override;

      /**
       * Member function overridden from ISmVoiceCallIf class
       *
       * @param[in]
       * @param[out]
       * @param[in,out]
       *
       * @return void
       */
      void transferAudioRequest(IN const BdAddress& deviceAddress, IN const std::string& audio) override;

      // **************************************************************************************************
      // Result calls

      /**
       * Member function overridden from ISmVoiceCallIf class
       *
       * @param[in]
       * @param[out]
       * @param[in,out]
       *
       * @return void
       */
      void acceptSCOConnectResult(IN std::shared_ptr<PmCoreIfMessage_AcceptSCOConnectResult> pmCoreIfMessage) override;

      /**
       * Member function overridden from ISmVoiceCallIf class
       *
       * @param[in]
       * @param[out]
       * @param[in,out]
       *
       * @return void
       */
      void dialResult(IN std::shared_ptr<PmCoreIfMessage_DialResult> pmCoreIfMessage) override;

      /**
       * Member function overridden from ISmVoiceCallIf class
       *
       * @param[in]
       * @param[out]
       * @param[in,out]
       *
       * @return void
       */
      void speedDialResult(IN std::shared_ptr<PmCoreIfMessage_SpeedDialResult> pmCoreIfMessage) override;

      /**
       * Member function overridden from ISmVoiceCallIf class
       *
       * @param[in]
       * @param[out]
       * @param[in,out]
       *
       * @return void
       */
      void redialResult(IN std::shared_ptr<PmCoreIfMessage_RedialResult> pmCoreIfMessage) override;

      /**
       * Member function overridden from ISmVoiceCallIf class
       *
       * @param[in]
       * @param[out]
       * @param[in,out]
       *
       * @return void
       */
      void swapCallResult(IN std::shared_ptr<PmCoreIfMessage_SwapCallResult> pmCoreIfMessage) override;

      /**
       * Member function overridden from ISmVoiceCallIf class
       *
       * @param[in]
       * @param[out]
       * @param[in,out]
       *
       * @return void
       */
      void releaseAndAcceptResult(IN std::shared_ptr<PmCoreIfMessage_ReleaseAndAcceptResult> pmCoreIfMessage) override;

      /**
       * Member function overridden from ISmVoiceCallIf class
       *
       * @param[in]
       * @param[out]
       * @param[in,out]
       *
       * @return void
       */
      void releaseAndSwapResult(IN std::shared_ptr<PmCoreIfMessage_ReleaseAndSwapResult> pmCoreIfMessage) override;

      /**
       * Member function overridden from ISmVoiceCallIf class
       *
       * @param[in]
       * @param[out]
       * @param[in,out]
       *
       * @return void
       */
      void holdAndAcceptResult(IN std::shared_ptr<PmCoreIfMessage_HoldAndAcceptResult> pmCoreIfMessage) override;

      /**
       * Member function overridden from ISmVoiceCallIf class
       *
       * @param[in]
       * @param[out]
       * @param[in,out]
       *
       * @return void
       */
      void hangupAllResult(IN std::shared_ptr<PmCoreIfMessage_HangupAllResult> pmCoreIfMessage) override;

      /**
       * Member function overridden from ISmVoiceCallIf class
       *
       * @param[in]
       * @param[out]
       * @param[in,out]
       *
       * @return void
       */
      void splitCallsResult(IN std::shared_ptr<PmCoreIfMessage_SplitCallsResult> pmCoreIfMessage) override;

      /**
       * Member function overridden from ISmVoiceCallIf class
       *
       * @param[in]
       * @param[out]
       * @param[in,out]
       *
       * @return void
       */
      void mergeCallsResult(IN std::shared_ptr<PmCoreIfMessage_MergeCallsResult> pmCoreIfMessage) override;

      /**
       * Member function overridden from ISmVoiceCallIf class
       *
       * @param[in]
       * @param[out]
       * @param[in,out]
       *
       * @return void
       */
      void hangupMultipartyResult(IN std::shared_ptr<PmCoreIfMessage_HangupMultipartyResult> pmCoreIfMessage) override;

      /**
       * Member function overridden from ISmVoiceCallIf class
       *
       * @param[in]
       * @param[out]
       * @param[in,out]
       *
       * @return void
       */
      void hangupCallResult(IN std::shared_ptr<PmCoreIfMessage_HangupCallResult> pmCoreIfMessage) override;

      /**
       * Member function overridden from ISmVoiceCallIf class
       *
       * @param[in]
       * @param[out]
       * @param[in,out]
       *
       * @return void
       */
      void acceptCallResult(IN std::shared_ptr<PmCoreIfMessage_AcceptCallResult> pmCoreIfMessage) override;

      /**
       * Member function overridden from ISmVoiceCallIf class
       *
       * @param[in]
       * @param[out]
       * @param[in,out]
       *
       * @return void
       */
      void holdIncomingCallResult(IN std::shared_ptr<PmCoreIfMessage_HoldIncomingCallResult> pmCoreIfMessage) override;

      /**
       * Member function overridden from ISmVoiceCallIf class
       *
       * @param[in]
       * @param[out]
       * @param[in,out]
       *
       * @return void
       */
      void transferAudioResult(IN std::shared_ptr<PmCoreIfMessage_TransferAudioResult> pmCoreIfMessage) override;

   private:
      ::std::map<BdAddress, PmCallHandler*> _deviceToCallHandlerSmMap; //map between device address and pm call handler
   };
}

#endif //YakinduSmVoiceCallIf_h
