/**
 * @file EvoBtStackWrapper.cpp
 *
 * @swcomponent PhoneCallManager
 *
 * @brief This file contains the definition of the EvoBtStackWrapper class
 *
 * @copyright (C) 2016 Robert Bosch GmbH.
 *            The reproduction, distribution and utilization of this file as
 *            well as the communication of its contents to others without express
 *            authorization is prohibited. Offenders will be held liable for the
 *            payment of damages. All rights reserved in the event of the grant
 *            of a patent, utility model or design.
 *
 * @details This file provides the request interfaces of EvoBtStackWrapper.
 *
 * @ingroup BtStackWrapper
 */

#include "EvoBtStackWrapper.h"
#include "PmCoreIfMessageCreator.h"
#include "PropertyUpdateNotifierToCore.h"
#include "PmCoreMainController.h"
#include "FwBluetoothStringUtils.h"
#include "FwStringUtils.h"
#include "ObjectPathInfoListHandler.h"

#include "EvolutionGeniviDbusHfpCallVolumeCallBackIf.h"
#include "EvolutionGeniviDbusHfpHandsfreeCallBackIf.h"
#include "EvolutionGeniviDbusHfpManagerCallBackIf.h"
#include "EvolutionGeniviDbusHfpModemCallBackIf.h"
#include "EvolutionGeniviDbusHfpNetworkOperatorCallBackIf.h"
#include "EvolutionGeniviDbusHfpNetworkRegistrationCallBackIf.h"
#include "EvolutionGeniviDbusHfpSiriCallBackIf.h"
#include "EvolutionGeniviDbusHfpVoiceCallManagerCallBackIf.h"
#include "EvolutionGeniviDbusHfpVoiceCallCallBackIf.h"

#include "EvolutionGeniviDbusHfpCallVolumeRequestIf.h"
#include "EvolutionGeniviDbusHfpHandsfreeRequestIf.h"
#include "EvolutionGeniviDbusHfpManagerRequestIf.h"
#include "EvolutionGeniviDbusHfpModemRequestIf.h"
#include "EvolutionGeniviDbusHfpNetworkOperatorRequestIf.h"
#include "EvolutionGeniviDbusHfpNetworkRegistrationRequestIf.h"
#include "EvolutionGeniviDbusHfpSiriRequestIf.h"
#include "EvolutionGeniviDbusHfpVoiceCallManagerRequestIf.h"
#include "EvolutionGeniviDbusHfpVoiceCallRequestIf.h"
#include "EvolutionGeniviDbusParser.h"
#include "PmAppTrace.h"

using namespace pmcore;

#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#define ETG_DEFAULT_TRACE_CLASS TR_CLASS_BT_STACK_WRAPPER
#ifdef VARIANT_S_FTR_ENABLE_FW_ETG_USAGE
#include "trcGenProj/Header/EvoBtStackWrapper.cpp.trc.h"
#define ETG_DEFAULT_TRACE_CLASS TR_CLASS_BT_STACK_WRAPPER
#endif
#endif

namespace evobtstackwrapper
{
   EvoBtStackWrapper::EvoBtStackWrapper()
   {
      ETG_TRACE_USR4(("EvoBtStackWrapper: Constructor"));

      ObjectPathInfoListHandler::getInstance();

      _hfpCallVolumeCallBack = new EvolutionGeniviDbusHfpCallVolumeCallBackIf(this);
      _hfpHandsfreeCallBack = new EvolutionGeniviDbusHfpHandsfreeCallBackIf(this);
      _hfpManagerCallBack = new EvolutionGeniviDbusHfpManagerCallBackIf(this);
      _hfpModemCallBack = new EvolutionGeniviDbusHfpModemCallBackIf(this);
      _hfpNetworkOperatorCallBack = new EvolutionGeniviDbusHfpNetworkOperatorCallBackIf(this);
      _hfpNetworkRegistrationCallBack = new EvolutionGeniviDbusHfpNetworkRegistrationCallBackIf(this);
      _hfpSiriCallBack = new EvolutionGeniviDbusHfpSiriCallBackIf(this);
      _hfpVoiceCallManagerCallBack = new EvolutionGeniviDbusHfpVoiceCallManagerCallBackIf(this);
      _hfpVoiceCallCallBack = new EvolutionGeniviDbusHfpVoiceCallCallBackIf(this);

      _hfpCallVolumeRequestIf = new EvolutionGeniviDbusHfpCallVolumeRequestIf();
      _hfpHandsfreeRequestIf = new EvolutionGeniviDbusHfpHandsfreeRequestIf();
      _hfpManagerRequestIf = new EvolutionGeniviDbusHfpManagerRequestIf();
      _hfpModemRequestIf = new EvolutionGeniviDbusHfpModemRequestIf();
      _hfpNetworkOperatorRequestIf = new EvolutionGeniviDbusHfpNetworkOperatorRequestIf();
      _hfpNetworkRegistrationRequestIf = new EvolutionGeniviDbusHfpNetworkRegistrationRequestIf();
      _hfpSiriRequestIf = new EvolutionGeniviDbusHfpSiriRequestIf();
      _hfpVoiceCallManagerRequestIf = new EvolutionGeniviDbusHfpVoiceCallManagerRequestIf();
      _hfpVoiceCallRequestIf = new EvolutionGeniviDbusHfpVoiceCallRequestIf();

      if((nullptr != _hfpCallVolumeRequestIf) && (nullptr != _hfpCallVolumeCallBack))
      {
         _hfpCallVolumeRequestIf->setCallbackIf(_hfpCallVolumeCallBack);
      }

      if((nullptr != _hfpHandsfreeRequestIf) && (nullptr != _hfpHandsfreeCallBack))
      {
         _hfpHandsfreeRequestIf->setCallbackIf(_hfpHandsfreeCallBack);
      }

      if((nullptr != _hfpManagerRequestIf) && (nullptr != _hfpManagerCallBack))
      {
         _hfpManagerRequestIf->setCallbackIf(_hfpManagerCallBack);
      }

      if((nullptr != _hfpModemRequestIf) && (nullptr != _hfpModemCallBack))
      {
         _hfpModemRequestIf->setCallbackIf(_hfpModemCallBack);
      }

      if((nullptr != _hfpNetworkOperatorRequestIf) && (nullptr != _hfpNetworkOperatorCallBack))
      {
         _hfpNetworkOperatorRequestIf->setCallbackIf(_hfpNetworkOperatorCallBack);
      }

      if((nullptr != _hfpNetworkRegistrationRequestIf) && (nullptr != _hfpNetworkRegistrationCallBack))
      {
         _hfpNetworkRegistrationRequestIf->setCallbackIf(_hfpNetworkRegistrationCallBack);
      }

      if((nullptr != _hfpSiriRequestIf) && (nullptr != _hfpSiriCallBack))
      {
         _hfpSiriRequestIf->setCallbackIf(_hfpSiriCallBack);
      }

      if((nullptr != _hfpVoiceCallManagerRequestIf) && (nullptr != _hfpVoiceCallManagerCallBack))
      {
         _hfpVoiceCallManagerRequestIf->setCallbackIf(_hfpVoiceCallManagerCallBack);
      }

      if((nullptr != _hfpVoiceCallRequestIf) && (nullptr != _hfpVoiceCallCallBack))
      {
         _hfpVoiceCallRequestIf->setCallbackIf(_hfpVoiceCallCallBack);
      }
   }

   EvoBtStackWrapper::~EvoBtStackWrapper()
   {
      ETG_TRACE_USR4(("EvoBtStackWrapper: Destructor"));

      if(nullptr != _hfpCallVolumeRequestIf)
      {
         delete _hfpCallVolumeRequestIf;
         _hfpCallVolumeRequestIf = nullptr;
      }

      if(nullptr != _hfpHandsfreeRequestIf)
      {
         delete _hfpHandsfreeRequestIf;
         _hfpHandsfreeRequestIf = nullptr;
      }

      if(nullptr != _hfpManagerRequestIf)
      {
         delete _hfpManagerRequestIf;
         _hfpManagerRequestIf = nullptr;
      }

      if(nullptr != _hfpModemRequestIf)
      {
         delete _hfpModemRequestIf;
         _hfpModemRequestIf = nullptr;
      }

      if(nullptr != _hfpNetworkOperatorRequestIf)
      {
         delete _hfpNetworkOperatorRequestIf;
         _hfpNetworkOperatorRequestIf = nullptr;
      }

      if(nullptr != _hfpNetworkRegistrationRequestIf)
      {
         delete _hfpNetworkRegistrationRequestIf;
         _hfpNetworkRegistrationRequestIf = nullptr;
      }

      if(nullptr != _hfpSiriRequestIf)
      {
         delete _hfpSiriRequestIf;
         _hfpSiriRequestIf = nullptr;
      }

      if(nullptr != _hfpVoiceCallManagerRequestIf)
      {
         delete _hfpVoiceCallManagerRequestIf;
         _hfpVoiceCallManagerRequestIf = nullptr;
      }

      if(nullptr != _hfpVoiceCallRequestIf)
      {
         delete _hfpVoiceCallRequestIf;
         _hfpVoiceCallRequestIf = nullptr;
      }

      if(nullptr != _hfpCallVolumeCallBack)
      {
         delete _hfpCallVolumeCallBack;
         _hfpCallVolumeCallBack = nullptr;
      }

      if(nullptr != _hfpHandsfreeCallBack)
      {
         delete _hfpHandsfreeCallBack;
         _hfpHandsfreeCallBack = nullptr;
      }

      if(nullptr != _hfpManagerCallBack)
      {
         delete _hfpManagerCallBack;
         _hfpManagerCallBack = nullptr;
      }

      if(nullptr != _hfpModemCallBack)
      {
         delete _hfpModemCallBack;
         _hfpModemCallBack = nullptr;
      }

      if(nullptr != _hfpNetworkOperatorCallBack)
      {
         delete _hfpNetworkOperatorCallBack;
         _hfpNetworkOperatorCallBack = nullptr;
      }

      if(nullptr != _hfpNetworkRegistrationCallBack)
      {
         delete _hfpNetworkRegistrationCallBack;
         _hfpNetworkRegistrationCallBack = nullptr;
      }

      if(nullptr != _hfpSiriCallBack)
      {
         delete _hfpSiriCallBack;
         _hfpSiriCallBack = nullptr;
      }

      if(nullptr != _hfpVoiceCallManagerCallBack)
      {
         delete _hfpVoiceCallManagerCallBack;
         _hfpVoiceCallManagerCallBack = nullptr;
      }

      if(nullptr != _hfpVoiceCallCallBack)
      {
         delete _hfpVoiceCallCallBack;
         _hfpVoiceCallCallBack = nullptr;
      }
      //clear the maps
      _dbusActToSetPropertyAttributes.clear();
      clearDbusActToTelephoneNumberMap();
      clearDbusAndPmCoreActMap();
   }

   void EvoBtStackWrapper::setVehicleConfiguration(const VehicleConfiguration& vehicleConfig)
   {
      ETG_TRACE_USR4(("EvoBtStackWrapper: setVehicleConfiguration(): %s", vehicleConfig._vehicleBTAddress.c_str()));

      _vehicleConfig = vehicleConfig;
   }

   void EvoBtStackWrapper::onDeviceConnected(const BdAddress& deviceAddressIn)
   {
      ETG_TRACE_USR4(("EvoBtStackWrapper: onDeviceConnected(): %s", deviceAddressIn.c_str()));

      BdAddress deviceAddress = deviceAddressIn;
      BdAddress vehicleBTAddress = _vehicleConfig._vehicleBTAddress;
      ObjectPath objPath;

      ObjectPathInfoListHandler::getInstance().getHandsfreeObjectPath(deviceAddressIn, objPath);

      if(true == objPath.empty())
      {
         ObjectPathInfoListHandler::getInstance().addDeviceInObjectPathInfoListHandler(deviceAddressIn);

         ::fw::convertString2UpperCase(vehicleBTAddress);
         ::fw::convertString2UpperCase(deviceAddress);

         // Handsfree Modem object path format : /hfp/xxxxxxxxxxxx_xxxxxxxxxxxx
         objPath = HANDSFREE_OBJECT_PATH_HFP + vehicleBTAddress + HANDSFREE_OBJECT_PATH_UNDERSCORE + deviceAddress;

         ObjectPathInfoListHandler::getInstance().setHandsfreeObjectPath(deviceAddressIn, objPath);

         if(nullptr != _hfpModemRequestIf)
         {
            _hfpModemRequestIf->modemCreated(objPath);
         }
         else
         {
            ETG_TRACE_ERR(("onDeviceConnected: _hfpModemRequestIf Instance is not valid"));
         }
      }

      deviceCreated(objPath);
   }

   void EvoBtStackWrapper::onDeviceDisconnected(const BdAddress& deviceAddress)
   {
      ETG_TRACE_USR4(("EvoBtStackWrapper: onDeviceDisconnected(): %s", deviceAddress.c_str()));

      ObjectPath objPath;
      ObjectPathInfoListHandler::getInstance().getHandsfreeObjectPath(deviceAddress, objPath);

      if(false == objPath.empty())
      {
         if(nullptr != _hfpModemRequestIf)
         {
            _hfpModemRequestIf->modemRemoved(objPath);
         }
         else
         {
            ETG_TRACE_ERR(("onDeviceDisconnected: _hfpModemRequestIf Instance is not valid"));
         }

         deviceRemoved(objPath);
      }

      ObjectPathInfoListHandler::getInstance().removeHandsfreeObjectPath(deviceAddress);
      ObjectPathInfoListHandler::getInstance().removeDeviceFromObjectPathInfoListHandler(deviceAddress);
   }

   void EvoBtStackWrapper::deviceCreated(const ObjectPath& objPath)
   {
      ETG_TRACE_USR4(("EvoBtStackWrapper: deviceCreated(): %s", objPath.c_str()));

      //set handsfree object path
      if(nullptr != _hfpHandsfreeRequestIf)
      {
         _hfpHandsfreeRequestIf->deviceCreated(objPath);
      }

      if(nullptr != _hfpVoiceCallManagerRequestIf)
      {
         _hfpVoiceCallManagerRequestIf->deviceCreated(objPath);
      }

      if(nullptr != _hfpSiriRequestIf)
      {
         _hfpSiriRequestIf->deviceCreated(objPath);
      }

      if(nullptr != _hfpNetworkRegistrationRequestIf)
      {
         _hfpNetworkRegistrationRequestIf->deviceCreated(objPath);
      }

      if(nullptr != _hfpCallVolumeRequestIf)
      {
         _hfpCallVolumeRequestIf->deviceCreated(objPath);
      }
   }

   void EvoBtStackWrapper::deviceRemoved(const ObjectPath& objPath)
   {
      ETG_TRACE_USR4(("EvoBtStackWrapper: deviceRemoved(): %s", objPath.c_str()));

      //remove handsfree object path
      if(nullptr != _hfpHandsfreeRequestIf)
      {
         _hfpHandsfreeRequestIf->deviceRemoved(objPath);
      }

      if(nullptr != _hfpVoiceCallManagerRequestIf)
      {
         _hfpVoiceCallManagerRequestIf->deviceRemoved(objPath);
      }

      if(nullptr != _hfpSiriRequestIf)
      {
         _hfpSiriRequestIf->deviceRemoved(objPath);
      }

      if(nullptr != _hfpNetworkRegistrationRequestIf)
      {
         _hfpNetworkRegistrationRequestIf->deviceRemoved(objPath);
      }

      if(nullptr != _hfpCallVolumeRequestIf)
      {
         _hfpCallVolumeRequestIf->deviceRemoved(objPath);
      }
   }

   void EvoBtStackWrapper::modemCreated(const BdAddress& deviceAddress, const ObjectPath& objPath)
   {
      ETG_TRACE_USR4(("EvoBtStackWrapper: modemCreated(): %s", objPath.c_str()));

      if(true == ::fw::isValidBdAddress(deviceAddress, false, false))
      {
         ObjectPathInfoListHandler::getInstance().addDeviceInObjectPathInfoListHandler(deviceAddress);
         ObjectPathInfoListHandler::getInstance().setHandsfreeObjectPath(deviceAddress, objPath);

         if(nullptr != _hfpModemRequestIf)
         {
            _hfpModemRequestIf->modemCreated(objPath);
         }
         else
         {
            ETG_TRACE_ERR(("modemCreated: Invalid instance of _hfpModemRequestIf"));
         }
      }
      else
      {
         ETG_TRACE_ERR(("modemCreated: Invalid BdAddress"));
      }
   }

   void EvoBtStackWrapper::modemRemoved(const ObjectPath& objPath)
   {
      ETG_TRACE_USR4(("EvoBtStackWrapper: modemRemoved(): %s", objPath.c_str()));

      if(nullptr != _hfpModemRequestIf)
      {
         _hfpModemRequestIf->modemRemoved(objPath);
      }
      else
      {
         ETG_TRACE_ERR(("modemRemoved: Invalid instance of _hfpModemRequestIf"));
      }

      BdAddress deviceAddress;
      ObjectPathInfoListHandler::getInstance().getDeviceAddress(objPath, deviceAddress);
      if(false == deviceAddress.empty())
      {
         ObjectPathInfoListHandler::getInstance().removeHandsfreeObjectPath(deviceAddress);
         ObjectPathInfoListHandler::getInstance().removeDeviceFromObjectPathInfoListHandler(deviceAddress);
      }
      else
      {
         ETG_TRACE_ERR(("modemRemoved: deviceAddress is Empty"));
      }
   }

   void EvoBtStackWrapper::networkOperatorCreated(const BdAddress& deviceAddress, const ObjectPath& objPath)
   {
      ETG_TRACE_USR4(("EvoBtStackWrapper: networkOperatorCreated(): %s", objPath.c_str()));

      //set operator object path
      if(nullptr != _hfpNetworkOperatorRequestIf)
      {
         NetworkOperatorInstance networkOperatorInstance = getInstanceFromObjectPath(objPath,
               NETWORKOPERATOR_OBJECTPATH_LENGTH);

         if(DEFAULT_INSTANCE_VALUE != networkOperatorInstance)
         {
            ObjectPathInfoListHandler::getInstance().setNetworkOperatorObjectPath(deviceAddress,
                  networkOperatorInstance, objPath);

            _hfpNetworkOperatorRequestIf->networkOperatorCreated(objPath);
         }
         else
         {
            ETG_TRACE_ERR(("networkOperatorCreated Instance is not valid"));
         }
      }
      else
      {
         ETG_TRACE_ERR(("networkOperatorCreated: _hfpNetworkOperatorRequestIf is NULL"));
      }
   }

   void EvoBtStackWrapper::networkOperatorRemoved(const ObjectPath& objPath)
   {
      ETG_TRACE_USR4(("EvoBtStackWrapper: networkOperatorRemoved(): %s", objPath.c_str()));

      //remove operator object path
      if(nullptr != _hfpNetworkOperatorRequestIf)
      {
         BdAddress deviceAddress;
         NetworkOperatorInstance networkOperatorInstance;

         ObjectPathInfoListHandler::getInstance().getDeviceAddressAndNetworkOperatorInstance(objPath,
                        deviceAddress, networkOperatorInstance);

         if(false == deviceAddress.empty())
         {
            ObjectPathInfoListHandler::getInstance().removeNetworkOperatorObjectPath(deviceAddress,
                  networkOperatorInstance);

            _hfpNetworkOperatorRequestIf->networkOperatorRemoved(objPath);
         }
         else
         {
            ETG_TRACE_ERR(("operatorRemoved(): DeviceAddress and NetworkOperatorInstance not found"));
         }
      }
      else
      {
         ETG_TRACE_ERR(("networkOperatorRemoved: _hfpNetworkOperatorRequestIf is NULL"));
      }
   }

   void EvoBtStackWrapper::voiceCallCreated(const ObjectPath& voiceCallMgrObjPath, const ObjectPath& voiceCallObjPath,
         BTSDbusPropertyList& propertyList)
   {
      ETG_TRACE_USR4(("EvoBtStackWrapper: voiceCallCreated(): %s", voiceCallObjPath.c_str()));

      //set voice call object path
      if(nullptr != _hfpVoiceCallRequestIf)
      {
         BdAddress deviceAddress;

         ObjectPathInfoListHandler::getInstance().getDeviceAddress(voiceCallMgrObjPath, deviceAddress);

         if(false == deviceAddress.empty())
         {
            CallInstance callInstance = getInstanceFromObjectPath(voiceCallObjPath, VOICECALL_OBJECTPATH_LENGTH);

            if(DEFAULT_INSTANCE_VALUE != callInstance)
            {
               ObjectPathInfoListHandler::getInstance().setVoiceCallObjectPath(deviceAddress, callInstance,
                     voiceCallObjPath);

               if (_hfpVoiceCallRequestIf->voiceCallCreated(voiceCallObjPath))
               {
                  BTSVoiceCallInfo btsVoiceCallInfo;
                  btsVoiceCallInfo._callInfo._instance = callInstance;

                  convertBTSDbusPropertyListToBTSVoiceCallInfo(propertyList, btsVoiceCallInfo);

                  //Send update to CallController
                  std::shared_ptr<PropertyDetails<BTSVoiceCallInfo>> propertyDetails =
                        std::make_shared<PropertyDetails<BTSVoiceCallInfo>>(deviceAddress, btsVoiceCallInfo);

                  PropertyUpdateNotifierToCore::getInstance().notifyControllers(BTS_SIGNAL_VOICE_CALL_ADDED, propertyDetails);
               }
               else
               {
                  ETG_TRACE_ERR(("Error in voiceCallCreated"));
               }
            }
            else
            {
               ETG_TRACE_ERR(("voiceCallCreated Instance is not valid"));
            }
         }
         else
         {
            ETG_TRACE_ERR(("voiceCallCreated(): DeviceAddress is empty"));
         }
      }
      else
      {
         ETG_TRACE_ERR(("voiceCallCreated: _hfpVoiceCallRequestIf is NULL"));
      }
   }

   void EvoBtStackWrapper::voiceCallRemoved(const ObjectPath& objPath)
   {
      ETG_TRACE_USR4(("EvoBtStackWrapper: voiceCallRemoved(): %s", objPath.c_str()));

      //remove voice call object path
      if(nullptr != _hfpVoiceCallRequestIf)
      {
         BdAddress deviceAddress;
         CallInstance callInstance;

         ObjectPathInfoListHandler::getInstance().getDeviceAddressAndCallInstance(objPath, deviceAddress, callInstance);

         if(false == deviceAddress.empty())
         {
            ObjectPathInfoListHandler::getInstance().removeVoiceCallObjectPath(deviceAddress, callInstance);

            _hfpVoiceCallRequestIf->voiceCallRemoved(objPath);

            //Send update to CallController
            std::shared_ptr<PropertyDetails<CallInstance>> propertyDetails =
                  std::make_shared<PropertyDetails<CallInstance>>(deviceAddress, callInstance);

            PropertyUpdateNotifierToCore::getInstance().notifyControllers(BTS_SIGNAL_VOICE_CALL_REMOVED,
                  propertyDetails);
         }
         else
         {
            ETG_TRACE_ERR(("voiceCallRemoved(): DeviceAddress and CallInstance not found"));
         }
      }
      else
      {
         ETG_TRACE_ERR(("voiceCallRemoved: _hfpVoiceCallRequestIf is NULL"));
      }
   }

   void EvoBtStackWrapper::onProxyServiceAvailabilityStatus(const ::ccdbusif::evolution::Interface interface,
         const ObjectPath& objPath, const bool serviceAvailability)
   {
      BdAddress deviceAddress;
      ObjectPathInfoListHandler::getInstance().getDeviceAddress(objPath, deviceAddress);

      if(false == deviceAddress.empty())
      {
         ProxyServiceAvailability proxyStatus;
         proxyStatus._interface = interface;
         proxyStatus._serviceAvailability = serviceAvailability;

         std::shared_ptr<PropertyDetails<ProxyServiceAvailability>> propertyDetails =
               std::make_shared<PropertyDetails<ProxyServiceAvailability>>(deviceAddress, proxyStatus);

         PropertyUpdateNotifierToCore::getInstance().notifyControllers(BTS_UPDATE_PROXY_SERVICE_AVAILABILITY,
               propertyDetails);
      }
      else
      {
         ETG_TRACE_ERR(("onProxyServiceAvailabilityStatus: no valid device address found"));
      }
   }

   BTSResult EvoBtStackWrapper::sendSetPropertyRequest(const ::ccdbusif::evolution::Interface interface,
         const BdAddress& deviceAddress, const ::std::string& property, const uint8 value)
   {
      BTSResult btsResult(BTS_REQ_FAILED, ::ccdbusif::evolution::ERROR_UNKNOWN, "");
      act_t dbusAct = ::ccdbusif::DEFAULT_ACT;
      ObjectPath objPath;

      ObjectPathInfoListHandler::getInstance().getHandsfreeObjectPath(deviceAddress, objPath);

      if(false == objPath.empty())
      {
         if(::ccdbusif::evolution::IF_CALL_VOLUME == interface)
         {
            if(nullptr != _hfpCallVolumeRequestIf)
            {
               dbusAct = _hfpCallVolumeRequestIf->sendSetPropertyRequest(objPath, property, value);

               if(::ccdbusif::DEFAULT_ACT != dbusAct)
               {
                  btsResult._btsRequestResult = BTS_REQ_SUCCESS;
               }
               else
               {
                  ETG_TRACE_ERR(("sendSetPropertyRequest: Failed to set property:%s for IF_CALL_VOLUME interface",property.c_str()));
               }
            }
            else
            {
               ETG_TRACE_ERR(("sendSetPropertyRequest: Invalid instance of _hfpCallVolumeRequestIf"));
            }
         }
      }
      else
      {
         ETG_TRACE_ERR(("sendSetPropertyRequest(int) No valid object path for BdAddress: %s", deviceAddress.c_str()));
      }

      return btsResult;
   }

   BTSResult EvoBtStackWrapper::sendSetPropertyRequest(const ::ccdbusif::evolution::Interface interface,
         const BdAddress& deviceAddress, const ::std::string& property, const bool value)
   {
      BTSResult btsResult(BTS_REQ_FAILED, ::ccdbusif::evolution::ERROR_UNKNOWN, "");
      act_t dbusAct = ::ccdbusif::DEFAULT_ACT;
      ObjectPath objPath;

      ObjectPathInfoListHandler::getInstance().getHandsfreeObjectPath(deviceAddress, objPath);

      if(false == objPath.empty())
      {
         if(::ccdbusif::evolution::IF_HANDSFREE == interface)
         {
            if(nullptr != _hfpHandsfreeRequestIf)
            {
               dbusAct = _hfpHandsfreeRequestIf->sendSetPropertyRequest(objPath, property, value);

               if(::ccdbusif::DEFAULT_ACT != dbusAct)
               {
                  btsResult._btsRequestResult = BTS_REQ_SUCCESS;

                  //add HF properties to dbusmap to handle response - for VR enable/disable error handling
                  SetPropertyAttributes setPropertyAttributes;
                  setPropertyAttributes._property = property;
                  setPropertyAttributes._value = value ? "true" : "false";

                  _dbusActToSetPropertyAttributes.emplace_hint(_dbusActToSetPropertyAttributes.end(),
                        dbusAct, setPropertyAttributes);
               }
               else
               {
                  ETG_TRACE_ERR(("sendSetPropertyRequest(bool): Failed to set property:%s for IF_HANDSFREE interface",property.c_str()));
               }
            }
            else
            {
               ETG_TRACE_ERR(("sendSetPropertyRequest: Invalid instance of _hfpHandsfreeRequestIf"));
            }
         }
         else if(::ccdbusif::evolution::IF_MODEM == interface)
         {
            if(nullptr != _hfpModemRequestIf)
            {
               dbusAct = _hfpModemRequestIf->sendSetPropertyRequest(objPath, property, value);

               if(::ccdbusif::DEFAULT_ACT != dbusAct)
               {
                  btsResult._btsRequestResult = BTS_REQ_SUCCESS;
               }
               else
               {
                  ETG_TRACE_ERR(("sendSetPropertyRequest: Failed to set property:%s for IF_MODEM interface",property.c_str()));
               }
            }
            else
            {
               ETG_TRACE_ERR(("sendSetPropertyRequest: Invalid instance of _hfpModemRequestIf"));
            }
         }
         else if(::ccdbusif::evolution::IF_SIRI == interface)
         {
            if(nullptr != _hfpSiriRequestIf)
            {
               dbusAct = _hfpSiriRequestIf->sendSetPropertyRequest(objPath, property, value);

               if(::ccdbusif::DEFAULT_ACT != dbusAct)
               {
                  btsResult._btsRequestResult = BTS_REQ_SUCCESS;
               }
               else
               {
                  ETG_TRACE_ERR(("sendSetPropertyRequest: Failed to set property:%s for IF_SIRI interface",property.c_str()));
               }
            }
            else
            {
               ETG_TRACE_ERR(("sendSetPropertyRequest: Invalid instance of _hfpSiriRequestIf"));
            }
         }
      }
      else
      {
         ETG_TRACE_ERR(("sendSetPropertyRequest(bool) No valid object path for BdAddress: %s", deviceAddress.c_str()));
      }

      return btsResult;
   }

   BTSResult EvoBtStackWrapper::sendSetPropertyRequest(const ::ccdbusif::evolution::Interface interface,
         const BdAddress& deviceAddress, const ::std::string& property, const ::std::string& value)
   {
      BTSResult btsResult(BTS_REQ_FAILED, ::ccdbusif::evolution::ERROR_UNKNOWN, "");
      act_t dbusAct = ::ccdbusif::DEFAULT_ACT;
      ObjectPath objPath;

      ObjectPathInfoListHandler::getInstance().getHandsfreeObjectPath(deviceAddress, objPath);

      if(false == objPath.empty())
      {
         if(::ccdbusif::evolution::IF_HANDSFREE == interface)
         {
            if(nullptr != _hfpHandsfreeRequestIf)
            {
               dbusAct = _hfpHandsfreeRequestIf->sendSetPropertyRequest(objPath, property, value);

               if(::ccdbusif::DEFAULT_ACT != dbusAct)
               {
                  btsResult._btsRequestResult = BTS_REQ_SUCCESS;

                  SetPropertyAttributes setPropertyAttributes;
                  setPropertyAttributes._property = property;
                  setPropertyAttributes._value = value;

                  _dbusActToSetPropertyAttributes.emplace_hint(_dbusActToSetPropertyAttributes.end(),
                        dbusAct, setPropertyAttributes);
               }
               else
               {
                  ETG_TRACE_ERR(("sendSetPropertyRequest(string): Failed to set property:%s for IF_HANDSFREE interface",property.c_str()));
               }
            }
            else
            {
               ETG_TRACE_ERR(("sendSetPropertyRequest: Invalid instance of _hfpSiriRequestIf"));
            }
         }
      }
      else
      {
         ETG_TRACE_ERR(("sendSetPropertyRequest(string): No valid object path for BdAddress: %s",
               deviceAddress.c_str()));
      }

      return btsResult;
   }

   void EvoBtStackWrapper::onSetPropertyResponse(const ObjectPath& objPath,
         const ::ccdbusif::evolution::Interface interface,
         const ErrorName& errorName, const ErrorMessage& errorMessage, const act_t dbusAct)
   {
      ETG_TRACE_USR4(("EvoBtStackWrapper::onSetPropertyResponse"));

      BdAddress deviceAddress;

      ObjectPathInfoListHandler::getInstance().getDeviceAddress(objPath, deviceAddress);
      if(false == deviceAddress.empty())
      {
         BTSResult btsResult(BTS_REQ_SUCCESS, ::ccdbusif::evolution::ERROR_UNKNOWN, "");

         if (!errorName.empty())
         {
            btsResult._btsRequestResult = BTS_REQ_FAILED;
            btsResult._errorCode = getErrorCode2Enum(errorName, errorMessage);
            btsResult._errorMessage = errorMessage;
         }

         if(::ccdbusif::evolution::IF_HANDSFREE == interface)
         {
            auto iter = _dbusActToSetPropertyAttributes.find(dbusAct);

            if (_dbusActToSetPropertyAttributes.end() != iter)
            {
               if ("Audio" == iter->second._property)
               {
                  AudioDirectionEnumType audioDirection =
                        ("AG" == iter->second._value) ? AUDIO_DIRECTION_AUDIOGATEWAY : AUDIO_DIRECTION_HANDSFREE;

                  std::shared_ptr<PmCoreIfMessage> pmCoreIfMessage = getNewPmCoreIfMessage_TransferAudioResult(
                        btsResult, deviceAddress, audioDirection,
                        ::ccdbusif::DEFAULT_ACT, PM_CORE_IF_MSG_ORIGIN_CLIENT);
                  pmCoreIfMessage->traceMessage();

                  PmCoreMainController::getInstance().handlePmCoreIfMessage(pmCoreIfMessage);
               }

               else if ("VoiceRecognition" == iter->second._property)
               {
                  StartStop vrStatus = ("true" == iter->second._value) ? true : false;
                  std::shared_ptr<PmCoreIfMessage> pmCoreIfMessage = getNewPmCoreIfMessage_StartStopVoiceRecognitionResult(
                        btsResult, deviceAddress, vrStatus, ::ccdbusif::DEFAULT_ACT, PM_CORE_IF_MSG_ORIGIN_CLIENT);
                  pmCoreIfMessage->traceMessage();

                  PmCoreMainController::getInstance().handlePmCoreIfMessage(pmCoreIfMessage);
               }

               _dbusActToSetPropertyAttributes.erase(iter);
            }
         }
      }
      else
      {
         ETG_TRACE_ERR(("onSetPropertyResponse: No valid BdAddress for object path: %s", objPath.c_str()));
      }
   }

   BTSResult EvoBtStackWrapper::sendRequestPhoneNumberRequest(const BdAddress& deviceAddress, const ActType act)
   {
      BTSResult btsResult(BTS_REQ_FAILED, ::ccdbusif::evolution::ERROR_UNKNOWN, "");
      act_t dbusAct = ::ccdbusif::DEFAULT_ACT;

      if(nullptr != _hfpHandsfreeRequestIf)
      {
         ObjectPath objPath;
         ObjectPathInfoListHandler::getInstance().getHandsfreeObjectPath(deviceAddress, objPath);

         if(false == objPath.empty())
         {
            dbusAct = _hfpHandsfreeRequestIf->sendRequestPhoneNumberRequest(objPath);

            if(::ccdbusif::DEFAULT_ACT != dbusAct)
            {
               mapDbusActToPmCoreAct(dbusAct, act);
               btsResult._btsRequestResult = BTS_REQ_SUCCESS;
            }
            else
            {
               ETG_TRACE_ERR(("EvoBtStackWrapper: Failed to sendRequestPhoneNumberRequest"));
            }
         }
         else
         {
            ETG_TRACE_ERR(("sendRequestPhoneNumberRequest:No object path for BdAddress:%s", deviceAddress.c_str()));
         }
      }
      else
      {
         ETG_TRACE_ERR(("sendRequestPhoneNumberRequest: _hfpHandsfreeRequestIf invalid instance"));
      }

      return btsResult;
   }

   void EvoBtStackWrapper::onRequestPhoneNumberError(const ObjectPath& objPath, const ErrorName& errorName,
         const ErrorMessage& errorMessage, const act_t act)
   {
      //Send Error response to PmCoreMainController
      BdAddress deviceAddress;

      ObjectPathInfoListHandler::getInstance().getDeviceAddress(objPath, deviceAddress);
      if(false == deviceAddress.empty())
      {
         BTSResult btsResult(BTS_REQ_FAILED, ::ccdbusif::evolution::ERROR_UNKNOWN, "");
         ActType pmCoreAct = PM_DEFAULT_ACT;
         TelephoneNumber telephoneNumber;

         btsResult._errorCode = getErrorCode2Enum(errorName, errorMessage);
         btsResult._errorMessage = errorMessage;

         retrievePmCoreActFromDbusAct(act, pmCoreAct);

         std::shared_ptr<PmCoreIfMessage> pmCoreIfMessage = getNewPmCoreIfMessage_RequestPhoneNumberResult(
               btsResult, deviceAddress, telephoneNumber, pmCoreAct, PM_CORE_IF_MSG_ORIGIN_CLIENT);
         pmCoreIfMessage->traceMessage();
         PmCoreMainController::getInstance().handlePmCoreIfMessage(pmCoreIfMessage);
      }
      else
      {
         ETG_TRACE_ERR(("onRequestPhoneNumberError: No valid BdAddress for object path: %s", objPath.c_str()));
      }
   }

   void EvoBtStackWrapper::onRequestPhoneNumberResponse(const ObjectPath& objPath, const std::string& telephoneNumber,
         const act_t act)
   {
      //Send Success response to PmCoreMainController
      BdAddress deviceAddress;

      ObjectPathInfoListHandler::getInstance().getDeviceAddress(objPath, deviceAddress);
      if(false == deviceAddress.empty())
      {
         BTSResult btsResult(BTS_REQ_SUCCESS, ::ccdbusif::evolution::ERROR_UNKNOWN, "");
         ActType pmCoreAct = PM_DEFAULT_ACT;

         retrievePmCoreActFromDbusAct(act, pmCoreAct);

         std::shared_ptr<PmCoreIfMessage> pmCoreIfMessage = getNewPmCoreIfMessage_RequestPhoneNumberResult(
               btsResult, deviceAddress, telephoneNumber, pmCoreAct, PM_CORE_IF_MSG_ORIGIN_CLIENT);
         pmCoreIfMessage->traceMessage();
         PmCoreMainController::getInstance().handlePmCoreIfMessage(pmCoreIfMessage);
      }
      else
      {
         ETG_TRACE_ERR(("onRequestPhoneNumberError: No valid BdAddress for object path: %s", objPath.c_str()));
      }
   }

   BTSResult EvoBtStackWrapper::sendGetSubscriberRequest(const BdAddress& deviceAddress, const ActType act)
   {
      BTSResult btsResult(BTS_REQ_FAILED, ::ccdbusif::evolution::ERROR_UNKNOWN, "");
      act_t dbusAct = ::ccdbusif::DEFAULT_ACT;

      if(nullptr != _hfpHandsfreeRequestIf)
      {
         ObjectPath objPath;
         ObjectPathInfoListHandler::getInstance().getHandsfreeObjectPath(deviceAddress, objPath);

         if(false == objPath.empty())
         {
            dbusAct = _hfpHandsfreeRequestIf->sendGetSubscriberRequest(objPath);

            if(::ccdbusif::DEFAULT_ACT != dbusAct)
            {
               mapDbusActToPmCoreAct(dbusAct, act);
               btsResult._btsRequestResult = BTS_REQ_SUCCESS;
            }
            else
            {
               ETG_TRACE_ERR(("EvoBtStackWrapper: Failed to sendGetSubscriberRequest"));
            }
         }
         else
         {
            ETG_TRACE_ERR(("sendGetSubscriberRequest: No object path for BdAddress:%s", deviceAddress.c_str()));
         }
      }
      else
      {
         ETG_TRACE_ERR(("sendGetSubscriberRequest: _hfpHandsfreeRequestIf invalid instance"));
      }

      return btsResult;
   }

   void EvoBtStackWrapper::onGetSubscriberError(const ObjectPath& objPath, const ErrorName& errorName,
         const ErrorMessage& errorMessage, const act_t act)
   {
      //Send Error response to PmCoreMainController
      BdAddress deviceAddress;

      ObjectPathInfoListHandler::getInstance().getDeviceAddress(objPath, deviceAddress);
      if(false == deviceAddress.empty())
      {
         BTSResult btsResult(BTS_REQ_FAILED, ::ccdbusif::evolution::ERROR_UNKNOWN, "");
         ActType pmCoreAct = PM_DEFAULT_ACT;
         NumberType  numberType = NUMBER_TYPE_UNKNOWN;
         TelephoneNumber telephoneNumber;

         btsResult._errorCode = getErrorCode2Enum(errorName, errorMessage);
         btsResult._errorMessage = errorMessage;

         retrievePmCoreActFromDbusAct(act, pmCoreAct);

         std::shared_ptr<PmCoreIfMessage> pmCoreIfMessage = getNewPmCoreIfMessage_GetSubscriberResult(btsResult,
               deviceAddress, numberType, telephoneNumber, pmCoreAct, PM_CORE_IF_MSG_ORIGIN_CLIENT);
         pmCoreIfMessage->traceMessage();
         PmCoreMainController::getInstance().handlePmCoreIfMessage(pmCoreIfMessage);
      }
      else
      {
         ETG_TRACE_ERR(("onGetSubscriberError: No valid BdAddress for object path: %s", objPath.c_str()));
      }
   }

   void EvoBtStackWrapper::onGetSubscriberResponse(const ObjectPath& objPath, const uint8_t type,
         const std::string& telephoneNumber, const act_t act)
   {
      //Send Success response to PmCoreMainController
      BdAddress deviceAddress;

      ObjectPathInfoListHandler::getInstance().getDeviceAddress(objPath, deviceAddress);
      if(false == deviceAddress.empty())
      {
         BTSResult btsResult(BTS_REQ_SUCCESS, ::ccdbusif::evolution::ERROR_UNKNOWN, "");
         ActType pmCoreAct = PM_DEFAULT_ACT;

         retrievePmCoreActFromDbusAct(act, pmCoreAct);

         std::shared_ptr<PmCoreIfMessage> pmCoreIfMessage = getNewPmCoreIfMessage_GetSubscriberResult(btsResult,
               deviceAddress, type, telephoneNumber, pmCoreAct, PM_CORE_IF_MSG_ORIGIN_CLIENT);
         pmCoreIfMessage->traceMessage();
         PmCoreMainController::getInstance().handlePmCoreIfMessage(pmCoreIfMessage);
      }
      else
      {
         ETG_TRACE_ERR(("onGetSubscriberResponse: No valid BdAddress for object path: %s", objPath.c_str()));
      }
   }

   BTSResult EvoBtStackWrapper::sendSendCustomATCommandRequest(const BdAddress& deviceAddress,
         const ::std::vector< uint8 >& data, const ActType act)
   {
      BTSResult btsResult(BTS_REQ_FAILED, ::ccdbusif::evolution::ERROR_UNKNOWN, "");
      act_t dbusAct = ::ccdbusif::DEFAULT_ACT;

      if(nullptr != _hfpHandsfreeRequestIf)
      {
         ObjectPath objPath;
         ObjectPathInfoListHandler::getInstance().getHandsfreeObjectPath(deviceAddress, objPath);

         if(false == objPath.empty())
         {
            dbusAct = _hfpHandsfreeRequestIf->sendSendCustomATCommandRequest(objPath, data);

            if(::ccdbusif::DEFAULT_ACT != dbusAct)
            {
               mapDbusActToPmCoreAct(dbusAct, act);
               btsResult._btsRequestResult = BTS_REQ_SUCCESS;
            }
            else
            {
               ETG_TRACE_ERR(("EvoBtStackWrapper: Failed to sendSendCustomATCommandRequest"));
            }
         }
         else
         {
            ETG_TRACE_ERR(("sendSendCustomATCommandRequest: No object path for BdAddress:%s", deviceAddress.c_str()));
         }
      }
      else
      {
         ETG_TRACE_ERR(("sendSendCustomATCommandRequest: _hfpHandsfreeRequestIf invalid instance"));
      }

      return btsResult;
   }

   void EvoBtStackWrapper::onSendCustomATCommandError(const ObjectPath& objPath, const ErrorName& errorName,
         const ErrorMessage& errorMessage, const act_t act)
   {
      //Send Error response to PmCoreMainController
      BdAddress deviceAddress;

      ObjectPathInfoListHandler::getInstance().getDeviceAddress(objPath, deviceAddress);
      if(false == deviceAddress.empty())
      {
         BTSResult btsResult(BTS_REQ_FAILED, ::ccdbusif::evolution::ERROR_UNKNOWN, "");
         ActType pmCoreAct = PM_DEFAULT_ACT;

         btsResult._errorCode = getErrorCode2Enum(errorName, errorMessage);
         btsResult._errorMessage = errorMessage;

         retrievePmCoreActFromDbusAct(act, pmCoreAct);

         std::shared_ptr<PmCoreIfMessage> pmCoreIfMessage = getNewPmCoreIfMessage_SendCustomATCommandResult(
               btsResult, deviceAddress, pmCoreAct, PM_CORE_IF_MSG_ORIGIN_CLIENT);
         pmCoreIfMessage->traceMessage();
         PmCoreMainController::getInstance().handlePmCoreIfMessage(pmCoreIfMessage);
      }
      else
      {
         ETG_TRACE_ERR(("onSendCustomATCommandError: No valid BdAddress for object path: %s", objPath.c_str()));
      }
   }

   void EvoBtStackWrapper::onSendCustomATCommandResponse(const ObjectPath& objPath, const act_t act)
   {
      //Send Success response to PmCoreMainController
      BdAddress deviceAddress;

      ObjectPathInfoListHandler::getInstance().getDeviceAddress(objPath, deviceAddress);
      if(false == deviceAddress.empty())
      {
         BTSResult btsResult(BTS_REQ_SUCCESS, ::ccdbusif::evolution::ERROR_UNKNOWN, "");
         ActType pmCoreAct = PM_DEFAULT_ACT;

         retrievePmCoreActFromDbusAct(act, pmCoreAct);

         std::shared_ptr<PmCoreIfMessage> pmCoreIfMessage = getNewPmCoreIfMessage_SendCustomATCommandResult(
              btsResult, deviceAddress, pmCoreAct, PM_CORE_IF_MSG_ORIGIN_CLIENT);
         pmCoreIfMessage->traceMessage();
         PmCoreMainController::getInstance().handlePmCoreIfMessage(pmCoreIfMessage);
      }
      else
      {
         ETG_TRACE_ERR(("onSendCustomATCommandResponse: No valid BdAddress for object path: %s", objPath.c_str()));
      }
   }

   BTSResult EvoBtStackWrapper::sendHFUpdateIndicatorsValueRequest(const BdAddress& deviceAddress, uint16 number,
         uint32 value, const ActType act)
   {
      BTSResult btsResult(BTS_REQ_FAILED, ::ccdbusif::evolution::ERROR_UNKNOWN, "");
      act_t dbusAct = ::ccdbusif::DEFAULT_ACT;

      if(nullptr != _hfpHandsfreeRequestIf)
      {
         ObjectPath objPath;
         ObjectPathInfoListHandler::getInstance().getHandsfreeObjectPath(deviceAddress, objPath);

         if(false == objPath.empty())
         {
            dbusAct = _hfpHandsfreeRequestIf->sendHFUpdateIndicatorsValueRequest(objPath, number, value);

            if(::ccdbusif::DEFAULT_ACT != dbusAct)
            {
               mapDbusActToPmCoreAct(dbusAct, act);
               btsResult._btsRequestResult = BTS_REQ_SUCCESS;
            }
            else
            {
               ETG_TRACE_ERR(("EvoBtStackWrapper: Failed to sendHFUpdateIndicatorsValueRequest"));
            }
         }
         else
         {
            ETG_TRACE_ERR(("sendHFUpdateIndicatorsValueRequest: No objpath for BdAddress: %s",deviceAddress.c_str()));
         }
      }
      else
      {
         ETG_TRACE_ERR(("sendHFUpdateIndicatorsValueRequest: _hfpHandsfreeRequestIf invalid instance"));
      }

      return btsResult;
   }

   void EvoBtStackWrapper::onHFUpdateIndicatorsValueError(const ObjectPath& objPath, const ErrorName& errorName,
         const ErrorMessage& errorMessage, const act_t act)
   {
      //Send Error response to PmCoreMainController
      BdAddress deviceAddress;

      ObjectPathInfoListHandler::getInstance().getDeviceAddress(objPath, deviceAddress);
      if(false == deviceAddress.empty())
      {
         BTSResult btsResult(BTS_REQ_FAILED, ::ccdbusif::evolution::ERROR_UNKNOWN, "");
         ActType pmCoreAct = PM_DEFAULT_ACT;

         btsResult._errorCode = getErrorCode2Enum(errorName, errorMessage);
         btsResult._errorMessage = errorMessage;

         retrievePmCoreActFromDbusAct(act, pmCoreAct);

         std::shared_ptr<PmCoreIfMessage> pmCoreIfMessage = getNewPmCoreIfMessage_HFUpdateIndicatorsValueResult(
               btsResult, deviceAddress, pmCoreAct, PM_CORE_IF_MSG_ORIGIN_CLIENT);
         pmCoreIfMessage->traceMessage();
         PmCoreMainController::getInstance().handlePmCoreIfMessage(pmCoreIfMessage);
      }
      else
      {
         ETG_TRACE_ERR(("onHFUpdateIndicatorsValueError: No valid BdAddress for object path: %s", objPath.c_str()));
      }
   }

   void EvoBtStackWrapper::onHFUpdateIndicatorsValueResponse(const ObjectPath& objPath, const act_t act)
   {
      //Send Success response to PmCoreMainController
      BdAddress deviceAddress;

      ObjectPathInfoListHandler::getInstance().getDeviceAddress(objPath, deviceAddress);
      if(false == deviceAddress.empty())
      {
         BTSResult btsResult(BTS_REQ_SUCCESS, ::ccdbusif::evolution::ERROR_UNKNOWN, "");
         ActType pmCoreAct = PM_DEFAULT_ACT;

         retrievePmCoreActFromDbusAct(act, pmCoreAct);

         std::shared_ptr<PmCoreIfMessage> pmCoreIfMessage = getNewPmCoreIfMessage_HFUpdateIndicatorsValueResult(
               btsResult, deviceAddress, pmCoreAct, PM_CORE_IF_MSG_ORIGIN_CLIENT);
         pmCoreIfMessage->traceMessage();
         PmCoreMainController::getInstance().handlePmCoreIfMessage(pmCoreIfMessage);
      }
      else
      {
         ETG_TRACE_ERR(("onHFUpdateIndicatorsValueResponse: No valid BdAddress for objpath: %s", objPath.c_str()));
      }
   }

   BTSResult EvoBtStackWrapper::sendGetModemsRequest(const ActType act)
   {
      BTSResult btsResult(BTS_REQ_FAILED, ::ccdbusif::evolution::ERROR_UNKNOWN, "");
      act_t dbusAct = ::ccdbusif::DEFAULT_ACT;

      if(nullptr != _hfpManagerRequestIf)
      {
         dbusAct = _hfpManagerRequestIf->sendGetModemsRequest();

         if(::ccdbusif::DEFAULT_ACT != dbusAct)
         {
            mapDbusActToPmCoreAct(dbusAct, act);
            btsResult._btsRequestResult = BTS_REQ_SUCCESS;
         }
         else
         {
            ETG_TRACE_ERR(("EvoBtStackWrapper: Failed to sendGetModemsRequest"));
         }
      }
      else
      {
         ETG_TRACE_ERR(("sendGetModemsRequest: _hfpManagerRequestIf invalid instance"));
      }

      return btsResult;
   }

   void EvoBtStackWrapper::onGetModemsError(const ObjectPath& objPath, const ErrorName& errorName,
         const ErrorMessage& errorMessage, const act_t act)
   {
      //Send Error response to PmCoreMainController
      BdAddress deviceAddress;

      ObjectPathInfoListHandler::getInstance().getDeviceAddress(objPath, deviceAddress);
      if(false == deviceAddress.empty())
      {
         BTSResult btsResult(BTS_REQ_FAILED, ::ccdbusif::evolution::ERROR_UNKNOWN, "");
         ActType pmCoreAct = PM_DEFAULT_ACT;

         btsResult._errorCode = getErrorCode2Enum(errorName, errorMessage);
         btsResult._errorMessage = errorMessage;

         retrievePmCoreActFromDbusAct(act, pmCoreAct);

         std::shared_ptr<PmCoreIfMessage> pmCoreIfMessage = getNewPmCoreIfMessage_GetModemsResult(btsResult,
               deviceAddress, pmCoreAct, PM_CORE_IF_MSG_ORIGIN_CLIENT);
         pmCoreIfMessage->traceMessage();
         PmCoreMainController::getInstance().handlePmCoreIfMessage(pmCoreIfMessage);
      }
      else
      {
         ETG_TRACE_ERR(("onGetModemsError: No valid BdAddress for object path: %s", objPath.c_str()));
      }
   }

   void EvoBtStackWrapper::onGetModemsResponse(const ObjectPath& objPath,
         /*const ::std::vector< GetModemsResponseModemsStruct >& modems,*/ const act_t act)
   {
      //Send Success response to PmCoreMainController
      BdAddress deviceAddress;

      ObjectPathInfoListHandler::getInstance().getDeviceAddress(objPath, deviceAddress);
      if(false == deviceAddress.empty())
      {
         BTSResult btsResult(BTS_REQ_SUCCESS, ::ccdbusif::evolution::ERROR_UNKNOWN, "");
         ActType pmCoreAct = PM_DEFAULT_ACT;

         retrievePmCoreActFromDbusAct(act, pmCoreAct);

         std::shared_ptr<PmCoreIfMessage> pmCoreIfMessage = getNewPmCoreIfMessage_GetModemsResult(btsResult,
               deviceAddress, pmCoreAct, PM_CORE_IF_MSG_ORIGIN_CLIENT);
         pmCoreIfMessage->traceMessage();
         PmCoreMainController::getInstance().handlePmCoreIfMessage(pmCoreIfMessage);
      }
      else
      {
         ETG_TRACE_ERR(("onGetModemsResponse: No valid BdAddress for object path: %s", objPath.c_str()));
      }
   }

   BTSResult EvoBtStackWrapper::sendAcceptSCOConnectRequest(const BdAddress& deviceAddress, const ActType act)
   {
      BTSResult btsResult(BTS_REQ_FAILED, ::ccdbusif::evolution::ERROR_UNKNOWN, "");
      act_t dbusAct = ::ccdbusif::DEFAULT_ACT;

      if(nullptr != _hfpModemRequestIf)
      {
         ObjectPath objPath;
         ObjectPathInfoListHandler::getInstance().getHandsfreeObjectPath(deviceAddress, objPath);

         if(false == objPath.empty())
         {
            dbusAct = _hfpModemRequestIf->sendAcceptSCOConnectRequest(objPath);

            if(::ccdbusif::DEFAULT_ACT != dbusAct)
            {
               mapDbusActToPmCoreAct(dbusAct, act);
               btsResult._btsRequestResult = BTS_REQ_SUCCESS;
            }
            else
            {
               ETG_TRACE_ERR(("EvoBtStackWrapper: Failed to sendAcceptSCOConnectRequest"));
            }
         }
         else
         {
            ETG_TRACE_ERR(("sendAcceptSCOConnectRequest: No object path for BdAddress:%s", deviceAddress.c_str()));
         }
      }
      else
      {
         ETG_TRACE_ERR(("sendAcceptSCOConnectRequest: _hfpModemRequestIf invalid instance"));
      }

      return btsResult;
   }

   void EvoBtStackWrapper::onAcceptSCOConnectError(const ObjectPath& objPath, const ErrorName& errorName,
         const ErrorMessage& errorMessage, const act_t act)
   {
      //Send Error response to PmCoreMainController
      BdAddress deviceAddress;

      ObjectPathInfoListHandler::getInstance().getDeviceAddress(objPath, deviceAddress);
      if(false == deviceAddress.empty())
      {
         BTSResult btsResult(BTS_REQ_FAILED, ::ccdbusif::evolution::ERROR_UNKNOWN, "");
         ActType pmCoreAct = PM_DEFAULT_ACT;

         btsResult._errorCode = getErrorCode2Enum(errorName, errorMessage);
         btsResult._errorMessage = errorMessage;

         retrievePmCoreActFromDbusAct(act, pmCoreAct);

         std::shared_ptr<PmCoreIfMessage> pmCoreIfMessage = getNewPmCoreIfMessage_AcceptSCOConnectResult(
               btsResult, deviceAddress, pmCoreAct, PM_CORE_IF_MSG_ORIGIN_CLIENT);
         pmCoreIfMessage->traceMessage();
         PmCoreMainController::getInstance().handlePmCoreIfMessage(pmCoreIfMessage);
      }
      else
      {
         ETG_TRACE_ERR(("onAcceptSCOConnectError: No valid BdAddress for object path: %s", objPath.c_str()));
      }
   }

   void EvoBtStackWrapper::onAcceptSCOConnectResponse(const ObjectPath& objPath, const act_t act)
   {
      //Send Success response to PmCoreMainController
      BdAddress deviceAddress;

      ObjectPathInfoListHandler::getInstance().getDeviceAddress(objPath, deviceAddress);
      if(false == deviceAddress.empty())
      {
         BTSResult btsResult(BTS_REQ_SUCCESS, ::ccdbusif::evolution::ERROR_UNKNOWN, "");
         ActType pmCoreAct = PM_DEFAULT_ACT;

         retrievePmCoreActFromDbusAct(act, pmCoreAct);

         std::shared_ptr<PmCoreIfMessage> pmCoreIfMessage = getNewPmCoreIfMessage_AcceptSCOConnectResult(btsResult,
               deviceAddress, pmCoreAct, PM_CORE_IF_MSG_ORIGIN_CLIENT);
         pmCoreIfMessage->traceMessage();
         PmCoreMainController::getInstance().handlePmCoreIfMessage(pmCoreIfMessage);
      }
      else
      {
         ETG_TRACE_ERR(("onAcceptSCOConnectResponse: No valid BdAddress for object path: %s", objPath.c_str()));
      }
   }

   void EvoBtStackWrapper::onSCOConnectRequestSignal(const ObjectPath& objPath, const uint8_t type)
   {
      BdAddress deviceAddress;

      ObjectPathInfoListHandler::getInstance().getDeviceAddress(objPath, deviceAddress);
      if(false == deviceAddress.empty())
      {
         std::shared_ptr<PropertyDetails<uint8_t>> propertyDetails =
               std::make_shared<PropertyDetails<uint8_t>>(deviceAddress, type);

         PropertyUpdateNotifierToCore::getInstance().notifyControllers(BTS_SIGNAL_SCO_CONNECT_REQUEST, propertyDetails);
      }
      else
      {
         ETG_TRACE_ERR(("onSCOConnectRequestSignal(): No device found"));
      }
   }

   BTSResult EvoBtStackWrapper::sendSiriEnableRequest(const BdAddress& deviceAddress, uint16 support_type, const ActType act)
   {
      BTSResult btsResult(BTS_REQ_FAILED, ::ccdbusif::evolution::ERROR_UNKNOWN, "");
      act_t dbusAct = ::ccdbusif::DEFAULT_ACT;

      if(nullptr != _hfpSiriRequestIf)
      {
         ObjectPath objPath;
         ObjectPathInfoListHandler::getInstance().getHandsfreeObjectPath(deviceAddress, objPath);

         if(false == objPath.empty())
         {
            dbusAct = _hfpSiriRequestIf->sendEnableRequest(objPath, _vehicleConfig._vendorID,
                         _vehicleConfig._productID, _vehicleConfig._productVersion,
                  support_type);

            if(::ccdbusif::DEFAULT_ACT != dbusAct)
            {
               mapDbusActToPmCoreAct(dbusAct, act);
               btsResult._btsRequestResult = BTS_REQ_SUCCESS;
            }
            else
            {
               ETG_TRACE_ERR(("EvoBtStackWrapper: Failed to sendSiriEnableRequest"));
            }
         }
         else
         {
            ETG_TRACE_ERR(("sendSiriEnableRequest: No object path for BdAddress:%s", deviceAddress.c_str()));
         }
      }
      else
      {
         ETG_TRACE_ERR(("sendSiriEnableRequest: _hfpSiriRequestIf invalid instance"));
      }

      return btsResult;
   }

   void EvoBtStackWrapper::onSiriEnableError(const ObjectPath& objPath, const ErrorName& errorName,
         const ErrorMessage& errorMessage, const act_t act)
   {
      //Send Error response to PmCoreMainController
      BdAddress deviceAddress;

      ObjectPathInfoListHandler::getInstance().getDeviceAddress(objPath, deviceAddress);
      if(false == deviceAddress.empty())
      {
         BTSResult btsResult(BTS_REQ_FAILED, ::ccdbusif::evolution::ERROR_UNKNOWN, "");
         ActType pmCoreAct = PM_DEFAULT_ACT;

         btsResult._errorCode = getErrorCode2Enum(errorName, errorMessage);
         btsResult._errorMessage = errorMessage;

         retrievePmCoreActFromDbusAct(act, pmCoreAct);

         std::shared_ptr<PmCoreIfMessage> pmCoreIfMessage = getNewPmCoreIfMessage_SiriEnableResult(btsResult,
               deviceAddress, pmCoreAct, PM_CORE_IF_MSG_ORIGIN_CLIENT);
         pmCoreIfMessage->traceMessage();
         PmCoreMainController::getInstance().handlePmCoreIfMessage(pmCoreIfMessage);
      }
      else
      {
         ETG_TRACE_ERR(("onSiriEnableError: No valid BdAddress for object path: %s", objPath.c_str()));
      }
   }

   void EvoBtStackWrapper::onSiriEnableResponse(const ObjectPath& objPath, const act_t act)
   {
      //Send Success response to PmCoreMainController
      BdAddress deviceAddress;

      ObjectPathInfoListHandler::getInstance().getDeviceAddress(objPath, deviceAddress);
      if(false == deviceAddress.empty())
      {
         BTSResult btsResult(BTS_REQ_SUCCESS, ::ccdbusif::evolution::ERROR_UNKNOWN, "");
         ActType pmCoreAct = PM_DEFAULT_ACT;

         retrievePmCoreActFromDbusAct(act, pmCoreAct);

         std::shared_ptr<PmCoreIfMessage> pmCoreIfMessage = getNewPmCoreIfMessage_SiriEnableResult(btsResult,
               deviceAddress, pmCoreAct, PM_CORE_IF_MSG_ORIGIN_CLIENT);
         pmCoreIfMessage->traceMessage();
         PmCoreMainController::getInstance().handlePmCoreIfMessage(pmCoreIfMessage);
      }
      else
      {
         ETG_TRACE_ERR(("onSiriEnableResponse: No valid BdAddress for object path: %s", objPath.c_str()));
      }
   }

   BTSResult EvoBtStackWrapper::sendSiriSetNRRequest(const BdAddress& deviceAddress, uint8 status, const ActType act)
   {
      BTSResult btsResult(BTS_REQ_FAILED, ::ccdbusif::evolution::ERROR_UNKNOWN, "");
      act_t dbusAct = ::ccdbusif::DEFAULT_ACT;

      if(nullptr != _hfpSiriRequestIf)
      {
         ObjectPath objPath;
         ObjectPathInfoListHandler::getInstance().getHandsfreeObjectPath(deviceAddress, objPath);

         if(false == objPath.empty())
         {
            dbusAct = _hfpSiriRequestIf->sendSetNRRequest(objPath, status);

            if(::ccdbusif::DEFAULT_ACT != dbusAct)
            {
               mapDbusActToPmCoreAct(dbusAct, act);
               btsResult._btsRequestResult = BTS_REQ_SUCCESS;
            }
            else
            {
               ETG_TRACE_ERR(("EvoBtStackWrapper: Failed to sendSiriSetNRRequest"));
            }
         }
         else
         {
            ETG_TRACE_ERR(("sendSiriSetNRRequest: No object path for BdAddress:%s", deviceAddress.c_str()));
         }
      }
      else
      {
         ETG_TRACE_ERR(("sendSiriSetNRRequest: _hfpSiriRequestIf invalid instance"));
      }

      return btsResult;
   }

   void EvoBtStackWrapper::onSiriSetNRError(const ObjectPath& objPath, const ErrorName& errorName,
         const ErrorMessage& errorMessage, const act_t act)
   {
      //Send Error response to PmCoreMainController
      BdAddress deviceAddress;

      ObjectPathInfoListHandler::getInstance().getDeviceAddress(objPath, deviceAddress);
      if(false == deviceAddress.empty())
      {
         BTSResult btsResult(BTS_REQ_FAILED, ::ccdbusif::evolution::ERROR_UNKNOWN, "");
         ActType pmCoreAct = PM_DEFAULT_ACT;

         btsResult._errorCode = getErrorCode2Enum(errorName, errorMessage);
         btsResult._errorMessage = errorMessage;

         retrievePmCoreActFromDbusAct(act, pmCoreAct);

         std::shared_ptr<PmCoreIfMessage> pmCoreIfMessage = getNewPmCoreIfMessage_SiriSetNRResult(btsResult,
               deviceAddress, pmCoreAct, PM_CORE_IF_MSG_ORIGIN_CLIENT);
         pmCoreIfMessage->traceMessage();
         PmCoreMainController::getInstance().handlePmCoreIfMessage(pmCoreIfMessage);
      }
      else
      {
         ETG_TRACE_ERR(("onSiriSetNRError: No valid BdAddress for object path: %s", objPath.c_str()));
      }
   }

   void EvoBtStackWrapper::onSiriSetNRResponse(const ObjectPath& objPath, const act_t act)
   {
      //Send Success response to PmCoreMainController
      BdAddress deviceAddress;

      ObjectPathInfoListHandler::getInstance().getDeviceAddress(objPath, deviceAddress);
      if(false == deviceAddress.empty())
      {
         BTSResult btsResult(BTS_REQ_SUCCESS, ::ccdbusif::evolution::ERROR_UNKNOWN, "");
         ActType pmCoreAct = PM_DEFAULT_ACT;

         retrievePmCoreActFromDbusAct(act, pmCoreAct);

         std::shared_ptr<PmCoreIfMessage> pmCoreIfMessage = getNewPmCoreIfMessage_SiriSetNRResult(btsResult,
               deviceAddress, pmCoreAct, PM_CORE_IF_MSG_ORIGIN_CLIENT);
         pmCoreIfMessage->traceMessage();
         PmCoreMainController::getInstance().handlePmCoreIfMessage(pmCoreIfMessage);
      }
      else
      {
         ETG_TRACE_ERR(("onSiriSetNRResponse: No valid BdAddress for object path: %s", objPath.c_str()));
      }
   }

   BTSResult EvoBtStackWrapper::sendGetCallsRequest(const BdAddress& deviceAddress, const ActType act)
   {
      BTSResult btsResult(BTS_REQ_FAILED, ::ccdbusif::evolution::ERROR_UNKNOWN, "");
      act_t dbusAct = ::ccdbusif::DEFAULT_ACT;

      if(nullptr != _hfpVoiceCallManagerRequestIf)
      {
         ObjectPath objPath;
         ObjectPathInfoListHandler::getInstance().getHandsfreeObjectPath(deviceAddress, objPath);

         if(false == objPath.empty())
         {
            dbusAct = _hfpVoiceCallManagerRequestIf->sendGetCallsRequest(objPath);

            if(::ccdbusif::DEFAULT_ACT != dbusAct)
            {
               mapDbusActToPmCoreAct(dbusAct, act);
               btsResult._btsRequestResult = BTS_REQ_SUCCESS;
            }
            else
            {
               ETG_TRACE_ERR(("EvoBtStackWrapper: Failed to sendGetCallsRequest"));
            }
         }
         else
         {
            ETG_TRACE_ERR(("sendGetCallsRequest: No object path for BdAddress:%s", deviceAddress.c_str()));
         }
      }
      else
      {
         ETG_TRACE_ERR(("sendGetCallsRequest: _hfpVoiceCallManagerRequestIf invalid instance"));
      }

      return btsResult;
   }

   void EvoBtStackWrapper::onGetCallsError(const ObjectPath& objPath, const ErrorName& errorName,
         const ErrorMessage& errorMessage, const act_t act)
   {
      //Send Error response to PmCoreMainController
      BdAddress deviceAddress;

      ObjectPathInfoListHandler::getInstance().getDeviceAddress(objPath, deviceAddress);
      if(false == deviceAddress.empty())
      {
         BTSResult btsResult(BTS_REQ_FAILED, ::ccdbusif::evolution::ERROR_UNKNOWN, "");
         ActType pmCoreAct = PM_DEFAULT_ACT;
         BTSVoiceCallInfoList btsVoiceCallInfoList;

         btsResult._errorCode = getErrorCode2Enum(errorName, errorMessage);
         btsResult._errorMessage = errorMessage;

         retrievePmCoreActFromDbusAct(act, pmCoreAct);

         std::shared_ptr<PmCoreIfMessage> pmCoreIfMessage = getNewPmCoreIfMessage_GetCallsResult(btsResult,
               deviceAddress, btsVoiceCallInfoList, pmCoreAct, PM_CORE_IF_MSG_ORIGIN_CLIENT);
         pmCoreIfMessage->traceMessage();
         PmCoreMainController::getInstance().handlePmCoreIfMessage(pmCoreIfMessage);
      }
      else
      {
         ETG_TRACE_ERR(("onGetCallsError: No valid BdAddress for object path: %s", objPath.c_str()));
      }
   }

   void EvoBtStackWrapper::onGetCallsResponse(const ObjectPath& objPath,
         const ::std::map< ObjectPath, BTSDbusPropertyList >& calls, const act_t act)
   {
      //set voice call object path
      if(nullptr != _hfpVoiceCallRequestIf)
      {
         //Send Success response to PmCoreMainController
         BdAddress deviceAddress;

         ObjectPathInfoListHandler::getInstance().getDeviceAddress(objPath, deviceAddress);
         if(false == deviceAddress.empty())
         {
            BTSResult btsResult(BTS_REQ_SUCCESS, ::ccdbusif::evolution::ERROR_UNKNOWN, "");
            ActType pmCoreAct = PM_DEFAULT_ACT;
            BTSVoiceCallInfoList btsVoiceCallInfoList;

            for(auto it = calls.begin(); it != calls.end(); it++)
            {
               CallInstance callInstance = getInstanceFromObjectPath(it->first, VOICECALL_OBJECTPATH_LENGTH);

               if(DEFAULT_INSTANCE_VALUE != callInstance)
               {
                  ObjectPathInfoListHandler::getInstance().setVoiceCallObjectPath(deviceAddress, callInstance,
                        it->first);

                  (void)_hfpVoiceCallRequestIf->voiceCallCreated(it->first);

                  BTSVoiceCallInfo btsVoiceCallInfo;
                  btsVoiceCallInfo._callInfo._instance = callInstance;

                  convertBTSDbusPropertyListToBTSVoiceCallInfo(it->second, btsVoiceCallInfo);

                  btsVoiceCallInfoList.push_back(btsVoiceCallInfo);
               }
            }

            retrievePmCoreActFromDbusAct(act, pmCoreAct);

            std::shared_ptr<PmCoreIfMessage> pmCoreIfMessage = getNewPmCoreIfMessage_GetCallsResult(btsResult,
                  deviceAddress, btsVoiceCallInfoList, pmCoreAct, PM_CORE_IF_MSG_ORIGIN_CLIENT);
            pmCoreIfMessage->traceMessage();
            PmCoreMainController::getInstance().handlePmCoreIfMessage(pmCoreIfMessage);
         }
         else
         {
            ETG_TRACE_ERR(("onGetCallsResponse: No valid BdAddress for object path: %s", objPath.c_str()));
         }
      }
      else
      {
         ETG_TRACE_ERR(("onGetCallsResponse: _hfpVoiceCallRequestIf invalid instance"));
      }
   }

   BTSResult EvoBtStackWrapper::sendDialRequest(const BdAddress& deviceAddress, const TelephoneNumber& number,
         const HideCallerId hide_callerid, const ActType act)
   {
      BTSResult btsResult(BTS_REQ_FAILED, ::ccdbusif::evolution::ERROR_UNKNOWN, "");
      act_t dbusAct = ::ccdbusif::DEFAULT_ACT;

      if(nullptr != _hfpVoiceCallManagerRequestIf)
      {
         ObjectPath objPath;
         ::std::string callerId;

         ObjectPathInfoListHandler::getInstance().getHandsfreeObjectPath(deviceAddress, objPath);

         if(false == objPath.empty())
         {
            convertCallerIdToDbusType(hide_callerid, callerId);

            dbusAct = _hfpVoiceCallManagerRequestIf->sendDialRequest(objPath, number, callerId);

            if(::ccdbusif::DEFAULT_ACT != dbusAct)
            {
               mapDbusActToPmCoreAct(dbusAct, act);
               mapDbusActToTelephoneNumber(dbusAct, number);
               btsResult._btsRequestResult = BTS_REQ_SUCCESS;
            }
            else
            {
               ETG_TRACE_ERR(("EvoBtStackWrapper: Failed to sendDialRequest"));
            }
         }
         else
         {
            ETG_TRACE_ERR(("sendDialRequest: No object path for BdAddress:%s", deviceAddress.c_str()));
         }
      }
      else
      {
         ETG_TRACE_ERR(("sendDialRequest: _hfpVoiceCallManagerRequestIf invalid instance"));
      }

      return btsResult;
   }

   void EvoBtStackWrapper::onDialError(const ObjectPath& objPath, const ErrorName& errorName,
         const ErrorMessage& errorMessage, const act_t act)
   {
      //Send Error response to PmCoreMainController
      BdAddress deviceAddress;

      ObjectPathInfoListHandler::getInstance().getDeviceAddress(objPath, deviceAddress);
      if(false == deviceAddress.empty())
      {
         BTSResult btsResult(BTS_REQ_FAILED, ::ccdbusif::evolution::ERROR_UNKNOWN, "");
         ActType pmCoreAct = PM_DEFAULT_ACT;
         CallInstance callInstance = CALL_INSTANCE_DEFAULT;
         TelephoneNumber dialedNumber = "";

         btsResult._errorCode = getErrorCode2Enum(errorName, errorMessage);
         btsResult._errorMessage = errorMessage;

         retrievePmCoreActFromDbusAct(act, pmCoreAct);
         retrieveTelephoneNumberFromDbusAct(act, dialedNumber);

         std::shared_ptr<PmCoreIfMessage> pmCoreIfMessage = getNewPmCoreIfMessage_DialResult(btsResult,
               deviceAddress, callInstance, dialedNumber, pmCoreAct, PM_CORE_IF_MSG_ORIGIN_CLIENT);
         pmCoreIfMessage->traceMessage();
         PmCoreMainController::getInstance().handlePmCoreIfMessage(pmCoreIfMessage);
      }
      else
      {
         ETG_TRACE_ERR(("onDialError: No valid BdAddress for object path: %s", objPath.c_str()));
      }
   }

   void EvoBtStackWrapper::onDialResponse(const ObjectPath& objPath, const std::string& path, const act_t act)
   {
      //Send Error response to PmCoreMainController
      BdAddress deviceAddress;

      ObjectPathInfoListHandler::getInstance().getDeviceAddress(objPath, deviceAddress);
      if(false == deviceAddress.empty())
      {
         BTSResult   btsResult(BTS_REQ_SUCCESS, ::ccdbusif::evolution::ERROR_UNKNOWN, "");
         ActType     pmCoreAct = PM_DEFAULT_ACT;
         CallInstance callInstance = CALL_INSTANCE_DEFAULT;
         TelephoneNumber dialedNumber = "";

         retrievePmCoreActFromDbusAct(act, pmCoreAct);
         retrieveTelephoneNumberFromDbusAct(act, dialedNumber);
         callInstance = getInstanceFromObjectPath(path, VOICECALL_OBJECTPATH_LENGTH);

         std::shared_ptr<PmCoreIfMessage> pmCoreIfMessage = getNewPmCoreIfMessage_DialResult(btsResult,
               deviceAddress, callInstance, dialedNumber, pmCoreAct, PM_CORE_IF_MSG_ORIGIN_CLIENT);
         pmCoreIfMessage->traceMessage();
         PmCoreMainController::getInstance().handlePmCoreIfMessage(pmCoreIfMessage);
      }
      else
      {
         ETG_TRACE_ERR(("onDialResponse: No valid BdAddress for object path: %s", objPath.c_str()));
      }
   }

   BTSResult EvoBtStackWrapper::sendMemDialRequest(const BdAddress& deviceAddress, const MemDialIndex numberIndex,
         const HideCallerId hideCallerId, const ActType act)
   {
      BTSResult btsResult(BTS_REQ_FAILED, ::ccdbusif::evolution::ERROR_UNKNOWN, "");
      act_t dbusAct = ::ccdbusif::DEFAULT_ACT;

      if(nullptr != _hfpVoiceCallManagerRequestIf)
      {
         ObjectPath objPath;
         ::std::string callerId;

         ObjectPathInfoListHandler::getInstance().getHandsfreeObjectPath(deviceAddress, objPath);

         if(false == objPath.empty())
         {
            convertCallerIdToDbusType(hideCallerId, callerId);

            dbusAct = _hfpVoiceCallManagerRequestIf->sendMemDialRequest(objPath, (uint16)numberIndex, callerId);

            if(::ccdbusif::DEFAULT_ACT != dbusAct)
            {
               mapDbusActToPmCoreAct(dbusAct, act);
               btsResult._btsRequestResult = BTS_REQ_SUCCESS;
            }
            else
            {
               ETG_TRACE_ERR(("EvoBtStackWrapper: Failed to sendMemDialRequest"));
            }
         }
         else
         {
            ETG_TRACE_ERR(("sendMemDialRequest: No object path for BdAddress:%s", deviceAddress.c_str()));
         }
      }
      else
      {
         ETG_TRACE_ERR(("sendMemDialRequest: _hfpVoiceCallManagerRequestIf invalid instance"));
      }

      return btsResult;
   }

   void EvoBtStackWrapper::onMemDialError(const ObjectPath& objPath, const ErrorName& errorName,
         const ErrorMessage& errorMessage, const act_t act)
   {
      //Send Error response to PmCoreMainController
      BdAddress deviceAddress;

      ObjectPathInfoListHandler::getInstance().getDeviceAddress(objPath, deviceAddress);
      if(false == deviceAddress.empty())
      {
         BTSResult btsResult(BTS_REQ_FAILED, ::ccdbusif::evolution::ERROR_UNKNOWN, "");
         ActType   pmCoreAct = PM_DEFAULT_ACT;
         CallInstance callInstance = CALL_INSTANCE_DEFAULT;

         btsResult._errorCode = getErrorCode2Enum(errorName, errorMessage);
         btsResult._errorMessage = errorMessage;

         retrievePmCoreActFromDbusAct(act, pmCoreAct);

         std::shared_ptr<PmCoreIfMessage> pmCoreIfMessage = getNewPmCoreIfMessage_SpeedDialResult(btsResult,
               deviceAddress, callInstance, pmCoreAct, PM_CORE_IF_MSG_ORIGIN_CLIENT);
         pmCoreIfMessage->traceMessage();
         PmCoreMainController::getInstance().handlePmCoreIfMessage(pmCoreIfMessage);
      }
      else
      {
         ETG_TRACE_ERR(("onMemDialError: No valid BdAddress for object path: %s", objPath.c_str()));
      }
   }

   void EvoBtStackWrapper::onMemDialResponse(const ObjectPath& objPath, const std::string& path, const act_t act)
   {
      //Send Success response to PmCoreMainController
      BdAddress deviceAddress;

      ObjectPathInfoListHandler::getInstance().getDeviceAddress(objPath, deviceAddress);
      if(false == deviceAddress.empty())
      {
         BTSResult btsResult(BTS_REQ_SUCCESS, ::ccdbusif::evolution::ERROR_UNKNOWN, "");
         ActType pmCoreAct = PM_DEFAULT_ACT;
         CallInstance callInstance = CALL_INSTANCE_DEFAULT;

         retrievePmCoreActFromDbusAct(act, pmCoreAct);
         callInstance = getInstanceFromObjectPath(path, VOICECALL_OBJECTPATH_LENGTH);

         std::shared_ptr<PmCoreIfMessage> pmCoreIfMessage = getNewPmCoreIfMessage_SpeedDialResult(btsResult,
               deviceAddress, callInstance, pmCoreAct, PM_CORE_IF_MSG_ORIGIN_CLIENT);
         pmCoreIfMessage->traceMessage();
         PmCoreMainController::getInstance().handlePmCoreIfMessage(pmCoreIfMessage);
      }
      else
      {
         ETG_TRACE_ERR(("onMemDialResponse: No valid BdAddress for object path: %s", objPath.c_str()));
      }
   }

   BTSResult EvoBtStackWrapper::sendRedialRequest(const BdAddress& deviceAddress, const HideCallerId hide_callerid,
         const ActType act)
   {
      BTSResult btsResult(BTS_REQ_FAILED, ::ccdbusif::evolution::ERROR_UNKNOWN, "");
      act_t dbusAct = ::ccdbusif::DEFAULT_ACT;

      if(nullptr != _hfpVoiceCallManagerRequestIf)
      {
         ObjectPath objPath;
         ::std::string callerId;

         ObjectPathInfoListHandler::getInstance().getHandsfreeObjectPath(deviceAddress, objPath);

         if(false == objPath.empty())
         {
            convertCallerIdToDbusType(hide_callerid, callerId);

            dbusAct = _hfpVoiceCallManagerRequestIf->sendRedialRequest(objPath, callerId);

            if(::ccdbusif::DEFAULT_ACT != dbusAct)
            {
               mapDbusActToPmCoreAct(dbusAct, act);
               btsResult._btsRequestResult = BTS_REQ_SUCCESS;
            }
            else
            {
               ETG_TRACE_ERR(("EvoBtStackWrapper: Failed to sendRedialRequest"));
            }
         }
         else
         {
            ETG_TRACE_ERR(("sendRedialRequest: No object path for BdAddress:%s", deviceAddress.c_str()));
         }
      }
      else
      {
         ETG_TRACE_ERR(("sendRedialRequest: _hfpVoiceCallManagerRequestIf invalid instance"));
      }

      return btsResult;
   }

   void EvoBtStackWrapper::onRedialError(const ObjectPath& objPath, const ErrorName& errorName,
         const ErrorMessage& errorMessage, const act_t act)
   {
      //Send Error response to PmCoreMainController
      BdAddress deviceAddress;

      ObjectPathInfoListHandler::getInstance().getDeviceAddress(objPath, deviceAddress);
      if(false == deviceAddress.empty())
      {
         BTSResult btsResult(BTS_REQ_FAILED, ::ccdbusif::evolution::ERROR_UNKNOWN, "");
         ActType pmCoreAct = PM_DEFAULT_ACT;
         CallInstance callInstance = CALL_INSTANCE_DEFAULT;

         btsResult._errorCode = getErrorCode2Enum(errorName, errorMessage);
         btsResult._errorMessage = errorMessage;

         retrievePmCoreActFromDbusAct(act, pmCoreAct);

         std::shared_ptr<PmCoreIfMessage> pmCoreIfMessage = getNewPmCoreIfMessage_RedialResult(btsResult,
               deviceAddress, callInstance, pmCoreAct, PM_CORE_IF_MSG_ORIGIN_CLIENT);
         pmCoreIfMessage->traceMessage();
         PmCoreMainController::getInstance().handlePmCoreIfMessage(pmCoreIfMessage);
      }
      else
      {
         ETG_TRACE_ERR(("onRedialError: No valid BdAddress for object path: %s", objPath.c_str()));
      }
   }

   void EvoBtStackWrapper::onRedialResponse(const ObjectPath& objPath, const std::string& path, const act_t act)
   {
      //Send Success response to PmCoreMainController
      BdAddress deviceAddress;

      ObjectPathInfoListHandler::getInstance().getDeviceAddress(objPath, deviceAddress);
      if(false == deviceAddress.empty())
      {
         BTSResult btsResult(BTS_REQ_SUCCESS, ::ccdbusif::evolution::ERROR_UNKNOWN, "");
         ActType pmCoreAct = PM_DEFAULT_ACT;
         CallInstance callInstance = CALL_INSTANCE_DEFAULT;

         retrievePmCoreActFromDbusAct(act, pmCoreAct);
         callInstance = getInstanceFromObjectPath(path, VOICECALL_OBJECTPATH_LENGTH);

         std::shared_ptr<PmCoreIfMessage> pmCoreIfMessage = getNewPmCoreIfMessage_RedialResult(btsResult,
               deviceAddress, callInstance, pmCoreAct, PM_CORE_IF_MSG_ORIGIN_CLIENT);
         pmCoreIfMessage->traceMessage();
         PmCoreMainController::getInstance().handlePmCoreIfMessage(pmCoreIfMessage);
      }
      else
      {
         ETG_TRACE_ERR(("onRedialResponse: No valid BdAddress for object path: %s", objPath.c_str()));
      }
   }

   BTSResult EvoBtStackWrapper::sendSwapCallsRequest(const BdAddress& deviceAddress, const ActType act)
   {
      BTSResult btsResult(BTS_REQ_FAILED, ::ccdbusif::evolution::ERROR_UNKNOWN, "");
      act_t dbusAct = ::ccdbusif::DEFAULT_ACT;

      if(nullptr != _hfpVoiceCallManagerRequestIf)
      {
         ObjectPath objPath;

         ObjectPathInfoListHandler::getInstance().getHandsfreeObjectPath(deviceAddress, objPath);

         if(false == objPath.empty())
         {
            dbusAct = _hfpVoiceCallManagerRequestIf->sendSwapCallsRequest(objPath);

            if(::ccdbusif::DEFAULT_ACT != dbusAct)
            {
               mapDbusActToPmCoreAct(dbusAct, act);
               btsResult._btsRequestResult = BTS_REQ_SUCCESS;
            }
            else
            {
               ETG_TRACE_ERR(("EvoBtStackWrapper: Failed to sendSwapCallsRequest"));
            }
         }
         else
         {
            ETG_TRACE_ERR(("sendSwapCallsRequest: No object path for BdAddress:%s", deviceAddress.c_str()));
         }
      }
      else
      {
         ETG_TRACE_ERR(("sendSwapCallsRequest: _hfpVoiceCallManagerRequestIf invalid instance"));
      }

      return btsResult;
   }

   void EvoBtStackWrapper::onSwapCallsError(const ObjectPath& objPath, const ErrorName& errorName,
         const ErrorMessage& errorMessage, const act_t act)
   {
      //Send Error response to PmCoreMainController
      BdAddress deviceAddress;

      ObjectPathInfoListHandler::getInstance().getDeviceAddress(objPath, deviceAddress);
      if(false == deviceAddress.empty())
      {
         BTSResult btsResult(BTS_REQ_FAILED, ::ccdbusif::evolution::ERROR_UNKNOWN, "");
         ActType pmCoreAct = PM_DEFAULT_ACT;

         btsResult._errorCode = getErrorCode2Enum(errorName, errorMessage);
         btsResult._errorMessage = errorMessage;

         retrievePmCoreActFromDbusAct(act, pmCoreAct);

         std::shared_ptr<PmCoreIfMessage> pmCoreIfMessage = getNewPmCoreIfMessage_SwapCallResult(btsResult,
               deviceAddress, pmCoreAct, PM_CORE_IF_MSG_ORIGIN_CLIENT);
         pmCoreIfMessage->traceMessage();
         PmCoreMainController::getInstance().handlePmCoreIfMessage(pmCoreIfMessage);
      }
      else
      {
         ETG_TRACE_ERR(("onSwapCallsError: No valid BdAddress for object path: %s", objPath.c_str()));
      }
   }

   void EvoBtStackWrapper::onSwapCallsResponse(const ObjectPath& objPath, const act_t act)
   {
      //Send Success response to PmCoreMainController
      BdAddress deviceAddress;

      ObjectPathInfoListHandler::getInstance().getDeviceAddress(objPath, deviceAddress);
      if(false == deviceAddress.empty())
      {
         BTSResult btsResult(BTS_REQ_SUCCESS, ::ccdbusif::evolution::ERROR_UNKNOWN, "");
         ActType pmCoreAct = PM_DEFAULT_ACT;

         retrievePmCoreActFromDbusAct(act, pmCoreAct);

         std::shared_ptr<PmCoreIfMessage> pmCoreIfMessage = getNewPmCoreIfMessage_SwapCallResult(btsResult,
               deviceAddress, pmCoreAct, PM_CORE_IF_MSG_ORIGIN_CLIENT);
         pmCoreIfMessage->traceMessage();
         PmCoreMainController::getInstance().handlePmCoreIfMessage(pmCoreIfMessage);
      }
      else
      {
         ETG_TRACE_ERR(("onSwapCallsResponse: No valid BdAddress for object path: %s", objPath.c_str()));
      }
   }

   BTSResult EvoBtStackWrapper::sendReleaseAndAnswerRequest(const BdAddress& deviceAddress, const ActType act)
   {
      BTSResult btsResult(BTS_REQ_FAILED, ::ccdbusif::evolution::ERROR_UNKNOWN, "");
      act_t dbusAct = ::ccdbusif::DEFAULT_ACT;

      if(nullptr != _hfpVoiceCallManagerRequestIf)
      {
         ObjectPath objPath;

         ObjectPathInfoListHandler::getInstance().getHandsfreeObjectPath(deviceAddress, objPath);

         if(false == objPath.empty())
         {
            dbusAct = _hfpVoiceCallManagerRequestIf->sendReleaseAndAnswerRequest(objPath);

            if(::ccdbusif::DEFAULT_ACT != dbusAct)
            {
               mapDbusActToPmCoreAct(dbusAct, act);
               btsResult._btsRequestResult = BTS_REQ_SUCCESS;
            }
            else
            {
               ETG_TRACE_ERR(("EvoBtStackWrapper: Failed to sendReleaseAndAnswerRequest"));
            }
         }
         else
         {
            ETG_TRACE_ERR(("sendReleaseAndAnswerRequest: No object path for BdAddress:%s", deviceAddress.c_str()));
         }
      }
      else
      {
         ETG_TRACE_ERR(("sendReleaseAndAnswerRequest: _hfpVoiceCallManagerRequestIf invalid instance"));
      }

      return btsResult;
   }

   void EvoBtStackWrapper::onReleaseAndAnswerError(const ObjectPath& objPath, const ErrorName& errorName,
         const ErrorMessage& errorMessage, const act_t act)
   {
      //Send Error response to PmCoreMainController
      BdAddress deviceAddress;

      ObjectPathInfoListHandler::getInstance().getDeviceAddress(objPath, deviceAddress);
      if(false == deviceAddress.empty())
      {
         BTSResult btsResult(BTS_REQ_FAILED, ::ccdbusif::evolution::ERROR_UNKNOWN, "");
         ActType pmCoreAct = PM_DEFAULT_ACT;

         btsResult._errorCode = getErrorCode2Enum(errorName, errorMessage);
         btsResult._errorMessage = errorMessage;

         retrievePmCoreActFromDbusAct(act, pmCoreAct);

         std::shared_ptr<PmCoreIfMessage> pmCoreIfMessage = getNewPmCoreIfMessage_ReleaseAndAcceptResult(
               btsResult, deviceAddress, pmCoreAct, PM_CORE_IF_MSG_ORIGIN_CLIENT);
         pmCoreIfMessage->traceMessage();
         PmCoreMainController::getInstance().handlePmCoreIfMessage(pmCoreIfMessage);
      }
      else
      {
         ETG_TRACE_ERR(("onReleaseAndAnswerError: No valid BdAddress for object path: %s", objPath.c_str()));
      }
   }

   void EvoBtStackWrapper::onReleaseAndAnswerResponse(const ObjectPath& objPath, const act_t act)
   {
      //Send Success response to PmCoreMainController
      BdAddress deviceAddress;

      ObjectPathInfoListHandler::getInstance().getDeviceAddress(objPath, deviceAddress);
      if(false == deviceAddress.empty())
      {
         BTSResult btsResult(BTS_REQ_SUCCESS, ::ccdbusif::evolution::ERROR_UNKNOWN, "");
         ActType pmCoreAct = PM_DEFAULT_ACT;

         retrievePmCoreActFromDbusAct(act, pmCoreAct);

         std::shared_ptr<PmCoreIfMessage> pmCoreIfMessage = getNewPmCoreIfMessage_SwapCallResult(btsResult,
               deviceAddress, pmCoreAct, PM_CORE_IF_MSG_ORIGIN_CLIENT);
         pmCoreIfMessage->traceMessage();
         PmCoreMainController::getInstance().handlePmCoreIfMessage(pmCoreIfMessage);
      }
      else
      {
         ETG_TRACE_ERR(("onReleaseAndAnswerResponse: No valid BdAddress for object path: %s", objPath.c_str()));
      }
   }

   BTSResult EvoBtStackWrapper::sendReleaseAndSwapRequest(const BdAddress& deviceAddress, const ActType act)
   {
      BTSResult btsResult(BTS_REQ_FAILED, ::ccdbusif::evolution::ERROR_UNKNOWN, "");
      act_t dbusAct = ::ccdbusif::DEFAULT_ACT;

      if(nullptr != _hfpVoiceCallManagerRequestIf)
      {
         ObjectPath objPath;

         ObjectPathInfoListHandler::getInstance().getHandsfreeObjectPath(deviceAddress, objPath);

         if(false == objPath.empty())
         {
            dbusAct = _hfpVoiceCallManagerRequestIf->sendReleaseAndSwapRequest(objPath);

            if(::ccdbusif::DEFAULT_ACT != dbusAct)
            {
               mapDbusActToPmCoreAct(dbusAct, act);
               btsResult._btsRequestResult = BTS_REQ_SUCCESS;
            }
            else
            {
               ETG_TRACE_ERR(("EvoBtStackWrapper: Failed to sendReleaseAndSwapRequest"));
            }
         }
         else
         {
            ETG_TRACE_ERR(("sendReleaseAndSwapRequest: No object path for BdAddress:%s", deviceAddress.c_str()));
         }
      }
      else
      {
         ETG_TRACE_ERR(("sendReleaseAndSwapRequest: _hfpVoiceCallManagerRequestIf invalid instance"));
      }

      return btsResult;
   }

   void EvoBtStackWrapper::onReleaseAndSwapError(const ObjectPath& objPath, const ErrorName& errorName,
         const ErrorMessage& errorMessage, const act_t act)
   {
      //Send Error response to PmCoreMainController
      BdAddress deviceAddress;

      ObjectPathInfoListHandler::getInstance().getDeviceAddress(objPath, deviceAddress);
      if(false == deviceAddress.empty())
      {
         BTSResult btsResult(BTS_REQ_FAILED, ::ccdbusif::evolution::ERROR_UNKNOWN, "");
         ActType pmCoreAct = PM_DEFAULT_ACT;

         btsResult._errorCode = getErrorCode2Enum(errorName, errorMessage);
         btsResult._errorMessage = errorMessage;

         retrievePmCoreActFromDbusAct(act, pmCoreAct);

         std::shared_ptr<PmCoreIfMessage> pmCoreIfMessage = getNewPmCoreIfMessage_ReleaseAndSwapResult(btsResult,
               deviceAddress, pmCoreAct, PM_CORE_IF_MSG_ORIGIN_CLIENT);
         pmCoreIfMessage->traceMessage();
         PmCoreMainController::getInstance().handlePmCoreIfMessage(pmCoreIfMessage);
      }
      else
      {
         ETG_TRACE_ERR(("onReleaseAndSwapError: No valid BdAddress for object path: %s", objPath.c_str()));
      }
   }

   void EvoBtStackWrapper::onReleaseAndSwapResponse(const ObjectPath& objPath, const act_t act)
   {
      //Send Success response to PmCoreMainController
      BdAddress deviceAddress;

      ObjectPathInfoListHandler::getInstance().getDeviceAddress(objPath, deviceAddress);
      if(false == deviceAddress.empty())
      {
         BTSResult btsResult(BTS_REQ_SUCCESS, ::ccdbusif::evolution::ERROR_UNKNOWN, "");
         ActType pmCoreAct = PM_DEFAULT_ACT;

         retrievePmCoreActFromDbusAct(act, pmCoreAct);

         std::shared_ptr<PmCoreIfMessage> pmCoreIfMessage = getNewPmCoreIfMessage_ReleaseAndSwapResult(btsResult,
               deviceAddress, pmCoreAct, PM_CORE_IF_MSG_ORIGIN_CLIENT);
         pmCoreIfMessage->traceMessage();
         PmCoreMainController::getInstance().handlePmCoreIfMessage(pmCoreIfMessage);
      }

      else
      {
         ETG_TRACE_ERR(("onReleaseAndSwapResponse: No valid BdAddress for object path: %s", objPath.c_str()));
      }
   }

   BTSResult EvoBtStackWrapper::sendHoldAndAnswerRequest(const BdAddress& deviceAddress, const ActType act)
   {
      BTSResult btsResult(BTS_REQ_FAILED, ::ccdbusif::evolution::ERROR_UNKNOWN, "");
      act_t dbusAct = ::ccdbusif::DEFAULT_ACT;

      if(nullptr != _hfpVoiceCallManagerRequestIf)
      {
         ObjectPath objPath;

         ObjectPathInfoListHandler::getInstance().getHandsfreeObjectPath(deviceAddress, objPath);

         if(false == objPath.empty())
         {
            dbusAct = _hfpVoiceCallManagerRequestIf->sendHoldAndAnswerRequest(objPath);

            if(::ccdbusif::DEFAULT_ACT != dbusAct)
            {
               mapDbusActToPmCoreAct(dbusAct, act);
               btsResult._btsRequestResult = BTS_REQ_SUCCESS;
            }
            else
            {
               ETG_TRACE_ERR(("EvoBtStackWrapper: Failed to sendHoldAndAnswerRequest"));
            }
         }
         else
         {
            ETG_TRACE_ERR(("sendHoldAndAnswerRequest: No object path for BdAddress:%s", deviceAddress.c_str()));
         }
      }
      else
      {
         ETG_TRACE_ERR(("sendHoldAndAnswerRequest: _hfpVoiceCallManagerRequestIf invalid instance"));
      }

      return btsResult;
   }

   void EvoBtStackWrapper::onHoldAndAnswerError(const ObjectPath& objPath, const ErrorName& errorName,
         const ErrorMessage& errorMessage, const act_t act)
   {
      //Send Error response to PmCoreMainController
      BdAddress deviceAddress;

      ObjectPathInfoListHandler::getInstance().getDeviceAddress(objPath, deviceAddress);
      if(false == deviceAddress.empty())
      {
         BTSResult btsResult(BTS_REQ_FAILED, ::ccdbusif::evolution::ERROR_UNKNOWN, "");
         ActType pmCoreAct = PM_DEFAULT_ACT;

         btsResult._errorCode = getErrorCode2Enum(errorName, errorMessage);
         btsResult._errorMessage = errorMessage;

         retrievePmCoreActFromDbusAct(act, pmCoreAct);

         std::shared_ptr<PmCoreIfMessage> pmCoreIfMessage = getNewPmCoreIfMessage_HoldAndAcceptResult(btsResult,
               deviceAddress, pmCoreAct, PM_CORE_IF_MSG_ORIGIN_CLIENT);
         pmCoreIfMessage->traceMessage();
         PmCoreMainController::getInstance().handlePmCoreIfMessage(pmCoreIfMessage);
      }
      else
      {
         ETG_TRACE_ERR(("onHoldAndAnswerError: No valid BdAddress for object path: %s", objPath.c_str()));
      }
   }

   void EvoBtStackWrapper::onHoldAndAnswerResponse(const ObjectPath& objPath, const act_t act)
   {
      //Send Success response to PmCoreMainController
      BdAddress deviceAddress;

      ObjectPathInfoListHandler::getInstance().getDeviceAddress(objPath, deviceAddress);
      if(false == deviceAddress.empty())
      {
         BTSResult btsResult(BTS_REQ_SUCCESS, ::ccdbusif::evolution::ERROR_UNKNOWN, "");
         ActType pmCoreAct = PM_DEFAULT_ACT;

         retrievePmCoreActFromDbusAct(act, pmCoreAct);

         std::shared_ptr<PmCoreIfMessage> pmCoreIfMessage = getNewPmCoreIfMessage_HoldAndAcceptResult(btsResult,
               deviceAddress, pmCoreAct, PM_CORE_IF_MSG_ORIGIN_CLIENT);
         pmCoreIfMessage->traceMessage();
         PmCoreMainController::getInstance().handlePmCoreIfMessage(pmCoreIfMessage);
      }
      else
      {
         ETG_TRACE_ERR(("onHoldAndAnswerResponse: No valid BdAddress for object path: %s", objPath.c_str()));
      }
   }

   BTSResult EvoBtStackWrapper::sendHangupAllRequest(const BdAddress& deviceAddress, const ActType act)
   {
      BTSResult btsResult(BTS_REQ_FAILED, ::ccdbusif::evolution::ERROR_UNKNOWN, "");
      act_t dbusAct = ::ccdbusif::DEFAULT_ACT;

      if(nullptr != _hfpVoiceCallManagerRequestIf)
      {
         ObjectPath objPath;

         ObjectPathInfoListHandler::getInstance().getHandsfreeObjectPath(deviceAddress, objPath);

         if(false == objPath.empty())
         {
            dbusAct = _hfpVoiceCallManagerRequestIf->sendHangupAllRequest(objPath);

            if(::ccdbusif::DEFAULT_ACT != dbusAct)
            {
               mapDbusActToPmCoreAct(dbusAct, act);
               btsResult._btsRequestResult = BTS_REQ_SUCCESS;
            }
            else
            {
               ETG_TRACE_ERR(("EvoBtStackWrapper: Failed to sendHangupAllRequest"));
            }
         }
         else
         {
            ETG_TRACE_ERR(("sendHangupAllRequest: No object path for BdAddress:%s", deviceAddress.c_str()));
         }
      }
      else
      {
         ETG_TRACE_ERR(("sendHangupAllRequest: _hfpVoiceCallManagerRequestIf invalid instance"));
      }

      return btsResult;
   }

   void EvoBtStackWrapper::onHangupAllError(const ObjectPath& objPath, const ErrorName& errorName,
         const ErrorMessage& errorMessage, const act_t act)
   {
      //Send Error response to PmCoreMainController
      BdAddress deviceAddress;

      ObjectPathInfoListHandler::getInstance().getDeviceAddress(objPath, deviceAddress);
      if(false == deviceAddress.empty())
      {
         BTSResult btsResult(BTS_REQ_FAILED, ::ccdbusif::evolution::ERROR_UNKNOWN, "");
         ActType pmCoreAct = PM_DEFAULT_ACT;

         btsResult._errorCode = getErrorCode2Enum(errorName, errorMessage);
         btsResult._errorMessage = errorMessage;

         retrievePmCoreActFromDbusAct(act, pmCoreAct);

         std::shared_ptr<PmCoreIfMessage> pmCoreIfMessage = getNewPmCoreIfMessage_HangupAllResult(btsResult,
               deviceAddress, pmCoreAct, PM_CORE_IF_MSG_ORIGIN_CLIENT);
         pmCoreIfMessage->traceMessage();
         PmCoreMainController::getInstance().handlePmCoreIfMessage(pmCoreIfMessage);
      }
      else
      {
         ETG_TRACE_ERR(("onHangupAllError: No valid BdAddress for object path: %s", objPath.c_str()));
      }
   }

   void EvoBtStackWrapper::onHangupAllResponse(const ObjectPath& objPath, const act_t act)
   {
      //Send Success response to PmCoreMainController
      BdAddress deviceAddress;

      ObjectPathInfoListHandler::getInstance().getDeviceAddress(objPath, deviceAddress);
      if(false == deviceAddress.empty())
      {
         BTSResult btsResult(BTS_REQ_SUCCESS, ::ccdbusif::evolution::ERROR_UNKNOWN, "");
         ActType pmCoreAct = PM_DEFAULT_ACT;

         retrievePmCoreActFromDbusAct(act, pmCoreAct);

         std::shared_ptr<PmCoreIfMessage> pmCoreIfMessage = getNewPmCoreIfMessage_HangupAllResult(btsResult,
               deviceAddress, pmCoreAct, PM_CORE_IF_MSG_ORIGIN_CLIENT);
         pmCoreIfMessage->traceMessage();
         PmCoreMainController::getInstance().handlePmCoreIfMessage(pmCoreIfMessage);
      }
      else
      {
         ETG_TRACE_ERR(("onHangupAllResponse: No valid BdAddress for object path: %s", objPath.c_str()));
      }
   }

   BTSResult EvoBtStackWrapper::sendPrivateChatRequest(const BdAddress& deviceAddress,
         const CallInstance callInstance, const ActType act)
   {
      BTSResult btsResult(BTS_REQ_FAILED, ::ccdbusif::evolution::ERROR_UNKNOWN, "");
      act_t dbusAct = ::ccdbusif::DEFAULT_ACT;

      if(nullptr != _hfpVoiceCallManagerRequestIf)
      {
         ObjectPath objPath;
         ObjectPath voiceCallObjPath;

         ObjectPathInfoListHandler::getInstance().getHandsfreeObjectPath(deviceAddress, objPath);

         if(false == objPath.empty())
         {
            ObjectPathInfoListHandler::getInstance().getVoiceCallObjectPath(deviceAddress, callInstance,
                  voiceCallObjPath);

            if(false == voiceCallObjPath.empty())
            {
               dbusAct = _hfpVoiceCallManagerRequestIf->sendPrivateChatRequest(objPath, voiceCallObjPath);

               if(::ccdbusif::DEFAULT_ACT != dbusAct)
               {
                  mapDbusActToPmCoreAct(dbusAct, act);
                  btsResult._btsRequestResult = BTS_REQ_SUCCESS;
               }
               else
               {
                  ETG_TRACE_ERR(("EvoBtStackWrapper: Failed to sendPrivateChatRequest"));
               }
            }
            else
            {
               ETG_TRACE_ERR(("sendPrivateChatRequest: VoiceCall ObjectPath not found"));
            }
         }
         else
         {
            ETG_TRACE_ERR(("sendPrivateChatRequest: No Handsfree obj path for BdAddress:%s", deviceAddress.c_str()));
         }
      }
      else
      {
         ETG_TRACE_ERR(("sendPrivateChatRequest: _hfpVoiceCallManagerRequestIf invalid instance"));
      }

      return btsResult;
   }

   void EvoBtStackWrapper::onPrivateChatError(const ObjectPath& objPath, const ErrorName& errorName,
         const ErrorMessage& errorMessage, const act_t act)
   {
      //Send Error response to PmCoreMainController
      BdAddress deviceAddress;

      ObjectPathInfoListHandler::getInstance().getDeviceAddress(objPath, deviceAddress);
      if(false == deviceAddress.empty())
      {
         BTSResult btsResult(BTS_REQ_FAILED, ::ccdbusif::evolution::ERROR_UNKNOWN, "");
         ActType pmCoreAct = PM_DEFAULT_ACT;
         CallInstanceList callInstanceList;

         btsResult._errorCode = getErrorCode2Enum(errorName, errorMessage);
         btsResult._errorMessage = errorMessage;

         retrievePmCoreActFromDbusAct(act, pmCoreAct);

         std::shared_ptr<PmCoreIfMessage> pmCoreIfMessage = getNewPmCoreIfMessage_SplitCallsResult(btsResult,
               deviceAddress, callInstanceList, pmCoreAct, PM_CORE_IF_MSG_ORIGIN_CLIENT);
         pmCoreIfMessage->traceMessage();
         PmCoreMainController::getInstance().handlePmCoreIfMessage(pmCoreIfMessage);
      }
      else
      {
         ETG_TRACE_ERR(("onPrivateChatError: No valid BdAddress for object path: %s", objPath.c_str()));
      }
   }

   void EvoBtStackWrapper::onPrivateChatResponse(const ObjectPath& objPath,
         const ::std::vector< ::std::string >& calls, const act_t act)
   {
      //Send Success response to PmCoreMainController
      BdAddress deviceAddress;

      ObjectPathInfoListHandler::getInstance().getDeviceAddress(objPath, deviceAddress);
      if(false == deviceAddress.empty())
      {
         BTSResult btsResult(BTS_REQ_SUCCESS, ::ccdbusif::evolution::ERROR_UNKNOWN, "");
         ActType pmCoreAct = PM_DEFAULT_ACT;
         CallInstanceList callInstanceList;

         retrievePmCoreActFromDbusAct(act, pmCoreAct);

         ::std::vector< ::std::string >::const_iterator it;
         for(it = calls.begin(); it != calls.end(); it++)
         {
            CallInstance callInstance;

            callInstance = getInstanceFromObjectPath(*it, VOICECALL_OBJECTPATH_LENGTH);

            if(callInstance != DEFAULT_INSTANCE_VALUE)
            {
               callInstanceList.push_back(callInstance);
            }
         }

         std::shared_ptr<PmCoreIfMessage> pmCoreIfMessage = getNewPmCoreIfMessage_SplitCallsResult(btsResult,
               deviceAddress, callInstanceList, pmCoreAct, PM_CORE_IF_MSG_ORIGIN_CLIENT);
         pmCoreIfMessage->traceMessage();
         PmCoreMainController::getInstance().handlePmCoreIfMessage(pmCoreIfMessage);
      }
      else
      {
         ETG_TRACE_ERR(("onPrivateChatResponse: No valid BdAddress for object path: %s", objPath.c_str()));
      }
   }

   BTSResult EvoBtStackWrapper::sendCreateMultipartyRequest(const BdAddress& deviceAddress, const ActType act)
   {
      BTSResult btsResult(BTS_REQ_FAILED, ::ccdbusif::evolution::ERROR_UNKNOWN, "");
      act_t dbusAct = ::ccdbusif::DEFAULT_ACT;

      if(nullptr != _hfpVoiceCallManagerRequestIf)
      {
         ObjectPath objPath;

         ObjectPathInfoListHandler::getInstance().getHandsfreeObjectPath(deviceAddress, objPath);

         if(false == objPath.empty())
         {
            dbusAct = _hfpVoiceCallManagerRequestIf->sendCreateMultipartyRequest(objPath);

            if(::ccdbusif::DEFAULT_ACT != dbusAct)
            {
               mapDbusActToPmCoreAct(dbusAct, act);
               btsResult._btsRequestResult = BTS_REQ_SUCCESS;
            }
            else
            {
               ETG_TRACE_ERR(("EvoBtStackWrapper: Failed to sendCreateMultipartyRequest"));
            }
         }
         else
         {
            ETG_TRACE_ERR(("sendCreateMultipartyRequest: No object path for BdAddress:%s", deviceAddress.c_str()));
         }
      }
      else
      {
         ETG_TRACE_ERR(("sendCreateMultipartyRequest: _hfpVoiceCallManagerRequestIf invalid instance"));
      }

      return btsResult;
   }

   void EvoBtStackWrapper::onCreateMultipartyError(const ObjectPath& objPath, const ErrorName& errorName,
         const ErrorMessage& errorMessage, const act_t act)
   {
      //Send Error response to PmCoreMainController
      BdAddress deviceAddress;

      ObjectPathInfoListHandler::getInstance().getDeviceAddress(objPath, deviceAddress);
      if(false == deviceAddress.empty())
      {
         BTSResult btsResult(BTS_REQ_FAILED, ::ccdbusif::evolution::ERROR_UNKNOWN, "");
         ActType pmCoreAct = PM_DEFAULT_ACT;
         CallInstanceList callInstanceList;

         btsResult._errorCode = getErrorCode2Enum(errorName, errorMessage);
         btsResult._errorMessage = errorMessage;

         retrievePmCoreActFromDbusAct(act, pmCoreAct);

         std::shared_ptr<PmCoreIfMessage> pmCoreIfMessage = getNewPmCoreIfMessage_MergeCallsResult(btsResult,
               deviceAddress, callInstanceList, pmCoreAct, PM_CORE_IF_MSG_ORIGIN_CLIENT);
         pmCoreIfMessage->traceMessage();
         PmCoreMainController::getInstance().handlePmCoreIfMessage(pmCoreIfMessage);
      }
      else
      {
         ETG_TRACE_ERR(("onCreateMultipartyError: No valid BdAddress for object path: %s", objPath.c_str()));
      }
   }

   void EvoBtStackWrapper::onCreateMultipartyResponse(const ObjectPath& objPath,
         const ::std::vector< ::std::string >& calls, const act_t act)
   {
      //Send Success response to PmCoreMainController
      BdAddress deviceAddress;

      ObjectPathInfoListHandler::getInstance().getDeviceAddress(objPath, deviceAddress);
      if(false == deviceAddress.empty())
      {
         BTSResult btsResult(BTS_REQ_SUCCESS, ::ccdbusif::evolution::ERROR_UNKNOWN, "");
         ActType pmCoreAct = PM_DEFAULT_ACT;
         CallInstanceList callInstanceList;

         retrievePmCoreActFromDbusAct(act, pmCoreAct);

         ::std::vector< ::std::string >::const_iterator it;
         for(it = calls.begin(); it != calls.end(); it++)
         {
            CallInstance callInstance;

            callInstance = getInstanceFromObjectPath(*it, VOICECALL_OBJECTPATH_LENGTH);

            if(callInstance != DEFAULT_INSTANCE_VALUE)
            {
               callInstanceList.push_back(callInstance);
            }
         }

         std::shared_ptr<PmCoreIfMessage> pmCoreIfMessage = getNewPmCoreIfMessage_MergeCallsResult(btsResult,
               deviceAddress, callInstanceList, pmCoreAct, PM_CORE_IF_MSG_ORIGIN_CLIENT);
         pmCoreIfMessage->traceMessage();
         PmCoreMainController::getInstance().handlePmCoreIfMessage(pmCoreIfMessage);
      }
      else
      {
         ETG_TRACE_ERR(("onCreateMultipartyResponse: No valid BdAddress for object path: %s", objPath.c_str()));
      }
   }

   BTSResult EvoBtStackWrapper::sendHangupMultipartyRequest(const BdAddress& deviceAddress, const ActType act)
   {
      BTSResult btsResult(BTS_REQ_FAILED, ::ccdbusif::evolution::ERROR_UNKNOWN, "");
      act_t dbusAct = ::ccdbusif::DEFAULT_ACT;

      if(nullptr != _hfpVoiceCallManagerRequestIf)
      {
         ObjectPath objPath;

         ObjectPathInfoListHandler::getInstance().getHandsfreeObjectPath(deviceAddress, objPath);

         if(false == objPath.empty())
         {
            dbusAct = _hfpVoiceCallManagerRequestIf->sendHangupMultipartyRequest(objPath);

            if(::ccdbusif::DEFAULT_ACT != dbusAct)
            {
               mapDbusActToPmCoreAct(dbusAct, act);
               btsResult._btsRequestResult = BTS_REQ_SUCCESS;
            }
            else
            {
               ETG_TRACE_ERR(("EvoBtStackWrapper: Failed to sendHangupMultipartyRequest"));
            }
         }
         else
         {
            ETG_TRACE_ERR(("sendHangupMultipartyRequest: No object path for BdAddress:%s", deviceAddress.c_str()));
         }
      }
      else
      {
         ETG_TRACE_ERR(("sendHangupMultipartyRequest: _hfpVoiceCallManagerRequestIf invalid instance"));
      }

      return btsResult;
   }

   void EvoBtStackWrapper::onHangupMultipartyError(const ObjectPath& objPath, const ErrorName& errorName,
         const ErrorMessage& errorMessage, const act_t act)
   {
      //Send Error response to PmCoreMainController
      BdAddress deviceAddress;

      ObjectPathInfoListHandler::getInstance().getDeviceAddress(objPath, deviceAddress);
      if(false == deviceAddress.empty())
      {
         BTSResult btsResult(BTS_REQ_FAILED, ::ccdbusif::evolution::ERROR_UNKNOWN, "");
         ActType pmCoreAct = PM_DEFAULT_ACT;

         btsResult._errorCode = getErrorCode2Enum(errorName, errorMessage);
         btsResult._errorMessage = errorMessage;

         retrievePmCoreActFromDbusAct(act, pmCoreAct);

         std::shared_ptr<PmCoreIfMessage> pmCoreIfMessage = getNewPmCoreIfMessage_HangupMultipartyResult(
               btsResult, deviceAddress, pmCoreAct, PM_CORE_IF_MSG_ORIGIN_CLIENT);
         pmCoreIfMessage->traceMessage();
         PmCoreMainController::getInstance().handlePmCoreIfMessage(pmCoreIfMessage);
      }
      else
      {
         ETG_TRACE_ERR(("onHangupMultipartyError: No valid BdAddress for object path: %s", objPath.c_str()));
      }
   }

   void EvoBtStackWrapper::onHangupMultipartyResponse(const ObjectPath& objPath, const act_t act)
   {
      //Send Success response to PmCoreMainController
      BdAddress deviceAddress;

      ObjectPathInfoListHandler::getInstance().getDeviceAddress(objPath, deviceAddress);
      if(false == deviceAddress.empty())
      {
         BTSResult btsResult(BTS_REQ_SUCCESS, ::ccdbusif::evolution::ERROR_UNKNOWN, "");
         ActType pmCoreAct = PM_DEFAULT_ACT;

         retrievePmCoreActFromDbusAct(act, pmCoreAct);

         std::shared_ptr<PmCoreIfMessage> pmCoreIfMessage = getNewPmCoreIfMessage_HangupMultipartyResult(
               btsResult, deviceAddress, pmCoreAct, PM_CORE_IF_MSG_ORIGIN_CLIENT);
         pmCoreIfMessage->traceMessage();
         PmCoreMainController::getInstance().handlePmCoreIfMessage(pmCoreIfMessage);
      }
      else
      {
         ETG_TRACE_ERR(("onHangupMultipartyResponse: No valid BdAddress for object path: %s", objPath.c_str()));
      }
   }

   BTSResult EvoBtStackWrapper::sendSendTonesRequest(const BdAddress& deviceAddress, const DTMFTones& sendTones,
         const ActType act)
   {
      BTSResult btsResult(BTS_REQ_FAILED, ::ccdbusif::evolution::ERROR_UNKNOWN, "");
      act_t dbusAct = ::ccdbusif::DEFAULT_ACT;

      if(nullptr != _hfpVoiceCallManagerRequestIf)
      {
         ObjectPath objPath;

         ObjectPathInfoListHandler::getInstance().getHandsfreeObjectPath(deviceAddress, objPath);

         if(false == objPath.empty())
         {
            dbusAct = _hfpVoiceCallManagerRequestIf->sendSendTonesRequest(objPath, sendTones);

            if(::ccdbusif::DEFAULT_ACT != dbusAct)
            {
               mapDbusActToPmCoreAct(dbusAct, act);
               btsResult._btsRequestResult = BTS_REQ_SUCCESS;
            }
            else
            {
               ETG_TRACE_ERR(("EvoBtStackWrapper: Failed to sendSendTonesRequest"));
            }
         }
         else
         {
            ETG_TRACE_ERR(("sendSendTonesRequest: No object path for BdAddress:%s", deviceAddress.c_str()));
         }
      }
      else
      {
         ETG_TRACE_ERR(("sendSendTonesRequest: _hfpVoiceCallManagerRequestIf invalid instance"));
      }

      return btsResult;
   }

   void EvoBtStackWrapper::onSendTonesError(const ObjectPath& objPath, const ErrorName& errorName,
         const ErrorMessage& errorMessage, const act_t act)
   {
      //Send Error response to PmCoreMainController
      BdAddress deviceAddress;

      ObjectPathInfoListHandler::getInstance().getDeviceAddress(objPath, deviceAddress);
      if(false == deviceAddress.empty())
      {
         BTSResult btsResult(BTS_REQ_FAILED, ::ccdbusif::evolution::ERROR_UNKNOWN, "");
         ActType pmCoreAct = PM_DEFAULT_ACT;

         btsResult._errorCode = getErrorCode2Enum(errorName, errorMessage);
         btsResult._errorMessage = errorMessage;

         retrievePmCoreActFromDbusAct(act, pmCoreAct);

         std::shared_ptr<PmCoreIfMessage> pmCoreIfMessage = getNewPmCoreIfMessage_SendDTMFResult(btsResult,
               deviceAddress, pmCoreAct, PM_CORE_IF_MSG_ORIGIN_CLIENT);
         pmCoreIfMessage->traceMessage();
         PmCoreMainController::getInstance().handlePmCoreIfMessage(pmCoreIfMessage);
      }
      else
      {
         ETG_TRACE_ERR(("onSendTonesError: No valid BdAddress for object path: %s", objPath.c_str()));
      }
   }

   void EvoBtStackWrapper::onSendTonesResponse(const ObjectPath& objPath, const act_t act)
   {
      //Send Success response to PmCoreMainController
      BdAddress deviceAddress;

      ObjectPathInfoListHandler::getInstance().getDeviceAddress(objPath, deviceAddress);
      if(false == deviceAddress.empty())
      {
         BTSResult btsResult(BTS_REQ_SUCCESS, ::ccdbusif::evolution::ERROR_UNKNOWN, "");
         ActType pmCoreAct = PM_DEFAULT_ACT;

         retrievePmCoreActFromDbusAct(act, pmCoreAct);

         std::shared_ptr<PmCoreIfMessage> pmCoreIfMessage = getNewPmCoreIfMessage_SendDTMFResult(btsResult,
               deviceAddress, pmCoreAct, PM_CORE_IF_MSG_ORIGIN_CLIENT);
         pmCoreIfMessage->traceMessage();
         PmCoreMainController::getInstance().handlePmCoreIfMessage(pmCoreIfMessage);
      }
      else
      {
         ETG_TRACE_ERR(("onSendTonesResponse: No valid BdAddress for object path: %s", objPath.c_str()));
      }
   }

   BTSResult EvoBtStackWrapper::sendHangupRequest(const BdAddress& deviceAddress, const CallInstance callInstance,
         const ActType act)
   {
      BTSResult btsResult(BTS_REQ_FAILED, ::ccdbusif::evolution::ERROR_UNKNOWN, "");
      act_t dbusAct = ::ccdbusif::DEFAULT_ACT;

      if(nullptr != _hfpVoiceCallRequestIf)
      {
         ObjectPath objPath;
         ObjectPathInfoListHandler::getInstance().getVoiceCallObjectPath(deviceAddress, callInstance, objPath);

         if(false == objPath.empty())
         {
            dbusAct = _hfpVoiceCallRequestIf->sendHangupRequest(objPath);

            if(::ccdbusif::DEFAULT_ACT != dbusAct)
            {
               mapDbusActToPmCoreAct(dbusAct, act);
               btsResult._btsRequestResult = BTS_REQ_SUCCESS;
            }
            else
            {
               ETG_TRACE_ERR(("EvoBtStackWrapper: Failed to sendHangupRequest"));
            }
         }
         else
         {
            ETG_TRACE_ERR(("sendHangupRequest: No object path for BdAddress:%s", deviceAddress.c_str()));
         }
      }
      else
      {
         ETG_TRACE_ERR(("sendHangupRequest: _hfpVoiceCallRequestIf invalid instance"));
      }

      return btsResult;
   }

   void EvoBtStackWrapper::onHangupError(const ObjectPath& objPath, const ErrorName& errorName,
         const ErrorMessage& errorMessage, const act_t act)
   {
      //Send Error response to PmCoreMainController
      BdAddress deviceAddress;
      CallInstance callInstance;

      ObjectPathInfoListHandler::getInstance().getDeviceAddressAndCallInstance(objPath, deviceAddress, callInstance);
      if(false == deviceAddress.empty())
      {
         BTSResult btsResult(BTS_REQ_FAILED, ::ccdbusif::evolution::ERROR_UNKNOWN, "");
         ActType pmCoreAct = PM_DEFAULT_ACT;

         btsResult._errorCode = getErrorCode2Enum(errorName, errorMessage);
         btsResult._errorMessage = errorMessage;

         retrievePmCoreActFromDbusAct(act, pmCoreAct);

         std::shared_ptr<PmCoreIfMessage> pmCoreIfMessage = getNewPmCoreIfMessage_HangupCallResult(btsResult,
               deviceAddress, callInstance, pmCoreAct, PM_CORE_IF_MSG_ORIGIN_CLIENT);
         pmCoreIfMessage->traceMessage();
         PmCoreMainController::getInstance().handlePmCoreIfMessage(pmCoreIfMessage);
      }
      else
      {
         ETG_TRACE_ERR(("onHangupError: No valid BdAddress for object path: %s", objPath.c_str()));
      }
   }

   void EvoBtStackWrapper::onHangupResponse(const ObjectPath& objPath, const act_t act)
   {
      ETG_TRACE_USR4(("EvoBtStackWrapper(): onHangupResponse"));
      //Send Success response to PmCoreMainController
      BdAddress deviceAddress;
      CallInstance callInstance;

      ObjectPathInfoListHandler::getInstance().getDeviceAddressAndCallInstance(objPath, deviceAddress, callInstance);
      if(false == deviceAddress.empty())
      {
         BTSResult btsResult(BTS_REQ_SUCCESS, ::ccdbusif::evolution::ERROR_UNKNOWN, "");
         ActType pmCoreAct = PM_DEFAULT_ACT;

         retrievePmCoreActFromDbusAct(act, pmCoreAct);

         std::shared_ptr<PmCoreIfMessage> pmCoreIfMessage = getNewPmCoreIfMessage_HangupCallResult(btsResult,
               deviceAddress, callInstance, pmCoreAct, PM_CORE_IF_MSG_ORIGIN_CLIENT);
         pmCoreIfMessage->traceMessage();
         PmCoreMainController::getInstance().handlePmCoreIfMessage(pmCoreIfMessage);
      }
      else
      {
         ETG_TRACE_ERR(("onHangupResponse: No valid BdAddress for object path: %s", objPath.c_str()));
      }
   }

   BTSResult EvoBtStackWrapper::sendAnswerRequest(const BdAddress& deviceAddress, const CallInstance callInstance,
         const ActType act)
   {
      BTSResult btsResult(BTS_REQ_FAILED, ::ccdbusif::evolution::ERROR_UNKNOWN, "");
      act_t dbusAct = ::ccdbusif::DEFAULT_ACT;

      if(nullptr != _hfpVoiceCallRequestIf)
      {
         ObjectPath objPath;
         ObjectPathInfoListHandler::getInstance().getVoiceCallObjectPath(deviceAddress, callInstance, objPath);

         if(false == objPath.empty())
         {
            dbusAct = _hfpVoiceCallRequestIf->sendAnswerRequest(objPath);

            if(::ccdbusif::DEFAULT_ACT != dbusAct)
            {
               mapDbusActToPmCoreAct(dbusAct, act);
               btsResult._btsRequestResult = BTS_REQ_SUCCESS;
            }
            else
            {
               ETG_TRACE_ERR(("EvoBtStackWrapper: Failed to sendAnswerRequest"));
            }
         }
         else
         {
            ETG_TRACE_ERR(("sendAnswerRequest: No object path for BdAddress:%s", deviceAddress.c_str()));
         }
      }
      else
      {
         ETG_TRACE_ERR(("sendAnswerRequest: _hfpVoiceCallRequestIf invalid instance"));
      }

      return btsResult;
   }

   void EvoBtStackWrapper::onAnswerError(const ObjectPath& objPath, const ErrorName& errorName,
         const ErrorMessage& errorMessage, const act_t act)
   {
      //Send Error response to PmCoreMainController
      BdAddress deviceAddress;
      CallInstance callInstance;

      ObjectPathInfoListHandler::getInstance().getDeviceAddressAndCallInstance(objPath, deviceAddress, callInstance);
      if(false == deviceAddress.empty())
      {
         BTSResult btsResult(BTS_REQ_FAILED, ::ccdbusif::evolution::ERROR_UNKNOWN, "");
         ActType pmCoreAct = PM_DEFAULT_ACT;

         btsResult._errorCode = getErrorCode2Enum(errorName, errorMessage);
         btsResult._errorMessage = errorMessage;

         retrievePmCoreActFromDbusAct(act, pmCoreAct);

         std::shared_ptr<PmCoreIfMessage> pmCoreIfMessage = getNewPmCoreIfMessage_AcceptCallResult(btsResult,
               deviceAddress, callInstance, pmCoreAct, PM_CORE_IF_MSG_ORIGIN_CLIENT);
         pmCoreIfMessage->traceMessage();
         PmCoreMainController::getInstance().handlePmCoreIfMessage(pmCoreIfMessage);
      }
      else
      {
         ETG_TRACE_ERR(("onAnswerError: No valid BdAddress for object path: %s", objPath.c_str()));
      }
   }

   void EvoBtStackWrapper::onAnswerResponse(const ObjectPath& objPath, const act_t act)
   {
      //Send Success response to PmCoreMainController
      BdAddress deviceAddress;
      CallInstance callInstance;

      ObjectPathInfoListHandler::getInstance().getDeviceAddressAndCallInstance(objPath, deviceAddress, callInstance);
      if(false == deviceAddress.empty())
      {
         BTSResult btsResult(BTS_REQ_SUCCESS, ::ccdbusif::evolution::ERROR_UNKNOWN, "");
         ActType pmCoreAct = PM_DEFAULT_ACT;

         retrievePmCoreActFromDbusAct(act, pmCoreAct);

         std::shared_ptr<PmCoreIfMessage> pmCoreIfMessage = getNewPmCoreIfMessage_AcceptCallResult(btsResult,
               deviceAddress, callInstance, pmCoreAct, PM_CORE_IF_MSG_ORIGIN_CLIENT);
         pmCoreIfMessage->traceMessage();
         PmCoreMainController::getInstance().handlePmCoreIfMessage(pmCoreIfMessage);
      }
      else
      {
         ETG_TRACE_ERR(("onAnswerResponse: No valid BdAddress for object path: %s", objPath.c_str()));
      }
   }

   BTSResult EvoBtStackWrapper::sendHoldRequest(const BdAddress& deviceAddress, const CallInstance callInstance,
         const ActType act)
   {
      BTSResult btsResult(BTS_REQ_FAILED, ::ccdbusif::evolution::ERROR_UNKNOWN, "");
      act_t dbusAct = ::ccdbusif::DEFAULT_ACT;

      if(nullptr != _hfpVoiceCallRequestIf)
      {
         ObjectPath objPath;
         ObjectPathInfoListHandler::getInstance().getVoiceCallObjectPath(deviceAddress, callInstance, objPath);

         if(false == objPath.empty())
         {
            dbusAct = _hfpVoiceCallRequestIf->sendHoldRequest(objPath);

            if(::ccdbusif::DEFAULT_ACT != dbusAct)
            {
               mapDbusActToPmCoreAct(dbusAct, act);
               btsResult._btsRequestResult = BTS_REQ_SUCCESS;
            }
            else
            {
               ETG_TRACE_ERR(("EvoBtStackWrapper: Failed to sendHoldRequest"));
            }
         }
         else
         {
            ETG_TRACE_ERR(("sendHoldRequest: No object path for BdAddress:%s", deviceAddress.c_str()));
         }
      }
      else
      {
         ETG_TRACE_ERR(("sendHoldRequest: _hfpVoiceCallRequestIf invalid instance"));
      }

      return btsResult;
   }

   void EvoBtStackWrapper::onHoldError(const ObjectPath& objPath, const ErrorName& errorName,
         const ErrorMessage& errorMessage, const act_t act)
   {
      //Send Error response to PmCoreMainController
      BdAddress deviceAddress;
      CallInstance callInstance;

      ObjectPathInfoListHandler::getInstance().getDeviceAddressAndCallInstance(objPath, deviceAddress, callInstance);
      if(false == deviceAddress.empty())
      {
         BTSResult btsResult(BTS_REQ_FAILED, ::ccdbusif::evolution::ERROR_UNKNOWN, "");
         ActType pmCoreAct = PM_DEFAULT_ACT;

         btsResult._errorCode = getErrorCode2Enum(errorName, errorMessage);
         btsResult._errorMessage = errorMessage;

         retrievePmCoreActFromDbusAct(act, pmCoreAct);

         std::shared_ptr<PmCoreIfMessage> pmCoreIfMessage = getNewPmCoreIfMessage_HoldIncomingCallResult(
               btsResult, deviceAddress, callInstance, pmCoreAct, PM_CORE_IF_MSG_ORIGIN_CLIENT);
         pmCoreIfMessage->traceMessage();
         PmCoreMainController::getInstance().handlePmCoreIfMessage(pmCoreIfMessage);
      }
      else
      {
         ETG_TRACE_ERR(("onHoldError: No valid BdAddress for object path: %s", objPath.c_str()));
      }
   }

   void EvoBtStackWrapper::onHoldResponse(const ObjectPath& objPath, const act_t act)
   {
      //Send Success response to PmCoreMainController
      BdAddress deviceAddress;
      CallInstance callInstance;

      ObjectPathInfoListHandler::getInstance().getDeviceAddressAndCallInstance(objPath, deviceAddress, callInstance);
      if(false == deviceAddress.empty())
      {
         BTSResult btsResult(BTS_REQ_SUCCESS, ::ccdbusif::evolution::ERROR_UNKNOWN, "");
         ActType pmCoreAct = PM_DEFAULT_ACT;

         retrievePmCoreActFromDbusAct(act, pmCoreAct);

         std::shared_ptr<PmCoreIfMessage> pmCoreIfMessage = getNewPmCoreIfMessage_HoldIncomingCallResult(
               btsResult, deviceAddress, callInstance, pmCoreAct, PM_CORE_IF_MSG_ORIGIN_CLIENT);
         pmCoreIfMessage->traceMessage();
         PmCoreMainController::getInstance().handlePmCoreIfMessage(pmCoreIfMessage);
      }
      else
      {
         ETG_TRACE_ERR(("onHoldResponse: No valid BdAddress for object path: %s", objPath.c_str()));
      }
   }

   void EvoBtStackWrapper::onSpeakerVolumeChanged(const ObjectPath& objPath, const uint8_t speakerVolume)
   {
      BdAddress deviceAddress;

      ObjectPathInfoListHandler::getInstance().getDeviceAddress(objPath, deviceAddress);
      if(false == deviceAddress.empty())
      {
         std::shared_ptr<PropertyDetails<uint8_t>> propertyDetails =
               std::make_shared<PropertyDetails<uint8_t>>(deviceAddress, speakerVolume);

         PropertyUpdateNotifierToCore::getInstance().notifyControllers(BTS_UPDATE_SPEAKER_VOLUME, propertyDetails);
      }
      else
      {
         ETG_TRACE_ERR(("onSpeakerVolumeChanged(): No device found"));
      }
   }

   void EvoBtStackWrapper::onMicrophoneVolumeChanged(const ObjectPath& objPath, const uint8_t microphoneVolume)
   {
      BdAddress deviceAddress;

      ObjectPathInfoListHandler::getInstance().getDeviceAddress(objPath, deviceAddress);
      if(false == deviceAddress.empty())
      {
         std::shared_ptr<PropertyDetails<uint8_t>> propertyDetails =
               std::make_shared<PropertyDetails<uint8_t>>(deviceAddress, microphoneVolume);

         PropertyUpdateNotifierToCore::getInstance().notifyControllers(BTS_UPDATE_MICROPHONE_VOLUME, propertyDetails);
      }
      else
      {
         ETG_TRACE_ERR(("onMicrophoneVolumeChanged(): No device found"));
      }
   }

   void EvoBtStackWrapper::onSupportedFeaturesChanged(const ObjectPath& objPath,
         const std::vector<std::string>& features)
   {
      BdAddress deviceAddress;

      ObjectPathInfoListHandler::getInstance().getDeviceAddress(objPath, deviceAddress);
      if(false == deviceAddress.empty())
      {
         SupportedFeatures supportedFeatures;
         convertSupportedFeaturesToPmCoreType(features, supportedFeatures);

         std::shared_ptr<PropertyDetails<SupportedFeatures>> propertyDetails =
               std::make_shared<PropertyDetails<SupportedFeatures>>(deviceAddress, supportedFeatures);

         PropertyUpdateNotifierToCore::getInstance().notifyControllers(BTS_UPDATE_SUPPORTED_FEATURES, propertyDetails);
      }
      else
      {
         ETG_TRACE_ERR(("onSupportedFeaturesChanged(): No device found"));
      }
   }

   void EvoBtStackWrapper::onInbandRingingChanged(const ObjectPath& objPath, const bool enabled)
   {
      BdAddress deviceAddress;

      ObjectPathInfoListHandler::getInstance().getDeviceAddress(objPath, deviceAddress);
      if(false == deviceAddress.empty())
      {
         std::shared_ptr<PropertyDetails<bool>> propertyDetails =
               std::make_shared<PropertyDetails<bool>>(deviceAddress, enabled);

         PropertyUpdateNotifierToCore::getInstance().notifyControllers(BTS_UPDATE_INBAND_RINGING, propertyDetails);
      }
      else
      {
         ETG_TRACE_ERR(("onInbandRingingChanged(): No device found"));
      }
   }

   void EvoBtStackWrapper::onVoiceRecognitionChanged(const ObjectPath& objPath, const bool enabled)
   {
      BdAddress deviceAddress;

      ObjectPathInfoListHandler::getInstance().getDeviceAddress(objPath, deviceAddress);
      if(false == deviceAddress.empty())
      {
         std::shared_ptr<PropertyDetails<bool>> propertyDetails =
               std::make_shared<PropertyDetails<bool>>(deviceAddress, enabled);

         PropertyUpdateNotifierToCore::getInstance().notifyControllers(BTS_UPDATE_VOICE_RECOGNITION_STATUS,
               propertyDetails);
      }
      else
      {
         ETG_TRACE_ERR(("onVoiceRecognitionChanged(): No device found"));
      }
   }

   void EvoBtStackWrapper::onEchoCancelingNoiseReductionChanged(const ObjectPath& objPath, const bool enabled)
   {
      BdAddress deviceAddress;

      ObjectPathInfoListHandler::getInstance().getDeviceAddress(objPath, deviceAddress);
      if(false == deviceAddress.empty())
      {
         std::shared_ptr<PropertyDetails<bool>> propertyDetails =
               std::make_shared<PropertyDetails<bool>>(deviceAddress, enabled);

         PropertyUpdateNotifierToCore::getInstance().notifyControllers(BTS_UPDATE_ECHO_CANCELING_NOISE_REDUCTION,
               propertyDetails);
      }
      else
      {
         ETG_TRACE_ERR(("onEchoCancelingNoiseReductionChanged(): No device found"));
      }
   }

   void EvoBtStackWrapper::onBatteryChargeLevelChanged(const ObjectPath& objPath, const uint8_t chargeLevel)
   {
      BdAddress deviceAddress;

      ObjectPathInfoListHandler::getInstance().getDeviceAddress(objPath, deviceAddress);
      if(false == deviceAddress.empty())
      {
         std::shared_ptr<PropertyDetails<BatteryChargeLevel>> propertyDetails =
               std::make_shared<PropertyDetails<BatteryChargeLevel>>(deviceAddress, BatteryChargeLevel(chargeLevel));

         PropertyUpdateNotifierToCore::getInstance().notifyControllers(BTS_UPDATE_BATTERY_CHARGE_LEVEL,
               propertyDetails);
      }
      else
      {
         ETG_TRACE_ERR(("onBatteryChargeLevelChanged(): No device found"));
      }
   }

   void EvoBtStackWrapper::onAudioChanged(const ObjectPath& objPath, const std::string& audio)
   {
      BdAddress deviceAddress;

      ObjectPathInfoListHandler::getInstance().getDeviceAddress(objPath, deviceAddress);
      if(false == deviceAddress.empty())
      {
         AudioDirection audioDirection;
         convertHfpAudioToPmCoreType(audio, audioDirection);

         std::shared_ptr<PropertyDetails<AudioDirection>> propertyDetails =
               std::make_shared<PropertyDetails<AudioDirection>>(deviceAddress, audioDirection);

         PropertyUpdateNotifierToCore::getInstance().notifyControllers(BTS_UPDATE_AUDIO, propertyDetails);
      }
      else
      {
         ETG_TRACE_ERR(("onAudioChanged(): No device found"));
      }
   }

   void EvoBtStackWrapper::onSCOTypeChanged(const ObjectPath& objPath, const uint8_t scoType)
   {
      BdAddress deviceAddress;

      ObjectPathInfoListHandler::getInstance().getDeviceAddress(objPath, deviceAddress);
      if(false == deviceAddress.empty())
      {
         SCOStatus scoStatus = SCO_DEFAULT;

         switch(scoType)
         {
            //TODO avoid magic number
            case 0: //Not connected
               scoStatus = SCO_NOT_ESTABLISHED;
               break;
            case 1: //NBS&S1~S3 setting supported
               scoStatus = SCO_NARROWBAND;
               break;
            case 2: //WBS
               scoStatus = SCO_WIDEBAND;
               break;
            case 3: //NBS&S4 setting supported
               scoStatus = SCO_NARROWBAND;
               break;
            default:
               scoStatus = SCO_DEFAULT;
               break;
         }

         std::shared_ptr<PropertyDetails<SCOConnection>> propertyDetails =
               std::make_shared<PropertyDetails<SCOConnection>>(deviceAddress, SCOConnection(scoStatus));

         PropertyUpdateNotifierToCore::getInstance().notifyControllers(BTS_UPDATE_SCO_TYPE, propertyDetails);
      }
      else
      {
         ETG_TRACE_ERR(("onSCOTypeChanged(): No device found"));
      }
   }

   void EvoBtStackWrapper::onAudioPathChanged(const ObjectPath& objPath, const uint8_t audioPath)
   {
      BdAddress deviceAddress;

      ObjectPathInfoListHandler::getInstance().getDeviceAddress(objPath, deviceAddress);
      if(false == deviceAddress.empty())
      {
         std::shared_ptr<PropertyDetails<AudioPath>> propertyDetails =
               std::make_shared<PropertyDetails<AudioPath>>(deviceAddress, audioPath);

         PropertyUpdateNotifierToCore::getInstance().notifyControllers(BTS_UPDATE_AUDIO_PATH, propertyDetails);
      }
      else
      {
         ETG_TRACE_ERR(("onAudioPathChanged(): No device found"));
      }
   }

   void EvoBtStackWrapper::onAllCallStatusChanged(const ObjectPath& objPath, const uint8_t allCallStatus)
   {
      BdAddress deviceAddress;

      ObjectPathInfoListHandler::getInstance().getDeviceAddress(objPath, deviceAddress);
      if(false == deviceAddress.empty())
      {
         std::shared_ptr<PropertyDetails<uint8_t>> propertyDetails =
               std::make_shared<PropertyDetails<uint8_t>>(deviceAddress, allCallStatus);

         PropertyUpdateNotifierToCore::getInstance().notifyControllers(BTS_UPDATE_ALL_CALL_STATUS, propertyDetails);
      }
      else
      {
         ETG_TRACE_ERR(("onAllCallStatusChanged(): No device found"));
      }
   }

   void EvoBtStackWrapper::onChldFeatureChanged(const ObjectPath& objPath,
         const ::std::vector<std::string>& childFeatureList)
   {
      BdAddress deviceAddress;

      ObjectPathInfoListHandler::getInstance().getDeviceAddress(objPath, deviceAddress);
      if(false == deviceAddress.empty())
      {
         std::shared_ptr<PropertyDetails< ::std::vector<std::string>>> propertyDetails =
               std::make_shared<PropertyDetails< ::std::vector<std::string>>>(deviceAddress, childFeatureList);

         PropertyUpdateNotifierToCore::getInstance().notifyControllers(BTS_UPDATE_CHLD_FEATURE, propertyDetails);
      }
      else
      {
         ETG_TRACE_ERR(("onChldFeatureChanged(): No device found"));
      }
   }

   void EvoBtStackWrapper::onAudioIndicatorChanged(const ObjectPath& objPath, const uint8_t audioIndicator)
   {
      BdAddress deviceAddress;

      ObjectPathInfoListHandler::getInstance().getDeviceAddress(objPath, deviceAddress);
      if(false == deviceAddress.empty())
      {
         std::shared_ptr<PropertyDetails<uint8_t>> propertyDetails =
               std::make_shared<PropertyDetails<uint8_t>>(deviceAddress, audioIndicator);

         PropertyUpdateNotifierToCore::getInstance().notifyControllers(BTS_UPDATE_AUDIO_INDICATOR, propertyDetails);
      }
      else
      {
         ETG_TRACE_ERR(("onAudioIndicatorChanged(): No device found"));
      }
   }

   //void onModemPoweredChanged(const ObjectPath& objPath)
   //{
   //}
   //void onModemNameChanged(const ObjectPath& objPath)
   //{
   //}
   //void onModemInterfacesChanged(const ObjectPath& objPath)
   //{
   //}
   //void onModemTypeChanged(const ObjectPath& objPath)
   //{
   //}
   //void onModemManufacturerChanged(const ObjectPath& objPath)
   //{
   //}
   //void onModemModelChanged(const ObjectPath& objPath)
   //{
   //}
   //void onModemRevisionChanged(const ObjectPath& objPath)
   //{
   //}
   //void onModemSerialChanged(const ObjectPath& objPath)
   //{
   //}

   void EvoBtStackWrapper::onNetworkStatusChanged(const ObjectPath& objPath, const std::string& status)
   {
      BdAddress deviceAddress;

      ObjectPathInfoListHandler::getInstance().getDeviceAddress(objPath, deviceAddress);
      if(false == deviceAddress.empty())
      {
         NetworkStatusType networkStatus;
         convertNetworkStatusToPmCoreType(status, networkStatus);

         std::shared_ptr<PropertyDetails<NetworkStatus>> propertyDetails =
               std::make_shared<PropertyDetails<NetworkStatus>>(deviceAddress, NetworkStatus(networkStatus));

         PropertyUpdateNotifierToCore::getInstance().notifyControllers(BTS_UPDATE_NETWORK_STATUS, propertyDetails);
      }
      else
      {
         ETG_TRACE_ERR(("onNetworkStatusChanged(): No device found"));
      }
   }

   void EvoBtStackWrapper::onNetworkMobileCountryCodeChanged(const ObjectPath& objPath,
         const std::string& mobileCountryCode)
   {
      BdAddress deviceAddress;

      ObjectPathInfoListHandler::getInstance().getDeviceAddress(objPath, deviceAddress);
      if(false == deviceAddress.empty())
      {
         std::shared_ptr<PropertyDetails<std::string>> propertyDetails =
               std::make_shared<PropertyDetails<std::string>>(deviceAddress, mobileCountryCode);

         PropertyUpdateNotifierToCore::getInstance().notifyControllers(BTS_UPDATE_MOBILE_COUNTRY_CODE,
               propertyDetails);
      }
      else
      {
         ETG_TRACE_ERR(("onNetworkMobileCountryCodeChanged(): No device found"));
      }
   }

   void EvoBtStackWrapper::onNetworkOperatorNameChanged(const ObjectPath& objPath, const std::string& operatorName)
   {
      BdAddress deviceAddress;

      ObjectPathInfoListHandler::getInstance().getDeviceAddress(objPath, deviceAddress);
      if(false == deviceAddress.empty())
      {
         std::shared_ptr<PropertyDetails<NetworkOperator>> propertyDetails =
               std::make_shared<PropertyDetails<NetworkOperator>>(deviceAddress, NetworkOperator(operatorName));

         PropertyUpdateNotifierToCore::getInstance().notifyControllers(BTS_UPDATE_CURRENT_OPERATOR_NAME,
               propertyDetails);
      }
      else
      {
         ETG_TRACE_ERR(("onNetworkOperatorNameChanged(): No device found"));
      }
   }

   void EvoBtStackWrapper::onNetworkSignalStrengthChanged(const ObjectPath& objPath, const uint8_t strength)
   {
      BdAddress deviceAddress;

      ObjectPathInfoListHandler::getInstance().getDeviceAddress(objPath, deviceAddress);
      if(false == deviceAddress.empty())
      {
         std::shared_ptr<PropertyDetails<SignalStrength>> propertyDetails =
               std::make_shared<PropertyDetails<SignalStrength>>(deviceAddress, SignalStrength(strength));

         PropertyUpdateNotifierToCore::getInstance().notifyControllers(BTS_UPDATE_SIGNAL_STRENGTH, propertyDetails);
      }
      else
      {
         ETG_TRACE_ERR(("onNetworkSignalStrengthChanged(): No device found"));
      }
   }

   void EvoBtStackWrapper::onNetworkCurrentOperatorChanged(const ObjectPath& objPath,
         const std::string& currentOperator)
   {
      if(false == currentOperator.empty())
      {
         BdAddress deviceAddress;

         ObjectPathInfoListHandler::getInstance().getDeviceAddress(objPath, deviceAddress);
         if(false == deviceAddress.empty())
         {
            networkOperatorCreated(deviceAddress, currentOperator);
         }
         else
         {
            ETG_TRACE_ERR(("onNetworkCurrentOperatorChanged(): No device found"));
         }
      }
      else
      {
         ETG_TRACE_ERR(("onNetworkCurrentOperatorChanged(): CurrentOperator ObjPath is empty"));
      }
   }

   void EvoBtStackWrapper::onSiriStatusChanged(const ObjectPath& objPath, const uint8_t status)
   {
      BdAddress deviceAddress;

      ObjectPathInfoListHandler::getInstance().getDeviceAddress(objPath, deviceAddress);
      if(false == deviceAddress.empty())
      {
         std::shared_ptr<PropertyDetails<uint8_t>> propertyDetails =
               std::make_shared<PropertyDetails<uint8_t>>(deviceAddress, status);

         PropertyUpdateNotifierToCore::getInstance().notifyControllers(BTS_UPDATE_SIRI_STATUS, propertyDetails);
      }
      else
      {
         ETG_TRACE_ERR(("onSiriStatusChanged(): No device found"));
      }
   }

   void EvoBtStackWrapper::onSiriNRChanged(const ObjectPath& objPath, const uint8_t noiseReduction)
   {
      BdAddress deviceAddress;

      ObjectPathInfoListHandler::getInstance().getDeviceAddress(objPath, deviceAddress);
      if(false == deviceAddress.empty())
      {
         std::shared_ptr<PropertyDetails<uint8_t>> propertyDetails =
               std::make_shared<PropertyDetails<uint8_t>>(deviceAddress, noiseReduction);

         PropertyUpdateNotifierToCore::getInstance().notifyControllers(BTS_UPDATE_NOISE_REDUCTION, propertyDetails);
      }
      else
      {
         ETG_TRACE_ERR(("onSiriNRChanged(): No device found"));
      }
   }

   void EvoBtStackWrapper::onSiriEyesFreeModeChanged(const ObjectPath& objPath, const bool enabled)
   {
      BdAddress deviceAddress;

      ObjectPathInfoListHandler::getInstance().getDeviceAddress(objPath, deviceAddress);
      if(false == deviceAddress.empty())
      {
         std::shared_ptr<PropertyDetails<bool>> propertyDetails =
               std::make_shared<PropertyDetails<bool>>(deviceAddress, enabled);

         PropertyUpdateNotifierToCore::getInstance().notifyControllers(BTS_UPDATE_EYES_FREE_MODE, propertyDetails);
      }
      else
      {
         ETG_TRACE_ERR(("onSiriEyesFreeModeChanged(): No device found"));
      }
   }

   void EvoBtStackWrapper::onSiriSupportedTypeChanged(const ObjectPath& objPath, const uint16_t supportedType)
   {
      BdAddress deviceAddress;

      ObjectPathInfoListHandler::getInstance().getDeviceAddress(objPath, deviceAddress);
      if(false == deviceAddress.empty())
      {
         std::shared_ptr<PropertyDetails<uint16_t>> propertyDetails =
               std::make_shared<PropertyDetails<uint16_t>>(deviceAddress, supportedType);

         PropertyUpdateNotifierToCore::getInstance().notifyControllers(BTS_UPDATE_SIRI_SUPPORTED_TYPE, propertyDetails);
      }
      else
      {
         ETG_TRACE_ERR(("onSiriSupportedTypeChanged(): No device found"));
      }
   }

   void EvoBtStackWrapper::onEmergencyNumbersChanged(const ObjectPath& objPath,
         const std::vector<std::string>& emergencyNumbers)
   {
      BdAddress deviceAddress;

      ObjectPathInfoListHandler::getInstance().getDeviceAddress(objPath, deviceAddress);
      if(false == deviceAddress.empty())
      {
         std::shared_ptr<PropertyDetails< ::std::vector<std::string>>> propertyDetails =
               std::make_shared<PropertyDetails< ::std::vector<std::string>>>(deviceAddress, emergencyNumbers);

         PropertyUpdateNotifierToCore::getInstance().notifyControllers(BTS_UPDATE_EMERGENCY_NUMBERS, propertyDetails);
      }
      else
      {
         ETG_TRACE_ERR(("onEmergencyNumbersChanged(): No device found"));
      }
   }

   void EvoBtStackWrapper::onVoiceCallLineIdentificationChanged(const ObjectPath& objPath,
         const std::string& lineIdentification)
   {
      BdAddress deviceAddress;
      CallInstance callInstance;

      ObjectPathInfoListHandler::getInstance().getDeviceAddressAndCallInstance(objPath, deviceAddress, callInstance);
      if(false == deviceAddress.empty())
      {
         VoiceCallProperty property;
         property._callInstance = callInstance;
         property._propertyId = VOICE_CALL_LINEIDENTIFICATION;
         property._value = (void*)&lineIdentification;

         std::shared_ptr<PropertyDetails<VoiceCallProperty>> propertyDetails =
               std::make_shared<PropertyDetails<VoiceCallProperty>>(deviceAddress, property);

         PropertyUpdateNotifierToCore::getInstance().notifyControllers(BTS_UPDATE_VOICE_CALL_STATUS,
               propertyDetails);
      }
      else
      {
         ETG_TRACE_ERR(("onVoiceCallLineIdentificationChanged(): No device found"));
      }
   }

   void EvoBtStackWrapper::onVoiceCallNameChanged(const ObjectPath& objPath, const std::string& name)
   {
      BdAddress deviceAddress;
      CallInstance callInstance;

      ObjectPathInfoListHandler::getInstance().getDeviceAddressAndCallInstance(objPath, deviceAddress, callInstance);
      if(false == deviceAddress.empty())
      {
         VoiceCallProperty property;
         property._callInstance = callInstance;
         property._propertyId = VOICE_CALL_NAME;
         property._value = (void*)&name;

         std::shared_ptr<PropertyDetails<VoiceCallProperty>> propertyDetails =
               std::make_shared<PropertyDetails<VoiceCallProperty>>(deviceAddress, property);

         PropertyUpdateNotifierToCore::getInstance().notifyControllers(BTS_UPDATE_VOICE_CALL_STATUS, propertyDetails);
      }
      else
      {
         ETG_TRACE_ERR(("onVoiceCallNameChanged(): No device found"));
      }
   }

   void EvoBtStackWrapper::onVoiceCallMultipartyChanged(const ObjectPath& objPath, const bool multiparty)
   {
      BdAddress deviceAddress;
      CallInstance callInstance;

      ObjectPathInfoListHandler::getInstance().getDeviceAddressAndCallInstance(objPath, deviceAddress, callInstance);
      if(false == deviceAddress.empty())
      {
         VoiceCallProperty property;
         property._callInstance = callInstance;
         property._propertyId = VOICE_CALL_MULTIPARTY;
         property._value = (void*)&multiparty;

         std::shared_ptr<PropertyDetails<VoiceCallProperty>> propertyDetails =
               std::make_shared<PropertyDetails<VoiceCallProperty>>(deviceAddress, property);

         PropertyUpdateNotifierToCore::getInstance().notifyControllers(BTS_UPDATE_VOICE_CALL_STATUS,
               propertyDetails);
      }
      else
      {
         ETG_TRACE_ERR(("onVoiceCallMultipartyChanged(): No device found"));
      }
   }

   void EvoBtStackWrapper::onVoiceCallStateChanged(const ObjectPath& objPath, const std::string& state)
   {
      BdAddress deviceAddress;
      CallInstance callInstance;

      ObjectPathInfoListHandler::getInstance().getDeviceAddressAndCallInstance(objPath, deviceAddress, callInstance);
      if(false == deviceAddress.empty())
      {
         VoiceCallProperty property;
         property._callInstance = callInstance;
         property._propertyId = VOICE_CALL_STATE;
         property._value = (void*)&state;

         std::shared_ptr<PropertyDetails<VoiceCallProperty>> propertyDetails =
               std::make_shared<PropertyDetails<VoiceCallProperty>>(deviceAddress, property);

         PropertyUpdateNotifierToCore::getInstance().notifyControllers(BTS_UPDATE_VOICE_CALL_STATUS,
               propertyDetails);
      }
      else
      {
         ETG_TRACE_ERR(("onVoiceCallStateChanged(): No device found"));
      }
   }

   void EvoBtStackWrapper::onVoiceCallEmergencyChanged(const ObjectPath& objPath, const bool emergency)
   {
      BdAddress deviceAddress;
      CallInstance callInstance;

      ObjectPathInfoListHandler::getInstance().getDeviceAddressAndCallInstance(objPath, deviceAddress, callInstance);
      if(false == deviceAddress.empty())
      {
         VoiceCallProperty property;
         property._callInstance = callInstance;
         property._propertyId = VOICE_CALL_EMERGENCY;
         property._value = (void*)&emergency;

         std::shared_ptr<PropertyDetails<VoiceCallProperty>> propertyDetails =
               std::make_shared<PropertyDetails<VoiceCallProperty>>(deviceAddress, property);

         PropertyUpdateNotifierToCore::getInstance().notifyControllers(BTS_UPDATE_VOICE_CALL_STATUS,
               propertyDetails);
      }
      else
      {
         ETG_TRACE_ERR(("onVoiceCallEmergencyChanged(): No device found"));
      }
   }

   void EvoBtStackWrapper::onVoiceCallDirectionChanged(const ObjectPath& objPath, const std::string& direction)
   {
      BdAddress deviceAddress;
      CallInstance callInstance;

      ObjectPathInfoListHandler::getInstance().getDeviceAddressAndCallInstance(objPath, deviceAddress, callInstance);
      if(false == deviceAddress.empty())
      {
         CallDirection callDirection;

         convertVoiceCallDirectionToPmCoreType(direction, callDirection);

         VoiceCallProperty property;
         property._callInstance = callInstance;
         property._propertyId = VOICE_CALL_DIRECTION;
         property._value = (void*)&callDirection;

         std::shared_ptr<PropertyDetails<VoiceCallProperty>> propertyDetails =
               std::make_shared<PropertyDetails<VoiceCallProperty>>(deviceAddress, property);

         PropertyUpdateNotifierToCore::getInstance().notifyControllers(BTS_UPDATE_VOICE_CALL_STATUS,
               propertyDetails);
      }
      else
      {
         ETG_TRACE_ERR(("onVoiceCallDirectionChanged(): No device found"));
      }
   }

   void EvoBtStackWrapper::onVoiceCallModeChanged(const ObjectPath& objPath, const std::string& mode)
   {
      BdAddress deviceAddress;
      CallInstance callInstance;

      ObjectPathInfoListHandler::getInstance().getDeviceAddressAndCallInstance(objPath, deviceAddress, callInstance);
      if(false == deviceAddress.empty())
      {
         CallMode callMode;

         convertVoiceCallModeToPmCoreType(mode, callMode);

         VoiceCallProperty property;
         property._callInstance = callInstance;
         property._propertyId = VOICE_CALL_MODE;
         property._value = (void*)&callMode;

         std::shared_ptr<PropertyDetails<VoiceCallProperty>> propertyDetails =
               std::make_shared<PropertyDetails<VoiceCallProperty>>(deviceAddress, property);

         PropertyUpdateNotifierToCore::getInstance().notifyControllers(BTS_UPDATE_VOICE_CALL_STATUS, propertyDetails);
      }
      else
      {
         ETG_TRACE_ERR(("onVoiceCallModeChanged(): No device found"));
      }
   }

   void EvoBtStackWrapper::onVoiceCallTypeChanged(const ObjectPath& objPath, const std::string& type)
   {
      BdAddress deviceAddress;
      CallInstance callInstance;

      ObjectPathInfoListHandler::getInstance().getDeviceAddressAndCallInstance(objPath, deviceAddress, callInstance);
      if(false == deviceAddress.empty())
      {
         NumberType numberType;

         convertVoiceCallTypeToPmCoreType(type, numberType);

         VoiceCallProperty property;
         property._callInstance = callInstance;
         property._propertyId = VOICE_CALL_TYPE;
         property._value = (void*)&numberType;

         std::shared_ptr<PropertyDetails<VoiceCallProperty>> propertyDetails =
               std::make_shared<PropertyDetails<VoiceCallProperty>>(deviceAddress, property);

         PropertyUpdateNotifierToCore::getInstance().notifyControllers(BTS_UPDATE_VOICE_CALL_STATUS, propertyDetails);
      }
      else
      {
         ETG_TRACE_ERR(("onVoiceCallTypeChanged(): No device found"));
      }
   }

   ::ccdbusif::evolution::ErrorCode EvoBtStackWrapper::getErrorCode2Enum(const ErrorName& errorName,
         const ErrorMessage& errorMessage)
   {
      ::ccdbusif::evolution::EvolutionGeniviDbusParser parser;
      parser.setTraces(true);

      return parser.getErrorCode2Enum(errorName, errorMessage);
   }

   uint8_t EvoBtStackWrapper::getInstanceFromObjectPath(const ObjectPath& objPath, const uint8_t length)
   {
      ETG_TRACE_USR1(("getInstanceFromObjectPath: %s", objPath.c_str()));
      uint8_t retValue = DEFAULT_INSTANCE_VALUE;

      if(static_cast<uint8_t>(objPath.size()) > length)
      {
         //Copy the Instance from ObjectPath
         size_t copySize = strlen(objPath.c_str() + length) + 1;

         char* instance = new char[copySize];
         if(nullptr != instance)
         {
            memmove(instance, objPath.c_str() + length, copySize-1);
            instance[copySize-1] = '\0';

            retValue = static_cast<uint8_t>(std::atoi(instance));

            if(!retValue) // If no numeric characters found, then update default value
            {
               retValue = DEFAULT_INSTANCE_VALUE;
            }

            delete[] instance;
         }
         else
         {
            ETG_TRACE_ERR(("getInstanceFromObjectPath: invalid instance"));
         }

      }

      return retValue;
   }

   void EvoBtStackWrapper::mapDbusActToPmCoreAct(const act_t dbusAct, const ActType pmCoreAct)
   {
      std::map<act_t, ActType>::iterator it = _dbusAndPmCoreActMap.find(dbusAct);
      if(it != _dbusAndPmCoreActMap.end())
      {
         //this should never happen
         ETG_TRACE_ERR(("Received DBus ACT is already available in the map"));
         it->second = pmCoreAct; //FIXME: fix this scenario
         //NORMAL_M_ASSERT_ALWAYS();
      }
      else
      {
         _dbusAndPmCoreActMap.insert(std::pair<act_t, ActType>(dbusAct, pmCoreAct));
      }
   }

   void EvoBtStackWrapper::retrievePmCoreActFromDbusAct(const act_t dbusAct, ActType& pmCoreAct)
   {
      std::map<act_t, ActType>::iterator it = _dbusAndPmCoreActMap.find(dbusAct);
      if(it != _dbusAndPmCoreActMap.end())
      {
         pmCoreAct = it->second;

         //clear the dbus act_t once the required response is received
         _dbusAndPmCoreActMap.erase(it);
      }
      else
      {
         //this should never happen
         ETG_TRACE_ERR(("Received DBus ACT is not available in the map"));
         //NORMAL_M_ASSERT_ALWAYS();
      }
   }

   void EvoBtStackWrapper::clearDbusAndPmCoreActMap()
   {
      _dbusAndPmCoreActMap.clear();
   }

   void EvoBtStackWrapper::mapDbusActToTelephoneNumber(const act_t dbusAct, const TelephoneNumber telephoneNumber)
   {
      std::map<act_t, TelephoneNumber>::iterator it = _dbusActToTelephoneNumber.find(dbusAct);
      if(it != _dbusActToTelephoneNumber.end())
      {
         //this should never happen
         ETG_TRACE_ERR(("Received DBus ACT is already available in the dbusActToTelephoneNumber map"));
         it->second = telephoneNumber; //FIXME: fix this scenario
         //NORMAL_M_ASSERT_ALWAYS();
      }
      else
      {
         _dbusActToTelephoneNumber.insert(std::pair<act_t, TelephoneNumber>(dbusAct, telephoneNumber));
      }
   }

   void EvoBtStackWrapper::retrieveTelephoneNumberFromDbusAct(const act_t dbusAct, TelephoneNumber& telephoneNumber)
   {
      std::map<act_t, TelephoneNumber>::iterator it = _dbusActToTelephoneNumber.find(dbusAct);
      if(it != _dbusActToTelephoneNumber.end())
      {
         telephoneNumber = it->second;

         //clear the dbus act_t once the required response is received
         _dbusActToTelephoneNumber.erase(it);
      }
      else
      {
         //this should never happen
         ETG_TRACE_ERR(("Received DBus ACT is not available in the dbusActToTelephoneNumber map"));
         //NORMAL_M_ASSERT_ALWAYS();
      }
   }

   void EvoBtStackWrapper::clearDbusActToTelephoneNumberMap()
   {
      _dbusActToTelephoneNumber.clear();
   }

   void EvoBtStackWrapper::convertSupportedFeaturesToPmCoreType(const std::vector<std::string>& supportedFeaturesIn,
               SupportedFeatures& supportedFeaturesOut)
   {
      std::vector<std::string>::const_iterator it;
      for(it = supportedFeaturesIn.begin(); it != supportedFeaturesIn.end(); it++)
      {
         if (0 == it->compare("Voice-recognition"))
         {
            supportedFeaturesOut._supportedFeatures.push_back("voice_recognition");
         }
         else if (0 == it->compare("Attach-voice-tag"))
         {
            supportedFeaturesOut._supportedFeatures.push_back("attach_voice_tag");
         }
         else if (0 == it->compare("3way"))
         {
            supportedFeaturesOut._supportedFeatures.push_back("3way");
         }
         else if (0 == it->compare("ECNR"))
         {
            supportedFeaturesOut._supportedFeatures.push_back("ecnr");
         }
         else if (0 == it->compare("Inband_ring"))
         {
            supportedFeaturesOut._supportedFeatures.push_back("inband_ringtone");
         }
         else if (0 == it->compare("Reject_incoming_call"))
         {
            supportedFeaturesOut._supportedFeatures.push_back("reject_incoming_call");
         }
         else if (0 == it->compare("Enhanced_call_status"))
         {
            supportedFeaturesOut._supportedFeatures.push_back("enhanced_call_status");
         }
         else if (0 == it->compare("Enhanced_call_control"))
         {
            supportedFeaturesOut._supportedFeatures.push_back("enhanced_call_control");
         }
         else if (0 == it->compare("Extended_error_codes"))
         {
            supportedFeaturesOut._supportedFeatures.push_back("extended_error_codes");
         }
         else if (0 == it->compare("Codec_negotiation"))
         {
            supportedFeaturesOut._supportedFeatures.push_back("codec_negotiation");
         }
      }
   }

   void EvoBtStackWrapper::convertHfpAudioToPmCoreType(const std::string& audio, AudioDirection& audioDirection)
   {
      audioDirection = AUDIO_DIRECTION_HANDSFREE;

      if (0 == audio.compare("HF"))
      {
         audioDirection = AUDIO_DIRECTION_HANDSFREE; // audio is in handsfree
      }
      else if (0 == audio.compare("AG"))
      {
         audioDirection = AUDIO_DIRECTION_AUDIOGATEWAY; // audio is in audio gateway
      }
      else
      {
         ETG_TRACE_ERR(("convertHfpAudioToPmCoreType: Unknown audio direction: %s", audio.c_str()));
      }
   }

   void EvoBtStackWrapper::convertNetworkStatusToPmCoreType(const std::string& status,
         NetworkStatusType& networkStatus)
   {
      networkStatus = NETWORK_STATUS_DEFAULT;

      if (0 == status.compare("Unregistered")) //Not registered to any network
      {
         networkStatus = NETWORK_STATUS_UNREGISTERED;
      }
      else if (0 == status.compare("Registered")) //Registered to home network
      {
         networkStatus = NETWORK_STATUS_REGISTERED;
      }
      else if (0 == status.compare("Searching")) //Not registered, but searching
      {
         networkStatus = NETWORK_STATUS_SEARCHING;
      }
      else if (0 == status.compare("Denied")) //Registration has been denied
      {
         networkStatus = NETWORK_STATUS_DENIED;
      }
      else if (0 == status.compare("Unknown")) //Status is unknown
      {
         networkStatus = NETWORK_STATUS_UNKNOWN;
      }
      else if (0 == status.compare("Roaming")) //Registered, but roaming
      {
         networkStatus = NETWORK_STATUS_ROAMING;
      }
      else
      {
         ETG_TRACE_ERR(("convertNetworkStatusToPmCoreType: unknown NetworkStatus: %s", status.c_str()));
      }
   }

   void EvoBtStackWrapper::convertVoiceCallDirectionToPmCoreType(const std::string& direction,
         ::pmcore::CallDirection& callDirection)
   {
      callDirection = CALL_DIRECTION_UNKNOWN;

      if (0 == direction.compare("Unknown"))
      {
         callDirection = CALL_DIRECTION_UNKNOWN;
      }
      else if (0 == direction.compare("Outgoing"))
      {
         callDirection = CALL_DIRECTION_OUTGOING;
      }
      else if (0 == direction.compare("Incoming"))
      {
         callDirection = CALL_DIRECTION_INCOMING;
      }
      else
      {
         ETG_TRACE_ERR(("convertVoiceCallDirectionToPmCoreType: unknown call direction: %s", direction.c_str()));
      }
   }

   void EvoBtStackWrapper::convertVoiceCallModeToPmCoreType(const std::string& mode, ::pmcore::CallMode& callMode)
   {
      callMode = CALL_MODE_UNKNOWN;

      if (0 == mode.compare("Unknown"))
      {
         callMode = CALL_MODE_UNKNOWN;
      }
      else if (0 == mode.compare("Voice"))
      {
         callMode = CALL_MODE_VOICE;
      }
      else if (0 == mode.compare("Data"))
      {
         callMode = CALL_MODE_DATA;
      }
      else if (0 == mode.compare("FAX"))
      {
         callMode = CALL_MODE_FAX;
      }
      else
      {
         ETG_TRACE_ERR(("convertVoiceCallModeToPmCoreType: unknown voice call mode: %s", mode.c_str()));
      }
   }

   void EvoBtStackWrapper::convertVoiceCallTypeToPmCoreType(const std::string& type, ::pmcore::NumberType& numberType)
   {
      numberType = NUMBER_TYPE_UNKNOWN;

      if (0 == type.compare("Unknown"))
      {
         numberType = NUMBER_TYPE_UNKNOWN;
      }
      else if (0 == type.compare("International"))
      {
         numberType = NUMBER_TYPE_INTERNATIONAL;
      }
      else if (0 == type.compare("National"))
      {
         numberType = NUMBER_TYPE_NATIONAL;
      }
      else if (0 == type.compare("Network"))
      {
         numberType = NUMBER_TYPE_NETWORK;
      }
      else if (0 == type.compare("Dedicated"))
      {
         numberType = NUMBER_TYPE_DEDICATED;
      }
      else
      {
         ETG_TRACE_ERR(("convertVoiceCallTypeToPmCoreType: unknown voice call type: %s", type.c_str()));
      }
   }

   void EvoBtStackWrapper::convertCallerIdToDbusType(const HideCallerId hideCallerId, std::string& callerId)
   {
      switch(hideCallerId)
      {
         case SHOW_CALLERID:
            callerId = "Enabled";
            break;
         case HIDE_CALLERID:
            callerId = "Disabled";
            break;
         case DEFAULT_CALLERID: //intentional fall through
         default:
            callerId = "Default";
            break;
      }
   }

   void EvoBtStackWrapper::convertBTSDbusPropertyListToBTSVoiceCallInfo(const BTSDbusPropertyList& propertyList,
         BTSVoiceCallInfo& btsVoiceCallInfo)
   {
      for(size_t i = 0; i < propertyList.size(); i++)
      {
         switch((::ccdbusif::evolution::HfpModemProperty)propertyList[i].propEnum)
         {
            case ::ccdbusif::evolution::HFP_VOICE_CALL_LINEIDENTIFICATION:
            {
               btsVoiceCallInfo._callInfo._telephoneNumber = propertyList[i].propData.getString();
            }
            break;
            case ::ccdbusif::evolution::HFP_VOICE_CALL_NAME:
            {
               btsVoiceCallInfo._contactName = propertyList[i].propData.getString();
            }
            break;
            case ::ccdbusif::evolution::HFP_VOICE_CALL_MULTIPARTY:
            {
               btsVoiceCallInfo._multiparty = propertyList[i].propData.getBool();
            }
            break;
            case ::ccdbusif::evolution::HFP_VOICE_CALL_STATE:
            {
               btsVoiceCallInfo._callInfo._state = propertyList[i].propData.getString();
            }
            break;
            case ::ccdbusif::evolution::HFP_VOICE_CALL_EMERGENCY:
            {
               btsVoiceCallInfo._emergency = propertyList[i].propData.getBool();
            }
            break;
            case ::ccdbusif::evolution::HFP_VOICE_CALL_DIRECTION:
            {
               convertVoiceCallDirectionToPmCoreType(propertyList[i].propData.getString(),
                     btsVoiceCallInfo._callInfo._direction);
            }
            break;
            case ::ccdbusif::evolution::HFP_VOICE_CALL_MODE:
            {
               convertVoiceCallModeToPmCoreType(propertyList[i].propData.getString(),
                     btsVoiceCallInfo._callInfo._mode);
            }
            break;
            case ::ccdbusif::evolution::HFP_VOICE_CALL_TYPE:
            {
               convertVoiceCallTypeToPmCoreType(propertyList[i].propData.getString(),
                     btsVoiceCallInfo._callInfo._type);
            }
            break;
            default:
            {
               ETG_TRACE_ERR(("Unknown VoiceCall Property: %d", propertyList[i].propEnum));
            }
            break;
         }
      }
   }

} //evobtstackwrapper
