/**
 * @file ObjectPathInfoListHandler.h
 *
 * @swcomponent PhoneCallManager
 *
 * @brief This file contains the definition of the ObjectPathInfoListHandler class
 *
 * @copyright (C) 2016 Robert Bosch GmbH.
 *            The reproduction, distribution and utilization of this file as
 *            well as the communication of its contents to others without express
 *            authorization is prohibited. Offenders will be held liable for the
 *            payment of damages. All rights reserved in the event of the grant
 *            of a patent, utility model or design.
 *
 * @details
 *
 * @ingroup BtStackWrapper
 */

#ifndef ObjectPathInfoListHandler_h
#define ObjectPathInfoListHandler_h

#include "PmInterfaceTypesInternal.h"
#include "ObjectPathInfo.h"
#include "PmSingleton.h"

namespace evobtstackwrapper {

/**
 *
 */
class ObjectPathInfoListHandler : public PmSingleton <ObjectPathInfoListHandler>
{
public:
   /**
    * Constructor of ObjectPathInfoListHandler class
    *
    * @param[in]
    * @param[out]
    * @param[in,out]
    *
    * @return
    */
   ObjectPathInfoListHandler();

   /**
    * Destructor of ObjectPathInfoListHandler class
    *
    * @param[in]
    * @param[out]
    * @param[in,out]
    *
    * @return
    */
   ~ObjectPathInfoListHandler();

   /**
   * Copy constructor
   *
   * @param[in] ObjectPathInfoListHandler& ref - Reference to ObjectPathInfoListHandler
   * @param[out]
   * @param[in,out]
   *
   * @return
   *
   */
   ObjectPathInfoListHandler(const ObjectPathInfoListHandler& ref);

  /**
   * operator= overloaded function
   *
   * @param[in]
   * @param[out]
   * @param[in,out]
   *
   * @return Returns the object of the ObjectPathInfoListHandler
   *
   */
   ObjectPathInfoListHandler& operator=(const ObjectPathInfoListHandler& ref);

   /**
    * This method is used to add the device in ObjectPathInfoList
    *
    * @param[in] deviceAddress: device address
    * @param[out]
    * @param[in,out]
    *
    * @return void
    */
   void addDeviceInObjectPathInfoListHandler(const ::pmcore::BdAddress& deviceAddress);

   /**
    * This method is used to remove the device in ObjectPathInfoList
    *
    * @param[in] deviceAddress: device address
    * @param[out]
    * @param[in,out]
    *
    * @return void
    */
   void removeDeviceFromObjectPathInfoListHandler(const ::pmcore::BdAddress& deviceAddress);

   /**
    * This method is used to set the Handsfree object path for the given device address
    *
    * @param[in] deviceAddress: device address
    * @param[in] objPath: handsfree object path
    * @param[out]
    * @param[in,out]
    *
    * @return void
    */
   void setHandsfreeObjectPath(const ::pmcore::BdAddress& deviceAddress, const ObjectPath& objPath);

   /**
    * This method is used to get the Handsfree object path
    *
    * @param[in] deviceAddress: device address
    * @param[out] objPath: handsfree object path
    * @param[in,out]
    *
    * @return void
    */
   void getHandsfreeObjectPath(const ::pmcore::BdAddress& deviceAddress, ObjectPath& objPath);

   /**
    * This method is used to get the device address from Handsfree object path
    *
    * @param[in] objPath: handsfree object path
    * @param[out] deviceAddress: device address
    * @param[in,out]
    *
    * @return void
    */
   void getDeviceAddress(const ObjectPath& objPath, ::pmcore::BdAddress& deviceAddress);

   /**
    * This method is used to clear the Handsfree object path for the given device address
    *
    * @param[in]
    * @param[out]
    * @param[in,out]
    *
    * @return void
    */
   void removeHandsfreeObjectPath(const ::pmcore::BdAddress& deviceAddress);

   /**
    * This method is used to set the VoiceCall object path
    *
    * @param[in] callinstance: call instance
    * @param[in] objPath: voice call object path
    * @param[out]
    * @param[in,out]
    *
    * @return void
    */
   void setVoiceCallObjectPath(const ::pmcore::BdAddress& deviceAddress, const CallInstance callInstance,
         const ObjectPath& objPath);

   /**
    * This method is used to get the VoiceCall object path from CallInstance of specific device
    *
    * @param[in] deviceAddress: device address
    * @param[in] callInstance: call instance
    * @param[out] objPath: voice call object path
    * @param[in,out]
    *
    * @return void
    */
   void getVoiceCallObjectPath(const ::pmcore::BdAddress& deviceAddress, const CallInstance callInstance,
         ObjectPath& objPath);

   /**
    * This method is used to get the CallInstance from VoiceCall object path of specific device
    *
    * @param[in] deviceAddress: device address
    * @param[in] objPath: voice call object path
    * @param[out] callInstance: call instance
    * @param[in,out]
    *
    * @return bool
    */
   bool getCallInstance(const ::pmcore::BdAddress& deviceAddress, const ObjectPath& objPath,
         CallInstance& callInstance); // TODO might not be required

   /**
    * This method is used to get the CallInstance from VoiceCall object path of specific device
    *
    * @param[in] objPath: voice call object path
    * @param[out] deviceAddress: device address
    * @param[out] callInstance: call instance
    * @param[in,out]
    *
    * @return bool
    */
   void getDeviceAddressAndCallInstance(const ObjectPath& objPath, ::pmcore::BdAddress& deviceAddress,
         CallInstance& callInstance);

   /**
    * This method is used to clear the VoiceCall object path for the given device address and call instance
    *
    * @param[in] deviceAddress: device address
    * @param[in] callInstance: call instance
    * @param[out]
    * @param[in,out]
    *
    * @return void
    */
   void removeVoiceCallObjectPath(const ::pmcore::BdAddress& deviceAddress, const CallInstance callInstance);

   /**
    * This method is used to set the NetworkOperator object path for the given device address and operator instance
    *
    * @param[in] deviceAddress: device address
    * @param[in] networkOperatorInstance: network operator instance
    * @param[in] objPath: network operator object path
    * @param[out]
    * @param[in,out]
    *
    * @return void
    */
   void setNetworkOperatorObjectPath(const ::pmcore::BdAddress& deviceAddress,
         const NetworkOperatorInstance networkOperatorInstance, const ObjectPath& objPath);

   /**
    * This method is used to get the NetworkOperator object path from NetworkOperatorinstance of specific device
    *
    * @param[in] deviceAddress: device address
    * @param[in] networkOperatorInstance: network operator instance
    * @param[out] objPath: network operator object path
    * @param[in,out]
    *
    * @return void
    */
   void getNetworkOperatorObjectPath(const ::pmcore::BdAddress& deviceAddress,
         const NetworkOperatorInstance networkOperatorInstance, ObjectPath& objPath);

   /**
    * This method is used to get the NetworkOperatorInstance from NetworkOperator object path of specific device
    *
    * @param[in] deviceAddress: device address
    * @param[in] objPath: network operator object path
    * @param[out] networkOperatorInstance: network operator instance
    * @param[in,out]
    *
    * @return bool
    */
   bool getNetworkOperatorInstance(const ::pmcore::BdAddress& deviceAddress, const ObjectPath& objPath,
         NetworkOperatorInstance& networkOperatorInstance); // TODO might not be required

   /**
    * This method is used to get the device address and NetworkOperatorInstance from NetworkOperator object path
    *
    * @param[in] objPath: network operator object path
    * @param[out] deviceAddress: device address
    * @param[out] networkOperatorInstance: network operator instance
    * @param[in,out]
    *
    * @return void
    */
   void getDeviceAddressAndNetworkOperatorInstance(const ObjectPath& objPath,
            ::pmcore::BdAddress& deviceAddress, NetworkOperatorInstance& networkOperatorInstance);

   /**
    * This method is used to clear the NetworkOperator object path for the device address and operator instance
    *
    * @param[in] deviceAddress: device address
    * @param[in] networkOperatorInstance: network operator instance
    * @param[out]
    * @param[in,out]
    *
    * @return void
    */
   void removeNetworkOperatorObjectPath(const ::pmcore::BdAddress& deviceAddress,
         const NetworkOperatorInstance networkOperatorInstance);

private:
   friend class PmSingleton<ObjectPathInfoListHandler>;

   ObjectPathInfoList _objectPathInfoList; /**< map of device address and object path info */
};

} //evobtstackwrapper

#endif //ObjectPathInfoListHandler_h
