/**
 * @file ObjectPathInfoListHandler.h
 *
 * @swcomponent PhoneCallManager
 *
 * @brief This file contains the definition of the ObjectPathInfoListHandler class
 *
 * @copyright (C) 2016 Robert Bosch GmbH.
 *            The reproduction, distribution and utilization of this file as
 *            well as the communication of its contents to others without express
 *            authorization is prohibited. Offenders will be held liable for the
 *            payment of damages. All rights reserved in the event of the grant
 *            of a patent, utility model or design.
 *
 * @details
 *
 * @ingroup BtStackWrapper
 */

#include "PmInterfaceTypesInternal.h"
#include "ObjectPathInfoListHandler.h"
#include "PmSingleton.h"

namespace evobtstackwrapper {

   ObjectPathInfoListHandler::ObjectPathInfoListHandler()
   {
   }

   ObjectPathInfoListHandler::~ObjectPathInfoListHandler()
   {
      _objectPathInfoList.clear();
   }

   void ObjectPathInfoListHandler::addDeviceInObjectPathInfoListHandler(const ::pmcore::BdAddress& deviceAddress)
   {
      ObjectPathInfoList::iterator it = _objectPathInfoList.find(deviceAddress);
      if(it != _objectPathInfoList.end())
      {
         //device already added in list
      }
      else
      {
         ObjectPathInfo objectPathInfo;
         _objectPathInfoList.insert(std::pair< ::pmcore::BdAddress, ObjectPathInfo>(deviceAddress, objectPathInfo));
      }
   }

   void ObjectPathInfoListHandler::removeDeviceFromObjectPathInfoListHandler(const ::pmcore::BdAddress& deviceAddress)
   {
      ObjectPathInfoList::iterator it = _objectPathInfoList.find(deviceAddress);
      if(it != _objectPathInfoList.end())
      {
         _objectPathInfoList.erase(it);
      }
      else
      {
         //device already removed from list
      }
   }

   void ObjectPathInfoListHandler::setHandsfreeObjectPath(const ::pmcore::BdAddress& deviceAddress,
         const ObjectPath& objPath)
   {
      ObjectPathInfoList::iterator it = _objectPathInfoList.find(deviceAddress);
      if(it != _objectPathInfoList.end())
      {
         it->second.setHandsfreeObjectPath(objPath);
      }
   }

   void ObjectPathInfoListHandler::getHandsfreeObjectPath(const ::pmcore::BdAddress& deviceAddress,
         ObjectPath& objPath)
   {
      ObjectPathInfoList::iterator it = _objectPathInfoList.find(deviceAddress);

      if(it != _objectPathInfoList.end())
      {
         objPath = it->second.getHandsfreeObjectPath();
      }
   }

   void ObjectPathInfoListHandler::getDeviceAddress(const ObjectPath& objPath, ::pmcore::BdAddress& deviceAddress)
   {
      ObjectPathInfoList::iterator it;

      for(it = _objectPathInfoList.begin(); (it != _objectPathInfoList.end()); it++)
      {
         if(it->second.getHandsfreeObjectPath() == objPath)
         {
            deviceAddress = it->first;
            break;
         }
      }
   }

   void ObjectPathInfoListHandler::removeHandsfreeObjectPath(const ::pmcore::BdAddress& deviceAddress)
   {
      ObjectPathInfoList::iterator it = _objectPathInfoList.find(deviceAddress);
      if(it != _objectPathInfoList.end())
      {
         it->second.removeHandsfreeObjectPath();
      }
   }

   void ObjectPathInfoListHandler::setVoiceCallObjectPath(const ::pmcore::BdAddress& deviceAddress,
         const CallInstance callInstance, const ObjectPath& objPath)
   {
      ObjectPathInfoList::iterator it = _objectPathInfoList.find(deviceAddress);
      if(it != _objectPathInfoList.end())
      {
         it->second.setVoiceCallObjectPath(callInstance, objPath);
      }
   }

   void ObjectPathInfoListHandler::getVoiceCallObjectPath(const ::pmcore::BdAddress& deviceAddress,
         const CallInstance callInstance, ObjectPath& objPath)
   {
      ObjectPathInfoList::iterator it = _objectPathInfoList.find(deviceAddress);

      if(it != _objectPathInfoList.end())
      {
         it->second.getVoiceCallObjectPath(callInstance, objPath);
      }
   }

   bool ObjectPathInfoListHandler::getCallInstance(const ::pmcore::BdAddress& deviceAddress,
         const ObjectPath& objPath, CallInstance& callInstance)
   {
      bool retValue = false;
      ObjectPathInfoList::iterator it = _objectPathInfoList.find(deviceAddress);

      if(it != _objectPathInfoList.end())
      {
         retValue = it->second.getCallInstance(objPath, callInstance);
      }

      return retValue;
   }

   void ObjectPathInfoListHandler::getDeviceAddressAndCallInstance(const ObjectPath& objPath,
         ::pmcore::BdAddress& deviceAddress, CallInstance& callInstance)
   {
      ObjectPathInfoList::iterator it;

      for(it = _objectPathInfoList.begin(); (it != _objectPathInfoList.end()); it++)
      {
         if(true == it->second.getCallInstance(objPath, callInstance))
         {
            deviceAddress = it->first;
            break;
         }
      }
   }

   void ObjectPathInfoListHandler::removeVoiceCallObjectPath(const ::pmcore::BdAddress& deviceAddress,
         const CallInstance callInstance)
   {
      ObjectPathInfoList::iterator it = _objectPathInfoList.find(deviceAddress);

      if(it != _objectPathInfoList.end())
      {
         it->second.removeVoiceCallObjectPath(callInstance);
      }
   }

   void ObjectPathInfoListHandler::setNetworkOperatorObjectPath(const ::pmcore::BdAddress& deviceAddress,
         const NetworkOperatorInstance networkOperatorInstance, const ObjectPath& objPath)
   {
      ObjectPathInfoList::iterator it = _objectPathInfoList.find(deviceAddress);
      if(it != _objectPathInfoList.end())
      {
         it->second.setNetworkOperatorObjectPath(networkOperatorInstance, objPath);
      }
   }

   void ObjectPathInfoListHandler::getNetworkOperatorObjectPath(const ::pmcore::BdAddress& deviceAddress,
         const NetworkOperatorInstance networkOperatorInstance, ObjectPath& objPath)
   {
      ObjectPathInfoList::iterator it = _objectPathInfoList.find(deviceAddress);

      if(it != _objectPathInfoList.end())
      {
         it->second.getNetworkOperatorObjectPath(networkOperatorInstance, objPath);
      }
   }

   bool ObjectPathInfoListHandler::getNetworkOperatorInstance(const ::pmcore::BdAddress& deviceAddress,
         const ObjectPath& objPath, NetworkOperatorInstance& networkOperatorInstance)
   {
      bool retValue = false;
      ObjectPathInfoList::iterator it = _objectPathInfoList.find(deviceAddress);

      if(it != _objectPathInfoList.end())
      {
         retValue = it->second.getNetworkOperatorInstance(objPath, networkOperatorInstance);
      }

      return retValue;
   }

   void ObjectPathInfoListHandler::getDeviceAddressAndNetworkOperatorInstance(const ObjectPath& objPath,
         ::pmcore::BdAddress& deviceAddress, NetworkOperatorInstance& networkOperatorInstance)
   {
      ObjectPathInfoList::iterator it;

      for(it = _objectPathInfoList.begin(); (it != _objectPathInfoList.end()); it++)
      {
         if(true == it->second.getNetworkOperatorInstance(objPath, networkOperatorInstance))
         {
            deviceAddress = it->first;
            break;
         }
      }
   }

   void ObjectPathInfoListHandler::removeNetworkOperatorObjectPath(const ::pmcore::BdAddress& deviceAddress,
         const NetworkOperatorInstance networkOperatorInstance)
   {
      ObjectPathInfoList::iterator it = _objectPathInfoList.find(deviceAddress);

      if(it != _objectPathInfoList.end())
      {
         it->second.removeNetworkOperatorObjectPath(networkOperatorInstance);
      }
   }

} //evobtstackwrapper
