/**
 * @file IBtStackWrapper.h
 *
 * @swcomponent PhoneCallManager
 *
 * @brief This file contains the definition of the IBtStackWrapper class
 *
 * @copyright (C) 2016 Robert Bosch GmbH.
 *            The reproduction, distribution and utilization of this file as
 *            well as the communication of its contents to others without express
 *            authorization is prohibited. Offenders will be held liable for the
 *            payment of damages. All rights reserved in the event of the grant
 *            of a patent, utility model or design.
 *
 * @details This file provides the request interfaces of BtStackWrapper.
 *
 * @ingroup BtStackWrapper
 */

#ifndef IBtStackWrapper_h
#define IBtStackWrapper_h

#include "EvolutionGeniviDbusTypes.h"
#include "BtStackWrapperTypes.h"
#include "PmCoreIfTypes.h"
#include "asf/core/Types.h"
#include <string>
#include <vector>

namespace evobtstackwrapper
{
   /**
    * IBtStackWrapper class definition
    */
   class IBtStackWrapper
   {
   public:

      /**
       * Destructor of IBtStackWrapper class
       *
       * @param[in]
       * @param[out]
       * @param[in,out]
       *
       * @return
       */
      virtual ~IBtStackWrapper() {}

      /**
       * This method is used to set Vehicle Configuration
       *
       * @param[in] vehicleConfig: vehicle configuration
       * @param[out]
       * @param[in,out]
       *
       * @return void
       */
      virtual void setVehicleConfiguration(const ::pmcore::VehicleConfiguration& vehicleConfig) = 0;

      /**
       * This method notifies the device connection to BtStackWrapper
       *
       * @param[in] deviceAddress: connected device address
       * @param[out]
       * @param[in,out]
       *
       * @return void
       */
      virtual void onDeviceConnected(const ::pmcore::BdAddress& deviceAddress) = 0;

      /**
       * This method notifies the device disconnection to BtStackWrapper
       *
       * @param[in] deviceAddress: disconnected device address
       * @param[out]
       * @param[in,out]
       *
       * @return void
       */
      virtual void onDeviceDisconnected(const ::pmcore::BdAddress& deviceAddress) = 0;

      /**
       * Handle signal for created device.
       *
       * @param[in] objPath: object path of created device
       * @param[out]
       * @param[in,out]
       *
       * @return void
       */
      virtual void deviceCreated(const ObjectPath& objPath) = 0;

      /**
       * Handle signal for removed device.
       *
       * @param[in] objPath: object path of removed device
       * @param[out]
       * @param[in,out]
       *
       * @return void
       */
      virtual void deviceRemoved(const ObjectPath& objPath) = 0;

      /**
       * Handle signal for created modem.
       *
       * @param[in] objPath: object path of created modem
       * @param[out]
       * @param[in,out]
       *
       * @return void
       */
      virtual void modemCreated(const ::pmcore::BdAddress& deviceAddress, const ObjectPath& objPath) = 0;

      /**
       * Handle signal for removed modem.
       *
       * @param[in] objPath: object path of removed modem
       * @param[out]
       * @param[in,out]
       *
       * @return void
       */
      virtual void modemRemoved(const ObjectPath& objPath) = 0;

      /**
       * Handle signal for created network operator.
       *
       * @param[in] objPath: object path of created operator
       * @param[out]
       * @param[in,out]
       *
       * @return void
       */
      virtual void networkOperatorCreated(const ::pmcore::BdAddress& deviceAddress, const ObjectPath& objPath) = 0;

      /**
       * Handle signal for removed network operator.
       *
       * @param[in] objPath: object path of removed operator
       * @param[out]
       * @param[in,out]
       *
       * @return void
       */
      virtual void networkOperatorRemoved(const ObjectPath& objPath) = 0;

      /**
       * Handle signal for created voice call.
       *
       * @param[in] objPath: object path of created call
       * @param[out]
       * @param[in,out]
       *
       * @return void
       */
      virtual void voiceCallCreated(const ObjectPath& voiceCallMgrObjPath, const ObjectPath& voiceCallObjPath,
            BTSDbusPropertyList& propertyList) = 0;

      /**
       * Handle signal for removed voice call.
       *
       * @param[in] objPath: object path of removed call
       * @param[out]
       * @param[in,out]
       *
       * @return void
       */
      virtual void voiceCallRemoved(const ObjectPath& objPath) = 0;

      /**
       * This function is used to update the BT Stack proxy availability status to PmCore.
       *
       * @param[in] interface: interface, which is updated
       * @param[in] interface: object path
       * @param[in] serviceAvailability: service availability status
       * @param[out]
       * @param[in,out]
       *
       * @return void
       */
      virtual void onProxyServiceAvailabilityStatus(const ::ccdbusif::evolution::Interface interface,
            const ObjectPath& objPath, const bool serviceAvailability) = 0;

      /**
       * Send SetProperty Request.
       *
       * @param[in] deviceAddress: device address
       * @param[in] property: property name to be changed
       * @param[in] value: value to be updated
       * @param[out]
       * @param[in,out]
       *
       * @return BTSResult
       */
      virtual ::pmcore::BTSResult sendSetPropertyRequest(const ::ccdbusif::evolution::Interface interface,
            const ::pmcore::BdAddress& deviceAddress, const ::std::string& property, const uint8 value) = 0;

      /**
       * Send SetProperty Request.
       *
       * @param[in] deviceAddress: device address
       * @param[in] property: property name to be changed
       * @param[in] value: value to be updated
       * @param[out]
       * @param[in,out]
       *
       * @return BTSResult
       */
      virtual ::pmcore::BTSResult sendSetPropertyRequest(const ::ccdbusif::evolution::Interface interface,
            const ::pmcore::BdAddress& deviceAddress, const ::std::string& property, const bool value) = 0;

      /**
       * Send SetProperty Request.
       *
       * @param[in] objPath: object path of device
       * @param[in] property: property name to be changed
       * @param[in] value: value to be updated
       * @param[out]
       * @param[in,out]
       *
       * @return BTSResult
       */
      virtual ::pmcore::BTSResult sendSetPropertyRequest(const ::ccdbusif::evolution::Interface interface,
            const ::pmcore::BdAddress& deviceAddress, const ::std::string& property, const ::std::string& value) = 0;

      /**
       * This method is called when the response for SetProperty is received for Handsfree proxy
       *
       * @param[in]  objPath - Object Path
       *             interface
       *             errorName
       *             errorMessage
       *             dbusAct
       * @param[out]
       * @param[in,out]
       *
       * @return void
       */
      virtual void onSetPropertyResponse(const ObjectPath& objPath,
            const ::ccdbusif::evolution::Interface interface,
            const ErrorName& errorName, const ErrorMessage& errorMessage, const act_t dbusAct) = 0;

      /**
       * Send RequestPhoneNumber Request.
       *
       * @param[in] deviceAddress: device address
       * @param[in] act: pmcore acknowledgment token
       * @param[out]
       * @param[in,out]
       *
       * @return BTSResult
       */
      virtual ::pmcore::BTSResult sendRequestPhoneNumberRequest(const ::pmcore::BdAddress& deviceAddress,
            const ::pmcore::ActType act) = 0;

      /**
       * Error callback for RequestPhoneNumber request
       *
       * @param[in] objPath: object path
       * @param[in] errorName: error name
       * @param[in] errorMessage: error message
       * @param[in] act: asynchronous completion token
       * @param[out]
       * @param[in,out]
       *
       * @return void
       */
      virtual void onRequestPhoneNumberError(const ObjectPath& objPath, const ErrorName& errorName,
            const ErrorMessage& errorMessage, const act_t act) = 0;

      /**
       * Response callback for RequestPhoneNumber request
       *
       * @param[in] objPath: object path
       * @param[in] telephoneNumber: telephone number
       * @param[in] act: asynchronous completion token
       * @param[out]
       * @param[in,out]
       *
       * @return void
       */
      virtual void onRequestPhoneNumberResponse(const ObjectPath& objPath, const std::string& telephoneNumber,
            const act_t act) = 0;

      /**
       * Send GetSubscriber Request.
       *
       * @param[in] deviceAddress: device address
       * @param[in] act: pmcore acknowledgment token
       * @param[out]
       * @param[in,out]
       *
       * @return BTSResult
       */
      virtual ::pmcore::BTSResult sendGetSubscriberRequest(const ::pmcore::BdAddress& deviceAddress,
            const ::pmcore::ActType act) = 0;

      /**
       * Error callback for GetSubscriber request
       *
       * @param[in] objPath: object path
       * @param[in] errorName: error name
       * @param[in] errorMessage: error message
       * @param[in] act: asynchronous completion token
       * @param[out]
       * @param[in,out]
       *
       * @return void
       */
      virtual void onGetSubscriberError(const ObjectPath& objPath, const ErrorName& errorName,
            const ErrorMessage& errorMessage, const act_t act) = 0;

      /**
       * Response callback for GetSubscriber request
       *
       * @param[in] objPath: object path
       * @param[in] telephoneNumber: telephone number
       * @param[in] act: asynchronous completion token
       * @param[out]
       * @param[in,out]
       *
       * @return void
       */
      virtual void onGetSubscriberResponse(const ObjectPath& objPath, const uint8_t type,
            const std::string& telephoneNumber, const act_t act) = 0;

      /**
       * Send SendCustomATCommand Request.
       *
       * @param[in] deviceAddress: device address
       * @param[in] data: custom AT command data
       * @param[in] act: pmcore acknowledgment token
       * @param[out]
       * @param[in,out]
       *
       * @return BTSResult
       */
      virtual ::pmcore::BTSResult sendSendCustomATCommandRequest(const ::pmcore::BdAddress& deviceAddress,
            const ::std::vector< uint8 >& data, const ::pmcore::ActType act) = 0;

      /**
       * Error callback for SendCustomATCommand request
       *
       * @param[in] objPath: object path
       * @param[in] errorName: error name
       * @param[in] errorMessage: error message
       * @param[in] act: asynchronous completion token
       * @param[out]
       * @param[in,out]
       *
       * @return void
       */
      virtual void onSendCustomATCommandError(const ObjectPath& objPath, const ErrorName& errorName,
            const ErrorMessage& errorMessage, const act_t act) = 0;

      /**
       * Response callback for SendCustomATCommand request
       *
       * @param[in] objPath: object path
       * @param[in] act: asynchronous completion token
       * @param[out]
       * @param[in,out]
       *
       * @return void
       */
      virtual void onSendCustomATCommandResponse(const ObjectPath& objPath, const act_t act) = 0;

      /**
       * Send HFUpdateIndicatorsValue Request.
       *
       * @param[in] deviceAddress: device address
       * @param[in] number: HF indicator number
       * @param[in] value: HF indicator value
       * @param[in] act: pmcore acknowledgment token
       * @param[out]
       * @param[in,out]
       *
       * @return BTSResult
       */
      virtual ::pmcore::BTSResult sendHFUpdateIndicatorsValueRequest(const ::pmcore::BdAddress& deviceAddress,
            uint16 number, uint32 value, const ::pmcore::ActType act) = 0;

      /**
       * Error callback for HFUpdateIndicatorsValue request
       *
       * @param[in] objPath: object path
       * @param[in] errorName: error name
       * @param[in] errorMessage: error message
       * @param[in] act: asynchronous completion token
       * @param[out]
       * @param[in,out]
       *
       * @return void
       */
      virtual void onHFUpdateIndicatorsValueError(const ObjectPath& objPath, const ErrorName& errorName,
            const ErrorMessage& errorMessage, const act_t act) = 0;

      /**
       * Response callback for HFUpdateIndicatorsValue request
       *
       * @param[in] objPath: object path
       * @param[in] act: asynchronous completion token
       * @param[out]
       * @param[in,out]
       *
       * @return void
       */
      virtual void onHFUpdateIndicatorsValueResponse(const ObjectPath& objPath, const act_t act) = 0;

      /**
       * Send GetModems Request.
       *
       * @param[in] act: pmcore acknowledgment token
       * @param[out]
       * @param[in,out]
       *
       * @return BTSResult
       */
      virtual ::pmcore::BTSResult sendGetModemsRequest(const ::pmcore::ActType act) = 0;

      /**
       * Error callback for GetModems request
       *
       * @param[in] objPath: object path
       * @param[in] errorName: error name
       * @param[in] errorMessage: error message
       * @param[in] act: asynchronous completion token
       * @param[out]
       * @param[in,out]
       *
       * @return void
       */
      virtual void onGetModemsError(const ObjectPath& objPath, const ErrorName& errorName,
            const ErrorMessage& errorMessage, const act_t act) = 0;

      /**
       * Response callback for GetModems request
       *
       * @param[in] objPath: object path
       * @param[in] act: asynchronous completion token
       * @param[out]
       * @param[in,out]
       *
       * @return void
       */
      virtual void onGetModemsResponse(const ObjectPath& objPath,
            /*const ::std::vector< GetModemsResponseModemsStruct >& modems,*/ const act_t act) = 0;

      /**
       * Send AcceptSCOConnect Request.
       *
       * @param[in] deviceAddress: device address
       * @param[in] act: pmcore acknowledgment token
       * @param[out]
       * @param[in,out]
       *
       * @return BTSResult
       */
      virtual ::pmcore::BTSResult sendAcceptSCOConnectRequest(const ::pmcore::BdAddress& deviceAddress,
            const ::pmcore::ActType act) = 0;

      /**
       * Error callback for AcceptSCOConnect request
       *
       * @param[in] objPath: object path
       * @param[in] errorName: error name
       * @param[in] errorMessage: error message
       * @param[in] act: asynchronous completion token
       * @param[out]
       * @param[in,out]
       *
       * @return void
       */
      virtual void onAcceptSCOConnectError(const ObjectPath& objPath, const ErrorName& errorName,
            const ErrorMessage& errorMessage, const act_t act) = 0;

      /**
       * Response callback for AcceptSCOConnect request
       *
       * @param[in] objPath: object path
       * @param[in] act: asynchronous completion token
       * @param[out]
       * @param[in,out]
       *
       * @return void
       */
      virtual void onAcceptSCOConnectResponse(const ObjectPath& objPath, const act_t act) = 0;

      /**
       * Callback for SCOConnectRequest signal
       *
       * @param[in] objPath: object path
       * @param[in] type: sco type
       * @param[out]
       * @param[in,out]
       *
       * @return void
       */
      virtual void onSCOConnectRequestSignal(const ObjectPath& objPath, const uint8_t type) = 0;

      /**
       * Send Siri Enable Request.
       *
       * @param[in] deviceAddress: device address
       * @param[in] vendor_id: vendor id of device
       * @param[in] product_id: product id of device
       * @param[in] version: version of device
       * @param[in] support_type: support type of device
       * @param[in] act: pmcore acknowledgment token
       * @param[out]
       * @param[in,out]
       *
       * @return BTSResult
       */
      virtual ::pmcore::BTSResult sendSiriEnableRequest(const ::pmcore::BdAddress& deviceAddress,
             uint16 support_type, const ::pmcore::ActType act) = 0;

      /**
       * Error callback for SiriEnable request
       *
       * @param[in] objPath: object path
       * @param[in] errorName: error name
       * @param[in] errorMessage: error message
       * @param[in] act: asynchronous completion token
       * @param[out]
       * @param[in,out]
       *
       * @return void
       */
      virtual void onSiriEnableError(const ObjectPath& objPath, const ErrorName& errorName,
            const ErrorMessage& errorMessage, const act_t act) = 0;

      /**
       * Response callback for SiriEnable request
       *
       * @param[in] objPath: object path
       * @param[in] act: asynchronous completion token
       * @param[out]
       * @param[in,out]
       *
       * @return void
       */
      virtual void onSiriEnableResponse(const ObjectPath& objPath, const act_t act) = 0;

      /**
       * Send Siri SetNR Request.
       *
       * @param[in] deviceAddress: device address
       * @param[in] status: NR status of the device
       * @param[in] act: pmcore acknowledgment token
       * @param[out]
       * @param[in,out]
       *
       * @return BTSResult
       */
      virtual ::pmcore::BTSResult sendSiriSetNRRequest(const ::pmcore::BdAddress& deviceAddress,
            uint8 status, const ::pmcore::ActType act) = 0;

      /**
       * Error callback for SiriSetNR request
       *
       * @param[in] objPath: object path
       * @param[in] errorName: error name
       * @param[in] errorMessage: error message
       * @param[in] act: asynchronous completion token
       * @param[out]
       * @param[in,out]
       *
       * @return void
       */
      virtual void onSiriSetNRError(const ObjectPath& objPath, const ErrorName& errorName,
            const ErrorMessage& errorMessage, const act_t act) = 0;

      /**
       * Response callback for SiriSetNR request
       *
       * @param[in] objPath: object path
       * @param[in] act: asynchronous completion token
       * @param[out]
       * @param[in,out]
       *
       * @return void
       */
      virtual void onSiriSetNRResponse(const ObjectPath& objPath, const act_t act) = 0;

      /**
       * Send GetCalls Request.
       *
       * @param[in] deviceAddress: device address
       * @param[in] act: pmcore acknowledgment token
       * @param[out]
       * @param[in,out]
       *
       * @return BTSResult
       */
      virtual ::pmcore::BTSResult sendGetCallsRequest(const ::pmcore::BdAddress& deviceAddress,
            const ::pmcore::ActType act) = 0;

      /**
       * Error callback for GetCalls request
       *
       * @param[in] objPath: object path
       * @param[in] errorName: error name
       * @param[in] errorMessage: error message
       * @param[in] act: asynchronous completion token
       * @param[out]
       * @param[in,out]
       *
       * @return void
       */
      virtual void onGetCallsError(const ObjectPath& objPath, const ErrorName& errorName,
            const ErrorMessage& errorMessage, const act_t act) = 0;

      /**
       * Response callback for GetCalls request
       *
       * @param[in] objPath: object path
       * @param[in] act: asynchronous completion token
       * @param[out]
       * @param[in,out]
       *
       * @return void
       */
      virtual void onGetCallsResponse(const ObjectPath& objPath,
            const ::std::map< ObjectPath, BTSDbusPropertyList >& calls, const act_t act) = 0;

      /**
       * Send Dial Request.
       *
       * @param[in] deviceAddress: device address
       * @param[in] number: number to be dialed
       * @param[in] hide_callerid: caller id state
       * @param[in] act: pmcore acknowledgment token
       * @param[out]
       * @param[in,out]
       *
       * @return BTSResult
       */
      virtual ::pmcore::BTSResult sendDialRequest(const ::pmcore::BdAddress& deviceAddress,
            const ::pmcore::TelephoneNumber& number, const ::pmcore::HideCallerId hide_callerid,
            const ::pmcore::ActType act) = 0;

      /**
       * Error callback for Dial request
       *
       * @param[in] objPath: object path
       * @param[in] errorName: error name
       * @param[in] errorMessage: error message
       * @param[in] act: asynchronous completion token
       * @param[out]
       * @param[in,out]
       *
       * @return void
       */
      virtual void onDialError(const ObjectPath& objPath, const ErrorName& errorName,
            const ErrorMessage& errorMessage, const act_t act) = 0;

      /**
       * Response callback for Dial request
       *
       * @param[in] objPath: object path
       * @param[in] path: voice call opject path
       * @param[in] act: asynchronous completion token
       * @param[out]
       * @param[in,out]
       *
       * @return void
       */
      virtual void onDialResponse(const ObjectPath& objPath, const std::string& path, const act_t act) = 0;

      /**
       * Send MemDial Request.
       *
       * @param[in] deviceAddress: device address
       * @param[in] numberindex: index number of the dial request
       * @param[in] hide_callerid: caller id state
       * @param[in] act: pmcore acknowledgment token
       * @param[out]
       * @param[in,out]
       *
       * @return BTSResult
       */
      virtual ::pmcore::BTSResult sendMemDialRequest(const ::pmcore::BdAddress& deviceAddress,
            ::pmcore::MemDialIndex numberIndex, const ::pmcore::HideCallerId hideCallerId,
             const ::pmcore::ActType act) = 0;

      /**
       * Error callback for MemDial request
       *
       * @param[in] objPath: object path
       * @param[in] errorName: error name
       * @param[in] errorMessage: error message
       * @param[in] act: asynchronous completion token
       * @param[out]
       * @param[in,out]
       *
       * @return void
       */
      virtual void onMemDialError(const ObjectPath& objPath, const ErrorName& errorName,
            const ErrorMessage& errorMessage, const act_t act) = 0;

      /**
       * Response callback for MemDial request
       *
       * @param[in] objPath: object path
       * @param[in] path: voice call opject path
       * @param[in] act: asynchronous completion token
       * @param[out]
       * @param[in,out]
       *
       * @return void
       */
      virtual void onMemDialResponse(const ObjectPath& objPath, const std::string& path, const act_t act) = 0;

      /**
       * Send Redial Request.
       *
       * @param[in] deviceAddress: device address
       * @param[in] hide_callerid: caller id state
       * @param[in] act: pmcore acknowledgment token
       * @param[out]
       * @param[in,out]
       *
       * @return BTSResult
       */
      virtual ::pmcore::BTSResult sendRedialRequest(const ::pmcore::BdAddress& deviceAddress,
            const ::pmcore::HideCallerId hide_callerid, const ::pmcore::ActType act) = 0;

      /**
       * Error callback for Redial request
       *
       * @param[in] objPath: object path
       * @param[in] errorName: error name
       * @param[in] errorMessage: error message
       * @param[in] act: asynchronous completion token
       * @param[out]
       * @param[in,out]
       *
       * @return void
       */
      virtual void onRedialError(const ObjectPath& objPath, const ErrorName& errorName,
            const ErrorMessage& errorMessage, const act_t act) = 0;

      /**
       * Response callback for Redial request
       *
       * @param[in] objPath: object path
       * @param[in] path: voice call opject path
       * @param[in] act: asynchronous completion token
       * @param[out]
       * @param[in,out]
       *
       * @return void
       */
      virtual void onRedialResponse(const ObjectPath& objPath, const std::string& path, const act_t act) = 0;

      /**
       * Send SwapCalls Request.
       *
       * @param[in] deviceAddress: device address
       * @param[in] act: pmcore acknowledgment token
       * @param[out]
       * @param[in,out]
       *
       * @return BTSResult
       */
      virtual ::pmcore::BTSResult sendSwapCallsRequest(const ::pmcore::BdAddress& deviceAddress,
            const ::pmcore::ActType act) = 0;

      /**
       * Error callback for SwapCalls request
       *
       * @param[in] objPath: object path
       * @param[in] errorName: error name
       * @param[in] errorMessage: error message
       * @param[in] act: asynchronous completion token
       * @param[out]
       * @param[in,out]
       *
       * @return void
       */
      virtual void onSwapCallsError(const ObjectPath& objPath, const ErrorName& errorName,
            const ErrorMessage& errorMessage, const act_t act) = 0;

      /**
       * Response callback for SwapCalls request
       *
       * @param[in] objPath: object path
       * @param[in] act: asynchronous completion token
       * @param[out]
       * @param[in,out]
       *
       * @return void
       */
      virtual void onSwapCallsResponse(const ObjectPath& objPath, const act_t act) = 0;

      /**
       * Send ReleaseAndAnswer Request.
       *
       * @param[in] deviceAddress: device address
       * @param[in] act: pmcore acknowledgment token
       * @param[out]
       * @param[in,out]
       *
       * @return BTSResult
       */
      virtual ::pmcore::BTSResult sendReleaseAndAnswerRequest(const ::pmcore::BdAddress& deviceAddress,
            const ::pmcore::ActType act) = 0;

      /**
       * Error callback for ReleaseAndAnswer request
       *
       * @param[in] objPath: object path
       * @param[in] errorName: error name
       * @param[in] errorMessage: error message
       * @param[in] act: asynchronous completion token
       * @param[out]
       * @param[in,out]
       *
       * @return void
       */
      virtual void onReleaseAndAnswerError(const ObjectPath& objPath, const ErrorName& errorName,
            const ErrorMessage& errorMessage, const act_t act) = 0;

      /**
       * Response callback for ReleaseAndAnswer request
       *
       * @param[in] objPath: object path
       * @param[in] act: asynchronous completion token
       * @param[out]
       * @param[in,out]
       *
       * @return void
       */
      virtual void onReleaseAndAnswerResponse(const ObjectPath& objPath, const act_t act) = 0;

      /**
       * Send ReleaseAndSwap Request.
       *
       * @param[in] deviceAddress: device address
       * @param[in] act: pmcore acknowledgment token
       * @param[out]
       * @param[in,out]
       *
       * @return BTSResult
       */
      virtual ::pmcore::BTSResult sendReleaseAndSwapRequest(const ::pmcore::BdAddress& deviceAddress,
            const ::pmcore::ActType act) = 0;

      /**
       * Error callback for ReleaseAndSwap request
       *
       * @param[in] objPath: object path
       * @param[in] errorName: error name
       * @param[in] errorMessage: error message
       * @param[in] act: asynchronous completion token
       * @param[out]
       * @param[in,out]
       *
       * @return void
       */
      virtual void onReleaseAndSwapError(const ObjectPath& objPath, const ErrorName& errorName,
            const ErrorMessage& errorMessage, const act_t act) = 0;

      /**
       * Response callback for ReleaseAndSwap request
       *
       * @param[in] objPath: object path
       * @param[in] act: asynchronous completion token
       * @param[out]
       * @param[in,out]
       *
       * @return void
       */
      virtual void onReleaseAndSwapResponse(const ObjectPath& objPath, const act_t act) = 0;

      /**
       * Send HoldAndAnswer Request.
       *
       * @param[in] deviceAddress: device address
       * @param[in] act: pmcore acknowledgment token
       * @param[out]
       * @param[in,out]
       *
       * @return BTSResult
       */
      virtual ::pmcore::BTSResult sendHoldAndAnswerRequest(const ::pmcore::BdAddress& deviceAddress,
            const ::pmcore::ActType act) = 0;

      /**
       * Error callback for HoldAndAnswer request
       *
       * @param[in] objPath: object path
       * @param[in] errorName: error name
       * @param[in] errorMessage: error message
       * @param[in] act: asynchronous completion token
       * @param[out]
       * @param[in,out]
       *
       * @return void
       */
      virtual void onHoldAndAnswerError(const ObjectPath& objPath, const ErrorName& errorName,
            const ErrorMessage& errorMessage, const act_t act) = 0;

      /**
       * Response callback for HoldAndAnswer request
       *
       * @param[in] objPath: object path
       * @param[in] act: asynchronous completion token
       * @param[out]
       * @param[in,out]
       *
       * @return void
       */
      virtual void onHoldAndAnswerResponse(const ObjectPath& objPath, const act_t act) = 0;

      /**
       * Send HangupAll Request.
       *
       * @param[in] deviceAddress: device address
       * @param[in] act: pmcore acknowledgment token
       * @param[out]
       * @param[in,out]
       *
       * @return BTSResult
       */
      virtual ::pmcore::BTSResult sendHangupAllRequest(const ::pmcore::BdAddress& deviceAddress,
            const ::pmcore::ActType act) = 0;

      /**
       * Error callback for HangupAll request
       *
       * @param[in] objPath: object path
       * @param[in] errorName: error name
       * @param[in] errorMessage: error message
       * @param[in] act: asynchronous completion token
       * @param[out]
       * @param[in,out]
       *
       * @return void
       */
      virtual void onHangupAllError(const ObjectPath& objPath, const ErrorName& errorName,
            const ErrorMessage& errorMessage, const act_t act) = 0;

      /**
       * Response callback for HangupAll request
       *
       * @param[in] objPath: object path
       * @param[in] act: asynchronous completion token
       * @param[out]
       * @param[in,out]
       *
       * @return void
       */
      virtual void onHangupAllResponse(const ObjectPath& objPath, const act_t act) = 0;

      /**
       * Send PrivateChat Request.
       *
       * @param[in] deviceAddress: device address
       * @param[in] call: call name to which private chat request has to be sent
       * @param[in] act: pmcore acknowledgment token
       * @param[out]
       * @param[in,out]
       *
       * @return BTSResult
       */
      virtual ::pmcore::BTSResult sendPrivateChatRequest(const ::pmcore::BdAddress& deviceAddress,
            const ::pmcore::CallInstance callInstance, const ::pmcore::ActType act) = 0;

      /**
       * Error callback for PrivateChat request
       *
       * @param[in] objPath: object path
       * @param[in] errorName: error name
       * @param[in] errorMessage: error message
       * @param[in] act: asynchronous completion token
       * @param[out]
       * @param[in,out]
       *
       * @return void
       */
      virtual void onPrivateChatError(const ObjectPath& objPath, const ErrorName& errorName,
            const ErrorMessage& errorMessage, const act_t act) = 0;

      /**
       * Response callback for PrivateChat request
       *
       * @param[in] objPath: object path
       * @param[in] calls: list of voice call object path
       * @param[in] act: asynchronous completion token
       * @param[out]
       * @param[in,out]
       *
       * @return void
       */
      virtual void onPrivateChatResponse(const ObjectPath& objPath, const ::std::vector< ::std::string >& calls,
            const act_t act) = 0;

      /**
       * Send CreateMultiparty Request.
       *
       * @param[in] deviceAddress: device address
       * @param[in] act: pmcore acknowledgment token
       * @param[out]
       * @param[in,out]
       *
       * @return BTSResult
       */
      virtual ::pmcore::BTSResult sendCreateMultipartyRequest(const ::pmcore::BdAddress& deviceAddress,
            const ::pmcore::ActType act) = 0;

      /**
       * Error callback for CreateMultiparty request
       *
       * @param[in] objPath: object path
       * @param[in] errorName: error name
       * @param[in] errorMessage: error message
       * @param[in] act: asynchronous completion token
       * @param[out]
       * @param[in,out]
       *
       * @return void
       */
      virtual void onCreateMultipartyError(const ObjectPath& objPath, const ErrorName& errorName,
            const ErrorMessage& errorMessage, const act_t act) = 0;

      /**
       * Response callback for CreateMultiparty request
       *
       * @param[in] objPath: object path
       * @param[in] calls: list of voice call object path
       * @param[in] act: asynchronous completion token
       * @param[out]
       * @param[in,out]
       *
       * @return void
       */
      virtual void onCreateMultipartyResponse(const ObjectPath& objPath, const ::std::vector< ::std::string >& calls,
            const act_t act) = 0;

      /**
       * Send HangupMultiparty Request.
       *
       * @param[in] deviceAddress: device address
       * @param[in] act: pmcore acknowledgment token
       * @param[out]
       * @param[in,out]
       *
       * @return BTSResult
       */
      virtual ::pmcore::BTSResult sendHangupMultipartyRequest(const ::pmcore::BdAddress& deviceAddress,
            const ::pmcore::ActType act) = 0;

      /**
       * Error callback for HangupMultiparty request
       *
       * @param[in] objPath: object path
       * @param[in] errorName: error name
       * @param[in] errorMessage: error message
       * @param[in] act: asynchronous completion token
       * @param[out]
       * @param[in,out]
       *
       * @return void
       */
      virtual void onHangupMultipartyError(const ObjectPath& objPath, const ErrorName& errorName,
            const ErrorMessage& errorMessage, const act_t act) = 0;

      /**
       * Response callback for HangupMultiparty request
       *
       * @param[in] objPath: object path
       * @param[in] act: asynchronous completion token
       * @param[out]
       * @param[in,out]
       *
       * @return void
       */
      virtual void onHangupMultipartyResponse(const ObjectPath& objPath, const act_t act) = 0;

      /**
       * Send SendTones Request.
       *
       * @param[in] deviceAddress: device address
       * @param[in] sendTones: sendTone name
       * @param[in] act: pmcore acknowledgment token
       * @param[out]
       * @param[in,out]
       *
       * @return BTSResult
       */
      virtual ::pmcore::BTSResult sendSendTonesRequest(const ::pmcore::BdAddress& deviceAddress,
            const ::pmcore::DTMFTones& sendTones, const ::pmcore::ActType act) = 0;

      /**
       * Error callback for SendTones request
       *
       * @param[in] objPath: object path
       * @param[in] errorName: error name
       * @param[in] errorMessage: error message
       * @param[in] act: asynchronous completion token
       * @param[out]
       * @param[in,out]
       *
       * @return void
       */
      virtual void onSendTonesError(const ObjectPath& objPath, const ErrorName& errorName,
            const ErrorMessage& errorMessage, const act_t act) = 0;

      /**
       * Response callback for SendTones request
       *
       * @param[in] objPath: object path
       * @param[in] act: asynchronous completion token
       * @param[out]
       * @param[in,out]
       *
       * @return void
       */
      virtual void onSendTonesResponse(const ObjectPath& objPath, const act_t act) = 0;

      /**
       * Send Hangup Request.
       *
       * @param[in] deviceAddress: device address
       * @param[in] callInstance: call instance
       * @param[in] act: pmcore acknowledgment token
       * @param[out]
       * @param[in,out]
       *
       * @return BTSResult
       */
      virtual ::pmcore::BTSResult sendHangupRequest(const ::pmcore::BdAddress& deviceAddress,
            const ::pmcore::CallInstance callInstance, const ::pmcore::ActType act) = 0;

      /**
       * Error callback for Hangup request
       *
       * @param[in] objPath: object path
       * @param[in] errorName: error name
       * @param[in] errorMessage: error message
       * @param[in] act: asynchronous completion token
       * @param[out]
       * @param[in,out]
       *
       * @return void
       */
      virtual void onHangupError(const ObjectPath& objPath, const ErrorName& errorName,
            const ErrorMessage& errorMessage, const act_t act) = 0;

      /**
       * Response callback for Hangup request
       *
       * @param[in] objPath: object path
       * @param[in] act: asynchronous completion token
       * @param[out]
       * @param[in,out]
       *
       * @return void
       */
      virtual void onHangupResponse(const ObjectPath& objPath, const act_t act) = 0;

      /**
       * Send Answer Request.
       *
       * @param[in] deviceAddress: device address
       * @param[in] callInstance: call instance
       * @param[in] act: pmcore acknowledgment token
       * @param[out]
       * @param[in,out]
       *
       * @return BTSResult
       */
      virtual ::pmcore::BTSResult sendAnswerRequest(const ::pmcore::BdAddress& deviceAddress,
            const ::pmcore::CallInstance callInstance, const ::pmcore::ActType act) = 0;

      /**
       * Error callback for Answer request
       *
       * @param[in] objPath: object path
       * @param[in] errorName: error name
       * @param[in] errorMessage: error message
       * @param[in] act: asynchronous completion token
       * @param[out]
       * @param[in,out]
       *
       * @return void
       */
      virtual void onAnswerError(const ObjectPath& objPath, const ErrorName& errorName,
            const ErrorMessage& errorMessage, const act_t act) = 0;

      /**
       * Response callback for Answer request
       *
       * @param[in] objPath: object path
       * @param[in] act: asynchronous completion token
       * @param[out]
       * @param[in,out]
       *
       * @return void
       */
      virtual void onAnswerResponse(const ObjectPath& objPath, const act_t act) = 0;

      /**
       * Send Hold Request.
       *
       * @param[in] deviceAddress: device address
       * @param[in] callInstance: call instance
       * @param[in] act: pmcore acknowledgment token
       * @param[out]
       * @param[in,out]
       *
       * @return BTSResult
       */
      virtual ::pmcore::BTSResult sendHoldRequest(const ::pmcore::BdAddress& deviceAddress,
            const ::pmcore::CallInstance callInstance, const ::pmcore::ActType act) = 0;

      /**
       * Error callback for Hold request
       *
       * @param[in] objPath: object path
       * @param[in] errorName: error name
       * @param[in] errorMessage: error message
       * @param[in] act: asynchronous completion token
       * @param[out]
       * @param[in,out]
       *
       * @return void
       */
      virtual void onHoldError(const ObjectPath& objPath, const ErrorName& errorName,
            const ErrorMessage& errorMessage, const act_t act) = 0;

      /**
       * Response callback for Hold request
       *
       * @param[in] objPath: object path
       * @param[in] act: asynchronous completion token
       * @param[out]
       * @param[in,out]
       *
       * @return void
       */
      virtual void onHoldResponse(const ObjectPath& objPath, const act_t act) = 0;

      /**
       * This method is used to notify the changes in SpeakerVolume
       *
       * @param[in] objPath: object path of CallVolume
       * @param[out] speakerVolume: speaker volume
       * @param[in,out]
       *
       * @return void
       */
      virtual void onSpeakerVolumeChanged(const ObjectPath& objPath, const uint8_t speakerVolume) = 0;

      /**
       * This method is used to notify the changes in MicrophoneVolume
       *
       * @param[in] objPath: object path of CallVolume
       * @param[out] microphoneVolume: microphone volume
       * @param[in,out]
       *
       * @return void
       */
      virtual void onMicrophoneVolumeChanged(const ObjectPath& objPath, const uint8_t microphoneVolume) = 0;

      /**
       * This method is used to notify the changes in SupportedFeatures
       *
       * @param[in] objPath: object path of Handsfree
       * @param[out] features: supported features
       * @param[in,out]
       *
       * @return void
       */
      virtual void onSupportedFeaturesChanged(const ObjectPath& objPath, const std::vector<std::string>& features) = 0;

      /**
       * This method is used to notify the changes in InbandRinging
       *
       * @param[in] objPath: object path of Handsfree
       * @param[out] enabled: inband ringing status
       * @param[in,out]
       *
       * @return void
       */
      virtual void onInbandRingingChanged(const ObjectPath& objPath, const bool enabled) = 0;

      /**
       * This method is used to notify the changes in VoiceRecognition
       *
       * @param[in] objPath: object path of Handsfree
       * @param[out] enabled: voice recognition status
       * @param[in,out]
       *
       * @return void
       */
      virtual void onVoiceRecognitionChanged(const ObjectPath& objPath, const bool enabled) = 0;

      /**
       * This method is used to notify the changes in EchoCancelingNoiseReduction
       *
       * @param[in] objPath: object path of Handsfree
       * @param[out] enabled: ecnr enabled status
       * @param[in,out]
       *
       * @return void
       */
      virtual void onEchoCancelingNoiseReductionChanged(const ObjectPath& objPath, const bool enabled) = 0;

      /**
       * This method is used to notify the changes in BatteryChargeLevel
       *
       * @param[in] objPath: object path of Handsfree
       * @param[out] chargeLevel: battery charge level
       * @param[in,out]
       *
       * @return void
       */
      virtual void onBatteryChargeLevelChanged(const ObjectPath& objPath, const uint8_t chargeLevel) = 0;

      /**
       * This method is used to notify the changes in Audio
       *
       * @param[in] objPath: object path of Handsfree
       * @param[out] audio: audio direction
       * @param[in,out]
       *
       * @return void
       */
      virtual void onAudioChanged(const ObjectPath& objPath, const std::string& audio) = 0;

      /**
       * This method is used to notify the changes in SCOType
       *
       * @param[in] objPath: object path of Handsfree
       * @param[out] scoType: sco type
       * @param[in,out]
       *
       * @return void
       */
      virtual void onSCOTypeChanged(const ObjectPath& objPath, const uint8_t scoType) = 0;

      /**
       * This method is used to notify the changes in Audio Path
       *
       * @param[in] objPath: object path of Handsfree
       * @param[out] audioPath: audio path
       * @param[in,out]
       *
       * @return void
       */
      virtual void onAudioPathChanged(const ObjectPath& objPath, const uint8_t audioPath) = 0;

      /**
       * This method is used to notify the changes in AllCallStatus
       *
       * @param[in] objPath: object path of Handsfree
       * @param[out] allCallStatus: all call status
       * @param[in,out]
       *
       * @return void
       */
      virtual void onAllCallStatusChanged(const ObjectPath& objPath, const uint8_t allCallStatus) = 0;

      /**
       * This method is used to notify the changes in ChldFeatureList
       *
       * @param[in] objPath: object path of Handsfree
       * @param[out] chldFeatureList: chld features list
       * @param[in,out]
       *
       * @return void
       */
      virtual void onChldFeatureChanged(const ObjectPath& objPath,
            const ::std::vector<std::string>& chldFeatureList) = 0;

      /**
       * This method is used to notify the changes in Audio Indicator
       *
       * @param[in] objPath: object path of Handsfree
       * @param[out] audioIndicator: audio indicator status
       * @param[in,out]
       *
       * @return void
       */
      virtual void onAudioIndicatorChanged(const ObjectPath& objPath, const uint8_t audioIndicator) = 0;

      //TODO: Below properties will be implemented if required in future
      //virtual void onModemPoweredChanged(const ObjectPath& objPath) = 0;

      //virtual void onModemNameChanged(const ObjectPath& objPath) = 0;

      //virtual void onModemInterfacesChanged(const ObjectPath& objPath) = 0;

      //virtual void onModemTypeChanged(const ObjectPath& objPath) = 0;

      //virtual void onModemManufacturerChanged(const ObjectPath& objPath) = 0;

      //virtual void onModemModelChanged(const ObjectPath& objPath) = 0;

      //virtual void onModemRevisionChanged(const ObjectPath& objPath) = 0;

      //virtual void onModemSerialChanged(const ObjectPath& objPath) = 0;

      /**
       * This method is used to notify the changes in Network Status
       *
       * @param[in] objPath: object path of network registration
       * @param[out] status: network status
       * @param[in,out]
       *
       * @return void
       */
      virtual void onNetworkStatusChanged(const ObjectPath& objPath, const std::string& status) = 0;

      /**
       * This method is used to notify the changes in Network MobileCountryCode
       *
       * @param[in] objPath: object path of network registration
       * @param[out] mobileCountryCode: mobile country code
       * @param[in,out]
       *
       * @return void
       */
      virtual void onNetworkMobileCountryCodeChanged(const ObjectPath& objPath,
            const std::string& mobileCountryCode) = 0;

      /**
       * This method is used to notify the changes in Network Operator Name
       *
       * @param[in] objPath: object path of network registration
       * @param[out] operatorName: network operator name
       * @param[in,out]
       *
       * @return void
       */
      virtual void onNetworkOperatorNameChanged(const ObjectPath& objPath, const std::string& operatorName) = 0;

      /**
       * This method is used to notify the changes in Network SignalStrength
       *
       * @param[in] objPath: object path of network registration
       * @param[out] strength: network signal strength
       * @param[in,out]
       *
       * @return void
       */
      virtual void onNetworkSignalStrengthChanged(const ObjectPath& objPath, const uint8_t strength) = 0;

      /**
       * This method is used to notify the changes in Network Current Operator
       *
       * @param[in] objPath: object path of network registration
       * @param[out] currentOperator: current operator opject path
       * @param[in,out]
       *
       * @return void
       */
      virtual void onNetworkCurrentOperatorChanged(const ObjectPath& objPath, const std::string& currentOperator) = 0;

      /**
       * This method is used to notify the changes in Siri Status
       *
       * @param[in] objPath: object path of siri
       * @param[out] status: Siri status
       * @param[in,out]
       *
       * @return void
       */
      virtual void onSiriStatusChanged(const ObjectPath& objPath, const uint8_t status) = 0;

      /**
       * This method is used to notify the changes in Siri NR
       *
       * @param[in] objPath: object path of siri
       * @param[out] noiseReduction: noise reduction status
       * @param[in,out]
       *
       * @return void
       */
      virtual void onSiriNRChanged(const ObjectPath& objPath, const uint8_t noiseReduction) = 0;

      /**
       * This method is used to notify the changes in Siri Eyes Free Mode
       *
       * @param[in] objPath: object path of siri
       * @param[out] enabled: EyesFreemode status
       * @param[in,out]
       *
       * @return void
       */
      virtual void onSiriEyesFreeModeChanged(const ObjectPath& objPath, const bool enabled) = 0;

      /**
       * This method is used to notify the changes in Siri Supported Type
       *
       * @param[in] objPath: object path of siri
       * @param[out] supportedType: supported type
       * @param[in,out]
       *
       * @return void
       */
      virtual void onSiriSupportedTypeChanged(const ObjectPath& objPath, const uint16_t supportedType) = 0;

      /**
       * This method is used to notify the changes in Emergency Numbers
       *
       * @param[in] objPath: object path of voice call manager
       * @param[out] emergencyNumbers: emergency numbers
       * @param[in,out]
       *
       * @return void
       */
      virtual void onEmergencyNumbersChanged(const ObjectPath& objPath,
            const std::vector<std::string>& emergencyNumbers) = 0;

      /**
       * This method is used to notify the changes in Voice Call Line Identification
       *
       * @param[in] objPath: object path of voice call
       * @param[out] lineIdentification: voice call line identification
       * @param[in,out]
       *
       * @return void
       */
      virtual void onVoiceCallLineIdentificationChanged(const ObjectPath& objPath,
            const std::string& lineIdentification) = 0;

      /**
       * This method is used to notify the changes in Voice Call Name
       *
       * @param[in] objPath: object path of voice call
       * @param[out] name: voice call name
       * @param[in,out]
       *
       * @return void
       */
      virtual void onVoiceCallNameChanged(const ObjectPath& objPath, const std::string& name) = 0;

      /**
       * This method is used to notify the changes in Voice Call Multiparty
       *
       * @param[in] objPath: object path of voice call
       * @param[out] multiparty: voice call multiparty
       * @param[in,out]
       *
       * @return void
       */
      virtual void onVoiceCallMultipartyChanged(const ObjectPath& objPath, const bool multiparty) = 0;

      /**
       * This method is used to notify the changes in Voice Call State
       *
       * @param[in] objPath: object path of voice call
       * @param[out] state: voice call state
       * @param[in,out]
       *
       * @return void
       */
      virtual void onVoiceCallStateChanged(const ObjectPath& objPath, const std::string& state) = 0;

      /**
       * This method is used to notify the changes in Voice Call Emergency
       *
       * @param[in] objPath: object path of voice call
       * @param[out] emergency: voice call emergency
       * @param[in,out]
       *
       * @return void
       */
      virtual void onVoiceCallEmergencyChanged(const ObjectPath& objPath, const bool emergency) = 0;

      /**
       * This method is used to notify the changes in Voice Call Direction
       *
       * @param[in] objPath: object path of voice call
       * @param[out] direction: voice call direction
       * @param[in,out]
       *
       * @return void
       */
      virtual void onVoiceCallDirectionChanged(const ObjectPath& objPath, const std::string& direction) = 0;

      /**
       * This method is used to notify the changes in Voice Call Mode
       *
       * @param[in] objPath: object path of voice call
       * @param[out] mode: voice call mode
       * @param[in,out]
       *
       * @return void
       */
      virtual void onVoiceCallModeChanged(const ObjectPath& objPath, const std::string& mode) = 0;

      /**
       * This method is used to notify the changes in Voice Call Type
       *
       * @param[in] objPath: object path of voice call
       * @param[out] type: voice call type
       * @param[in,out]
       *
       * @return void
       */
      virtual void onVoiceCallTypeChanged(const ObjectPath& objPath, const std::string& type) = 0;
   };

} //evobtstackwrapper

#endif //IBtStackWrapper_h
