/**
 * @file EvoBtStackWrapper.h
 *
 * @swcomponent PhoneCallManager
 *
 * @brief This file contains the definition of the EvoBtStackWrapper class
 *
 * @copyright (C) 2016 Robert Bosch GmbH.
 *            The reproduction, distribution and utilization of this file as
 *            well as the communication of its contents to others without express
 *            authorization is prohibited. Offenders will be held liable for the
 *            payment of damages. All rights reserved in the event of the grant
 *            of a patent, utility model or design.
 *
 * @details This file provides the request interfaces of EvoBtStackWrapper.
 *
 * @ingroup BtStackWrapper
 */

#ifndef EvoBtStackWrapper_h
#define EvoBtStackWrapper_h

#include "IBtStackWrapper.h"
#include "PmSingleton.h"

namespace evobtstackwrapper
{
   struct SetPropertyAttributes{
      std::string _property;
      std::string _value;
   };

   class IEvolutionGeniviDbusHfpCallVolumeCallBackIf;
   class IEvolutionGeniviDbusHfpHandsfreeCallBackIf;
   class IEvolutionGeniviDbusHfpManagerCallBackIf;
   class IEvolutionGeniviDbusHfpModemCallBackIf;
   class IEvolutionGeniviDbusHfpNetworkOperatorCallBackIf;
   class IEvolutionGeniviDbusHfpNetworkRegistrationCallBackIf;
   class IEvolutionGeniviDbusHfpSiriCallBackIf;
   class IEvolutionGeniviDbusHfpVoiceCallManagerCallBackIf;
   class IEvolutionGeniviDbusHfpVoiceCallCallBackIf;

   class IEvolutionGeniviDbusHfpCallVolumeRequestIf;
   class IEvolutionGeniviDbusHfpHandsfreeRequestIf;
   class IEvolutionGeniviDbusHfpManagerRequestIf;
   class IEvolutionGeniviDbusHfpModemRequestIf;
   class IEvolutionGeniviDbusHfpNetworkOperatorRequestIf;
   class IEvolutionGeniviDbusHfpNetworkRegistrationRequestIf;
   class IEvolutionGeniviDbusHfpSiriRequestIf;
   class IEvolutionGeniviDbusHfpVoiceCallManagerRequestIf;
   class IEvolutionGeniviDbusHfpVoiceCallRequestIf;

   /**
    * EvoBtStackWrapper class definition
    */
   class EvoBtStackWrapper final : public IBtStackWrapper, public PmSingleton<EvoBtStackWrapper>
   {
   public:
      /**
       * Constructor of EvoBtStackWrapper class
       *
       * @param[in]
       * @param[out]
       * @param[in,out]
       *
       * @return
       */
      EvoBtStackWrapper();

      /**
       * Destructor of EvoBtStackWrapper class
       *
       * @param[in]
       * @param[out]
       * @param[in,out]
       *
       * @return
       */
      ~EvoBtStackWrapper();

      /**
      * Copy constructor
      *
      * @param[in] EvoBtStackWrapper& ref - Reference to EvoBtStackWrapper
      * @param[out]
      * @param[in,out]
      *
      * @return
      */
      EvoBtStackWrapper(const EvoBtStackWrapper& ref);

     /**
      * operator= overloaded function
      *
      * @param[in]
      * @param[out]
      * @param[in,out]
      *
      * @return Returns the object of the EvoBtStackWrapper
      */
      EvoBtStackWrapper& operator=(const EvoBtStackWrapper& ref);

      /**
       * Member function overridden from IBtStackWrapper class
       *
       * @param[in]
       * @param[out]
       * @param[in,out]
       *
       * @return void
       */
      void setVehicleConfiguration(IN const ::pmcore::VehicleConfiguration& vehicleConfig) override;

      /**
       * Member function overridden from IBtStackWrapper class
       *
       * @param[in]
       * @param[out]
       * @param[in,out]
       *
       * @return void
       */
      void onDeviceConnected(IN const ::pmcore::BdAddress& deviceAddress) override;

      /**
       * Member function overridden from IBtStackWrapper class
       *
       * @param[in]
       * @param[out]
       * @param[in,out]
       *
       * @return void
       */
      void onDeviceDisconnected(IN const ::pmcore::BdAddress& deviceAddress) override;

      /**
       * Member function overridden from IBtStackWrapper class
       *
       * @param[in]
       * @param[out]
       * @param[in,out]
       *
       * @return void
       */
      void deviceCreated(IN const ObjectPath& objPath) override;

      /**
       * Member function overridden from IBtStackWrapper class
       *
       * @param[in]
       * @param[out]
       * @param[in,out]
       *
       * @return void
       */
      void deviceRemoved(IN const ObjectPath& objPath) override;

      /**
       * Member function overridden from IBtStackWrapper class
       *
       * @param[in]
       * @param[out]
       * @param[in,out]
       *
       * @return void
       */
      void modemCreated(IN const ::pmcore::BdAddress& deviceAddress, IN const ObjectPath& objPath) override;

      /**
       * Member function overridden from IBtStackWrapper class
       *
       * @param[in]
       * @param[out]
       * @param[in,out]
       *
       * @return void
       */
      void modemRemoved(IN const ObjectPath& objPath) override;

      /**
       * Member function overridden from IBtStackWrapper class
       *
       * @param[in]
       * @param[out]
       * @param[in,out]
       *
       * @return void
       */
      void networkOperatorCreated(IN const ::pmcore::BdAddress& deviceAddress, IN const ObjectPath& objPath) override;

      /**
       * Member function overridden from IBtStackWrapper class
       *
       * @param[in]
       * @param[out]
       * @param[in,out]
       *
       * @return void
       */
      void networkOperatorRemoved(IN const ObjectPath& objPath) override;

      /**
       * Member function overridden from IBtStackWrapper class
       *
       * @param[in]
       * @param[out]
       * @param[in,out]
       *
       * @return void
       */
      void voiceCallCreated(IN const ObjectPath& voiceCallMgrObjPath, IN const ObjectPath& voiceCallObjPath,
            BTSDbusPropertyList& propertyList) override;

      /**
       * Member function overridden from IBtStackWrapper class
       *
       * @param[in]
       * @param[out]
       * @param[in,out]
       *
       * @return void
       */
      void voiceCallRemoved(IN const ObjectPath& objPath) override;

      /**
       * Member function overridden from IBtStackWrapper class
       *
       * @param[in]
       * @param[out]
       * @param[in,out]
       *
       * @return void
       */
      void onProxyServiceAvailabilityStatus(const ::ccdbusif::evolution::Interface interface,
            const ObjectPath& objPath, const bool serviceAvailability) override;

      /**
       * Member function overridden from IBtStackWrapper class
       *
       * @param[in]
       * @param[out]
       * @param[in,out]
       *
       * @return BTSResult
       */
      ::pmcore::BTSResult sendSetPropertyRequest(IN const ::ccdbusif::evolution::Interface interface,
            IN const ::pmcore::BdAddress& deviceAddress, IN const ::std::string& property,
            IN const uint8 value) override;

      /**
       * Member function overridden from IBtStackWrapper class
       *
       * @param[in]
       * @param[out]
       * @param[in,out]
       *
       * @return BTSResult
       */
      ::pmcore::BTSResult sendSetPropertyRequest(IN const ::ccdbusif::evolution::Interface interface,
            IN const ::pmcore::BdAddress& deviceAddress, IN const ::std::string& property,
            IN const bool value) override;

      /**
       * Member function overridden from IBtStackWrapper class
       *
       * @param[in]
       * @param[out]
       * @param[in,out]
       *
       * @return BTSResult
       */
      ::pmcore::BTSResult sendSetPropertyRequest(IN const ::ccdbusif::evolution::Interface interface,
            IN const ::pmcore::BdAddress& deviceAddress, IN const ::std::string& property,
            IN const ::std::string& value) override;

      /**
       * This method is called when the response for SetProperty is received for Handsfree proxy
       *
       * @param[in] SetPropertyAttributes: It contains the property and the value that was
       *                                   set during the SetProperty method call
       * @param[out]
       * @param[in,out]
       *
       * @return void
       */
      void onSetPropertyResponse(const ObjectPath& objPath, const ::ccdbusif::evolution::Interface interface,
            const ErrorName& errorName, const ErrorMessage& errorMessage, const act_t dbusAct) override;

      /**
       * Member function overridden from IBtStackWrapper class
       *
       * @param[in]
       * @param[out]
       * @param[in,out]
       *
       * @return BTSResult
       */
      ::pmcore::BTSResult sendRequestPhoneNumberRequest(IN const ::pmcore::BdAddress& deviceAddress,
            IN const ::pmcore::ActType act) override;

      /**
       * Member function overridden from IBtStackWrapper class
       *
       * @param[in]
       * @param[out]
       * @param[in,out]
       *
       * @return void
       */
      void onRequestPhoneNumberError(IN const ObjectPath& objPath, IN const ErrorName& errorName,
            IN const ErrorMessage& errorMessage, IN const act_t act) override;

      /**
       * Member function overridden from IBtStackWrapper class
       *
       * @param[in]
       * @param[out]
       * @param[in,out]
       *
       * @return void
       */
      void onRequestPhoneNumberResponse(IN const ObjectPath& objPath, IN const std::string& telephoneNumber,
            IN const act_t act) override;

      /**
       * Member function overridden from IBtStackWrapper class
       *
       * @param[in]
       * @param[out]
       * @param[in,out]
       *
       * @return BTSResult
       */
      ::pmcore::BTSResult sendGetSubscriberRequest(IN const ::pmcore::BdAddress& deviceAddress,
            IN const ::pmcore::ActType act) override;

      /**
       * Member function overridden from IBtStackWrapper class
       *
       * @param[in]
       * @param[out]
       * @param[in,out]
       *
       * @return void
       */
      void onGetSubscriberError(IN const ObjectPath& objPath, IN const ErrorName& errorName,
            IN const ErrorMessage& errorMessage, IN const act_t act) override;

      /**
       * Member function overridden from IBtStackWrapper class
       *
       * @param[in]
       * @param[out]
       * @param[in,out]
       *
       * @return void
       */
      void onGetSubscriberResponse(IN const ObjectPath& objPath, IN const uint8_t type,
            IN const std::string& telephoneNumber, IN const act_t act) override;

      /**
       * Member function overridden from IBtStackWrapper class
       *
       * @param[in]
       * @param[out]
       * @param[in,out]
       *
       * @return BTSResult
       */
      ::pmcore::BTSResult sendSendCustomATCommandRequest(IN const ::pmcore::BdAddress& deviceAddress,
            IN const ::std::vector< uint8 >& data, IN const ::pmcore::ActType act) override;

      /**
       * Member function overridden from IBtStackWrapper class
       *
       * @param[in]
       * @param[out]
       * @param[in,out]
       *
       * @return void
       */
      void onSendCustomATCommandError(IN const ObjectPath& objPath, IN const ErrorName& errorName,
            IN const ErrorMessage& errorMessage, IN const act_t act) override;

      /**
       * Member function overridden from IBtStackWrapper class
       *
       * @param[in]
       * @param[out]
       * @param[in,out]
       *
       * @return void
       */
      void onSendCustomATCommandResponse(IN const ObjectPath& objPath, IN const act_t act) override;

      /**
       * Member function overridden from IBtStackWrapper class
       *
       * @param[in]
       * @param[out]
       * @param[in,out]
       *
       * @return BTSResult
       */
      ::pmcore::BTSResult sendHFUpdateIndicatorsValueRequest(IN const ::pmcore::BdAddress& deviceAddress,
            IN uint16 number, IN uint32 value, IN const ::pmcore::ActType act) override;

      /**
       * Member function overridden from IBtStackWrapper class
       *
       * @param[in]
       * @param[out]
       * @param[in,out]
       *
       * @return void
       */
      void onHFUpdateIndicatorsValueError(IN const ObjectPath& objPath, IN const ErrorName& errorName,
            IN const ErrorMessage& errorMessage, IN const act_t act) override;

      /**
       * Member function overridden from IBtStackWrapper class
       *
       * @param[in]
       * @param[out]
       * @param[in,out]
       *
       * @return void
       */
      void onHFUpdateIndicatorsValueResponse(IN const ObjectPath& objPath, IN const act_t act) override;

      /**
       * Member function overridden from IBtStackWrapper class
       *
       * @param[in]
       * @param[out]
       * @param[in,out]
       *
       * @return BTSResult
       */
      ::pmcore::BTSResult sendGetModemsRequest(IN const ::pmcore::ActType act) override;

      /**
       * Member function overridden from IBtStackWrapper class
       *
       * @param[in]
       * @param[out]
       * @param[in,out]
       *
       * @return void
       */
      void onGetModemsError(IN const ObjectPath& objPath, IN const ErrorName& errorName,
            IN const ErrorMessage& errorMessage, IN const act_t act) override;

      /**
       * Member function overridden from IBtStackWrapper class
       *
       * @param[in]
       * @param[out]
       * @param[in,out]
       *
       * @return void
       */
      void onGetModemsResponse(IN const ObjectPath& objPath,
            /*const ::std::vector< GetModemsResponseModemsStruct >& modems,*/ IN const act_t act) override;

      /**
       * Member function overridden from IBtStackWrapper class
       *
       * @param[in]
       * @param[out]
       * @param[in,out]
       *
       * @return BTSResult
       */
      ::pmcore::BTSResult sendAcceptSCOConnectRequest(IN const ::pmcore::BdAddress& deviceAddress,
            IN const ::pmcore::ActType act) override;

      /**
       * Member function overridden from IBtStackWrapper class
       *
       * @param[in]
       * @param[out]
       * @param[in,out]
       *
       * @return void
       */
      void onAcceptSCOConnectError(IN const ObjectPath& objPath, IN const ErrorName& errorName,
            IN const ErrorMessage& errorMessage, IN const act_t act) override;

      /**
       * Member function overridden from IBtStackWrapper class
       *
       * @param[in]
       * @param[out]
       * @param[in,out]
       *
       * @return void
       */
      void onAcceptSCOConnectResponse(IN const ObjectPath& objPath, IN const act_t act) override;

      /**
       * Member function overridden from IBtStackWrapper class
       *
       * @param[in]
       * @param[out]
       * @param[in,out]
       *
       * @return void
       */
      void onSCOConnectRequestSignal(IN const ObjectPath& objPath, IN const uint8_t type) override;

      /**
       * Member function overridden from IBtStackWrapper class
       *
       * @param[in]
       * @param[out]
       * @param[in,out]
       *
       * @return BTSResult
       */
      ::pmcore::BTSResult sendSiriEnableRequest(IN const ::pmcore::BdAddress& deviceAddress,
            IN uint16 support_type, IN const ::pmcore::ActType act) override;

      /**
       * Member function overridden from IBtStackWrapper class
       *
       * @param[in]
       * @param[out]
       * @param[in,out]
       *
       * @return void
       */
      void onSiriEnableError(IN const ObjectPath& objPath, IN const ErrorName& errorName,
            IN const ErrorMessage& errorMessage, IN const act_t act) override;

      /**
       * Member function overridden from IBtStackWrapper class
       *
       * @param[in]
       * @param[out]
       * @param[in,out]
       *
       * @return void
       */
      void onSiriEnableResponse(IN const ObjectPath& objPath, IN const act_t act) override;

      /**
       * Member function overridden from IBtStackWrapper class
       *
       * @param[in]
       * @param[out]
       * @param[in,out]
       *
       * @return BTSResult
       */
      ::pmcore::BTSResult sendSiriSetNRRequest(IN const ::pmcore::BdAddress& deviceAddress, IN uint8 status,
            IN const ::pmcore::ActType act) override;

      /**
       * Member function overridden from IBtStackWrapper class
       *
       * @param[in]
       * @param[out]
       * @param[in,out]
       *
       * @return void
       */
      void onSiriSetNRError(IN const ObjectPath& objPath, IN const ErrorName& errorName,
            IN const ErrorMessage& errorMessage, IN const act_t act) override;

      /**
       * Member function overridden from IBtStackWrapper class
       *
       * @param[in]
       * @param[out]
       * @param[in,out]
       *
       * @return void
       */
      void onSiriSetNRResponse(IN const ObjectPath& objPath, IN const act_t act) override;

      /**
       * Member function overridden from IBtStackWrapper class
       *
       * @param[in]
       * @param[out]
       * @param[in,out]
       *
       * @return BTSResult
       */
      ::pmcore::BTSResult sendGetCallsRequest(IN const ::pmcore::BdAddress& deviceAddress,
            IN const ::pmcore::ActType act) override;

      /**
       * Member function overridden from IBtStackWrapper class
       *
       * @param[in]
       * @param[out]
       * @param[in,out]
       *
       * @return void
       */
      void onGetCallsError(IN const ObjectPath& objPath, IN const ErrorName& errorName,
            IN const ErrorMessage& errorMessage, IN const act_t act) override;

      /**
       * Member function overridden from IBtStackWrapper class
       *
       * @param[in]
       * @param[out]
       * @param[in,out]
       *
       * @return void
       */
      void onGetCallsResponse(IN const ObjectPath& objPath,
            IN const ::std::map< ObjectPath, BTSDbusPropertyList >& calls, IN const act_t act) override;

      /**
       * Member function overridden from IBtStackWrapper class
       *
       * @param[in]
       * @param[out]
       * @param[in,out]
       *
       * @return BTSResult
       */
      ::pmcore::BTSResult sendDialRequest(IN const ::pmcore::BdAddress& deviceAddress,
            IN const ::pmcore::TelephoneNumber& number, IN const ::pmcore::HideCallerId hide_callerid,
            IN const ::pmcore::ActType act) override;

      /**
       * Member function overridden from IBtStackWrapper class
       *
       * @param[in]
       * @param[out]
       * @param[in,out]
       *
       * @return void
       */
      void onDialError(IN const ObjectPath& objPath, IN const ErrorName& errorName,
            IN const ErrorMessage& errorMessage, IN const act_t act) override;

      /**
       * Member function overridden from IBtStackWrapper class
       *
       * @param[in]
       * @param[out]
       * @param[in,out]
       *
       * @return void
       */
      void onDialResponse(IN const ObjectPath& objPath, IN const std::string& path, IN const act_t act) override;

      /**
       * Member function overridden from IBtStackWrapper class
       *
       * @param[in]
       * @param[out]
       * @param[in,out]
       *
       * @return BTSResult
       */
      ::pmcore::BTSResult sendMemDialRequest(IN const ::pmcore::BdAddress& deviceAddress,
            IN const ::pmcore::MemDialIndex numberIndex, IN const ::pmcore::HideCallerId hideCallerId,
            IN const ::pmcore::ActType act) override;

      /**
       * Member function overridden from IBtStackWrapper class
       *
       * @param[in]
       * @param[out]
       * @param[in,out]
       *
       * @return void
       */
      void onMemDialError(IN const ObjectPath& objPath, IN const ErrorName& errorName,
            IN const ErrorMessage& errorMessage, IN const act_t act) override;

      /**
       * Member function overridden from IBtStackWrapper class
       *
       * @param[in]
       * @param[out]
       * @param[in,out]
       *
       * @return void
       */
      void onMemDialResponse(IN const ObjectPath& objPath, IN const std::string& path, IN const act_t act) override;

      /**
       * Member function overridden from IBtStackWrapper class
       *
       * @param[in]
       * @param[out]
       * @param[in,out]
       *
       * @return BTSResult
       */
      ::pmcore::BTSResult sendRedialRequest(IN const ::pmcore::BdAddress& deviceAddress,
            IN const ::pmcore::HideCallerId hide_callerid, IN const ::pmcore::ActType act) override;

      /**
       * Member function overridden from IBtStackWrapper class
       *
       * @param[in]
       * @param[out]
       * @param[in,out]
       *
       * @return void
       */
      void onRedialError(IN const ObjectPath& objPath, IN const ErrorName& errorName,
            IN const ErrorMessage& errorMessage, IN const act_t act) override;

      /**
       * Member function overridden from IBtStackWrapper class
       *
       * @param[in]
       * @param[out]
       * @param[in,out]
       *
       * @return void
       */
      void onRedialResponse(IN const ObjectPath& objPath, IN const std::string& path, IN const act_t act) override;

      /**
       * Member function overridden from IBtStackWrapper class
       *
       * @param[in]
       * @param[out]
       * @param[in,out]
       *
       * @return BTSResult
       */
      ::pmcore::BTSResult sendSwapCallsRequest(IN const ::pmcore::BdAddress& deviceAddress,
            IN const ::pmcore::ActType act) override;

      /**
       * Member function overridden from IBtStackWrapper class
       *
       * @param[in]
       * @param[out]
       * @param[in,out]
       *
       * @return void
       */
      void onSwapCallsError(IN const ObjectPath& objPath, IN const ErrorName& errorName,
            IN const ErrorMessage& errorMessage, IN const act_t act) override;

      /**
       * Member function overridden from IBtStackWrapper class
       *
       * @param[in]
       * @param[out]
       * @param[in,out]
       *
       * @return void
       */
      void onSwapCallsResponse(IN const ObjectPath& objPath, IN const act_t act) override;

      /**
       * Member function overridden from IBtStackWrapper class
       *
       * @param[in]
       * @param[out]
       * @param[in,out]
       *
       * @return BTSResult
       */
      ::pmcore::BTSResult sendReleaseAndAnswerRequest(IN const ::pmcore::BdAddress& deviceAddress,
            IN const ::pmcore::ActType act) override;

      /**
       * Member function overridden from IBtStackWrapper class
       *
       * @param[in]
       * @param[out]
       * @param[in,out]
       *
       * @return void
       */
      void onReleaseAndAnswerError(IN const ObjectPath& objPath, IN const ErrorName& errorName,
            IN const ErrorMessage& errorMessage, IN const act_t act) override;

      /**
       * Member function overridden from IBtStackWrapper class
       *
       * @param[in]
       * @param[out]
       * @param[in,out]
       *
       * @return void
       */
      void onReleaseAndAnswerResponse(IN const ObjectPath& objPath, IN const act_t act) override;

      /**
       * Member function overridden from IBtStackWrapper class
       *
       * @param[in]
       * @param[out]
       * @param[in,out]
       *
       * @return BTSResult
       */
      ::pmcore::BTSResult sendReleaseAndSwapRequest(IN const ::pmcore::BdAddress& deviceAddress,
            IN const ::pmcore::ActType act) override;

      /**
       * Member function overridden from IBtStackWrapper class
       *
       * @param[in]
       * @param[out]
       * @param[in,out]
       *
       * @return void
       */
      void onReleaseAndSwapError(IN const ObjectPath& objPath, IN const ErrorName& errorName,
            IN const ErrorMessage& errorMessage, IN const act_t act) override;

      /**
       * Member function overridden from IBtStackWrapper class
       *
       * @param[in]
       * @param[out]
       * @param[in,out]
       *
       * @return void
       */
      void onReleaseAndSwapResponse(IN const ObjectPath& objPath, IN const act_t act) override;

      /**
       * Member function overridden from IBtStackWrapper class
       *
       * @param[in]
       * @param[out]
       * @param[in,out]
       *
       * @return BTSResult
       */
      ::pmcore::BTSResult sendHoldAndAnswerRequest(IN const ::pmcore::BdAddress& deviceAddress,
            IN const ::pmcore::ActType act) override;

      /**
       * Member function overridden from IBtStackWrapper class
       *
       * @param[in]
       * @param[out]
       * @param[in,out]
       *
       * @return void
       */
      void onHoldAndAnswerError(IN const ObjectPath& objPath, IN const ErrorName& errorName,
            IN const ErrorMessage& errorMessage, IN const act_t act) override;

      /**
       * Member function overridden from IBtStackWrapper class
       *
       * @param[in]
       * @param[out]
       * @param[in,out]
       *
       * @return void
       */
      void onHoldAndAnswerResponse(IN const ObjectPath& objPath, IN const act_t act) override;

      /**
       * Member function overridden from IBtStackWrapper class
       *
       * @param[in]
       * @param[out]
       * @param[in,out]
       *
       * @return BTSResult
       */
      ::pmcore::BTSResult sendHangupAllRequest(IN const ::pmcore::BdAddress& deviceAddress,
            IN const ::pmcore::ActType act) override;

      /**
       * Member function overridden from IBtStackWrapper class
       *
       * @param[in]
       * @param[out]
       * @param[in,out]
       *
       * @return void
       */
      void onHangupAllError(IN const ObjectPath& objPath, IN const ErrorName& errorName,
            IN const ErrorMessage& errorMessage, IN const act_t act) override;

      /**
       * Member function overridden from IBtStackWrapper class
       *
       * @param[in]
       * @param[out]
       * @param[in,out]
       *
       * @return void
       */
      void onHangupAllResponse(IN const ObjectPath& objPath, IN const act_t act) override;

      /**
       * Member function overridden from IBtStackWrapper class
       *
       * @param[in]
       * @param[out]
       * @param[in,out]
       *
       * @return BTSResult
       */
      ::pmcore::BTSResult sendPrivateChatRequest(IN const ::pmcore::BdAddress& deviceAddress,
            IN const ::pmcore::CallInstance callInstance, IN const ::pmcore::ActType act) override;

      /**
       * Member function overridden from IBtStackWrapper class
       *
       * @param[in]
       * @param[out]
       * @param[in,out]
       *
       * @return void
       */
      void onPrivateChatError(IN const ObjectPath& objPath, IN const ErrorName& errorName,
            IN const ErrorMessage& errorMessage, IN const act_t act) override;

      /**
       * Member function overridden from IBtStackWrapper class
       *
       * @param[in]
       * @param[out]
       * @param[in,out]
       *
       * @return void
       */
      void onPrivateChatResponse(IN const ObjectPath& objPath, IN const ::std::vector< ::std::string >& calls,
            IN const act_t act) override;

      /**
       * Member function overridden from IBtStackWrapper class
       *
       * @param[in]
       * @param[out]
       * @param[in,out]
       *
       * @return BTSResult
       */
      ::pmcore::BTSResult sendCreateMultipartyRequest(IN const ::pmcore::BdAddress& deviceAddress,
            IN const ::pmcore::ActType act) override;

      /**
       * Member function overridden from IBtStackWrapper class
       *
       * @param[in]
       * @param[out]
       * @param[in,out]
       *
       * @return void
       */
      void onCreateMultipartyError(IN const ObjectPath& objPath, IN const ErrorName& errorName,
            IN const ErrorMessage& errorMessage, IN const act_t act) override;

      /**
       * Member function overridden from IBtStackWrapper class
       *
       * @param[in]
       * @param[out]
       * @param[in,out]
       *
       * @return void
       */
      void onCreateMultipartyResponse(IN const ObjectPath& objPath, IN const ::std::vector< ::std::string >& calls,
            IN const act_t act) override;

      /**
       * Member function overridden from IBtStackWrapper class
       *
       * @param[in]
       * @param[out]
       * @param[in,out]
       *
       * @return BTSResult
       */
      ::pmcore::BTSResult sendHangupMultipartyRequest(IN const ::pmcore::BdAddress& deviceAddress,
            IN const ::pmcore::ActType act) override;

      /**
       * Member function overridden from IBtStackWrapper class
       *
       * @param[in]
       * @param[out]
       * @param[in,out]
       *
       * @return void
       */
      void onHangupMultipartyError(IN const ObjectPath& objPath, IN const ErrorName& errorName,
            IN const ErrorMessage& errorMessage, IN const act_t act) override;

      /**
       * Member function overridden from IBtStackWrapper class
       *
       * @param[in]
       * @param[out]
       * @param[in,out]
       *
       * @return void
       */
      void onHangupMultipartyResponse(IN const ObjectPath& objPath, IN const act_t act) override;

      /**
       * Member function overridden from IBtStackWrapper class
       *
       * @param[in]
       * @param[out]
       * @param[in,out]
       *
       * @return BTSResult
       */
      ::pmcore::BTSResult sendSendTonesRequest(IN const ::pmcore::BdAddress& deviceAddress,
            IN const ::pmcore::DTMFTones& sendTones, IN const ::pmcore::ActType act) override;

      /**
       * Member function overridden from IBtStackWrapper class
       *
       * @param[in]
       * @param[out]
       * @param[in,out]
       *
       * @return void
       */
      void onSendTonesError(IN const ObjectPath& objPath, IN const ErrorName& errorName,
            IN const ErrorMessage& errorMessage, IN const act_t act) override;

      /**
       * Member function overridden from IBtStackWrapper class
       *
       * @param[in]
       * @param[out]
       * @param[in,out]
       *
       * @return void
       */
      void onSendTonesResponse(IN const ObjectPath& objPath, IN const act_t act) override;

      /**
       * Member function overridden from IBtStackWrapper class
       *
       * @param[in]
       * @param[out]
       * @param[in,out]
       *
       * @return BTSResult
       */
      ::pmcore::BTSResult sendHangupRequest(IN const ::pmcore::BdAddress& deviceAddress,
            IN const ::pmcore::CallInstance callInstance, IN const ::pmcore::ActType act) override;

      /**
       * Member function overridden from IBtStackWrapper class
       *
       * @param[in]
       * @param[out]
       * @param[in,out]
       *
       * @return void
       */
      void onHangupError(IN const ObjectPath& objPath, IN const ErrorName& errorName,
            IN const ErrorMessage& errorMessage, IN const act_t act) override;

      /**
       * Member function overridden from IBtStackWrapper class
       *
       * @param[in]
       * @param[out]
       * @param[in,out]
       *
       * @return void
       */
      void onHangupResponse(IN const ObjectPath& objPath, IN const act_t act) override;

      /**
       * Member function overridden from IBtStackWrapper class
       *
       * @param[in]
       * @param[out]
       * @param[in,out]
       *
       * @return BTSResult
       */
      ::pmcore::BTSResult sendAnswerRequest(IN const ::pmcore::BdAddress& deviceAddress,
            IN const ::pmcore::CallInstance callInstance, IN const ::pmcore::ActType act) override;

      /**
       * Member function overridden from IBtStackWrapper class
       *
       * @param[in]
       * @param[out]
       * @param[in,out]
       *
       * @return void
       */
      void onAnswerError(IN const ObjectPath& objPath, IN const ErrorName& errorName,
            IN const ErrorMessage& errorMessage, IN const act_t act) override;

      /**
       * Member function overridden from IBtStackWrapper class
       *
       * @param[in]
       * @param[out]
       * @param[in,out]
       *
       * @return void
       */
      void onAnswerResponse(IN const ObjectPath& objPath, IN const act_t act) override;

      /**
       * Member function overridden from IBtStackWrapper class
       *
       * @param[in]
       * @param[out]
       * @param[in,out]
       *
       * @return BTSResult
       */
      ::pmcore::BTSResult sendHoldRequest(IN const ::pmcore::BdAddress& deviceAddress,
            IN const ::pmcore::CallInstance callInstance, IN const ::pmcore::ActType act) override;

      /**
       * Member function overridden from IBtStackWrapper class
       *
       * @param[in]
       * @param[out]
       * @param[in,out]
       *
       * @return void
       */
      void onHoldError(IN const ObjectPath& objPath, IN const ErrorName& errorName,
            IN const ErrorMessage& errorMessage, IN const act_t act) override;

      /**
       * Member function overridden from IBtStackWrapper class
       *
       * @param[in]
       * @param[out]
       * @param[in,out]
       *
       * @return void
       */
      void onHoldResponse(IN const ObjectPath& objPath, IN const act_t act) override;

      /**
       * Member function overridden from IBtStackWrapper class
       *
       * @param[in]
       * @param[out]
       * @param[in,out]
       *
       * @return void
       */
      void onSpeakerVolumeChanged(IN const ObjectPath& objPath, IN const uint8_t speakerVolume) override;

      /**
       * Member function overridden from IBtStackWrapper class
       *
       * @param[in]
       * @param[out]
       * @param[in,out]
       *
       * @return void
       */
      void onMicrophoneVolumeChanged(IN const ObjectPath& objPath, IN const uint8_t microphoneVolume) override;

      /**
       * Member function overridden from IBtStackWrapper class
       *
       * @param[in]
       * @param[out]
       * @param[in,out]
       *
       * @return void
       */
      void onSupportedFeaturesChanged(IN const ObjectPath& objPath,
            IN const std::vector<std::string>& features) override;

      /**
       * Member function overridden from IBtStackWrapper class
       *
       * @param[in]
       * @param[out]
       * @param[in,out]
       *
       * @return void
       */
      void onInbandRingingChanged(IN const ObjectPath& objPath, IN const bool enabled) override;

      /**
       * Member function overridden from IBtStackWrapper class
       *
       * @param[in]
       * @param[out]
       * @param[in,out]
       *
       * @return void
       */
      void onVoiceRecognitionChanged(IN const ObjectPath& objPath, IN const bool enabled) override;

      /**
       * Member function overridden from IBtStackWrapper class
       *
       * @param[in]
       * @param[out]
       * @param[in,out]
       *
       * @return void
       */
      void onEchoCancelingNoiseReductionChanged(IN const ObjectPath& objPath, IN const bool enabled) override;

      /**
       * Member function overridden from IBtStackWrapper class
       *
       * @param[in]
       * @param[out]
       * @param[in,out]
       *
       * @return void
       */
      void onBatteryChargeLevelChanged(IN const ObjectPath& objPath, IN const uint8_t chargeLevel) override;

      /**
       * Member function overridden from IBtStackWrapper class
       *
       * @param[in]
       * @param[out]
       * @param[in,out]
       *
       * @return void
       */
      void onAudioChanged(IN const ObjectPath& objPath, IN const std::string& audio) override;

      /**
       * Member function overridden from IBtStackWrapper class
       *
       * @param[in]
       * @param[out]
       * @param[in,out]
       *
       * @return void
       */
      void onSCOTypeChanged(IN const ObjectPath& objPath, IN const uint8_t scoType) override;

      /**
       * Member function overridden from IBtStackWrapper class
       *
       * @param[in]
       * @param[out]
       * @param[in,out]
       *
       * @return void
       */
      void onAudioPathChanged(IN const ObjectPath& objPath, IN const uint8_t audioPath) override;

      /**
       * Member function overridden from IBtStackWrapper class
       *
       * @param[in]
       * @param[out]
       * @param[in,out]
       *
       * @return void
       */
      void onAllCallStatusChanged(IN const ObjectPath& objPath, IN const uint8_t allCallStatus) override;

      /**
       * Member function overridden from IBtStackWrapper class
       *
       * @param[in]
       * @param[out]
       * @param[in,out]
       *
       * @return void
       */
      void onChldFeatureChanged(IN const ObjectPath& objPath,
            IN const ::std::vector<std::string>& childFeatureList) override;

      /**
       * Member function overridden from IBtStackWrapper class
       *
       * @param[in]
       * @param[out]
       * @param[in,out]
       *
       * @return void
       */
      void onAudioIndicatorChanged(IN const ObjectPath& objPath, IN const uint8_t audioIndicator) override;

      //void onModemPoweredChanged(IN const ObjectPath& objPath) override;

      //void onModemNameChanged(IN const ObjectPath& objPath) override;

      //void onModemInterfacesChanged(IN const ObjectPath& objPath) override;

      //void onModemTypeChanged(IN const ObjectPath& objPath) override;

      //void onModemManufacturerChanged(IN const ObjectPath& objPath) override;

      //void onModemModelChanged(IN const ObjectPath& objPath) override;

      //void onModemRevisionChanged(IN const ObjectPath& objPath) override;

      //void onModemSerialChanged(IN const ObjectPath& objPath) override;

      /**
       * Member function overridden from IBtStackWrapper class
       *
       * @param[in]
       * @param[out]
       * @param[in,out]
       *
       * @return void
       */
      void onNetworkStatusChanged(IN const ObjectPath& objPath, IN const std::string& status) override;

      /**
       * Member function overridden from IBtStackWrapper class
       *
       * @param[in]
       * @param[out]
       * @param[in,out]
       *
       * @return void
       */
      void onNetworkMobileCountryCodeChanged(IN const ObjectPath& objPath,
            IN const std::string& mobileCountryCode) override;

      /**
       * Member function overridden from IBtStackWrapper class
       *
       * @param[in]
       * @param[out]
       * @param[in,out]
       *
       * @return void
       */
      void onNetworkOperatorNameChanged(IN const ObjectPath& objPath, IN const std::string& operatorName) override;

      /**
       * Member function overridden from IBtStackWrapper class
       *
       * @param[in]
       * @param[out]
       * @param[in,out]
       *
       * @return void
       */
      void onNetworkSignalStrengthChanged(IN const ObjectPath& objPath, IN const uint8_t strength) override;

      /**
       * Member function overridden from IBtStackWrapper class
       *
       * @param[in]
       * @param[out]
       * @param[in,out]
       *
       * @return void
       */
      void onNetworkCurrentOperatorChanged(IN const ObjectPath& objPath,
            IN const std::string& currentOperator) override;

      /**
       * Member function overridden from IBtStackWrapper class
       *
       * @param[in]
       * @param[out]
       * @param[in,out]
       *
       * @return void
       */
      void onSiriStatusChanged(IN const ObjectPath& objPath, IN const uint8_t status) override;

      /**
       * Member function overridden from IBtStackWrapper class
       *
       * @param[in]
       * @param[out]
       * @param[in,out]
       *
       * @return void
       */
      void onSiriNRChanged(IN const ObjectPath& objPath, IN const uint8_t noiseReduction) override;

      /**
       * Member function overridden from IBtStackWrapper class
       *
       * @param[in]
       * @param[out]
       * @param[in,out]
       *
       * @return void
       */
      void onSiriEyesFreeModeChanged(IN const ObjectPath& objPath, IN const bool enabled) override;

      /**
       * Member function overridden from IBtStackWrapper class
       *
       * @param[in]
       * @param[out]
       * @param[in,out]
       *
       * @return void
       */
      void onSiriSupportedTypeChanged(IN const ObjectPath& objPath, IN const uint16_t supportedType) override;

      /**
       * Member function overridden from IBtStackWrapper class
       *
       * @param[in]
       * @param[out]
       * @param[in,out]
       *
       * @return void
       */
      void onEmergencyNumbersChanged(IN const ObjectPath& objPath,
            IN const std::vector<std::string>& emergencyNumbers) override;

      /**
       * Member function overridden from IBtStackWrapper class
       *
       * @param[in]
       * @param[out]
       * @param[in,out]
       *
       * @return void
       */
      void onVoiceCallLineIdentificationChanged(IN const ObjectPath& objPath,
            IN const std::string& lineIdentification) override;

      /**
       * Member function overridden from IBtStackWrapper class
       *
       * @param[in]
       * @param[out]
       * @param[in,out]
       *
       * @return void
       */
      void onVoiceCallNameChanged(IN const ObjectPath& objPath, IN const std::string& name) override;

      /**
       * Member function overridden from IBtStackWrapper class
       *
       * @param[in]
       * @param[out]
       * @param[in,out]
       *
       * @return void
       */
      void onVoiceCallMultipartyChanged(IN const ObjectPath& objPath, IN const bool multiparty) override;

      /**
       * Member function overridden from IBtStackWrapper class
       *
       * @param[in]
       * @param[out]
       * @param[in,out]
       *
       * @return void
       */
      void onVoiceCallStateChanged(IN const ObjectPath& objPath, IN const std::string& state) override;

      /**
       * Member function overridden from IBtStackWrapper class
       *
       * @param[in]
       * @param[out]
       * @param[in,out]
       *
       * @return void
       */
      void onVoiceCallEmergencyChanged(IN const ObjectPath& objPath, IN const bool emergency) override;

      /**
       * Member function overridden from IBtStackWrapper class
       *
       * @param[in]
       * @param[out]
       * @param[in,out]
       *
       * @return void
       */
      void onVoiceCallDirectionChanged(IN const ObjectPath& objPath, IN const std::string& direction) override;

      /**
       * Member function overridden from IBtStackWrapper class
       *
       * @param[in]
       * @param[out]
       * @param[in,out]
       *
       * @return void
       */
      void onVoiceCallModeChanged(IN const ObjectPath& objPath, IN const std::string& mode) override;

      /**
       * Member function overridden from IBtStackWrapper class
       *
       * @param[in]
       * @param[out]
       * @param[in,out]
       *
       * @return void
       */
      void onVoiceCallTypeChanged(IN const ObjectPath& objPath, IN const std::string& type) override;

   private:
      friend class PmSingleton<EvoBtStackWrapper>;

      /**
       * This method is used to convert the error name and error message to error code
       *
       * @param[in] errorName: error name
       * @param[in] errorMessage: errorMessage
       * @param[out]
       * @param[in,out]
       *
       * @return ErrorCode - error code enum
       */
      ::ccdbusif::evolution::ErrorCode getErrorCode2Enum(IN const ErrorName& errorName,
            IN const ErrorMessage& errorMessage);

      /**
       * This method is used to parse the voice call (or) network operator instance from given object path
       *
       * @param[in] objPath: voice call (or) network operator object path
       * @param[in] length: object path length
       * @param[out]
       * @param[in,out]
       *
       * @return uint8_t - voice call (or) network operator instance
       */
      uint8_t getInstanceFromObjectPath(IN const ObjectPath& objPath, IN const uint8_t length);

      /**
       * This method is used to store the mapping between dbus and pmcore acknowledgment token
       *
       * @param[in] dbusAct: dbus asynchronous completion token
       * @param[in] pmCoreAct: pmcore acknowledgment token
       * @param[out]
       * @param[in,out]
       *
       * @return void
       */
      void mapDbusActToPmCoreAct(IN const act_t dbusAct, IN const ::pmcore::ActType pmCoreAct);

      /**
       * This method is used to retrieve the pmcore acknowledgment token from dbus act
       *
       * @param[in] dbusAct: dbus asynchronous completion token
       * @param[out] pmCoreAct: pmcore acknowledgment token
       * @param[in,out]
       *
       * @return void
       */
      void retrievePmCoreActFromDbusAct(IN const act_t dbusAct, OUT ::pmcore::ActType& pmCoreAct);

      /**
       * This method is used to clear the mapping between dbus and pmcore acknowledgment token
       *
       * @param[in]
       * @param[out]
       * @param[in,out]
       *
       * @return void
       */
      void clearDbusAndPmCoreActMap();

      /**
       * This method is used to store the mapping between dbus and Telephone Number for dial request
       *
       * @param[in] dbusAct: dbus asynchronous completion token
       * @param[in] telephoneNumber: Telephone Number for dial request
       * @param[out]
       * @param[in,out]
       *
       * @return void
       */
      void mapDbusActToTelephoneNumber(IN const act_t dbusAct, IN const ::pmcore::TelephoneNumber telephoneNumber);

      /**
       * This method is used to retrieve the telephone number from dbus act
       *
       * @param[in] dbusAct: dbus asynchronous completion token
       * @param[out] pmCoreAct: telephone number for dial request
       * @param[in,out]
       *
       * @return void
       */
      void retrieveTelephoneNumberFromDbusAct(IN const act_t dbusAct, OUT ::pmcore::TelephoneNumber& telephoneNumber);

      /**
       * This method is used to clear the mapping between dbus and telephone number for dial request
       *
       * @param[in]
       * @param[out]
       * @param[in,out]
       *
       * @return void
       */
      void clearDbusActToTelephoneNumberMap();

      /**
       * This method is used to convert the dbus SupportedFeatures type to PmCore type
       *
       * @param[in] supportedFeaturesIn: dbus supported features list
       * @param[out] supportedFeaturesOut: pmcore supported features type
       * @param[in,out]
       *
       * @return void
       */
      void convertSupportedFeaturesToPmCoreType(const std::vector<std::string>& supportedFeaturesIn,
                     pmcore::SupportedFeatures& supportedFeaturesOut);

      /**
       * This method is used to convert the dbus HfpAudio type to PmCore type
       *
       * @param[in] audio: dbus hfp audio type
       * @param[out] audioDirection: pmcore audio direction type
       * @param[in,out]
       *
       * @return void
       */
      void convertHfpAudioToPmCoreType(IN const std::string& audio, OUT ::pmcore::AudioDirection& audioDirection);

      /**
       * This method is used to convert the dbus NetworkStatus type to PmCore type
       *
       * @param[in] status: dbus network status type
       * @param[out] networkStatus: pmcore network status type
       * @param[in,out]
       *
       * @return void
       */
      void convertNetworkStatusToPmCoreType(IN const std::string& status,
            OUT ::pmcore::NetworkStatusType& networkStatus);

      /**
       * This method is used to convert the dbus VoiceCallDirection type to PmCore type
       *
       * @param[in] direction: dbus voice call direction type
       * @param[out] callDirection: pmcore voice call direction type
       * @param[in,out]
       *
       * @return void
       */
      void convertVoiceCallDirectionToPmCoreType(IN const std::string& direction,
            OUT ::pmcore::CallDirection& callDirection);

      /**
       * This method is used to convert the dbus VoiceCallMode type to PmCore type
       *
       * @param[in] mode: dbus voice call mode type
       * @param[out] callMode: pmcore voice call mode type
       * @param[in,out]
       *
       * @return void
       */
      void convertVoiceCallModeToPmCoreType(IN const std::string& mode, OUT ::pmcore::CallMode& callMode);

      /**
       * This method is used to convert the dbus VoiceCallType type to PmCore type
       *
       * @param[in] type: dbus voice call type
       * @param[out] numberType: pmcore voice call number type
       * @param[in,out]
       *
       * @return void
       */
      void convertVoiceCallTypeToPmCoreType(IN const std::string& type, OUT ::pmcore::NumberType& numberType);

      /**
       * This method is used to convert the PmCore CallerId type to dbus type
       *
       * @param[in] hideCallerId: pmcore hide caller id type
       * @param[out] callerId: dbus hide caller Id type
       * @param[in,out]
       *
       * @return void
       */
      void convertCallerIdToDbusType(IN const ::pmcore::HideCallerId hideCallerId, OUT std::string& callerId);

      /**
       * This method is used to convert the BTSDbusPropertyList type to BTSVoiceCallInfo type
       *
       * @param[in] propertyList: DBus property list
       * @param[out] btsVoiceCallInfo: BTS voice call info
       * @param[in,out]
       *
       * @return void
       */
      void convertBTSDbusPropertyListToBTSVoiceCallInfo(const BTSDbusPropertyList& propertyList,
            ::pmcore::BTSVoiceCallInfo& btsVoiceCallInfo);

      ::pmcore::VehicleConfiguration _vehicleConfig;              /**< vehicle BT configuration */
      std::map<act_t, ::pmcore::ActType>  _dbusAndPmCoreActMap;   /**< map for dbus and pmcore acknowledgment token */
      std::map<act_t, SetPropertyAttributes>  _dbusActToSetPropertyAttributes; /**< map for dbus Act and SetPropertyAttributes */
      std::map<act_t, ::pmcore::TelephoneNumber> _dbusActToTelephoneNumber; /**< map for dbus act and Telephone Number for dial request */

      IEvolutionGeniviDbusHfpCallVolumeCallBackIf*          _hfpCallVolumeCallBack;
      IEvolutionGeniviDbusHfpHandsfreeCallBackIf*           _hfpHandsfreeCallBack;
      IEvolutionGeniviDbusHfpManagerCallBackIf*             _hfpManagerCallBack;
      IEvolutionGeniviDbusHfpModemCallBackIf*               _hfpModemCallBack;
      IEvolutionGeniviDbusHfpNetworkOperatorCallBackIf*     _hfpNetworkOperatorCallBack;
      IEvolutionGeniviDbusHfpNetworkRegistrationCallBackIf* _hfpNetworkRegistrationCallBack;
      IEvolutionGeniviDbusHfpSiriCallBackIf*                _hfpSiriCallBack;
      IEvolutionGeniviDbusHfpVoiceCallManagerCallBackIf*    _hfpVoiceCallManagerCallBack;
      IEvolutionGeniviDbusHfpVoiceCallCallBackIf*           _hfpVoiceCallCallBack;

      IEvolutionGeniviDbusHfpCallVolumeRequestIf*           _hfpCallVolumeRequestIf;
      IEvolutionGeniviDbusHfpHandsfreeRequestIf*            _hfpHandsfreeRequestIf;
      IEvolutionGeniviDbusHfpManagerRequestIf*              _hfpManagerRequestIf;
      IEvolutionGeniviDbusHfpModemRequestIf*                _hfpModemRequestIf;
      IEvolutionGeniviDbusHfpNetworkOperatorRequestIf*      _hfpNetworkOperatorRequestIf;
      IEvolutionGeniviDbusHfpNetworkRegistrationRequestIf*  _hfpNetworkRegistrationRequestIf;
      IEvolutionGeniviDbusHfpSiriRequestIf*                 _hfpSiriRequestIf;
      IEvolutionGeniviDbusHfpVoiceCallManagerRequestIf*     _hfpVoiceCallManagerRequestIf;
      IEvolutionGeniviDbusHfpVoiceCallRequestIf*            _hfpVoiceCallRequestIf;
   };

} //evobtstackwrapper

#endif //EvoBtStackWrapper_h
