/**
 * @file EvolutionGeniviDbusHfpVoiceCallRequestIf.cpp
 *
 * @swcomponent PhoneCallManager
 *
 * @brief Interface definition for EvolutionGenivi DBUS Proxy for HfpVoiceCall
 *
 * @copyright (C) 2016 Robert Bosch GmbH.
 *            The reproduction, distribution and utilization of this file as
 *            well as the communication of its contents to others without express
 *            authorization is prohibited. Offenders will be held liable for the
 *            payment of damages. All rights reserved in the event of the grant
 *            of a patent, utility model or design.
 *
 * @details
 *
 * @ingroup BtStackWrapper
 */

#include "EvolutionGeniviDbusHfpVoiceCallRequestIf.h"
#include "IEvolutionGeniviDbusHfpVoiceCallCallBackIf.h"
#include "CcDbusIfTypes.h"
#include "CcDbusIfUtils.h"
#include "EvolutionGeniviDbusParser.h"
#include "PmAppTrace.h"

using namespace ::org::ofono::VoiceCall;
using namespace ::asf::core;

#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#define ETG_DEFAULT_TRACE_CLASS TR_CLASS_BT_STACK_WRAPPER
#ifdef VARIANT_S_FTR_ENABLE_FW_ETG_USAGE
#include "trcGenProj/Header/EvolutionGeniviDbusHfpVoiceCallRequestIf.cpp.trc.h"
#define ETG_DEFAULT_TRACE_CLASS TR_CLASS_BT_STACK_WRAPPER
#endif
#endif

namespace evobtstackwrapper {

EvolutionGeniviDbusHfpVoiceCallRequestIf::EvolutionGeniviDbusHfpVoiceCallRequestIf()
{
   // _hfpVoiceCallProxy is set later
   _callbackIf = nullptr;
}

EvolutionGeniviDbusHfpVoiceCallRequestIf::~EvolutionGeniviDbusHfpVoiceCallRequestIf()
{
   _callbackIf = nullptr;
}

void EvolutionGeniviDbusHfpVoiceCallRequestIf::createProxy(void)
{
   // these proxies have to be created during runtime
}

void EvolutionGeniviDbusHfpVoiceCallRequestIf::destroyProxy(void)
{
   ETG_TRACE_USR3(("destroyProxy(): VoiceCall"));

   // these proxies have to be destroyed during runtime
   // destroy all proxies now if still available
   _hfpVoiceCallProxy.resetAllProxiesAndClear();
}

void EvolutionGeniviDbusHfpVoiceCallRequestIf::createRuntimeProxy(const ::std::string& objPath)
{
   ETG_TRACE_USR3(("createRuntimeProxy(): VoiceCall: ObjPath: %s", objPath.c_str()));

   if(nullptr != _callbackIf)
   {
      ::ccdbusif::evolution::EvolutionGeniviDbusParser parser;
      ::com::bosch::cm::asf::lang::dbus::Connectors::DBusConnector connectorOptions;
      ::ccdbusif::convertBusType2ConnectorOption(connectorOptions,
            parser.getInterface2BusType(::ccdbusif::evolution::IF_VOICE_CALL));

      ::boost::shared_ptr< VoiceCallProxy > hfpVoiceCallProxy = VoiceCallProxy::createProxy(
            "pmAppEvoHfpVoiceCallPort",
            parser.getInterface2BusName(::ccdbusif::evolution::IF_VOICE_CALL),
            objPath,
            connectorOptions,
            *this);

      _hfpVoiceCallProxy.addProxyInstance(objPath, hfpVoiceCallProxy);
   }
}

void EvolutionGeniviDbusHfpVoiceCallRequestIf::destroyRuntimeProxy(const ::std::string& objPath)
{
   ETG_TRACE_USR3(("destroyRuntimeProxy(): VoiceCall: ObjPath: %s", objPath.c_str()));

   // HINT: destroying runtime proxy is not possible because ASF is storing a proxy "reference" internally;
   //       therefore reset() call will not causing destruction of proxy
   // HINT: due to this we explicitly have to call sendDeregisterAll() to force removing of signal registration
   //       (DBUS: member=RemoveMatch)
   // HINT: sendDeregisterAll() is automatically called during destruction of proxy

   ::boost::shared_ptr< VoiceCallProxy > hfpVoiceCallProxy;
   if(true == _hfpVoiceCallProxy.isProxyAvailable(hfpVoiceCallProxy, objPath))
   {
      hfpVoiceCallProxy->sendDeregisterAll();
   }
   else
   {
      ETG_TRACE_ERR(("VoiceCall Proxy not exist for requested ObjPath"));
   }

   _hfpVoiceCallProxy.removeProxyInstance(objPath);
}

void EvolutionGeniviDbusHfpVoiceCallRequestIf::setCallbackIf(IEvolutionGeniviDbusHfpVoiceCallCallBackIf* callbackIf)
{
   _callbackIf = callbackIf;

   // callback interface is now available => but proxy will be created later
}

bool EvolutionGeniviDbusHfpVoiceCallRequestIf::voiceCallCreated(const ::std::string& objPath)
{
   bool proxyCreated = false;
   // create runtime proxy
   if(nullptr != _callbackIf)
   {
      if(false == _hfpVoiceCallProxy.isInstanceAvailable(objPath))
      {
         createRuntimeProxy(objPath);
         proxyCreated = true;
      }
      else
      {
         ETG_TRACE_ERR(("VoiceCall Proxy already existing for requested ObjPath"));
      }
   }
   return proxyCreated;
}

void EvolutionGeniviDbusHfpVoiceCallRequestIf::voiceCallRemoved(const ::std::string& objPath)
{
   // destroy runtime proxy
   if(nullptr != _callbackIf)
   {
      destroyRuntimeProxy(objPath);
   }
}

void EvolutionGeniviDbusHfpVoiceCallRequestIf::onAvailable(const boost::shared_ptr<Proxy>& proxy,
      const ServiceStateChange& stateChange)
{
   ETG_TRACE_USR3(("onAvailable(): VoiceCall"));

   for(unsigned int i = 0; i < _hfpVoiceCallProxy.getNumberOfProxyInstances(); i++)
   {
      ::boost::shared_ptr< VoiceCallProxy >& hfpVoiceCallProxy = _hfpVoiceCallProxy[i];
      if((nullptr != hfpVoiceCallProxy.get()) && (proxy == hfpVoiceCallProxy))
      {
         if(nullptr != _callbackIf)
         {
            hfpVoiceCallProxy->sendPropertyChangedRegister(*_callbackIf);

            _callbackIf->onProxyAvailable(stateChange.getPreviousState(), stateChange.getCurrentState(),
                  hfpVoiceCallProxy->getDBusObjectPath());

            //GetProperties is not required for VoiceCall interface as of now
            //sendGetPropertiesRequest(hfpVoiceCallProxy->getDBusObjectPath());
         }
         break;
      }
   }
}

void EvolutionGeniviDbusHfpVoiceCallRequestIf::onUnavailable(const boost::shared_ptr<Proxy>& proxy,
      const ServiceStateChange& stateChange)
{
   ETG_TRACE_USR3(("onUnavailable(): VoiceCall"));

   for(unsigned int i = 0; i < _hfpVoiceCallProxy.getNumberOfProxyInstances(); i++)
   {
      ::boost::shared_ptr< VoiceCallProxy >& hfpVoiceCallProxy = _hfpVoiceCallProxy[i];
      if((nullptr != hfpVoiceCallProxy.get()) && (proxy == hfpVoiceCallProxy))
      {
         hfpVoiceCallProxy->sendDeregisterAll();

         if(nullptr != _callbackIf)
         {
            _callbackIf->onProxyUnavailable(stateChange.getPreviousState(), stateChange.getCurrentState(),
                  hfpVoiceCallProxy->getDBusObjectPath());
         }
         break;
      }
   }
}

act_t EvolutionGeniviDbusHfpVoiceCallRequestIf::sendGetPropertiesRequest(const ::std::string& objPath)
{
   act_t act = ::ccdbusif::DEFAULT_ACT;
   ::boost::shared_ptr< VoiceCallProxy > proxy;

   if(true == _hfpVoiceCallProxy.isInstanceAvailable(proxy, objPath))
   {
      if((nullptr != _callbackIf) && (nullptr != proxy.get()))
      {
         act = proxy->sendGetPropertiesRequest(*_callbackIf);
      }
   }

   return act;
}

act_t EvolutionGeniviDbusHfpVoiceCallRequestIf::sendHangupRequest(const ::std::string& objPath)
{
   act_t act = ::ccdbusif::DEFAULT_ACT;
   ::boost::shared_ptr< VoiceCallProxy > proxy;

   if(true == _hfpVoiceCallProxy.isInstanceAvailable(proxy, objPath))
   {
      if((nullptr != _callbackIf) && (nullptr != proxy.get()))
      {
         act = proxy->sendHangupRequest(*_callbackIf);
      }
   }

   return act;
}

act_t EvolutionGeniviDbusHfpVoiceCallRequestIf::sendAnswerRequest(const ::std::string& objPath)
{
   act_t act = ::ccdbusif::DEFAULT_ACT;
   ::boost::shared_ptr< VoiceCallProxy > proxy;

   if(true == _hfpVoiceCallProxy.isInstanceAvailable(proxy, objPath))
   {
      if((nullptr != _callbackIf) && (nullptr != proxy.get()))
      {
         act = proxy->sendAnswerRequest(*_callbackIf);
      }
   }

   return act;
}

act_t EvolutionGeniviDbusHfpVoiceCallRequestIf::sendHoldRequest(const ::std::string& objPath)
{
   act_t act = ::ccdbusif::DEFAULT_ACT;
   ::boost::shared_ptr< VoiceCallProxy > proxy;

   if(true == _hfpVoiceCallProxy.isInstanceAvailable(proxy, objPath))
   {
      if((nullptr != _callbackIf) && (nullptr != proxy.get()))
      {
         act = proxy->sendHoldRequest(*_callbackIf);
      }
   }

   return act;
}

} //evobtstackwrapper
