/**
 * @file EvolutionGeniviDbusHfpSiriRequestIf.cpp
 *
 * @swcomponent PhoneCallManager
 *
 * @brief Interface definition for EvolutionGenivi DBUS Proxy for HfpSiri
 *
 * @copyright (C) 2016 Robert Bosch GmbH.
 *            The reproduction, distribution and utilization of this file as
 *            well as the communication of its contents to others without express
 *            authorization is prohibited. Offenders will be held liable for the
 *            payment of damages. All rights reserved in the event of the grant
 *            of a patent, utility model or design.
 *
 * @details
 *
 * @ingroup BtStackWrapper
 */

#include "EvolutionGeniviDbusHfpSiriRequestIf.h"
#include "IEvolutionGeniviDbusHfpSiriCallBackIf.h"
#include "CcDbusIfTypes.h"
#include "CcDbusIfUtils.h"
#include "EvolutionGeniviDbusParser.h"
#include "PmAppTrace.h"

using namespace ::org::ofono::evo::Siri;
using namespace ::asf::core;

#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#define ETG_DEFAULT_TRACE_CLASS TR_CLASS_BT_STACK_WRAPPER
#ifdef VARIANT_S_FTR_ENABLE_FW_ETG_USAGE
#include "trcGenProj/Header/EvolutionGeniviDbusHfpSiriRequestIf.cpp.trc.h"
#define ETG_DEFAULT_TRACE_CLASS TR_CLASS_BT_STACK_WRAPPER
#endif
#endif

namespace evobtstackwrapper {

EvolutionGeniviDbusHfpSiriRequestIf::EvolutionGeniviDbusHfpSiriRequestIf()
{
   // _hfpSiriProxy is set later
   _callbackIf = nullptr;
}

EvolutionGeniviDbusHfpSiriRequestIf::~EvolutionGeniviDbusHfpSiriRequestIf()
{
   _callbackIf = nullptr;
}

void EvolutionGeniviDbusHfpSiriRequestIf::createProxy(void)
{
   // these proxies have to be created during runtime
}

void EvolutionGeniviDbusHfpSiriRequestIf::destroyProxy(void)
{
   ETG_TRACE_USR3(("destroyProxy(): Siri"));

   // these proxies have to be destroyed during runtime
   // destroy all proxies now if still available
   _hfpSiriProxy.resetAllProxiesAndClear();
}

void EvolutionGeniviDbusHfpSiriRequestIf::createRuntimeProxy(const ::std::string& objPath)
{
   ETG_TRACE_USR3(("createRuntimeProxy(): Siri: ObjPath: %s", objPath.c_str()));

   if(nullptr != _callbackIf)
   {
      if(false == _hfpSiriProxy.isInstanceAvailable(objPath))
      {
         ::ccdbusif::evolution::EvolutionGeniviDbusParser parser;
         ::com::bosch::cm::asf::lang::dbus::Connectors::DBusConnector connectorOptions;
         ::ccdbusif::convertBusType2ConnectorOption(connectorOptions,
               parser.getInterface2BusType(::ccdbusif::evolution::IF_SIRI));

         ::boost::shared_ptr< SiriProxy > hfpSiriProxy = SiriProxy::createProxy(
               "pmAppEvoHfpSiri",
               parser.getInterface2BusName(::ccdbusif::evolution::IF_SIRI),
               objPath,
               connectorOptions,
               *this);

         _hfpSiriProxy.addProxyInstance(objPath, hfpSiriProxy);
      }
      else
      {
         ETG_TRACE_ERR(("Siri Proxy already existing for requested ObjPath"));
      }
   }
}

void EvolutionGeniviDbusHfpSiriRequestIf::destroyRuntimeProxy(const ::std::string& objPath)
{
   ETG_TRACE_USR3(("destroyRuntimeProxy(): Siri: ObjPath: %s", objPath.c_str()));

   // HINT: destroying runtime proxy is not possible because ASF is storing a proxy "reference" internally;
   //       therefore reset() call will not causing destruction of proxy
   // HINT: due to this we explicitly have to call sendDeregisterAll() to force removing of signal registration
   //       (DBUS: member=RemoveMatch)
   // HINT: sendDeregisterAll() is automatically called during destruction of proxy

   ::boost::shared_ptr< SiriProxy > hfpSiriProxy;
   if(true == _hfpSiriProxy.isProxyAvailable(hfpSiriProxy, objPath))
   {
      hfpSiriProxy->sendDeregisterAll();
   }
   else
   {
      ETG_TRACE_ERR(("Siri Proxy not exist for requested ObjPath"));
   }

   _hfpSiriProxy.removeProxyInstance(objPath);
}

void EvolutionGeniviDbusHfpSiriRequestIf::setCallbackIf(IEvolutionGeniviDbusHfpSiriCallBackIf* callbackIf)
{
   _callbackIf = callbackIf;

   // callback interface is now available => but proxy will be created later
}

void EvolutionGeniviDbusHfpSiriRequestIf::deviceCreated(const ::std::string& objPath)
{
   // create runtime proxy
   if(nullptr != _callbackIf)
   {
      createRuntimeProxy(objPath);
   }
}

void EvolutionGeniviDbusHfpSiriRequestIf::deviceRemoved(const ::std::string& objPath)
{
   // destroy runtime proxy
   if(nullptr != _callbackIf)
   {
      destroyRuntimeProxy(objPath);
   }
}

void EvolutionGeniviDbusHfpSiriRequestIf::onAvailable(const boost::shared_ptr<Proxy>& proxy,
      const ServiceStateChange& stateChange)
{
   ETG_TRACE_USR3(("onAvailable(): Siri"));

   for(unsigned int i = 0; i < _hfpSiriProxy.getNumberOfProxyInstances(); i++)
   {
      ::boost::shared_ptr< SiriProxy >& hfpSiriProxy = _hfpSiriProxy[i];
      if((nullptr != hfpSiriProxy.get()) && (proxy == hfpSiriProxy))
      {
         if(nullptr != _callbackIf)
         {
            hfpSiriProxy->sendPropertyChangedRegister(*_callbackIf);

            _callbackIf->onProxyAvailable(stateChange.getPreviousState(), stateChange.getCurrentState(),
                  hfpSiriProxy->getDBusObjectPath());

            //GetProperties
            sendGetPropertiesRequest(hfpSiriProxy->getDBusObjectPath());
         }
         break;
      }
   }
}

void EvolutionGeniviDbusHfpSiriRequestIf::onUnavailable(const boost::shared_ptr<Proxy>& proxy,
      const ServiceStateChange& stateChange)
{
   ETG_TRACE_USR3(("onUnavailable(): Siri"));

   for(unsigned int i = 0; i < _hfpSiriProxy.getNumberOfProxyInstances(); i++)
   {
      ::boost::shared_ptr< SiriProxy >& hfpSiriProxy = _hfpSiriProxy[i];
      if((nullptr != hfpSiriProxy.get()) && (proxy == hfpSiriProxy))
      {
         hfpSiriProxy->sendDeregisterAll();

         if(nullptr != _callbackIf)
         {
            _callbackIf->onProxyUnavailable(stateChange.getPreviousState(), stateChange.getCurrentState(),
                  hfpSiriProxy->getDBusObjectPath());
         }
         break;
      }
   }
}

act_t EvolutionGeniviDbusHfpSiriRequestIf::sendGetPropertiesRequest(const ::std::string& objPath)
{
   act_t act = ::ccdbusif::DEFAULT_ACT;
   ::boost::shared_ptr< SiriProxy > proxy;

   if(true == _hfpSiriProxy.isInstanceAvailable(proxy, objPath))
   {
      if((nullptr != _callbackIf) && (nullptr != proxy.get()))
      {
         act = proxy->sendGetPropertiesRequest(*_callbackIf);
      }
   }

   return act;
}

act_t EvolutionGeniviDbusHfpSiriRequestIf::sendEnableRequest(const ::std::string& objPath, uint16 vendor_id,
      uint16 product_id, uint16 version, uint16 support_type)
{
   act_t act = ::ccdbusif::DEFAULT_ACT;
   ::boost::shared_ptr< SiriProxy > proxy;

   if(true == _hfpSiriProxy.isInstanceAvailable(proxy, objPath))
   {
      if((nullptr != _callbackIf) && (nullptr != proxy.get()))
      {
         act = proxy->sendEnableRequest(*_callbackIf, vendor_id, product_id, version, support_type);
      }
   }

   return act;
}

act_t EvolutionGeniviDbusHfpSiriRequestIf::sendSetNRRequest(const ::std::string& objPath, uint8 status)
{
   act_t act = ::ccdbusif::DEFAULT_ACT;
   ::boost::shared_ptr< SiriProxy > proxy;

   if(true == _hfpSiriProxy.isInstanceAvailable(proxy, objPath))
   {
      if((nullptr != _callbackIf) && (nullptr != proxy.get()))
      {
         act = proxy->sendSetNRRequest(*_callbackIf, status);
      }
   }

   return act;
}

act_t EvolutionGeniviDbusHfpSiriRequestIf::sendSetPropertyRequest(const ::std::string& objPath,
      const ::std::string& property, const bool value)
{
   act_t act = ::ccdbusif::DEFAULT_ACT;
   ::boost::shared_ptr< SiriProxy > proxy;

   if(true == _hfpSiriProxy.isInstanceAvailable(proxy, objPath))
   {
      if((nullptr != _callbackIf) && (nullptr != proxy.get()))
      {
         ::asf::dbus::DBusVariant variant;
         ::ccdbusif::DbusVariant vValue;
         vValue.setBool(value);
         ::ccdbusif::evolution::EvolutionGeniviDbusParser parser;
         parser.addVariantValue(variant, vValue);

         act = proxy->sendSetPropertyRequest(*_callbackIf, property, variant);
      }
   }

   return act;
}

} //evobtstackwrapper
