/**
 * @file EvolutionGeniviDbusHfpSiriCallBackIf.cpp
 *
 * @swcomponent PhoneCallManager
 *
 * @brief Interface definition for EvolutionGenivi DBUS callback for HfpSiri
 *
 * @copyright (C) 2016 Robert Bosch GmbH.
 *            The reproduction, distribution and utilization of this file as
 *            well as the communication of its contents to others without express
 *            authorization is prohibited. Offenders will be held liable for the
 *            payment of damages. All rights reserved in the event of the grant
 *            of a patent, utility model or design.
 *
 * @details
 *
 * @ingroup BtStackWrapper
 */

#include "EvolutionGeniviDbusHfpSiriCallBackIf.h"
#include "IBtStackWrapper.h"
#include "EvolutionGeniviDbusParser.h"
#include "PmAppTrace.h"

using namespace ::org::ofono::evo::Siri;
using namespace ::asf::core;

#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#define ETG_DEFAULT_TRACE_CLASS TR_CLASS_BT_STACK_WRAPPER
#ifdef VARIANT_S_FTR_ENABLE_FW_ETG_USAGE
#include "trcGenProj/Header/EvolutionGeniviDbusHfpSiriCallBackIf.cpp.trc.h"
#define ETG_DEFAULT_TRACE_CLASS TR_CLASS_BT_STACK_WRAPPER
#endif
#endif

namespace evobtstackwrapper {

EvolutionGeniviDbusHfpSiriCallBackIf::EvolutionGeniviDbusHfpSiriCallBackIf()
{
   _btStackWrapper = nullptr;
}

EvolutionGeniviDbusHfpSiriCallBackIf::EvolutionGeniviDbusHfpSiriCallBackIf(IBtStackWrapper* btStackWrapper)
{
   _btStackWrapper = btStackWrapper;
}

EvolutionGeniviDbusHfpSiriCallBackIf::~EvolutionGeniviDbusHfpSiriCallBackIf()
{
   _btStackWrapper = nullptr;
}

void EvolutionGeniviDbusHfpSiriCallBackIf::onProxyAvailable(
      const ServiceState previousState, const ServiceState currentState, const ::std::string& objPath)
{
   (void)(previousState);
   (void)(currentState);
   (void)(objPath);

   if(nullptr != _btStackWrapper)
   {
      _btStackWrapper->onProxyServiceAvailabilityStatus(::ccdbusif::evolution::IF_SIRI, objPath, true);
   }
}

void EvolutionGeniviDbusHfpSiriCallBackIf::onProxyUnavailable(
      const ServiceState previousState, const ServiceState currentState, const ::std::string& objPath)
{
   (void)(previousState);
   (void)(currentState);
   (void)(objPath);

   if(nullptr != _btStackWrapper)
   {
      _btStackWrapper->onProxyServiceAvailabilityStatus(::ccdbusif::evolution::IF_SIRI, objPath, false);
   }
}

void EvolutionGeniviDbusHfpSiriCallBackIf::onGetPropertiesError(
      const ::boost::shared_ptr< SiriProxy >& proxy, const ::boost::shared_ptr< GetPropertiesError >& error)
{
   ETG_TRACE_USR4(("onGetPropertiesError(): Siri"));

   (void)(proxy);
   (void)(error);

   //TODO: implement
}

void EvolutionGeniviDbusHfpSiriCallBackIf::onGetPropertiesResponse(
      const ::boost::shared_ptr< SiriProxy >& proxy, const ::boost::shared_ptr< GetPropertiesResponse >& response)
{
   ETG_TRACE_USR4(("onGetPropertiesResponse(): Siri"));

   ::ccdbusif::evolution::EvolutionGeniviDbusParser parser;
   parser.setTraces(true);
   BTSDbusPropertyList outPropertyList;

   const std::string& objPath = proxy->getDBusObjectPath();

   ETG_TRACE_USR4(("Siri: ObjPath: %s", objPath.c_str()));

   if(true == parser.findAndParseProperty(outPropertyList, response->getPropertiesMutable(),
         parser.getHfpSiriProperty2String(::ccdbusif::evolution::HFP_SIRI_STATUS),
         (int)::ccdbusif::evolution::IF_SIRI))
   {
      onSiriStatusChanged(objPath, outPropertyList);
   }

   outPropertyList.clear();

   if(true == parser.findAndParseProperty(outPropertyList, response->getPropertiesMutable(),
         parser.getHfpSiriProperty2String(::ccdbusif::evolution::HFP_SIRI_NR),
         (int)::ccdbusif::evolution::IF_SIRI))
   {
      onSiriNRChanged(objPath, outPropertyList);
   }

   outPropertyList.clear();

   if(true == parser.findAndParseProperty(outPropertyList, response->getPropertiesMutable(),
         parser.getHfpSiriProperty2String(::ccdbusif::evolution::HFP_SIRI_EYESFREEMODE),
         (int)::ccdbusif::evolution::IF_SIRI))
   {
      onSiriEyesFreeModeChanged(objPath, outPropertyList);
   }

   outPropertyList.clear();

   if(true == parser.findAndParseProperty(outPropertyList, response->getPropertiesMutable(),
         parser.getHfpSiriProperty2String(::ccdbusif::evolution::HFP_SIRI_SUPPORTEDTYPE),
         (int)::ccdbusif::evolution::IF_SIRI))
   {
      onSiriSupportedTypeChanged(objPath, outPropertyList);
   }

   outPropertyList.clear();
}

void EvolutionGeniviDbusHfpSiriCallBackIf::onEnableError(const ::boost::shared_ptr< SiriProxy >& proxy,
      const ::boost::shared_ptr< EnableError >& error)
{
   const std::string& objPath = proxy->getDBusObjectPath();
   const std::string& errorName = error->getName();
   const std::string& errorMessage = error->getMessage();
   const act_t act = error->getAct();

   if(nullptr != _btStackWrapper)
   {
      _btStackWrapper->onSiriEnableError(objPath, errorName, errorMessage, act);
   }
}

void EvolutionGeniviDbusHfpSiriCallBackIf::onEnableResponse(const ::boost::shared_ptr< SiriProxy >& proxy,
      const ::boost::shared_ptr< EnableResponse >& response)
{
   const std::string& objPath = proxy->getDBusObjectPath();
   const act_t act = response->getAct();

   if(nullptr != _btStackWrapper)
   {
      _btStackWrapper->onSiriEnableResponse(objPath, act);
   }
}

void EvolutionGeniviDbusHfpSiriCallBackIf::onSetNRError(const ::boost::shared_ptr< SiriProxy >& proxy,
      const ::boost::shared_ptr< SetNRError >& error)
{
   const std::string& objPath = proxy->getDBusObjectPath();
   const std::string& errorName = error->getName();
   const std::string& errorMessage = error->getMessage();
   const act_t act = error->getAct();

   if(nullptr != _btStackWrapper)
   {
      _btStackWrapper->onSiriSetNRError(objPath, errorName, errorMessage, act);
   }
}

void EvolutionGeniviDbusHfpSiriCallBackIf::onSetNRResponse(const ::boost::shared_ptr< SiriProxy >& proxy,
      const ::boost::shared_ptr< SetNRResponse >& response)
{
   const std::string& objPath = proxy->getDBusObjectPath();
   const act_t act = response->getAct();

   if(nullptr != _btStackWrapper)
   {
      _btStackWrapper->onSiriSetNRResponse(objPath, act);
   }
}

void EvolutionGeniviDbusHfpSiriCallBackIf::onSetPropertyError(const ::boost::shared_ptr< SiriProxy >& proxy,
      const ::boost::shared_ptr< SetPropertyError >& error)
{
   ETG_TRACE_USR4(("onSetPropertyError(): Siri"));

   (void)(proxy);
   (void)(error);

   //TODO: implement
   //const std::string& objPath = proxy->getDBusObjectPath();
   //const std::string& errorName = error->getName();
   //const std::string& errorMessage = error->getMessage();
   //const act_t act = error->getAct();
}

void EvolutionGeniviDbusHfpSiriCallBackIf::onSetPropertyResponse(const ::boost::shared_ptr< SiriProxy >& proxy,
      const ::boost::shared_ptr< SetPropertyResponse >& response)
{
   ETG_TRACE_USR4(("onSetPropertyResponse(): Siri"));

   (void)(proxy);
   (void)(response);

   //TODO: implement
   //const std::string& objPath = proxy->getDBusObjectPath();
   //const act_t act = response->getAct();

}

void EvolutionGeniviDbusHfpSiriCallBackIf::onPropertyChangedError(const ::boost::shared_ptr< SiriProxy >& proxy,
      const ::boost::shared_ptr< PropertyChangedError >& error)
{
   (void)(proxy);
   (void)(error);
   // is never triggered
   ETG_TRACE_ERR(("EvolutionGeniviDbusHfpSiriCallBackIf::onPropertyChangedError"));
}

void EvolutionGeniviDbusHfpSiriCallBackIf::onPropertyChangedSignal(const ::boost::shared_ptr< SiriProxy >& proxy,
      const ::boost::shared_ptr< PropertyChangedSignal >& signal)
{
   ETG_TRACE_USR4(("onPropertyChangedSignal(): Siri"));

   //check received properties for Status, NR, EyesFreeMode and SupportedType
   ::ccdbusif::evolution::EvolutionGeniviDbusParser parser;
   parser.setTraces(true);
   ::ccdbusif::evolution::HfpSiriProperty propEnum = parser.getHfpSiriProperty2Enum(signal->getProperty());

   BTSDbusPropertyList outPropertyList;
   parser.parseProperty(outPropertyList, signal->getValueMutable(), signal->getProperty(),
         (int)::ccdbusif::evolution::IF_SIRI);

   const std::string& objPath = proxy->getDBusObjectPath();

   ETG_TRACE_USR4(("Siri: ObjPath: %s", objPath.c_str()));

   if(::ccdbusif::evolution::HFP_SIRI_STATUS == propEnum)
   {
      onSiriStatusChanged(objPath, outPropertyList);
   }
   else if(::ccdbusif::evolution::HFP_SIRI_NR == propEnum)
   {
      onSiriNRChanged(objPath, outPropertyList);
   }
   else if(::ccdbusif::evolution::HFP_SIRI_EYESFREEMODE == propEnum)
   {
      onSiriEyesFreeModeChanged(objPath, outPropertyList);
   }
   else if(::ccdbusif::evolution::HFP_SIRI_SUPPORTEDTYPE == propEnum)
   {
      onSiriSupportedTypeChanged(objPath, outPropertyList);
   }

   outPropertyList.clear();
}

void EvolutionGeniviDbusHfpSiriCallBackIf::onSiriStatusChanged(const std::string& objPath,
      const BTSDbusPropertyList outPropertyList)
{
   ETG_TRACE_USR4(("Siri: Property: Status"));

   if(OUTPROPERTY_LIST_SIZE == outPropertyList.size())
   {
      const ::ccdbusif::DbusVariantProperty& outProperty = outPropertyList[0];

      const uint8_t& status = outProperty.propData.getUInt8();

      ETG_TRACE_USR4(("Siri: Value: %d", status));

      if(nullptr != _btStackWrapper)
      {
         _btStackWrapper->onSiriStatusChanged(objPath, status);
      }
   }
   else
   {
      ETG_TRACE_ERR(("EvolutionGeniviDbusHfpSiriCallBackIf::onSiriStatusChanged:Property list size exceeds max limit"));
   }
}

void EvolutionGeniviDbusHfpSiriCallBackIf::onSiriNRChanged(const std::string& objPath,
      const BTSDbusPropertyList outPropertyList)
{
   ETG_TRACE_USR4(("onSiriNRChanged: Siri: Property: NR"));

   if(OUTPROPERTY_LIST_SIZE == outPropertyList.size())
   {
      const ::ccdbusif::DbusVariantProperty& outProperty = outPropertyList[0];

      const uint8_t& noiseReduction = outProperty.propData.getUInt8();

      ETG_TRACE_USR4(("Siri: Value: %d", noiseReduction));

      if(nullptr != _btStackWrapper)
      {
         _btStackWrapper->onSiriNRChanged(objPath, noiseReduction);
      }
   }
   else
   {
      ETG_TRACE_ERR(("EvolutionGeniviDbusHfpSiriCallBackIf::onSiriNRChanged:Property list size exceeds max limit"));
   }
}

void EvolutionGeniviDbusHfpSiriCallBackIf::onSiriEyesFreeModeChanged(const std::string& objPath,
      const BTSDbusPropertyList outPropertyList)
{
   ETG_TRACE_USR4(("Siri: Property: EyesFreeMode"));

   if(OUTPROPERTY_LIST_SIZE == outPropertyList.size())
   {
      const ::ccdbusif::DbusVariantProperty& outProperty = outPropertyList[0];

      const bool eyesFreeMode = outProperty.propData.getBool();

      ETG_TRACE_USR4(("Siri: Value: %d", eyesFreeMode));

      if(nullptr != _btStackWrapper)
      {
         _btStackWrapper->onSiriEyesFreeModeChanged(objPath, eyesFreeMode);
      }
   }
   else
   {
      ETG_TRACE_ERR(("EvolutionGeniviDbusHfpSiriCallBackIf::onSiriEyesFreeModeChanged:Property list size exceeds max limit"));
   }
}

void EvolutionGeniviDbusHfpSiriCallBackIf::onSiriSupportedTypeChanged(const std::string& objPath,
      const BTSDbusPropertyList outPropertyList)
{
   ETG_TRACE_USR4(("Siri: Property: SupportedType"));

   if(OUTPROPERTY_LIST_SIZE == outPropertyList.size())
   {
      const ::ccdbusif::DbusVariantProperty& outProperty = outPropertyList[0];

      const uint16_t& supportedType = outProperty.propData.getUInt16();

      ETG_TRACE_USR4(("Siri: Value: %d", supportedType));

      if(nullptr != _btStackWrapper)
      {
         _btStackWrapper->onSiriSupportedTypeChanged(objPath, supportedType);
      }
   }
   else
   {
      ETG_TRACE_ERR(("EvolutionGeniviDbusHfpSiriCallBackIf::onSiriSupportedTypeChanged:Property list size exceeds max limit"));
   }
}

} //evobtstackwrapper
