/**
 * @file EvolutionGeniviDbusHfpNetworkRegistrationCallBackIf.cpp
 *
 * @swcomponent PhoneCallManager
 *
 * @brief Interface definition for EvolutionGenivi DBUS callback for HfpNetworkRegistration
 *
 * @copyright (C) 2016 Robert Bosch GmbH.
 *            The reproduction, distribution and utilization of this file as
 *            well as the communication of its contents to others without express
 *            authorization is prohibited. Offenders will be held liable for the
 *            payment of damages. All rights reserved in the event of the grant
 *            of a patent, utility model or design.
 *
 * @details
 *
 * @ingroup BtStackWrapper
 */

#include "EvolutionGeniviDbusHfpNetworkRegistrationCallBackIf.h"
#include "IBtStackWrapper.h"
#include "EvolutionGeniviDbusParser.h"
#include "PmAppTrace.h"

using namespace ::org::ofono::NetworkRegistration;
using namespace ::asf::core;

#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#define ETG_DEFAULT_TRACE_CLASS TR_CLASS_BT_STACK_WRAPPER
#ifdef VARIANT_S_FTR_ENABLE_FW_ETG_USAGE
#include "trcGenProj/Header/EvolutionGeniviDbusHfpNetworkRegistrationCallBackIf.cpp.trc.h"
#define ETG_DEFAULT_TRACE_CLASS TR_CLASS_BT_STACK_WRAPPER
#endif
#endif

namespace evobtstackwrapper {

EvolutionGeniviDbusHfpNetworkRegistrationCallBackIf::EvolutionGeniviDbusHfpNetworkRegistrationCallBackIf()
{
   _btStackWrapper = nullptr;
}

EvolutionGeniviDbusHfpNetworkRegistrationCallBackIf::EvolutionGeniviDbusHfpNetworkRegistrationCallBackIf(
      IBtStackWrapper* btStackWrapper)
{
   _btStackWrapper = btStackWrapper;
}

EvolutionGeniviDbusHfpNetworkRegistrationCallBackIf::~EvolutionGeniviDbusHfpNetworkRegistrationCallBackIf()
{
   _btStackWrapper = nullptr;
}

void EvolutionGeniviDbusHfpNetworkRegistrationCallBackIf::onProxyAvailable(
      const ServiceState previousState, const ServiceState currentState, const ::std::string& objPath)
{
   (void)(previousState);
   (void)(currentState);
   (void)(objPath);

   //TODO: implement
}

void EvolutionGeniviDbusHfpNetworkRegistrationCallBackIf::onProxyUnavailable(
      const ServiceState previousState, const ServiceState currentState, const ::std::string& objPath)
{
   (void)(previousState);
   (void)(currentState);
   (void)(objPath);

   //TODO: implement
}

void EvolutionGeniviDbusHfpNetworkRegistrationCallBackIf::onGetPropertiesError(
      const ::boost::shared_ptr< NetworkRegistrationProxy >& proxy,
      const ::boost::shared_ptr< GetPropertiesError >& error)
{
   ETG_TRACE_USR4(("onGetPropertiesError(): NetworkRegistration"));

   (void)(proxy);
   (void)(error);

   //TODO: implement
}

void EvolutionGeniviDbusHfpNetworkRegistrationCallBackIf::onGetPropertiesResponse(
      const ::boost::shared_ptr< NetworkRegistrationProxy >& proxy,
      const ::boost::shared_ptr< GetPropertiesResponse >& response)
{
   ETG_TRACE_USR4(("onGetPropertiesResponse(): NetworkRegistration"));

   //check received properties for Status, MobileCountryCode, Name, Strength and CurrentOperator
   ::ccdbusif::evolution::EvolutionGeniviDbusParser parser;
   parser.setTraces(true);
   BTSDbusPropertyList outPropertyList;

   const std::string& objPath = proxy->getDBusObjectPath();

   ETG_TRACE_USR4(("GetProperties: NetworkRegistration: ObjPath: %s", objPath.c_str()));

   if(true == parser.findAndParseProperty(outPropertyList, response->getPropertiesMutable(),
         parser.getHfpNetworkRegistrationProperty2String(::ccdbusif::evolution::HFP_NETWORK_REGISTRATION_STATUS),
         (int)::ccdbusif::evolution::IF_NETWORK_REGISTRATION))
   {
      onNetworkStatusChanged(objPath, outPropertyList);
   }

   outPropertyList.clear();

   if(true == parser.findAndParseProperty(outPropertyList, response->getPropertiesMutable(),
         parser.getHfpNetworkRegistrationProperty2String(
               ::ccdbusif::evolution::HFP_NETWORK_REGISTRATION_MOBILECOUNTRYCODE),
         (int)::ccdbusif::evolution::IF_NETWORK_REGISTRATION))
   {
      onNetworkMobileCountryCodeChanged(objPath, outPropertyList);
   }

   outPropertyList.clear();

   if(true == parser.findAndParseProperty(outPropertyList, response->getPropertiesMutable(),
         parser.getHfpNetworkRegistrationProperty2String(::ccdbusif::evolution::HFP_NETWORK_REGISTRATION_NAME),
         (int)::ccdbusif::evolution::IF_NETWORK_REGISTRATION))
   {
      onNetworkOperatorNameChanged(objPath, outPropertyList);
   }

   outPropertyList.clear();

   if(true == parser.findAndParseProperty(outPropertyList, response->getPropertiesMutable(),
         parser.getHfpNetworkRegistrationProperty2String(::ccdbusif::evolution::HFP_NETWORK_REGISTRATION_STRENGTH),
         (int)::ccdbusif::evolution::IF_NETWORK_REGISTRATION))
   {
      onNetworkSignalStrengthChanged(objPath, outPropertyList);
   }

   outPropertyList.clear();

   if(true == parser.findAndParseProperty(outPropertyList, response->getPropertiesMutable(),
         parser.getHfpNetworkRegistrationProperty2String(
               ::ccdbusif::evolution::HFP_NETWORK_REGISTRATION_CURRENTOPERATOR),
         (int)::ccdbusif::evolution::IF_NETWORK_REGISTRATION))
   {
      onNetworkCurrentOperatorChanged(objPath, outPropertyList);
   }

   outPropertyList.clear();
}

void EvolutionGeniviDbusHfpNetworkRegistrationCallBackIf::onPropertyChangedError(
      const ::boost::shared_ptr< NetworkRegistrationProxy >& proxy,
      const ::boost::shared_ptr< PropertyChangedError >& error)
{
   (void)(proxy);
   (void)(error);
   // is never triggered
   ETG_TRACE_ERR(("EvolutionGeniviDbusHfpNetworkRegistrationCallBackIf::onPropertyChangedError"));
}

void EvolutionGeniviDbusHfpNetworkRegistrationCallBackIf::onPropertyChangedSignal(
      const ::boost::shared_ptr< NetworkRegistrationProxy >& proxy,
      const ::boost::shared_ptr< PropertyChangedSignal >& signal)
{
   ETG_TRACE_USR4(("onPropertyChangedSignal(): NetworkRegistration"));

   //check received properties for Status, MobileCountryCode, Name, Strength and CurrentOperator
   ::ccdbusif::evolution::EvolutionGeniviDbusParser parser;
   parser.setTraces(true);
   ::ccdbusif::evolution::HfpNetworkRegistrationProperty propEnum =
         parser.getHfpNetworkRegistrationProperty2Enum(signal->getName());

   BTSDbusPropertyList outPropertyList;
   parser.parseProperty(outPropertyList, signal->getValueMutable(), signal->getName(),
         (int)::ccdbusif::evolution::IF_NETWORK_REGISTRATION);

   const std::string& objPath = proxy->getDBusObjectPath();

   ETG_TRACE_USR4(("NetworkRegistration: ObjPath: %s", objPath.c_str()));

   if(::ccdbusif::evolution::HFP_NETWORK_REGISTRATION_STATUS == propEnum)
   {
      onNetworkStatusChanged(objPath, outPropertyList);
   }
   else if(::ccdbusif::evolution::HFP_NETWORK_REGISTRATION_MOBILECOUNTRYCODE == propEnum)
   {
      onNetworkMobileCountryCodeChanged(objPath, outPropertyList);
   }
   else if(::ccdbusif::evolution::HFP_NETWORK_REGISTRATION_NAME == propEnum)
   {
      onNetworkOperatorNameChanged(objPath, outPropertyList);
   }
   else if(::ccdbusif::evolution::HFP_NETWORK_REGISTRATION_STRENGTH == propEnum)
   {
      onNetworkSignalStrengthChanged(objPath, outPropertyList);
   }
   else if(::ccdbusif::evolution::HFP_NETWORK_REGISTRATION_CURRENTOPERATOR == propEnum)
   {
      onNetworkCurrentOperatorChanged(objPath, outPropertyList);
   }

   outPropertyList.clear();
}

void EvolutionGeniviDbusHfpNetworkRegistrationCallBackIf::onNetworkStatusChanged(const std::string& objPath,
      const BTSDbusPropertyList outPropertyList)
{
   ETG_TRACE_USR4(("NetworkRegistration: Property: Status"));

   if(OUTPROPERTY_LIST_SIZE == outPropertyList.size())
   {
      const ::ccdbusif::DbusVariantProperty& outProperty = outPropertyList[0];

      const std::string& status = outProperty.propData.getString();

      ETG_TRACE_USR4(("NetworkRegistration: Value: %s", status.c_str()));

      if(nullptr != _btStackWrapper)
      {
         _btStackWrapper->onNetworkStatusChanged(objPath, status);
      }
   }
   else
   {
      ETG_TRACE_ERR(("EvolutionGeniviDbusHfpNetworkRegistrationCallBackIf::onNetworkStatusChanged:Property list size exceeds max limit"));
   }
}

void EvolutionGeniviDbusHfpNetworkRegistrationCallBackIf::onNetworkMobileCountryCodeChanged(const std::string& objPath,
      const BTSDbusPropertyList outPropertyList)
{
   ETG_TRACE_USR4(("NetworkRegistration: Property: MobileCountryCode"));

   if(OUTPROPERTY_LIST_SIZE == outPropertyList.size())
   {
      const ::ccdbusif::DbusVariantProperty& outProperty = outPropertyList[0];

      const std::string& mobileCountryCode = outProperty.propData.getString();

      ETG_TRACE_USR4(("NetworkRegistration: Value: %s", mobileCountryCode.c_str()));

      if(nullptr != _btStackWrapper)
      {
         _btStackWrapper->onNetworkMobileCountryCodeChanged(objPath, mobileCountryCode);
      }
   }
   else
   {
      ETG_TRACE_ERR(("EvolutionGeniviDbusHfpNetworkRegistrationCallBackIf::onNetworkMobileCountryCodeChanged:Property list size exceeds max limit"));
   }
}

void EvolutionGeniviDbusHfpNetworkRegistrationCallBackIf::onNetworkOperatorNameChanged(const std::string& objPath,
      const BTSDbusPropertyList outPropertyList)
{
   ETG_TRACE_USR4(("NetworkRegistration: Property: Name"));

   if(OUTPROPERTY_LIST_SIZE == outPropertyList.size())
   {
      const ::ccdbusif::DbusVariantProperty& outProperty = outPropertyList[0];

      const std::string& name = outProperty.propData.getString();

      ETG_TRACE_USR4(("NetworkRegistration: Value: %s", name.c_str()));

      if(nullptr != _btStackWrapper)
      {
         _btStackWrapper->onNetworkOperatorNameChanged(objPath, name);
      }
   }
   else
   {
      ETG_TRACE_ERR(("EvolutionGeniviDbusHfpNetworkRegistrationCallBackIf::onNetworkOperatorNameChanged:Property list size exceeds max limit"));
   }
}

void EvolutionGeniviDbusHfpNetworkRegistrationCallBackIf::onNetworkSignalStrengthChanged(const std::string& objPath,
      const BTSDbusPropertyList outPropertyList)
{
   ETG_TRACE_USR4(("NetworkRegistration: Property: Strength"));

   if(OUTPROPERTY_LIST_SIZE == outPropertyList.size())
   {
      const ::ccdbusif::DbusVariantProperty& outProperty = outPropertyList[0];

      const uint8_t& strength = outProperty.propData.getUInt8();

      ETG_TRACE_USR4(("NetworkRegistration: Value: %d", strength));

      if(nullptr != _btStackWrapper)
      {
         _btStackWrapper->onNetworkSignalStrengthChanged(objPath, strength);
      }
   }
   else
   {
      ETG_TRACE_ERR(("EvolutionGeniviDbusHfpNetworkRegistrationCallBackIf::onNetworkSignalStrengthChanged:Property list size exceeds max limit"));
   }
}

void EvolutionGeniviDbusHfpNetworkRegistrationCallBackIf::onNetworkCurrentOperatorChanged(const std::string& objPath,
      const BTSDbusPropertyList outPropertyList)
{
   ETG_TRACE_USR4(("NetworkRegistration: Property: CurrentOperator"));

   if(OUTPROPERTY_LIST_SIZE == outPropertyList.size())
   {
      const ::ccdbusif::DbusVariantProperty& outProperty = outPropertyList[0];

      const std::string& currentOperator = outProperty.propData.getString();

      ETG_TRACE_USR4(("NetworkRegistration: Value: %s", currentOperator.c_str()));

      if(nullptr != _btStackWrapper)
      {
         _btStackWrapper->onNetworkCurrentOperatorChanged(objPath, currentOperator);
      }
   }
   else
   {
      ETG_TRACE_ERR(("EvolutionGeniviDbusHfpNetworkRegistrationCallBackIf::onNetworkCurrentOperatorChanged:Property list size exceeds max limit"));
   }
}

} //evobtstackwrapper
