/**
 * @file EvolutionGeniviDbusHfpManagerCallBackIf.cpp
 *
 * @swcomponent PhoneCallManager
 *
 * @brief Interface definition for EvolutionGenivi DBUS callback for HfpManager
 *
 * @copyright (C) 2016 Robert Bosch GmbH.
 *            The reproduction, distribution and utilization of this file as
 *            well as the communication of its contents to others without express
 *            authorization is prohibited. Offenders will be held liable for the
 *            payment of damages. All rights reserved in the event of the grant
 *            of a patent, utility model or design.
 *
 * @details
 *
 * @ingroup BtStackWrapper
 */

#include "EvolutionGeniviDbusHfpManagerCallBackIf.h"
#include "IEvolutionGeniviDbusHfpHandsfreeRequestIf.h"
#include "IEvolutionGeniviDbusHfpModemRequestIf.h"
#include "IBtStackWrapper.h"
#include "EvolutionGeniviDbusParser.h"
#include "BtStackWrapperTypesInternal.h"
#include "FwStringUtils.h"
#include "PmAppTrace.h"

using namespace ::org::ofono::Manager;
using namespace ::asf::core;

#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#define ETG_DEFAULT_TRACE_CLASS TR_CLASS_BT_STACK_WRAPPER
#ifdef VARIANT_S_FTR_ENABLE_FW_ETG_USAGE
#include "trcGenProj/Header/EvolutionGeniviDbusHfpManagerCallBackIf.cpp.trc.h"
#define ETG_DEFAULT_TRACE_CLASS TR_CLASS_BT_STACK_WRAPPER
#endif
#endif

namespace evobtstackwrapper {

EvolutionGeniviDbusHfpManagerCallBackIf::EvolutionGeniviDbusHfpManagerCallBackIf()
{
   _btStackWrapper = nullptr;
}

EvolutionGeniviDbusHfpManagerCallBackIf::EvolutionGeniviDbusHfpManagerCallBackIf(IBtStackWrapper* btStackWrapper)
{
   _btStackWrapper = btStackWrapper;
}

EvolutionGeniviDbusHfpManagerCallBackIf::~EvolutionGeniviDbusHfpManagerCallBackIf()
{
   _btStackWrapper = nullptr;
}

void EvolutionGeniviDbusHfpManagerCallBackIf::onProxyAvailable(const ServiceState previousState,
      const ServiceState currentState, const ::std::string& objPath)
{
   (void)(previousState);
   (void)(currentState);
   (void)(objPath);

   //TODO: implement
}

void EvolutionGeniviDbusHfpManagerCallBackIf::onProxyUnavailable(const ServiceState previousState,
      const ServiceState currentState, const ::std::string& objPath)
{
   (void)(previousState);
   (void)(currentState);
   (void)(objPath);

   //TODO: implement
}

void EvolutionGeniviDbusHfpManagerCallBackIf::onGetModemsError(const ::boost::shared_ptr< ManagerProxy >& proxy,
      const ::boost::shared_ptr< GetModemsError >& error)
{
   const std::string& objPath = proxy->getDBusObjectPath();
   const std::string& errorName = error->getName();
   const std::string& errorMessage = error->getMessage();
   const act_t act = error->getAct();

   if(nullptr != _btStackWrapper)
   {
      _btStackWrapper->onGetModemsError(objPath, errorName, errorMessage, act);
   }
}

void EvolutionGeniviDbusHfpManagerCallBackIf::onGetModemsResponse(const ::boost::shared_ptr< ManagerProxy >& proxy,
      const ::boost::shared_ptr< GetModemsResponse >& response)
{
   ETG_TRACE_USR4(("onGetModemsResponse received"));

   const std::string& objPath = proxy->getDBusObjectPath();
   ::std::vector< GetModemsResponseModemsStruct >& modems = response->getModemsMutable();
   //const act_t act = response->getAct();

   ETG_TRACE_USR4(("onGetModemsResponse: Manager: ObjPath: %s", objPath.c_str()));

   for(::std::vector< GetModemsResponseModemsStruct >::iterator it = modems.begin(); it != modems.end(); ++it)
   {
      ETG_TRACE_USR4(("onGetModemsResponse: Modem: ObjPath: %s", it->getElem1().c_str()));

      handleModemAdded(it->getElem1(), it->getElem2Mutable(), ::ccdbusif::BUS_TYPE_SYSTEM, proxy->getDBusBusName(),
            proxy->getDBusObjectPath(), proxy->getInterfaceName());
   }
}

void EvolutionGeniviDbusHfpManagerCallBackIf::onModemAddedError(const ::boost::shared_ptr< ManagerProxy >& proxy,
      const ::boost::shared_ptr< ModemAddedError >& error)
{
   (void)(proxy);
   (void)(error);
   // is never triggered
   ETG_TRACE_ERR(("EvolutionGeniviDbusHfpManagerCallBackIf::onModemAddedError"));
}

void EvolutionGeniviDbusHfpManagerCallBackIf::onModemAddedSignal(const ::boost::shared_ptr< ManagerProxy >& proxy,
      const ::boost::shared_ptr< ModemAddedSignal >& signal)
{
   ETG_TRACE_USR4(("onModemAddedSignal received"));

   handleModemAdded(signal->getPath(), signal->getPropertiesMutable(), ::ccdbusif::BUS_TYPE_SYSTEM,
         proxy->getDBusBusName(), proxy->getDBusObjectPath(), proxy->getInterfaceName());
}

void EvolutionGeniviDbusHfpManagerCallBackIf::onModemRemovedError(const ::boost::shared_ptr< ManagerProxy >& proxy,
      const ::boost::shared_ptr< ModemRemovedError >& error)
{
   (void)(proxy);
   (void)(error);
   // is never triggered
   ETG_TRACE_ERR(("EvolutionGeniviDbusHfpManagerCallBackIf::onModemRemovedError"));
}

void EvolutionGeniviDbusHfpManagerCallBackIf::onModemRemovedSignal(const ::boost::shared_ptr< ManagerProxy >& proxy,
      const ::boost::shared_ptr< ModemRemovedSignal >& signal)
{
   ETG_TRACE_USR4(("onModemRemovedSignal received"));

   (void)(proxy);

   // destroy proxy
   if(nullptr != _btStackWrapper)
   {
      _btStackWrapper->modemRemoved(signal->getPath());
      _btStackWrapper->deviceRemoved(signal->getPath());
   }
}

void EvolutionGeniviDbusHfpManagerCallBackIf::convertBdAddress2InternalValue(::std::string& stringOutput,
      const ::std::string& stringInput)
{
   const ::std::string bdAddressEvolution = "AA:BB:CC:DD:EE:FF";
   const ::std::string bdAddressInternal = "AABBCCDDEEFF";

   if(bdAddressEvolution.size() == stringInput.size())
   {
      stringOutput.reserve(bdAddressInternal.size());

      for(size_t i = 0; i < stringInput.size(); i++)
      {
         if(':' != stringInput[i])
         {
            stringOutput.push_back(stringInput[i]);
         }
      }

      ::fw::convertString2LowerCase(stringOutput);
   }
   else
   {
      // wrong size
      ETG_TRACE_ERR(("EvolutionGeniviDbusHfpManagerCallBackIf::convertBdAddress2InternalValue:Incorrect BT Address size"));
      stringOutput.reserve(bdAddressInternal.size());
      for(size_t i = 0; i < bdAddressInternal.size(); i++)
      {
         stringOutput.push_back('0');
      }
   }
}

void EvolutionGeniviDbusHfpManagerCallBackIf::handleModemAdded(const ::std::string& modemObjPath,
      ::std::map< ::std::string, ::asf::dbus::DBusVariant >& inProperties, const ::ccdbusif::DbusBusType busType,
       const ::std::string& busName, const ::std::string& objPath, const ::std::string& interfaceName)
{
   ::ccdbusif::evolution::EvolutionGeniviDbusParser parser;
   parser.setTraces(true);

   BTSDbusPropertyList ifProperties;
   ::std::vector<int> matchingInterfaces;
   matchingInterfaces.push_back((int)::ccdbusif::evolution::IF_MODEM);

   parser.parseProperties(ifProperties, matchingInterfaces, inProperties, busType, busName, objPath, interfaceName);

   // send mapping
   std::string deviceAddress;
   bool powered = false;

   for(size_t i = 0; i < ifProperties.size(); i++)
   {
      if(::ccdbusif::evolution::HFP_MODEM_SERIAL == (::ccdbusif::evolution::HfpModemProperty)ifProperties[i].propEnum)
      {
         convertBdAddress2InternalValue(deviceAddress, ifProperties[i].propData.getString());
      }
      else if(::ccdbusif::evolution::HFP_MODEM_POWERED ==
            (::ccdbusif::evolution::HfpModemProperty)ifProperties[i].propEnum)
      {
         powered = ifProperties[i].propData.getBool();
      }
   }

   // create proxy
   if((nullptr != _btStackWrapper) && (false == deviceAddress.empty()) && powered)
   {
      _btStackWrapper->modemCreated(deviceAddress, modemObjPath);
   }

   // send data
   /*for(size_t i = 0; i < ifProperties.size(); i++)
   {
      if(::ccdbusif::evolution::HFP_MODEM_INTERFACES ==
       (::ccdbusif::evolution::HfpModemProperty)ifProperties[i].propEnum)
      {
         const ::std::vector<std::string>& strArray = ifProperties[i].propData.getStringArray();
         ::ccdbusif::evolution::Interface item;

         for(size_t j = 0; j < strArray.size(); j++)
         {
            item = parser.getInterface2Enum(strArray[j]);

            if(::ccdbusif::evolution::IF_HANDSFREE == item)
            {
               // create proxy
               if(nullptr != _btStackWrapper)
               {
                  _btStackWrapper->deviceCreated(modemObjPath);
               }
            }
         }

         break;
      }
   }*/
}

} //evobtstackwrapper
