/**
 * @file IEvolutionGeniviDbusHfpSiriRequestIf.h
 *
 * @swcomponent PhoneCallManager
 *
 * @brief Interface definition for EvolutionGenivi DBUS send request for HfpSiri
 *
 * @copyright (C) 2016 Robert Bosch GmbH.
 *            The reproduction, distribution and utilization of this file as
 *            well as the communication of its contents to others without express
 *            authorization is prohibited. Offenders will be held liable for the
 *            payment of damages. All rights reserved in the event of the grant
 *            of a patent, utility model or design.
 *
 * @details
 *
 * @ingroup BtStackWrapper
 */

#ifndef IEvolutionGeniviDbusHfpSiriRequestIf_h
#define IEvolutionGeniviDbusHfpSiriRequestIf_h

#include <stdbool.h>
#include <string>

#include "asf/core/Types.h"

namespace evobtstackwrapper {

// class forward declarations
class IEvolutionGeniviDbusHfpSiriCallBackIf;

/**
 * Send request interface class for HfpSiri.
 */
class IEvolutionGeniviDbusHfpSiriRequestIf
{
public:
   /**
    * Destructor of IEvolutionGeniviDbusHfpSiriRequestIf class
    *
    * @param[in]
    * @param[out]
    * @param[in,out]
    *
    * @return
    */
   virtual ~IEvolutionGeniviDbusHfpSiriRequestIf() {}

   /**
    * Set callback handler.
    *
    * @param[in] callbackIf: callback handler
    * @param[out]
    * @param[in,out]
    *
    * @return void
    */
   virtual void setCallbackIf(IEvolutionGeniviDbusHfpSiriCallBackIf* callbackIf) = 0;

   /**
    * Handle signal for created device.
    *
    * @param[in] objPath: object path of created device
    * @param[out]
    * @param[in,out]
    *
    * @return void
    */
   virtual void deviceCreated(const ::std::string& objPath) = 0;

   /**
    * Handle signal for removed device.
    *
    * @param[in] objPath: object path of removed device
    * @param[out]
    * @param[in,out]
    *
    * @return void
    */
   virtual void deviceRemoved(const ::std::string& objPath) = 0;

   /**
    * Send GetProperties Request.
    *
    * @param[in] objPath: object path of device
    * @param[out]
    * @param[in,out]
    *
    * @return asynchronous completion token
    */
   virtual act_t sendGetPropertiesRequest(const ::std::string& objPath) = 0;

   /**
    * Send Enable Request.
    *
    * @param[in] objPath: object path of device
    * @param[in] vendor_id: vendor id of device
    * @param[in] product_id: product id of device
    * @param[in] version: version of device
    * @param[in] support_type: support type of device
    * @param[out]
    * @param[in,out]
    *
    * @return asynchronous completion token
    */
   virtual act_t sendEnableRequest(const ::std::string& objPath, uint16 vendor_id, uint16 product_id, uint16 version,
         uint16 support_type) = 0;

   /**
    * Send SetNR Request.
    *
    * @param[in] objPath: object path of device
    * @param[in] status: NR status of the device
    * @param[out]
    * @param[in,out]
    *
    * @return asynchronous completion token
    */
   virtual act_t sendSetNRRequest(const ::std::string& objPath, uint8 status) = 0;

   /**
    * Send SetProperty Request.
    *
    * @param[in] objPath: object path of device
    * @param[in] property: property name to be updated
    * @param[in] value: property value to be updated
    * @param[out]
    * @param[in,out]
    *
    * @return asynchronous completion token
    */
   virtual act_t sendSetPropertyRequest(const ::std::string& objPath, const ::std::string& property,
         const bool value) = 0;
};

} //evobtstackwrapper

#endif //IEvolutionGeniviDbusHfpSiriRequestIf_h
