/**
 * @file IEvolutionGeniviDbusHfpHandsfreeRequestIf.h
 *
 * @swcomponent PhoneCallManager
 *
 * @brief Interface definition for EvolutionGenivi DBUS send request for HfpHandsfree
 *
 * @copyright (C) 2016 Robert Bosch GmbH.
 *            The reproduction, distribution and utilization of this file as
 *            well as the communication of its contents to others without express
 *            authorization is prohibited. Offenders will be held liable for the
 *            payment of damages. All rights reserved in the event of the grant
 *            of a patent, utility model or design.
 *
 * @details
 *
 * @ingroup BtStackWrapper
 */

#ifndef IEvolutionGeniviDbusHfpHandsfreeRequestIf_h
#define IEvolutionGeniviDbusHfpHandsfreeRequestIf_h

#include <stdbool.h>
#include <string>
#include <vector>

#include "asf/core/Types.h"

namespace evobtstackwrapper {

// class forward declarations
class IEvolutionGeniviDbusHfpHandsfreeCallBackIf;

/**
 * Send request interface class for HfpHandsfree.
 */
class IEvolutionGeniviDbusHfpHandsfreeRequestIf
{
public:
   /**
    * Destructor of IEvolutionGeniviDbusHfpHandsfreeRequestIf class
    *
    * @param[in]
    * @param[out]
    * @param[in,out]
    *
    * @return
    */
   virtual ~IEvolutionGeniviDbusHfpHandsfreeRequestIf() {}

   /**
    * Set callback handler.
    *
    * @param[in] callbackIf: callback handler
    * @param[out]
    * @param[in,out]
    *
    * @return void
    */
   virtual void setCallbackIf(IEvolutionGeniviDbusHfpHandsfreeCallBackIf* callbackIf) = 0;

   /**
    * Handle signal for created device.
    *
    * @param[in] objPath: object path of created device
    * @param[out]
    * @param[in,out]
    *
    * @return void
    */
   virtual void deviceCreated(const ::std::string& objPath) = 0;

   /**
    * Handle signal for removed device.
    *
    * @param[in] objPath: object path of removed device
    * @param[out]
    * @param[in,out]
    *
    * @return void
    */
   virtual void deviceRemoved(const ::std::string& objPath) = 0;

   /**
    * Send GetProperties Request.
    *
    * @param[in] objPath: object path of device
    * @param[out]
    * @param[in,out]
    *
    * @return asynchronous completion token
    */
   virtual act_t sendGetPropertiesRequest(const ::std::string& objPath) = 0;

   /**
    * Send SetProperty Request.
    *
    * @param[in] objPath: object path of device
    * @param[in] property: property name
    * @param[in] value: property value
    * @param[out]
    * @param[in,out]
    *
    * @return asynchronous completion token
    */
   virtual act_t sendSetPropertyRequest(const ::std::string& objPath, const ::std::string& property,
         const bool value) = 0;

   /**
    * Send SetProperty Request.
    *
    * @param[in] objPath: object path of device
    * @param[in] property: property name
    * @param[in] value: property value
    * @param[out]
    * @param[in,out]
    *
    * @return asynchronous completion token
    */
   virtual act_t sendSetPropertyRequest(const ::std::string& objPath, const ::std::string& property,
         const ::std::string& value) = 0;

   /**
    * Send RequestPhoneNumber Request.
    *
    * @param[in] objPath: object path of device
    * @param[out]
    * @param[in,out]
    *
    * @return asynchronous completion token
    */
   virtual act_t sendRequestPhoneNumberRequest(const ::std::string& objPath) = 0;

   /**
    * Send GetSubscriber Request.
    *
    * @param[in] objPath: object path of device
    * @param[out]
    * @param[in,out]
    *
    * @return asynchronous completion token
    */
   virtual act_t sendGetSubscriberRequest(const ::std::string& objPath) = 0;

   /**
    * Send SendCustomATCommand Request.
    *
    * @param[in] objPath: object path of device
    * @param[in] data: data
    * @param[out]
    * @param[in,out]
    *
    * @return asynchronous completion token
    */
   virtual act_t sendSendCustomATCommandRequest(const ::std::string& objPath, const ::std::vector< uint8 >& data) = 0;

   /**
    * Send HFUpdateIndicatorsValue Request.
    *
    * @param[in] objPath: object path of device
    * @param[in] number : number
    * @param[in] value  : value
    * @param[out]
    * @param[in,out]
    *
    * @return asynchronous completion token
    */
   virtual act_t sendHFUpdateIndicatorsValueRequest(const ::std::string& objPath, uint16 number, uint32 value) = 0;
};

} //evobtstackwrapper

#endif //IEvolutionGeniviDbusHfpHandsfreeRequestIf_h
